/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie.storage.ldb;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.protobuf.ByteString;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.Unpooled;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.DefaultThreadFactory;
import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.PrimitiveIterator;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.StampedLock;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.CheckpointSource;
import org.apache.bookkeeper.bookie.Checkpointer;
import org.apache.bookkeeper.bookie.CompactableLedgerStorage;
import org.apache.bookkeeper.bookie.EntryLocation;
import org.apache.bookkeeper.bookie.GarbageCollectionStatus;
import org.apache.bookkeeper.bookie.GarbageCollectorThread;
import org.apache.bookkeeper.bookie.LastAddConfirmedUpdateNotification;
import org.apache.bookkeeper.bookie.LedgerCache;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.bookie.LedgerEntryPage;
import org.apache.bookkeeper.bookie.LedgerStorage;
import org.apache.bookkeeper.bookie.StateManager;
import org.apache.bookkeeper.bookie.storage.EntryLogger;
import org.apache.bookkeeper.bookie.storage.ldb.DbLedgerStorageDataFormats;
import org.apache.bookkeeper.bookie.storage.ldb.DbLedgerStorageStats;
import org.apache.bookkeeper.bookie.storage.ldb.EntryLocationIndex;
import org.apache.bookkeeper.bookie.storage.ldb.KeyValueStorage;
import org.apache.bookkeeper.bookie.storage.ldb.KeyValueStorageRocksDB;
import org.apache.bookkeeper.bookie.storage.ldb.LedgerMetadataIndex;
import org.apache.bookkeeper.bookie.storage.ldb.ReadCache;
import org.apache.bookkeeper.bookie.storage.ldb.TransientLedgerInfo;
import org.apache.bookkeeper.bookie.storage.ldb.WriteCache;
import org.apache.bookkeeper.common.util.Watcher;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.stats.ThreadRegistry;
import org.apache.bookkeeper.util.MathUtils;
import org.apache.bookkeeper.util.collections.ConcurrentLongHashMap;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.mutable.MutableLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SingleDirectoryDbLedgerStorage
implements CompactableLedgerStorage {
    private final EntryLogger entryLogger;
    private final LedgerMetadataIndex ledgerIndex;
    private final EntryLocationIndex entryLocationIndex;
    private final ConcurrentLongHashMap<TransientLedgerInfo> transientLedgerInfoCache;
    private final GarbageCollectorThread gcThread;
    protected volatile WriteCache writeCache;
    protected volatile WriteCache writeCacheBeingFlushed;
    private final ReadCache readCache;
    private final StampedLock writeCacheRotationLock = new StampedLock();
    protected final ReentrantLock flushMutex = new ReentrantLock();
    protected final AtomicBoolean hasFlushBeenTriggered = new AtomicBoolean(false);
    private final AtomicBoolean isFlushOngoing = new AtomicBoolean(false);
    private static String dbStoragerExecutorName = "db-storage";
    private final ExecutorService executor = Executors.newSingleThreadExecutor((ThreadFactory)new DefaultThreadFactory(dbStoragerExecutorName){

        protected Thread newThread(Runnable r, String name) {
            return super.newThread(ThreadRegistry.registerThread((Runnable)r, (String)dbStoragerExecutorName), name);
        }
    });
    private final ScheduledExecutorService cleanupExecutor = Executors.newSingleThreadScheduledExecutor((ThreadFactory)new DefaultThreadFactory("db-storage-cleanup"));
    private final CopyOnWriteArrayList<LedgerStorage.LedgerDeletionListener> ledgerDeletionListeners = Lists.newCopyOnWriteArrayList();
    private CheckpointSource checkpointSource = CheckpointSource.DEFAULT;
    private CheckpointSource.Checkpoint lastCheckpoint = CheckpointSource.Checkpoint.MIN;
    private final long writeCacheMaxSize;
    private final long readCacheMaxSize;
    private final int readAheadCacheBatchSize;
    private final long readAheadCacheBatchBytesSize;
    private final long maxThrottleTimeNanos;
    private final DbLedgerStorageStats dbLedgerStorageStats;
    private static final long DEFAULT_MAX_THROTTLE_TIME_MILLIS = TimeUnit.SECONDS.toMillis(10L);
    private final long maxReadAheadBytesSize;
    private final Counter flushExecutorTime;
    private final boolean singleLedgerDirs;
    private static final Logger log = LoggerFactory.getLogger(SingleDirectoryDbLedgerStorage.class);
    private static final Map<LedgerStorage.StorageState, Integer> stateBitmaps = ImmutableMap.of((Object)((Object)LedgerStorage.StorageState.NEEDS_INTEGRITY_CHECK), (Object)1);

    public SingleDirectoryDbLedgerStorage(ServerConfiguration conf, LedgerManager ledgerManager, LedgerDirsManager ledgerDirsManager, LedgerDirsManager indexDirsManager, EntryLogger entryLogger, StatsLogger statsLogger, ByteBufAllocator allocator, long writeCacheSize, long readCacheSize, int readAheadCacheBatchSize, long readAheadCacheBatchBytesSize) throws IOException {
        String ledgerBaseDir;
        Preconditions.checkArgument((ledgerDirsManager.getAllLedgerDirs().size() == 1 ? 1 : 0) != 0, (Object)"Db implementation only allows for one storage dir");
        String indexBaseDir = ledgerBaseDir = ledgerDirsManager.getAllLedgerDirs().get(0).getPath();
        if (CollectionUtils.isEmpty(indexDirsManager.getAllLedgerDirs()) || ledgerBaseDir.equals(indexDirsManager.getAllLedgerDirs().get(0).getPath())) {
            log.info("indexDir is equals ledgerBaseDir, creating single directory db ledger storage on {}", (Object)indexBaseDir);
        } else {
            indexBaseDir = indexDirsManager.getAllLedgerDirs().get(0).getPath();
            log.info("indexDir is specified a separate dir, creating single directory db ledger storage on {}", (Object)indexBaseDir);
        }
        StatsLogger ledgerIndexDirStatsLogger = statsLogger.scopeLabel("ledgerDir", ledgerBaseDir).scopeLabel("indexDir", indexBaseDir);
        this.writeCacheMaxSize = writeCacheSize;
        this.writeCache = new WriteCache(allocator, this.writeCacheMaxSize / 2L);
        this.writeCacheBeingFlushed = new WriteCache(allocator, this.writeCacheMaxSize / 2L);
        this.singleLedgerDirs = conf.getLedgerDirs().length == 1;
        this.readCacheMaxSize = readCacheSize;
        this.readAheadCacheBatchSize = readAheadCacheBatchSize;
        this.readAheadCacheBatchBytesSize = readAheadCacheBatchBytesSize;
        this.maxReadAheadBytesSize = this.readCacheMaxSize / 2L;
        long maxThrottleTimeMillis = conf.getLong("dbStorage_maxThrottleTimeMs", DEFAULT_MAX_THROTTLE_TIME_MILLIS);
        this.maxThrottleTimeNanos = TimeUnit.MILLISECONDS.toNanos(maxThrottleTimeMillis);
        this.readCache = new ReadCache(allocator, this.readCacheMaxSize);
        this.ledgerIndex = new LedgerMetadataIndex(conf, KeyValueStorageRocksDB.factory, indexBaseDir, ledgerIndexDirStatsLogger);
        this.entryLocationIndex = new EntryLocationIndex(conf, KeyValueStorageRocksDB.factory, indexBaseDir, ledgerIndexDirStatsLogger);
        this.transientLedgerInfoCache = ConcurrentLongHashMap.newBuilder().expectedItems(16384).concurrencyLevel(Runtime.getRuntime().availableProcessors() * 2).build();
        this.cleanupExecutor.scheduleAtFixedRate(this::cleanupStaleTransientLedgerInfo, 10L, 10L, TimeUnit.MINUTES);
        this.entryLogger = entryLogger;
        this.gcThread = new GarbageCollectorThread(conf, ledgerManager, ledgerDirsManager, this, entryLogger, ledgerIndexDirStatsLogger);
        this.dbLedgerStorageStats = new DbLedgerStorageStats(ledgerIndexDirStatsLogger, () -> this.writeCache.size() + this.writeCacheBeingFlushed.size(), () -> this.writeCache.count() + this.writeCacheBeingFlushed.count(), () -> this.readCache.size(), () -> this.readCache.count());
        this.flushExecutorTime = ledgerIndexDirStatsLogger.getThreadScopedCounter("db-storage-thread-time");
        this.executor.submit(() -> this.flushExecutorTime.addLatency(0L, TimeUnit.NANOSECONDS));
        ledgerDirsManager.addLedgerDirsListener(this.getLedgerDirsListener());
        if (!ledgerBaseDir.equals(indexBaseDir)) {
            indexDirsManager.addLedgerDirsListener(this.getLedgerDirsListener());
        }
    }

    @Override
    public void initialize(ServerConfiguration conf, LedgerManager ledgerManager, LedgerDirsManager ledgerDirsManager, LedgerDirsManager indexDirsManager, StatsLogger statsLogger, ByteBufAllocator allocator) throws IOException {
    }

    @Override
    public void setStateManager(StateManager stateManager) {
    }

    @Override
    public void setCheckpointSource(CheckpointSource checkpointSource) {
        this.checkpointSource = checkpointSource;
    }

    @Override
    public void setCheckpointer(Checkpointer checkpointer) {
    }

    private void cleanupStaleTransientLedgerInfo() {
        this.transientLedgerInfoCache.removeIf((ledgerId, ledgerInfo) -> {
            boolean isStale = ledgerInfo.isStale();
            if (isStale) {
                ledgerInfo.close();
            }
            return isStale;
        });
    }

    @Override
    public void start() {
        this.gcThread.start();
    }

    @Override
    public void forceGC() {
        this.gcThread.enableForceGC();
    }

    @Override
    public void forceGC(boolean forceMajor, boolean forceMinor) {
        this.gcThread.enableForceGC(forceMajor, forceMinor);
    }

    @Override
    public boolean isInForceGC() {
        return this.gcThread.isInForceGC();
    }

    @Override
    public void suspendMinorGC() {
        this.gcThread.suspendMinorGC();
    }

    @Override
    public void suspendMajorGC() {
        this.gcThread.suspendMajorGC();
    }

    @Override
    public void resumeMinorGC() {
        this.gcThread.resumeMinorGC();
    }

    @Override
    public void resumeMajorGC() {
        this.gcThread.resumeMajorGC();
    }

    @Override
    public boolean isMajorGcSuspended() {
        return this.gcThread.isMajorGcSuspend();
    }

    @Override
    public boolean isMinorGcSuspended() {
        return this.gcThread.isMinorGcSuspend();
    }

    @Override
    public void entryLocationCompact() {
        if (this.entryLocationIndex.isCompacting()) {
            log.info("Compacting directory {}, skipping this entryLocationCompaction this time.", (Object)this.entryLocationIndex.getEntryLocationDBPath());
            return;
        }
        this.cleanupExecutor.execute(() -> {
            try {
                log.info("Trigger entry location index RocksDB compact.");
                this.entryLocationIndex.compact();
            }
            catch (Throwable t) {
                log.warn("Failed to trigger entry location index RocksDB compact", t);
            }
        });
    }

    @Override
    public boolean isEntryLocationCompacting() {
        return this.entryLocationIndex.isCompacting();
    }

    @Override
    public List<String> getEntryLocationDBPath() {
        return Lists.newArrayList((Object[])new String[]{this.entryLocationIndex.getEntryLocationDBPath()});
    }

    @Override
    public void shutdown() throws InterruptedException {
        try {
            this.flush();
            this.gcThread.shutdown();
            this.entryLogger.close();
            this.cleanupExecutor.shutdown();
            this.cleanupExecutor.awaitTermination(1L, TimeUnit.SECONDS);
            this.ledgerIndex.close();
            this.entryLocationIndex.close();
            this.writeCache.close();
            this.writeCacheBeingFlushed.close();
            this.readCache.close();
            this.executor.shutdown();
        }
        catch (IOException e) {
            log.error("Error closing db storage", (Throwable)e);
        }
    }

    @Override
    public boolean ledgerExists(long ledgerId) throws IOException {
        try {
            DbLedgerStorageDataFormats.LedgerData ledgerData = this.ledgerIndex.get(ledgerId);
            if (log.isDebugEnabled()) {
                log.debug("Ledger exists. ledger: {} : {}", (Object)ledgerId, (Object)ledgerData.getExists());
            }
            return ledgerData.getExists();
        }
        catch (Bookie.NoLedgerException nle) {
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean entryExists(long ledgerId, long entryId) throws IOException, BookieException {
        boolean inCache;
        if (entryId == -1L) {
            return false;
        }
        long stamp = this.writeCacheRotationLock.tryOptimisticRead();
        WriteCache localWriteCache = this.writeCache;
        WriteCache localWriteCacheBeingFlushed = this.writeCacheBeingFlushed;
        if (!this.writeCacheRotationLock.validate(stamp)) {
            stamp = this.writeCacheRotationLock.readLock();
            try {
                localWriteCache = this.writeCache;
                localWriteCacheBeingFlushed = this.writeCacheBeingFlushed;
            }
            finally {
                this.writeCacheRotationLock.unlockRead(stamp);
            }
        }
        boolean bl = inCache = localWriteCache.hasEntry(ledgerId, entryId) || localWriteCacheBeingFlushed.hasEntry(ledgerId, entryId) || this.readCache.hasEntry(ledgerId, entryId);
        if (inCache) {
            return true;
        }
        long entryLocation = this.entryLocationIndex.getLocation(ledgerId, entryId);
        if (entryLocation != 0L) {
            return true;
        }
        this.throwIfLimbo(ledgerId);
        return false;
    }

    @Override
    public boolean isFenced(long ledgerId) throws IOException, BookieException {
        boolean isFenced = this.ledgerIndex.get(ledgerId).getFenced();
        if (log.isDebugEnabled()) {
            log.debug("ledger: {}, isFenced: {}.", (Object)ledgerId, (Object)isFenced);
        }
        if (!isFenced) {
            this.throwIfLimbo(ledgerId);
        }
        return isFenced;
    }

    @Override
    public boolean setFenced(long ledgerId) throws IOException {
        TransientLedgerInfo ledgerInfo;
        boolean changed;
        if (log.isDebugEnabled()) {
            log.debug("Set fenced. ledger: {}", (Object)ledgerId);
        }
        if ((changed = this.ledgerIndex.setFenced(ledgerId)) && null != (ledgerInfo = this.transientLedgerInfoCache.get(ledgerId))) {
            ledgerInfo.notifyWatchers(Long.MAX_VALUE);
        }
        return changed;
    }

    @Override
    public void setMasterKey(long ledgerId, byte[] masterKey) throws IOException {
        if (log.isDebugEnabled()) {
            log.debug("Set master key. ledger: {}", (Object)ledgerId);
        }
        this.ledgerIndex.setMasterKey(ledgerId, masterKey);
    }

    @Override
    public byte[] readMasterKey(long ledgerId) throws IOException, BookieException {
        if (log.isDebugEnabled()) {
            log.debug("Read master key. ledger: {}", (Object)ledgerId);
        }
        return this.ledgerIndex.get(ledgerId).getMasterKey().toByteArray();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long addEntry(ByteBuf entry) throws IOException, BookieException {
        long startTime = MathUtils.nowInNano();
        long ledgerId = entry.getLong(entry.readerIndex());
        long entryId = entry.getLong(entry.readerIndex() + 8);
        long lac = entry.getLong(entry.readerIndex() + 16);
        if (log.isDebugEnabled()) {
            log.debug("Add entry. {}@{}, lac = {}", new Object[]{ledgerId, entryId, lac});
        }
        long stamp = this.writeCacheRotationLock.tryOptimisticRead();
        boolean inserted = false;
        inserted = this.writeCache.put(ledgerId, entryId, entry);
        if (!this.writeCacheRotationLock.validate(stamp)) {
            stamp = this.writeCacheRotationLock.readLock();
            try {
                inserted = this.writeCache.put(ledgerId, entryId, entry);
            }
            finally {
                this.writeCacheRotationLock.unlockRead(stamp);
            }
        }
        if (!inserted) {
            this.triggerFlushAndAddEntry(ledgerId, entryId, entry);
        }
        this.updateCachedLacIfNeeded(ledgerId, lac);
        this.recordSuccessfulEvent(this.dbLedgerStorageStats.getAddEntryStats(), startTime);
        return entryId;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void triggerFlushAndAddEntry(long ledgerId, long entryId, ByteBuf entry) throws IOException, BookieException {
        long throttledStartTime = MathUtils.nowInNano();
        this.dbLedgerStorageStats.getThrottledWriteRequests().inc();
        long absoluteTimeoutNanos = System.nanoTime() + this.maxThrottleTimeNanos;
        while (System.nanoTime() < absoluteTimeoutNanos) {
            if (!this.isFlushOngoing.get() && this.hasFlushBeenTriggered.compareAndSet(false, true)) {
                log.info("Write cache is full, triggering flush");
                this.executor.execute(() -> {
                    long startTime = System.nanoTime();
                    try {
                        this.flush();
                    }
                    catch (IOException e) {
                        log.error("Error during flush", (Throwable)e);
                    }
                    finally {
                        this.flushExecutorTime.addLatency(MathUtils.elapsedNanos((long)startTime), TimeUnit.NANOSECONDS);
                    }
                });
            }
            long stamp = this.writeCacheRotationLock.readLock();
            try {
                if (this.writeCache.put(ledgerId, entryId, entry)) {
                    this.recordSuccessfulEvent(this.dbLedgerStorageStats.getThrottledWriteStats(), throttledStartTime);
                    return;
                }
            }
            finally {
                this.writeCacheRotationLock.unlockRead(stamp);
            }
            try {
                Thread.sleep(1L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IOException("Interrupted when adding entry " + ledgerId + "@" + entryId);
            }
        }
        this.dbLedgerStorageStats.getRejectedWriteRequests().inc();
        this.recordFailedEvent(this.dbLedgerStorageStats.getThrottledWriteStats(), throttledStartTime);
        throw new BookieException.OperationRejectedException();
    }

    @Override
    public ByteBuf getEntry(long ledgerId, long entryId) throws IOException, BookieException {
        long startTime = MathUtils.nowInNano();
        try {
            ByteBuf entry = this.doGetEntry(ledgerId, entryId);
            this.recordSuccessfulEvent(this.dbLedgerStorageStats.getReadEntryStats(), startTime);
            return entry;
        }
        catch (IOException e) {
            this.recordFailedEvent(this.dbLedgerStorageStats.getReadEntryStats(), startTime);
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ByteBuf doGetEntry(long ledgerId, long entryId) throws IOException, BookieException {
        long entryLocation;
        ByteBuf entry;
        if (log.isDebugEnabled()) {
            log.debug("Get Entry: {}@{}", (Object)ledgerId, (Object)entryId);
        }
        if (entryId == -1L) {
            return this.getLastEntry(ledgerId);
        }
        long stamp = this.writeCacheRotationLock.tryOptimisticRead();
        WriteCache localWriteCache = this.writeCache;
        WriteCache localWriteCacheBeingFlushed = this.writeCacheBeingFlushed;
        if (!this.writeCacheRotationLock.validate(stamp)) {
            stamp = this.writeCacheRotationLock.readLock();
            try {
                localWriteCache = this.writeCache;
                localWriteCacheBeingFlushed = this.writeCacheBeingFlushed;
            }
            finally {
                this.writeCacheRotationLock.unlockRead(stamp);
            }
        }
        if ((entry = localWriteCache.get(ledgerId, entryId)) != null) {
            this.dbLedgerStorageStats.getWriteCacheHitCounter().inc();
            return entry;
        }
        entry = localWriteCacheBeingFlushed.get(ledgerId, entryId);
        if (entry != null) {
            this.dbLedgerStorageStats.getWriteCacheHitCounter().inc();
            return entry;
        }
        this.dbLedgerStorageStats.getWriteCacheMissCounter().inc();
        entry = this.readCache.get(ledgerId, entryId);
        if (entry != null) {
            this.dbLedgerStorageStats.getReadCacheHitCounter().inc();
            return entry;
        }
        this.dbLedgerStorageStats.getReadCacheMissCounter().inc();
        long locationIndexStartNano = MathUtils.nowInNano();
        try {
            entryLocation = this.entryLocationIndex.getLocation(ledgerId, entryId);
            if (entryLocation == 0L) {
                this.throwIfLimbo(ledgerId);
                throw new Bookie.NoEntryException(ledgerId, entryId);
            }
        }
        finally {
            this.dbLedgerStorageStats.getReadFromLocationIndexTime().addLatency(MathUtils.elapsedNanos((long)locationIndexStartNano), TimeUnit.NANOSECONDS);
        }
        long readEntryStartNano = MathUtils.nowInNano();
        try {
            entry = this.entryLogger.readEntry(ledgerId, entryId, entryLocation);
        }
        finally {
            this.dbLedgerStorageStats.getReadFromEntryLogTime().addLatency(MathUtils.elapsedNanos((long)readEntryStartNano), TimeUnit.NANOSECONDS);
        }
        this.readCache.put(ledgerId, entryId, entry);
        long nextEntryLocation = entryLocation + 4L + (long)entry.readableBytes();
        this.fillReadAheadCache(ledgerId, entryId + 1L, nextEntryLocation);
        return entry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void fillReadAheadCache(long orginalLedgerId, long firstEntryId, long firstEntryLocation) {
        long readAheadStartNano = MathUtils.nowInNano();
        int count = 0;
        long size = 0L;
        try {
            long firstEntryLogId;
            long currentEntryLogId = firstEntryLogId = firstEntryLocation >> 32;
            long currentEntryLocation = firstEntryLocation;
            while (this.chargeReadAheadCache(count, size) && currentEntryLogId == firstEntryLogId) {
                ByteBuf entry = this.entryLogger.readEntry(orginalLedgerId, firstEntryId, currentEntryLocation);
                try {
                    long currentEntryLedgerId = entry.getLong(0);
                    long currentEntryId = entry.getLong(8);
                    if (currentEntryLedgerId == orginalLedgerId) {
                        this.readCache.put(orginalLedgerId, currentEntryId, entry);
                        ++count;
                        ++firstEntryId;
                        size += (long)entry.readableBytes();
                        currentEntryLogId = (currentEntryLocation += (long)(4 + entry.readableBytes())) >> 32;
                        continue;
                    }
                    break;
                }
                finally {
                    ReferenceCountUtil.release((Object)entry);
                }
            }
        }
        catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Exception during read ahead for ledger: {}: e", (Object)orginalLedgerId, (Object)e);
            }
        }
        finally {
            this.dbLedgerStorageStats.getReadAheadBatchCountStats().registerSuccessfulValue((long)count);
            this.dbLedgerStorageStats.getReadAheadBatchSizeStats().registerSuccessfulValue(size);
            this.dbLedgerStorageStats.getReadAheadTime().addLatency(MathUtils.elapsedNanos((long)readAheadStartNano), TimeUnit.NANOSECONDS);
        }
    }

    protected boolean chargeReadAheadCache(int currentReadAheadCount, long currentReadAheadBytes) {
        boolean chargeSizeCondition;
        boolean bl = chargeSizeCondition = currentReadAheadCount < this.readAheadCacheBatchSize && currentReadAheadBytes < this.maxReadAheadBytesSize;
        if (chargeSizeCondition && this.readAheadCacheBatchBytesSize > 0L) {
            chargeSizeCondition = currentReadAheadBytes < this.readAheadCacheBatchBytesSize;
        }
        return chargeSizeCondition;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ByteBuf getLastEntry(long ledgerId) throws IOException, BookieException {
        this.throwIfLimbo(ledgerId);
        long stamp = this.writeCacheRotationLock.readLock();
        try {
            ByteBuf entry = this.writeCache.getLastEntry(ledgerId);
            if (entry != null) {
                if (log.isDebugEnabled()) {
                    long foundLedgerId = entry.readLong();
                    long entryId = entry.readLong();
                    entry.resetReaderIndex();
                    if (log.isDebugEnabled()) {
                        log.debug("Found last entry for ledger {} in write cache: {}@{}", new Object[]{ledgerId, foundLedgerId, entryId});
                    }
                }
                this.dbLedgerStorageStats.getWriteCacheHitCounter().inc();
                ByteBuf foundLedgerId = entry;
                return foundLedgerId;
            }
            entry = this.writeCacheBeingFlushed.getLastEntry(ledgerId);
            if (entry != null) {
                if (log.isDebugEnabled()) {
                    entry.readLong();
                    long entryId = entry.readLong();
                    entry.resetReaderIndex();
                    if (log.isDebugEnabled()) {
                        log.debug("Found last entry for ledger {} in write cache being flushed: {}", (Object)ledgerId, (Object)entryId);
                    }
                }
                this.dbLedgerStorageStats.getWriteCacheHitCounter().inc();
                ByteBuf byteBuf = entry;
                return byteBuf;
            }
        }
        finally {
            this.writeCacheRotationLock.unlockRead(stamp);
        }
        this.dbLedgerStorageStats.getWriteCacheMissCounter().inc();
        long locationIndexStartNano = MathUtils.nowInNano();
        long lastEntryId = this.entryLocationIndex.getLastEntryInLedger(ledgerId);
        if (log.isDebugEnabled()) {
            log.debug("Found last entry for ledger {} in db: {}", (Object)ledgerId, (Object)lastEntryId);
        }
        long entryLocation = this.entryLocationIndex.getLocation(ledgerId, lastEntryId);
        this.dbLedgerStorageStats.getReadFromLocationIndexTime().addLatency(MathUtils.elapsedNanos((long)locationIndexStartNano), TimeUnit.NANOSECONDS);
        long readEntryStartNano = MathUtils.nowInNano();
        ByteBuf content = this.entryLogger.readEntry(ledgerId, lastEntryId, entryLocation);
        this.dbLedgerStorageStats.getReadFromEntryLogTime().addLatency(MathUtils.elapsedNanos((long)readEntryStartNano), TimeUnit.NANOSECONDS);
        return content;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    boolean isFlushRequired() {
        long stamp = this.writeCacheRotationLock.readLock();
        try {
            boolean bl = !this.writeCache.isEmpty();
            return bl;
        }
        finally {
            this.writeCacheRotationLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void checkpoint(CheckpointSource.Checkpoint checkpoint) throws IOException {
        CheckpointSource.Checkpoint thisCheckpoint = this.checkpointSource.newCheckpoint();
        if (this.lastCheckpoint.compareTo(checkpoint) > 0) {
            return;
        }
        this.flushMutex.lock();
        long startTime = -1L;
        try {
            startTime = MathUtils.nowInNano();
        }
        catch (Throwable e) {
            this.flushMutex.unlock();
            throw new IOException(e);
        }
        try {
            if (this.writeCache.isEmpty()) {
                return;
            }
            this.swapWriteCache();
            long sizeToFlush = this.writeCacheBeingFlushed.size();
            if (log.isDebugEnabled()) {
                log.debug("Flushing entries. count: {} -- size {} Mb", (Object)this.writeCacheBeingFlushed.count(), (Object)((double)sizeToFlush / 1024.0 / 1024.0));
            }
            KeyValueStorage.Batch batch = this.entryLocationIndex.newBatch();
            this.writeCacheBeingFlushed.forEach((ledgerId, entryId, entry) -> {
                long location = this.entryLogger.addEntry(ledgerId, entry);
                this.entryLocationIndex.addLocation(batch, ledgerId, entryId, location);
            });
            long entryLoggerStart = MathUtils.nowInNano();
            this.entryLogger.flush();
            this.recordSuccessfulEvent(this.dbLedgerStorageStats.getFlushEntryLogStats(), entryLoggerStart);
            long batchFlushStartTime = MathUtils.nowInNano();
            batch.flush();
            batch.close();
            this.recordSuccessfulEvent(this.dbLedgerStorageStats.getFlushLocationIndexStats(), batchFlushStartTime);
            if (log.isDebugEnabled()) {
                log.debug("DB batch flushed time : {} s", (Object)((double)MathUtils.elapsedNanos((long)batchFlushStartTime) / (double)TimeUnit.SECONDS.toNanos(1L)));
            }
            long ledgerIndexStartTime = MathUtils.nowInNano();
            this.ledgerIndex.flush();
            this.recordSuccessfulEvent(this.dbLedgerStorageStats.getFlushLedgerIndexStats(), ledgerIndexStartTime);
            this.lastCheckpoint = thisCheckpoint;
            this.writeCacheBeingFlushed.clear();
            double flushTimeSeconds = (double)MathUtils.elapsedNanos((long)startTime) / (double)TimeUnit.SECONDS.toNanos(1L);
            double flushThroughput = (double)sizeToFlush / 1024.0 / 1024.0 / flushTimeSeconds;
            if (log.isDebugEnabled()) {
                log.debug("Flushing done time {} s -- Written {} MB/s", (Object)flushTimeSeconds, (Object)flushThroughput);
            }
            this.recordSuccessfulEvent(this.dbLedgerStorageStats.getFlushStats(), startTime);
            this.dbLedgerStorageStats.getFlushSizeStats().registerSuccessfulValue(sizeToFlush);
        }
        catch (IOException e) {
            this.recordFailedEvent(this.dbLedgerStorageStats.getFlushStats(), startTime);
            throw e;
        }
        finally {
            try {
                this.cleanupExecutor.execute(() -> {
                    try {
                        if (log.isDebugEnabled()) {
                            log.debug("Removing deleted ledgers from db indexes");
                        }
                        this.entryLocationIndex.removeOffsetFromDeletedLedgers();
                        this.ledgerIndex.removeDeletedLedgers();
                    }
                    catch (Throwable t) {
                        log.warn("Failed to cleanup db indexes", t);
                    }
                });
                this.isFlushOngoing.set(false);
            }
            finally {
                this.flushMutex.unlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void swapWriteCache() {
        long stamp = this.writeCacheRotationLock.writeLock();
        try {
            WriteCache tmp = this.writeCacheBeingFlushed;
            this.writeCacheBeingFlushed = this.writeCache;
            this.writeCache = tmp;
            this.hasFlushBeenTriggered.set(false);
        }
        finally {
            try {
                this.isFlushOngoing.set(true);
            }
            finally {
                this.writeCacheRotationLock.unlockWrite(stamp);
            }
        }
    }

    @Override
    public void flush() throws IOException {
        CheckpointSource.Checkpoint cp = this.checkpointSource.newCheckpoint();
        this.checkpoint(cp);
        if (this.singleLedgerDirs) {
            this.checkpointSource.checkpointComplete(cp, true);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void deleteLedger(long ledgerId) throws IOException {
        if (log.isDebugEnabled()) {
            log.debug("Deleting ledger {}", (Object)ledgerId);
        }
        long stamp = this.writeCacheRotationLock.readLock();
        try {
            this.writeCache.deleteLedger(ledgerId);
        }
        finally {
            this.writeCacheRotationLock.unlockRead(stamp);
        }
        this.entryLocationIndex.delete(ledgerId);
        this.ledgerIndex.delete(ledgerId);
        int size = this.ledgerDeletionListeners.size();
        for (int i = 0; i < size; ++i) {
            LedgerStorage.LedgerDeletionListener listener = this.ledgerDeletionListeners.get(i);
            listener.ledgerDeleted(ledgerId);
        }
        TransientLedgerInfo tli = this.transientLedgerInfoCache.remove(ledgerId);
        if (tli != null) {
            tli.close();
        }
    }

    @Override
    public Iterable<Long> getActiveLedgersInRange(long firstLedgerId, long lastLedgerId) throws IOException {
        return this.ledgerIndex.getActiveLedgersInRange(firstLedgerId, lastLedgerId);
    }

    @Override
    public void updateEntriesLocations(Iterable<EntryLocation> locations) throws IOException {
        this.flushMutex.lock();
        this.flushMutex.unlock();
        this.entryLocationIndex.updateLocations(locations);
    }

    @VisibleForTesting
    EntryLogger getEntryLogger() {
        return this.entryLogger;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long getLastAddConfirmed(long ledgerId) throws IOException, BookieException {
        long lac;
        this.throwIfLimbo(ledgerId);
        TransientLedgerInfo ledgerInfo = this.transientLedgerInfoCache.get(ledgerId);
        long l = lac = null != ledgerInfo ? ledgerInfo.getLastAddConfirmed() : Long.MIN_VALUE;
        if (lac == Long.MIN_VALUE) {
            ByteBuf bb = this.getEntry(ledgerId, -1L);
            try {
                bb.skipBytes(16);
                lac = bb.readLong();
                lac = this.getOrAddLedgerInfo(ledgerId).setLastAddConfirmed(lac);
            }
            finally {
                ReferenceCountUtil.release((Object)bb);
            }
        }
        return lac;
    }

    @Override
    public boolean waitForLastAddConfirmedUpdate(long ledgerId, long previousLAC, Watcher<LastAddConfirmedUpdateNotification> watcher) throws IOException {
        return this.getOrAddLedgerInfo(ledgerId).waitForLastAddConfirmedUpdate(previousLAC, watcher);
    }

    @Override
    public void cancelWaitForLastAddConfirmedUpdate(long ledgerId, Watcher<LastAddConfirmedUpdateNotification> watcher) throws IOException {
        this.getOrAddLedgerInfo(ledgerId).cancelWaitForLastAddConfirmedUpdate(watcher);
    }

    @Override
    public void setExplicitLac(long ledgerId, ByteBuf lac) throws IOException {
        TransientLedgerInfo ledgerInfo = this.getOrAddLedgerInfo(ledgerId);
        ledgerInfo.setExplicitLac(lac);
        this.ledgerIndex.setExplicitLac(ledgerId, lac);
        ledgerInfo.notifyWatchers(Long.MAX_VALUE);
    }

    @Override
    public ByteBuf getExplicitLac(long ledgerId) throws IOException, BookieException {
        TransientLedgerInfo ledgerInfo;
        this.throwIfLimbo(ledgerId);
        if (log.isDebugEnabled()) {
            log.debug("getExplicitLac ledger {}", (Object)ledgerId);
        }
        if ((ledgerInfo = this.getOrAddLedgerInfo(ledgerId)).getExplicitLac() != null) {
            if (log.isDebugEnabled()) {
                log.debug("getExplicitLac ledger {} returned from TransientLedgerInfo", (Object)ledgerId);
            }
            return ledgerInfo.getExplicitLac();
        }
        DbLedgerStorageDataFormats.LedgerData ledgerData = this.ledgerIndex.get(ledgerId);
        if (!ledgerData.hasExplicitLac()) {
            if (log.isDebugEnabled()) {
                log.debug("getExplicitLac ledger {} missing from LedgerData", (Object)ledgerId);
            }
            return null;
        }
        if (log.isDebugEnabled()) {
            log.debug("getExplicitLac ledger {} returned from LedgerData", (Object)ledgerId);
        }
        ByteString persistedLac = ledgerData.getExplicitLac();
        ledgerInfo.setExplicitLac(Unpooled.wrappedBuffer((byte[])persistedLac.toByteArray()));
        return ledgerInfo.getExplicitLac();
    }

    private TransientLedgerInfo getOrAddLedgerInfo(long ledgerId) {
        return this.transientLedgerInfoCache.computeIfAbsent(ledgerId, l -> new TransientLedgerInfo(l, this.ledgerIndex));
    }

    private void updateCachedLacIfNeeded(long ledgerId, long lac) {
        TransientLedgerInfo tli = this.transientLedgerInfoCache.get(ledgerId);
        if (tli != null) {
            tli.setLastAddConfirmed(lac);
        }
    }

    @Override
    public void flushEntriesLocationsIndex() throws IOException {
    }

    @SuppressFBWarnings(value={"RCN_REDUNDANT_NULLCHECK_WOULD_HAVE_BEEN_A_NPE"})
    public long addLedgerToIndex(long ledgerId, boolean isFenced, byte[] masterKey, LedgerCache.PageEntriesIterable pages) throws Exception {
        DbLedgerStorageDataFormats.LedgerData ledgerData = DbLedgerStorageDataFormats.LedgerData.newBuilder().setExists(true).setFenced(isFenced).setMasterKey(ByteString.copyFrom((byte[])masterKey)).build();
        this.ledgerIndex.set(ledgerId, ledgerData);
        MutableLong numberOfEntries = new MutableLong();
        KeyValueStorage.Batch batch = this.entryLocationIndex.newBatch();
        for (LedgerCache.PageEntries page : pages) {
            LedgerEntryPage lep = page.getLEP();
            Throwable throwable = null;
            try {
                lep.getEntries((entryId, location) -> {
                    this.entryLocationIndex.addLocation(batch, ledgerId, entryId, location);
                    numberOfEntries.increment();
                    return true;
                });
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (lep == null) continue;
                if (throwable != null) {
                    try {
                        lep.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                lep.close();
            }
        }
        this.ledgerIndex.flush();
        batch.flush();
        batch.close();
        return numberOfEntries.longValue();
    }

    @Override
    public void registerLedgerDeletionListener(LedgerStorage.LedgerDeletionListener listener) {
        this.ledgerDeletionListeners.add(listener);
    }

    public EntryLocationIndex getEntryLocationIndex() {
        return this.entryLocationIndex;
    }

    private void recordSuccessfulEvent(OpStatsLogger logger, long startTimeNanos) {
        logger.registerSuccessfulEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
    }

    private void recordFailedEvent(OpStatsLogger logger, long startTimeNanos) {
        logger.registerFailedEvent(MathUtils.elapsedNanos((long)startTimeNanos), TimeUnit.NANOSECONDS);
    }

    @Override
    public List<GarbageCollectionStatus> getGarbageCollectionStatus() {
        return Collections.singletonList(this.gcThread.getGarbageCollectionStatus());
    }

    @Override
    public PrimitiveIterator.OfLong getListOfEntriesOfLedger(long ledgerId) throws IOException {
        throw new UnsupportedOperationException("getListOfEntriesOfLedger method is currently unsupported for SingleDirectoryDbLedgerStorage");
    }

    private LedgerDirsManager.LedgerDirsListener getLedgerDirsListener() {
        return new LedgerDirsManager.LedgerDirsListener(){

            @Override
            public void diskAlmostFull(File disk) {
                if (SingleDirectoryDbLedgerStorage.this.gcThread.isForceGCAllowWhenNoSpace()) {
                    SingleDirectoryDbLedgerStorage.this.gcThread.enableForceGC();
                } else {
                    SingleDirectoryDbLedgerStorage.this.gcThread.suspendMajorGC();
                }
            }

            @Override
            public void diskFull(File disk) {
                if (SingleDirectoryDbLedgerStorage.this.gcThread.isForceGCAllowWhenNoSpace()) {
                    SingleDirectoryDbLedgerStorage.this.gcThread.enableForceGC();
                } else {
                    SingleDirectoryDbLedgerStorage.this.gcThread.suspendMajorGC();
                    SingleDirectoryDbLedgerStorage.this.gcThread.suspendMinorGC();
                }
            }

            @Override
            public void allDisksFull(boolean highPriorityWritesAllowed) {
                if (SingleDirectoryDbLedgerStorage.this.gcThread.isForceGCAllowWhenNoSpace()) {
                    SingleDirectoryDbLedgerStorage.this.gcThread.enableForceGC();
                } else {
                    SingleDirectoryDbLedgerStorage.this.gcThread.suspendMajorGC();
                    SingleDirectoryDbLedgerStorage.this.gcThread.suspendMinorGC();
                }
            }

            @Override
            public void diskWritable(File disk) {
                if (SingleDirectoryDbLedgerStorage.this.gcThread.isForceGCAllowWhenNoSpace()) {
                    SingleDirectoryDbLedgerStorage.this.gcThread.disableForceGC();
                } else {
                    SingleDirectoryDbLedgerStorage.this.gcThread.resumeMajorGC();
                    SingleDirectoryDbLedgerStorage.this.gcThread.resumeMinorGC();
                }
            }

            @Override
            public void diskJustWritable(File disk) {
                if (SingleDirectoryDbLedgerStorage.this.gcThread.isForceGCAllowWhenNoSpace()) {
                    SingleDirectoryDbLedgerStorage.this.gcThread.enableForceGC();
                } else {
                    SingleDirectoryDbLedgerStorage.this.gcThread.resumeMinorGC();
                }
            }
        };
    }

    @Override
    public void setLimboState(long ledgerId) throws IOException {
        if (log.isDebugEnabled()) {
            log.debug("setLimboState. ledger: {}", (Object)ledgerId);
        }
        this.ledgerIndex.setLimbo(ledgerId);
    }

    @Override
    public boolean hasLimboState(long ledgerId) throws IOException {
        if (log.isDebugEnabled()) {
            log.debug("hasLimboState. ledger: {}", (Object)ledgerId);
        }
        return this.ledgerIndex.get(ledgerId).getLimbo();
    }

    @Override
    public void clearLimboState(long ledgerId) throws IOException {
        if (log.isDebugEnabled()) {
            log.debug("clearLimboState. ledger: {}", (Object)ledgerId);
        }
        this.ledgerIndex.clearLimbo(ledgerId);
    }

    private void throwIfLimbo(long ledgerId) throws IOException, BookieException {
        if (this.hasLimboState(ledgerId)) {
            if (log.isDebugEnabled()) {
                log.debug("Accessing ledger({}) in limbo state, throwing exception", (Object)ledgerId);
            }
            throw BookieException.create(-111);
        }
    }

    @Override
    public EnumSet<LedgerStorage.StorageState> getStorageStateFlags() throws IOException {
        int flags = this.ledgerIndex.getStorageStateFlags();
        EnumSet<LedgerStorage.StorageState> flagsEnum = EnumSet.noneOf(LedgerStorage.StorageState.class);
        for (Map.Entry<LedgerStorage.StorageState, Integer> e : stateBitmaps.entrySet()) {
            int value = e.getValue();
            if ((flags & value) == value) {
                flagsEnum.add(e.getKey());
            }
            flags &= ~value;
        }
        Preconditions.checkState((flags == 0 ? 1 : 0) != 0, (Object)("Unknown storage state flag found " + flags));
        return flagsEnum;
    }

    @Override
    public void setStorageStateFlag(LedgerStorage.StorageState flag) throws IOException {
        Preconditions.checkArgument((boolean)stateBitmaps.containsKey((Object)flag), (Object)("Unsupported flag " + (Object)((Object)flag)));
        int flagInt = stateBitmaps.get((Object)flag);
        int newFlags;
        int curFlags;
        while (!this.ledgerIndex.setStorageStateFlags(curFlags = this.ledgerIndex.getStorageStateFlags(), newFlags = curFlags | flagInt)) {
            log.info("Conflict updating storage state flags {} -> {}, retrying", (Object)curFlags, (Object)newFlags);
        }
        return;
    }

    @Override
    public void clearStorageStateFlag(LedgerStorage.StorageState flag) throws IOException {
        Preconditions.checkArgument((boolean)stateBitmaps.containsKey((Object)flag), (Object)("Unsupported flag " + (Object)((Object)flag)));
        int flagInt = stateBitmaps.get((Object)flag);
        int newFlags;
        int curFlags;
        while (!this.ledgerIndex.setStorageStateFlags(curFlags = this.ledgerIndex.getStorageStateFlags(), newFlags = curFlags & ~flagInt)) {
            log.info("Conflict updating storage state flags {} -> {}, retrying", (Object)curFlags, (Object)newFlags);
        }
        return;
    }

    @VisibleForTesting
    DbLedgerStorageStats getDbLedgerStorageStats() {
        return this.dbLedgerStorageStats;
    }

    public static interface LedgerLoggerProcessor {
        public void process(long var1, long var3, long var5);
    }
}

