/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.bmc.auth.internal;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.oracle.bmc.auth.exception.InstancePrincipalUnavailableException;
import com.oracle.bmc.auth.internal.JWK;
import com.oracle.bmc.auth.internal.X509CertificateWithOriginalPem;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.RSAPublicKeySpec;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.x500.AttributeTypeAndValue;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AuthUtils {
    private static final Logger LOG = LoggerFactory.getLogger(AuthUtils.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final char[] HEX_ARRAY = "0123456789ABCDEF".toCharArray();

    public static String getFingerPrint(X509Certificate certificate) {
        Preconditions.checkNotNull((Object)certificate);
        try {
            byte[] encodedCertificate = AuthUtils.getEncodedCertificate(certificate);
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            md.update(encodedCertificate);
            String fingerprint = AuthUtils.getHex(md.digest());
            return AuthUtils.formatStringWithSeparator(fingerprint);
        }
        catch (NoSuchAlgorithmException | CertificateEncodingException e) {
            throw new Error(e.getMessage());
        }
    }

    private static String formatStringWithSeparator(String fingerprint) {
        int length = fingerprint.length();
        char[] format = new char[length * 3 / 2 - 1];
        int j = 0;
        for (int i = 0; i < length - 2; i += 2) {
            format[j++] = fingerprint.charAt(i);
            format[j++] = fingerprint.charAt(i + 1);
            format[j++] = 58;
        }
        format[j++] = fingerprint.charAt(length - 2);
        format[j] = fingerprint.charAt(length - 1);
        return String.valueOf(format);
    }

    private static String getHex(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; ++j) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_ARRAY[v & 0xF];
        }
        return new String(hexChars);
    }

    public static Optional<RSAPublicKey> toPublicKeyFromJson(String json) {
        Preconditions.checkArgument((!StringUtils.isBlank((CharSequence)json) ? 1 : 0) != 0);
        Optional<JWK> jwk = AuthUtils.toJwk(json);
        if (!jwk.isPresent()) {
            return Optional.absent();
        }
        return AuthUtils.toPublicKeyFromJwk((JWK)jwk.get());
    }

    public static Optional<JWK> toJwk(String json) {
        Preconditions.checkArgument((!StringUtils.isBlank((CharSequence)json) ? 1 : 0) != 0);
        try {
            JWK jwk = (JWK)OBJECT_MAPPER.readValue(json, JWK.class);
            return Optional.of((Object)jwk);
        }
        catch (IOException e) {
            LOG.debug("Exception reading or de-serializing jwk", (Throwable)e);
            return Optional.absent();
        }
    }

    public static Optional<RSAPublicKey> toPublicKeyFromJwk(JWK jwk) {
        Preconditions.checkNotNull((Object)jwk);
        try {
            RSAPublicKey key = (RSAPublicKey)KeyFactory.getInstance("RSA").generatePublic(new RSAPublicKeySpec(new BigInteger(1, Base64.decodeBase64((String)jwk.getModulus())), new BigInteger(1, Base64.decodeBase64((String)jwk.getPublicExponent()))));
            return Optional.of((Object)key);
        }
        catch (Exception ex) {
            LOG.debug("Failed to construct public key from JWK", (Throwable)ex);
            return Optional.absent();
        }
    }

    public static byte[] toByteArrayFromRSAPrivateKey(RSAPrivateKey key) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (JcaPEMWriter writer = new JcaPEMWriter((Writer)new OutputStreamWriter((OutputStream)baos, StandardCharsets.UTF_8));){
            writer.writeObject((Object)key);
            writer.flush();
        }
        catch (IOException e) {
            throw new IllegalStateException("Unable to write PEM object", e);
        }
        return baos.toByteArray();
    }

    public static String base64EncodeNoChunking(RSAPublicKey publicKey) {
        return new String(Base64.encodeBase64((byte[])publicKey.getEncoded(), (boolean)false), StandardCharsets.UTF_8);
    }

    public static String base64EncodeNoChunking(X509Certificate certificate) throws CertificateEncodingException {
        return new String(Base64.encodeBase64((byte[])AuthUtils.getEncodedCertificate(certificate), (boolean)false), StandardCharsets.UTF_8);
    }

    private static byte[] getEncodedCertificate(X509Certificate certificate) throws CertificateEncodingException {
        if (certificate instanceof X509CertificateWithOriginalPem) {
            return AuthUtils.getEncodedCertificateFromPem(((X509CertificateWithOriginalPem)certificate).getPemEncodedCertificate());
        }
        return certificate.getEncoded();
    }

    private static byte[] getEncodedCertificateFromPem(String pemEncodedCertificate) {
        return Base64.decodeBase64((String)pemEncodedCertificate.replace("-----BEGIN CERTIFICATE-----", "").replace("-----END CERTIFICATE-----", ""));
    }

    public static String getTenantIdFromCertificate(X509Certificate certificate) {
        Preconditions.checkNotNull((Object)certificate);
        X500Name name = new X500Name(certificate.getSubjectX500Principal().getName());
        Optional tenancyId = AuthUtils.getValue(name, BCStyle.OU, "opc-tenant").or(AuthUtils.getValue(name, BCStyle.O, "opc-identity"));
        if (tenancyId.isPresent()) {
            return (String)tenancyId.get();
        }
        throw new InstancePrincipalUnavailableException("The certificate does not contain tenant id.");
    }

    private static Optional<String> getValue(X500Name name, ASN1ObjectIdentifier id, String key) {
        String prefix = key + ":";
        for (RDN rdn : name.getRDNs(id)) {
            for (AttributeTypeAndValue typeAndValue : rdn.getTypesAndValues()) {
                String value = typeAndValue.getValue().toString();
                if (!value.startsWith(prefix)) continue;
                return Optional.of((Object)value.substring(prefix.length()));
            }
        }
        return Optional.absent();
    }

    private AuthUtils() {
    }
}

