/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.spark.sql.extension

import org.apache.gluten.expression._
import org.apache.gluten.extension.ExpressionExtensionTrait

import org.apache.spark.sql.catalyst.expressions.{Attribute, Expression}

case class CustomAddExpressionTransformer(
    substraitExprName: String,
    left: ExpressionTransformer,
    right: ExpressionTransformer,
    original: Expression)
  extends BinaryExpressionTransformer

case class CustomerExpressionTransformer() extends ExpressionExtensionTrait {

  lazy val expressionSigs = Seq(
    Sig[CustomAdd]("add")
  )

  /** Generate the extension expressions list, format: Sig[XXXExpression]("XXXExpressionName") */
  override def expressionSigList: Seq[Sig] = expressionSigs

  /** Replace extension expression to transformer. */
  override def replaceWithExtensionExpressionTransformer(
      substraitExprName: String,
      expr: Expression,
      attributeSeq: Seq[Attribute]): ExpressionTransformer = expr match {
    case custom: CustomAdd =>
      CustomAddExpressionTransformer(
        substraitExprName,
        ExpressionConverter.replaceWithExpressionTransformer(custom.left, attributeSeq),
        ExpressionConverter.replaceWithExpressionTransformer(custom.right, attributeSeq),
        custom
      )
    case other =>
      throw new UnsupportedOperationException(
        s"${expr.getClass} or $expr is not currently supported.")
  }
}
