/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.common.persistence;

import java.io.IOException;
import java.util.concurrent.Callable;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.persistence.ResourceStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExponentialBackoffRetry {
    private static final Logger logger = LoggerFactory.getLogger(ExponentialBackoffRetry.class);
    private final ResourceStore store;
    private final KylinConfig config;
    private final int baseSleepTimeMs;
    private final int maxSleepTimeMs;
    private long firstSleepTime;
    private int retryCount;

    public ExponentialBackoffRetry(ResourceStore store) {
        this.store = store;
        this.config = store.getConfig();
        this.baseSleepTimeMs = this.config.getResourceStoreReconnectBaseMs();
        this.maxSleepTimeMs = this.config.getResourceStoreReconnectMaxMs();
        this.retryCount = 0;
    }

    public <V> V doWithRetry(Callable<V> callable) throws IOException {
        V result = null;
        boolean done = false;
        while (!done) {
            try {
                result = callable.call();
                done = true;
            }
            catch (Throwable ex) {
                boolean shouldRetry = this.checkIfAllowRetry(ex);
                if (shouldRetry) continue;
                this.throwIOException(ex);
            }
        }
        return result;
    }

    private void throwIOException(Throwable ex) throws IOException {
        if (ex instanceof IOException) {
            throw (IOException)ex;
        }
        if (ex instanceof RuntimeException) {
            throw (RuntimeException)ex;
        }
        if (ex instanceof Error) {
            throw (Error)ex;
        }
        throw new IOException(ex);
    }

    private boolean checkIfAllowRetry(Throwable ex) {
        if (this.config.isResourceStoreReconnectEnabled() && this.store.isUnreachableException(ex)) {
            if (this.isTimeOut(this.config.getResourceStoreReconnectTimeoutMs())) {
                logger.error("Reconnect to resource store timeout, abandoning...", ex);
                return false;
            }
            long waitMs = this.getSleepTimeMs();
            long seconds = waitMs / 1000L;
            logger.info("Will try to re-connect after {} seconds.", (Object)seconds);
            try {
                Thread.sleep(waitMs);
            }
            catch (InterruptedException e) {
                throw new RuntimeException("Current thread for resource store's CRUD is interrupted, abandoning...");
            }
            this.increaseRetryCount();
            return true;
        }
        return false;
    }

    private long getSleepTimeMs() {
        long ms;
        if (this.retryCount == 0) {
            this.firstSleepTime = System.currentTimeMillis();
        }
        if ((ms = (long)this.baseSleepTimeMs * (1L << this.retryCount)) > (long)this.maxSleepTimeMs) {
            ms = this.maxSleepTimeMs;
        }
        return ms;
    }

    private void increaseRetryCount() {
        ++this.retryCount;
    }

    private boolean isTimeOut(long timeoutMs) {
        return this.retryCount != 0 && System.currentTimeMillis() - this.firstSleepTime >= timeoutMs;
    }
}

