/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.metrics.lib.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import org.apache.kylin.metrics.lib.ActiveReservoirListener;
import org.apache.kylin.metrics.lib.Record;
import org.apache.kylin.metrics.lib.impl.AbstractActiveReservoir;
import org.apache.kylin.shaded.com.google.common.collect.Lists;
import org.apache.kylin.shaded.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockingReservoir
extends AbstractActiveReservoir {
    private static final Logger logger = LoggerFactory.getLogger(BlockingReservoir.class);
    private static final int MAX_QUEUE_SIZE = 500000;
    private final BlockingQueue<Record> recordsQueue;
    private final Thread scheduledReporter;
    private final int minReportSize;
    private final int maxReportSize;
    private final long maxReportTime;
    private List<Record> records;

    public BlockingReservoir() {
        this(100, 500);
    }

    public BlockingReservoir(int minReportSize, int maxReportSize) {
        this(minReportSize, maxReportSize, 10);
    }

    public BlockingReservoir(int minReportSize, int maxReportSize, int maxReportTime) {
        this(minReportSize, maxReportSize, maxReportTime, 500000);
    }

    public BlockingReservoir(int minReportSize, int maxReportSize, int maxReportTime, int maxQueueSize) {
        Preconditions.checkArgument(minReportSize > 0, "minReportSize should be larger than 0");
        Preconditions.checkArgument(maxReportSize >= minReportSize, "maxReportSize should not be less than minBatchSize");
        Preconditions.checkArgument(maxReportTime > 0, "maxReportTime should be larger than 0");
        this.maxReportSize = maxReportSize;
        this.maxReportTime = (long)(maxReportTime * 60) * 1000L;
        this.recordsQueue = maxQueueSize <= 0 ? new LinkedBlockingQueue<Record>() : new LinkedBlockingQueue(maxQueueSize);
        this.minReportSize = minReportSize;
        this.listeners = Lists.newArrayList();
        this.records = Lists.newArrayListWithExpectedSize(this.maxReportSize);
        this.scheduledReporter = new ThreadFactoryBuilder().setNameFormat("metrics-blocking-reservoir-scheduler-%d").build().newThread(new ReporterRunnable());
    }

    @Override
    public void update(Record record) {
        if (!this.isReady) {
            logger.info("Current reservoir is not ready for update record");
            return;
        }
        try {
            this.recordsQueue.put(record);
        }
        catch (InterruptedException e) {
            logger.warn("Thread is interrupted during putting value to blocking queue.", e);
        }
        catch (IllegalArgumentException e) {
            logger.warn("The record queue may be full");
        }
    }

    @Override
    public int size() {
        return this.recordsQueue.size();
    }

    private void onRecordUpdate(boolean ifAll) {
        if (ifAll) {
            this.records = Lists.newArrayList();
            this.recordsQueue.drainTo(this.records);
            logger.info("Will report {} metrics records", (Object)this.records.size());
        } else {
            this.records.clear();
            this.recordsQueue.drainTo(this.records, this.maxReportSize);
            logger.info("Will report {} metrics records, remaining {} records", (Object)this.records.size(), (Object)this.size());
        }
        boolean ifSucceed = true;
        for (ActiveReservoirListener listener : this.listeners) {
            if (this.notifyListenerOfUpdatedRecord(listener, this.records)) continue;
            ifSucceed = false;
            logger.warn("It fails to notify listener " + listener.toString() + " of updated record size " + this.records.size());
        }
        if (!ifSucceed) {
            this.notifyListenerHAOfUpdatedRecord(this.records);
        }
    }

    private boolean notifyListenerOfUpdatedRecord(ActiveReservoirListener listener, List<Record> records) {
        return listener.onRecordUpdate(records);
    }

    private boolean notifyListenerHAOfUpdatedRecord(List<Record> records) {
        logger.info("The HA listener " + this.listenerHA.toString() + " for updated record size " + records.size() + " will be started");
        if (!this.notifyListenerOfUpdatedRecord(this.listenerHA, records)) {
            logger.error("The HA listener also fails!!!");
            return false;
        }
        return true;
    }

    @VisibleForTesting
    void notifyUpdate() {
        this.onRecordUpdate(false);
    }

    @VisibleForTesting
    void setReady() {
        super.start();
    }

    @Override
    public void start() {
        this.setReady();
        this.scheduledReporter.start();
    }

    @Override
    public void stop() {
        super.stop();
        this.scheduledReporter.interrupt();
        try {
            this.scheduledReporter.join();
        }
        catch (InterruptedException e) {
            logger.warn("Interrupted during join");
            throw new RuntimeException(e);
        }
    }

    class ReporterRunnable
    implements Runnable {
        ReporterRunnable() {
        }

        @Override
        public void run() {
            long startTime = System.currentTimeMillis();
            while (BlockingReservoir.this.isReady) {
                if (BlockingReservoir.this.size() <= 0) {
                    logger.info("There's no record in the blocking queue.");
                    this.sleep();
                    startTime = System.currentTimeMillis();
                    continue;
                }
                if (BlockingReservoir.this.size() < BlockingReservoir.this.minReportSize && System.currentTimeMillis() - startTime < BlockingReservoir.this.maxReportTime) {
                    logger.info("The number of records in the blocking queue is less than {} and the duration from last reporting is less than {} ms. Will delay to report!", (Object)BlockingReservoir.this.minReportSize, (Object)BlockingReservoir.this.maxReportTime);
                    this.sleep();
                    continue;
                }
                BlockingReservoir.this.onRecordUpdate(false);
                startTime = System.currentTimeMillis();
            }
            BlockingReservoir.this.onRecordUpdate(true);
            logger.info("Reporter finishes reporting metrics.");
        }

        private void sleep() {
            try {
                Thread.sleep(60000L);
            }
            catch (InterruptedException e) {
                logger.warn("Interrupted during running");
            }
        }
    }
}

