/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.stream.core.storage;

import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.input.ReversedLinesFileReader;
import org.apache.kylin.common.util.Bytes;
import org.apache.kylin.common.util.JsonUtil;
import org.apache.kylin.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.kylin.shaded.com.google.common.collect.Lists;
import org.apache.kylin.stream.core.exception.IllegalStorageException;
import org.apache.kylin.stream.core.storage.CheckPoint;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CheckPointStore {
    static final int CP_FILE_MAX_NUM = 5;
    private static final String CP_START = "====================";
    private static final String CP_END = "####################";
    private static final long DAY_TIMESTAMP_BASE = 86400000L;
    private static final String CP_FILE_PREFIX = "CP-";
    private static Logger logger = LoggerFactory.getLogger(CheckPointStore.class);
    private final File checkPointFolder;
    private final String cubeName;
    private final int maxNumOfCPFile;

    public CheckPointStore(String cubeName, File checkPointParent) {
        this(cubeName, checkPointParent, 5);
    }

    public CheckPointStore(String cubeName, File checkPointParent, int keepCPFileNum) {
        this.cubeName = cubeName;
        this.checkPointFolder = new File(checkPointParent, ".cp");
        if (this.checkPointFolder.exists() && !this.checkPointFolder.isDirectory()) {
            this.checkPointFolder.delete();
        }
        if (!this.checkPointFolder.exists()) {
            this.checkPointFolder.mkdirs();
        }
        this.maxNumOfCPFile = keepCPFileNum;
    }

    @VisibleForTesting
    File[] getCheckPointFiles() {
        File[] cpFiles = this.checkPointFolder.listFiles(new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                return pathname.getName().startsWith(CheckPointStore.CP_FILE_PREFIX);
            }
        });
        if (cpFiles.length == 0) {
            return null;
        }
        return cpFiles;
    }

    private File getLatestCheckPointFile() {
        File[] cpFiles = this.getCheckPointFiles();
        if (cpFiles == null || cpFiles.length == 0) {
            return null;
        }
        File latestCheckPointFile = cpFiles[0];
        long latestTimestamp = this.getTimestampFromFileName(latestCheckPointFile.getName());
        for (int i = 1; i < cpFiles.length; ++i) {
            long curTimestamp = this.getTimestampFromFileName(cpFiles[i].getName());
            if (curTimestamp <= latestTimestamp) continue;
            latestTimestamp = curTimestamp;
            latestCheckPointFile = cpFiles[i];
        }
        return latestCheckPointFile;
    }

    private File getCheckPointFile(CheckPoint cp) {
        File checkPointFile = new File(this.checkPointFolder, this.getFileNameFromTimestamp(cp.getCheckPointTime()));
        if (!checkPointFile.exists()) {
            try {
                checkPointFile.createNewFile();
                this.deleteOldCPFiles();
            }
            catch (IOException e) {
                throw new IllegalStorageException(e);
            }
        }
        return checkPointFile;
    }

    @VisibleForTesting
    void deleteOldCPFiles() {
        File[] cpFiles = this.getCheckPointFiles();
        if (cpFiles == null || cpFiles.length <= this.maxNumOfCPFile) {
            return;
        }
        ArrayList<File> cpFileList = Lists.newArrayList(cpFiles);
        Collections.sort(cpFileList, new Comparator<File>(){

            @Override
            public int compare(File o1, File o2) {
                return o1.getName().compareTo(o2.getName());
            }
        });
        for (File file : cpFileList.subList(0, cpFileList.size() - this.maxNumOfCPFile)) {
            logger.info("going to delete checkpoint file " + file.getName());
            System.out.println("going to delete checkpoint file " + file.getName());
            file.delete();
        }
    }

    private String getFileNameFromTimestamp(long timestamp) {
        return CP_FILE_PREFIX + timestamp / 86400000L * 86400000L;
    }

    private long getTimestampFromFileName(String cpFileName) {
        return Long.valueOf(cpFileName.substring(CP_FILE_PREFIX.length()));
    }

    public void saveCheckPoint(CheckPoint cp) {
        try (FileOutputStream outputStream = FileUtils.openOutputStream((File)this.getCheckPointFile(cp), (boolean)true);){
            String jsonCP = JsonUtil.writeValueAsIndentString(cp);
            outputStream.write(Bytes.toBytes(this.wrapCheckPointString(jsonCP)));
            outputStream.flush();
        }
        catch (Exception e) {
            logger.error("CheckPoint error for cube " + this.cubeName, e);
        }
    }

    private String wrapCheckPointString(String checkpoint) {
        String lineSeparator = System.lineSeparator();
        return CP_START + lineSeparator + checkpoint + lineSeparator + CP_END + lineSeparator;
    }

    public CheckPoint getLatestCheckPoint() {
        return this.getLatestCheckPoint(this.getLatestCheckPointFile());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private CheckPoint getLatestCheckPoint(File checkPointFile) {
        if (checkPointFile == null) {
            return null;
        }
        CheckPoint cp = null;
        try (ReversedLinesFileReader fileReader = new ReversedLinesFileReader(checkPointFile, 4096, Charset.forName("UTF-8"));){
            String line = fileReader.readLine();
            while (!CP_END.equals(line) && line != null) {
                line = fileReader.readLine();
            }
            if (line == null) {
                CheckPoint checkPoint = null;
                return checkPoint;
            }
            LinkedList<String> stack = Lists.newLinkedList();
            line = fileReader.readLine();
            while (!CP_START.equals(line) && line != null) {
                stack.push(line);
                line = fileReader.readLine();
            }
            if (line == null) {
                CheckPoint checkPoint = null;
                return checkPoint;
            }
            StringBuilder cpJson = new StringBuilder();
            while (!stack.isEmpty()) {
                cpJson.append((String)stack.pop());
            }
            cp = JsonUtil.readValue(cpJson.toString(), CheckPoint.class);
            return cp;
        }
        catch (IOException e) {
            logger.error("error when parse checkpoint");
        }
        return cp;
    }

    public void removeAllCheckPoints() {
        File[] cpFiles = this.getCheckPointFiles();
        if (cpFiles == null || cpFiles.length == 0) {
            return;
        }
        for (File cpFile : cpFiles) {
            this.removeCheckPoints(cpFile);
        }
    }

    private void removeCheckPoints(File checkPointFile) {
        try {
            FileUtils.write((File)checkPointFile, (CharSequence)"");
        }
        catch (IOException e) {
            logger.error("error when remove all checkpoints");
        }
    }
}

