/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.maven.it;

import java.io.File;

import org.junit.jupiter.api.Test;

/**
 * This is a test set for <a href="https://github.com/apache/maven/issues/11399">GH-11399</a>.
 *
 * Verifies that using flatten-maven-plugin with updatePomFile=true does not cause a false
 * parent cycle detection error during install phase. The issue occurred when the plugin
 * updated the POM file reference, causing the consumer POM builder to incorrectly detect
 * a cycle between the project and its parent.
 *
 * @see <a href="https://github.com/mojohaus/flatten-maven-plugin">flatten-maven-plugin</a>
 */
public class MavenITgh11399FlattenPluginParentCycleTest extends AbstractMavenIntegrationTestCase {

    public MavenITgh11399FlattenPluginParentCycleTest() {
        super("(4.0.0-rc-3,)");
    }

    /**
     * Verify that flatten-maven-plugin with updatePomFile=true and parent expansion
     * does not cause a false parent cycle detection error during install.
     *
     * The error was:
     * "The parents form a cycle: org.apache:apache:35 -> /path/to/pom.xml -> org.apache:apache:35"
     *
     * @throws Exception in case of failure
     */
    @Test
    public void testFlattenPluginWithParentExpansionDoesNotCauseCycle() throws Exception {
        File testDir = extractResources("/gh-11399-flatten-plugin-parent-cycle");

        Verifier verifier = newVerifier(testDir.getAbsolutePath());
        verifier.setAutoclean(false);
        verifier.deleteArtifacts("org.apache.maven.its.mng8750");
        verifier.addCliArgument("install");
        verifier.execute();
        verifier.verifyErrorFreeLog();

        // Verify that the flattened POM was created
        verifier.verifyFilePresent("target/.flattened-pom.xml");
    }
}

