﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class UntagOpenIDConnectProviderRequest : public IAMRequest {
 public:
  AWS_IAM_API UntagOpenIDConnectProviderRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UntagOpenIDConnectProvider"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ARN of the OIDC provider in IAM from which you want to remove tags.</p>
   * <p>This parameter allows (through its <a
   * href="http://wikipedia.org/wiki/regex">regex pattern</a>) a string of characters
   * consisting of upper and lowercase alphanumeric characters with no spaces. You
   * can also include any of the following characters: _+=,.@-</p>
   */
  inline const Aws::String& GetOpenIDConnectProviderArn() const { return m_openIDConnectProviderArn; }
  inline bool OpenIDConnectProviderArnHasBeenSet() const { return m_openIDConnectProviderArnHasBeenSet; }
  template <typename OpenIDConnectProviderArnT = Aws::String>
  void SetOpenIDConnectProviderArn(OpenIDConnectProviderArnT&& value) {
    m_openIDConnectProviderArnHasBeenSet = true;
    m_openIDConnectProviderArn = std::forward<OpenIDConnectProviderArnT>(value);
  }
  template <typename OpenIDConnectProviderArnT = Aws::String>
  UntagOpenIDConnectProviderRequest& WithOpenIDConnectProviderArn(OpenIDConnectProviderArnT&& value) {
    SetOpenIDConnectProviderArn(std::forward<OpenIDConnectProviderArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of key names as a simple array of strings. The tags with matching keys
   * are removed from the specified OIDC provider.</p>
   */
  inline const Aws::Vector<Aws::String>& GetTagKeys() const { return m_tagKeys; }
  inline bool TagKeysHasBeenSet() const { return m_tagKeysHasBeenSet; }
  template <typename TagKeysT = Aws::Vector<Aws::String>>
  void SetTagKeys(TagKeysT&& value) {
    m_tagKeysHasBeenSet = true;
    m_tagKeys = std::forward<TagKeysT>(value);
  }
  template <typename TagKeysT = Aws::Vector<Aws::String>>
  UntagOpenIDConnectProviderRequest& WithTagKeys(TagKeysT&& value) {
    SetTagKeys(std::forward<TagKeysT>(value));
    return *this;
  }
  template <typename TagKeysT = Aws::String>
  UntagOpenIDConnectProviderRequest& AddTagKeys(TagKeysT&& value) {
    m_tagKeysHasBeenSet = true;
    m_tagKeys.emplace_back(std::forward<TagKeysT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_openIDConnectProviderArn;

  Aws::Vector<Aws::String> m_tagKeys;
  bool m_openIDConnectProviderArnHasBeenSet = false;
  bool m_tagKeysHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
