/**
 * Licensed to the University Corporation for Advanced Internet
 * Development, Inc. (UCAID) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for
 * additional information regarding copyright ownership.
 *
 * UCAID licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the
 * License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */

#include "internal.h"
#include <saml/saml2/core/Protocols.h>
#include <saml/util/SAMLConstants.h>
#include <xmltooling/signature/Signature.h>

using namespace opensaml::saml2p;
using namespace opensaml::saml2;


class Response20Test : public CxxTest::TestSuite, public SAMLObjectBaseTestCase {
    XMLCh* expectedID; 
    XMLCh* expectedInResponseTo; 
    XMLCh* expectedVersion; 
    XMLCh* expectedConsent; 
    XMLCh* expectedDestination; 
    scoped_ptr<XMLDateTime> expectedIssueInstant;

    // Assertion marshaller autogenerates ID, Version and IssueInstant if they are nullptr,
    // so have to agree on something to put in the control XML
    XMLCh* assertionID1, * assertionID2, * assertionID3;

public:
    void setUp() {
        expectedID = XMLString::transcode("def456"); 
        expectedInResponseTo = XMLString::transcode("abc123"); 
        expectedVersion = XMLString::transcode("2.0"); 
        expectedConsent = XMLString::transcode("urn:string:consent"); 
        expectedDestination = XMLString::transcode("http://sp.example.org/endpoint"); 
        expectedIssueInstant.reset(new XMLDateTime(XMLString::transcode("2006-02-21T16:40:00.000Z")));
        expectedIssueInstant->parseDateTime();

        assertionID1 = XMLString::transcode("test1"); 
        assertionID2= XMLString::transcode("test2"); 
        assertionID3 = XMLString::transcode("test3"); 

        singleElementFile = data_path + "saml2/core/impl/Response.xml";
        singleElementOptionalAttributesFile = data_path + "saml2/core/impl/ResponseOptionalAttributes.xml";
        childElementsFile  = data_path + "saml2/core/impl/ResponseChildElements.xml";    
        SAMLObjectBaseTestCase::setUp();
    }
    
    void tearDown() {
        XMLString::release(&expectedID);
        XMLString::release(&expectedInResponseTo);
        XMLString::release(&expectedVersion);
        XMLString::release(&expectedConsent);
        XMLString::release(&expectedDestination);
        XMLString::release(&assertionID1);
        XMLString::release(&assertionID2);
        XMLString::release(&assertionID3);
        expectedIssueInstant.reset();
        SAMLObjectBaseTestCase::tearDown();
    }

    void testSingleElementUnmarshall() {
        scoped_ptr<XMLObject> xo(unmarshallElement(singleElementFile));
        Response* response = dynamic_cast<Response*>(xo.get());
        TS_ASSERT(response!=nullptr);

        assertEquals("ID attribute", expectedID, response->getID());
        assertEquals("Version attribute", expectedVersion, response->getVersion());
        TSM_ASSERT_EQUALS("IssueInstant attribute", expectedIssueInstant->getEpoch(), response->getIssueInstant()->getEpoch());

        TS_ASSERT(response->getIssuer()==nullptr);
        TS_ASSERT(response->getSignature()==nullptr);
        TS_ASSERT(response->getExtensions()==nullptr);
        TS_ASSERT(response->getStatus()==nullptr);
        TSM_ASSERT_EQUALS("# of Assertion child elements", 0, response->getAssertions().size());
        TSM_ASSERT_EQUALS("# of EncryptedAssertion child elements", 0, response->getEncryptedAssertions().size());
    }

    void testSingleElementOptionalAttributesUnmarshall() {
        scoped_ptr<XMLObject> xo(unmarshallElement(singleElementOptionalAttributesFile));
        Response* response = dynamic_cast<Response*>(xo.get());
        TS_ASSERT(response!=nullptr);

        assertEquals("Consent attribute", expectedConsent, response->getConsent());
        assertEquals("Destination attribute", expectedDestination, response->getDestination());
        assertEquals("InResponseTo attribute", expectedInResponseTo, response->getInResponseTo());

        TS_ASSERT(response->getIssuer()==nullptr);
        TS_ASSERT(response->getSignature()==nullptr);
        TS_ASSERT(response->getExtensions()==nullptr);
        TS_ASSERT(response->getStatus()==nullptr);
        TSM_ASSERT_EQUALS("# of Assertion child elements", 0, response->getAssertions().size());
        TSM_ASSERT_EQUALS("# of EncryptedAssertion child elements", 0, response->getEncryptedAssertions().size());
    }

    void testChildElementsUnmarshall() {
        scoped_ptr<XMLObject> xo(unmarshallElement(childElementsFile));
        Response* response= dynamic_cast<Response*>(xo.get());
        TS_ASSERT(response!=nullptr);

        TS_ASSERT(response->getIssuer()!=nullptr);
        TS_ASSERT(response->getSignature()!=nullptr);
        TS_ASSERT(response->getExtensions()!=nullptr);
        TS_ASSERT(response->getStatus()!=nullptr);
        TSM_ASSERT_EQUALS("# of Assertion child elements", 3, response->getAssertions().size());
        TSM_ASSERT_EQUALS("# of EncryptedAssertion child elements", 1, response->getEncryptedAssertions().size());
    }

    void testSingleElementMarshall() {
        Response* response = ResponseBuilder::buildResponse();
        TS_ASSERT(response!=nullptr);

        response->setID(expectedID);
        response->setIssueInstant(expectedIssueInstant.get());
        //response->setVersion(expectedVersion);
        assertEquals(expectedDOM, response);
    }

    void testSingleElementOptionalAttributesMarshall() {
        Response* response = ResponseBuilder::buildResponse();
        TS_ASSERT(response!=nullptr);

        response->setID(expectedID);
        response->setInResponseTo(expectedInResponseTo);
        response->setIssueInstant(expectedIssueInstant.get());
        //response->setVersion(expectedVersion);
        response->setConsent(expectedConsent);
        response->setDestination(expectedDestination);
        response->setInResponseTo(expectedInResponseTo);
        assertEquals(expectedOptionalAttributesDOM, response);
    }

    void testChildElementsMarshall() {
        Response* response = ResponseBuilder::buildResponse();
        TS_ASSERT(response!=nullptr);

        response->setID(expectedID);
        response->setIssueInstant(expectedIssueInstant.get());
        // Do this just so don't have to redeclare the saml namespace prefix on every child element in the control XML file
        Namespace ns(samlconstants::SAML20_NS, samlconstants::SAML20_PREFIX);
        response->addNamespace(ns);
        response->setIssuer(IssuerBuilder::buildIssuer());
        // If the form of the default, basic, empty signature that is emittted changes wrt whitespace, etc,
        // this will probably break the test.  In that case need to fix the control XML.
        response->setSignature(xmlsignature::SignatureBuilder::buildSignature());
        response->setExtensions(ExtensionsBuilder::buildExtensions());
        response->setStatus(StatusBuilder::buildStatus());

        Assertion* assertion=nullptr;

        assertion = AssertionBuilder::buildAssertion();
        assertion->setIssueInstant(expectedIssueInstant.get());
        assertion->setID(assertionID1);
        response->getAssertions().push_back(assertion);

        assertion = AssertionBuilder::buildAssertion();
        assertion->setIssueInstant(expectedIssueInstant.get());
        assertion->setID(assertionID2);
        response->getAssertions().push_back(assertion);

        response->getEncryptedAssertions().push_back((EncryptedAssertionBuilder::buildEncryptedAssertion()));

        assertion = AssertionBuilder::buildAssertion();
        assertion->setIssueInstant(expectedIssueInstant.get());
        assertion->setID(assertionID3);
        response->getAssertions().push_back(assertion);


        assertEquals(expectedChildElementsDOM, response);
    }

};
