############################################################################
##      Copyright (C) 2005 Subredu Manuel  <diablo@iasi.roedu.net>.        #
##                                                                         #
## This program is free software; you can redistribute it and/or modify    #
## it under the terms of the GNU General Public License v2 as published by #
## the Free Software Foundation.                                           #
##                                                                         #
## This program is distributed in the hope that it will be useful,         #
## but WITHOUT ANY WARRANTY; without even the implied warranty of          #
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           #
## GNU General Public License for more details.                            #
##                                                                         #
## You should have received a copy of the GNU General Public License       #
## along with this program; if not, write to the Free Software             #
## Foundation, Inc., 59 Temple Place - Suite 330, Boston,                  #
## MA 02111-1307,USA.                                                      #
############################################################################

package RoPkg::Simba::Excludes;

use strict;
use warnings;

use Scalar::Util qw(blessed);

use RoPkg::Exceptions;
use RoPkg::DBCollection;

use vars qw($VERSION);
$VERSION='0.2.2';

use base qw(RoPkg::DBCollection);

sub new {
  my ($class, %opt) = @_;
  my $self;

  $self = $class->SUPER::new(%opt);
  $self->{table} = 'Excludes';

  return $self;
}

sub Count {
  my ($self, $fields) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw(
      error    => 'Called outside class instance',
      pkg_name => 'RoPkg::Simba::Excludes',
    );
  }

  return $self->_count($fields);
}

sub Get {
  my ($self, $fields, $orderby) = @_;
  my @elist;

  if (!blessed($self)) {
    OutsideClass->throw(
      error    => 'Called outside class instance',
      pkg_name => 'RoPkg::Simba::Excludes'
    );
  }
  
  @elist = $self->_get('RoPkg::Simba::Exclude', $fields, $orderby);
  foreach(@elist) {
    $_->Load();
  }

  return @elist;
}

1;


__END__

=head1 NAME

RoPkg::Simba::Excludes

=head1 VERSION

0.2.2

=head1 DESCRIPTION

RoPkg::Simba::Excludes is a class used to count and get
the excludes from the database. 

=head1 SYNOPSIS

 !#/usr/bin/perl
 
 use RoPkg::DB;
 use RoPkg::Simba::Excludes;
 
 sub main {
   my $dbp = new RoPkg::DB();
   $dbp->Add('dbi:mysql:database=mysql;host=localhost',
             'root',
             '',
             'local');

   my $m = new RoPkg::Simba::Excludes(dbo => $dbp, dbo_method => 'db_local');

   print $m->Count,' excludes found in database',$/;
 }
 
 main();

=head1 SUBROUTINES/METHODS

=head2 new()

The class constructor. At this moment, it just calls
RoPkg::DBObject->new() . Please read the RoPkg::DBObject
manual page for more information about the new() parameters.

=head2 Count($fields)

Returns the number of excludes from database. The B<$fields> parameter
contains a list of WHERE clauses. For more details of B<$fields>
see L<SQL::Abstract>.

=head2 Get($fields, $orderby)

Returns a array of I<RoPkg::Simba::exclude list> objects. The excludes are 
read from the database.  The B<$fields> parameter
contains a list of WHERE clauses. For more details of B<$fields>
see L<SQL::Abstract>. B<$orderby> is a reference to a array holding
the fields specification used for sorting the data.

=head1 DIAGNOSTICS

Unpack the source, and use 'make test' command

=head1 CONFIGURATION AND ENVIRONMENT

This module does not use any configuration files or
environment variables.

=head1 DEPENDENCIES

RoPkg::DBCollection and RoPkg::Exceptions

=head1 INCOMPATIBILITIES

None known to the author

=head1 BUGS AND LIMITATIONS

None known to the author

=head1 PERL CRITIC

This module is perl critic level 2 compliant

=head1 SEE ALSO

L<RoPkg::Simba> L<RoPkg::Simba::Exclude> L<SQL::Abstract>

=head1 AUTHOR

Subredu Manuel <diablo@iasi.roedu.net>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2005 Subredu Manuel.  All Rights Reserved.
This module is free software; you can redistribute it 
and/or modify it under the same terms as Perl itself.
The LICENSE file contains the full text of the license.

=cut
