import QtQuick
import QtQuick.Controls
import QtQuick.Layouts
import QtQuick.Effects

import org.deepin.dtk as D
import org.deepin.dtk.style as DS
import org.deepin.dcc
import org.deepin.dcc.passkey

D.DialogWindow {
    id: dialog
    icon: "preferences-system"
    modality: Qt.WindowModal
    width: 360

    Item {
        implicitWidth: parent.width
        implicitHeight: 570 - DS.Style.dialogWindow.titleBarHeight - DS.Style.dialogWindow.contentHMargin

        ColumnLayout {
            anchors.fill: parent

            D.Label {
                Layout.fillWidth: true
                horizontalAlignment: Text.AlignHCenter
                wrapMode: Text.WordWrap
                font: D.DTK.fontManager.t5
                text: dccData.model.existPin ? qsTr("Change PIN") : qsTr("Set PIN")
            }

            StackLayout {
                id: stackLayout
                Layout.fillWidth: true
                Layout.fillHeight: true

                currentIndex: {
                    switch (dccData.model.setPinDialogStyle) {
                        case Common.SetPinStyle:
                        case Common.ChangePinStyle:
                            return 0
                        case Common.SetFailedStyle:
                            return 1
                        default:
                            return 1
                    }
                }

                ColumnLayout {

                    Image {
                        Layout.alignment: Qt.AlignCenter
                        width: 180
                        height: 180
                        source: "qrc:/icons/deepin/builtin/texts/set_password_180px.svg"
                    }

                    D.Label {
                        Layout.fillWidth: true
                        wrapMode: Text.WordWrap
                        font: D.DTK.fontManager.t6
                        text: qsTr("Old PIN")
                        visible: dccData.model.existPin
                    }

                    D.PasswordEdit {
                        id: oldPinEdit
                        Layout.fillWidth: true
                        visible: dccData.model.existPin
                        alertDuration: 3000
                        onEditingFinished: {
                            oldPinEdit.showAlert = false
                            oldPinEdit.alertText = ""
                            if (text.length === 0) {
                                oldPinEdit.showAlert = true
                                oldPinEdit.alertText = qsTr("PIN cannot be empty")
                            } else if (text.length < 4) {
                                oldPinEdit.showAlert = true
                                oldPinEdit.alertText = qsTr("PIN length must not be less than 4 characters")
                            } else if (text.length > 63) {
                                oldPinEdit.showAlert = true
                                oldPinEdit.alertText = qsTr("PIN length cannot be more than 63 characters")
                            }
                        }
                    }

                    D.Label {
                        Layout.fillWidth: true
                        wrapMode: Text.WordWrap
                        font: D.DTK.fontManager.t6
                        text: qsTr("New PIN")
                    }

                    D.PasswordEdit {
                        id: newPinEdit
                        Layout.fillWidth: true
                        alertDuration: 3000

                        onEditingFinished: {
                            newPinEdit.showAlert = false
                            newPinEdit.alertText = ""
                            if (text.length === 0) {
                                newPinEdit.showAlert = true
                                newPinEdit.alertText = qsTr("PIN cannot be empty")
                            } else if (text.length < 4) {
                                newPinEdit.showAlert = true
                                newPinEdit.alertText = qsTr("PIN length must not be less than 4 characters")
                            } else if (text.length > 63) {
                                newPinEdit.showAlert = true
                                newPinEdit.alertText = qsTr("PIN length cannot be more than 63 characters")
                            }
                        }
                    }

                    D.Label {
                        Layout.fillWidth: true
                        wrapMode: Text.WordWrap
                        font: D.DTK.fontManager.t6
                        text: qsTr("Repeat PIN")
                    }

                    D.PasswordEdit {
                        id: repeatPinEdit
                        Layout.fillWidth: true
                        alertDuration: 3000
                        onTextChanged: {
                            if (newPinEdit.text !== repeatPinEdit.text) {
                                repeatPinEdit.showAlert = true
                                repeatPinEdit.alertText = qsTr("PIN inconsistency")
                            } else {
                                repeatPinEdit.showAlert = false
                                repeatPinEdit.alertText = ""
                            }
                        }
                    }

                    Item {
                        Layout.fillHeight: true
                    }

                    RowLayout {
                        Layout.fillWidth: true
                        Layout.bottomMargin: 10

                        D.Button {
                            text: qsTr("Cancel")
                            Layout.fillWidth: true
                            onClicked: {
                                oldPinEdit.text = ""
                                newPinEdit.text = ""
                                repeatPinEdit.text = ""
                                dialog.close()
                            }
                        }

                        D.Button {
                            text: dccData.model.existPin ? qsTr("Change") : qsTr("Set")
                            Layout.fillWidth: true
                            onClicked: {
                                if (dccData.model.existPin && oldPinEdit.text.length === 0) {
                                    return;
                                }

                                if (newPinEdit.text.length === 0 || repeatPinEdit.text.length === 0) {
                                    return;
                                }

                                if (newPinEdit.text !== repeatPinEdit.text) {
                                    return;
                                }

                                if (newPinEdit.text.length > 0) {
                                    dccData.worker.handleSetPasskeyPin(oldPinEdit.text, newPinEdit.text)
                                }
                            }
                        }
                    }

                    Connections {
                        target: dccData.worker
                        function onSetPinCompleted(success, pinErr, errMsg) {
                            if (success) {
                                dialog.close()
                            } else {
                                if (pinErr) {
                                    oldPinEdit.showAlert = true
                                    oldPinEdit.alertText = errMsg

                                    oldPinEdit.text = ""
                                    newPinEdit.text = ""
                                    repeatPinEdit.text = ""
                                }
                            }
                        }
                    }
                }

                ColumnLayout {

                    Item {
                        Layout.preferredHeight: 40
                    }

                    D.DciIcon {
                        name: "icon_fail"
                        sourceSize: Qt.size(96, 96)
                        Layout.alignment: Qt.AlignHCenter
                    }

                    Item {
                        Layout.preferredHeight: 20
                    }

                    D.Label {
                        Layout.fillWidth: true
                        horizontalAlignment: Text.AlignHCenter
                        wrapMode: Text.WordWrap
                        font: D.DTK.fontManager.t6
                        text: dccData.model.existPin ? qsTr("Change Failed") : qsTr("Setting Failure")
                    }

                    Item {
                        Layout.fillHeight: true
                    }

                    RowLayout {
                        Layout.fillWidth: true
                        Layout.bottomMargin: 10

                        D.Button {
                            text: qsTr("Cancel")
                            Layout.fillWidth: true
                            onClicked: {
                                dialog.close()
                            }
                        }
                    }
                }
            }
        }
    }
}
