var tld = {
  preserveFQDNs: false,

  // remove trailing dots from FQDNs
  normalize(d) { return d.endsWith(".") ? d.slice(0, -1) : d },

  isIp(d) { return this._ipRx.test(d); },

  getDomain(aDomain) {
    if (aDomain === "localhost" || this.isIp(aDomain)) return aDomain;

    domain = this.normalize(aDomain);

    let resultDomain = this.preserveFQDNs ? aDomain : domain;

    var pos = domain.search(this._tldEx);
    if(pos === -1 ) {
      pos = domain.search(this._tldRx);
      if (pos === -1) {
        // TLD not in the public suffix list, fall back to the "one-dot rule"
        // (for a.b.c.tld, assume the domain to be "c.tld")
        pos = domain.lastIndexOf(".");
        if (pos === -1) {
          // No dots at all? Likely a private domain in a LAN.
          return resultDomain;
        }
      }
      pos = domain.lastIndexOf(".", pos - 1) + 1;
    } else if(domain[pos] == ".") {
      ++pos;
    }
    return pos <= 0 ? resultDomain : resultDomain.substring(pos);
  },

  getPublicSuffix(aDomain) {
    if (this.isIp(aDomain)) return "";

    domain = this.normalize(aDomain);
    let resultDomain = this.preserveFQDNs ? aDomain : domain;

    var pos = domain.search(this._tldEx);
    if(pos < 0) {
      pos = domain.search(this._tldRx);
      if(pos >= 0 && domain[pos] == ".") pos++;
    } else {
      pos = domain.indexOf(".", pos + 1) + 1;
    }
    return pos < 0 ? "" : resultDomain.substring(pos);
  },

  _ipRx: /^(?:0\.|[1-9]\d{0,2}\.){3}(?:0|[1-9]\d{0,2})$|:.*:/i,

  _tldRx: /(?:\.|^)(?:s(?:h(?:i(?:(?:m(?:o(?:(?:go\.fukushim|fusa\.chib|ji\.okinaw|ichi\.nar)a|k(?:awa\.hokkaido|itayama\.nara)|n(?:oseki\.yamaguchi|ita\.gunma)|tsu(?:ke\.tochig|ma\.ibarak)i|da(?:te\.ibaraki|\.shizuoka)|suwa\.nagano)|a(?:m(?:aki\.hokkaido|oto\.osaka)|(?:(?:ne\.shima)?n|\.mi)e|bara\.nagasaki|da\.shizuoka)|izu\.(?:hokkaido|shizuoka))|n(?:(?:a(?:nomachi\.nagan|gawa\.toky)|onsen\.hyog)o|g(?:u\.(?:(?:wakayam|fukuok)a|hyogo)|o\.aomori)|j(?:o\.(?:yamagat|okayam|nar)a|uku\.tokyo)|to(?:ku\.hokkaido|mi\.miyazaki|\.gunma)|(?:yoshitomi\.fukuok|ichi\.hiroshim)a|shi(?:notsu\.hokkaido|ro\.aichi)|kamigoto\.nagasaki)|r(?:a(?:k(?:awa\.(?:fukushima|gifu)|o\.chiba)|(?:hama\.wakayam|taka\.yamagat)a|o(?:i\.hokkaido|ka\.saitama)|nuka\.hokkaido)|o(?:i(?:shi\.(?:miyagi|saga)|\.chiba)|sato\.ibaraki)|iuchi\.hokkaido)|b(?:u(?:kawa\.gunma|ya\.tokyo)|ata\.(?:niigata|miyagi)|e(?:cha|tsu)\.hokkaido)|(?:chi(?:kashuku\.miyag|nohe\.aomor)|iba\.miyazak|tara\.aich)i|o(?:(?:gama\.miya|ya\.tochi)gi|jiri\.nagano)|s(?:(?:hikui\.tokushim|ui\.chib)a|o\.hyogo)|zu(?:(?:oka\.shizu)?oka|kuishi\.iwate)|(?:jonawate\.osak|g)a|wa\.iwate)\.jp|k(?:(?:a(?:(?:ma\.miyag|tsu\.aich)i|(?:be|oi)\.hokkaido|\.ishikawa)|okuchuo\.ehime|i\.saitama)\.jp|sha)|ftedit\.io|a)|o(?:(?:(?:bara\.hiroshi|o\.okaya)m|nai\.(?:yamagat|fukuok))a\.jp|w(?:a\.(?:(?:fukushi|gun)ma|yamanashi)\.jp|\.aero|time)?|p(?:\.(?:h[tu]|pl|ro)|ping)?|uji|es)|a(?:(?:r(?:i\.hokkaido\.j)?|kotan\.hokkaido\.j)p|cknet\.nu|ngrila|w)|e(?:ll(?:\.museum)?|rbrooke\.museum)|unan\.yamaguchi\.jp|riram|\.cn)?|a(?:n(?:d(?:(?:(?:e(?:\.(?:m[oø]re-og-romsdal|vestfold)|fjord)|nes(?:sj[oø]en)?|[oø]y)\.n|cats\.i)o|vik(?:coromant)?|a\.hyogo\.jp|iego\.museum)|(?:(?:agochi\.tokushim|jo\.niigat|uki\.kagaw|go\.nar)a|n(?:ohe\.aomori|an\.hyogo))\.jp|t(?:a(?:(?:barbara|cruz|fe)\.museum|maria\.br)|oandre\.br)|o(?:\.tochigi\.jp|k\.pl|fi)|francisco\.museum)|k(?:(?:a(?:i(?:\.(?:(?:ibarak|fuku)i|osaka)|minato\.tottori)|(?:(?:\.hiroshi|do\.saita)m|ta\.yamagat)a|e\.(?:nagano|chiba)|hogi\.gifu|ki\.nagano|wa\.kochi)|egawa\.yamagata|yo\.kyoto)\.jp|u(?:ra(?:(?:\.(?:tochigi|chiba)|gawa\.ibaraki|i\.nara)\.jp)?|(?:ho)?\.nagano\.jp))|(?:(?:ga(?:(?:mihara\.kanagaw|e\.yamagat)a|(?:\.saga)?)|y(?:ama\.(?:saitam|osak)a|o\.hyogo)|do\.niigata|bae\.fukui)\.j)?p|i(?:(?:t(?:(?:ama\.sait)?ama|o\.miyazaki)|k(?:ai\.nagasaki|i\.oita)|gawa\.fukuoka|jo\.ehime)\.jp|ntlouis\.museum)|l(?:(?:(?:a(?:ngen|t)|tdal)\.n|ud\.b)o|vador(?:dali\.museum|\.br)|e(?:m\.museum|rno\.it)?|zburg\.museum|on)|s(?:(?:a(?:guri\.fukuoka|yama\.hyogo)|ebo\.nagasaki)\.jp|katchewan\.museum|sari\.it)?|m(?:(?:egawa\.fukushim|ukawa\.kanagaw)a\.jp|s(?:club|ung)|nanger\.no|pa\.br)|r(?:(?:ufutsu|oma)\.hokkaido\.jp|(?:d(?:egn|ini)a)?\.it|psborg\.no|l)|t(?:(?:(?:osho\.okay|te\.sait)a|sumasendai\.kagoshi)ma\.jp|x\.museum)|v(?:e(?:s-the-whales\.com)?|annahga\.museum|ona\.it)|\.(?:gov\.(?:au|pl)|(?:edu\.)?au|c(?:om|r)|it)|o(?:(?:bernardo|gonca)\.br|tome\.st)|-east-1\.elasticbeanstalk\.com|(?:u(?:herad|da)\.n|x)o|fe(?:ty(?:\.aero)?)?|arland)?|e(?:[sw]|r(?:v(?:e(?:(?:(?:(?:counterstri|qua)k|exchang)e|h(?:alflife|umour|ttp)|p(?:ics|2p)|sarcasm|irc)\.com|b(?:bs\.(?:com|net|org)|eer\.com|log\.net)|m(?:inecraft\.net|p3\.com)|ftp\.(?:com|net|org)|game\.(?:com|org))|ice(?:s(?:\.aero)?|\.gov\.uk))|a(?:nishi)?\.hiroshima\.jp)|(?:i(?:rou?\.niigata|hi\.nagasaki|ka\.kyoto|yo\.ehime)|ki(?:(?:gahara)?\.gifu|kawa\.niigata)|m(?:boku\.akita|ine\.miyagi))\.jp|l(?:ls(?:-(?:for-(?:less|u)\.com|it\.net)|yourhome\.org)|fip\.(?:info|biz|com|net|org)|(?:j(?:ord|e)|bu)?\.no|ect)|t(?:(?:o(?:uchi\.okayama|\.aichi)|agaya\.tokyo)\.jp|t(?:le(?:ment|rs)\.museum|su\.osaka\.jp))|c(?:ur(?:ity(?:tactics\.com)?|e)|\.ps)|\.(?:(?:gov|leg)\.br|eu\.org|net)|a(?:port\.museum|rch|t)|n(?:nan\.osaka\.jp|er)|v(?:astopol\.ua|en)|x(?:\.(?:hu|pl)|y)?|bastopol\.ua|jny\.pl|oul\.kr|ek)?|3(?:-(?:website(?:-(?:ap-(?:southeast-[12]|northeast-1)|us-(?:west-[12]|east-1)|(?:eu-we|sa-ea)st-1)|\.(?:ap-(?:northeast-2|south-1)|eu-(?:central-1|west-[23])|ca-central-1|us-east-2))|(?:(?:fips-us-gov-we|sa-ea)st|ca-central)-1|ap-(?:south(?:east-[12]|-1)|northeast-[12])|e(?:u-(?:west-[123]|central-1)|xternal-1)|us-(?:gov-west-1|west-[12]|east-2))\.amazonaws\.com|\.(?:(?:(?:dualstack\.(?:ap-(?:south(?:east-[12]|-1)|northeast-[12])|eu-(?:west-[123]|central-1)|(?:ca-central|sa-east)-1|us-east-[12])|eu-(?:central-1|west-[23])|us-east-2)\.|(?:ap-(?:northeast-2|south-1)\.)?)amazonaws\.com|c(?:n-north-1\.amazonaws\.com\.cn|a-central-1\.amazonaws\.com)))|t(?:a(?:t(?:e(?:(?:(?:ofdelaware)?\.museu|far)m|bank)|i(?:c(?:-access\.net|\.land)|on\.museum)|(?:helle)?\.no|oil)|r(?:(?:ostwo\.gov|achowice|gard)\.pl|nberg\.museum|hub)?|(?:(?:v(?:anger|ern)|nge)\.n|ge\.nodeart\.i)o|l(?:bans\.museum|owa-wola\.pl)|d(?:t\.museum|a)|ples)|o(?:r(?:(?:d(?:al)?|-elvdal|fjord)\.no|e(?:\.(?:bb|dk|nf|ro|st|ve))?|j\.farm|age)|ckhol(?:m\.museu)?m|kke\.no)|u(?:ff(?:-4-sale\.(?:org|us)|toread\.com)|d(?:(?:ent\.aer|i)o|y)|ttgart\.museum)|j(?:o(?:rdal(?:shalsen)?\.no|hn\.museum)|ørdal(?:shalsen)?\.no)|e(?:i(?:(?:nkjer|gen)\.no|ermark\.museum)|am\.museum)|r(?:(?:anda?|yn)\.no|eam)|petersburg\.museum|c(?:group)?|\.no|yle)?|o(?:(?:(?:(?:o\.kagoshi|ja\.okaya)m|degaura\.chib|eda\.fukuok)a|betsu\.hokkaido|wa\.ibaraki)\.jp|r(?:(?:-(?:(?:aur|o)dal|varanger|fron)|(?:tlan|fol)d|reisa|um)\.no|ocaba\.br)|u(?:nd(?:andvision\.museum|cast\.me)|th(?:carolina|west)\.museum)|n(?:dr(?:e-land\.no|io\.it)|g(?:dalen\.no)?|i\.nara\.jp|y)|l(?:u(?:nd\.no|tions)|ogne\.museum|a(?:\.no|r))|c(?:i(?:ety\.museum|al)|hi\.su|\.lk|cer)|s(?:(?:nowiec)?\.pl|a\.chiba\.jp)|k(?:a\.saitama\.jp|ndal\.no)|m(?:a\.fukushima\.jp|na\.no)|ft(?:ware(?:\.aero)?|bank)|\.(?:gov\.pl|it)|gn(?:dal|e)\.no|pot\.pl|hu|y)?|u(?:(?:k(?:agawa\.fukushima|umo\.kochi)|gi(?:nami\.tokyo|to\.saitama)|s(?:ono\.shizuoka|aki\.kochi))\.jp|m(?:(?:i(?:da\.tokyo|ta\.iwate)|oto\.(?:kumamot|hyog)o)\.jp|y\.ua)|z(?:u(?:k(?:a\.mie\.jp|i)|\.ishikawa\.jp)|aka\.nagano\.jp)|r(?:ge(?:onshall\.museum|ry)|rey\.museum|nadal\.no|f)|i(?:(?:fu\.ibaraki|ta\.osaka)\.jp|sse\.museum)|n(?:agawa\.hokkaido\.jp|(?:ndal|d)\.no)|e(?:\.fukuoka\.jp|dtirol\.it)|wa(?:\.nagano\.jp|lki\.pl)|l(?:(?:dal|a)\.no|i\.hu)|pp(?:l(?:ies|y)|ort)|̈dtirol\.it|cks)?|c(?:[ab]|h(?:o(?:ko(?:laden\.museum|keks\.net)|ol(?:\.(?:museum|n[az]|za))?|enbrunn\.museum|larships)|\.(?:[qs]a|i[dr]|l[ky]|ae|jo|ng|zm)|w(?:eiz\.museum|arz)|lesisches\.museum|aeffler|midt|ule)|i(?:en(?:ce(?:(?:(?:and(?:indust|histo)|histo)ry|(?:snaturelle)?s|-fiction|centers?)?\.museum)?|tist\.aero)|\.eg)|r(?:app(?:er-site\.net|ing\.cc)|ysec\.com)|\.(?:(?:gov|leg)\.br|k[er]|u[gs]|cn|tz)|o(?:t(?:land\.museum)?|r)|johnson)?|k(?:(?:a(?:n(?:land|it)|un)|j(?:erv[oø]y|[aå]k)|edsmo(?:korset)?|ånland|ánit)\.no|i(?:(?:e(?:rv[aá]|n)|ptvet)\.no|\.(?:museum|no)|n)?|o(?:(?:\.gov|czow)\.pl|le\.museum|dje\.no)|y(?:diving\.aero|pe)?|\.(?:eu\.org|ca)|lep\.pl)?|i(?:(?:benik\.museu|mple-url\.co)m|l(?:k(?:\.museum)?|jan\.no)|n(?:a(?:app\.com)?|gles)|r(?:acusa\.it|dal\.no)|te(?:s\.static\.land)?|c(?:il(?:ia|y))?\.it|e(?:llak\.no|na\.it)|\.(?:eu\.org|it)|gdal\.no)?|p(?:ace(?:(?:-to-rent\.co|\.museu)m|kit\.io)?|\.(?:(?:gov|leg)\.br|it)|y(?:deberg\.no|\.museum)|dns\.(?:org|de|eu)|o(?:rt(?:\.hu)?|t)|jelkavik\.no|readbetting|b\.[rs]u|iegel)|y(?:no(?:logy(?:-d(?:iskstation|s)\.d|\.m)|-ds\.d)e|dney(?:\.museum)?|kkylven\.no|tes\.net|mantec|stems)?|(?:ø(?:r(?:-(?:(?:aur|o)dal|varanger|fron)|reisa|fold|um)|ndre-land|gne|mna)|ál[aá]t)\.no|w(?:i(?:(?:noujscie|ebodzin|dnica)\.pl|ftcover|ss)|e(?:etpepper\.org|den\.museum)|atch)|v(?:(?:e(?:lvik|io)|albard)\.no|izzera\.museum|n-repos\.de|\.it)?|l(?:a(?:ttum\.no|sk\.pl)|d\.(?:do|pa)|[gz]\.br|upsk\.pl|ing)?|n(?:(?:[aå](?:ase|sa)|illfjord|oasa)\.no|\.cn|cf)?|z(?:(?:cz(?:ecin|ytno)|kola)\.pl|ex\.hu)?|s(?:l\.origin\.cdn77-secure\.org|\.it)|quare(?:7\.(?:net|ch|de)|\.museum)|r(?:[lt]|\.(?:gov\.pl|it)|v\.br)?|m(?:[oø]la\.no|\.ua|art|ile)?|d(?:\.(?:cn|us)|n\.gov\.pl)?|b(?:[is]|\.ua)?|\.(?:bg|se)|f(?:\.no|r)|j(?:c\.br)?|x(?:\.cn)?|g)|k(?:a(?:(?:m(?:i(?:(?:t(?:onda\.wakayam|sue\.oit)|i(?:zumi\.sait|chi\.toy)am|mine\.sag|oka\.akit)a|k(?:awa\.(?:h(?:okkaid|yog)o|saitama)|(?:itayama\.nar|oani\.akit)a)|s(?:u(?:nagawa\.hokkaido|\.ibaraki)|hihoro\.hokkaido|ato\.saitama)|(?:amakusa\.kumamot|furano\.hokkaid|gori\.hyog)o|no(?:yama\.yamagata|kawa\.tochigi)|\.(?:miyag|koch)i|jima\.ehime)|o(?:\.(?:niigata|kyoto)|enai\.hokkaido|gawa\.chiba)|a(?:(?:kura\.kanagaw|gaya\.chib)a|ishi\.iwate)|e(?:oka\.kyoto|yama\.mie))|wa(?:(?:(?:(?:zu\.shiz|ra\.fuk)uo|chinagano\.osa)k|(?:jima\.saita|ba\.gun)m)a|n(?:(?:abe\.kagoshim|ehon\.shizuok)a|ishi\.(?:(?:yamagat|nar)a|hyogo))|(?:tana\.nagasak|hara\.tottor|saki\.miyag)i|g(?:oe\.(?:saitama|mie)|uchi\.saitama)|k(?:ami\.na(?:gano|ra)|ita\.ishikawa)|m(?:inami\.miyazaki|ata\.fukushima)|i\.(?:iwate|nara)|ue\.gifu)|n(?:(?:a(?:(?:zawa\.ishik|g)aw|n\.osak)|o(?:ya\.kagoshim|nji\.kagaw)|na(?:mi\.shizuok|\.gunm)|maki\.nar|zaki\.sag|ra\.gunm)a|e(?:yama\.(?:fukushim|yamagat)a|gasaki\.iwate)|i(?:e\.aichi|\.gifu)|uma\.tochigi)|g(?:a(?:mi(?:(?:ishi\.fukushi|no\.okaya)ma|\.kochi)|(?:\.ishika)?wa)|(?:oshima\.kag)?oshima)|i(?:(?:(?:nan\.(?:tokushi|wakaya)|ta\.hiroshi)m|sei\.kanagaw|zuka\.osak)a|\.yamanashi)|k(?:amigahara\.gifu|egawa\.shizuoka|inoki\.shimane|ogawa\.hyogo|uda\.miyagi)|d(?:o(?:gawa\.miyazaki|ma\.osaka)|ena\.okinawa)|ho(?:ku\.(?:ishikaw|yamagat)|\.fukuok)a|yabe\.hokkaido)\.jp|s(?:(?:u(?:ga(?:\.(?:fukuoka|hyogo)|i\.aichi)|(?:kabe\.saitam|ya\.fukuok)a|migaura\.ibaraki)|hi(?:(?:wa(?:zaki\.niigat|ra\.osak|\.chib)|(?:har|b)a\.nar)a|ma\.(?:ibaraki|saga))|a(?:ma(?:\.ibaraki|tsu\.gifu)|oka\.okayama|hara\.gifu|i\.hyogo))\.jp|zuby\.pl)|r(?:a(?:s(?:uyama\.tochigi\.jp|jo(?:hka|k)\.no)|t(?:su\.saga\.jp|e\.museum)|(?:ganda|col)\.su)|i(?:(?:wa\.niigata|ya\.aichi)\.jp|katur\.museum)|u(?:izawa\.nagano|mai\.iwate)\.jp|(?:m[oø]|lso)y\.no|(?:pacz|tuzy)\.pl|elia\.su)|t(?:(?:su(?:(?:ragi\.(?:wakayam|nar)|ura\.chib)a|shika\.tokyo|yama\.fukui)|a(?:shina\.gunm|gami\.akit|no\.osak)a)\.jp|o(?:ri\.chiba\.jp|wice\.pl))|z(?:(?:o\.saitam|uno\.akit)a\.jp|imierz-dolny\.pl)|l(?:(?:mykia\.[rs]|uga\.s)u|isz\.pl)|u(?:tokeino\.no|fen)|fjord\.no)|o(?:(?:u(?:(?:(?:yama\.kagoshi|nosu\.saita)m|hoku\.sag)a|zushima\.tokyo)|(?:(?:ori\.fukushi|ya\.wakaya)m|r(?:iyama\.fukushim|yo\.nar))a|t(?:o(?:\.(?:shiga|tokyo)|hira\.kagawa|ura\.tottori)|a\.aichi)|g(?:a(?:\.(?:fukuoka|ibaraki)|nei\.tokyo)|e\.tottori)|k(?:(?:onoe\.oit|a\.shig)a|ubunji\.tokyo)|(?:(?:chi\.ko)?c|fu\.yamanas)hi|daira\.tokyo)\.jp|m(?:a(?:(?:(?:gane\.nagan|e\.toky)o|ki\.aichi)\.jp|tsu(?:(?:shima\.tokushim|\.ishikaw)a\.jp)?)|o(?:ro\.nagano|no\.mie)\.jp|mun(?:alforbund\.se|e\.no)|(?:forb|vux)\.se)|s(?:h(?:(?:i(?:mizu\.hokkaido|gaya\.saitama)|u\.yamanashi)\.jp|er)|(?:(?:a(?:i\.shizuok|ka\.akit)|ei\.shig)a|uge\.yamanashi)\.jp)|n(?:(?:s(?:ulat\.gov|kowola)|in)\.pl|an\.(?:aichi|shiga)\.jp|gs(?:vinger|berg)\.no|yvelo\.hu)|z(?:a(?:(?:gawa)?\.wakayam|ki\.chib)a\.jp|ow\.com)|b(?:ayashi\.miyazaki\.jp|ierzyce\.pl)|e(?:benhavn\.museum|ln(?:\.museum)?)|lobrzeg\.pl|pervik\.no)|i(?:t(?:a(?:(?:(?:shiobara\.fukushi|yama\.wakaya)m|(?:nakagusuku|daito)\.okinaw)a|ga(?:wa\.(?:miyazak|koch)i|ta\.(?:gifu|saga))|ka(?:ta\.(?:fukushima|miyazaki)|mi\.iwate)|h(?:iroshima\.hokkaido|ata\.saga)|m(?:oto\.saitama|i\.hokkaido)|a(?:iki\.nagano|kita\.akita)|\.(?:(?:kyot|toky)o|osaka)|ura\.miyazaki)\.jp|chen)|(?:y(?:o(?:s(?:(?:ato\.hokkaid|e\.toky)o|u\.aichi)|kawa\.kanagawa)|ama\.saga)|s(?:o(?:(?:fukushima)?\.nagano|saki\.mie)|(?:hiwada\.osak|arazu\.chib)a)|k(?:u(?:gawa\.shizuoka|chi\.kumamoto)|onai\.hokkaido)|ho(?:ku\.ehim|\.mi)e|bichuo\.okayama|jo\.miyazaki|zu\.kyoto)\.jp|n(?:(?:(?:okawa\.wakaya|ko\.kagoshi)m|\.okinaw)a\.jp|d(?:er|le))|m(?:(?:i(?:no\.wakayam|tsu\.chib)a|obetsu\.hokkaido)\.jp)?|r(?:(?:yu\.gunma|a\.aichi)\.jp|ovograd\.ua|kenes\.no)|w(?:a\.mie\.jp|i(?:\.nz)?)|cks-ass\.(?:net|org)|ds\.(?:museum|us)|(?:ev\.u)?a)?|u(?:(?:(?:m(?:a(?:(?:gaya\.saitam|tori\.osak)a|no\.(?:hiroshima|mie)|(?:moto\.kuma)?moto|kogen\.ehime)|e(?:jima\.okinaw|nan\.okayam)a|iyama\.kyoto)|d(?:amatsu\.yamaguchi|oyama\.wakayama)|j(?:u(?:kuri\.chib|\.oit)a|i\.iwate)|(?:i\.hiroshi|ki\.saita)ma|(?:zumaki\.iwat|wana\.mi)e|chinotsu\.nagasaki)\.j|okgrou)p|r(?:(?:o(?:(?:gi\.fukuok|be\.toyam|taki\.nar)a|is(?:hi\.aomor|o\.tochig)i|matsunai\.hokkaido)|(?:a(?:shiki\.okayam|te\.fukuok)|e\.hiroshim|ume\.fukuok)a|iyama\.hokkaido)\.jp|gan\.su)|n(?:(?:i(?:(?:gami\.okinaw|mi\.fukushim|saki\.oit)a|t(?:omi\.miyazaki|achi\.tokyo))|neppu\.hokkaido|ohe\.iwate)\.jp|st(?:unddesign|sammlung)?\.museum)|s(?:(?:hi(?:m(?:oto\.wakayama|a\.miyazaki)|ro\.hokkaido)|(?:atsu\.(?:gunm|shig)|u\.oit)a)\.jp|tanai\.[rs]u)|t(?:chan\.hokkaido\.jp|no\.pl))|12\.(?:(?:m[adeinost]|n[cehjmvy]|a[klrsz]|c[aot]|o[hkr]|w[aiy]|d[ce]|g[au]|k[sy]|p[ar]|fl|la|ri|sc|ut)\.us|i(?:l(?:\.us)?|[adn]\.us)|v(?:i(?:\.us)?|[at]\.us)|t(?:[nx]\.us|r)|ec)|r(?:(?:(?:istians[au]n|ødshera)d|o(?:kstadelva|dsherad)|åanghke)\.no|a(?:(?:ger[oø]|anghke)\.no|snodar\.su|kow\.pl)|\.(?:eu\.org|com|it|ua)|ym\.ua|e?d)?|y(?:o(?:t(?:a(?:n(?:abe|go)|mba)\.kyoto\.jp|o(?:\.jp)?)|(?:wa\.(?:hokkaido|akita)|nan\.chiba)\.jp)|uragi\.saga\.jp|iv\.ua|\.us)?|e(?:(?:mbuchi\.hokkaido|isen\.fukuoka)\.jp|rry(?:propertie|logistic|hotel)s|p(?:no\.pl|\.tr)|ymachine\.de|trzyn\.pl)?|v(?:(?:i(?:n(?:nherad|esdal)|t(?:s[oø]y|eseid))|a(?:(?:fjor|lsun)d|nangen|m)|æ(?:nangen|fjord))\.no|\.ua)|h(?:(?:meln(?:itskiy|ytskyi)|erson)?\.ua|a(?:rk[io]v\.ua|kassia\.su))|n(?:ightpoint\.systems|owsitall\.info|x-server\.net)?|l(?:(?:[aæ]bu|epp)\.no|odzko\.pl)|(?:árášjohka|åfjord)\.no|m(?:psp\.gov\.pl|\.ua)?|p(?:psp\.gov\.pl|mg|n)?|w(?:(?:ps)?p\.gov\.pl)?|\.(?:bg|se)|g(?:\.kr)?|s\.u[as]|ddi|fh|z)|m(?:[hqv]|i(?:n(?:(?:a(?:m(?:i(?:(?:(?:yamashiro\.ky|oguni\.kumam)ot|m(?:inowa|aki)\.nagan|furano\.hokkaid)o|(?:tane\.kagoshim|uonuma\.niigat|daito\.okinaw|boso\.chib)a|(?:-alps\.yamanash|sanriku\.miyag|echizen\.fuku)i|a(?:(?:iki\.nagan|waji\.hyog)o|shigara\.kanagawa)|\.(?:(?:tokushim|fukuok)a|kyoto)|i(?:zu\.shizuoka|se\.mie))|ata\.kumamoto)|(?:kami\.gun|no\.saita)ma|to\.(?:osaka|tokyo))|o(?:(?:kamo)?\.gifu|bu\.yamanashi|wa\.nagano|h\.osaka))\.jp|i(?:(?:server\.co|ng\.museu)m)?|e(?:rs\.museum|\.nu)|nesota\.museum|com\.tn|t)|(?:y(?:a(?:k(?:o(?:\.(?:fukuoka|iwate)|nojo\.miyazaki)|e\.nara)|z(?:(?:aki\.miyaz)?aki|u\.kyoto)|(?:shiro\.saitam|waka\.fukuok)a|ma\.(?:fukuoka|mie)|da\.nagano|gi)|o(?:shi\.(?:(?:(?:hiro|toku)shi|saita)ma|aichi)|ta\.nagano))|h(?:a(?:ma\.(?:(?:wakayam|chib)a|(?:aich|fuku)i|mie)|r(?:a\.(?:hiroshima|kochi)|u\.fukushima))|o\.ibaraki)|zu(?:maki\.fukuoka|sawa\.iwate|nami\.gifu|ho\.tokyo)|k(?:a(?:sa\.hokkaido|wa\.yamagata)|i\.hyogo)|ma(?:ta\.miyazaki|\.tokushima)|fune\.kumamoto|ura\.kanagawa|bu\.tochigi)\.jp|s(?:(?:a(?:to\.(?:s(?:aitama|himane)|(?:wakayam|akit)a|miyagi)|(?:sa\.tott|wa\.aom)ori|ki\.o(?:kayam|sak)a)|hima\.(?:fukushim|shizuok)a|ugi\.mie)\.jp|s(?:oula|ile)\.museum|confused\.org)|l(?:\.(?:t[jmorwz]|a[celrz]|b[aory]|k[gmrz]|m[gvyz]|n[gioz]|p[ehly]|c[lno]|g[eht]|i[dnq]|s[hty]|z[amw]|[dj]o|e[cg]|r[uw]|v[ce]|hn|lv|qa|uy)|(?:itary|l)\.museum|ano?\.it)?|t(?:(?:o(?:(?:u\.yamaguch|\.ibarak)i|yo\.kagawa)|a(?:k(?:a\.tokyo|e\.gifu)|ne\.akita))\.jp|su(?:(?:ke\.niigat|e\.nar)a\.jp|bishi))?|d(?:(?:tre-gauldal|sund)\.no|ori\.(?:chib|gunm)a\.jp|atlantic\.museum)|c(?:ro(?:light\.aero|soft)|higan\.museum)|a(?:s(?:a\.nagano\.jp|ta\.pl)|mi)|e(?:l(?:ec|no)\.pl|\.jp)|\.(?:it|th|us))|a(?:t(?:su(?:(?:b(?:ushi\.saitam|ara\.osak)|d(?:a\.kanagaw|o\.chib)|zaki\.shizuok)a|s(?:hi(?:ge\.tokushima|ma\.miyagi)|aka\.mie)|m(?:oto\.(?:kagoshima|nagano)|ae\.hokkaido)|(?:(?:yama|no)\.ehim|e\.shiman)e|ura\.nagasaki|kawa\.nagano)\.jp|t(?:a-varjjat\.no|el)|era\.it|\.br)|r(?:i(?:n(?:ga\.br|e\.ru)|tim[eo]\.museum)|u(?:game\.kagawa|mori\.miyagi)\.jp|(?:yl(?:hurst|and)|burg)\.museum|ke(?:t(?:ing|s)?|r\.no)|(?:(?:che)?\.i|riot)t|nardal\.no|shalls)|s(?:(?:hik(?:(?:e\.hokkaid|i\.kumamot)o|o\.tochigi)|(?:uda\.shiman|aki\.ehim)e)\.jp|(?:fjorden|oy)\.no|sa-?carrara\.it|erati)|n(?:(?:iwa\.okayam|no\.kagaw)a\.jp|(?:chester|sions?|x)\.museum|a(?:gement|us\.br)|g(?:yshlak\.su|o)|tova\.it|dal\.no)?|(?:k(?:(?:urazaki\.kagoshim|inohara\.shizuok)a\.j|eu)|(?:murogawa\.yamagat|ebashi\.gunm)a\.j)p|l(?:(?:atvuopmi|selv|vik)\.no|(?:opolska|bork)\.pl|lorca\.museum)|i(?:(?:bara\.shiga|zuru\.kyoto)\.jp|ntenance\.aero|l\.pl|son|f)|c(?:e(?:rata\.it|io\.br)|hida\.tokyo\.jp|apa\.br|ys)|d(?:rid(?:\.museum)?|\.museum)|p(?:\.fastly(?:lb)?\.net)?|\.(?:(?:gov|leg)\.br|us)|yfirst\.(?:info|org)|z(?:owsze|ury)\.pl|gazine\.aero|ori\.nz)?|o(?:[ei]|r(?:(?:i(?:y(?:a(?:\.ibaraki|ma\.shiga)|oshi\.akita)|(?:machi\.shizuo|guchi\.osa)ka|oka\.iwate)|o(?:tsuka\.miyazaki|yama\.saitama))\.jp|dovia\.[rs]u|ena\.br|tgage|mon)|n(?:(?:za(?:(?:-e-della)?-|e(?:della)?)?brian)?za\.it|(?:t(?:icello|real)|mouth)\.museum|ey(?:\.museum)?|ster|ash)|t(?:o(?:(?:bu\.okinawa|yama\.kochi|su\.gifu)\.jp|rcycle(?:\.museum|s))?|egi\.tochigi\.jp)|s(?:(?:(?:kene)?s|j[oø]en|vik)\.no|eushi\.hokkaido\.jp|cow(?:\.museum)?)|d(?:e(?:lling\.aero|rn\.museum|na\.it)|a(?:len\.no)?|um\.no|\.gi)|b(?:i(?:\.(?:n[ag]|t[tz]|gp|ke)|l[ey])?|ara\.chiba\.jp)|(?:chizuki\.nagano|ka\.tochigi)\.jp|m(?:betsu\.hokkaido\.jp|a\.museum)?|v(?:i(?:miento\.bo|star|e))?|(?:[aå]reke|-i-rana)\.no|l(?:(?:ise)?\.it|de\.no)|\.(?:cn|it|us)|zilla-iot\.org|onscale\.net|par)?|y(?:(?:a(?:ctivedirectory|sustor)|qnapcloud|ravendb|iphost|vnc)\.com|d(?:atto\.(?:com|net)|issent\.net|dns\.rocks|robo\.com|s\.me)|-(?:(?:(?:route|vigo)r|gateway|wan)\.de|firewall\.org)|s(?:ecuritycamera\.(?:com|net|org)|hopblocks\.com)|p(?:e(?:p\.link|ts\.ws)|hotos\.cc|sx\.net|i\.co)|f(?:tp\.(?:biz|org)|irewall\.org|ritz\.net)|m(?:ailer\.com\.tw|ediapc\.net)|t(?:uleap\.com|is\.ru)|(?:jino\.r|cd\.e)u|\.(?:eu\.org|id)|oko\.niigata\.jp|home-server\.de|effect\.net|kolaiv\.ua|wire\.org)?|u(?:s(?:e(?:um(?:(?:vereniging|center)\.museum|\.(?:m[vw]|no|om|tt))?|et\.museum)|ashi(?:murayama|no)\.tokyo\.jp|ic(?:a\.(?:ar|bo)|\.museum)|\.(?:mi\.us|br))|r(?:(?:a(?:(?:yama\.yama|kami\.nii)gata|ta\.miyagi)|o(?:ran\.hokkaido|to\.kochi))\.jp|mansk\.su)|(?:(?:gi\.tokushim|ika\.niigat)a|k(?:awa\.hokkaid|o\.kyot)o)\.jp|n(?:akata\.fukuoka\.jp|cie\.museum|i\.il)|t(?:su(?:zawa\.chiba|\.aomori)\.jp|ual)|(?:en(?:chen|ster)|lhouse)\.museum|os[aá]t\.no|p\.gov\.pl)?|e(?:d(?:i(?:a(?:\.(?:museum|aero|hu|pl))?|c(?:al\.museum|ina\.bo)|zinhistorisches\.museum|o-?campidano\.it)|\.(?:p(?:[al]|ro)|e[ce]|s[ad]|br|ht|ly|om)|ecin\.(?:fr|km))?|i(?:n-(?:iserv|vigor)\.de|wa\.(?:gunma|mie)\.jp)|l(?:(?:[oø]y|and|dal|hus)\.no|bourne)|m(?:orial(?:\.museum)?|set\.net|e)|r(?:[aå]ker\.no|seine\.nu|ckmsd)|\.(?:eu\.org|u[ks]|it|ke|tz)|s(?:averde\.museum|sina\.it)|t(?:eorapp\.com|life)|e(?:res\.museum|t)|guro\.tokyo\.jp|x\.com|nu?)?|(?:á(?:tta-várjjat|latvuopmi)|å(?:lselv|søy)|j[oø]ndalen)\.no|s(?:\.(?:(?:(?:gov|leg)\.b|k)r|it|us)|k\.[rs]u|d)?|t(?:[nr]|\.(?:(?:gov|leg)\.br|eu\.org|it|us))?|c(?:\.(?:eu\.org|it)|kinsey)?|g(?:\.(?:gov|leg)\.br)?|k(?:\.(?:eu\.org|ua))?|l(?:b(?:fan\.org)?|s)?|r(?:agowo\.pl|\.no)?|b(?:\.(?:ca|it)|a)|d(?:\.(?:ci|us))?|n(?:\.(?:it|us))?|ma(?:fan\.biz)?|w(?:\.gov\.pl)?|(?:āori\.n)?z|\.(?:bg|se)|p(?:\.br)?|x(?:\.na)?)|t(?:[flz]|a(?:(?:k(?:a(?:ha(?:(?:(?:ru\.miyaz|gi\.ibar)ak|ma\.(?:aich|fuku))i|(?:shi\.okayam|ta\.yamagat)a)|(?:t(?:suki\.(?:osak|shig)|a\.fukuok|ori\.nar)|ishi\.osak|oka\.toyam)a|s(?:a(?:go\.hyogo|ki\.gunma)|hima\.shiga|u\.hokkaido)|(?:n(?:ezawa\.tochig|abe\.miyazak)|zaki\.miyazak)i|m(?:ori\.(?:kumamot|nagan)o|atsu\.kagawa)|(?:(?:razuka)?\.hyog|gi\.nagan)o|yama\.(?:g(?:unma|ifu)|nagano))|(?:e(?:t(?:omi\.okinaw|a\.oit)|hara\.hiroshim)|o\.chib|u\.sag)a|i(?:(?:no(?:ue\.hokkaid|\.hyog)|kawa\.hokkaid)o|\.mie)|ko\.aomori)|m(?:a(?:k(?:awa\.fukushima|i\.mie)|(?:mura\.gun|no\.okaya)ma|tsukuri\.ibaraki|yu\.shimane|\.tokyo)|ba\.hyogo)|ga(?:(?:mi\.niigat|wa\.fukuok)a|jo\.miyagi)|d(?:o(?:tsu\.kagawa|\.mie)|aoka\.osaka)|chi(?:arai\.fukuoka|kawa\.tokyo)|ji(?:ri\.osaka|mi\.gifu)|waramoto\.nara|hara\.aichi)\.jp|i(?:(?:(?:(?:ji\.waka|ra\.to)yam|nai\.niigat)a|shi(?:\.(?:hyogo|osaka)|n\.fukushima)|ki\.(?:hokkaido|mie)|wa\.miyagi|to\.tokyo)\.jp|fun-dns\.de|pei)|t(?:(?:e(?:(?:yama\.(?:toyam|chib)|bayashi\.gunm)a|shina\.nagano)|suno\.(?:nagan|hyog)o)\.jp|a(?:motors|r)|too)|r(?:a(?:(?:ma\.okinaw|\.sag)a\.jp|nto\.it)|u(?:mizu\.kagoshima|i\.gifu)\.jp|g(?:i\.pl|et)|nobrzeg\.pl)|n(?:a(?:(?:be\.(?:wakayama|kyoto)|gura\.fukushima)\.jp|(?:nger)?\.no)|ohata\.iwate\.jp|k\.museum)|b(?:(?:ayama\.yamanas|use\.yamaguc)hi\.jp)?|s(?:(?:\.(?:edu|gov))?\.a|hkent\.s)u|x(?:i(?:\.br)?)?|a?\.it|obao|lk)|o(?:(?:m(?:i(?:(?:gusuku\.okinaw|sato\.chib|oka\.gunm)a|ya\.miyagi|\.nagano|ka\.gifu)|(?:obe\.ibarak|e\.miyag)i|a(?:koma|r)i\.hokkaido)|b(?:e(?:tsu\.hokkaido|\.ehime)|ishima\.aichi|a\.mie)|h(?:(?:nosho\.chib|o\.fukuok)a|ma\.hokkaido)|chi(?:(?:gi\.tochi)?gi|o\.niigata)|ei\.aichi)\.jp|y(?:o(?:(?:(?:(?:hashi|kawa|ake)\.ai|\.ko)chi|n(?:(?:aka|o)\.osaka|e\.aichi)|(?:ura\.hokkaid|oka\.hyog)o|sato\.shiga)\.jp|t(?:(?:omi\.hokkaido|su\.fukuoka)\.jp|a(?:\.(?:yamagu|ai)chi\.jp)?))|a(?:(?:ko)?\.hokkaido|(?:ma\.toya)?ma)\.jp|s)|k(?:(?:a(?:(?:machi\.niigat|shiki\.okinaw)a|i\.(?:ibarak|aich)i)|u(?:(?:shima\.toku)?shima|yama\.yamaguchi)|o(?:rozawa\.saitama|name\.aichi)|i(?:gawa\.saitama|\.gifu))\.jp|yo(?:\.jp)?|ke\.no)|n(?:(?:(?:a(?:ki\.okinaw|mi\.toyam)|dabayashi\.osak)a|o(?:sho\.kagawa|\.iwate)|e\.ibaraki)\.jp|sberg\.no)|g(?:(?:a(?:(?:ne\.chib|\.toyam)a|kushi\.nagano)|(?:itsu\.nagasak|o\.aich)i|ura\.nagano)\.jp|liatti\.su)|s(?:(?:a(?:shimizu)?\.kochi|u\.saga)\.jp|hi(?:ma\.tokyo\.jp|ba)|(?:cana)?\.it)|r(?:i(?:no\.(?:museum|it)|de\.ibaraki\.jp)|a(?:hime\.shiga\.jp|y)|sken\.no)|w(?:n(?:(?:news-staging\.co|\.museu)m)?|ada\.aomori\.jp)|u(?:r(?:ism\.(?:pl|tn)|s)|ch\.museum)|t(?:(?:tori\.tot)?tori\.jp|al)|z(?:awa\.yamagata\.jp|sde\.hu)|\.(?:(?:gov|leg)\.br|it)|da(?:\.saitama\.jp|y)|o(?:n\.ehime\.jp|ls)|p(?:ology\.museum)?|lga\.no)?|r(?:a(?:n(?:i(?:-(?:andria-barlett|barletta-andri)|andriabarlett|barlettaandri)a\.it|sport(?:\.museum|e\.bo)|(?:[boø]y|a)\.no)|vel(?:ers(?:insurance)?|\.(?:pl|tt)|channel)?|d(?:ing(?:\.aero)?|e(?:r\.aero)?)|in(?:er\.aero|ing)|fficplex\.cloud|eumtgerade\.de|pani\.it|\.kp)|e(?:(?:nt(?:in(?:o(?:-(?:s(?:u[ë]?d)?-?tirol|a(?:lto)?-?adige)|s(?:u[ë]?d)?-?tirol|a(?:lto)?-?adige)?|-?su[ë]?d-?tirol)|o)|viso)\.it|e\.museum)|o(?:(?:ms[aoø]|ndheim|andin|gstad)\.no|lley\.museum|itsk\.su)|(?:øgstad|ysil|æna)\.no|ust(?:(?:ee)?\.museum)?|\.(?:eu\.org|it|no)|ieste\.it|d\.br|v)?|s(?:u(?:ru(?:g(?:a(?:shima\.saitama|\.fukui)|i\.ishikawa)|(?:\.yamanash|ta\.aomor)i|oka\.yamagata)|k(?:u(?:(?:i\.kanagaw|mi\.oit)a|ba\.ibaraki)|i(?:gata\.hokkaido|yono\.gunma))|(?:ga(?:ru\.aomor|\.tochig)|shima\.(?:nagasak|aich)|chiura\.ibarak)i|b(?:a(?:ta\.ishikaw|me\.niigat)a|etsu\.hokkaido)|(?:(?:yama\.okaya|magoi\.gun)m|iki\.fukuok)a|n(?:o\.(?:miyazak|koch)i|an\.niigata)|(?:wano\.shiman|\.mi)e)\.jp|elinograd\.su|\.it)|e(?:l(?:e(?:(?:kommunikat|vis)ion\.museum|\.amune\.org|fonica|city)|\.tr)?|n(?:(?:(?:kawa|ri)\.nar|ei\.fukushim|do\.yamagat)a\.jp|nis)|c(?:h(?:nology(?:\.museum)?)?|\.(?:mi\.us|ve)|nologia\.bo)|s(?:t(?:\.(?:ru|tj)|-iserv\.de)|hikaga\.hokkaido\.jp)|(?:a(?:ches-yoga\.co)?|x(?:tile|as)\.museu)m|r(?:n(?:opil\.ua|i\.it)|amo\.it|mez\.su)|m(?:p(?:io-?olbia\.it|-dns\.com)|asek)|\.(?:it|ua)|o\.br|va)|u(?:r(?:e(?:k\.pl|n\.tn)|ystyka\.pl|\.[ab]r|in\.it)|s(?:cany\.it|hu)|n(?:k\.org|es)|xfamily\.org|[lv]a\.su|be|i)|i(?:me(?:\.(?:museum|no)|keeping\.museum)|n(?:gvoll|n)\.no|(?:cket|p)s|(?:end|a)a|r(?:es|ol)|ffany)|h(?:e(?:at(?:er(?:\.museum)?|re)|workpc\.com|\.br)|ruhere\.net|d)?|m(?:\.(?:[nr]o|m[cg]|cy|fr|hu|km|pl|se|za)|p\.br|all)?|v(?:\.(?:b[bor]|i[mt]|t[rz]|na|sd)|edestrand\.no|s)?|y(?:(?:s(?:v[aæ]r|fjord|nes)|nset|dal)\.no|chy\.pl)|j(?:(?:eldsund|[oø]me)\.no|(?:max)?x|\.cn)?|w(?:mail\.(?:net|org|cc)|\.cn)?|c(?:m\.museum|p4\.me|i)?|(?:3l3p0rt\.ne|p\.i)t|k(?:sat\.bo|maxx)?|n(?:\.(?:it|us))?|g(?:ory\.pl)?|\.(?:bg|se)|ønsberg\.no|t(?:\.im)?|x\.us|dk?)|a(?:s(?:(?:a(?:hi(?:\.(?:(?:yamagat|toyam|chib)a|ibaraki|nagano|mie)|kawa\.hokkaido)|(?:k(?:a(?:wa\.fukushi|\.saita)|uchi\.okaya)|minami\.hiroshi)ma|go\.hyogo)|o\.kumamoto|uke\.aichi)\.jp|s(?:o(?:\.(?:eu\.org|[mn]c|bj|ci|dz|fr|gp|ht|km|re)|ciat(?:ion\.(?:museum|aero)|es))|a(?:ssination\.museum|bu\.hokkaido\.jp)|(?:isi\.museu|\.k)m|edic\.fr|n\.lk)|h(?:(?:i(?:ya\.(?:fukuoka|hyogo)|betsu\.hokkaido|kaga\.tochigi)|oro\.hokkaido)\.jp|gabad\.su)|(?:k(?:[oø]y|voll|er|im)|eral)\.no|t(?:ronomy\.museum|i\.it)|n(?:\.(?:au|lv)|es\.no)|coli-?piceno\.it|matart\.museum|[di]a|\.us)?|r(?:t(?:s(?:\.(?:museum|[cr]o|nf|ve)|andcrafts\.museum)|(?:anddesign|gallery|center|deco)\.museum|\.(?:museum|d[oz]|br|ht|pl|sn)|e(?:ducation\.museum|\.bo)?)?|a(?:(?:kawa\.(?:saitama|tokyo)|i\.shizuoka|o\.kumamoto)\.jp|mco|b)|ch(?:aeolog(?:ical|y)\.museum|i(?:tecture\.museum)?)|i(?:d(?:agaw)?a\.wakayam|(?:ake|ta)\.sag)a\.jp|e(?:(?:mark|ndal)\.no|zzo\.it)|\.(?:com|it|us)|boretum\.museum|m(?:enia\.su|y)|(?:dal|na)\.no|khangelsk\.su|q\.br|pa)?|n(?:d(?:r(?:ia(?:-(?:barletta-trani|trani-barletta)|barlettatrani|tranibarletta)\.it|oid)|(?:asuolo|ebu|øy)\.no|o(?:\.nara\.jp|y\.no)|\.museum)|a(?:n(?:\.(?:tokushima|nagano)\.jp|i\.br)|mizu\.ishikawa\.jp|lytics)|n(?:-arbor\.mi\.us|aka\.gunma\.jp|efrank\.museum)|t(?:hro(?:pology)?|iques)\.museum|(?:pachi\.gifu|jo\.aichi)\.jp|(?:cona)?\.it|quan|z)|c(?:\.(?:g(?:ov\.br|n)|l(?:eg\.br|k)|i[dlmnr]|m[aeuwz]|c[inry]|r[suw]|t[hjz]|z[amw]|a[et]|k[er]|n[iz]|p[ar]|s[ez]|u[gk]|be|jp|vn)|c(?:(?:ident-(?:investiga|preven)tion\.ae|t\.p)ro|e(?:sscam\.org|nture)|ountants?)|a(?:dem(?:y(?:\.museum)?|ia\.bo)|\.pro)|t(?:(?:\.edu)?\.au|ive|or)|hi\.nagano\.jp|o)?|i(?:(?:(?:zu(?:(?:wakamatsu|bange)\.fu|mi(?:sato\.fu|\.to))kushim|kawa\.kanagaw)a|(?:betsu\.hokkaid|oi\.hyog)o|s(?:ai\.aichi|ho\.shiga)|nan\.ehime|chi)\.jp|r(?:(?:-(?:traffic-control|surveillance)|(?:craf|por)t|line)\.aero|t(?:raffic\.aero|el)|(?:guard)?\.museum|force|bus)|d\.pl|p\.ee|go?)?|l(?:a(?:(?:bam|sk)a\.museum|headju\.no|nd\.fi)|\.(?:(?:gov|leg)\.br|eu\.org|it|no|us)|pha(?:\.bounty-full|-myqnapcloud)\.com|(?:(?:gard|vdal)\.n|farome)o|t(?:o-?adige\.it|a\.no|\.za)|s(?:t(?:ahaug\.no|om)|ace)|es(?:sandria\.it|und\.no)|l(?:finanz|state|y)|i(?:baba|pay)|waysdata\.net)?|m(?:a(?:(?:kusa\.kumamot|gasaki\.hyog)o|\.(?:shimane|aichi)|mi\.kagoshima)\.jp|e(?:rican(?:(?:a(?:ntiques|rt)?)?\.museum|express|family)|x)|b(?:ulance\.(?:museum|aero)|er\.museum)|(?:(?:li|ot)\.n|usement\.aer)o|(?:sterda(?:m\.museu)?|fa)m|(?:\.(?:gov|leg))?\.br|i(?:\.ibaraki\.jp|ca))?|p(?:p(?:\.(?:os(?:\.stg)?\.fedoraproject\.org|lmpm\.com)|s(?:\.(?:fbsbx\.com|lair\.io)|pot\.com)|l(?:inzi\.com|e)|chizi\.com)?|-(?:south(?:east-[12]|-1)|northeast-[123])\.elasticbeanstalk\.com|\.(?:gov\.(?:br|pl)|leg\.br|it)|ar(?:ecida\.br|tments))|u(?:s(?:t(?:r(?:alia\.museum|heim\.no)|in\.museum|evoll\.no)|post)|t(?:o(?:motive\.museum|\.pl|s)?|hor(?:\.aero)?)|(?:r(?:(?:skog-h[oø])?land|e)|kra)\.no|d(?:i(?:ble|o)?|nedaln\.no)|gustow\.pl|\.eu\.org|ction)?|k(?:(?:a(?:(?:bira\.hokkaid|shi\.hyog)o|iwa\.okayama|gi\.shimane)|i(?:(?:shima|runo)\.tokyo|(?:ta\.aki)?ta|\.kochi)|(?:keshi\.hokkaid|o\.hyog)o|une\.kagoshima)\.jp|(?:noluokta|rehamn)\.no|tyubinsk\.su|\.us|dn)|g(?:r(?:i(?:c(?:ulture\.museum|\.za)|gento\.it|net\.tn)|o\.(?:bo|pl)|ar\.hu|\.br)|e(?:matsu\.nagano\.jp|n(?:ts\.aero|cy))|a(?:(?:no)?\.niigata\.jp|khan)|uni\.okinawa\.jp|denes\.no|\.it)?|b(?:(?:i(?:ra\.hokkaido|ko\.chiba)|eno\.osaka)\.jp|a(?:shiri\.hokkaido\.jp|rth)|u(?:\.yamaguchi\.jp|dhabi)|o(?:\.pa|gado)|r(?:uzzo)?\.it|b(?:ott|vie)?|c(?:\.br)?|khazia\.su|\.ca|le)|t(?:(?:su(?:gi\.kanagawa|ma\.hokkaido)|ami\.shizuoka)\.jp|\.(?:eu\.org|it)|-band-camp\.net|h(?:\.cx|leta)|lanta\.museum|torney|m\.pl|o\.br)?|o(?:(?:(?:ga(?:shima\.toky|ki\.hyog)|ki\.nagan)o|(?:mori\.ao)?mori)\.jp|(?:st(?:a(?:-?valley)?|e))?\.it|l)?|v(?:o(?:cat\.(?:pro|fr)|ues\.fr)|e(?:r[oø]y\.no|llino\.it)|ia(?:tion\.museum|nca)|\.(?:it|tr))|z(?:u(?:re(?:(?:websites|-mobile)\.net|container\.io)?|mino\.nagano\.jp)|erbaijan\.su|\.us)?|e(?:ro(?:(?:batic|drome|club)\.aero|\.(?:mv|tt)|port\.fr)?|(?:\.or)?g|jrie\.no|tna)?|d(?:ac(?:hi\.tokyo\.jp)?|ul(?:t\.h)?t|v\.(?:br|mz)|ygeya\.[rs]u|m\.br|\.jp|s)?|ya(?:(?:gawa\.k|se\.kan)agawa|\.miyazaki|be\.kyoto)\.jp|\.(?:s(?:sl\.fastly\.net|e)|prod\.fastly\.net|bg)|q(?:u(?:ar(?:ium\.museum|elle)|ila\.it)|\.it)?|f(?:rica(?:\.com)?|amilycompany|jord\.no|l)?|a(?:r(?:borte\.no|p)|a(?:\.pro)?|\.no)|w(?:aji\.hyogo\.jp|s)?|x(?:is\.museum|a)?|h\.(?:cn|no)|éroport\.ci|ju\.br)|n(?:a(?:k(?:(?:a(?:g(?:awa\.(?:(?:tokushim|fukuok)a|(?:hokkaid|nagan)o)|usuku\.okinawa|yo\.kyoto)|n(?:o(?:(?:to\.ishikaw|jo\.gunm)a|\.(?:nagan|toky)o)|iikawa\.toyama)|t(?:ombetsu\.hokkaido|ane\.kagoshima|sugawa\.gifu)|m(?:(?:ichi\.yamanas|ura\.koc)hi|a\.fukuoka)|(?:yama\.yamagat|i\.kanagaw)a|\.(?:hiroshima|ibaraki)|satsunai\.hokkaido|domari\.aomori)|ijin\.okinawa)\.jp|lo\.pl)|g(?:(?:a(?:(?:r(?:eyam)?a\.chib|hama\.shig|i\.yamagat)a|s(?:(?:aki\.nagas)?aki|u\.kumamoto)|no(?:(?:\.nagano)?|hara\.gunma)|to(?:\.yamaguchi|ro\.saitama)|oka(?:kyo\.kyoto|\.niigata)|wa\.nagano)|i(?:so\.nagano|\.okayama))\.jp|o(?:\.okinawa\.jp|ya))|t(?:u(?:r(?:a(?:l(?:(?:history(?:museum)?|sciences)\.museum|\.bo))?|(?:historisches|e)\.museum|bruksgymn\.se)|urwetenschappen\.museum)|i(?:on(?:al(?:firearms|heritage)?\.museum|wide)|veamerican\.museum)|ori\.miyagi\.jp|al\.br|\.tn)|n(?:(?:(?:(?:go\.fukushi|moku\.gun)m|yo\.yamagat|jo\.okinaw)a|(?:bu\.(?:yamanash|tottor)|koku\.koch)i|a(?:e\.hokkaido|o\.ishikawa)|t(?:an\.kyoto|o\.toyama)|poro\.hokkaido)\.jp|nestad\.no)|m(?:e(?:(?:ga(?:ta\.ibaraki|wa\.saitama)|rikawa\.toyama)\.jp|\.(?:m[kvy]|t[jrt]|[hp]r|e[gt]|n[ag]|az|cy|jo|qa|vn))?|i(?:e\.fukushima|kata\.ehime)\.jp|(?:s(?:skogan|os)|dalseid)\.no)|(?:(?:chikatsuura\.wakayam|oshima\.kagaw)a|ha(?:\.okinawa|ri\.kochi)|su(?:shiobara)?\.tochigi|(?:yoro|ie)\.hokkaido)\.jp|r(?:(?:u(?:sawa\.yamanashi|to\.tokushima)|a(?:shino\.chiba|(?:\.nara)?)|ita\.chiba)\.jp|(?:vi(?:ika|k)|oy)\.no)|v(?:(?:igation\.aer|uotna\.n)o|al\.museum|oi\.su|y)|u(?:mburg\.museum|stdal\.no)|(?:p(?:les|oli))?\.it|b(?:ari\.mie\.jp)?|amesjevuemie\.no|lchik\.[rs]u|dex)?|o(?:(?:(?:s(?:e(?:gawa\.nar|\.osak)|hiro\.akit)|noichi\.ishikaw)a|b(?:oribetsu\.hokkaido|eoka\.miyazaki)|g(?:ata\.fukuoka|i\.tochigi)|zawaonsen\.nagano)\.jp|m(?:\.(?:a[defgil]|g[delt]|p[aelw]|n[ciu]|r[eos]|[qz]a|c[lo]|k[em]|m[gk]|s[it]|t[jm]|u[gy]|v[cg]|es|fr|hn|im|li)|i\.ishikawa\.jp|bre\.bo|e\.pt)|r(?:(?:d(?:-(?:(?:aur|o)dal|fron)|re(?:-land|isa)|kapp|dal)|e-og-uvdal)\.no|t(?:h(?:-kazakhstan\.su|westernmutual|\.museum)|on)|folk\.museum)|t(?:o(?:(?:gawa\.shig|\.ishikaw)a\.jp|dden\.no)|(?:icias\.b|teroy\.n)o|aires\.(?:fr|km)|\.br)|w(?:-dns\.(?:net|org|top)|aruda\.pl|\.sh|ruz|tv)?|-ip\.(?:c(?:o\.uk|a)|info|biz|net|org)|d(?:a\.(?:chiba|iwate)\.jp|um\.[ci]o)|h(?:o(?:st\.me|\.st)|eji\.aomori\.jp)|\.(?:eu\.org|com|it)|v(?:\.[rs]u|ara\.it)|ip\.(?:me|us)|kia)?|i(?:s(?:hi(?:(?:a(?:(?:wakura\.okaya|izu\.fukushi)m|(?:rita\.sa|zai\.shi)g)|(?:\.(?:fukuo|osa)|izu\.shizuo)k|go\.fukushim)a|ka(?:t(?:sura\.yamanash|a\.tochig)i|wa\.yamagata)|no(?:omote\.kagoshima|shima\.shimane|miya\.hyogo)|(?:mera\.miyazak|tosa\.koch)i|o(?:koppe\.hokkaido|\.aichi)|hara\.(?:kumamoto|okinawa)|waki\.hyogo)\.jp|s(?:hin\.aichi\.jp|edal\.no|a[ny])|\.za)|(?:i(?:(?:(?:mi\.okay|za\.sait)am|(?:gata\.nii)?gat)a|kappu\.hokkaido|hama\.ehime)|(?:rasaki\.yamanas|yodogawa\.koc)hi)\.jp|k(?:(?:i\.hokkaido|ko\.tochigi|aho\.akita)\.jp|o(?:laev\.ua|n)|e)|c(?:hinan\.(?:miyazak|tottor)i\.jp|\.(?:in|tj)|o)|n(?:o(?:miya\.kanagawa|he\.iwate)\.jp|ja)|e(?:ruchomosci\.pl|pce\.museum)|t(?:tedal\.no|eroi\.br)|d\.io)?|e(?:t(?:\.(?:e(?:[ct]|(?:u\.or)?g)|m[aeklostuvwxyz]|b[abhmnorstz]|p[aehklnrsty]|s[abcdghloty]|a[cefgilruz]|g[eglnprtuy]|t[hjmnortw]|c[imnouwy]|i[dlmnqrs]|l[abckrvy]|k[ginwyz]|n[fgirz]|v[ceinu]|u[akyz]|d[moz]|h[knt]|j[eo]|r[uw]|z[am]|om|qa|ws)|(?:-freaks|lify)\.com|(?:ban|wor)k|flix)?|w(?:(?:jersey|mexico|port|york)\.museum|h(?:ampshire\.museum|olland)|s(?:paper\.museum|\.hu)?)?|(?:s(?:\.(?:akershus|buskerud)|odd(?:tang)?en|se(?:by|t)|na)|dre-eiker)\.no|(?:muro\.hokkaido|yagawa\.osaka)\.jp|(?:braska\.museu|at-url\.co)m|r(?:ima\.tokyo\.jp|dpol\.ovh)|\.(?:k[er]|u[gs]|jp|pw|tz)|u(?:es\.museum|star)|x(?:(?:tdirec)?t|us)|c)?|y(?:m\.(?:l[acitu]|m[enx]|s[kux]|[kn]z|b[yz]|g[ry]|p[et]|ie|ro|tw)|c(?:\.m(?:useum|n))?|uzen\.toyama\.jp|ny\.museum|sa\.pl|\.us)|u(?:ma(?:ta\.(?:hokkaido|gunma)|zu\.shizuoka)\.jp|(?:ern|rem)berg\.museum|\.(?:ca|it)|oro\.it)?|(?:(?:(?:øtte|æ)røy|ååmesjevuemie|ávuotna)\.n|4t\.c)o|t(?:\.(?:(?:edu\.)?au|[nr]o|ca)|dll\.top|r\.br|t)|g(?:o(?:\.(?:lk|ph|za))?|\.eu\.org|rok\.io)?|h(?:(?:(?:-serv\.co|s)\.u)?k|lfan\.net|\.us)|s(?:w(?:\.edu)?\.au|update\.info|n\.us|\.ca)|f(?:l(?:fan\.org)?|shost\.com|\.ca)?|l(?:\.(?:eu\.org|ca|no))?|r(?:w(?:\.museum)?|a)?|c(?:\.(?:tr|us))?|z(?:\.eu\.org)?|m\.(?:cn|us)|\.(?:bg|se)|b(?:\.c)?a|[djv]\.us|x\.cn)|c(?:[gmwx]|o(?:m(?:\.(?:s[abcdeghlnotvy]|b[abhimnorstyz]|m[gklostuvwxy]|g[ehilnprtuy]|p[aefhklrsty]|a[cfgilruwz]|c[imnouwy]|k[gimpwyz]|l[abckrvy]|t[jmnortw]|e[cegst]|n[afgir]|v[ceinu]|d[emoz]|h[knrt]|i[moqs]|r[eouw]|u[agyz]|[oz]m|fr|jo|qa|ws)|m(?:uni(?:cations?\.museum|ty(?:\.museum)?)|bank)|p(?:uter(?:(?:history)?\.museum)?|a(?:ny|re))|unicações\.museum|(?:o\.i|cas)t|sec)?|n(?:t(?:emporary(?:art)?\.museum|r(?:ol\.aero|actors)|a(?:gem\.br|ct))|s(?:ul(?:t(?:ing(?:\.aero)?|ant\.aero)|ado\.st)|truction)|f(?:erence\.aero|\.(?:au|lv))|vent\.museum|dos)|\.(?:c(?:[ailrz]|o?m)|k(?:rd?|e)|i[dlmnrt]|m[aeguwz]|n[ailoz]|t[hjmtz]|u[agksz]|a[egot]|b[biw]|g[gly]|p[lnw]|z[amw]|j[ep]|l[cs]|r[sw]|s[tz]|v[ei]|dk|hu|om)|l(?:o(?:(?:nialwilliamsburg|radoplateau)\.museum|gne)|le(?:ge(?:fan\.org)?|ction\.museum)|(?:umb(?:ia|us)|dwar)\.museum)|u(?:n(?:t(?:ry(?:estate\.museum)?|y\.museum)|cil\.aero)|chpotatofries\.org|pons?|rses)|o(?:p(?:\.(?:[ht]t|m[vw]|br|km|py)|erativa\.bo)?|king(?:channel)?|l)|r(?:(?:reios-e-telecomunicações|poration|vette)\.museum|sica)|a(?:(?:staldefence|l)\.museum|ch)|d(?:es(?:pot\.com)?|y\.museum)|s(?:tume\.museum|enza\.it)|penhagen\.museum|g\.mi\.us|ffee)?|h(?:i(?:(?:ku(?:s(?:hino\.fukuoka|ei\.ibaraki)|ho(?:ku\.nagano|\.fukuoka)|(?:[gj]o|zen)\.fukuoka|ma\.nagano)|(?:hayaakasaka\.osak|gasaki\.kanagaw|b)a|(?:jiwa\.nagasak|zu\.tottor)i|t(?:ose\.hokkaido|a\.aichi)|yoda\.(?:gunma|tokyo)|ppubetsu\.hokkaido)\.jp|r(?:urgiens-dentistes(?:-en-france)?\.fr|opractic\.museum|yu\.aichi\.jp)|c(?:hibu\.saitama\.jp|ago\.museum)|ldren(?:s(?:garden)?)?\.museum|n(?:o\.nagano\.jp|tai)|mkent\.su|eti\.it)|a(?:m(?:pionship\.aero|bagri\.fr)|n(?:nel(?:sdvr\.net)?|el)|t(?:tanooga\.museum)?|r(?:ter\.aero|ity)|se)|e(?:r(?:n(?:i(?:(?:go|hi)v|vtsi)|ovtsy)|kas?sy)\.ua|(?:sapeakebay|ltenham)\.museum|ap)|u(?:o\.(?:(?:(?:fukuo|osa)k|chib)a|yamanashi|tokyo)\.jp|ng(?:buk|nam)\.kr|rch)|o(?:(?:(?:yo\.kumamot|fu\.toky)o|(?:s[eh]i|nan)\.chiba)\.jp|colate\.museum)|r(?:ist(?:iansburg\.museum|mas)|ysler|ome)|\.(?:eu\.org|it)|tr\.k12\.ma\.us)?|a(?:r(?:r(?:ara-?massa\.it|ier\.museum)|t(?:oonart\.museum|ier)|bonia-?iglesias\.it|e(?:ers?)?|go\.aero|avan|d?s)?|m(?:p(?:i(?:na(?:grande|s)\.br|dano-?medio\.it)|(?:obasso|ania)\.it)?|bridge\.museum|dvr\.org|\.it|era)?|s(?:a(?:delamoneda\.museum|cam\.net)?|t(?:res|le)\.museum|e(?:rta\.it|ih)?|ino(?:\.hu)?|h)|l(?:(?:(?:tanissett|abri)a)?\.it|ifornia\.museum|vinklein|l)?|(?:-central-1\.elasticbeanstalk\.co|daques\.museu)m|t(?:an(?:zaro|ia)\.it|ering(?:\.aero)?|holic)?|p(?:e(?:breton\.museum|town)|ital(?:one)?)|n(?:(?:ada)?\.museum|cerresearch|on)|(?:hcesuolo\.n|a\.aer)o|\.(?:eu\.org|it|na|us)|b(?:le-modem\.org)?|gliari\.it|xias\.br|fe)?|l(?:o(?:ud(?:(?:(?:f(?:unctions|ront)|ycluster|eity)\.ne|a(?:ccess\.(?:hos|ne)|pp\.ne))t|ns\.(?:c(?:lub|c)|in(?:fo)?|p(?:ro|w)|asia|biz|org|eu|us)|\.(?:(?:fedoraproject\.or|goo)g|metacentrum\.cz)|control(?:app|led)\.com|66\.ws)?|ck\.museum|thing)|i(?:n(?:ton\.museum|i(?:que|c))|ck)|ub(?:\.(?:aero|tw)|med)?|e(?:verapps\.io|aning)|a(?:n\.rip|ims)|\.it)?|i(?:vil(?:(?:i[sz]ation|war)\.museum|aviation\.aero)|s(?:co(?:freak\.com)?|tron\.nl)|t(?:y(?:\.hu|eats)?|adel|ic?)|n(?:cinnati|ema)\.museum|e(?:ncia\.bo|szyn\.pl)|rc(?:us\.museum|le)|priani|m\.br|\.it)?|u(?:st(?:om(?:er\.(?:speedpartner\.de|enonic\.io)|\.metacentrum\.cz)|\.(?:d(?:isrec|ev)|testing|prod)\.thingdust\.io)|ltur(?:al(?:center)?|e)\.museum|i(?:aba\.br|sinella)|ritiba\.br|pcake\.is|neo\.it)?|r(?:e(?:dit(?:union|card)?|ation\.museum|mona\.it|w\.aero)|a(?:ft(?:s\.museum|ing\.xyz)|nbrook\.museum)|i(?:\.(?:br|nz)|mea\.ua|cket)|o(?:tone\.it|wn)|\.(?:it|ua)|uises?|s)?|c(?:\.(?:(?:m[adeinost]|a[klrsz]|i[adln]|w[aivy]|c[aot]|o[hkr]|v[ait]|[hr]i|d[ce]|g[au]|k[sy]|p[ar]|s[cd]|t[nx]|fl|la)\.us|n(?:[cdehjmvy]\.us|a)|u(?:t\.us|a))|i\.fr)?|e(?:[bo]|r(?:t(?:ification\.aero|mgr\.org)|n)|(?:ltic\.museu|chire\.co)m|\.(?:(?:gov|leg)\.br|it)|nter(?:\.museum)?|sena-?forlì?\.it)|n(?:(?:-north(?:west)?-1\.eb\.amazonaws\.com\.c|py\.gd)n|\.(?:eu\.org|com|it|ua)|[gt]\.br)?|y(?:o(?:n\.(?:link|site)|u)|(?:\.eu\.or|a\.g)g|mru(?:\.museum)?|ber\.museum)?|z(?:\.(?:eu\.org|it)|e(?:ladz|st)\.pl)?|\.(?:(?:cdn77\.or|b)g|la|se)|d(?:n77-ssl\.net|\.eu\.org)?|b(?:[ans]|g\.ru|\.it|re)|t\.(?:it|us)|s(?:\.it|c)|v(?:\.ua)?|pa\.pro|66\.me|f[ad]?|k\.ua|q\.cn)|h(?:i(?:(?:(?:gashi(?:(?:(?:hiroshima\.hiroshi|chichibu\.saita|agatsuma\.gun)m|\.(?:fuku(?:shim|ok)|okinaw)|n(?:aruse\.aki|e\.yamaga)t|o(?:saka\.osak|mi\.shig))a|k(?:a(?:g(?:ura\.hokkaido|awa\.kagawa)|wa\.hokkaido)|urume\.tokyo)|y(?:o(?:dogawa\.osak|shino\.nar)a|ama(?:to\.toky|\.kyot)o)|m(?:atsu(?:shima\.miyagi|yama\.saitama)|urayama\.tokyo)|s(?:umiyoshi\.osaka|hirakawa\.gifu)|izu(?:mo\.shimane|\.shizuoka)|(?:tsuno\.ko|ura\.ai)chi)|r(?:a(?:(?:t(?:suka\.kanagaw|a\.fukushim)|kata\.osak|ra\.okinaw)a|(?:do\.nagasak|nai\.aomor)i|izumi\.iwate|ya\.nagano)|o(?:(?:gawa\.wakayam|kawa\.fukuok)a|no\.(?:fukushima|iwate)|s(?:aki\.aomori|hima)|o\.hokkaido))|k(?:a(?:ri\.yamaguchi|wa\.shimane)|imi\.shimane|one\.shiga)|da(?:ka\.(?:(?:wakay|sait)ama|hokkaido|kochi)|\.gifu)|no(?:(?:hara|de)\.tokyo|\.to(?:ttori|kyo))|m(?:e(?:shima\.oita|ji\.hyogo)|i\.toyama)|(?:oki\.kagoshim|zen\.sag|ji\.oit)a)\.j|pho)p|s(?:to(?:r(?:i(?:c(?:al(?:society)?|houses)|sch(?:es)?)|y(?:ofscience)?)|ire)\.museum|a(?:yama\.fukuoka\.jp|mitsu))|t(?:a(?:chi(?:(?:(?:o(?:miy|t)|nak)a)?\.ibaraki\.jp)?|\.oita\.jp)|ra\.no)|c(?:hiso\.gifu\.jp|am\.net)|\.(?:cn|us)|v)|a(?:(?:k(?:u(?:(?:san|i)\.ishikawa|ba\.nagano)|o(?:date\.hokkaido|ne\.kanagawa)|ata\.fukuoka)|chi(?:(?:oji|jo)\.tokyo|rogata\.akita|nohe\.aomori)|ya(?:kawa\.yamanashi|shima\.okayama)|ebaru\.okinawa|zu\.aichi)\.jp|m(?:a(?:(?:t(?:onbetsu\.hokkaido|ama\.saga)|matsu\.shizuoka|da\.shimane)\.jp|r(?:oy)?\.no)|m(?:arfeasta|erfest)\.no|burg(?:\.museum)?|-radio-op\.net|ura\.tokyo\.jp)|n(?:(?:a(?:m(?:igawa\.chiba|aki\.iwate)|wa\.fukushima)|(?:n(?:o\.saitam|an\.osak)|yu\.saitam)a)\.jp|d(?:a\.aichi\.jp|son\.museum)|g(?:gliding\.aero|out))|s(?:h(?:i(?:m(?:oto\.wakayama|a\.gifu)|kami\.aomori)\.jp|bang\.sh)|u(?:ra(?:-app\.io|\.app)|da\.saitama\.jp)|am(?:i\.nagasaki|a\.oita)\.jp|vik\.no)|r(?:a(?:\.nagano\.jp|m\.no)|vestcelebration\.museum|(?:sta|ei)d\.no|ima\.hyogo\.jp)|t(?:(?:o(?:gay|yam)a\.saita|sukaichi\.hiroshi)ma\.jp|tfjelldal\.no)|b(?:(?:oro\.hokkaido|ikino\.osaka)\.jp|mer\.no)|g(?:(?:i\.yamaguch|a\.tochig)i\.jp|ebostad\.no)|l(?:loffame\.museum|(?:den|sa)\.no|f\.host)|d(?:ano\.kanagawa\.jp|sel\.no)|p(?:pou\.akita\.jp|mir\.no)|i(?:bara\.shizuoka\.jp|r)|u(?:gesund\.no|s)|waii\.museum|\.(?:cn|no))|o(?:me(?:s(?:e(?:curity(?:ma|p)c\.com|nse))?|lin(?:ux\.(?:com|net|org)|k\.one)|unix\.(?:com|net|org)|d(?:ns\.org|epot)|ftp\.(?:net|org)|office\.gov\.uk|-webserver\.de|\.dyndns\.org|built\.aero|ip\.net|goods)|n(?:(?:(?:j(?:o\.(?:saitam|akit)|yo\.akit)|go\.hiroshim)a|betsu\.hokkaido|ai\.ehime)\.jp|e(?:foss\.no|ywell)|da)|k(?:u(?:to\.(?:yamanashi|hokkaido)|ryu\.hokkaido)\.jp|k(?:aido\.jp|sund\.no))|r(?:o(?:(?:kanai|nobe)\.hokkaido\.jp|logy\.museum)|(?:nindal|ten)\.no|se)|l(?:(?:mestrand|t[aå]len|e)?\.no|dings|iday)|s(?:t(?:ing(?:-cluster\.nl)?)?|pital)|b(?:by-site\.(?:com|org)|[oø]l\.no)|t(?:el(?:\.(?:hu|lk|tz)|e?s)|mail)?|f(?:u\.yamaguchi\.jp|\.no)|y(?:landet|anger)\.no|use(?:\.museum)?|pto\.(?:org|me)|ckey|w)|e(?:r(?:o(?:y\.(?:more-og-romsdal|nordland)\.no|ku(?:app|ssl)\.com)|(?:øy\.(?:møre-og-romsdal|nordland)|ad)\.no|e(?:-for-more\.info)?|itage\.museum|mes)|alth(?:\.(?:museum|nz|vn)|-carereform\.com|care)?|m(?:bygdsforbund\.museum|(?:sedal|nes?)\.no)|l(?:sinki(?:\.museum)?|las\.museum|p)|(?:kinan\.aichi|guri\.nara)\.jp|imatunduhren\.museum|pforge\.org|\.cn)|(?:(?:ø(?:y(?:landet|anger)|nefoss)|á(?:(?:bme|pmi)r|mmárfeasta)|j(?:elmeland|artdal)|ægebostad|valer|å)\.n|zc\.i)o|u(?:\.(?:eu\.org|com|net)|issier-justice\.fr|manities\.museum|r(?:dal|um)\.no|ghes)?|y(?:u(?:ga\.miyazaki\.jp|ndai)|llestad\.no|ogo\.jp|att)|k(?:\.(?:c(?:om|n)|org)|t)?|b(?:\.c(?:ldmail\.ru|n)|o)|r(?:\.eu\.org)?|dfc(?:bank)?|l\.(?:cn|no)|s(?:\.kr|bc)|\.(?:bg|se)|(?:n\.c)?n|m(?:\.no)?|gtv|t)|i(?:[oq]|s(?:-(?:a(?:-(?:(?:(?:h(?:ard-work|unt)e|financialadviso)r|d(?:e(?:mocrat|signer)|octor)|t(?:e(?:acher|chie)|herapist)|r(?:epublican|ockstar)|n(?:ascarfan|urse)|anarchist|musician)\.com|c(?:(?:(?:ubicle-sla|onservati)ve|pa)\.com|a(?:ndidate\.org|terer\.com)|hef\.(?:com|net|org)|elticsfan\.org)|l(?:i(?:ber(?:tarian|al)\.com|nux-user\.org)|(?:a(?:ndscap|wy)er|lama)\.com)|p(?:(?:ersonaltrain|hotograph|lay)er\.com|a(?:inter\.com|tsfan\.org))|b(?:(?:(?:ookkeep|logg)er|ulls-fan)\.com|ruinsfan\.org)|s(?:o(?:cialist\.com|xfan\.org)|tudent\.com)|g(?:eek\.(?:com|net|org)|(?:reen|uru)\.com)|knight\.org)|n-(?:a(?:c(?:t(?:ress|or)|countant)|(?:narch|rt)ist)|en(?:tertain|gine)er)\.com)|(?:into-(?:(?:car(?:toon)?|game)s|anime)|(?:(?:not-)?certifie|with-theban)d|uberleet|gone)\.com|(?:very-(?:(?:goo|ba)d|sweet|evil|nice)|found)\.org|s(?:aved\.org|lick\.com)|l(?:eet\.com|ost\.org)|by\.us)|(?:hi(?:ka(?:w(?:a\.(?:fukushim|okinaw))?a|ri\.hokkaido)|nomaki\.miyagi|gaki\.okinawa)|umi\.chiba)\.jp|a(?:-(?:geek\.(?:com|net|org)|hockeynut\.com)|\.(?:kagoshima\.jp|us)|haya\.nagasaki\.jp)|e(?:(?:(?:(?:n\.kagoshi|saki\.gun)m|hara\.kanagaw)a|\.mie)\.jp|(?:rnia\.i|lec)t)|s(?:marterthanyou\.com|hiki\.aichi\.jp)|t(?:(?:eingeek|mein)\.de|anbul)?|l(?:eofman\.museum|a\.pr)|\.(?:eu\.org|gov\.pl|it)|maili)?|n(?:(?:a(?:(?:(?:mi\.(?:waka|to)ya|washiro\.fukushi)m|tsuki\.fukuok)a|(?:shiki\.ibarak|zawa\.aich)i|\.(?:ibaraki|saitama|nagano)|g(?:awa\.hyog|i\.toky)o|be\.mie)|(?:uyama\.ai|o\.ko)chi|zai\.chiba|e\.kyoto)\.jp|t(?:e(?:r(?:n(?:et-dns\.de|ational)|active\.museum)|l(?:ligence\.museum)?)|\.(?:eu\.org|a[rz]|c[io]|l[ak]|m[vw]|r[uw]|t[jt]|v[en]|bo|is|ni|pt)|l\.tn|uit)?|f(?:o(?:\.(?:n[afir]|t[nrtz]|a[tuz]|p[klr]|b[bo]|c[ox]|e[ct]|h[tu]|k[ei]|v[en]|gu|la|mv|ro|sd|zm))?|\.(?:br|cu|mk|ua)|initi)|d(?:i(?:an(?:a(?:polis)?|market)?\.museum|gena\.bo)|\.(?:[it]n|br|gt|kw)|ustri(?:a\.bo|es)|er[oø]y\.no)|\.(?:n(?:[ai]|et)|eu\.org|u[as]|rs|th)|s(?:ur(?:ance(?:\.aero)?|e)|titute)|-(?:the-band\.net|addr\.arpa)|g(?:atlan\.hu|\.pa)?|c(?:heon\.kr|\.hk)?|vestments|k)?|w(?:a(?:t(?:(?:suki\.saitam|a\.shizuok)a|(?:e\.iwat)?e)|k(?:u(?:ni\.yamagu|ra\.ai)chi|i\.fukushima)|m(?:izawa\.hokkaido|a\.ibaraki)|n(?:ai\.hokkaido|uma\.miyagi)|fune\.tochigi|de\.wakayama|izumi\.iwate)\.jp|i\.nz)|t(?:a(?:(?:k(?:o\.ibaraki|ura\.gunma)|(?:bashi\.toky|mi\.hyog)o|yanagi\.aomori|no\.tokushima)\.jp|u)|o(?:igawa\.niigat|man\.okinaw|\.shizuok)a\.jp|\.(?:eu\.org|ao)|s\.me|v)?|c(?:[eu]|hi(?:ka(?:wa(?:\.(?:chiba|hyogo)|misato\.yamanashi)|i\.tochigi)|no(?:miya\.(?:aichi|chiba)|(?:seki|he)\.iwate)|(?:ba\.tokushim|hara\.chib)a)\.jp|\.gov\.pl|bc)|k(?:(?:eda\.(?:(?:hokkaid|nagan)o|fukui|osaka|gifu)|usaka\.nagano|oma\.nara)\.jp|a(?:(?:(?:ruga\.nar|wa\.akit)a|ta\.ehime)\.jp|no)|i\.(?:nagasaki\.jp|fi))|z(?:(?:u(?:m(?:i(?:(?:otsu|sano)\.osak|\.(?:kagoshim|osak)|zaki\.fukushim)a|o(?:zaki\.niigata|\.shimane))|(?:nokuni)?\.shizuoka)|ena\.okinawa)\.jp|\.hr)|m(?:a(?:(?:kane\.hokkaido|bari\.ehime|ri\.saga)\.jp|geandsound\.museum|mat)|izu\.toyama\.jp|(?:peria)?\.it|mo(?:bilien)?|b\.br|db)?|(?:i(?:zu(?:ka\.fukuoka|na\.nagano)|d(?:e\.yamagata|a\.nagano)|(?:ji|ya)ma\.nagano|tate\.fukushima)|heya\.okinawa|yo\.ehime)\.jp|b(?:(?:ara(?:ki(?:\.(?:ibaraki|osaka))?|\.okayama)|igawa\.gifu)\.jp|estad\.no|m)|d(?:\.(?:l[vy]|au|ir|us)|e\.kyoto\.jp|v\.(?:hk|tw)|rett\.no|f\.il)?|l(?:lustration\.museum|\.(?:eu\.org|us)|ovecollege\.info|awa\.pl)?|v(?:ano(?:-frankivsk\.ua|vo\.su)|(?:e(?:land\.n|c)|gu\.n)o)|r(?:(?:aq|on)\.museum|uma\.saitama\.jp|is(?:\.arpa|h))?|p(?:i(?:fony\.net|ranga)|6\.arpa)|a(?:mallama\.com|\.us)|glesias-?carbonia\.it|e(?:\.eu\.org|ee)?|\.(?:[bn]g|ph|se)|f(?:\.ua|m)|234\.me)|b(?:[fw]|a(?:r(?:sy(?:\.(?:s(?:upport|hop|ite)|m(?:e(?:nu)?|obi)|i(?:n(?:fo)?|o)|c(?:o\.uk|lub)|o(?:nline|rg)|p(?:ro|ub)|net|bg|de|eu|uk)|online\.co(?:\.uk|m)|center\.com)|(?:(?:letta(?:-trani-|trani)andria|i)\.i|efoo)t|c(?:elona(?:\.museum)?|lay(?:card|s))|re(?:l?l-of-knowledge\.info|au\.bj)|u(?:eri\.br|m\.no)|(?:du\.n|\.pr)o|gains)?|l(?:s(?:an(?:-su[ë]?dtirol)?\.it|fjord\.no)|l(?:ooning\.aer|angen\.n)o|e(?:strand\.no|\.museum)|a(?:shov\.su|t\.no)|timore\.museum)|s(?:e(?:ball(?:\.museum)?|l\.museum)|(?:ilicata)?\.it|hkiria\.[rs]u|ketball)|n(?:d(?:(?:ai\.fukushima|o\.ibaraki)\.jp)?|a(?:narepublic|mex)|k)|t(?:o\.tochigi\.jp|hs\.museum|sfjord\.no)|(?:(?:jddar|mble)\.n|ckplaneapp\.i)o|h(?:c?cavuotna\.no|n\.museum)|da(?:joz\.museum|ddja\.no)|\.(?:(?:gov|leg)\.br|it)|u(?:ern\.museum|haus)|b(?:ia-gora\.pl|y)|id(?:ar\.no|u)|ghdad\.museum|yern)?|o(?:l(?:(?:zano(?:-altoadige)?|ogna)\.it|dlygoingnowhere\.org|eslawiec\.pl|ivia\.bo|t\.hu)|t(?:an(?:ic(?:al(?:garden)?|garden)|y)\.museum)?|u(?:n(?:ty-full\.com|ceme\.net)|tique)|\.(?:(?:nordland|telemark)\.no|it)|s(?:t(?:on(?:\.museum)?|ik)|ch)|zen(?:-su[ë]?dtirol)?\.it|o(?:k(?:ing)?|mla\.net)?|a(?:vista\.br|ts)|(?:d[oø]|kn)\.no|n(?:n\.museum|d)|x(?:fuse\.io)?|m(?:lo\.no)?|ehringer|fa)?|i(?:z(?:\.(?:p[klr]|t[jrt]|a[tz]|m[vw]|n[ir]|bb|cy|dk|et|id|ki|ua|vn|zm)|en\.okayama\.jp)?|r(?:(?:thplace|dart)\.museum|atori\.hokkaido\.jp|kenes\.no|\.ru)|e(?:l(?:awa\.pl|la\.it)|i\.hokkaido\.jp|szczady\.pl|v[aá]t\.no)|b(?:ai\.hokkaido\.jp|le(?:\.museum)?)|(?:l(?:bao|l)\.museu|tballoon\.co)m|(?:fuka|horo)\.hokkaido\.jp|al(?:owieza|ystok)\.pl|n(?:dal\.no|go?)|o(?:\.br)?|\.it|ke|d)?|r(?:o(?:(?:wsersafetymark\.i|nnoy(?:sund)?\.n)o|ke(?:r(?:\.aero)?|-it\.net)|ad(?:cast\.museum|way)|ther)|u(?:ssel(?:s(?:\.museum)?|\.museum)|(?:xelles|nel)\.museum|munddal\.no)|a(?:nd(?:ywinevalley\.museum|\.se)|sil(?:\.museum|ia\.me)|desco)|i(?:(?:tish(?:columbia)?|stol)\.museum|ndisi\.it|dgestone)|e(?:manger\.no|scia\.it)|y(?:ansk\.su|ne\.no)|ønnøy(?:sund)?\.no|\.(?:com|it))?|e(?:r(?:g(?:(?:en)?\.no|bau\.museum|amo\.it)|l(?:in(?:\.museum)?|ev[aå]g\.no)|(?:keley|n)\.museum)|a(?:r(?:alv[aá]hki|du)\.no|u(?:xarts\.museum|ty)|ts)|l(?:l(?:evue\.museum|uno\.it)|(?:em\.b|\.t)r|au\.pw)|t(?:a(?:\.bounty-full|inabox)\.com|ter-than\.tv)?|e(?:ldengeluid\.museum|p\.pl|r)|s(?:t(?:buy)?|kidy\.pl)|n(?:evento\.it|tley)|ppu\.oita\.jp|\.eu\.org|dzin\.pl|iarn\.no)?|l(?:o(?:g(?:s(?:pot\.(?:c(?:[afhlvz]|o(?:m(?:\.(?:e[egs]|a[ru]|b[ry]|c[oy]|mt|ng|tr|uy))?|\.(?:i[dl]|at|ke|nz|uk|za)))|m[dkrxy]|s[egikn]|b[aegj]|i[enst]|r[eosu]|a[elm]|h[kru]|l[itu]|[gk]r|d[ek]|f[ir]|n[lo]|p[et]|t[dw]|jp|qa|ug|vn)|ite\.(?:org|xyz)|yte\.com)|dns\.(?:com|net|org)|\.b[or])?|xcms\.com|ckbuster|omberg)|a(?:ck(?:baudcdn\.net|friday)?|nco)|\.it|ue)|u(?:n(?:go(?:takada|ono)\.oita|kyo\.tokyo)\.jp|s(?:(?:hey)?\.museum|an\.kr|iness)|lsan(?:-su[ë]?dtirol)?\.it|ild(?:ing\.museum|ers)?|z(?:en\.fukuoka\.jp|z)|d(?:ejju\.no|apest)|y(?:shouses\.net)?|rghof\.museum|khara\.su|gatti|\.no)|(?:á(?:(?:jdda|idá)r|hc?cavuotna|lát)|ø(?:\.(?:nordland|telemark)|mlo)|å(?:tsfjord|dåddjå)|ærum)\.no|y(?:(?:dgoszcz|tom)\.pl|(?:gland|kle)\.no|en\.site)?|j(?:(?:ark[oø]y|erkreim|ugn)\.no|\.cn)?|\.(?:s(?:sl\.fastly\.net|e)|b[gr])|m(?:[sw]|oattachments\.org|d\.br)?|c(?:[gn]|i\.dnstrace\.pro|\.ca)|n(?:pparibas|r\.la|\.it|l)?|g(?:\.(?:eu\.org|it))?|placed\.(?:com|net|de)|b(?:[ct]|s\.tr|va)?|h(?:z\.br|arti)?|s(?:b\.br|\.it)?|z(?:\.it|h)?|(?:t\.i)?t|v(?:\.nl)?|d\.se)|o(?:r(?:g(?:\.(?:m[aegklnostuvwxyz]|s[abcdeghlnotvyz]|b[abhimnorstwz]|g[eghilnprtuy]|p[aefhklnrsty]|a[cefgilruz]|k[gimnpwyz]|l[abckrsvy]|t[jmnortw]|c[inouwy]|i[lmnqrs]|e[cegst]|n[agirz]|u[agkyz]|v[ceinu]|h[kntu]|d[moz]|r[osu]|z[amw]|j[eo]|om|qa|ws)|anic)?|\.(?:c[ir]|i[dt]|k[er]|t[hz]|u[gs]|at|bi|jp|mu|na|pw)|(?:k(?:anger|dal)|s(?:kog|ta)|land)\.no|a(?:\.gunma\.jp|(?:cl|ng)e)|egon(?:trail)?\.museum|i(?:stano\.it|gins))|s(?:a(?:k(?:i(?:kamijima\.hiroshima|\.miyagi)\.jp|a(?:(?:sayama\.osaka)?\.jp)?)|sco\.br)|t(?:r(?:o(?:w(?:wlkp|iec)|(?:lek|d)a)\.pl|e-toten\.no)|er[oø]y\.no)|h(?:i(?:ma\.(?:yamaguchi|tokyo)|no\.yamanashi)|u\.iwate)\.jp|(?:\.h(?:ordaland|edmark)|(?:[oø]yr|l)o)\.no|e(?:to\.nagasaki\.jp|n\.no))|t(?:a(?:(?:k(?:i\.(?:(?:saitam|chib)a|nagano)|e\.hiroshima)|r(?:u\.hokkaid|i\.nagan)o|\.(?:gunma|tokyo)|ma\.fukushima)\.jp|go\.museum)|su(?:k(?:i\.(?:yamanas|koc)hi\.jp|a)|(?:chi\.iwate|\.shiga)\.jp)|o(?:(?:(?:fuk|b)e|ineppu)\.hokkaido|\.fukuoka|yo\.kochi)\.jp|(?:\.i)?t|her\.nf)|k(?:(?:a(?:ya(?:(?:ma\.okaya)?ma|\.nagano)|wa\.(?:fukuoka|kochi)|gaki\.fukuoka|zaki\.aichi)|u(?:izumo\.shimane|ma\.fukushima|tama\.tokyo)|e(?:gawa\.saitama|to\.hokkaido)|oppe\.hokkaido)\.jp|i(?:n(?:awa(?:(?:\.okinawa)?\.jp)?|oshima\.shimane\.jp)|\.fukuoka\.jp)|snes\.no|\.us)|n(?:(?:o(?:(?:michi\.hiroshim|jo\.fukuok)a|\.(?:fuku(?:shima|i)|hyogo))|(?:juku\.chib|na\.okinaw)a|agawa\.miyagi)\.jp|-(?:aptible\.com|the-web\.tv|web\.fr)|t(?:ario\.museu|hewifi\.co)m|g(?:a\.fukuoka\.jp|\.br)?|l(?:ine(?:\.museum)?)?|(?:yoursid)?e|\.ca|ion)|g(?:(?:a(?:wa(?:\.(?:ibaraki|saitama|nagano)|ra\.miyagi)|(?:(?:ta)?\.akit|no\.saitam)a|sawara\.tokyo|ki\.gifu)|(?:o(?:ri\.fukuok|se\.saitam)|i(?:mi\.okinaw|\.sag))a|u(?:ni\.(?:kumamoto|yamagata)|chi\.aichi))\.jp|\.(?:ao|it)|liastra\.it)|m(?:(?:i(?:(?:hachiman\.shig|gawa\.chib|ya\.saitam)a|\.n(?:iigata|agano)|tama\.ibaraki)|u(?:ra\.nagasaki|ta\.fukuoka)|otego\.fukushima)\.jp|a(?:(?:chi\.(?:nagano|saga)|ezaki\.shizuoka)\.jp|svuotna\.no|ha\.museum)|e(?:\.tokyo\.jp|ga))?|b(?:(?:a(?:ma\.(?:nagasak|fuku)i|nazawa\.yamagata)|u(?:se\.nagano|\.aichi))\.jp|i(?:(?:hiro|ra)\.hokkaido\.jp)?|ninsk\.su|server)|i(?:(?:s(?:hida\.yamagat|o\.kanagaw)|(?:ta\.oi)?t|zumi\.gunm|\.kanagaw)a\.jp|r(?:ase\.aomori\.jp|m\.gov\.pl))|h(?:(?:(?:i(?:ra\.(?:tochi|miya)g|\.fuku)|tawara\.tochig|aru\.aich)i|kura\.yamagata|da\.shimane)\.jp|\.us)|y(?:(?:a(?:ma(?:zaki\.kyoto|\.tochigi)|be\.toyama)|odo\.nara)\.jp|(?:stre-slidre|garden|er)\.no|\.lc)|p(?:e(?:n(?:craft\.hosting|air\.museum)?|raunite\.com)|p(?:eg[aå]rd|dal)\.no|o(?:czno|le)\.pl)|l(?:a(?:yan(?:group)?|wa\.pl)|(?:sztyn|ecko|kusz)\.pl|bia-?tempio\.it|(?:\.n|l)o|dnavy)|(?:(?:ji(?:ya\.niigat|\.nar)|e\.yamagat)a|a(?:mishirasato\.chiba|rai\.ibaraki))\.jp|f(?:f(?:ic(?:e(?:-on-the\.net)?|ial\.academy)|\.ai)?|unato\.iwate\.jp|\.(?:by|no))|u(?:m(?:u\.hokkaido\.jp|\.gov\.pl)|(?:chi\.sag|da\.nar)a\.jp|tsystemscloud\.com)|w(?:n(?:(?:provider\.co|\.p)m|ip\.net)|a(?:riasahi\.aich|ni\.aomor)i\.jp)|d(?:a(?:wara\.kanagaw|te\.akit)a\.jp|(?:es?sa)?\.ua|da\.no|o\.br)|z(?:(?:u\.(?:kumamoto|ehime)|ora\.hokkaido)\.jp|\.au)|c(?:eanographi(?:que|c)\.museum|hi\.kochi\.jp)|o(?:(?:shik|kuw)a\.nagano\.jp|guy\.com|o)|v(?:(?:re-eiker|erhalla)\.no|h)|x(?:ford\.museum|\.rs)|\.(?:bg|se))|f(?:u(?:(?:k(?:u(?:(?:(?:yama\.hiroshi|mitsu\.toya)m|(?:roi\.shizu)?ok|domi\.sag)a|s(?:hima(?:\.(?:fukushima|hokkaido))?|aki\.hyogo)|chi(?:yama\.kyoto|\.fukuoka)|(?:i\.fuku)?i)|a(?:gawa\.hokkaido|ya\.saitama))|chu\.(?:to(?:yama|kyo)|hiroshima)|dai\.iwate)\.jp|ji(?:(?:(?:(?:(?:(?:nomiy|ed)a)?\.shizuo|idera\.osa)k|oka\.gunm)a|s(?:a(?:wa\.(?:kanagawa|iwate)|to\.akita)|hiro\.ibaraki)|kawa(?:\.(?:yamanashi|shizuoka)|guchiko\.yamanashi)|mi(?:\.(?:saitama|nagano)|no\.saitama)|yoshida\.yamanashi)\.jp|xerox|tsu)|r(?:(?:u(?:dono\.fukushima|bira\.hokkaido|kawa\.miyagi)|ano\.hokkaido)\.jp|niture(?:\.museum)?)|t(?:(?:(?:aba\.fukushim|tsu\.chib)a|su\.nagasaki)\.jp|ure(?:host|mail)ing\.at|bol)|n(?:a(?:gata\.yamagat|hashi\.toyam|bashi\.chib)a\.jp|d(?:acio\.museum)?)?|e(?:fuki\.yamanashi\.jp|ttertdasnetz\.de|l\.aero)|s(?:(?:sa\.tokyo|o\.aichi)\.jp|a\.no)|o(?:isku|ssko)\.no)|r(?:o(?:m(?:-(?:(?:i[adln]|w[aivy]|o[hkr]|[hr]i|d[ce]|k[sy]|p[ar]|s[cd]|t[nx]|v[at]|fl|ga|ut)\.com|m(?:[adinost]\.com|e\.org)|n(?:[cdehjmv]\.com|y\.net)|a(?:[klr]\.com|z\.net)|c(?:[at]\.com|o\.net)|la\.net)|\.hr)|g(?:\.museum|n\.no|ans)|s(?:inone\.it|ta\.no)|(?:land|ya)\.no|nt(?:doo|ie)r)|e(?:e(?:box(?:-os\.(?:com|fr)|os\.(?:com|fr))|d(?:dns\.(?:org|us)|esktop\.org)|(?:tls\.fastly\.ne|site\.hos)t|masonry\.museum)?|i(?:(?:ght\.aer|\.n)o|burg\.museum)|drikstad\.no|senius)|i(?:uli-?v(?:e(?:nezia)?)?-?giulia\.it|bourg\.museum)|an(?:(?:ziskaner|caise|kfurt)\.museum|a\.no)|\.(?:eu\.org|it)|(?:æn|øy)a\.no|l)?|i(?:r(?:e(?:wall-gateway\.(?:com|net|de)|baseapp\.com|nze\.it|stone)?|m(?:\.(?:[cr]o|dk|ht|in|nf|ve)|dale))|n(?:(?:earts?|land)\.museum|a(?:nc(?:ial|e)|l)|\.(?:ec|tn)|n[oø]y\.no)|l(?:m(?:\.(?:museum|hu))?|atelia\.museum|egear\.me)|e(?:ld\.museum|\.ee)|\.(?:eu\.org|cr|it)|t(?:jar\.no|ness)?|gueres\.museum|d(?:elity|o)|sh(?:ing)?|at)?|o(?:r(?:-(?:(?:(?:mor|som|th)e|better)\.biz|our\.info)|t(?:(?:missoula|worth)\.museum|al\.br)|got\.h(?:er|is)\.name|um(?:z\.info|\.hu)?|lì?-?cesena\.it|sa(?:nd\.no|le)|d(?:e\.no)?|ce\.museum|ex)|(?:l(?:kebib|lda)l|snes)\.no|o(?:d(?:network)?|tball)?|undation(?:\.museum)?|[tz]\.br|ggia\.it|x)?|a(?:r(?:m(?:e(?:quipment\.museum|rs(?:\.museum)?)|(?:stead)?\.museum)?|sund\.no|\.br)|s(?:t(?:(?:panel\.direc|lylb\.ne)t|vps-server\.com)?|hion)|m(?:ily(?:ds\.(?:com|net|org)|\.museum)?|\.pk)|n(?:tasyleague\.cc|s)?|i(?:rwinds|th|l)|uske\.no|ge)|l(?:o(?:r(?:i(?:da\.museum|pa\.br|st)|[aoø]\.no|ence\.it)|g\.br|wers)|a(?:(?:tanger|kstad)?\.no|nders\.museum)|y(?:nnh(?:osting\.net|ub\.com))?|(?:e(?:kkefjord|sberg)|å)\.no|i(?:ght(?:\.aero|s)|(?:ck)?r)|t\.cloud\.muni\.cz|\.us)|e(?:d(?:ora(?:infracloud|people)\.org|e(?:ration\.aero|x)|je\.no|\.us)|r(?:r(?:ar(?:a\.it|i)|ero)|mo\.it)|(?:ste-ip\.ne|\.i)t|t(?:sund)?\.no|ira\.br|edback)|h(?:s(?:k\.se|\.no)|app\.xyz|v?\.se)|y(?:(?:lkesbib|resda)l\.no|i)|(?:(?:nd|st)\.b|bx-?os\.f)r|j(?:(?:aler|ell)\.no|\.cn)|m(?:\.(?:br|it|no))?|t(?:paccess\.cc|r)|(?:v?g|c)\.it|\.(?:bg|se)|ørde\.no)|g(?:[fhpqt]|o(?:v(?:\.(?:m[aegklnorsuvwyz]|n(?:(?:c\.t)?r|g)|b[abfhmnrstyz]|s[abcdghltxy]|a[ceflrsuz]|c[dlmnouxy]|k[gimnpwyz]|l[abckrtvy]|p[hklnrsty]|t[jlmnortw]|g[ehinruy]|i[elnqrst]|e[cegt]|d[moz]|r[suw]|v[cen]|z[amw]|u[ak]|hk|jo|om|qa|ws)|ernment\.aero|t\.nz)?|t(?:(?:emba\.shizuoka|o\.nagasaki|su\.shimane)\.jp|dns\.(?:c(?:om|h)|org)|pantheon\.com)?|\.(?:(?:dyndns\.or|u)g|(?:gov|leg)\.br|t[hjz]|c[ir]|i[dt]|k[er]|jp|pw)|b(?:\.(?:p[aek]|[bd]o|e[cs]|ar|cl|gt|hn|mx|ni|sv|ve)|o\.wakayama\.jp)|o(?:g(?:le(?:(?:apis|code)\.com)?)?|d(?:hands|year))?|s(?:(?:e(?:n\.niigat|\.nar)a|hiki\.hyogo)\.jp|\.pk)|l(?:f(?:fan\.us)?|d(?:point)?|\.no)|r(?:ge\.museum|izia\.it|lice\.pl)|uv\.(?:bj|ci|fr|ht|km|ml|rw|sn)|k(?:ase\.miyazaki\.jp|\.pk)|n(?:ohe\.aomori\.jp|\.pk)|d(?:o\.gifu\.jp|addy)|i(?:ania\.br|p\.de)|jome\.akita\.jp|p(?:\.pk)?)|r(?:o(?:u(?:ndhandling\.aero|p(?:\.aero)?)|ks-th(?:is|e)\.info|n(?:dar\.za|g\.no)|zny\.[rs]u|sseto\.it|cery)|a(?:n(?:drapids\.museum|(?:vin|e)?\.no)|t(?:angen\.no|is)|z\.museum|jewo\.pl|inger|phics)|i(?:w\.gov\.pl|mstad\.no|pe)|\.(?:eu\.org|com|it|jp)|u(?:e\.no|\.br)|e(?:ta\.fr|en)|p\.lk)?|e(?:o(?:rg(?:ia\.(?:museum|su)|e)|metre-expert\.fr|logy\.museum)|n(?:\.(?:mi\.us|in|nz|tr)|kai\.saga\.jp|t(?:ing)?|ov?a\.it)|e(?:k(?:galaxy\.com|\.nz)|lvinck\.museum)|t(?:myip\.com|s-it\.net)|mological\.museum|isei\.kochi\.jp|\.it|a)?|a(?:m(?:e(?:-(?:server\.cc|host\.org)|s(?:\.hu)?|\.tw)?|(?:agori\.aichi|o\.shiga)\.jp|vik\.no)|l(?:l(?:ery(?:\.museum)?|up|o)|sa\.no)?|(?:u(?:sdal|lar)|ivuotna)\.no|ng(?:aviika\.no|won\.kr)|rden(?:\.museum)?|teway\.museum|\.us|p)?|i(?:t(?:hub(?:usercontent\.com|\.io)|-repos\.de|lab\.io)|n(?:o(?:wan|za)\.okinawa|an\.gifu)\.jp|e(?:htavuoatna\.no|ssen\.museum)|f(?:(?:u\.gif)?u\.jp|ts?)|(?:ldesk[aå]l|ske)\.no|v(?:ing|es)|ize\.com)?|l(?:o(?:b(?:al(?:\.(?:prod|ssl)\.fastly\.net)?|o)|ppen\.no|gow\.pl)|a(?:s(?:s(?:\.museum)?|\.museum)|de)|i(?:ding\.aero|wice\.pl)|e(?:eze\.com)?|ug\.org\.uk)?|u(?:(?:(?:shikami\.okinaw|nm)a|jo\.gifu)\.jp|(?:ovdageaidnu|len)\.no|a(?:m\.gu|rdian)|ernsey\.museum|i(?:tars|de)|b\.uy|\.us|cci|ge|ru)?|s(?:\.(?:(?:s(?:[ft]|valbard)|o(?:[fl]|slo)|jan-mayen|a[ah]|h[lm]|n[lt]|t[mr]|v[af]|bu|fm|mr|rl)\.no|cn)|m\.pl)?|(?:j(?:e(?:r(?:drum|stad)|mnes|sdal)|[oø]vik)|á(?:(?:ŋgaviik|ivuotn)a|lsá))\.no|y(?:eong(?:buk|nam|gi)\.kr|okuto\.kumamoto\.jp)?|d(?:(?:a(?:nsk)?|ynia)\.pl|(?:\.c)?n)?|w(?:iddle\.co\.uk|angju\.kr)?|m(?:[ox]|(?:ina\.p|ai)l|bh)?|b(?:\.(?:com|net)|iz)?|n(?:iezno\.pl)?|g(?:f\.br|ee)?|\.(?:bg|se)|[xz]\.cn|v\.a[ot]|12\.br|c\.ca)|p(?:a(?:r(?:is(?:\.(?:eu\.org|museum))?|a(?:chut|glid)ing\.aero|t(?:(?:ner)?s|i\.se|y)|(?:och\.k12\.ma\.u)?s|liament\.(?:cy|nz)|ma\.it)|l(?:m(?:springs\.museum|as\.br)|e(?:o\.museum|rmo\.it)|ace\.museum)|s(?:s(?:enger-association\.aero|agens)|adena\.museum)|n(?:a(?:ma\.museum|sonic)|theonsite\.io|erai)|ge(?:(?:speedmobilizer|frontapp)\.com)?|\.(?:gov\.(?:br|pl)|leg\.br|it|us)|d(?:erborn\.museum|(?:ov|u)a\.it)|cific\.museum|tria\.bo|via\.it|y)?|r(?:o(?:\.(?:[bp]r|[ht]t|az|cy|ec|mv|na|om|vn)|d(?:uction(?:\.aero|s))?|f(?:esional\.bo|\.pr)?|t(?:onet\.io|ection)|pert(?:ies|y)|ject\.museum|chowice\.pl|gressive|mo)?|es(?:s(?:\.(?:m(?:useum|a)|aero|cy|se)|e\.(?:ci|fr|km|ml))?|(?:ervation|idio)\.museum)|i(?:v(?:atizehealthinsurance\.net|\.(?:at|hu|me|no|pl))|(?:\.e|m)e|ncipe\.st)|\.(?:(?:gov|leg)\.br|it|us)|u(?:(?:szkow\.p|dentia)l)?|a(?:merica|to\.it|xi)|d\.(?:fr|km|mg)|zeworsk\.pl)?|o(?:r(?:t(?:(?:l(?:ligat|and)|al)\.museum|\.fr)|s(?:ang(?:er|u)|grunn|áŋgu)\.no|denone\.it|n)|l(?:i(?:ti(?:ca\.bo|e)|ce\.uk)|\.(?:dz|ht|tr)|kowice\.pl|tava\.ua)|d(?:(?:lasi|hal)e\.pl|zone\.(?:net|org))|st(?:s-and-telecommunications\.museum)?|i(?:nt(?:2this\.com|to\.us)|vron\.org)|(?:(?:mor(?:ski|z)e|wiat|znan)\.p|h)l|t(?:ager\.org|enza\.it)|k(?:rovsk\.su|er)|\.(?:gov\.pl|it)|npes\.id|a\.br)|i(?:(?:e(?:dmont|monte)\.i|a(?:cenza\.i|ge))t|l(?:ot(?:s\.museum|\.aero)|a\.pl)|(?:ttsburgh\.museu|xolino\.co)m|\.(?:(?:gov|leg)\.br|it)|s(?:(?:toi)?a\.it|z\.pl)|n(?:[gk]|b\.gov\.pl)?|c(?:t(?:ures|et)|s)|ppu\.hokkaido\.jp|mienta\.org|w\.gov\.pl|oneer|zza|d)|h(?:o(?:to(?:graphy(?:\.museum)?|s)?|enix\.museum|ne)|il(?:a(?:delphi(?:aare)?a|tely)\.museum|ips)|armac(?:ien(?:s\.km|\.fr)|y(?:\.museum)?)|ysio|d)?|l(?:a(?:(?:n(?:t(?:ation|s)|etarium)|za)\.museum|y(?:station)?|ce)|u(?:rinacional\.bo|mbing|s)|c\.(?:co\.im|ly|uk)|\.(?:eu\.org|ua)|o\.ps)?|e(?:r(?:so\.(?:[st]n|ht)|\.(?:la|nf|sg)|ugia\.it)|\.(?:(?:(?:gov|leg)\.b|k)r|ca|it)|(?:s(?:aro-?urbino|cara)\.i)?t|nza\.su)?|u(?:b(?:l(?:i(?:shproxy\.co|c\.museu)m|\.pt)|ol\.museum|\.sa)?|(?:p\.gov|lawy)\.pl|(?:g(?:lia)?)?\.it|eblo\.bo)|v(?:t\.(?:k12\.ma\.us|ge)|h\.br|\.it)|s(?:(?:se|p)\.gov\.pl|[ci]\.br)?|t(?:\.(?:eu\.org|it)|plus\.fit)?|g(?:(?:afan\.ne|\.i)t|fog\.com)|c(?:\.(?:it|pl)|loud\.host|cw)|p(?:\.(?:az|ru|se|ua)|g\.br)|b\.(?:(?:gov|leg)\.br|ao)|y(?:atigorsk\.ru)?|n(?:\.it|c)?|\.(?:bg|se)|m(?:n\.it)?|f(?:izer)?|[dz]\.it|wc?|k)|d(?:[jmz]|e(?:v(?:ices\.resinstaging\.io|-myqnapcloud\.com|\.static\.land|elopment\.run)?|l(?:(?:menhorst|aware)\.museum|l(?:-?ogliastra\.it)?|ivery|oitte|ta)|s(?:i(?:gn(?:\.(?:museum|aero))?)?|a\.id)|f(?:inima\.(?:net|io)|ense\.tn|\.br)|p(?:o(?:t\.museum|rte\.bo)|\.no)|(?:corativearts|troit)\.museum|n(?:mark\.museum|t(?:ist|al))|mo(?:cra(?:cia\.bo|t)|n\.nl)|\.(?:co(?:ol|m)|eu\.org|us)|a(?:l(?:er|s)?|tnu\.no)|bian\.net|dyn\.io|gree)?|y(?:n(?:dns(?:-(?:(?:offic|remot|fre|hom)e|at-(?:home|work)|w(?:iki|ork|eb)|server|blog|mail|pics|ip)\.com|\.(?:ddnss\.de|info|biz|org|tv|ws)|1\.de)|a(?:mi(?:sches-dns\.de|c-dns\.info)|lias\.(?:com|net|org)|thome\.net)|\.(?:(?:cosidn|ddns)s|home-webserver)\.de|-(?:(?:ip24|vpn)\.de|o-saur\.com)|v(?:6\.net|pn\.de)|serv\.org|ns\.com|u\.net)|r[oø]y\.no|\.fi)|a(?:[dy]|t(?:to(?:(?:relay|web)\.com|local\.(?:com|net))|e(?:\.(?:fukushima|hokkaido)\.jp)?|a(?:base\.museum)?|ing|sun)|(?:i(?:(?:wa\.hiroshim|sen\.akit|to\.osak)a|go\.ibaraki)|zaifu\.fukuoka)\.jp|(?:l(?:las|i)\.museu|mnserver\.co)m|vve(?:nj[aá]rg|siid)a\.no|(?:e(?:jeon|gu)\.k|bu)r|(?:plie\.m|nc)e|gestan\.[rs]u)|o(?:[gt]|n(?:texist\.(?:com|net|org)|ostia\.museum|etsk\.ua|na\.no)|es(?:ntexist\.(?:com|org)|-it\.net)|m(?:inic\.ua|ains)|omdns\.(?:com|org)|shi\.yamanashi\.jp|lls\.museum|c(?:tor|s)|vre\.no|wnload|dge|ha)?|i(?:s(?:co(?:ver(?:y\.museum)?|unt)|kstation\.(?:org|eu|me)|h)|(?:nosaur\.museu|tchyourip\.co)m|vt(?:tasvuot|asvuod)na\.no|e(?:lddanuorri\.no|t)|rect(?:ory)?|amonds|gital|y)|n(?:s(?:alias\.(?:com|net|org)|dojo\.(?:com|net|org)|up(?:dater\.de|\.net)|(?:home\.d|for\.m)e|iskinky\.com|king\.ch)|i(?:propetrovsk\.ua|\.us)|(?:epropetrovsk)?\.ua|p)|r(?:a(?:y(?:d(?:dns\.com|ns\.de)|-dns\.de)|(?:ngedal|mmen)\.no)|eamhosters\.com|ud\.(?:io|us)|[oø]bak\.no|\.(?:na|tr)|ive)|d(?:ns(?:(?:(?:fre|liv)e|geek|king)\.com|s\.(?:org|de)|\.(?:net|me))|r\.museum|-dns\.de|s)|s(?:cloud\.(?:m(?:obi|e)|biz)|mynas\.(?:com|net|org)|t\.mi\.us)|u(?:r(?:ham\.museum|ban)|ck(?:dns\.org)?|n(?:lop|s)|pont|bai)|v(?:r(?:cam\.info|dns\.org)?|ag)|(?:gca\.aer|ønna\.n)o|(?:lugoleka\.p|h)l|f\.(?:gov|leg)\.br|k(?:\.eu\.org)?|c(?:\.us|lk)|\.(?:bg|se)|p\.ua|tv)|l(?:[bkr]|i(?:b\.(?:(?:m[adeinost]|n[cdehjmvy]|a[klrsz]|i[adln]|c[aot]|o[hkr]|v[ait]|w[aiy]|[hr]i|d[ce]|g[au]|k[sy]|p[ar]|s[cd]|t[nx]|fl|la|ut)\.us|ee)|n(?:k(?:yard(?:-cloud\.ch|\.cloud)|itools\.space)?|d(?:e(?:snes\.no)?|[aå]s\.no)|coln(?:\.museum)?|z\.museum)|m(?:a(?:-city\.(?:rocks|at|ch|de)|nowa\.pl|\.zone)|ited|o)|v(?:ing(?:(?:history)?\.museum)?|orno\.it|e)|ll(?:e(?:hammer|sand)\.no|y)|ke(?:s(?:candy|-pie)\.com)?|fe(?:(?:insuranc|styl)e)?|g(?:(?:uria)?\.it|hting)|er(?:ne)?\.no|(?:xi|d)l|aison|\.it|psy)?|a(?:n(?:c(?:as(?:hire\.museum|ter)|ome|ia)|d(?:-4-sale\.us|es\.museum|rover)?|gev[aå]g\.no|s\.museum|bib\.se|xess)|(?:va(?:ngen|gis)|akesvuemie|hppi)\.no|(?:(?:-spezi|quil)a|z(?:io)?)\.it|r(?:(?:dal|vik)\.no|sson\.museum)|t(?:in(?:a\.it|o)|robe)?|(?:bou?r|jolla)\.museum|w(?:\.(?:pro|za)|yer)?|s(?:pezia\.it|alle)|m(?:borghini|er)|(?:dbroke|\.u)s|kas\.hu|py\.pl|caixa)?|o(?:(?:s(?:angeles\.museu|eyourip\.co)|(?:yalist|uvre)\.museu)m|c(?:alh(?:ost\.daplie\.me|istory\.museum)|ker|us)|g(?:i(?:stics\.aero|nto\.me)|oip\.(?:com|de))|m(?:bard(?:ia|y)\.it|\.(?:it|no)|za\.pl)|nd(?:on(?:\.museum)?|rina\.br)|(?:renskog|ppa)\.no|a(?:b[aá]t\.no|ns?)|di(?:ngen\.no|\.it)|t(?:en\.no|t[eo])|(?:wicz\.p)?l|(?:\.i|f)t|ve)|e(?:i(?:(?:r(?:fjord|vik)|kanger)\.no|tungsen\.de)|a(?:s(?:ing\.aero|e)|[nŋ]gaviika\.no)|(?:k(?:svik|a)|vanger|rdal|sja)\.no|b(?:timnetz\.de|esby\.no|ork\.pl)|g(?:(?:nica\.p|a)l|\.br|o)|c(?:c[eo]\.it|lerc)|n(?:vik\.no|ug\.su)|wismiller\.museum|zajsk\.pl|l\.br|\.it|frak|xus)|u(?:c(?:(?:ani|c)a\.it|erne\.museum)|x(?:e(?:mbourg\.museum)?|ury)|n(?:d(?:\.no|beck)|ner\.no)|g(?:s?\.org\.uk|ansk\.ua)|(?:r[oø]y|ster)\.no|\.(?:eu\.org|it)|(?:bin|kow)\.pl|zern\.museum|tsk\.ua|pin)?|t(?:d(?:\.(?:c(?:o\.im|y)|[hl]k|u[ak]|gi)|a)?|\.(?:eu\.org|it|ua))?|(?:ø(?:(?:ding|t)en|renskog)|áhppi|ærdal)\.no|v(?:\.(?:eu\.org|ua)|iv\.ua)?|c(?:ube-server\.de|\.it)?|y(?:ng(?:dal|en)\.no)?|g(?:\.(?:jp|ua)|bt)|p(?:lfinancia)?l|-o-g-i-n\.de|\.(?:bg|se)|äns\.museum|n\.cn|d?s|lc)|e(?:d(?:u(?:\.(?:e(?:[cest]|(?:u\.or)?g)|k(?:[gimnpwyz]|rd)|m[egklnostvwxyz]|b[abhimnorstz]|p[aefhklnrsty]|g[ehilnprtuy]|s[abcdglntvy]|l[abckrvy]|a[cflruz]|t[jmortw]|c[inouw]|i[nqst]|v[cenu]|d[moz]|h[knt]|n[gir]|r[suw]|u[ay]|z[am]|jo|om|qa|ws)|cat(?:ion(?:(?:al)?\.museum)?|or\.aero)|net\.tn)?|\.(?:c[ir]|ao|jp|pw)|ogawa\.tokyo\.jp|eka)|n(?:g(?:ine(?:er(?:\.aero|ing)?|\.aero)|\.(?:pro|br)|land\.museum|erdal\.no)|(?:vironment(?:alconservation)?|cyclopedic)\.museum|t(?:er(?:tainment\.aero|prises)|omology\.museum)|dof(?:internet\.(?:net|org)|theinternet\.org)|(?:iwa\.hokkaido|a\.gifu)\.jp|e(?:bakk\.no|rgy)|(?:na)?\.it|onic\.io|s\.tn)|s(?:t(?:-(?:(?:a-la-ma(?:is|si)|le-patr)on|mon-blogueur)\.com|ate(?:\.museum)?|\.pr)|\.(?:(?:(?:gov|leg)\.b|k)r|eu\.org)|a(?:shi|n)\.hokkaido\.jp|sex\.museum|urance|p\.br|q)?|m(?:b(?:etsu\.hokkaido\.jp|roidery\.museum|aixada\.st|\.kw)|(?:ilia-?romagna|r)\.it|er(?:gency\.aero|ck)|p(?:resa\.bo|\.br)|ail)|u(?:-(?:(?:west-[123]|central-1)\.elasticbeanstalk|[1234]\.evennode)\.com|\.(?:(?:meteorapp\.)?com|int|org)|rovision|n\.eg|s)?|x(?:p(?:ert(?:s-comptables\.fr)?|ress(?:\.aero)?|osed)|(?:hibition|eter)\.museum|change(?:\.aero)?|traspace|net\.su)|a(?:st(?:(?:africa|coast)\.museum|-kazakhstan\.su)|t(?:ing-organic\.net|on\.mi\.us)?|rth)|i(?:(?:d(?:s(?:(?:ber|ko)g|voll)|fjord)?|gersund)\.no|heiji\.fukui\.jp|senbahn\.museum)|l(?:ve(?:ndrell\.museum|rum\.no)|b(?:urg\.museum|lag\.pl)|asticbeanstalk\.com|k\.pl)|t(?:ajima\.hiroshima\.jp|hnology\.museum|i(?:salat|\.br)|ne(?:dal)?\.no|c\.br)?|b(?:i(?:n(?:a\.kanagawa|o\.miyazaki)\.jp|z\.tw)|etsu\.hokkaido\.jp)|v(?:e(?:n(?:(?:(?:ass|ášš)i|es)\.no|ts)|rbank)|je-og-hornnes\.no)|c(?:o(?:(?:log|nom)ia\.bo|\.br)?|hizen\.fukui\.jp|n\.br)?|r(?:i(?:mo\.hokkaido\.jp|csson)|oti[ck]a\.hu|ni)|g(?:yptian\.museum|ersund\.no)?|p(?:ilepsy\.museum|ost|son)|1(?:64\.arpa|2\.ve)|quipment(?:\.aero)?|e(?:\.eu\.org)?|\.(?:bg|se)|kloges\.cy|hime\.jp|4\.cz)|r(?:e(?:s(?:i(?:stance\.museum|ndevice\.io)|earch\.(?:museum|aero)|\.(?:aero|in)|(?:tauran)?t)|a(?:l(?:estate(?:\.pl)?|t(?:or|y)|m\.cz)|d(?:(?:-books|myblog)\.org)?)|c(?:(?:reation\.ae|ht\.p)ro|\.(?:[cr]o|br|nf|ve)|i(?:fe\.br|pes))|n(?:(?:ne(?:s[oø]y|bu)|dalen)\.no|t(?:als)?)?|d(?:irectme\.net|umbrella|stone|\.sv)?|g(?:gio-?(?:calabr|emil)ia\.it|\.dk)|p(?:body\.aero|ublican|\.kp|air|ort)|l(?:\.(?:ht|pl)|iance)|vi(?:sta\.bo|ews?)|bun\.hokkaido\.jp|\.(?:it|kr)|i(?:sen?|t)|motewd\.com|klam\.hu|xroth|hab)?|i(?:(?:(?:ku(?:zentakata\.iwate|betsu\.hokkaido)|fu\.miyagi|tto\.shiga)\.j)?p|s(?:hir(?:ifuj)?i\.hokkaido\.jp|(?:[oø]r|sa)\.no)|o(?:(?:(?:branc|pret)o)?\.br|dejaneiro\.museum)?|n(?:g(?:e(?:rike|bu)|saker)|dal)\.no|(?:ghtathom|ik\.e)e|c(?:h(?:ardli)?|oh)|(?:min|et)i\.it|\.(?:it|us)|beirao\.br|vne\.ua|l)|a(?:n(?:(?:koshi\.hokkaido|zan\.saitama)\.jp|(?:daberg|a)\.no)|d(?:o(?:m\.pl|y\.no)|(?:øy|e)\.no|io(?:\.br)?)|(?:h(?:kkeravju|olt)|kkestad|lingen|uma)\.no|ven(?:db\.(?:community|run|me)|na\.it)|i(?:l(?:road|way)\.museum|sa\.no|d)|c(?:kmaze\.(?:com|net)|ing)|(?:gusa)?\.it|wa-maz\.pl|s\.ru)|o(?:(?:(?:y(?:rvik|ken)|llag|ros|an|st)\.n|torcraft\.aer|d(?:oy\.n|e))o|c(?:he(?:ster\.museum|r)|k(?:art\.museum|s))|m(?:a\.(?:museum|it)|s(?:kog|a)\.no|e\.it)|\.(?:(?:gov|leg)\.br|eu\.org|i[mt])|v(?:igo\.it|no\.ua)|kunohe\.aomori\.jp|uter\.management|gers|om)?|(?:ø(?:y(?:rvik|ken)|mskog|døy|ros|st)|á(?:hkkerávju|isa)|å(?:holt|de)|ælingen|l)\.no|y(?:u(?:(?:gasaki\.ibaraki|oh\.shiga)\.jp|kyu)|okami\.saitama\.jp|bnik\.pl|gge\.no)|u(?:\.(?:eu\.org|com|net)|ssia\.museum|ovat\.no|gby|hr|n)?|n(?:\.(?:(?:gov|leg)\.br|it)|(?:[su]|rt)\.tn)|s(?:\.(?:gov|leg)\.br|c\.cdn77\.org|vp)?|\.(?:cdn77\.net|bg|se)|z(?:gw\.gov|eszow)\.pl|[jr]\.(?:gov|leg)\.br|(?:[cg]\.|m\.?)it|hcloud\.com|v\.ua|we?)|y(?:a(?:m(?:a(?:(?:t(?:o(?:\.(?:k(?:anagawa|umamoto)|fukushima)|(?:koriyam|takad)a\.nara)|suri\.fukushima)|n(?:a(?:(?:shi\.yamana)?|kako\.yamana)shi|o(?:be\.yamagata|uchi\.nagano))|g(?:a(?:ta(?:\.(?:yamagata|ibaraki|nagano|gifu))?|\.kumamoto)|uchi)|da\.(?:(?:fukuok|toyam)a|iwate)|(?:kita\.kanagaw|zoe\.nar)a|moto\.miyagi|shina\.kyoto)\.jp|xun)|e\.fukuoka\.jp)|(?:s(?:u(?:gi\.shimane|oka\.nagano|da\.kochi|\.shiga)|hi(?:o\.saitama|ro\.hyogo)|aka\.nagano)|t(?:su(?:shiro\.kumamoto|ka\.shimane)|omi\.aichi)|k(?:umo\.(?:hokkaido|shimane)|age\.okayama)|wa(?:ta(?:hama\.ehime|\.kyoto)|ra\.ibaraki)|i(?:zu\.shizuoka|ta\.tochigi)|bu(?:ki\.fukushima|\.hyogo)|o(?:tsu\.gifu|\.osaka)|ese\.okinawa|zu\.tottori)\.jp|ch(?:i(?:yo\.(?:ibaraki|chiba)|mata\.chiba)\.jp|ts)|n(?:a(?:izu\.fukushim|gawa\.fukuok)a\.jp|dex)|h(?:(?:iko\.niigata|aba\.iwate)\.jp|oo)|lta\.ua)|o(?:(?:n(?:a(?:g(?:uni\.okinawa|o\.tottori)|baru\.okinawa)|(?:ezawa\.yamagat|o\.saitam)a)|i(?:chi\.hokkaido|ta\.niigata)|tsukaido\.chiba)\.jp|k(?:o(?:(?:s(?:hibahikari\.chib|uka\.kanagaw)|ze\.saitam|te\.akit)a\.jp|hama)|(?:a(?:(?:wa)?\.hyogo|ichiba\.chiba)|kaichi\.mie)\.jp)|s(?:hi(?:(?:(?:kawa|mi)\.saita|oka\.gun)m|da\.s(?:hizuok|aitam)|no(?:gari\.sag|\.nar))a\.jp|emite\.museum)|r(?:(?:ii\.saitama|o\.gifu)\.jp|k(?:shire)?\.museum)|m(?:itan\.okinawa\.jp|bo\.me)|u(?:t(?:h\.museum|ube))?|lasite\.com|dobashi|ga)|u(?:(?:(?:(?:za(?:wa\.nii|\.yama)ga|fu\.oi)t|gawa(?:ra\.kanagaw|\.fukushim)|r(?:ihonjo\.akit|a\.wakayam)|asa\.wakayam)a|k(?:uhashi\.fukuoka|i\.ibaraki)|su(?:i\.kagoshima|hara\.kochi)|u\.yamaguchi)\.jp|n)|bo\.(?:(?:scienc|trad)e|review|faith|party)|\.(?:bg|se)|k\.ca|n\.cn|t)|u(?:s(?:(?:-(?:east-(?:1\.(?:elasticbeanstalk|amazonaws)|2\.elasticbeanstalk)|(?:gov-west-1|west-[12])\.elasticbeanstalk|[1234]\.evennode)\.co|(?:c(?:ountryestat|ultur)e|decorativearts|livinghistory|garden)\.museu)m|h(?:i(?:ku\.ibaraki\.jp|story\.museum)|uaia\.museum)|a(?:(?:ntique|rt)s\.museum|\.(?:oita\.jp|museum))|\.(?:(?:eu\.)?org|gov\.pl|com|na)|u(?:i\.fukuok|ki\.oit)a\.jp|r\.cloud\.muni\.cz|er\.party\.eus|tka\.pl)?|n(?:(?:azuki\.toyama|zen\.nagasaki|nan\.shimane)\.jp|i(?:v(?:ersity(?:\.museum)?|\.sn)|on\.aero|com)|(?:usualperson\.co|dersea\.museu)m|(?:j[aá]rga\.n)?o)|r(?:(?:a(?:(?:soe\.okinaw|wa\.saitam|yasu\.chib)a|(?:kawa|usu)\.hokkaido)|uma\.okinawa|yu\.hokkaido|eshino\.mie)\.jp|bino-?pesaro\.it|[in]\.arpa|l\.tw)|t(?:a(?:z(?:u\.kagawa\.jp|as\.hu)|shinai\.hokkaido\.jp|h\.museum)|s(?:unomiya\.tochigi\.jp|ira\.no)|o\.kumamoto\.jp|wente\.io|\.us)|(?:e(?:no(?:hara\.yamanashi|\.gunma)|da\.nagano)|ji(?:(?:tawara)?\.kyoto|ie\.tochigi))\.jp|c(?:hi(?:n(?:ada\.ishik|omi\.kag)awa|hara\.ibaraki|ko\.ehime)\.jp|onnect)|m(?:i(?:\.fukuoka\.jp|g\.gov\.pl)|aji\.kochi\.jp|b(?:ria)?\.it|\.gov\.pl)|k(?:i(?:ha\.fukuoka|\.kumamoto)\.jp|\.(?:eu\.org|com|net)|lugs\.org)?|l(?:(?:lens(?:aker|vang)|vik)\.no|m\.museum|san\.kr)|d(?:(?:ono\.mie|a\.nara)\.jp|i(?:ne\.it|\.br)|\.it)|(?:2(?:-local)?\.xnbay\.co|(?:hren|vic)\.museu)m|b(?:e(?:\.yamaguchi\.jp|r\.space)|ank|s)|o(?:(?:numa\.niigat|zu\.toyam)a\.jp|l)|z(?:(?:hgorod)?\.ua|s\.gov\.pl)?|w(?:ajima\.ehime\.jp|\.gov\.pl)|p(?:(?:ow|po)\.gov\.pl|s)|g(?:(?:im)?\.gov\.pl)?|\.(?:bg|se)|y(?:\.com)?|fcfan\.org|a)|v(?:i(?:r(?:tu(?:e(?:eldomein\.nl|l\.museum)|al(?:-?user\.de|\.museum))|gin(?:ia\.museum)?)|c(?:(?:\.(?:edu|gov))?\.au|enza\.it)|n(?:n(?:ytsi|ic)a\.ua|dafjord\.no)?|k(?:ing(?:\.museum)?|(?:na)?\.no)|(?:bo-?valentia|terbo)\.it|s(?:ta(?:print)?|ion|a)|lla(?:ge\.museum|s)|p(?:sinaapp\.com)?|deo(?:\.hu)?|\.(?:it|us)|v[ao]|x\.br|ajes|g)?|a(?:l(?:le(?:(?:(?:-(?:d-?)?|d-?)?aosta|́?e(?:(?:-d)?-|d)?aoste)\.it|y\.museum|\.no)|er\.(?:hedmark|ostfold)\.no|-?d-?aosta\.it)|n(?:g(?:\.no|uard)|taa\.museum|ylven\.no|a)|(?:g(?:an?|soy)|ds[oø]|apste|ksdal)\.no|r(?:(?:d[oø]|ggat|oy)\.no|ese\.it)|por(?:cloud\.io|\.cloud)|\.(?:it|no|us)|cations|o\.it)?|e(?:r(?:s(?:ailles\.museum|icherung)|(?:(?:bani|on)a|celli)\.it|mögensberat(?:ung|er)|(?:dal|ran)\.no|isign)|(?:st(?:(?:v(?:ago|ågø)|b)y|re-(?:slidre|toten)|nes)|velstad|fsn)\.no|n(?:(?:e(?:zia|to)|ice)?\.it|nesla\.no|tures)|g(?:a(?:(?:rshei)?\.no|s)|årshei\.no)|t(?:erinaire\.(?:fr|km)|\.br)?|\.it)?|o(?:l(?:k(?:enkunde\.museum|swagen)|(?:da\.n|v)o|ogda\.su|yn\.ua)|(?:ss(?:evangen)?|agat)\.no|t(?:[eo]|ing)|yage|dka)|(?:(?:å(?:ler\.(?:hedmark|østfold)|g(?:søy|an|å))|árggát|ærøy|f)\.n|-info\.inf)o|l(?:a(?:di(?:kavkaz|mir)\.[rs]u|anderen(?:\.museum)?)|og\.br)|pn(?:dns\.net|plus\.to)|(?:[brsv]|da)\.it|t\.(?:it|us)|g(?:s\.no)?|c(?:\.it)?|n(?:\.ua)?|u(?:elos)?|\.bg)|w(?:a(?:(?:k(?:a(?:sa\.(?:tottor|fuku)i|(?:yama\.waka)?yama)|kanai\.hokkaido|uya\.miyagi|e\.okayama)|ji(?:ki\.tokushim|ma\.ishikaw)a|(?:zuka\.kyot|da\.nagan)o)\.jp|t(?:ch(?:(?:-and-|and)clock\.museum|es)?|ar(?:i\.miyagi|ai\.mie)\.jp)|s(?:h(?:ingtondc\.museum|tenaw\.mi\.us)|samu\.hokkaido\.jp)|r(?:abi\.saitama\.jp|m(?:ia\.pl|an)|szawa\.pl|\.museum)|l(?:es(?:\.museum)?|lonie\.museum|brzych\.pl|mart|ter)|n(?:ouchi\.gifu\.jp|g(?:gou)?)|\.(?:(?:(?:edu|gov)\.)?au|us)|w\.pl)|e(?:b(?:\.(?:[bcd]o|[lp]k|n[fi]|t[jr]|gu|id|ve|za)|ho(?:p\.(?:info|biz|net|org|me)|sting\.be)|s(?:pace\.rocks|ite)|redirect\.org|cam|er)|d(?:eploy\.(?:io|me|sh)|ding)?|llbeingzone\.(?:co\.uk|eu)|st(?:fale|er)n\.museum|ather(?:channel)?|i(?:bo|r)|grow\.pl|\.bs)|i(?:l(?:liam(?:sburg\.museum|hill)|dlife\.museum)|n(?:d(?:mill\.museum|ows)|b\.gov\.pl|ners|e)?|t(?:h(?:youtub|googl)e\.com|d\.gov\.pl)|(?:[fw]|ih|os)\.gov\.pl|e(?:lun\.pl|n)|ki(?:\.b[or])?|\.us)|o(?:r(?:k(?:i(?:nggroup\.aero|sboring\.com)|s(?:hop\.museum|\.aero)?)?|se-than\.tv|ld)|l(?:terskluwer|omin\.pl)|dzislaw\.pl|odside|w)|(?:(?:zmiuw|uoz)\.gov|locl(?:awek)?)\.pl|r(?:itesthisblog\.com|oc(?:law)?\.pl)|s(?:(?:kr|a)\.gov\.pl|\.na)?|h(?:aling\.museum|oswho)|m(?:flabs\.org|e)|pdevcloud\.com|\.(?:bg|se)|[vy]\.us|ww\.ro|t[cf]|f)|j(?:o(?:(?:(?:etsu\.niigat|hana\.toyam)a|so\.ibaraki)\.jp|urnal(?:is(?:m\.museum|t\.aero)|\.aero)|b(?:oji\.iwate\.jp|s(?:\.tt)?|urg)|r(?:peland\.no|\.br)|(?:lster|ndal)\.no|y(?:o\.kyoto\.jp)?|inville\.br|gasz\.hu|t)?|e(?:w(?:ish(?:art)?\.museum|elry(?:\.museum)?)|(?:fferson|rusalem)\.museum|(?:on(?:buk|nam)|ju)\.kr|(?:ssheim|vnaker)\.no|lenia-gora\.pl|tzt|ep)?|u(?:(?:d(?:ygarland|aica)|if)\.museum|e(?:disches\.museum|gos)|(?:nipe|s\.b)r|r\.pro)|a(?:m(?:ison\.museum|byl\.su|pa\.br)|(?:b\.b|gua)r|n-mayen\.no|worzno\.pl|va)|p(?:\.(?:eu\.org|net)|morgan|n\.com|rs)?|i(?:nsekikogen\.hiroshima\.jp|o)|d(?:evcloud\.com|f\.br)|ø(?:rpeland|lster)\.no|l(?:[cl]|\.cn)|s\.(?:org|cn)|fk\.museum|gora\.pl|c[bp]|x\.cn|\.bg|mp|nj)|[^\.]+\.(?:s(?:en(?:siosite\.cloud|dai\.jp)|t(?:atics\.cloud|olos\.io)|pectrum\.myjino\.ru|apporo\.jp|5y\.io|ch\.uk)|c(?:ompute(?:\.(?:amazonaws\.com(?:\.cn)?|estate)|-1\.amazonaws\.com)|ns\.joyent\.com|ryptonomic\.net|k)|e(?:x\.(?:futurecms|ortsinfo)\.at|lb\.amazonaws\.com(?:\.cn)?|r)|k(?:(?:itakyushu|awasaki|obe)\.jp|unden\.ortsinfo\.at|h)|a(?:lces\.network|dvisor\.ws|wdev\.ca)|tr(?:ansurl\.(?:be|eu|nl)|iton\.zone)|(?:(?:host|land)ing|vps)\.myjino\.ru|(?:(?:quipelements|0emm)\.co|j)m|p(?:latform(?:sh\.site|\.sh)|g)|n(?:(?:agoya\.j)?p|om\.br)|m(?:agentosite\.cloud|m)|f(?:[jk]|uturecms\.at)|y(?:okohama\.jp|e)|in\.futurecms\.at|uberspace\.de|otap\.co|bd)|z(?:[mw]|a(?:p(?:orizhzh(?:ia|e)\.ua|to\.(?:org|xyz)|pos)|(?:ma(?:mi\.okin|\.kanag)awa|o\.miyagi)\.jp|(?:chpomor|kopane|gan)\.pl|\.(?:com|net|org|bz)|r(?:ow\.pl|a))|o(?:olog(?:ical|y)\.museum|ne(?:\.id)?)|u(?:shi\.kanagawa\.jp|erich)|(?:h(?:itomi|ytomy)r|t)\.ua|e(?:ntsuji\.kagawa\.jp|ro)|gor(?:zelec|a)\.pl|p\.(?:gov\.pl|ua)|\.(?:bg|se)|ip(?:po)?|lg\.br|j\.cn)|(?:ø(?:y(?:stre-slidre|garden|er)|r(?:s(?:kog|ta)|land)|stre-toten|vre-eiker|ksnes)|å(?:l(?:(?:esun|går)d)?|s(?:eral|nes)?|m(?:li|ot)|krehamn|fjord|rdal)|á(?:l(?:aheadju|tá)|kŋoluokta)|čáhcesuolo)\.no|(?:(?:[富岡]|和歌)山|(?:[広徳]|鹿児)島|(?:神奈|石)川|山[口形梨]|福[井岡島]|[佐滋]賀|宮[城崎]|愛[媛知]|長[崎野]|北海道|三重|京都|兵庫|千葉|埼玉|奈良|岐阜|岩手|島根|東京|栃木|沖縄|熊本|秋田|群馬|茨城|青森|静岡|高知|鳥取)\.jp|q(?:u(?:e(?:bec(?:\.museum)?|st)|icksytes\.com)|ld(?:\.(?:edu|gov))?\.au|(?:-a\.eu\.or|\.b)g|(?:h\.c|po)n|a(?:2\.com)?|c\.c(?:om|a)|sl\.br|vc)|x(?:e(?:n(?:apponazure|\.prgmr)\.com|rox)|(?:i(?:hua)?|[jz]\.c)n|s4all\.space|\.(?:bg|se)|(?:bo|x)x|nbay\.com|443\.pw|finity|yz)|ا(?:ل(?:سعود(?:ي[ةه]|ی[ةۃ])|(?:عليا|ارد|يم)ن|جزائر|مغرب)|(?:تصال|مار)ات|يران(?:\.ir)?|یران(?:\.ir)?|بوظبي|رامكو)|1(?:2hp\.(?:at|ch|de)|337\.pictures|kapp\.com|6-b\.it|\.bg)|(?:(?:(?:องค์ก|ทหา)ร|ธุรกิจ|รัฐบาล|ศึกษา|เน็ต)\.)?ไทย|2(?:0(?:00\.hu|38\.io)|ix\.(?:at|ch|de)|\.bg)|о(?:(?:бр|д)\.срб|рг(?:\.срб)?|нлайн)|4(?:lima\.(?:at|ch|de)|u\.com|\.bg)|网(?:[址店站]|络(?:\.(?:cn|hk))?|絡\.hk)|3(?:utilities\.com|2-b\.it|\.bg)|網(?:絡\.(?:cn|hk|香港)|络\.hk|路\.tw)|م(?:و(?:بايلي|قع)|ليسيا|صر)|公(?:司(?:\.(?:cn|hk|香港))?|益)|組(?:織\.(?:hk|tw|香港)|织\.hk)|ب(?:ا(?:زار|رت)|ھارت|يتك)|(?:ירושלים|иком)\.museum|政(?:府(?:\.(?:hk|香港))?|务)|组(?:织(?:\.hk|机构)|織\.hk)|(?:پا[كک]ستا|فلسطي)ن|0(?:01www\.com|\.bg)|9(?:guacu\.br|\.bg)|м(?:о(?:сква|н)|кд)|大(?:[分阪]\.jp|众汽车|拿)|ع(?:ر(?:اق|ب)|مان)|भार(?:त(?:म्)?|ोत)|6(?:4-b\.it|\.bg)|இ(?:ந்தியா|லங்கை)|(?:[个箇]人|敎育)\.hk|سو(?:ري[اة]|دان)|商(?:[城店标]|業\.tw)|香(?:川\.jp|格里拉|港)|у(?:пр\.срб|кр)|新(?:潟\.jp|加坡|闻)|(?:ак|пр)\.срб|к(?:атолик|ом)|ك(?:اثوليك|وم)|中(?:[信国國]|文网)|個人\.(?:hk|香港)|教育\.(?:hk|香港)|(?:グーグ|セー)ル|с(?:айт|рб)|சிங்கப்பூர்|嘉里(?:大酒店)?|[578]\.bg|б(?:ел|г)|р(?:ус|ф)|ভা[রৰ]ত|ファッション|همراه|संगठन|বাংলা|భారత్|ഭാരതം|台[湾灣]|手[机表]|澳[門门]|닷[넷컴]|дети|تونس|شبكة|ڀارت|ਭਾਰਤ|ભારત|ଭାରତ|ಭಾರತ|ලංකා|クラウド|ポイント|電訊盈科|қаз|հայ|קום|قطر|कॉम|नेट|คอม|みんな|ストア|天主教|我爱你|淡马锡|诺基亚|飞利浦|ελ|ею|გე|コム|世界|企业|佛山|信息|健康|八卦|在线|娱乐|家電|工行|广东|微博|慈善|招聘|时尚|書籍|机构|游戏|点看|珠宝|移动|联通|臺灣|谷歌|购物|通販|集团|食品|餐厅|삼성|한국)$/
  ,
  _tldEx: /(?:\.|^)(?:city\.(?:k(?:itakyushu|awasaki|obe)|(?:yokoham|nagoy)a|s(?:apporo|endai))\.jp|www\.ck)$/
};
