/**
 * Copyright (c) 2011-2019 EclipseSource Muenchen GmbH and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Stefan Dirix - initial API and implementation
 */
package org.eclipse.emf.ecp.emf2web.json.controller.xtend;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.Diagnostician;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecp.view.spi.categorization.model.VCategorization;
import org.eclipse.emf.ecp.view.spi.categorization.model.VCategorizationElement;
import org.eclipse.emf.ecp.view.spi.categorization.model.VCategory;
import org.eclipse.emf.ecp.view.spi.label.model.VLabel;
import org.eclipse.emf.ecp.view.spi.model.VContainer;
import org.eclipse.emf.ecp.view.spi.model.VControl;
import org.eclipse.emf.ecp.view.spi.model.VElement;
import org.eclipse.emf.ecp.view.spi.model.VView;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;

/**
 * Processes {@link VView}s before ui schema generation
 * 
 * @author Stefan Dirix <sdirix@eclipsesource.com>
 */
@SuppressWarnings("all")
public class ViewCleaner {
  /**
   * Number of maximum validation and cleaning iterations on any given view
   */
  private final static int VALIDATION_TRIES = 5;
  
  /**
   * Remove invalid and not supported {@link VElement}s.
   */
  public static void cleanView(final VView view) {
    ViewCleaner.removeInvalidElements(view, ViewCleaner.VALIDATION_TRIES);
    final List<? extends Class<? extends VElement>> whiteList = Collections.<Class<? extends VElement>>unmodifiableList(CollectionLiterals.<Class<? extends VElement>>newArrayList(VLabel.class, VCategorizationElement.class, VCategorization.class, VCategory.class, VView.class, VControl.class, VContainer.class));
    ViewCleaner.removeUnsupportedElements(view, whiteList);
  }
  
  /**
   * Removing invalid elements from the view might trigger new validation errors.
   * Therefore we should try multiple times to validate and clean the view model in case it is invalid.
   */
  private static void removeInvalidElements(final VView view, final int tries) {
    for (int i = 0; (i < tries); i++) {
      {
        boolean wasInvalid = ViewCleaner.removeInvalidElements(view);
        if ((!wasInvalid)) {
          return;
        }
      }
    }
  }
  
  private static boolean removeInvalidElements(final VView view) {
    boolean _xblockexpression = false;
    {
      final Diagnostic validation = Diagnostician.INSTANCE.validate(view);
      int _severity = validation.getSeverity();
      boolean _equals = (_severity == Diagnostic.ERROR);
      if (_equals) {
        List<Diagnostic> _children = validation.getChildren();
        for (final Diagnostic diagnostic : _children) {
          ViewCleaner.removeInvalidElements(diagnostic);
        }
      }
      int _severity_1 = validation.getSeverity();
      _xblockexpression = (_severity_1 == Diagnostic.ERROR);
    }
    return _xblockexpression;
  }
  
  private static void removeInvalidElements(final Diagnostic diagnostic) {
    if ((((diagnostic.getSeverity() == Diagnostic.ERROR) && (!diagnostic.getData().isEmpty())) && 
      EObject.class.isInstance(diagnostic.getData().get(0)))) {
      Object _get = diagnostic.getData().get(0);
      EcoreUtil.delete(((EObject) _get));
    }
    List<Diagnostic> _children = diagnostic.getChildren();
    for (final Diagnostic childDiagnostic : _children) {
      ViewCleaner.removeInvalidElements(childDiagnostic);
    }
  }
  
  private static void removeUnsupportedElements(final VView view, final List<? extends Class<?>> whiteList) {
    final LinkedList<EObject> toDelete = new LinkedList<EObject>();
    final TreeIterator<EObject> allContents = view.eAllContents();
    while (allContents.hasNext()) {
      {
        final EObject next = allContents.next();
        boolean _isInstance = VElement.class.isInstance(next);
        if (_isInstance) {
          boolean supported = false;
          for (final Class<?> element : whiteList) {
            boolean _isInstance_1 = element.isInstance(next);
            if (_isInstance_1) {
              supported = true;
            }
          }
          if ((!supported)) {
            toDelete.add(next);
          }
        }
      }
    }
    EcoreUtil.deleteAll(toDelete, false);
  }
}
