/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.k3.al.annotationprocessor;

import com.google.common.base.Objects;
import com.google.common.collect.ImmutableList;
import java.util.Collection;
import org.eclipse.xtend.lib.macro.AbstractFieldProcessor;
import org.eclipse.xtend.lib.macro.TransformationContext;
import org.eclipse.xtend.lib.macro.declaration.AnnotationReference;
import org.eclipse.xtend.lib.macro.declaration.AnnotationTypeDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableFieldDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableMethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableTypeDeclaration;
import org.eclipse.xtend.lib.macro.declaration.Type;
import org.eclipse.xtend.lib.macro.declaration.TypeReference;
import org.eclipse.xtend.lib.macro.declaration.Visibility;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Opposite annotation processing
 */
@SuppressWarnings("all")
public class OppositeProcessor extends AbstractFieldProcessor {
  protected MutableTypeDeclaration containingType;
  
  protected MutableFieldDeclaration field;
  
  protected MutableTypeDeclaration oppositeType;
  
  protected MutableFieldDeclaration oppositeField;
  
  @Extension
  protected TransformationContext context;
  
  protected static final String GENERATED_PREFIX = "__K3_";
  
  /**
   * Weaves opposite behavior
   */
  @Override
  public void doTransform(final MutableFieldDeclaration field, final TransformationContext ctx) {
    this.context = ctx;
    final Function1<AnnotationReference, Boolean> _function = (AnnotationReference it) -> {
      AnnotationTypeDeclaration _annotationTypeDeclaration = it.getAnnotationTypeDeclaration();
      Type _type = this.context.newTypeReference(Opposite.class).getType();
      return Boolean.valueOf(Objects.equal(_annotationTypeDeclaration, _type));
    };
    final Object oppositeRefName = IterableExtensions.findFirst(field.getAnnotations(), _function).getValue("value");
    boolean _isCollection = this.isCollection(field.getType());
    if (_isCollection) {
      int _size = field.getType().getActualTypeArguments().size();
      boolean _notEquals = (_size != 1);
      if (_notEquals) {
        this.context.addError(field, "Only collections with 1 type parameter are supported");
        return;
      }
      this.oppositeType = this.context.findClass(IterableExtensions.<TypeReference>head(field.getType().getActualTypeArguments()).getName());
    } else {
      this.oppositeType = this.context.findClass(field.getType().getName());
    }
    this.field = field;
    this.containingType = field.getDeclaringType();
    final Function1<MutableFieldDeclaration, Boolean> _function_1 = (MutableFieldDeclaration f) -> {
      return Boolean.valueOf(f.getSimpleName().equals(oppositeRefName));
    };
    this.oppositeField = IterableExtensions.findFirst(this.oppositeType.getDeclaredFields(), _function_1);
    boolean _check = this.check();
    if (_check) {
      this.field.setVisibility(Visibility.PRIVATE);
      this.generateInitializer();
      this.generateGetterMethod();
      this.generateSetterProxyMethod();
      this.generateResetMethod();
      this.generateSetMethod();
    }
  }
  
  protected void generateInitializer() {
    boolean _isCollection = this.isCollection(this.field.getType());
    if (_isCollection) {
      final String t = IterableExtensions.<TypeReference>head(this.field.getType().getActualTypeArguments()).getSimpleName();
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("new java.util.ArrayList<");
          _builder.append(t);
          _builder.append(">()");
        }
      };
      this.field.setInitializer(_client);
    } else {
      StringConcatenationClient _client_1 = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("null");
        }
      };
      this.field.setInitializer(_client_1);
    }
  }
  
  /**
   * Generates a simple getter method
   */
  protected void generateGetterMethod() {
    final String f = this.field.getSimpleName();
    boolean _isCollection = this.isCollection(this.field.getType());
    if (_isCollection) {
      final Procedure1<MutableMethodDeclaration> _function = (MutableMethodDeclaration it) -> {
        it.setVisibility(Visibility.PUBLIC);
        it.setReturnType(this.context.newTypeReference(ImmutableList.class, IterableExtensions.<TypeReference>head(this.field.getType().getActualTypeArguments())));
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return com.google.common.collect.ImmutableList.copyOf(");
            _builder.append(f);
            _builder.append(") ;");
          }
        };
        it.setBody(_client);
      };
      this.containingType.addMethod(this.getGetterName(this.field), _function);
    } else {
      final Procedure1<MutableMethodDeclaration> _function_1 = (MutableMethodDeclaration it) -> {
        it.setVisibility(Visibility.PUBLIC);
        it.setReturnType(this.field.getType());
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return ");
            _builder.append(f);
            _builder.append(" ;");
          }
        };
        it.setBody(_client);
      };
      this.containingType.addMethod(this.getGetterName(this.field), _function_1);
    }
  }
  
  /**
   * Generates a proxy setter for the annotated
   * field in order to inject the opposite behavior
   */
  protected void generateSetterProxyMethod() {
    final String f = this.field.getSimpleName();
    final String o = this.oppositeField.getSimpleName();
    final TypeReference t = this.oppositeField.getType();
    boolean _isCollection = this.isCollection(this.field.getType());
    if (_isCollection) {
      String _firstUpper = StringExtensions.toFirstUpper(this.field.getSimpleName());
      String _plus = ("add" + _firstUpper);
      final Procedure1<MutableMethodDeclaration> _function = (MutableMethodDeclaration it) -> {
        it.setVisibility(Visibility.PUBLIC);
        it.addParameter("obj", IterableExtensions.<TypeReference>head(this.field.getType().getActualTypeArguments()));
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("if (!");
            _builder.append(f);
            _builder.append(".contains(obj)) {");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("if (obj != null)");
            _builder.newLine();
            _builder.append("\t\t");
            _builder.append("obj.");
            _builder.append(OppositeProcessor.GENERATED_PREFIX, "\t\t");
            _builder.append(o, "\t\t");
            _builder.append("_set(this) ;");
            _builder.newLineIfNotEmpty();
            _builder.newLine();
            _builder.append("\t");
            _builder.append(f, "\t");
            _builder.append(".add(obj) ;");
            _builder.newLineIfNotEmpty();
            _builder.append("}");
            _builder.newLine();
          }
        };
        it.setBody(_client);
      };
      this.containingType.addMethod(_plus, _function);
    } else {
      final Procedure1<MutableMethodDeclaration> _function_1 = (MutableMethodDeclaration it) -> {
        it.setVisibility(Visibility.PUBLIC);
        it.addParameter("obj", this.field.getType());
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("if (obj != ");
            _builder.append(f);
            _builder.append(") {");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("if (");
            _builder.append(f, "\t");
            _builder.append(" != null)");
            _builder.newLineIfNotEmpty();
            _builder.append("\t\t");
            _builder.append(f, "\t\t");
            _builder.append(".");
            _builder.append(OppositeProcessor.GENERATED_PREFIX, "\t\t");
            _builder.append(o, "\t\t");
            _builder.append("_reset(");
            {
              boolean _isCollection = OppositeProcessor.this.isCollection(t);
              if (_isCollection) {
                _builder.append("this");
              }
            }
            _builder.append(") ;");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("if (obj != null)");
            _builder.newLine();
            _builder.append("\t\t");
            _builder.append("obj.");
            _builder.append(OppositeProcessor.GENERATED_PREFIX, "\t\t");
            _builder.append(o, "\t\t");
            _builder.append("_set(this) ;");
            _builder.newLineIfNotEmpty();
            _builder.newLine();
            _builder.append("\t");
            _builder.append(f, "\t");
            _builder.append(" = obj ;");
            _builder.newLineIfNotEmpty();
            _builder.append("}");
            _builder.newLine();
          }
        };
        it.setBody(_client);
      };
      this.containingType.addMethod(this.getSetterName(this.field), _function_1);
    }
  }
  
  /**
   * Reset annotated field value
   */
  protected void generateResetMethod() {
    final String f = this.field.getSimpleName();
    final String o = this.oppositeField.getSimpleName();
    final TypeReference t = this.oppositeField.getType();
    boolean _isCollection = this.isCollection(this.field.getType());
    if (_isCollection) {
      String _simpleName = this.field.getSimpleName();
      String _plus = (OppositeProcessor.GENERATED_PREFIX + _simpleName);
      String _plus_1 = (_plus + "_reset");
      final Procedure1<MutableMethodDeclaration> _function = (MutableMethodDeclaration it) -> {
        it.setVisibility(Visibility.PUBLIC);
        it.addParameter("obj", IterableExtensions.<TypeReference>head(this.field.getType().getActualTypeArguments()));
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("if (");
            _builder.append(f);
            _builder.append(".contains(obj))");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append(f, "\t");
            _builder.append(".remove(obj) ;");
            _builder.newLineIfNotEmpty();
          }
        };
        it.setBody(_client);
      };
      this.containingType.addMethod(_plus_1, _function);
      String _firstUpper = StringExtensions.toFirstUpper(this.field.getSimpleName());
      String _plus_2 = ("remove" + _firstUpper);
      final Procedure1<MutableMethodDeclaration> _function_1 = (MutableMethodDeclaration it) -> {
        it.setVisibility(Visibility.PUBLIC);
        it.addParameter("obj", IterableExtensions.<TypeReference>head(this.field.getType().getActualTypeArguments()));
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("if (obj != null)");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("obj.");
            _builder.append(OppositeProcessor.GENERATED_PREFIX, "\t");
            _builder.append(o, "\t");
            _builder.append("_reset(");
            {
              boolean _isCollection = OppositeProcessor.this.isCollection(t);
              if (_isCollection) {
                _builder.append("this");
              }
            }
            _builder.append(") ;");
            _builder.newLineIfNotEmpty();
            _builder.newLine();
            _builder.append(f);
            _builder.append(".remove(obj) ;");
            _builder.newLineIfNotEmpty();
          }
        };
        it.setBody(_client);
      };
      this.containingType.addMethod(_plus_2, _function_1);
    } else {
      final Procedure1<MutableMethodDeclaration> _function_2 = (MutableMethodDeclaration it) -> {
        it.setVisibility(Visibility.PUBLIC);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append(f);
            _builder.append(" = null ;");
          }
        };
        it.setBody(_client);
      };
      this.containingType.addMethod(((OppositeProcessor.GENERATED_PREFIX + f) + "_reset"), _function_2);
    }
  }
  
  /**
   * Set field value
   */
  protected void generateSetMethod() {
    final String f = this.field.getSimpleName();
    final String o = this.oppositeField.getSimpleName();
    final TypeReference t = this.oppositeField.getType();
    boolean _isCollection = this.isCollection(this.field.getType());
    if (_isCollection) {
      final Procedure1<MutableMethodDeclaration> _function = (MutableMethodDeclaration it) -> {
        it.setVisibility(Visibility.PUBLIC);
        it.addParameter("obj", IterableExtensions.<TypeReference>head(this.field.getType().getActualTypeArguments()));
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append(f);
            _builder.append(".add(obj) ;");
            _builder.newLineIfNotEmpty();
          }
        };
        it.setBody(_client);
      };
      this.containingType.addMethod(((OppositeProcessor.GENERATED_PREFIX + f) + "_set"), _function);
    } else {
      final Procedure1<MutableMethodDeclaration> _function_1 = (MutableMethodDeclaration it) -> {
        it.setVisibility(Visibility.PUBLIC);
        it.addParameter("obj", this.field.getType());
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("if (");
            _builder.append(f);
            _builder.append(" != null)");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append(f, "\t");
            _builder.append(".");
            _builder.append(OppositeProcessor.GENERATED_PREFIX, "\t");
            _builder.append(o, "\t");
            _builder.append("_reset(");
            {
              boolean _isCollection = OppositeProcessor.this.isCollection(t);
              if (_isCollection) {
                _builder.append("this");
              }
            }
            _builder.append(") ;");
            _builder.newLineIfNotEmpty();
            _builder.newLine();
            _builder.append(f);
            _builder.append(" = obj ;");
            _builder.newLineIfNotEmpty();
          }
        };
        it.setBody(_client);
      };
      this.containingType.addMethod(((OppositeProcessor.GENERATED_PREFIX + f) + "_set"), _function_1);
    }
  }
  
  /**
   * Checks whether the opposite references
   * are properly defined
   * @return true if the opposite references are properly defined
   */
  protected boolean check() {
    if ((this.field.getType().isPrimitive() || this.field.getType().isWrapper())) {
      String _simpleName = this.field.getType().getSimpleName();
      String _plus = ("Can\'t declare a primitive type " + _simpleName);
      String _plus_1 = (_plus + " as opposite");
      this.context.addError(this.field, _plus_1);
      return false;
    }
    if ((this.oppositeField == null)) {
      this.context.addError(this.field, "Referenced opposite attribute doesn\'t exist");
      return false;
    }
    if ((((!this.isCollection(this.oppositeField.getType())) && (!Objects.equal(this.oppositeField.getType(), this.context.newTypeReference(this.containingType)))) || (this.isCollection(this.oppositeField.getType()) && (!Objects.equal(IterableExtensions.<TypeReference>head(this.oppositeField.getType().getActualTypeArguments()), this.context.newTypeReference(this.containingType)))))) {
      String _simpleName_1 = this.oppositeField.getType().getSimpleName();
      String _plus_2 = ("The opposite attribute type (" + _simpleName_1);
      String _plus_3 = (_plus_2 + ") doesn\'t match");
      this.context.addError(this.field, _plus_3);
      return false;
    }
    if ((IterableExtensions.exists(this.field.getAnnotations(), ((Function1<AnnotationReference, Boolean>) (AnnotationReference it) -> {
      AnnotationTypeDeclaration _annotationTypeDeclaration = it.getAnnotationTypeDeclaration();
      Type _type = this.context.newTypeReference(Composition.class).getType();
      return Boolean.valueOf(Objects.equal(_annotationTypeDeclaration, _type));
    })) && 
      IterableExtensions.exists(this.oppositeField.getAnnotations(), ((Function1<AnnotationReference, Boolean>) (AnnotationReference it) -> {
        AnnotationTypeDeclaration _annotationTypeDeclaration = it.getAnnotationTypeDeclaration();
        Type _type = this.context.newTypeReference(Composition.class).getType();
        return Boolean.valueOf(Objects.equal(_annotationTypeDeclaration, _type));
      })))) {
      this.context.addError(this.field, "Can\'t declare as opposites two composition references");
      return false;
    }
    final Function1<AnnotationReference, Boolean> _function = (AnnotationReference it) -> {
      return Boolean.valueOf((Objects.equal(it.getAnnotationTypeDeclaration(), this.context.newTypeReference(Opposite.class).getType()) && 
        it.getValue("value").equals(this.field.getSimpleName())));
    };
    boolean _exists = IterableExtensions.exists(this.oppositeField.getAnnotations(), _function);
    boolean _not = (!_exists);
    if (_not) {
      this.context.addError(this.field, "The opposite attribute must be marked as opposite of this attribute");
      return false;
    }
    return true;
  }
  
  /**
   * Checks whether the specified TypeReference refers
   * to a collection
   * @param type type reference
   * @return true if the type is a collection
   */
  protected boolean isCollection(final TypeReference type) {
    return this.context.newTypeReference(Collection.class, this.context.newWildcardTypeReference()).isAssignableFrom(type);
  }
  
  protected String getGetterName(final MutableFieldDeclaration f) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("get");
    String _firstUpper = StringExtensions.toFirstUpper(f.getSimpleName());
    _builder.append(_firstUpper);
    return _builder.toString();
  }
  
  protected String getSetterName(final MutableFieldDeclaration f) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("set");
    String _firstUpper = StringExtensions.toFirstUpper(f.getSimpleName());
    _builder.append(_firstUpper);
    return _builder.toString();
  }
}
