/**
 * Copyright (c) 2017 Inria.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.xdsmlframework.test.lib;

import com.google.common.base.Objects;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.zip.ZipFile;
import org.eclipse.core.commands.Command;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.expressions.IEvaluationContext;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.ui.IDebugUIConstants;
import org.eclipse.gemoc.xdsmlframework.ide.ui.XDSMLFrameworkUI;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.internal.framework.EquinoxBundle;
import org.eclipse.osgi.storage.BundleInfo;
import org.eclipse.pde.core.target.ITargetDefinition;
import org.eclipse.pde.core.target.ITargetLocation;
import org.eclipse.pde.core.target.ITargetPlatformService;
import org.eclipse.pde.core.target.LoadTargetDefinitionJob;
import org.eclipse.pde.internal.core.target.TargetPlatformService;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.ISources;
import org.eclipse.ui.IWindowListener;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.commands.ICommandService;
import org.eclipse.ui.dialogs.IOverwriteQuery;
import org.eclipse.ui.handlers.IHandlerService;
import org.eclipse.ui.internal.wizards.datatransfer.ZipLeveledStructureProvider;
import org.eclipse.ui.intro.IIntroPart;
import org.eclipse.ui.wizards.datatransfer.ImportOperation;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.ui.testing.util.IResourcesSetupUtil;
import org.eclipse.xtext.ui.testing.util.JavaProjectSetupUtil;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IntegerRange;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.junit.Assert;
import org.osgi.framework.Bundle;

/**
 * Class containing helper methods for testing a workspace in a GEMOC Language workbench
 */
@SuppressWarnings("all")
public class WorkspaceTestHelper {
  public static final String CMD_PROJECT_CLEAN = "org.eclipse.ui.project.cleanAction";

  /**
   * Value to use by default for SWTBotPreferences.TIMEOUT
   * It must be used for any GEMOC test using swtbot
   * we should increase this value up to the value where we don't have any false failed tests
   */
  public static final int SWTBotPreferencesTIMEOUT_4_GEMOC = 20000;

  /**
   * Value to use by default for SWTBotPreferences.PLAYBACK_DELAY
   * It must be used for any GEMOC test using swtbot
   * we should increase this value up to the value where we don't have any false failed tests
   */
  public static final int SWTBotPreferencesPLAYBACK_DELAY_4_GEMOC = 100;

  public void waitWorkbench() {
    try {
      final ArrayList<Boolean> workbenchAvailable = CollectionLiterals.<Boolean>newArrayList(Boolean.valueOf(false));
      while ((workbenchAvailable.get(0)).booleanValue()) {
        {
          Display _default = Display.getDefault();
          _default.syncExec(new Runnable() {
            @Override
            public void run() {
              if ((PlatformUI.isWorkbenchRunning() && (PlatformUI.getWorkbench().getActiveWorkbenchWindow() != null))) {
                workbenchAvailable.set(0, Boolean.valueOf(true));
              }
            }
          });
          InputOutput.<String>println("[WorkspaceTestHelper] waiting for Workbench availability");
          Thread.sleep(1000);
        }
      }
      WorkspaceTestHelper.reallyWaitForJobs(50);
      InputOutput.<String>println("[WorkspaceTestHelper] Workbench is available");
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public void init() {
    Display _default = Display.getDefault();
    _default.syncExec(new Runnable() {
      @Override
      public void run() {
        try {
          PlatformUI.getWorkbench().showPerspective(XDSMLFrameworkUI.ID_PERSPECTIVE, PlatformUI.getWorkbench().getActiveWorkbenchWindow());
          WorkspaceTestHelper.this.closeWelcomePage();
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      }
    });
  }

  public IProject getProject(final String projectName) {
    return ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
  }

  public boolean projectExists(final String projectName) {
    return this.getProject(projectName).exists();
  }

  private IProject deployProjectResult = null;

  public IProject deployProject(final String projectName, final String zipLocation) {
    this.deployProjectResult = null;
    final ArrayList<Throwable> thrownException = CollectionLiterals.<Throwable>newArrayList();
    final Runnable _function = () -> {
      try {
        final IProject newProject = JavaProjectSetupUtil.createSimpleProject(projectName);
        final ZipFile zip = new ZipFile(zipLocation);
        final ZipLeveledStructureProvider structureProvider = new ZipLeveledStructureProvider(zip);
        final IOverwriteQuery queryOverwrite = new IOverwriteQuery() {
          @Override
          public String queryOverwrite(final String file) {
            return IOverwriteQuery.ALL;
          }
        };
        IPath _fullPath = newProject.getProject().getFullPath();
        Object _root = structureProvider.getRoot();
        ImportOperation _importOperation = new ImportOperation(_fullPath, _root, structureProvider, queryOverwrite);
        NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
        _importOperation.run(_nullProgressMonitor);
        zip.close();
        this.deployProjectResult = newProject.getProject();
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          thrownException.add(e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    };
    Display.getDefault().syncExec(_function);
    final Function1<Throwable, Boolean> _function_1 = (Throwable e) -> {
      try {
        throw new Exception(e);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    IterableExtensions.<Throwable>forall(thrownException, _function_1);
    return this.deployProjectResult;
  }

  public void closeWelcomePage() {
    IIntroPart _intro = PlatformUI.getWorkbench().getIntroManager().getIntro();
    boolean _tripleNotEquals = (_intro != null);
    if (_tripleNotEquals) {
      PlatformUI.getWorkbench().getIntroManager().closeIntro(PlatformUI.getWorkbench().getIntroManager().getIntro());
    }
  }

  public void assertNoMarkers() {
    final Consumer<IProject> _function = (IProject project) -> {
      this.printAllMarkers(project);
    };
    ((List<IProject>)Conversions.doWrapArray(ResourcesPlugin.getWorkspace().getRoot().getProjects())).forEach(_function);
    final Consumer<IProject> _function_1 = (IProject project) -> {
      try {
        final Consumer<IMarker> _function_2 = (IMarker m) -> {
          try {
            Object _attribute = m.getAttribute(IMarker.MESSAGE);
            String _plus = ("Unexpected error marker: " + _attribute);
            String _plus_1 = (_plus + " on ");
            IPath _fullPath = m.getResource().getFullPath();
            String _plus_2 = (_plus_1 + _fullPath);
            Object _attribute_1 = m.getAttribute(IMarker.SEVERITY);
            boolean _equals = Objects.equal(_attribute_1, Integer.valueOf(IMarker.SEVERITY_ERROR));
            Assert.assertFalse(_plus_2, _equals);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        };
        ((List<IMarker>)Conversions.doWrapArray(project.findMarkers(IMarker.PROBLEM, true, IResource.DEPTH_INFINITE))).forEach(_function_2);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    ((List<IProject>)Conversions.doWrapArray(ResourcesPlugin.getWorkspace().getRoot().getProjects())).forEach(_function_1);
  }

  public void assertNoMarkers(final String filename) {
    try {
      IWorkspaceRoot _root = ResourcesPlugin.getWorkspace().getRoot();
      Path _path = new Path(filename);
      final IFile mlgFile = _root.getFile(_path);
      this.printAllMarkers(mlgFile);
      final Consumer<IMarker> _function = (IMarker it) -> {
        try {
          Object _attribute = it.getAttribute(IMarker.MESSAGE);
          String _plus = ("Unexpected error marker: " + _attribute);
          String _plus_1 = (_plus + " on file ");
          String _plus_2 = (_plus_1 + filename);
          Object _attribute_1 = it.getAttribute(IMarker.SEVERITY);
          boolean _equals = Objects.equal(_attribute_1, Integer.valueOf(IMarker.SEVERITY_ERROR));
          Assert.assertFalse(_plus_2, _equals);
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      };
      ((List<IMarker>)Conversions.doWrapArray(mlgFile.findMarkers(IMarker.PROBLEM, true, IResource.DEPTH_INFINITE))).forEach(_function);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public void printAllMarkers(final IResource res) {
    try {
      final Consumer<IMarker> _function = (IMarker it) -> {
        try {
          String _switchResult = null;
          Object _attribute = it.getAttribute(IMarker.SEVERITY);
          boolean _matched = false;
          if (Objects.equal(_attribute, IMarker.SEVERITY_ERROR)) {
            _matched=true;
            _switchResult = "error";
          }
          if (!_matched) {
            if (Objects.equal(_attribute, IMarker.SEVERITY_WARNING)) {
              _matched=true;
              _switchResult = "warning";
            }
          }
          if (!_matched) {
            if (Objects.equal(_attribute, IMarker.SEVERITY_INFO)) {
              _matched=true;
              _switchResult = "info";
            }
          }
          if (!_matched) {
            _switchResult = "";
          }
          final String severity = _switchResult;
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Found ");
          _builder.append(severity);
          _builder.append(" marker \"");
          Object _attribute_1 = it.getAttribute(IMarker.MESSAGE);
          _builder.append(_attribute_1);
          _builder.append("\" on ");
          String _name = res.getName();
          _builder.append(_name);
          InputOutput.<String>println(_builder.toString());
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      };
      ((List<IMarker>)Conversions.doWrapArray(res.findMarkers(IMarker.PROBLEM, true, IResource.DEPTH_INFINITE))).forEach(_function);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Usage : projectName/folder/file
   */
  public void assertFileExists(final String filename) {
    final IWorkspace ws = ResourcesPlugin.getWorkspace();
    IWorkspaceRoot _root = ws.getRoot();
    Path _path = new Path(filename);
    Assert.assertTrue(
      ("Cannot find file " + filename), 
      _root.getFile(_path).exists());
  }

  /**
   * Usage : projectName/folder/file
   */
  public void assertFileDontExists(final String filename) {
    final IWorkspace ws = ResourcesPlugin.getWorkspace();
    IWorkspaceRoot _root = ws.getRoot();
    Path _path = new Path(filename);
    Assert.assertFalse(
      ("Found file " + filename), 
      _root.getFile(_path).exists());
  }

  /**
   * Usage : projectName/folder
   */
  public void assertFolderExists(final String foldername) {
    final IWorkspace ws = ResourcesPlugin.getWorkspace();
    IWorkspaceRoot _root = ws.getRoot();
    Path _path = new Path(foldername);
    Assert.assertTrue(
      ("Cannot find file " + foldername), 
      _root.getFolder(_path).exists());
  }

  /**
   * Check if {@link project} exist
   */
  public void assertProjectExists(final String project) {
    final IWorkspace ws = ResourcesPlugin.getWorkspace();
    Assert.assertTrue(
      ("Cannot find project " + project), 
      ws.getRoot().getProject(project).exists());
  }

  /**
   * Check if {@link project} doesn't exist
   */
  public void assertProjectDoesntExists(final String project) {
    final IWorkspace ws = ResourcesPlugin.getWorkspace();
    Assert.assertFalse(
      ("Can find project " + project), 
      ws.getRoot().getProject(project).exists());
  }

  /**
   * search in the given list if it contains the searchedString, report the error with a message
   */
  public void assertContains(final String baseMessage, final String searchedString, final List<String> list) {
    Assert.assertTrue(((baseMessage + " ") + searchedString), list.contains(searchedString));
  }

  /**
   * Creates and launches a new run configuration for {@link project}
   * using {@link mainClass} as the main Java class.
   * 
   * @return the produced console output
   */
  public String runMainClass(final IProject project, final String mainClass) {
    try {
      final String outputFileName = "output.txt";
      final ILaunchManager manager = DebugPlugin.getDefault().getLaunchManager();
      final ILaunchConfigurationType type = manager.getLaunchConfigurationType(IJavaLaunchConfigurationConstants.ID_JAVA_APPLICATION);
      final ILaunchConfigurationWorkingCopy newLaunchConfig = type.newInstance(project, "RunMainTransfo");
      newLaunchConfig.setAttribute(IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME, project.getName());
      newLaunchConfig.setAttribute(IJavaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, mainClass);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("${workspace_loc:/");
      String _name = project.getName();
      _builder.append(_name);
      _builder.append("/");
      _builder.append(outputFileName);
      _builder.append("}");
      newLaunchConfig.setAttribute(IDebugUIConstants.ATTR_CAPTURE_IN_FILE, _builder);
      newLaunchConfig.doSave();
      final IFile outputFile = project.getFile(outputFileName);
      boolean _exists = outputFile.exists();
      boolean _not = (!_exists);
      if (_not) {
        byte[] _bytes = "".getBytes();
        ByteArrayInputStream _byteArrayInputStream = new ByteArrayInputStream(_bytes);
        outputFile.create(_byteArrayInputStream, true, null);
      }
      final ILaunch launch = newLaunchConfig.launch(ILaunchManager.RUN_MODE, null);
      final long TIMEOUT = ((1000 * 60) * 5);
      final long startTime = System.currentTimeMillis();
      while (((!launch.isTerminated()) && (System.currentTimeMillis() < (startTime + TIMEOUT)))) {
        try {
          Thread.sleep(50);
        } catch (final Throwable _t) {
          if (_t instanceof InterruptedException) {
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      }
      outputFile.refreshLocal(IResource.DEPTH_ONE, null);
      final InputStream bi = outputFile.getContents();
      final byte[] buffer = new byte[bi.available()];
      bi.read(buffer);
      final String out = new String(buffer);
      return out;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public void debug() {
    final Consumer<IProject> _function = (IProject project) -> {
      try {
        String _name = project.getName();
        String _plus = ("Error markers: " + _name);
        InputOutput.<String>println(_plus);
        final Consumer<IMarker> _function_1 = (IMarker marker) -> {
          try {
            String _name_1 = marker.getResource().getName();
            String _plus_1 = ("   Resource: " + _name_1);
            InputOutput.<String>println(_plus_1);
            Object _attribute = marker.getAttribute(IMarker.LOCATION);
            String _plus_2 = ("   Location: " + _attribute);
            InputOutput.<String>println(_plus_2);
            Object _attribute_1 = marker.getAttribute(IMarker.MESSAGE);
            String _plus_3 = ("   Message: " + _attribute_1);
            String _plus_4 = (_plus_3 + "\n");
            InputOutput.<String>println(_plus_4);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        };
        ((List<IMarker>)Conversions.doWrapArray(project.findMarkers(IMarker.PROBLEM, true, IResource.DEPTH_INFINITE))).forEach(_function_1);
        InputOutput.<String>println("\n");
        String _name_1 = project.getName();
        String _plus_1 = ("Classpath: " + _name_1);
        InputOutput.<String>println(_plus_1);
        final IJavaProject jProject = JavaCore.create(project);
        final StringBuffer cp = new StringBuffer();
        final Consumer<IClasspathEntry> _function_2 = (IClasspathEntry entry) -> {
          String _string = entry.getPath().toString();
          String _plus_2 = (" " + _string);
          String _plus_3 = (_plus_2 + "\n");
          cp.append(_plus_3);
        };
        ((List<IClasspathEntry>)Conversions.doWrapArray(jProject.getResolvedClasspath(false))).forEach(_function_2);
        final String str = cp.toString();
        InputOutput.<String>println(str);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    ((List<IProject>)Conversions.doWrapArray(ResourcesPlugin.getWorkspace().getRoot().getProjects())).forEach(_function);
  }

  /**
   * Sets a target platform in the test platform to get workspace builds OK
   * with PDE
   * 
   * @throws Exception
   */
  public void setTargetPlatform() throws Exception {
    final ITargetPlatformService tpService = TargetPlatformService.getDefault();
    final ITargetDefinition targetDef = tpService.newTarget();
    targetDef.setName("Tycho platform");
    final Bundle[] bundles = Platform.getBundle("org.eclipse.core.runtime").getBundleContext().getBundles();
    final List<ITargetLocation> bundleContainers = new ArrayList<ITargetLocation>();
    final Set<File> dirs = new HashSet<File>();
    for (final Bundle bundle : bundles) {
      {
        final EquinoxBundle bundleImpl = ((EquinoxBundle) bundle);
        Object _revisionInfo = bundleImpl.getModule().getCurrentRevision().getRevisionInfo();
        final BundleInfo.Generation generation = ((BundleInfo.Generation) _revisionInfo);
        final File file = generation.getBundleFile().getBaseFile();
        final File folder = file.getParentFile();
        boolean _contains = dirs.contains(folder);
        boolean _not = (!_contains);
        if (_not) {
          dirs.add(folder);
          bundleContainers.add(tpService.newDirectoryLocation(folder.getAbsolutePath()));
        }
      }
    }
    final ITargetLocation[] bundleContainersArray = ((ITargetLocation[])Conversions.unwrapArray(bundleContainers, ITargetLocation.class));
    targetDef.setTargetLocations(bundleContainersArray);
    targetDef.setArch(Platform.getOSArch());
    targetDef.setOS(Platform.getOS());
    targetDef.setWS(Platform.getWS());
    targetDef.setNL(Platform.getNL());
    tpService.saveTargetDefinition(targetDef);
    final Job job = new LoadTargetDefinitionJob(targetDef);
    job.schedule();
    job.join();
  }

  /**
   * call a command the selection file, if file is null or empty it will call the command without selection
   */
  public static void invokeCommandOnSelectedFile(final String commandId, final String file) {
    try {
      final IWorkspace ws = ResourcesPlugin.getWorkspace();
      final IWorkbench wb = PlatformUI.getWorkbench();
      ICommandService _service = wb.<ICommandService>getService(ICommandService.class);
      final ICommandService commandService = ((ICommandService) _service);
      IHandlerService _service_1 = wb.<IHandlerService>getService(IHandlerService.class);
      final IHandlerService handlerService = ((IHandlerService) _service_1);
      final Command command = commandService.getCommand(commandId);
      final ExecutionEvent executionEvent = handlerService.createExecutionEvent(command, null);
      Object _applicationContext = executionEvent.getApplicationContext();
      final IEvaluationContext context = ((IEvaluationContext) _applicationContext);
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(file);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        IWorkspaceRoot _root = ws.getRoot();
        Path _path = new Path(file);
        final IFile iFile = _root.getFile(_path);
        IEvaluationContext _parent = context.getParent();
        StructuredSelection _structuredSelection = new StructuredSelection(iFile);
        _parent.addVariable(ISources.ACTIVE_MENU_SELECTION_NAME, _structuredSelection);
      }
      command.executeWithChecks(executionEvent);
      IResourcesSetupUtil.reallyWaitForAutoBuild();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public ClassLoader createClassLoader(final IJavaProject project) {
    try {
      final String[] classPathEntries = JavaRuntime.computeDefaultRuntimeClassPath(project);
      final ArrayList<URL> urlList = new ArrayList<URL>();
      for (int i = 0; (i < classPathEntries.length); i++) {
        {
          final String entry = classPathEntries[i];
          final Path path = new Path(entry);
          final URL url = path.toFile().toURI().toURL();
          urlList.add(url);
        }
      }
      final ClassLoader parentClassLoader = project.getClass().getClassLoader();
      return new URLClassLoader(((URL[])Conversions.unwrapArray(urlList, URL.class)), parentClassLoader);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public void waitFileExistOrAssert(final String fileName, final int retry, final long sleep) {
    final IWorkspace ws = ResourcesPlugin.getWorkspace();
    ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, retry, true);
    for (final Integer i : _doubleDotLessThan) {
      {
        IWorkspaceRoot _root = ws.getRoot();
        Path _path = new Path(fileName);
        boolean _exists = _root.getFile(_path).exists();
        if (_exists) {
          return;
        }
        try {
          Thread.sleep(sleep);
        } catch (final Throwable _t) {
          if (_t instanceof InterruptedException) {
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      }
    }
    this.assertFileExists(fileName);
  }

  /**
   * relaunch the  waitForJobs several times in case some other background task
   * also wait for idle time to triggers new jobs
   */
  public static void reallyWaitForJobs(final int retry) {
    try {
      IntegerRange _upTo = new IntegerRange(0, retry);
      for (final Integer i : _upTo) {
        {
          WorkspaceTestHelper.waitForJobs();
          Thread.sleep(100);
        }
      }
      WorkspaceTestHelper.waitForJobs();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public static void waitForJobs() {
    try {
      final int delay = 1000;
      int retry = 600;
      boolean _isIdle = Job.getJobManager().isIdle();
      boolean _not = (!_isIdle);
      if (_not) {
        WorkspaceTestHelper.delay(delay);
      }
      while ((!Job.getJobManager().isIdle())) {
        {
          final Job currentJob = Job.getJobManager().currentJob();
          if (((retry % 10) == 0)) {
            final Job[] jobs = Job.getJobManager().find(null);
            for (final Job job : jobs) {
              {
                if ((job == currentJob)) {
                  String _name = job.getName();
                  String _plus = ("[waitForJobs](current) " + _name);
                  String _plus_1 = (_plus + " (rule=");
                  ISchedulingRule _rule = job.getRule();
                  String _plus_2 = (_plus_1 + _rule);
                  String _plus_3 = (_plus_2 + ")");
                  InputOutput.<String>println(_plus_3);
                } else {
                  String _name_1 = job.getName();
                  String _plus_4 = ("[waitForJobs] " + _name_1);
                  String _plus_5 = (_plus_4 + " (rule=");
                  ISchedulingRule _rule_1 = job.getRule();
                  String _plus_6 = (_plus_5 + _rule_1);
                  String _plus_7 = (_plus_6 + ")");
                  InputOutput.<String>println(_plus_7);
                }
                if (((retry <= 300) && ((((job.getName().contains("Git Repository Change Scanner") || 
                  job.getName().contains("Periodic workspace save")) || 
                  job.getName().contains("Workbench Auto-Save Job")) || 
                  job.getName().contains("Compacting resource model")) || 
                  job.getName().contains("Open Blocked Dialog")))) {
                  String _name_2 = job.getName();
                  String _plus_8 = ("[waitForJobs] CANCELLING job " + _name_2);
                  String _plus_9 = (_plus_8 + " (rule=");
                  ISchedulingRule _rule_2 = job.getRule();
                  String _plus_10 = (_plus_9 + _rule_2);
                  String _plus_11 = (_plus_10 + ")");
                  InputOutput.<String>println(_plus_11);
                  job.cancel();
                }
              }
            }
            final Function1<Job, Boolean> _function = (Job j) -> {
              ISchedulingRule _rule = j.getRule();
              return Boolean.valueOf((_rule != null));
            };
            final Function1<Job, ISchedulingRule> _function_1 = (Job j) -> {
              return j.getRule();
            };
            final Iterable<ISchedulingRule> rules = IterableExtensions.<Job, ISchedulingRule>map(IterableExtensions.<Job>filter(((Iterable<Job>)Conversions.doWrapArray(jobs)), _function), _function_1);
            final Consumer<ISchedulingRule> _function_2 = (ISchedulingRule r1) -> {
              final Consumer<ISchedulingRule> _function_3 = (ISchedulingRule r2) -> {
                if (((!Objects.equal(r1, r2)) && r1.isConflicting(r2))) {
                  InputOutput.<String>println(((("[waitForJobs] conflicting rules: " + r1) + " / ") + r2));
                }
              };
              rules.forEach(_function_3);
            };
            rules.forEach(_function_2);
          }
          WorkspaceTestHelper.delay(delay);
          retry = (retry - 1);
          if ((retry <= 0)) {
            InputOutput.<String>println((("waitForJobs timed out after " + Integer.valueOf((delay * 600))) + "ms"));
            InputOutput.<String>println("[waitForJobs] CANCELLING all jobs ");
            final Job[] jobs_1 = Job.getJobManager().find(null);
            for (final Job job_1 : jobs_1) {
              {
                String _name = job_1.getName();
                String _plus = ("[waitForJobs] CANCELLING job " + _name);
                String _plus_1 = (_plus + " (rule=");
                ISchedulingRule _rule = job_1.getRule();
                String _plus_2 = (_plus_1 + _rule);
                String _plus_3 = (_plus_2 + ")");
                InputOutput.<String>println(_plus_3);
                job_1.cancel();
              }
            }
            throw new InterruptedException((("waitForJobs timed out after " + Integer.valueOf((delay * 600))) + "ms"));
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private static boolean closed = false;

  public static void delay(final long waitTimeMillis) {
    final Display display = Display.getCurrent();
    IWorkbench _workbench = PlatformUI.getWorkbench();
    _workbench.addWindowListener(
      new IWindowListener() {
        @Override
        public void windowActivated(final IWorkbenchWindow window) {
        }

        @Override
        public void windowClosed(final IWorkbenchWindow window) {
          WorkspaceTestHelper.closed = true;
        }

        @Override
        public void windowDeactivated(final IWorkbenchWindow window) {
        }

        @Override
        public void windowOpened(final IWorkbenchWindow window) {
        }
      });
    if ((display != null)) {
      long _currentTimeMillis = System.currentTimeMillis();
      final long endTimeMillis = (_currentTimeMillis + (waitTimeMillis / 2));
      while (((System.currentTimeMillis() < endTimeMillis) && (!WorkspaceTestHelper.closed))) {
        boolean _readAndDispatch = display.readAndDispatch();
        boolean _not = (!_readAndDispatch);
        if (_not) {
          display.sleep();
        }
      }
      display.update();
      try {
        Thread.sleep((waitTimeMillis / 2));
      } catch (final Throwable _t) {
        if (_t instanceof InterruptedException) {
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } else {
      try {
        Thread.sleep(waitTimeMillis);
      } catch (final Throwable _t) {
        if (_t instanceof InterruptedException) {
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
  }

  public static void forceCleanPreviousWorkspaceContent() {
    try {
      IResourcesSetupUtil.cleanWorkspace();
      final String dirName = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();
      System.out.println((("workspace content (" + dirName) + ")"));
      final ArrayList<java.nio.file.Path> dirsToDelete = CollectionLiterals.<java.nio.file.Path>newArrayList();
      final Consumer<java.nio.file.Path> _function = (java.nio.file.Path path) -> {
        System.out.println(("   " + path));
        boolean _endsWith = path.endsWith(".metadata");
        boolean _not = (!_endsWith);
        if (_not) {
          dirsToDelete.add(path);
        }
      };
      Files.list(new File(dirName).toPath()).forEach(_function);
      final Consumer<java.nio.file.Path> _function_1 = (java.nio.file.Path dir) -> {
        try {
          System.out.println((("Deleting " + dir) + "..."));
          final Function<java.nio.file.Path, File> _function_2 = (java.nio.file.Path p) -> {
            return p.toFile();
          };
          final Consumer<File> _function_3 = (File f) -> {
            f.delete();
          };
          Files.walk(dir).sorted(Comparator.<java.nio.file.Path>reverseOrder()).<File>map(_function_2).forEach(_function_3);
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      };
      dirsToDelete.forEach(_function_1);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
