/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.executionframework.debugger;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import fr.inria.diverse.melange.resource.MelangeResourceImpl;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.function.BiPredicate;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gemoc.commons.eclipse.xtext.NameHelper;
import org.eclipse.gemoc.dsl.debug.ide.IDSLDebugger;
import org.eclipse.gemoc.dsl.debug.ide.event.IDSLDebugEventProcessor;
import org.eclipse.gemoc.executionframework.engine.core.EngineStoppedException;
import org.eclipse.gemoc.trace.commons.model.helper.StepHelper;
import org.eclipse.gemoc.trace.commons.model.trace.MSE;
import org.eclipse.gemoc.trace.commons.model.trace.MSEOccurrence;
import org.eclipse.gemoc.trace.commons.model.trace.ParallelStep;
import org.eclipse.gemoc.trace.commons.model.trace.Step;
import org.eclipse.gemoc.xdsmlframework.api.core.IExecutionEngine;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class GenericSequentialModelDebugger extends AbstractGemocDebugger {
  public static class MSEFrameInformation {
    public final EObject caller;

    public final String prettyLabel;

    public MSEFrameInformation(final EObject caller, final String prettyLabel) {
      this.caller = caller;
      this.prettyLabel = prettyLabel;
    }
  }

  private static class ToPushPop {
    public Step<?> step;

    public boolean push;

    public ToPushPop(final Step<?> step, final boolean push) {
      this.step = step;
      this.push = push;
    }
  }

  /**
   * A fake instruction to prevent the stepping return to stop on each event.
   */
  private static final EObject FAKE_INSTRUCTION = EcorePackage.eINSTANCE;

  private List<GenericSequentialModelDebugger.ToPushPop> toPushPop = new ArrayList<GenericSequentialModelDebugger.ToPushPop>();

  protected final String threadName = "Model debugging";

  protected int nbStackFrames = 0;

  protected boolean executionTerminated = false;

  public GenericSequentialModelDebugger(final IDSLDebugEventProcessor target, final IExecutionEngine<?> engine) {
    super(target, engine);
  }

  /**
   * This method is eventually called within a new engine thread. (non-Javadoc)
   * 
   * @see IDSLDebugger#start()
   */
  @Override
  public void start() {
    this.engine.start();
  }

  @Override
  public void disconnect() {
    return;
  }

  protected void setupStepReturnPredicateBreak() {
    abstract class __GenericSequentialModelDebugger_1 implements BiPredicate<IExecutionEngine<?>, Step<?>> {
      Step<?> steppedReturn;
    }

    final IExecutionEngine<?> seqEngine = ((IExecutionEngine<?>) this.engine);
    final Deque<Step<?>> stack = seqEngine.getCurrentStack();
    int _size = stack.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      final Iterator<Step<?>> it = stack.iterator();
      it.next();
      __GenericSequentialModelDebugger_1 ___GenericSequentialModelDebugger_1 = new __GenericSequentialModelDebugger_1() {
        {
          steppedReturn = it.next();
        }
        @Override
        public boolean test(final IExecutionEngine<?> t, final Step<?> u) {
          boolean _contains = seqEngine.getCurrentStack().contains(this.steppedReturn);
          return (!_contains);
        }
      };
      this.addPredicateBreak(___GenericSequentialModelDebugger_1);
    }
  }

  @Override
  public void steppingReturn(final String threadName) {
    super.steppingReturn(threadName);
    this.setupStepReturnPredicateBreak();
  }

  protected void setupStepOverPredicateBreak() {
    abstract class __GenericSequentialModelDebugger_2 implements BiPredicate<IExecutionEngine<?>, Step<?>> {
      IExecutionEngine<?> seqEngine;

      Step<?> steppedOver;
    }

    __GenericSequentialModelDebugger_2 ___GenericSequentialModelDebugger_2 = new __GenericSequentialModelDebugger_2() {
      {
        seqEngine = ((IExecutionEngine<?>) GenericSequentialModelDebugger.this.engine);

        steppedOver = this.seqEngine.getCurrentStep();
      }
      @Override
      public boolean test(final IExecutionEngine<?> t, final Step<?> u) {
        boolean _contains = this.seqEngine.getCurrentStack().contains(this.steppedOver);
        return (!_contains);
      }
    };
    this.addPredicateBreak(___GenericSequentialModelDebugger_2);
  }

  @Override
  public void steppingOver(final String threadName) {
    super.steppingOver(threadName);
    this.setupStepOverPredicateBreak();
  }

  @Override
  public boolean canStepInto(final String threadName, final EObject instruction) {
    final EObject currentInstruction = this.currentInstructions.get(threadName);
    final Step<?> currentStep = this.engine.getCurrentStep();
    final boolean correctObject = Objects.equal(currentInstruction, instruction);
    final boolean canStepInto = (!(currentStep instanceof ParallelStep<?, ?>));
    return (correctObject && canStepInto);
  }

  @Override
  public void steppingInto(final String threadName) {
    super.steppingInto(threadName);
    this.addPredicateBreak(new BiPredicate<IExecutionEngine<?>, Step<?>>() {
      @Override
      public boolean test(final IExecutionEngine<?> t, final Step<?> u) {
        return true;
      }
    });
  }

  @Override
  public void pushStackFrame(final String threadName, final String frameName, final EObject context, final EObject instruction) {
    super.pushStackFrame(threadName, frameName, context, instruction);
    this.nbStackFrames++;
  }

  @Override
  public void popStackFrame(final String threadName) {
    super.popStackFrame(threadName);
    this.nbStackFrames--;
  }

  protected String prettyFrameName(final MSEOccurrence mseoccurrence, final boolean implicit) {
    if ((mseoccurrence != null)) {
      final MSE mse = mseoccurrence.getMse();
      final Function1<Object, String> _function = (Object p) -> {
        return NameHelper.prettyObjectName(p);
      };
      final String args = IterableExtensions.join(ListExtensions.<Object, String>map(mseoccurrence.getParameters(), _function), ", ");
      String _prettyFrameName = this.prettyFrameName(mse, implicit);
      String _plus = (_prettyFrameName + "(");
      String _plus_1 = (_plus + args);
      return (_plus_1 + ")");
    }
    return null;
  }

  protected String prettyFrameName(final MSE mse, final boolean implicit) {
    if ((mse != null)) {
      EObject caller = mse.getCaller();
      final String objectName = NameHelper.prettyObjectName(caller);
      String _xifexpression = null;
      if (implicit) {
        EOperation _action = mse.getAction();
        String _name = null;
        if (_action!=null) {
          _name=_action.getName();
        }
        _xifexpression = (_name + "_implicitStep");
      } else {
        EOperation _action_1 = mse.getAction();
        String _name_1 = null;
        if (_action_1!=null) {
          _name_1=_action_1.getName();
        }
        _xifexpression = _name_1;
      }
      final String opName = _xifexpression;
      final String prettyName = ((objectName + "#") + opName);
      return prettyName;
    }
    return null;
  }

  protected GenericSequentialModelDebugger.MSEFrameInformation getMSEFrameInformation(final Step<?> step) {
    final MSEOccurrence mseOccurrence = step.getMseoccurrence();
    final EObject container = step.eContainer();
    String prettyName = "";
    EObject caller = null;
    if ((mseOccurrence != null)) {
      caller = mseOccurrence.getMse().getCaller();
      prettyName = this.prettyFrameName(mseOccurrence, false);
    } else {
      if ((((container != null) && (container instanceof Step<?>)) && (((Step<?>) container).getMseoccurrence() != null))) {
        final MSE parentMSE = ((Step<?>) container).getMseoccurrence().getMse();
        caller = parentMSE.getCaller();
        String _prettyFrameName = this.prettyFrameName(parentMSE, true);
        String _plus = (_prettyFrameName + "()");
        prettyName = _plus;
      } else {
        if ((step instanceof ParallelStep<?, ?>)) {
          caller = step;
          StringConcatenation _builder = new StringConcatenation();
          _builder.append(" ");
          String _stepName = StepHelper.getStepName(step);
          _builder.append(_stepName, " ");
          _builder.append(" (");
          final Function1<MSE, String> _function = (MSE it) -> {
            return it.getName();
          };
          String _join = IterableExtensions.join(ListExtensions.<MSE, String>map(StepHelper.getMSEs(step), _function), ", ");
          _builder.append(_join, " ");
          _builder.append(")");
          prettyName = _builder.toString();
        } else {
          caller = step;
          prettyName = "Unknown step";
        }
      }
    }
    return new GenericSequentialModelDebugger.MSEFrameInformation(caller, prettyName);
  }

  @Override
  public void updateStack(final String threadName, final EObject instruction) {
    final Deque<Step<?>> virtualStack = new ArrayDeque<Step<?>>();
    for (final GenericSequentialModelDebugger.ToPushPop m : this.toPushPop) {
      if (m.push) {
        virtualStack.push(m.step);
      } else {
        boolean _isEmpty = virtualStack.isEmpty();
        if (_isEmpty) {
          this.popStackFrame(threadName);
        } else {
          virtualStack.pop();
        }
      }
    }
    final Iterator<Step<?>> iterator = virtualStack.descendingIterator();
    while (iterator.hasNext()) {
      {
        final Step<?> step = iterator.next();
        final GenericSequentialModelDebugger.MSEFrameInformation info = this.getMSEFrameInformation(step);
        this.pushStackFrame(threadName, info.prettyLabel, info.caller, info.caller);
      }
    }
    this.setCurrentInstruction(threadName, instruction);
    this.toPushPop.clear();
  }

  @Override
  public void updateData(final String threadName, final EObject instruction) {
    EObject realInstruction = instruction;
    if ((instruction == null)) {
      this.updateVariables(threadName);
      this.updateStack(threadName, null);
      return;
    }
    if ((instruction instanceof Step<?>)) {
      final Step<?> step = ((Step<?>) instruction);
      MSEOccurrence _mseoccurrence = step.getMseoccurrence();
      boolean _tripleNotEquals = (_mseoccurrence != null);
      if (_tripleNotEquals) {
        realInstruction = step.getMseoccurrence().getMse().getCaller();
      }
    } else {
      if ((instruction instanceof MSEOccurrence)) {
        realInstruction = ((MSEOccurrence)instruction).getMse().getCaller();
      }
    }
    super.updateData(threadName, realInstruction);
  }

  @Override
  public boolean shouldBreak(final EObject instruction) {
    if ((instruction instanceof Step<?>)) {
      return this.shouldBreakStep(((Step<?>) instruction));
    } else {
      boolean _equals = Objects.equal(instruction, GenericSequentialModelDebugger.FAKE_INSTRUCTION);
      if (_equals) {
        return true;
      }
    }
    return false;
  }

  private boolean hasRegularBreakpointTrue(final EObject o) {
    EObject target = o;
    final Resource res = o.eResource();
    if ((res != null)) {
      final MelangeResourceImpl mr = IterableExtensions.<MelangeResourceImpl>head(Iterables.<MelangeResourceImpl>filter(res.getResourceSet().getResources(), MelangeResourceImpl.class));
      if ((mr != null)) {
        final String uriFragment = res.getURIFragment(o);
        target = mr.getWrappedResource().getEObject(uriFragment);
      }
    }
    return (super.shouldBreak(target) && ((Boolean.valueOf(((String) this.getBreakpointAttributes(target, GemocBreakpoint.BREAK_ON_LOGICAL_STEP)))).booleanValue() || 
      (Boolean.valueOf(((String) this.getBreakpointAttributes(target, GemocBreakpoint.BREAK_ON_MSE_OCCURRENCE)))).booleanValue()));
  }

  private boolean shouldBreakStep(final Step<?> step) {
    boolean _shouldBreakPredicates = this.shouldBreakPredicates(this.engine, step);
    if (_shouldBreakPredicates) {
      return true;
    }
    MSEOccurrence _mseoccurrence = step.getMseoccurrence();
    boolean _tripleNotEquals = (_mseoccurrence != null);
    if (_tripleNotEquals) {
      final MSE mse = step.getMseoccurrence().getMse();
      boolean _hasRegularBreakpointTrue = this.hasRegularBreakpointTrue(mse);
      if (_hasRegularBreakpointTrue) {
        return true;
      }
      final EObject caller = mse.getCaller();
      boolean _hasRegularBreakpointTrue_1 = this.hasRegularBreakpointTrue(caller);
      if (_hasRegularBreakpointTrue_1) {
        return true;
      }
    }
    return false;
  }

  @Override
  public EObject getNextInstruction(final String threadName, final EObject currentInstruction, final IDSLDebugger.Stepping stepping) {
    return GenericSequentialModelDebugger.FAKE_INSTRUCTION;
  }

  @Override
  public void engineStarted(final IExecutionEngine<?> executionEngine) {
    this.spawnRunningThread(this.threadName, this.engine.getExecutionContext().getResourceModel().getContents().get(0));
  }

  @Override
  public void engineStopped(final IExecutionEngine<?> engine) {
    boolean _isTerminated = this.isTerminated(this.threadName);
    boolean _not = (!_isTerminated);
    if (_not) {
      this.terminated(this.threadName);
    }
  }

  @Override
  public void aboutToExecuteStep(final IExecutionEngine<?> executionEngine, final Step<?> step) {
    final GenericSequentialModelDebugger.ToPushPop stackModification = new GenericSequentialModelDebugger.ToPushPop(step, true);
    this.toPushPop.add(stackModification);
    final boolean shallcontinue = this.control(this.threadName, step);
    if ((!shallcontinue)) {
      throw new EngineStoppedException("Debug thread has stopped.");
    }
  }

  @Override
  public void stepExecuted(final IExecutionEngine<?> engine, final Step<?> step) {
    final GenericSequentialModelDebugger.ToPushPop stackModification = new GenericSequentialModelDebugger.ToPushPop(step, false);
    this.toPushPop.add(stackModification);
  }

  @Override
  public void engineAboutToStop(final IExecutionEngine<?> engine) {
    this.executionTerminated = true;
  }

  @Override
  public void terminate() {
    super.terminate();
    this.engine.stop();
  }
}
