/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.builder.ModelTypingSpaceBuilder;
import fr.inria.diverse.melange.builder.OperatorBuilder;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.lib.ModelUtils;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.ClassBinding;
import fr.inria.diverse.melange.metamodel.melange.Import;
import fr.inria.diverse.melange.metamodel.melange.Inheritance;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.LanguageOperator;
import fr.inria.diverse.melange.metamodel.melange.MelangePackage;
import fr.inria.diverse.melange.metamodel.melange.Merge;
import fr.inria.diverse.melange.metamodel.melange.Operator;
import fr.inria.diverse.melange.metamodel.melange.PackageBinding;
import fr.inria.diverse.melange.metamodel.melange.PropertyBinding;
import fr.inria.diverse.melange.metamodel.melange.Slice;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.Stack;
import java.util.function.Consumer;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class AspectCopier2 {
  @Inject
  private ModelUtils modelUtils;

  @Inject
  private AspectRenamer renamer;

  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;

  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;

  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;

  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;

  @Inject
  @Extension
  private AspectExtensions aspectExtension;

  @Inject
  private ModelTypingSpaceBuilder modelTypingSpaceBuilder;

  /**
   * Copy all aspects from {@link sourceLang} into the Eclipse
   * project of {@link targetLang}
   */
  public SetMultimap<String, String> copyAspect(final Language sourceLang, final Language targetLang) {
    Stack<List<PackageBinding>> _stack = new Stack<List<PackageBinding>>();
    return this.copyAspect(sourceLang, targetLang, _stack, null);
  }

  private SetMultimap<String, String> copyAspect(final Language l, final Language targetLang, final Stack<List<PackageBinding>> stack, final List<EClass> sliceClasses) {
    final SetMultimap<String, String> realPkgNames = HashMultimap.<String, String>create();
    final Consumer<LanguageOperator> _function = (LanguageOperator depLang) -> {
      if ((depLang instanceof Inheritance)) {
        final SetMultimap<String, String> deepMapping = this.copyAspect(((Inheritance)depLang).getTargetLanguage(), targetLang, stack, sliceClasses);
        realPkgNames.putAll(deepMapping);
      } else {
        if ((depLang instanceof Slice)) {
          stack.push(((Slice)depLang).getMappingRules());
          final List<OperatorBuilder<? extends Operator>> opBuilders = this.modelTypingSpaceBuilder.findBuilder(l).getSubBuilders();
          final Function1<OperatorBuilder<? extends Operator>, Boolean> _function_1 = (OperatorBuilder<? extends Operator> it) -> {
            return Boolean.valueOf(Objects.equal(it.source, depLang));
          };
          final OperatorBuilder<? extends Operator> sliceBuilder = IterableExtensions.<OperatorBuilder<? extends Operator>>findFirst(opBuilders, _function_1);
          final Function1<EPackage, List<EClass>> _function_2 = (EPackage it) -> {
            return this._ecoreExtensions.getAllClasses(it);
          };
          final List<EClass> opSliceClasses = IterableExtensions.<EClass>toList(Iterables.<EClass>concat(IterableExtensions.<EPackage, List<EClass>>map(sliceBuilder.getModel(), _function_2)));
          final List<EClass> renamedSlice = this.intersection(sliceClasses, opSliceClasses);
          final List<EClass> newSlice = this.reverseRenaming(renamedSlice, ((Slice)depLang).getTargetLanguage(), ((Slice)depLang).getMappingRules());
          final SetMultimap<String, String> deepMapping_1 = this.copyAspect(((Slice)depLang).getTargetLanguage(), targetLang, stack, newSlice);
          stack.pop();
          final Consumer<String> _function_3 = (String pkgName) -> {
            final Set<String> emfPkgs = deepMapping_1.get(pkgName);
            final String newPkgName = this.renameWith(pkgName, ((Slice)depLang).getMappingRules());
            realPkgNames.putAll(newPkgName, emfPkgs);
          };
          deepMapping_1.keySet().forEach(_function_3);
        } else {
          if ((depLang instanceof Merge)) {
            stack.push(((Merge)depLang).getMappingRules());
            final SetMultimap<String, String> deepMapping_2 = this.copyAspect(((Merge)depLang).getTargetLanguage(), targetLang, stack, sliceClasses);
            stack.pop();
            final Consumer<String> _function_4 = (String pkgName) -> {
              final Set<String> emfPkgs = deepMapping_2.get(pkgName);
              final String newPkgName = this.renameWith(pkgName, ((Merge)depLang).getMappingRules());
              realPkgNames.putAll(newPkgName, emfPkgs);
            };
            deepMapping_2.keySet().forEach(_function_4);
          }
        }
      }
    };
    Iterables.<LanguageOperator>filter(l.getOperators(), LanguageOperator.class).forEach(_function);
    realPkgNames.putAll(this.mappingFromSyntax(l));
    realPkgNames.putAll(this.mappingFromLanguageOp(l));
    this.copyLocalAspects(l, targetLang, realPkgNames, stack, sliceClasses);
    return realPkgNames;
  }

  /**
   * Get package renaming from 'with' operators.
   * 
   * Return the associations of EPackage with generated Java Class
   * (use full qualified names)
   */
  private SetMultimap<String, String> mappingFromSyntax(final Language sourceLang) {
    final HashMultimap<String, String> res = HashMultimap.<String, String>create();
    final Consumer<Import> _function = (Import ecore) -> {
      String _elvis = null;
      String _head = IterableExtensions.<String>head(ecore.getGenmodelUris());
      if (_head != null) {
        _elvis = _head;
      } else {
        String _substring = ecore.getEcoreUri().substring(0, ecore.getEcoreUri().lastIndexOf("."));
        String _plus = (_substring + ".genmodel");
        _elvis = _plus;
      }
      final String genModelUri = _elvis;
      final GenModel genmodel = this.modelUtils.loadGenmodel(genModelUri);
      final Consumer<GenPackage> _function_1 = (GenPackage genPkg) -> {
        final String emfFqn = genPkg.getQualifiedPackageName();
        final String packageFqn = this.renameWith(this._ecoreExtensions.getUniqueId(genPkg.getEcorePackage()), ecore.getMappingRules());
        res.put(packageFqn, emfFqn);
      };
      this._ecoreExtensions.getAllGenPkgs(genmodel).forEach(_function_1);
    };
    Iterables.<Import>filter(sourceLang.getOperators(), Import.class).forEach(_function);
    return res;
  }

  /**
   * Get package renaming from 'merge', 'slice' and 'inherits' operators.
   * 
   * Return the associations of EPackage with generated Java Class
   * (use full qualified names)
   */
  private SetMultimap<String, String> mappingFromLanguageOp(final Language sourceLanguage) {
    final HashMultimap<String, String> res = HashMultimap.<String, String>create();
    final Consumer<LanguageOperator> _function = (LanguageOperator it) -> {
      res.putAll(this.mappingFromLanguageOp(it));
    };
    Iterables.<LanguageOperator>filter(sourceLanguage.getOperators(), LanguageOperator.class).forEach(_function);
    return res;
  }

  /**
   * Get package renaming from {@link op}.
   * 
   * Return the associations of EPackage with generated Java Class
   * (use full qualified names)
   */
  private SetMultimap<String, String> mappingFromLanguageOp(final LanguageOperator op) {
    final HashMultimap<String, String> res = HashMultimap.<String, String>create();
    final Consumer<EPackage> _function = (EPackage pkg) -> {
      QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(op.getOwningLanguage()).toLowerCase();
      String _plus = (_lowerCase + ".");
      String _uniqueId = this._ecoreExtensions.getUniqueId(pkg);
      final String emfFqn = (_plus + _uniqueId);
      String _switchResult = null;
      boolean _matched = false;
      if (op instanceof Inheritance) {
        _matched=true;
        _switchResult = this._ecoreExtensions.getUniqueId(pkg);
      }
      if (!_matched) {
        if (op instanceof Slice) {
          _matched=true;
          _switchResult = this.renameWith(this._ecoreExtensions.getUniqueId(pkg), ((Slice)op).getMappingRules());
        }
      }
      if (!_matched) {
        if (op instanceof Merge) {
          _matched=true;
          _switchResult = this.renameWith(this._ecoreExtensions.getUniqueId(pkg), ((Merge)op).getMappingRules());
        }
      }
      final String packageFqn = _switchResult;
      res.put(packageFqn, emfFqn);
    };
    this._modelingElementExtensions.getPkgs(op.getOwningLanguage().getSyntax()).forEach(_function);
    return res;
  }

  /**
   * Return the new name for the EPackage {@link pkFng}.
   * If nothing match in {@link rules}, return {@link pkFng}.
   */
  private String renameWith(final String pkgFqn, final List<PackageBinding> rules) {
    final Function1<PackageBinding, Boolean> _function = (PackageBinding it) -> {
      String _from = it.getFrom();
      return Boolean.valueOf(Objects.equal(_from, pkgFqn));
    };
    final PackageBinding firstMatch = IterableExtensions.<PackageBinding>findFirst(rules, _function);
    if ((firstMatch != null)) {
      return firstMatch.getTo();
    }
    return pkgFqn;
  }

  /**
   * Copy aspects declared in {@link currentLang} into the project of {@link targetLang}
   */
  private void copyLocalAspects(final Language currentLang, final Language targetLang, final SetMultimap<String, String> realPkgNames, final Stack<List<PackageBinding>> stack, final List<EClass> sliceClasses) {
    try {
      List<Aspect> _xifexpression = null;
      if ((sliceClasses != null)) {
        final Function1<Aspect, Boolean> _function = (Aspect asp) -> {
          final Function1<EClass, Boolean> _function_1 = (EClass it) -> {
            QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(it);
            EClass _aspectedClass = asp.getAspectedClass();
            QualifiedName _fullyQualifiedName_1 = null;
            if (_aspectedClass!=null) {
              _fullyQualifiedName_1=this._iQualifiedNameProvider.getFullyQualifiedName(_aspectedClass);
            }
            return Boolean.valueOf(Objects.equal(_fullyQualifiedName, _fullyQualifiedName_1));
          };
          return Boolean.valueOf(IterableExtensions.<EClass>exists(sliceClasses, _function_1));
        };
        _xifexpression = IterableExtensions.<Aspect>toList(IterableExtensions.<Aspect>filter(this._languageExtensions.getLocalSemantics(currentLang), _function));
      } else {
        _xifexpression = this._languageExtensions.getLocalSemantics(currentLang);
      }
      final List<Aspect> localAspSliced = _xifexpression;
      final Function1<Aspect, JvmTypeReference> _function_1 = (Aspect it) -> {
        return it.getSource().getAspectTypeRef();
      };
      final List<JvmTypeReference> localWeaveRef = ListExtensions.<Aspect, JvmTypeReference>map(localAspSliced, _function_1);
      final Set<IFile> toBeCopied = CollectionLiterals.<IFile>newHashSet();
      final IWorkspaceRoot ws = ResourcesPlugin.getWorkspace().getRoot();
      ws.accept(new IResourceVisitor() {
        @Override
        public boolean visit(final IResource resource) throws CoreException {
          if ((resource instanceof IFile)) {
            final Function1<JvmTypeReference, Boolean> _function = (JvmTypeReference ref) -> {
              String _replace = ref.getIdentifier().replace(".", "/");
              final String pattern1 = (_replace + ".java");
              String _replace_1 = AspectCopier2.this.getContextFqn(ref).replace(".", "/");
              final String pattern2 = (_replace_1 + ".java");
              String _replace_2 = AspectCopier2.this.getPropertiesFqn(ref).replace(".", "/");
              final String pattern3 = (_replace_2 + ".java");
              return Boolean.valueOf((((((((IFile)resource).getLocationURI().getPath().endsWith(("src-gen/" + pattern1)) || ((IFile)resource).getLocationURI().getPath().endsWith(("src-gen/" + pattern2))) || ((IFile)resource).getLocationURI().getPath().endsWith(("src-gen/" + pattern3))) || ((IFile)resource).getLocationURI().getPath().endsWith(("xtend-gen/" + pattern1))) || ((IFile)resource).getLocationURI().getPath().endsWith(("xtend-gen/" + pattern2))) || ((IFile)resource).getLocationURI().getPath().endsWith(("xtend-gen/" + pattern3))));
            };
            final JvmTypeReference firstMatch = IterableExtensions.<JvmTypeReference>findFirst(localWeaveRef, _function);
            if ((firstMatch != null)) {
              toBeCopied.add(((IFile)resource));
            }
            return false;
          }
          return true;
        }
      });
      final IProject targetProject = ws.getProject(this._languageExtensions.getExternalRuntimeName(targetLang));
      String _replace = this._languageExtensions.getAspectsNamespace(targetLang).replace(".", "/");
      String _plus = ("src-gen/" + _replace);
      final IFolder targetAspectFolder = targetProject.getFolder(_plus);
      this.createFolder(targetAspectFolder);
      final Consumer<IFile> _function_2 = (IFile file) -> {
        try {
          final IFile destFile = targetAspectFolder.getFile(file.getName());
          destFile.create(file.getContents(), true, null);
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      };
      toBeCopied.forEach(_function_2);
      final List<PackageBinding> finalRenaming = AspectCopier2.flatten(stack);
      final ArrayList<PackageBinding> toRemove = CollectionLiterals.<PackageBinding>newArrayList();
      final ArrayList<PackageBinding> toAdd = CollectionLiterals.<PackageBinding>newArrayList();
      final Consumer<String> _function_3 = (String pkgName) -> {
        final Set<String> possibleEmfNames = realPkgNames.get(pkgName);
        final Function1<PackageBinding, Boolean> _function_4 = (PackageBinding it) -> {
          String _from = it.getFrom();
          return Boolean.valueOf(Objects.equal(_from, pkgName));
        };
        final PackageBinding firstMatch = IterableExtensions.<PackageBinding>findFirst(finalRenaming, _function_4);
        if ((firstMatch != null)) {
          final Consumer<String> _function_5 = (String emfName) -> {
            final PackageBinding copy = EcoreUtil.<PackageBinding>copy(firstMatch);
            copy.setFrom(emfName);
            toAdd.add(copy);
          };
          possibleEmfNames.forEach(_function_5);
          toRemove.add(firstMatch);
        } else {
          final Consumer<String> _function_6 = (String emfName) -> {
            EObject _create = MelangePackage.eINSTANCE.getEFactoryInstance().create(MelangePackage.eINSTANCE.getPackageBinding());
            final PackageBinding newRule = ((PackageBinding) _create);
            newRule.setFrom(emfName);
            newRule.setTo(pkgName);
            toAdd.add(newRule);
          };
          possibleEmfNames.forEach(_function_6);
        }
      };
      realPkgNames.keySet().forEach(_function_3);
      finalRenaming.removeAll(toRemove);
      finalRenaming.addAll(toAdd);
      final Consumer<PackageBinding> _function_4 = (PackageBinding rule) -> {
        QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(targetLang).toLowerCase();
        String _plus_1 = (_lowerCase + ".");
        String _to = rule.getTo();
        String _plus_2 = (_plus_1 + _to);
        rule.setTo(_plus_2);
      };
      finalRenaming.forEach(_function_4);
      final Function1<Language, List<Aspect>> _function_5 = (Language it) -> {
        return this._languageExtensions.getLocalSemantics(it);
      };
      final Function1<Aspect, String> _function_6 = (Aspect it) -> {
        JvmType _type = it.getSource().getAspectTypeRef().getType();
        return ((JvmDeclaredType) _type).getPackageName();
      };
      final Set<String> sourceAspectNs = IterableExtensions.<String>toSet(IterableExtensions.<Aspect, String>map(Iterables.<Aspect>concat(IterableExtensions.<Language, List<Aspect>>map(this._languageExtensions.getAllDependencies(currentLang), _function_5)), _function_6));
      final RenamingRuleManager renamingManager = new RenamingRuleManager(finalRenaming, localAspSliced, this.aspectExtension);
      this.renamer.processRenaming(localAspSliced, targetLang, sourceAspectNs, CollectionLiterals.<RenamingRuleManager>newArrayList(renamingManager));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private String getContextFqn(final JvmTypeReference aspRef) {
    final String prefix = aspRef.getIdentifier().replaceAll("\\.", "/");
    final String targetCls = this.aspectExtension.getSimpleAspectAnnotationValue(aspRef);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(prefix);
    _builder.append(targetCls);
    _builder.append("AspectContext");
    return _builder.toString();
  }

  private String getPropertiesFqn(final JvmTypeReference aspRef) {
    final String prefix = aspRef.getIdentifier().replaceAll("\\.", "/");
    final String targetCls = this.aspectExtension.getSimpleAspectAnnotationValue(aspRef);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(prefix);
    _builder.append(targetCls);
    _builder.append("AspectProperties");
    return _builder.toString();
  }

  /**
   * Merge sequentially all stages of the {@link stack} to get the
   * final transitive renaming
   */
  public static List<PackageBinding> flatten(final Stack<List<PackageBinding>> stack) {
    final ArrayList<PackageBinding> res = CollectionLiterals.<PackageBinding>newArrayList();
    final Consumer<List<PackageBinding>> _function = (List<PackageBinding> rules) -> {
      AspectCopier2.apply(res, rules);
    };
    ListExtensions.<List<PackageBinding>>reverseView(stack).forEach(_function);
    return res;
  }

  /**
   * Merge {@link newRules} in {@link base} to have a transitive renaming
   */
  private static void apply(final List<PackageBinding> base, final List<PackageBinding> newRules) {
    final Function1<PackageBinding, Integer> _function = (PackageBinding it) -> {
      return Integer.valueOf(it.getFrom().length());
    };
    final Consumer<PackageBinding> _function_1 = (PackageBinding newRule) -> {
      final Function1<PackageBinding, Boolean> _function_2 = (PackageBinding it) -> {
        return Boolean.valueOf((Objects.equal(it.getTo(), newRule.getFrom()) || it.getTo().startsWith((newRule.getFrom() + "."))));
      };
      final Iterable<PackageBinding> toRename = IterableExtensions.<PackageBinding>filter(base, _function_2);
      final Consumer<PackageBinding> _function_3 = (PackageBinding baseRule) -> {
        baseRule.setTo(baseRule.getTo().replaceFirst(newRule.getFrom(), newRule.getTo()));
      };
      toRename.forEach(_function_3);
      final Function1<PackageBinding, Boolean> _function_4 = (PackageBinding it) -> {
        String _to = it.getTo();
        String _to_1 = newRule.getTo();
        return Boolean.valueOf(Objects.equal(_to, _to_1));
      };
      final PackageBinding toUpdate = IterableExtensions.<PackageBinding>findFirst(base, _function_4);
      if ((toUpdate != null)) {
        final Consumer<ClassBinding> _function_5 = (ClassBinding newCls) -> {
          final Function1<ClassBinding, Boolean> _function_6 = (ClassBinding it) -> {
            String _to = it.getTo();
            String _from = newCls.getFrom();
            return Boolean.valueOf(Objects.equal(_to, _from));
          };
          final ClassBinding clsRename = IterableExtensions.<ClassBinding>findFirst(toUpdate.getClasses(), _function_6);
          if ((clsRename != null)) {
            clsRename.setTo(newCls.getTo());
            final Consumer<PropertyBinding> _function_7 = (PropertyBinding newProp) -> {
              final Function1<PropertyBinding, Boolean> _function_8 = (PropertyBinding it) -> {
                String _to = it.getTo();
                String _from = newProp.getFrom();
                return Boolean.valueOf(Objects.equal(_to, _from));
              };
              final PropertyBinding propRename = IterableExtensions.<PropertyBinding>findFirst(clsRename.getProperties(), _function_8);
              if ((propRename != null)) {
                propRename.setTo(newProp.getTo());
              } else {
                clsRename.getProperties().add(newProp);
              }
            };
            newCls.getProperties().forEach(_function_7);
          } else {
            toUpdate.getClasses().add(newCls);
          }
        };
        newRule.getClasses().forEach(_function_5);
      }
      boolean _isEmpty = IterableExtensions.isEmpty(toRename);
      if (_isEmpty) {
        base.add(EcoreUtil.<PackageBinding>copy(newRule));
      }
    };
    ListExtensions.<PackageBinding>reverseView(IterableExtensions.<PackageBinding, Integer>sortBy(newRules, _function)).forEach(_function_1);
  }

  private void createFolder(final IFolder folder) {
    try {
      final IContainer parent = folder.getParent();
      if ((parent instanceof IFolder)) {
        this.createFolder(((IFolder)parent));
      }
      boolean _exists = folder.exists();
      boolean _not = (!_exists);
      if (_not) {
        folder.create(false, true, null);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Return the subset of l2 contained in l1
   * Return l2 if l1 is null
   * l1 is in the post mapping space
   * l2 is in the pre mapping space
   */
  private List<EClass> intersection(final List<EClass> l1, final List<EClass> l2) {
    if ((l1 == null)) {
      return l2;
    }
    final ArrayList<EClass> res = CollectionLiterals.<EClass>newArrayList();
    final Function1<EClass, Boolean> _function = (EClass cls2) -> {
      final Function1<EClass, Boolean> _function_1 = (EClass cls1) -> {
        String _uniqueId = this._ecoreExtensions.getUniqueId(cls1);
        String _uniqueId_1 = this._ecoreExtensions.getUniqueId(cls2);
        return Boolean.valueOf(Objects.equal(_uniqueId, _uniqueId_1));
      };
      return Boolean.valueOf(IterableExtensions.<EClass>exists(l1, _function_1));
    };
    Iterables.<EClass>addAll(res, 
      IterableExtensions.<EClass>filter(l2, _function));
    return res;
  }

  /**
   * Return classes from {@link origin} corresponding to {@link classes} through {@link mapping}
   */
  private List<EClass> reverseRenaming(final List<EClass> classes, final Language origin, final List<PackageBinding> mapping) {
    final ArrayList<EClass> res = CollectionLiterals.<EClass>newArrayList();
    final Iterable<EClass> originPool = this._modelingElementExtensions.getAllClasses(origin.getSyntax());
    final Consumer<EClass> _function = (EClass cls) -> {
      final String pkg = this._ecoreExtensions.getUniqueId(cls.getEPackage());
      final Function1<PackageBinding, Boolean> _function_1 = (PackageBinding pkgRule) -> {
        String _to = pkgRule.getTo();
        return Boolean.valueOf(Objects.equal(_to, pkg));
      };
      final PackageBinding pkgRule = IterableExtensions.<PackageBinding>findFirst(mapping, _function_1);
      EList<ClassBinding> _classes = null;
      if (pkgRule!=null) {
        _classes=pkgRule.getClasses();
      }
      ClassBinding _findFirst = null;
      if (_classes!=null) {
        final Function1<ClassBinding, Boolean> _function_2 = (ClassBinding clsRule) -> {
          String _to = clsRule.getTo();
          String _name = cls.getName();
          return Boolean.valueOf(Objects.equal(_to, _name));
        };
        _findFirst=IterableExtensions.<ClassBinding>findFirst(_classes, _function_2);
      }
      final ClassBinding clsRule = _findFirst;
      String _xifexpression = null;
      if ((clsRule != null)) {
        String _from = pkgRule.getFrom();
        String _plus = (_from + ".");
        String _from_1 = clsRule.getFrom();
        _xifexpression = (_plus + _from_1);
      } else {
        String _xifexpression_1 = null;
        if ((pkgRule != null)) {
          String _from_2 = pkgRule.getFrom();
          String _plus_1 = (_from_2 + ".");
          String _name = cls.getName();
          _xifexpression_1 = (_plus_1 + _name);
        } else {
          _xifexpression_1 = this._ecoreExtensions.getUniqueId(cls);
        }
        _xifexpression = _xifexpression_1;
      }
      final String realClsName = _xifexpression;
      final Function1<EClass, Boolean> _function_3 = (EClass originCls) -> {
        String _uniqueId = this._ecoreExtensions.getUniqueId(originCls);
        return Boolean.valueOf(Objects.equal(_uniqueId, realClsName));
      };
      final EClass originCls = IterableExtensions.<EClass>findFirst(originPool, _function_3);
      if ((originCls != null)) {
        res.add(originCls);
      }
    };
    classes.forEach(_function);
    return res;
  }
}
