/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.ast;

import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;
import com.google.inject.Inject;
import fr.inria.diverse.melange.builder.ModelTypingSpaceBuilder;
import fr.inria.diverse.melange.builder.OperatorBuilder;
import fr.inria.diverse.melange.eclipse.EclipseProjectHelper;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.lib.MatchingHelper;
import fr.inria.diverse.melange.metamodel.melange.Annotation;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.ClassBinding;
import fr.inria.diverse.melange.metamodel.melange.ExternalLanguage;
import fr.inria.diverse.melange.metamodel.melange.Import;
import fr.inria.diverse.melange.metamodel.melange.ImportDsl;
import fr.inria.diverse.melange.metamodel.melange.Inheritance;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.LanguageOperator;
import fr.inria.diverse.melange.metamodel.melange.Mapping;
import fr.inria.diverse.melange.metamodel.melange.MelangeFactory;
import fr.inria.diverse.melange.metamodel.melange.Merge;
import fr.inria.diverse.melange.metamodel.melange.Metamodel;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import fr.inria.diverse.melange.metamodel.melange.Operator;
import fr.inria.diverse.melange.metamodel.melange.PackageBinding;
import fr.inria.diverse.melange.metamodel.melange.Slice;
import fr.inria.diverse.melange.metamodel.melange.Weave;
import fr.inria.diverse.melange.utils.AspectCopier2;
import fr.inria.diverse.melange.utils.RenamingRuleManager;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.Stack;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gemoc.dsl.Dsl;
import org.eclipse.gemoc.dsl.Entry;
import org.eclipse.gemoc.dsl.impl.DslFactoryImpl;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.JvmUnknownTypeReference;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.validation.EObjectDiagnosticImpl;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * A collection of utilities around {@link Language}s
 */
@SuppressWarnings("all")
public class LanguageExtensions {
  @Inject
  @Extension
  private ASTHelper _aSTHelper;

  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;

  @Inject
  @Extension
  private MetamodelExtensions _metamodelExtensions;

  @Inject
  @Extension
  private NamingHelper _namingHelper;

  @Inject
  @Extension
  private AspectExtensions aspectExtension;

  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;

  @Inject
  @Extension
  private EclipseProjectHelper _eclipseProjectHelper;

  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;

  @Inject
  private AspectCopier2 copier2;

  @Inject
  private JvmTypesBuilder typesBuilder;

  @Inject
  private JvmTypeReferenceBuilder.Factory builderFactory;

  @Inject
  private MatchingHelper matchingHelper;

  @Inject
  private ModelTypingSpaceBuilder modelTypingSpaceBuilder;

  @Inject
  private EclipseProjectHelper eclipseHelper;

  private static final Logger log = Logger.getLogger(LanguageExtensions.class);

  private static final String ASPECT_MAIN_ANNOTATION = "fr.inria.diverse.k3.al.annotationprocessor.Main";

  /**
   * Checks whether {@link l} or one of its dependencies (languages it depends
   * on) cannot be built
   */
  public boolean isInError(final Language l) {
    final ArrayList<Language> langs = CollectionLiterals.<Language>newArrayList();
    langs.add(l);
    Set<Language> _allDependencies = this.getAllDependencies(l);
    Iterables.<Language>addAll(langs, _allDependencies);
    final Function1<Language, Boolean> _function = (Language lang) -> {
      final Function1<EObjectDiagnosticImpl, Boolean> _function_1 = (EObjectDiagnosticImpl it) -> {
        return Boolean.valueOf(this._ecoreExtensions.isContainedBy(it.getProblematicObject(), lang));
      };
      return Boolean.valueOf(IterableExtensions.<EObjectDiagnosticImpl>exists(Iterables.<EObjectDiagnosticImpl>filter(lang.eResource().getErrors(), EObjectDiagnosticImpl.class), _function_1));
    };
    return IterableExtensions.<Language>exists(langs, _function);
  }

  /**
   * Returns the set of immediate super-languages of {@link l}, excluding
   * {@link l}
   */
  public Set<Language> getSuperLanguages(final Language l) {
    final Function1<Inheritance, Language> _function = (Inheritance it) -> {
      return it.getTargetLanguage();
    };
    return IterableExtensions.<Language>toSet(IterableExtensions.<Inheritance, Language>map(Iterables.<Inheritance>filter(l.getOperators(), Inheritance.class), _function));
  }

  /**
   * Returns the transitive set of all the super-languages of {@link l},
   * excluding {@link l}
   */
  public Set<Language> getAllSuperLanguages(final Language l) {
    final HashSet<Language> ret = CollectionLiterals.<Language>newHashSet();
    Set<Language> _superLanguages = this.getSuperLanguages(l);
    Iterables.<Language>addAll(ret, _superLanguages);
    final Function1<Language, Set<Language>> _function = (Language it) -> {
      return this.getAllSuperLanguages(it);
    };
    Iterable<Language> _flatten = Iterables.<Language>concat(IterableExtensions.<Language, Set<Language>>map(this.getSuperLanguages(l), _function));
    Iterables.<Language>addAll(ret, _flatten);
    return ret;
  }

  /**
   * Returns the set of all {@link Language}s {@code l} depends on, ie. all
   * the languages from which it is derived through {@link Inheritance},
   * {@link Merge}, or {@link Slice}
   */
  public Set<Language> getAllDependencies(final Language l) {
    final HashSet<Language> ret = CollectionLiterals.<Language>newHashSet();
    final Function1<LanguageOperator, Language> _function = (LanguageOperator it) -> {
      return it.getTargetLanguage();
    };
    Iterable<Language> _map = IterableExtensions.<LanguageOperator, Language>map(Iterables.<LanguageOperator>filter(l.getOperators(), LanguageOperator.class), _function);
    Iterables.<Language>addAll(ret, _map);
    final Function1<LanguageOperator, Set<Language>> _function_1 = (LanguageOperator it) -> {
      return this.getAllDependencies(it.getTargetLanguage());
    };
    Iterable<Language> _flatten = Iterables.<Language>concat(IterableExtensions.<LanguageOperator, Set<Language>>map(Iterables.<LanguageOperator>filter(l.getOperators(), LanguageOperator.class), _function_1));
    Iterables.<Language>addAll(ret, _flatten);
    return ret;
  }

  /**
   * Checks whether the given {@link Language} {@code l} is correctly defined
   * and can be processed
   */
  public boolean isValid(final Language l) {
    return ((((!StringExtensions.isNullOrEmpty(l.getName())) && (l.getSyntax() != null)) && this._metamodelExtensions.isValid(l.getSyntax())) && IterableExtensions.<Aspect>forall(l.getSemantics(), ((Function1<Aspect, Boolean>) (Aspect it) -> {
      return Boolean.valueOf(this.aspectExtension.isValid(it));
    })));
  }

  /**
   * Checks whether the given {@link Language} {@code l} is correctly defined
   * and can be typed
   */
  public boolean isTypable(final Language l) {
    return ((((!StringExtensions.isNullOrEmpty(l.getName())) && (l.getSyntax() != null)) && (!IterableExtensions.isEmpty(IterableExtensions.<EPackage>filterNull(this._modelingElementExtensions.getPkgs(l.getSyntax()))))) && IterableExtensions.<Aspect>forall(l.getSemantics(), ((Function1<Aspect, Boolean>) (Aspect it) -> {
      return Boolean.valueOf(this.aspectExtension.isValid(it));
    })));
  }

  /**
   * Checks whether the given {@link Language} {@code l} has a valid syntax
   * definition (either directly defined or inherited it from one of its
   * dependencies).
   */
  public boolean hasSyntax(final Language l) {
    return ((!IterableExtensions.isEmpty(Iterables.<Import>filter(l.getOperators(), Import.class))) || IterableExtensions.<Language>exists(this.getAllDependencies(l), ((Function1<Language, Boolean>) (Language it) -> {
      return Boolean.valueOf(this.hasSyntax(it));
    })));
  }

  /**
   * Returns the list of all {@link JvmTypeReference}s pointing to the aspects
   * woven on the {@link Language} {@code l}, ordered by priority, without
   * duplicates.
   * 
   * The priority order is:
   * <ul>
   *   <li>Aspects explicitly defined in {@code l} declaration, in
   *   top->bottom order</li>
   *   <li>From {@link Merge} and {@link Slice} relations, in top->bottom
   *   order</li>
   *   <li>From {@link Inheritance} relations, in the left->right order</li>
   * </ul>
   */
  public List<Aspect> getAllAspects(final Language l) {
    final HashSet<Aspect> res = CollectionLiterals.<Aspect>newHashSet();
    List<Aspect> _localSemantics = this.getLocalSemantics(l);
    Iterables.<Aspect>addAll(res, _localSemantics);
    final Function1<Operator, List<Aspect>> _function = (Operator op) -> {
      List<Aspect> _xifexpression = null;
      if ((op instanceof Slice)) {
        _xifexpression = this.getAllAspects(((Slice)op).getTargetLanguage());
      } else {
        List<Aspect> _xifexpression_1 = null;
        if ((op instanceof Merge)) {
          _xifexpression_1 = this.getAllAspects(((Merge)op).getTargetLanguage());
        } else {
          _xifexpression_1 = CollectionLiterals.<Aspect>newArrayList();
        }
        _xifexpression = _xifexpression_1;
      }
      return _xifexpression;
    };
    Iterable<Aspect> _flatten = Iterables.<Aspect>concat(ListExtensions.<Operator, List<Aspect>>map(l.getOperators(), _function));
    Iterables.<Aspect>addAll(res, _flatten);
    final Function1<Inheritance, List<Aspect>> _function_1 = (Inheritance it) -> {
      return this.getAllAspects(it.getTargetLanguage());
    };
    Iterable<Aspect> _flatten_1 = Iterables.<Aspect>concat(IterableExtensions.<Inheritance, List<Aspect>>map(Iterables.<Inheritance>filter(l.getOperators(), Inheritance.class), _function_1));
    Iterables.<Aspect>addAll(res, _flatten_1);
    return IterableExtensions.<Aspect>toList(res);
  }

  /**
   * Makes no sense to me :) Comment that later on :/
   */
  public List<Aspect> allSemantics(final Language l) {
    boolean _isGeneratedByMelange = this.isGeneratedByMelange(l);
    if (_isGeneratedByMelange) {
      return l.getSemantics();
    } else {
      final ArrayList<Aspect> res = CollectionLiterals.<Aspect>newArrayList();
      final Consumer<Aspect> _function = (Aspect a1) -> {
        if (((!IterableExtensions.<Aspect>exists(res, ((Function1<Aspect, Boolean>) (Aspect a2) -> {
          String _identifier = a2.getAspectTypeRef().getIdentifier();
          String _identifier_1 = a1.getAspectTypeRef().getIdentifier();
          return Boolean.valueOf(Objects.equal(_identifier, _identifier_1));
        }))) && ((!this.aspectExtension.hasAspectAnnotation(a1)) || IterableExtensions.<EClass>exists(this._ecoreExtensions.getAllClasses(IterableExtensions.<EPackage>head(this._modelingElementExtensions.getPkgs(l.getSyntax()))), ((Function1<EClass, Boolean>) (EClass cls) -> {
          String _name = cls.getName();
          EClass _aspectedClass = a1.getAspectedClass();
          String _name_1 = null;
          if (_aspectedClass!=null) {
            _name_1=_aspectedClass.getName();
          }
          return Boolean.valueOf(Objects.equal(_name, _name_1));
        }))))) {
          res.add(a1);
        }
      };
      ListExtensions.<Aspect>reverseView(l.getSemantics()).forEach(_function);
      return res;
    }
  }

  /**
   * Returns the list of {@link Aspect}s created from {@link Weave} operators.
   * The order is the same as in the {@link Language} declaration
   */
  public List<Aspect> getLocalSemantics(final Language l) {
    final Function1<Aspect, Boolean> _function = (Aspect asp) -> {
      Language _owningLanguage = asp.getOwningLanguage();
      Language _owningLanguage_1 = asp.getSource().getOwningLanguage();
      return Boolean.valueOf((_owningLanguage == _owningLanguage_1));
    };
    return IterableExtensions.<Aspect>toList(IterableExtensions.<Aspect>filter(l.getSemantics(), _function));
  }

  /**
   * Returns the set of {@link Aspect}s of the {@link Language} {@code l}
   * that are woven on the {@link EClass} {@code cls} of its syntax, or on
   * one of its super-classes.
   */
  public Set<Aspect> findAspectsOn(final Language l, final EClass cls) {
    final Function1<Aspect, Boolean> _function = (Aspect asp) -> {
      boolean _and = false;
      EClass _aspectedClass = asp.getAspectedClass();
      String _name = null;
      if (_aspectedClass!=null) {
        _name=_aspectedClass.getName();
      }
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(_name);
      boolean _not = (!_isNullOrEmpty);
      if (!_not) {
        _and = false;
      } else {
        boolean _or = false;
        EClass _aspectedClass_1 = asp.getAspectedClass();
        QualifiedName _fullyQualifiedName = null;
        if (_aspectedClass_1!=null) {
          _fullyQualifiedName=this._iQualifiedNameProvider.getFullyQualifiedName(_aspectedClass_1);
        }
        QualifiedName _fullyQualifiedName_1 = this._iQualifiedNameProvider.getFullyQualifiedName(cls);
        boolean _equals = Objects.equal(_fullyQualifiedName, _fullyQualifiedName_1);
        if (_equals) {
          _or = true;
        } else {
          final Function1<EClass, Boolean> _function_1 = (EClass it) -> {
            EClass _aspectedClass_2 = asp.getAspectedClass();
            QualifiedName _fullyQualifiedName_2 = null;
            if (_aspectedClass_2!=null) {
              _fullyQualifiedName_2=this._iQualifiedNameProvider.getFullyQualifiedName(_aspectedClass_2);
            }
            QualifiedName _fullyQualifiedName_3 = this._iQualifiedNameProvider.getFullyQualifiedName(it);
            return Boolean.valueOf(Objects.equal(_fullyQualifiedName_2, _fullyQualifiedName_3));
          };
          boolean _exists = IterableExtensions.<EClass>exists(cls.getEAllSuperTypes(), _function_1);
          _or = _exists;
        }
        _and = _or;
      }
      return Boolean.valueOf(_and);
    };
    return IterableExtensions.<Aspect>toSet(IterableExtensions.<Aspect>filter(this.allSemantics(l), _function));
  }

  /**
   * Checks whether the given {@link Language} {@code l} implements the
   * interface described by the {@link ModelType} {@code mt}.
   * 
   * @see MatchingHelper#match
   */
  public boolean doesImplement(final Language l, final ModelType mt) {
    final Function1<Mapping, Boolean> _function = (Mapping it) -> {
      ModelType _to = it.getTo();
      return Boolean.valueOf(Objects.equal(_to, mt));
    };
    return this.matchingHelper.match(
      IterableExtensions.<EPackage>toList(this._modelingElementExtensions.getPkgs(l.getSyntax())), 
      IterableExtensions.<EPackage>toList(this._modelingElementExtensions.getPkgs(mt)), 
      IterableExtensions.<Mapping>findFirst(l.getMappings(), _function));
  }

  /**
   * Checks whether the given {@link Language} {@code l} has an adapter
   * for the {@link EClassifier} {@code cls} towards the {@link ModelType}
   * {@code mt}, ie. whether {@code l} implements {@code mt} and {@code cls}
   * is common to {@code l} and {@code mt}.
   */
  public boolean hasAdapterFor(final Language l, final ModelType mt, final EClassifier cls) {
    return this.hasAdapterFor(l, mt, this._ecoreExtensions.getUniqueId(cls));
  }

  /**
   * Checks whether the given {@link Language} {@code l} has an adapter
   * for the {@link EClassifier} whose name is {@code find} towards the
   * {@link ModelType} {@code mt}, ie. whether {@code l} implements {@code mt}
   * and {@code find} is common to {@code l} and {@code mt}.
   * 
   * @param find
   * 		Simple or qualified EClass name
   */
  public boolean hasAdapterFor(final Language l, final ModelType mt, final String find) {
    final EClass syntaxFind = this._modelingElementExtensions.findClass(l.getSyntax(), find);
    return (((IterableExtensions.<ModelType>exists(l.getImplements(), ((Function1<ModelType, Boolean>) (ModelType it) -> {
      String _name = it.getName();
      String _name_1 = mt.getName();
      return Boolean.valueOf(Objects.equal(_name, _name_1));
    })) && (syntaxFind != null)) && (this._modelingElementExtensions.findClass(mt, find) != null)) && this._ecoreExtensions.isAbstractable(syntaxFind));
  }

  /**
   * Generates and serializes the Ecore file defining the syntax of the
   * {@link Language} {@code l}.
   * 
   * @see #getExternalEcoreUri
   * @see #getExternalPackageUri
   */
  public void createExternalEcore(final Language l) {
    this._modelingElementExtensions.createEcore(l.getSyntax(), this.getExternalEcoreUri(l), this.getExternalPackageUri(l), false);
  }

  /**
   * Generates and serializes the Genmodel file defining the syntax of the
   * {@link Language} {@code l}.
   * 
   * @see #getExternalEcoreUri
   * @see #getExternalGenmodelUri
   * @see #getExternalPackageUri
   */
  public void createExternalGenmodel(final Language l) {
    this._metamodelExtensions.createGenmodel(l.getSyntax(), this.getExternalEcoreUri(l), this.getExternalGenmodelUri(l), 
      this.getExternalGenerationPath(l));
  }

  /**
   * Get the name of the project that will contain all the runtime of
   * the given {@link Language} {@code l} (Ecore, Genmodel, Java runtime, etc.)
   */
  public String getExternalRuntimeName(final Language l) {
    return this._iQualifiedNameProvider.getFullyQualifiedName(l).toLowerCase().toString();
  }

  /**
   * Returns the URI of the {@link Language} {@code l} once it is generated.
   * The returned URI is of the form {@code http://$languageNameInLowerCase/}.
   */
  public String getExternalPackageUri(final Language l) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("http://");
    QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(l).toLowerCase();
    _builder.append(_lowerCase);
    _builder.append("/");
    return _builder.toString();
  }

  public String getExternalEcorePath(final Language l) {
    final IProject project = this._eclipseProjectHelper.getProject(l.eResource());
    if (((project != null) && Objects.equal(this.getExternalRuntimeName(l), project.getName()))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("../");
      String _externalRuntimeName = this.getExternalRuntimeName(l);
      _builder.append(_externalRuntimeName);
      _builder.append("/model-gen/");
      String _name = l.getName();
      _builder.append(_name);
      _builder.append(".ecore");
      return _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("../");
      String _externalRuntimeName_1 = this.getExternalRuntimeName(l);
      _builder_1.append(_externalRuntimeName_1);
      _builder_1.append("/model/");
      String _name_1 = l.getName();
      _builder_1.append(_name_1);
      _builder_1.append(".ecore");
      return _builder_1.toString();
    }
  }

  public String getExternalGenmodelPath(final Language l) {
    final IProject project = this._eclipseProjectHelper.getProject(l.eResource());
    if (((project != null) && Objects.equal(this.getExternalRuntimeName(l), project.getName()))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("../");
      String _externalRuntimeName = this.getExternalRuntimeName(l);
      _builder.append(_externalRuntimeName);
      _builder.append("/model-gen/");
      String _name = l.getName();
      _builder.append(_name);
      _builder.append(".genmodel");
      return _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("../");
      String _externalRuntimeName_1 = this.getExternalRuntimeName(l);
      _builder_1.append(_externalRuntimeName_1);
      _builder_1.append("/model/");
      String _name_1 = l.getName();
      _builder_1.append(_name_1);
      _builder_1.append(".genmodel");
      return _builder_1.toString();
    }
  }

  public String getExternalGenerationPath(final Language l) {
    final IProject project = this._eclipseProjectHelper.getProject(l.eResource());
    if (((project != null) && Objects.equal(this.getExternalRuntimeName(l), project.getName()))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("../");
      String _externalRuntimeName = this.getExternalRuntimeName(l);
      _builder.append(_externalRuntimeName);
      _builder.append("/src-model-gen/");
      return _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("../");
      String _externalRuntimeName_1 = this.getExternalRuntimeName(l);
      _builder_1.append(_externalRuntimeName_1);
      _builder_1.append("/src/");
      return _builder_1.toString();
    }
  }

  public String getExternalEcoreUri(final Language l) {
    final IProject project = this._eclipseProjectHelper.getProject(l.eResource());
    if (((project != null) && Objects.equal(this.getExternalRuntimeName(l), project.getName()))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("platform:/resource/");
      String _externalRuntimeName = this.getExternalRuntimeName(l);
      _builder.append(_externalRuntimeName);
      _builder.append("/model-gen/");
      String _name = l.getName();
      _builder.append(_name);
      _builder.append(".ecore");
      return _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("platform:/resource/");
      String _externalRuntimeName_1 = this.getExternalRuntimeName(l);
      _builder_1.append(_externalRuntimeName_1);
      _builder_1.append("/model/");
      String _name_1 = l.getName();
      _builder_1.append(_name_1);
      _builder_1.append(".ecore");
      return _builder_1.toString();
    }
  }

  public String getExternalGenmodelUri(final Language l) {
    final IProject project = this._eclipseProjectHelper.getProject(l.eResource());
    if (((project != null) && Objects.equal(this.getExternalRuntimeName(l), project.getName()))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("platform:/resource/");
      String _externalRuntimeName = this.getExternalRuntimeName(l);
      _builder.append(_externalRuntimeName);
      _builder.append("/model-gen/");
      String _name = l.getName();
      _builder.append(_name);
      _builder.append(".genmodel");
      return _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("platform:/resource/");
      String _externalRuntimeName_1 = this.getExternalRuntimeName(l);
      _builder_1.append(_externalRuntimeName_1);
      _builder_1.append("/model/");
      String _name_1 = l.getName();
      _builder_1.append(_name_1);
      _builder_1.append(".genmodel");
      return _builder_1.toString();
    }
  }

  /**
   * Returns the fully qualified name of the package that contains all
   * the (possibly type-group-copied) aspects of the {@link Language} {@code l}
   */
  public String getAspectsNamespace(final Language l) {
    return this._iQualifiedNameProvider.getFullyQualifiedName(l).append("aspects").toLowerCase().toString();
  }

  /**
   * Checks whether the runtime of the given {@link Language} {@code l} must
   * be generated by Melange, ie. if it consists of multiple syntax imports or
   * if it is constructed from others {@link Language}s.
   */
  public boolean isGeneratedByMelange(final Language l) {
    return (!((l instanceof ExternalLanguage) || (l instanceof ImportDsl)));
  }

  /**
   * Checks whether the runtime project for the {@link Language} {@code l}
   * has already been generated, ie. if we can find its project, Ecore/Genmodel
   * files and Java code.
   */
  public boolean getRuntimeHasBeenGenerated(final Language l) {
    boolean _isGeneratedByMelange = this.isGeneratedByMelange(l);
    if (_isGeneratedByMelange) {
      final ArrayList<String> segments = CollectionLiterals.<String>newArrayList();
      GenModel _head = IterableExtensions.<GenModel>head(this._modelingElementExtensions.getGenmodels(l.getSyntax()));
      EList<GenPackage> _genPackages = null;
      if (_head!=null) {
        _genPackages=_head.getGenPackages();
      }
      GenPackage _head_1 = null;
      if (_genPackages!=null) {
        _head_1=IterableExtensions.<GenPackage>head(_genPackages);
      }
      final GenPackage gp = _head_1;
      final IProject project = this._eclipseProjectHelper.getProject(l.eResource());
      if (((gp == null) || (project == null))) {
        return false;
      }
      if (((gp.getBasePackage() != null) && (gp.getBasePackage().length() > 0))) {
        String _basePackage = gp.getBasePackage();
        segments.add(_basePackage);
      }
      String _name = gp.getEcorePackage().getName();
      segments.add(_name);
      final String fqn = QualifiedName.create(segments).toString().replace(".", "/");
      if (((project.getFile(this.getExternalEcorePath(l)).exists() && project.getFile(this.getExternalGenmodelPath(l)).exists()) && project.getFolder((this.getExternalGenerationPath(l) + fqn)).exists())) {
        return true;
      } else {
        return false;
      }
    } else {
      return true;
    }
  }

  /**
   * Copy (and possibly transform) all the aspects defined on the
   * {@link Language} {@code l} into its runtime project and update
   * {@code l}'s semantics with the newly generated aspects.
   */
  public void createExternalAspects(final Language l) {
    this.copier2.copyAspect(l, l);
  }

  /**
   * Returns the transitive set of namespaces defined by the {@link Language}
   * {@code l} (ie. all the EMF namespaces generated from its syntax or the
   * syntax of one of its dependencies).
   */
  public SetMultimap<String, String> collectTargetedPackages(final Language l) {
    final SetMultimap<String, String> res = HashMultimap.<String, String>create();
    final Function1<Import, Set<GenPackage>> _function = (Import it) -> {
      return this._modelingElementExtensions.getAllGenPkgs(it);
    };
    final Function1<GenPackage, Boolean> _function_1 = (GenPackage it) -> {
      EPackage _eSuperPackage = it.getEcorePackage().getESuperPackage();
      return Boolean.valueOf((_eSuperPackage == null));
    };
    final Consumer<GenPackage> _function_2 = (GenPackage it) -> {
      res.put(this._ecoreExtensions.getUniqueId(it.getEcorePackage()), this._namingHelper.getPackageNamespace(it));
    };
    IterableExtensions.<GenPackage>filter(Iterables.<GenPackage>concat(IterableExtensions.<Import, Set<GenPackage>>map(Iterables.<Import>filter(l.getOperators(), Import.class), _function)), _function_1).forEach(_function_2);
    final Function1<GenPackage, Boolean> _function_3 = (GenPackage it) -> {
      EPackage _eSuperPackage = it.getEcorePackage().getESuperPackage();
      return Boolean.valueOf((_eSuperPackage == null));
    };
    final Consumer<GenPackage> _function_4 = (GenPackage it) -> {
      res.put(this._ecoreExtensions.getUniqueId(it.getEcorePackage()), this._namingHelper.getPackageNamespace(it));
    };
    IterableExtensions.<GenPackage>filter(this._modelingElementExtensions.getAllGenPkgs(l.getSyntax()), _function_3).forEach(_function_4);
    final Function1<Language, SetMultimap<String, String>> _function_5 = (Language it) -> {
      return this.collectTargetedPackages(it);
    };
    final Consumer<SetMultimap<String, String>> _function_6 = (SetMultimap<String, String> it) -> {
      res.putAll(it);
    };
    IterableExtensions.<Language, SetMultimap<String, String>>map(this.getAllDependencies(l), _function_5).forEach(_function_6);
    return res;
  }

  /**
   * Checks whether the given {@link Language} {@code l} contains {@link Aspect}s
   * that are not directly defined on its own syntax and must be copied.
   */
  public boolean hasCopiedAspects(final Language l) {
    return (this.isGeneratedByMelange(l) && (this.getAllAspects(l).size() > 0));
  }

  /**
   * For each {@link Language} in the {@link ModelTypingSpace} {@code root},
   * gather {@link Aspect}s from all its dependencies to form its complete
   * semantics.
   */
  public void makeAllSemantics(final ModelTypingSpace root) {
    this.clearSemantics(root);
    final ArrayList<Language> processed = CollectionLiterals.<Language>newArrayList();
    final Function1<Language, Boolean> _function = (Language it) -> {
      return Boolean.valueOf(this.isGeneratedByMelange(it));
    };
    final Function1<Language, Boolean> _function_1 = (Language it) -> {
      boolean _contains = processed.contains(it);
      return Boolean.valueOf((!_contains));
    };
    final Consumer<Language> _function_2 = (Language lang) -> {
      this.makeAllSemantics(lang, processed);
      this.ensureUniqueAspects(lang);
    };
    IterableExtensions.<Language>filter(IterableExtensions.<Language>filter(this._aSTHelper.getLanguages(root), _function), _function_1).forEach(_function_2);
  }

  /**
   * For each {@link Language} in the {@link ModelTypingSpace} {@code root},
   * clear all its semantics and only keep those from the {@link Weave} operator.
   */
  public void clearSemantics(final ModelTypingSpace root) {
    final Consumer<Language> _function = (Language lang) -> {
      final List<Aspect> localAspects = this.getLocalSemantics(lang);
      lang.getSemantics().clear();
      EList<Aspect> _semantics = lang.getSemantics();
      Iterables.<Aspect>addAll(_semantics, localAspects);
    };
    this._aSTHelper.getLanguages(root).forEach(_function);
  }

  private void makeAllSemantics(final Language language, final List<Language> processed) {
    boolean _contains = processed.contains(language);
    boolean _not = (!_contains);
    if (_not) {
      processed.add(language);
      final Function1<Language, Boolean> _function = (Language it) -> {
        return Boolean.valueOf(((!processed.contains(it)) && this.isGeneratedByMelange(it)));
      };
      final Consumer<Language> _function_1 = (Language superLang) -> {
        this.makeAllSemantics(superLang, processed);
      };
      IterableExtensions.<Language>filter(this.getAllDependencies(language), _function).forEach(_function_1);
      this.makeAllSemantics(language);
    }
  }

  /**
   * Gathers the semantics of all the dependencies of the {@link Language}
   * {@code language}, copy the appropriate {@link Aspect}s and update the
   * local semantics of {@code language} to point to the copied aspects.
   * <br>
   * Note: assumes that the semantics of each of the dependencies of
   * {@code language} has been built.
   */
  private void makeAllSemantics(final Language language) {
    this.updateLocalAspects(language);
    final Function1<LanguageOperator, Boolean> _function = (LanguageOperator it) -> {
      return Boolean.valueOf(((it instanceof Merge) || (it instanceof Slice)));
    };
    final Consumer<LanguageOperator> _function_1 = (LanguageOperator op) -> {
      List<Aspect> aspects = this.getOrderedAspects(op.getTargetLanguage());
      if ((op instanceof Slice)) {
        final List<OperatorBuilder<? extends Operator>> opBuilders = this.modelTypingSpaceBuilder.findBuilder(language).getSubBuilders();
        final Function1<OperatorBuilder<? extends Operator>, Boolean> _function_2 = (OperatorBuilder<? extends Operator> it) -> {
          return Boolean.valueOf(Objects.equal(it.source, op));
        };
        final OperatorBuilder<? extends Operator> sliceBuilder = IterableExtensions.<OperatorBuilder<? extends Operator>>findFirst(opBuilders, _function_2);
        Iterable<EClass> _xifexpression = null;
        if ((sliceBuilder != null)) {
          final Function1<EPackage, List<EClass>> _function_3 = (EPackage it) -> {
            return this._ecoreExtensions.getAllClasses(it);
          };
          _xifexpression = Iterables.<EClass>concat(IterableExtensions.<EPackage, List<EClass>>map(sliceBuilder.getModel(), _function_3));
        } else {
          _xifexpression = CollectionLiterals.<EClass>newArrayList();
        }
        final Iterable<EClass> sliceClasses = _xifexpression;
        final Function1<Aspect, Boolean> _function_4 = (Aspect asp) -> {
          final Function1<EClass, Boolean> _function_5 = (EClass it) -> {
            String _name = it.getName();
            EClass _aspectedClass = asp.getAspectedClass();
            String _name_1 = null;
            if (_aspectedClass!=null) {
              _name_1=_aspectedClass.getName();
            }
            return Boolean.valueOf(Objects.equal(_name, _name_1));
          };
          return Boolean.valueOf(IterableExtensions.<EClass>exists(sliceClasses, _function_5));
        };
        aspects = IterableExtensions.<Aspect>toList(IterableExtensions.<Aspect>filter(aspects, _function_4));
      }
      final Consumer<Aspect> _function_5 = (Aspect asp) -> {
        Metamodel _syntax = language.getSyntax();
        EClass _findClassWithMapping = this.findClassWithMapping(asp, op);
        QualifiedName _fullyQualifiedName = null;
        if (_findClassWithMapping!=null) {
          _fullyQualifiedName=this._iQualifiedNameProvider.getFullyQualifiedName(_findClassWithMapping);
        }
        String _string = null;
        if (_fullyQualifiedName!=null) {
          _string=_fullyQualifiedName.toString();
        }
        final EClass localAspectedClass = this._modelingElementExtensions.findClass(_syntax, _string);
        Aspect _createAspect = MelangeFactory.eINSTANCE.createAspect();
        final Procedure1<Aspect> _function_6 = (Aspect it) -> {
          it.setAspectedClass(localAspectedClass);
          it.setAspectTypeRef(this.typesBuilder.cloneWithProxies(asp.getAspectTypeRef()));
          it.setEcoreFragment(EcoreUtil.<EPackage>copy(asp.getEcoreFragment()));
          it.setSource(asp.getSource());
        };
        final Aspect newAsp = ObjectExtensions.<Aspect>operator_doubleArrow(_createAspect, _function_6);
        EList<Aspect> _semantics = language.getSemantics();
        _semantics.add(newAsp);
        this.aspectExtension.tryUpdateAspect(newAsp);
      };
      aspects.forEach(_function_5);
    };
    IterableExtensions.<LanguageOperator>filter(Iterables.<LanguageOperator>filter(ListExtensions.<Operator>reverseView(language.getOperators()), LanguageOperator.class), _function).forEach(_function_1);
    final Consumer<Language> _function_2 = (Language superLang) -> {
      final Consumer<Aspect> _function_3 = (Aspect asp) -> {
        Metamodel _syntax = language.getSyntax();
        EClass _aspectedClass = asp.getAspectedClass();
        QualifiedName _fullyQualifiedName = null;
        if (_aspectedClass!=null) {
          _fullyQualifiedName=this._iQualifiedNameProvider.getFullyQualifiedName(_aspectedClass);
        }
        String _string = null;
        if (_fullyQualifiedName!=null) {
          _string=_fullyQualifiedName.toString();
        }
        final EClass localAspectedClass = this._modelingElementExtensions.findClass(_syntax, _string);
        Aspect _createAspect = MelangeFactory.eINSTANCE.createAspect();
        final Procedure1<Aspect> _function_4 = (Aspect it) -> {
          it.setAspectedClass(localAspectedClass);
          it.setAspectTypeRef(this.typesBuilder.cloneWithProxies(asp.getAspectTypeRef()));
          it.setEcoreFragment(EcoreUtil.<EPackage>copy(asp.getEcoreFragment()));
          it.setSource(asp.getSource());
        };
        final Aspect newAsp = ObjectExtensions.<Aspect>operator_doubleArrow(_createAspect, _function_4);
        EList<Aspect> _semantics = language.getSemantics();
        _semantics.add(newAsp);
        this.aspectExtension.tryUpdateAspect(newAsp);
      };
      this.getOrderedAspects(superLang).forEach(_function_3);
    };
    ListExtensions.<Language>reverseView(IterableExtensions.<Language>toList(this.getSuperLanguages(language))).forEach(_function_2);
  }

  /**
   * Ensure that a language doesn't contain multiple times the same aspect
   * (eg. in the case of multiple inheritance/overriding)
   */
  private void ensureUniqueAspects(final Language language) {
    final LinkedHashMap<String, Aspect> noDuplicates = CollectionLiterals.<String, Aspect>newLinkedHashMap();
    final Consumer<Aspect> _function = (Aspect asp) -> {
      noDuplicates.put(asp.getAspectTypeRef().getIdentifier(), asp);
    };
    language.getSemantics().forEach(_function);
    language.getSemantics().clear();
    EList<Aspect> _semantics = language.getSemantics();
    Collection<Aspect> _values = noDuplicates.values();
    Iterables.<Aspect>addAll(_semantics, _values);
  }

  /**
   * Get {@link Aspect}s from the {@link Language} {@code language} ordered
   * by their overriding priority.
   */
  private List<Aspect> getOrderedAspects(final Language language) {
    List<Aspect> _xifexpression = null;
    boolean _isGeneratedByMelange = this.isGeneratedByMelange(language);
    if (_isGeneratedByMelange) {
      _xifexpression = language.getSemantics();
    } else {
      _xifexpression = ListExtensions.<Aspect>reverseView(language.getSemantics());
    }
    return _xifexpression;
  }

  /**
   * Try to update each of the local {@link Aspect}s of the {@link Language}
   * {@code language} to point to their generated equivalent.
   */
  private void updateLocalAspects(final Language language) {
    final Consumer<Aspect> _function = (Aspect asp) -> {
      Metamodel _syntax = language.getSyntax();
      EClass _aspectedClass = asp.getAspectedClass();
      QualifiedName _fullyQualifiedName = null;
      if (_aspectedClass!=null) {
        _fullyQualifiedName=this._iQualifiedNameProvider.getFullyQualifiedName(_aspectedClass);
      }
      String _string = null;
      if (_fullyQualifiedName!=null) {
        _string=_fullyQualifiedName.toString();
      }
      asp.setAspectedClass(this._modelingElementExtensions.findClass(_syntax, _string));
      this.aspectExtension.tryUpdateAspect(asp);
    };
    ListExtensions.<Aspect>reverseView(this.getLocalSemantics(language)).forEach(_function);
  }

  /**
   * Get a {@link JvmTypeReference} to the copied class corresponding to
   * {@code aspectSimpleName} in the project generated for the
   * {@link Language} {@code language}, or null if it cannot be found.
   */
  public JvmTypeReference getCopiedAspectRefFor(final Language language, final String aspectSimpleName) {
    final JvmTypeReferenceBuilder typeRefBuilder = this.builderFactory.create(language.eResource().getResourceSet());
    StringConcatenation _builder = new StringConcatenation();
    String _aspectsNamespace = this.getAspectsNamespace(language);
    _builder.append(_aspectsNamespace);
    _builder.append(".");
    _builder.append(aspectSimpleName);
    final JvmTypeReference newRef = typeRefBuilder.typeRef(_builder.toString());
    JvmTypeReference _xifexpression = null;
    if ((newRef instanceof JvmUnknownTypeReference)) {
      _xifexpression = null;
    } else {
      _xifexpression = newRef;
    }
    return _xifexpression;
  }

  /**
   * Return the set of {@link JvmOperation} tagged with @Main found
   * in {@code language}'s {@link Aspect}s.
   */
  public Set<JvmOperation> getEntryPoints(final Language language) {
    final Function1<Aspect, JvmType> _function = (Aspect it) -> {
      return it.getAspectTypeRef().getType();
    };
    final Function1<JvmDeclaredType, Iterable<JvmOperation>> _function_1 = (JvmDeclaredType it) -> {
      return it.getDeclaredOperations();
    };
    final Function1<JvmOperation, Boolean> _function_2 = (JvmOperation it) -> {
      final Function1<JvmAnnotationReference, Boolean> _function_3 = (JvmAnnotationReference it_1) -> {
        String _qualifiedName = it_1.getAnnotation().getQualifiedName();
        return Boolean.valueOf(Objects.equal(_qualifiedName, LanguageExtensions.ASPECT_MAIN_ANNOTATION));
      };
      return Boolean.valueOf(IterableExtensions.<JvmAnnotationReference>exists(it.getAnnotations(), _function_3));
    };
    return IterableExtensions.<JvmOperation>toSet(IterableExtensions.<JvmOperation>filter(Iterables.<JvmOperation>concat(IterableExtensions.<JvmDeclaredType, Iterable<JvmOperation>>map(Iterables.<JvmDeclaredType>filter(ListExtensions.<Aspect, JvmType>map(this.allSemantics(language), _function), JvmDeclaredType.class), _function_1)), _function_2));
  }

  /**
   * Search in {@link op}.owningLanguage for the class on which {@link asp}
   * is weaved, taking in account the mapping
   * 
   * @param asp An aspect from op.targetLanguage
   * @param op A Merge or Slice operator
   */
  public EClass findClassWithMapping(final Aspect asp, final LanguageOperator op) {
    EClass _aspectedClass = asp.getAspectedClass();
    QualifiedName _fullyQualifiedName = null;
    if (_aspectedClass!=null) {
      _fullyQualifiedName=this._iQualifiedNameProvider.getFullyQualifiedName(_aspectedClass);
    }
    String _string = null;
    if (_fullyQualifiedName!=null) {
      _string=_fullyQualifiedName.toString();
    }
    String classFqName = _string;
    EList<PackageBinding> _xifexpression = null;
    if ((op instanceof Merge)) {
      _xifexpression = ((Merge)op).getMappingRules();
    } else {
      EList<PackageBinding> _xifexpression_1 = null;
      if ((op instanceof Slice)) {
        _xifexpression_1 = ((Slice)op).getMappingRules();
      }
      _xifexpression = _xifexpression_1;
    }
    final EList<PackageBinding> rules = _xifexpression;
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(rules);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      classFqName = this.rename(classFqName, rules);
    }
    return this._modelingElementExtensions.findClass(op.getOwningLanguage().getSyntax(), classFqName);
  }

  /**
   * Return null if {@link op} has no mapping
   */
  public RenamingRuleManager createRenamingManager(final LanguageOperator op) {
    EList<PackageBinding> _xifexpression = null;
    if ((op instanceof Merge)) {
      _xifexpression = ((Merge)op).getMappingRules();
    } else {
      EList<PackageBinding> _xifexpression_1 = null;
      if ((op instanceof Slice)) {
        _xifexpression_1 = ((Slice)op).getMappingRules();
      }
      _xifexpression = _xifexpression_1;
    }
    final EList<PackageBinding> rules = _xifexpression;
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(rules);
    if (_isNullOrEmpty) {
      return null;
    } else {
      return new RenamingRuleManager(rules, Collections.<Aspect>unmodifiableList(CollectionLiterals.<Aspect>newArrayList()), this.aspectExtension);
    }
  }

  /**
   * Returns mappings from syntax, merge & slice operators
   */
  public List<PackageBinding> collectMappings(final Language l) {
    final ArrayList<PackageBinding> res = CollectionLiterals.<PackageBinding>newArrayList();
    final Function1<Import, EList<PackageBinding>> _function = (Import it) -> {
      return it.getMappingRules();
    };
    Iterables.<PackageBinding>addAll(res, Iterables.<PackageBinding>concat(IterableExtensions.<Import, EList<PackageBinding>>map(Iterables.<Import>filter(l.getOperators(), Import.class), _function)));
    final Function1<Merge, EList<PackageBinding>> _function_1 = (Merge it) -> {
      return it.getMappingRules();
    };
    Iterables.<PackageBinding>addAll(res, Iterables.<PackageBinding>concat(IterableExtensions.<Merge, EList<PackageBinding>>map(Iterables.<Merge>filter(l.getOperators(), Merge.class), _function_1)));
    final Function1<Slice, EList<PackageBinding>> _function_2 = (Slice it) -> {
      return it.getMappingRules();
    };
    Iterables.<PackageBinding>addAll(res, Iterables.<PackageBinding>concat(IterableExtensions.<Slice, EList<PackageBinding>>map(Iterables.<Slice>filter(l.getOperators(), Slice.class), _function_2)));
    return res;
  }

  public String rename(final String qualifiedClsName, final List<PackageBinding> rules) {
    if (((qualifiedClsName == null) || (!qualifiedClsName.contains(".")))) {
      return qualifiedClsName;
    }
    final String pkgName = qualifiedClsName.substring(0, qualifiedClsName.lastIndexOf("."));
    int _lastIndexOf = qualifiedClsName.lastIndexOf(".");
    int _plus = (_lastIndexOf + 1);
    final String simpleName = qualifiedClsName.substring(_plus);
    final Function1<PackageBinding, Boolean> _function = (PackageBinding it) -> {
      return Boolean.valueOf(pkgName.endsWith(it.getFrom()));
    };
    final Iterable<PackageBinding> candidateRules = IterableExtensions.<PackageBinding>filter(rules, _function);
    final Function1<PackageBinding, Boolean> _function_1 = (PackageBinding it) -> {
      final Function1<ClassBinding, Boolean> _function_2 = (ClassBinding it_1) -> {
        String _from = it_1.getFrom();
        return Boolean.valueOf(Objects.equal(_from, simpleName));
      };
      return Boolean.valueOf(IterableExtensions.<ClassBinding>exists(it.getClasses(), _function_2));
    };
    final PackageBinding res = IterableExtensions.<PackageBinding>findFirst(candidateRules, _function_1);
    if ((res != null)) {
      String _to = res.getTo();
      String _plus_1 = (_to + ".");
      final Function1<ClassBinding, Boolean> _function_2 = (ClassBinding it) -> {
        String _from = it.getFrom();
        return Boolean.valueOf(Objects.equal(_from, simpleName));
      };
      String _to_1 = IterableExtensions.<ClassBinding>findFirst(res.getClasses(), _function_2).getTo();
      return (_plus_1 + _to_1);
    } else {
      boolean _isEmpty = IterableExtensions.isEmpty(candidateRules);
      boolean _not = (!_isEmpty);
      if (_not) {
        String _to_2 = IterableExtensions.<PackageBinding>head(candidateRules).getTo();
        String _plus_2 = (_to_2 + ".");
        return (_plus_2 + simpleName);
      }
    }
    return qualifiedClsName;
  }

  private Set<String> collectAspectDependencies(final Language l) {
    final Set<Language> scope = this.getAllDependencies(l);
    scope.add(l);
    final Function1<Language, EList<Operator>> _function = (Language it) -> {
      return it.getOperators();
    };
    final Function1<Weave, Boolean> _function_1 = (Weave it) -> {
      String _aspectWildcardImport = it.getAspectWildcardImport();
      return Boolean.valueOf((_aspectWildcardImport == null));
    };
    final Function1<Weave, JvmTypeReference> _function_2 = (Weave it) -> {
      return it.getAspectTypeRef();
    };
    final Set<JvmTypeReference> originalAspects = IterableExtensions.<JvmTypeReference>toSet(IterableExtensions.<Weave, JvmTypeReference>map(IterableExtensions.<Weave>filter(Iterables.<Weave>filter((Iterables.<Operator>concat(IterableExtensions.<Language, EList<Operator>>map(scope, _function))), Weave.class), _function_1), _function_2));
    final Function1<Language, EList<Operator>> _function_3 = (Language it) -> {
      return it.getOperators();
    };
    final Function1<Import, String> _function_4 = (Import it) -> {
      return it.getEcoreUri();
    };
    final Set<String> originalEcores = IterableExtensions.<String>toSet(IterableExtensions.<Import, String>map(Iterables.<Import>filter((Iterables.<Operator>concat(IterableExtensions.<Language, EList<Operator>>map(scope, _function_3))), Import.class), _function_4));
    final Function1<String, IFile> _function_5 = (String ecoreURI) -> {
      IFile _xblockexpression = null;
      {
        final URI uri = URI.createURI(ecoreURI);
        final String filePath = uri.toPlatformString(true);
        IFile _xifexpression = null;
        if ((filePath != null)) {
          IFile _xblockexpression_1 = null;
          {
            final IPath path = new Path(filePath);
            _xblockexpression_1 = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
          }
          _xifexpression = _xblockexpression_1;
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    };
    final Function1<IFile, IProject> _function_6 = (IFile it) -> {
      return it.getProject();
    };
    final Set<IProject> ecoreProjects = IterableExtensions.<IProject>toSet(IterableExtensions.<IFile, IProject>map(IterableExtensions.<IFile>filterNull(IterableExtensions.<String, IFile>map(originalEcores, _function_5)), _function_6));
    final HashSet<IProject> k3Projects = new HashSet<IProject>();
    final IWorkspaceRoot ws = ResourcesPlugin.getWorkspace().getRoot();
    try {
      ws.accept(new IResourceVisitor() {
        @Override
        public boolean visit(final IResource resource) throws CoreException {
          if ((resource instanceof IFile)) {
            final Function1<JvmTypeReference, Boolean> _function = (JvmTypeReference ref) -> {
              boolean _xblockexpression = false;
              {
                String _replace = ref.getIdentifier().replace(".", "/");
                final String pattern = (_replace + ".java");
                _xblockexpression = ((IFile)resource).getLocationURI().getPath().endsWith(pattern);
              }
              return Boolean.valueOf(_xblockexpression);
            };
            final JvmTypeReference firstMatch = IterableExtensions.<JvmTypeReference>findFirst(originalAspects, _function);
            if ((firstMatch != null)) {
              k3Projects.add(((IFile)resource).getProject());
            }
            return false;
          }
          return true;
        }
      });
    } catch (final Throwable _t) {
      if (_t instanceof CoreException) {
        final CoreException e = (CoreException)_t;
        LanguageExtensions.log.error("Unexpected exception while visiting workspace", e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    final Function1<IProject, Iterable<String>> _function_7 = (IProject it) -> {
      return this._eclipseProjectHelper.getDependencies(it);
    };
    final Set<String> allRequiredBundles = IterableExtensions.<String>toSet(Iterables.<String>concat(IterableExtensions.<IProject, Iterable<String>>map(k3Projects, _function_7)));
    final Function1<IProject, String> _function_8 = (IProject it) -> {
      return it.getName();
    };
    CollectionExtensions.<String>removeAll(allRequiredBundles, IterableExtensions.<IProject, String>map(ecoreProjects, _function_8));
    return allRequiredBundles;
  }

  /**
   * Collect required bundles from Aspect's projects (Ecore bundles excluded)
   * and add them to the runtime project of {@link l}.
   */
  public void addRequireBundleForAspects(final Language l) {
    final IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(this.getExternalRuntimeName(l));
    this.eclipseHelper.addDependencies(project, this.collectAspectDependencies(l));
  }

  /**
   * Return all local aspects extracted from {@link l}'s dependencies associated with renaming rules
   * from aspect.owningLanguage to {@link l}
   */
  public List<Pair<Aspect, List<PackageBinding>>> getAllAspectsWithRenaming(final Language l, final Stack<List<PackageBinding>> stack) {
    final ArrayList<Pair<Aspect, List<PackageBinding>>> res = new ArrayList<Pair<Aspect, List<PackageBinding>>>();
    final List<PackageBinding> renaming = AspectCopier2.flatten(stack);
    final Consumer<Aspect> _function = (Aspect asp) -> {
      Pair<Aspect, List<PackageBinding>> _pair = new Pair<Aspect, List<PackageBinding>>(asp, renaming);
      res.add(_pair);
    };
    this.getLocalSemantics(l).forEach(_function);
    final Consumer<Operator> _function_1 = (Operator op) -> {
      if ((op instanceof Slice)) {
        stack.push(((Slice)op).getMappingRules());
        res.addAll(this.getAllAspectsWithRenaming(((Slice)op).getTargetLanguage(), stack));
        stack.pop();
      } else {
        if ((op instanceof Merge)) {
          stack.push(((Merge)op).getMappingRules());
          res.addAll(this.getAllAspectsWithRenaming(((Merge)op).getTargetLanguage(), stack));
          stack.pop();
        }
      }
    };
    l.getOperators().forEach(_function_1);
    final Function1<Inheritance, List<Pair<Aspect, List<PackageBinding>>>> _function_2 = (Inheritance it) -> {
      return this.getAllAspectsWithRenaming(it.getTargetLanguage(), stack);
    };
    Iterable<Pair<Aspect, List<PackageBinding>>> _flatten = Iterables.<Pair<Aspect, List<PackageBinding>>>concat(IterableExtensions.<Inheritance, List<Pair<Aspect, List<PackageBinding>>>>map(Iterables.<Inheritance>filter(l.getOperators(), Inheritance.class), _function_2));
    Iterables.<Pair<Aspect, List<PackageBinding>>>addAll(res, _flatten);
    return IterableExtensions.<Pair<Aspect, List<PackageBinding>>>toList(res);
  }

  public boolean isXmof(final Language l) {
    return ((l instanceof ExternalLanguage) && (l.getXmof() != null));
  }

  public Dsl toDsl(final Language l) {
    final Dsl dsl = DslFactoryImpl.eINSTANCE.createDsl();
    dsl.setName(this._iQualifiedNameProvider.getFullyQualifiedName(l).toString());
    final Entry ecoreEntry = DslFactoryImpl.eINSTANCE.createEntry();
    ecoreEntry.setKey("ecore");
    ecoreEntry.setValue(l.getSyntax().getEcoreUri());
    EList<Entry> _entries = dsl.getEntries();
    _entries.add(ecoreEntry);
    boolean _isEmpty = l.getSemantics().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final Entry k3Entry = DslFactoryImpl.eINSTANCE.createEntry();
      k3Entry.setKey("k3");
      final Function1<Aspect, String> _function = (Aspect asp) -> {
        return asp.getAspectTypeRef().getQualifiedName();
      };
      k3Entry.setValue(
        IterableExtensions.join(ListExtensions.<Aspect, String>map(l.getSemantics(), _function), ","));
      EList<Entry> _entries_1 = dsl.getEntries();
      _entries_1.add(k3Entry);
    }
    EList<Annotation> _annotations = l.getAnnotations();
    for (final Annotation annot : _annotations) {
      {
        final Entry entry = DslFactoryImpl.eINSTANCE.createEntry();
        entry.setKey(annot.getKey());
        entry.setValue(annot.getValue());
        EList<Entry> _entries_2 = dsl.getEntries();
        _entries_2.add(entry);
      }
    }
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(l.getEcl());
    if (_isNullOrEmpty) {
      final Entry metaprogEnry = DslFactoryImpl.eINSTANCE.createEntry();
      metaprogEnry.setKey("metaprog");
      metaprogEnry.setValue("org.eclipse.gemoc.metaprog.kermeta3");
      boolean _existMetaprogApproach = this.existMetaprogApproach(metaprogEnry.getValue());
      if (_existMetaprogApproach) {
        EList<Entry> _entries_2 = dsl.getEntries();
        _entries_2.add(metaprogEnry);
      }
    } else {
      final Entry eclEntry = DslFactoryImpl.eINSTANCE.createEntry();
      eclEntry.setKey("ecl");
      eclEntry.setValue(IterableExtensions.join(l.getEcl(), ","));
      EList<Entry> _entries_3 = dsl.getEntries();
      _entries_3.add(eclEntry);
      final Entry metaprogEnry_1 = DslFactoryImpl.eINSTANCE.createEntry();
      metaprogEnry_1.setKey("metaprog");
      metaprogEnry_1.setValue("org.eclipse.gemoc.metaprog.moccml");
      boolean _existMetaprogApproach_1 = this.existMetaprogApproach(metaprogEnry_1.getValue());
      if (_existMetaprogApproach_1) {
        EList<Entry> _entries_4 = dsl.getEntries();
        _entries_4.add(metaprogEnry_1);
      }
    }
    return dsl;
  }

  public boolean existMetaprogApproach(final String metaprogApproachName) {
    ArrayList<String> approachesList = new ArrayList<String>();
    IConfigurationElement[] exts = Platform.getExtensionRegistry().getConfigurationElementsFor("org.eclipse.gemoc.gemoc_language_workbench.metaprog");
    for (final IConfigurationElement elem : exts) {
      approachesList.add(elem.getAttribute("name"));
    }
    return approachesList.contains(metaprogApproachName);
  }

  public void createDsl(final Language l) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("platform:/resource/");
    String _externalRuntimeName = this.getExternalRuntimeName(l);
    _builder.append(_externalRuntimeName);
    _builder.append("/");
    String _name = l.getName();
    _builder.append(_name);
    _builder.append(".dsl");
    final String uri = _builder.toString();
    final ResourceSetImpl resSet = new ResourceSetImpl();
    final Resource res = resSet.createResource(URI.createURI(uri));
    EList<EObject> _contents = res.getContents();
    Dsl _dsl = this.toDsl(l);
    _contents.add(_dsl);
    try {
      final HashMap<Object, Object> options = CollectionLiterals.<Object, Object>newHashMap();
      res.save(options);
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(l);
        String _plus = ("Error while serializing DSL file for" + _fullyQualifiedName);
        LanguageExtensions.log.error(_plus, e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
}
