/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.xdsmlframework.api.engine_addon.modelchangelistener;

import com.google.common.base.Objects;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.gemoc.xdsmlframework.api.engine_addon.modelchangelistener.FieldModelChange;
import org.eclipse.gemoc.xdsmlframework.api.engine_addon.modelchangelistener.ModelChange;
import org.eclipse.gemoc.xdsmlframework.api.engine_addon.modelchangelistener.NewObjectModelChange;
import org.eclipse.gemoc.xdsmlframework.api.engine_addon.modelchangelistener.NonCollectionFieldModelChange;
import org.eclipse.gemoc.xdsmlframework.api.engine_addon.modelchangelistener.PotentialCollectionFieldModelChange;
import org.eclipse.gemoc.xdsmlframework.api.engine_addon.modelchangelistener.RemovedObjectModelChange;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class BatchModelChangeListener {
  private final EContentAdapter adapter;
  
  private final Map<Object, List<Notification>> changes = new HashMap<Object, List<Notification>>();
  
  private final Set<Object> registeredObservers = new HashSet<Object>();
  
  private final Set<Resource> observedResources = new HashSet<Resource>();
  
  public BatchModelChangeListener(final Set<Resource> resources) {
    this.adapter = new EContentAdapter() {
      @Override
      public void notifyChanged(final Notification notification) {
        super.notifyChanged(notification);
        for (final Object obs : BatchModelChangeListener.this.registeredObservers) {
          BatchModelChangeListener.this.changes.get(obs).add(notification);
        }
      }
    };
    this.observedResources.addAll(resources);
    final Consumer<Resource> _function = (Resource r) -> {
      if ((r != null)) {
        r.eAdapters().add(this.adapter);
      }
    };
    this.observedResources.forEach(_function);
  }
  
  /**
   * When an observer asks for the changes, we process all the notifications gathered for it since the last time.
   */
  public List<ModelChange> getChanges(final Object addon) {
    final List<ModelChange> result = new ArrayList<ModelChange>();
    final List<Notification> allNotifs = this.changes.get(addon);
    boolean _contains = this.registeredObservers.contains(addon);
    if (_contains) {
      ArrayList<Notification> _arrayList = new ArrayList<Notification>();
      this.changes.put(addon, _arrayList);
    }
    final Map<EObject, Map<EStructuralFeature, List<Notification>>> sortedNotifications = new HashMap<EObject, Map<EStructuralFeature, List<Notification>>>();
    final Map<Resource, List<Notification>> resourcesNotifications = new HashMap<Resource, List<Notification>>();
    for (final Notification notification : allNotifs) {
      {
        final int eventType = notification.getEventType();
        if (((eventType < Notification.EVENT_TYPE_COUNT) && (!notification.isTouch()))) {
          if (((notification.getNotifier() instanceof EObject) && 
            (notification.getFeature() instanceof EStructuralFeature))) {
            Object _feature = notification.getFeature();
            final EStructuralFeature feature = ((EStructuralFeature) _feature);
            Object _notifier = notification.getNotifier();
            final EObject changedObject = ((EObject) _notifier);
            boolean _containsKey = sortedNotifications.containsKey(changedObject);
            boolean _not = (!_containsKey);
            if (_not) {
              HashMap<EStructuralFeature, List<Notification>> _hashMap = new HashMap<EStructuralFeature, List<Notification>>();
              sortedNotifications.put(changedObject, _hashMap);
            }
            final Map<EStructuralFeature, List<Notification>> objectsNotifications = sortedNotifications.get(changedObject);
            boolean _containsKey_1 = objectsNotifications.containsKey(feature);
            boolean _not_1 = (!_containsKey_1);
            if (_not_1) {
              ArrayList<Notification> _arrayList_1 = new ArrayList<Notification>();
              objectsNotifications.put(feature, _arrayList_1);
            }
            final List<Notification> fieldNotifications = objectsNotifications.get(feature);
            BatchModelChangeListener.<Notification>addUnique(fieldNotifications, notification);
          } else {
            Object _notifier_1 = notification.getNotifier();
            if ((_notifier_1 instanceof Resource)) {
              Object _notifier_2 = notification.getNotifier();
              final Resource resource = ((Resource) _notifier_2);
              boolean _containsKey_2 = resourcesNotifications.containsKey(resource);
              boolean _not_2 = (!_containsKey_2);
              if (_not_2) {
                ArrayList<Notification> _arrayList_2 = new ArrayList<Notification>();
                resourcesNotifications.put(resource, _arrayList_2);
              }
              final List<Notification> resourceNotifications = resourcesNotifications.get(resource);
              BatchModelChangeListener.<Notification>addUnique(resourceNotifications, notification);
            }
          }
        }
      }
    }
    final HashSet<EObject> newObjects = new HashSet<EObject>();
    final HashSet<EObject> removedObjects = new HashSet<EObject>();
    final HashSet<EObject> eventuallyRemoved = new HashSet<EObject>();
    Set<Resource> _keySet = resourcesNotifications.keySet();
    for (final Resource resource : _keySet) {
      {
        final List<Notification> resourceNotifications = resourcesNotifications.get(resource);
        for (final Notification notif : resourceNotifications) {
          BatchModelChangeListener.manageCollectionContainmentNotification(eventuallyRemoved, removedObjects, newObjects, notif);
        }
      }
    }
    Set<EObject> _keySet_1 = sortedNotifications.keySet();
    for (final EObject object : _keySet_1) {
      {
        final Map<EStructuralFeature, List<Notification>> featureMap = sortedNotifications.get(object);
        Set<EStructuralFeature> _keySet_2 = featureMap.keySet();
        for (final EStructuralFeature feature : _keySet_2) {
          {
            final List<Notification> notifs = featureMap.get(feature);
            boolean _isMany = feature.isMany();
            boolean _not = (!_isMany);
            if (_not) {
              final Object previousValue = IterableExtensions.<Notification>head(notifs).getOldValue();
              final Object newValue = IterableExtensions.<Notification>last(notifs).getNewValue();
              if ((feature instanceof EReference)) {
                boolean _notEquals = (!Objects.equal(previousValue, newValue));
                if (_notEquals) {
                  NonCollectionFieldModelChange _nonCollectionFieldModelChange = new NonCollectionFieldModelChange(object, feature);
                  result.add(_nonCollectionFieldModelChange);
                  boolean _isContainment = ((EReference) feature).isContainment();
                  if (_isContainment) {
                    if (((previousValue != null) && (previousValue instanceof EObject))) {
                      BatchModelChangeListener.addToRemovedObjects(eventuallyRemoved, removedObjects, newObjects, 
                        ((EObject) previousValue));
                    }
                    if (((newValue != null) && (newValue instanceof EObject))) {
                      BatchModelChangeListener.addToNewObjects(eventuallyRemoved, removedObjects, newObjects, ((EObject) newValue));
                    }
                  }
                }
              } else {
                boolean _xifexpression = false;
                if ((previousValue == null)) {
                  _xifexpression = (newValue != null);
                } else {
                  boolean _equals = previousValue.equals(newValue);
                  _xifexpression = (!_equals);
                }
                if (_xifexpression) {
                  NonCollectionFieldModelChange _nonCollectionFieldModelChange_1 = new NonCollectionFieldModelChange(object, feature);
                  result.add(_nonCollectionFieldModelChange_1);
                }
              }
            } else {
              PotentialCollectionFieldModelChange _potentialCollectionFieldModelChange = new PotentialCollectionFieldModelChange(object, feature, notifs);
              result.add(_potentialCollectionFieldModelChange);
              for (final Notification notif : notifs) {
                if (((feature instanceof EReference) && ((EReference) feature).isContainment())) {
                  BatchModelChangeListener.manageCollectionContainmentNotification(eventuallyRemoved, removedObjects, newObjects, notif);
                }
              }
            }
          }
        }
      }
    }
    for (final EObject newObject : newObjects) {
      NewObjectModelChange _newObjectModelChange = new NewObjectModelChange(newObject);
      result.add(0, _newObjectModelChange);
    }
    for (final EObject removedObject : removedObjects) {
      RemovedObjectModelChange _removedObjectModelChange = new RemovedObjectModelChange(removedObject);
      result.add(0, _removedObjectModelChange);
    }
    final Predicate<ModelChange> _function = (ModelChange c) -> {
      return ((c instanceof FieldModelChange) && ((newObjects.contains(c.getChangedObject()) || removedObjects.contains(c.getChangedObject())) || eventuallyRemoved.contains(c.getChangedObject())));
    };
    result.removeIf(_function);
    return result;
  }
  
  public boolean registerObserver(final Object observer) {
    final boolean res = this.registeredObservers.add(observer);
    if (res) {
      ArrayList<Notification> _arrayList = new ArrayList<Notification>();
      this.changes.put(observer, _arrayList);
    }
    return res;
  }
  
  public void removeObserver(final Object observer) {
    this.changes.remove(observer);
    this.registeredObservers.remove(observer);
  }
  
  private static void addToNewObjects(final Collection<EObject> eventuallyRemoved, final Collection<EObject> removedObjects, final Collection<EObject> newObjects, final EObject object) {
    eventuallyRemoved.remove(object);
    if ((object != null)) {
      final boolean hasMoved = removedObjects.remove(object);
      if ((!hasMoved)) {
        newObjects.add(object);
      }
    }
  }
  
  private static void addToRemovedObjects(final Collection<EObject> eventuallyRemoved, final Collection<EObject> removedObjects, final Collection<EObject> newObjects, final EObject object) {
    eventuallyRemoved.add(object);
    if ((object != null)) {
      final boolean hasMoved = newObjects.remove(object);
      if ((!hasMoved)) {
        removedObjects.add(object);
      }
    }
  }
  
  private static void manageCollectionContainmentNotification(final Collection<EObject> eventuallyRemoved, final Collection<EObject> removedObjects, final Collection<EObject> newObjects, final Notification notif) {
    int _eventType = notif.getEventType();
    switch (_eventType) {
      case Notification.ADD:
        Object _newValue = notif.getNewValue();
        BatchModelChangeListener.addToNewObjects(eventuallyRemoved, removedObjects, newObjects, ((EObject) _newValue));
        break;
      case Notification.ADD_MANY:
        Object _newValue_1 = notif.getNewValue();
        for (final EObject add : ((List<EObject>) _newValue_1)) {
          BatchModelChangeListener.addToNewObjects(eventuallyRemoved, removedObjects, newObjects, add);
        }
        break;
      case Notification.REMOVE:
        Object _oldValue = notif.getOldValue();
        BatchModelChangeListener.addToRemovedObjects(eventuallyRemoved, removedObjects, newObjects, ((EObject) _oldValue));
        break;
      case Notification.REMOVE_MANY:
        Object _oldValue_1 = notif.getOldValue();
        for (final EObject remove : ((List<EObject>) _oldValue_1)) {
          BatchModelChangeListener.addToNewObjects(eventuallyRemoved, removedObjects, newObjects, remove);
        }
        break;
    }
  }
  
  public void cleanUp() {
    final Function1<Resource, Boolean> _function = (Resource r) -> {
      return Boolean.valueOf((r != null));
    };
    Iterable<Resource> _filter = IterableExtensions.<Resource>filter(this.observedResources, _function);
    for (final Resource r : _filter) {
      r.eAdapters().remove(this.adapter);
    }
  }
  
  private static <T extends Object> boolean addUnique(final List<T> list, final T o) {
    boolean _xifexpression = false;
    boolean _contains = list.contains(o);
    boolean _not = (!_contains);
    if (_not) {
      _xifexpression = list.add(o);
    }
    return _xifexpression;
  }
}
