/*
 * Decompiled with CFR 0.152.
 */
package jdk.internal.jimage;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.IntBuffer;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;
import jdk.internal.jimage.BasicImageReader;
import jdk.internal.jimage.ImageLocation;

public final class ImageReader
implements AutoCloseable {
    private final SharedImageReader reader;
    private volatile boolean closed;

    private ImageReader(SharedImageReader reader) {
        this.reader = reader;
    }

    public static ImageReader open(Path imagePath, ByteOrder byteOrder) throws IOException {
        Objects.requireNonNull(imagePath);
        Objects.requireNonNull(byteOrder);
        return SharedImageReader.open(imagePath, byteOrder);
    }

    public static ImageReader open(Path imagePath) throws IOException {
        return ImageReader.open(imagePath, ByteOrder.nativeOrder());
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            throw new IOException("image file already closed");
        }
        this.reader.close(this);
        this.closed = true;
    }

    private void ensureOpen() throws IOException {
        if (this.closed) {
            throw new IOException("image file closed");
        }
    }

    private void requireOpen() {
        if (this.closed) {
            throw new IllegalStateException("image file closed");
        }
    }

    public Node findNode(String name) throws IOException {
        this.ensureOpen();
        return this.reader.findNode(name);
    }

    public byte[] getResource(Node node) throws IOException {
        this.ensureOpen();
        return this.reader.getResource(node);
    }

    public static void releaseByteBuffer(ByteBuffer buffer) {
        BasicImageReader.releaseByteBuffer(buffer);
    }

    public ByteBuffer getResourceBuffer(Node node) {
        this.requireOpen();
        if (!node.isResource()) {
            throw new IllegalArgumentException("Not a resource node: " + node);
        }
        return this.reader.getResourceBuffer(node.getLocation());
    }

    private static final class SharedImageReader
    extends BasicImageReader {
        private static final Map<Path, SharedImageReader> OPEN_FILES = new HashMap<Path, SharedImageReader>();
        private static final String MODULES_ROOT = "/modules";
        private static final String PACKAGES_ROOT = "/packages";
        private static final int INITIAL_NODE_CACHE_CAPACITY = 2000;
        private final Set<ImageReader> openers = new HashSet<ImageReader>();
        private final BasicFileAttributes imageFileAttributes;
        private final Map<String, Node> nodes;
        private final int modulesStringOffset;
        private final int packagesStringOffset;

        private SharedImageReader(Path imagePath, ByteOrder byteOrder) throws IOException {
            super(imagePath, byteOrder);
            this.imageFileAttributes = Files.readAttributes(imagePath, BasicFileAttributes.class, new LinkOption[0]);
            this.nodes = new HashMap<String, Node>(2000);
            this.modulesStringOffset = this.getModuleOffset("/modules/java.base");
            this.packagesStringOffset = this.getModuleOffset("/packages/java.lang");
            Directory packages = this.newDirectory(PACKAGES_ROOT);
            this.nodes.put(packages.getName(), packages);
            Directory modules = this.newDirectory(MODULES_ROOT);
            this.nodes.put(modules.getName(), modules);
            Directory root = this.newDirectory("/");
            root.setChildren(Arrays.asList(packages, modules));
            this.nodes.put(root.getName(), root);
        }

        private int getModuleOffset(String path) {
            ImageLocation location = this.findLocation(path);
            assert (location != null) : "Cannot find expected jimage location: " + path;
            int offset = location.getModuleOffset();
            assert (offset != 0) : "Invalid module offset for jimage location: " + path;
            return offset;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private static ImageReader open(Path imagePath, ByteOrder byteOrder) throws IOException {
            Objects.requireNonNull(imagePath);
            Objects.requireNonNull(byteOrder);
            Map<Path, SharedImageReader> map = OPEN_FILES;
            synchronized (map) {
                SharedImageReader reader = OPEN_FILES.get(imagePath);
                if (reader == null) {
                    reader = new SharedImageReader(imagePath, byteOrder);
                    OPEN_FILES.put(imagePath, reader);
                } else if (reader.getByteOrder() != byteOrder) {
                    throw new IOException("\"" + reader.getName() + "\" is not an image file");
                }
                ImageReader image = new ImageReader(reader);
                reader.openers.add(image);
                return image;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void close(ImageReader image) throws IOException {
            Objects.requireNonNull(image);
            Map<Path, SharedImageReader> map = OPEN_FILES;
            synchronized (map) {
                if (!this.openers.remove(image)) {
                    throw new IOException("image file already closed");
                }
                if (this.openers.isEmpty()) {
                    this.close();
                    this.nodes.clear();
                    if (!OPEN_FILES.remove(this.getImagePath(), this)) {
                        throw new IOException("image file not found in open list");
                    }
                }
            }
        }

        synchronized Node findNode(String name) {
            Node node = this.nodes.get(name);
            if (node == null) {
                if (name.startsWith("/modules/")) {
                    node = this.buildModulesNode(name);
                } else if (name.startsWith("/packages/")) {
                    node = this.buildPackagesNode(name);
                }
                if (node != null) {
                    this.nodes.put(node.getName(), node);
                }
            } else if (!node.isCompleted()) {
                assert (node instanceof Directory) : "Invalid incomplete node: " + node;
                this.completeDirectory((Directory)node);
            }
            assert (node == null || node.isCompleted()) : "Incomplete node: " + node;
            return node;
        }

        private Node buildModulesNode(String name) {
            assert (name.startsWith("/modules/")) : "Invalid module node name: " + name;
            ImageLocation loc = this.findLocation(name);
            if (loc != null) {
                assert (name.equals(loc.getFullName())) : "Mismatched location for directory: " + name;
                assert (this.isModulesSubdirectory(loc)) : "Invalid modules directory: " + name;
                return this.completeModuleDirectory(this.newDirectory(name), loc);
            }
            loc = this.findLocation(name.substring(MODULES_ROOT.length()));
            return loc != null && this.isResource(loc) ? this.newResource(name, loc) : null;
        }

        private Node buildPackagesNode(String name) {
            ImageLocation loc;
            int packageStart = PACKAGES_ROOT.length() + 1;
            int packageEnd = name.indexOf(47, packageStart);
            if (packageEnd == -1) {
                ImageLocation loc2 = this.findLocation(name);
                return loc2 != null ? this.completePackageDirectory(this.newDirectory(name), loc2) : null;
            }
            String dirName = name.substring(0, packageEnd);
            if (!this.nodes.containsKey(dirName) && (loc = this.findLocation(dirName)) != null) {
                this.nodes.put(dirName, this.completePackageDirectory(this.newDirectory(dirName), loc));
                return this.nodes.get(name);
            }
            return null;
        }

        private void completeDirectory(Directory dir) {
            String name = dir.getName();
            assert (name.startsWith(MODULES_ROOT) || name.startsWith(PACKAGES_ROOT));
            ImageLocation loc = this.findLocation(name);
            assert (loc != null && name.equals(loc.getFullName())) : "Invalid location for name: " + name;
            if (name.charAt(1) == 'm') {
                this.completeModuleDirectory(dir, loc);
            } else {
                this.completePackageDirectory(dir, loc);
            }
            assert (dir.isCompleted()) : "Directory must be complete by now: " + dir;
        }

        private Directory completeModuleDirectory(Directory dir, ImageLocation loc) {
            assert (dir.getName().equals(loc.getFullName())) : "Mismatched location for directory: " + dir;
            List<Node> children = this.createChildNodes(loc, childLoc -> {
                if (this.isModulesSubdirectory((ImageLocation)childLoc)) {
                    return this.nodes.computeIfAbsent(childLoc.getFullName(), this::newDirectory);
                }
                String resourceName = childLoc.getFullName(true);
                return this.nodes.computeIfAbsent(resourceName, n -> this.newResource((String)n, (ImageLocation)childLoc));
            });
            dir.setChildren(children);
            return dir;
        }

        private Directory completePackageDirectory(Directory dir, ImageLocation loc) {
            List<Node> children;
            assert (dir.getName().equals(loc.getFullName())) : "Mismatched location for directory: " + dir;
            if (dir.getName().equals(PACKAGES_ROOT)) {
                children = this.createChildNodes(loc, c -> this.nodes.computeIfAbsent(c.getFullName(), this::newDirectory));
            } else {
                IntBuffer intBuffer = this.getOffsetBuffer(loc);
                int offsetCount = intBuffer.capacity();
                assert ((offsetCount & 1) == 0) : "Offset count must be even: " + offsetCount;
                children = new ArrayList<Node>(offsetCount / 2);
                for (int i = 1; i < offsetCount; i += 2) {
                    String moduleName = this.getString(intBuffer.get(i));
                    children.add(this.nodes.computeIfAbsent(dir.getName() + "/" + moduleName, n -> this.newLinkNode((String)n, "/modules/" + moduleName)));
                }
            }
            dir.setChildren(children);
            return dir;
        }

        private List<Node> createChildNodes(ImageLocation loc, Function<ImageLocation, Node> newChildFn) {
            IntBuffer offsets = this.getOffsetBuffer(loc);
            int childCount = offsets.capacity();
            ArrayList<Node> children = new ArrayList<Node>(childCount);
            for (int i = 0; i < childCount; ++i) {
                children.add(newChildFn.apply(this.getLocation(offsets.get(i))));
            }
            return children;
        }

        private IntBuffer getOffsetBuffer(ImageLocation dir) {
            assert (!this.isResource(dir)) : "Not a directory: " + dir.getFullName();
            byte[] offsets = this.getResource(dir);
            ByteBuffer buffer = ByteBuffer.wrap(offsets);
            buffer.order(this.getByteOrder());
            return buffer.asIntBuffer();
        }

        private boolean isResource(ImageLocation loc) {
            int moduleOffset = loc.getModuleOffset();
            return moduleOffset != 0 && moduleOffset != this.modulesStringOffset && moduleOffset != this.packagesStringOffset;
        }

        private boolean isModulesSubdirectory(ImageLocation loc) {
            return loc.getModuleOffset() == this.modulesStringOffset;
        }

        private Directory newDirectory(String name) {
            return new Directory(name, this.imageFileAttributes);
        }

        private Resource newResource(String name, ImageLocation loc) {
            assert (name.equals(loc.getFullName(true))) : "Mismatched location for resource: " + name;
            return new Resource(name, loc, this.imageFileAttributes);
        }

        private LinkNode newLinkNode(String name, String targetName) {
            return new LinkNode(name, () -> this.findNode(targetName), this.imageFileAttributes);
        }

        private byte[] getResource(Node node) throws IOException {
            if (node.isResource()) {
                return super.getResource(node.getLocation());
            }
            throw new IOException("Not a resource: " + node);
        }
    }

    public static abstract class Node {
        private final String name;
        private final BasicFileAttributes fileAttrs;

        protected Node(String name, BasicFileAttributes fileAttrs) {
            this.name = Objects.requireNonNull(name);
            this.fileAttrs = Objects.requireNonNull(fileAttrs);
        }

        boolean isCompleted() {
            return true;
        }

        ImageLocation getLocation() {
            throw new IllegalStateException("not a resource: " + this.getName());
        }

        public final String getName() {
            return this.name;
        }

        public final BasicFileAttributes getFileAttributes() {
            return this.fileAttrs;
        }

        public final Node resolveLink() {
            return this.resolveLink(false);
        }

        public Node resolveLink(boolean recursive) {
            return this;
        }

        public boolean isLink() {
            return false;
        }

        public boolean isDirectory() {
            return false;
        }

        public boolean isResource() {
            return false;
        }

        public Stream<String> getChildNames() {
            throw new IllegalStateException("not a directory: " + this.getName());
        }

        public long size() {
            return 0L;
        }

        public long compressedSize() {
            return 0L;
        }

        public String extension() {
            return null;
        }

        public final String toString() {
            return this.getName();
        }

        public final int hashCode() {
            return this.name.hashCode();
        }

        public final boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (other instanceof Node) {
                return this.name.equals(((Node)other).name);
            }
            return false;
        }
    }

    private static class LinkNode
    extends Node {
        private final Supplier<Node> link;

        private LinkNode(String name, Supplier<Node> link, BasicFileAttributes fileAttrs) {
            super(name, fileAttrs);
            this.link = link;
        }

        @Override
        public Node resolveLink(boolean recursive) {
            return this.link.get();
        }

        @Override
        public boolean isLink() {
            return true;
        }
    }

    private static class Resource
    extends Node {
        private final ImageLocation loc;

        private Resource(String name, ImageLocation loc, BasicFileAttributes fileAttrs) {
            super(name, fileAttrs);
            this.loc = loc;
        }

        @Override
        ImageLocation getLocation() {
            return this.loc;
        }

        @Override
        public boolean isResource() {
            return true;
        }

        @Override
        public long size() {
            return this.loc.getUncompressedSize();
        }

        @Override
        public long compressedSize() {
            return this.loc.getCompressedSize();
        }

        @Override
        public String extension() {
            return this.loc.getExtension();
        }
    }

    private static final class Directory
    extends Node {
        private List<Node> children = null;

        private Directory(String name, BasicFileAttributes fileAttrs) {
            super(name, fileAttrs);
        }

        @Override
        boolean isCompleted() {
            return this.children != null;
        }

        @Override
        public boolean isDirectory() {
            return true;
        }

        @Override
        public Stream<String> getChildNames() {
            if (this.children != null) {
                return this.children.stream().map(Node::getName);
            }
            throw new IllegalStateException("Cannot get child nodes of an incomplete directory: " + this.getName());
        }

        private void setChildren(List<Node> children) {
            assert (this.children == null) : this + ": Cannot set child nodes twice!";
            this.children = Collections.unmodifiableList(children);
        }
    }
}

