/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.internal.rest.auth;

import java.security.Principal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import org.eclipse.kura.audit.AuditConstants;
import org.eclipse.kura.audit.AuditContext;
import org.eclipse.kura.internal.rest.auth.CertificateAuthenticationProvider;
import org.eclipse.kura.internal.rest.auth.ConfigurationAdminHelper;
import org.eclipse.kura.internal.rest.auth.RestSessionHelper;
import org.eclipse.kura.internal.rest.auth.dto.AuthenticationInfoDTO;
import org.eclipse.kura.internal.rest.auth.dto.AuthenticationResponseDTO;
import org.eclipse.kura.internal.rest.auth.dto.IdentityInfoDTO;
import org.eclipse.kura.internal.rest.auth.dto.UpdatePasswordDTO;
import org.eclipse.kura.internal.rest.auth.dto.UsernamePasswordDTO;
import org.eclipse.kura.internal.rest.auth.dto.XsrfTokenDTO;
import org.eclipse.kura.internal.rest.provider.RestServiceOptions;
import org.eclipse.kura.request.handler.jaxrs.DefaultExceptionHandler;
import org.eclipse.kura.util.useradmin.UserAdminHelper;
import org.eclipse.kura.util.validation.PasswordStrengthValidators;
import org.eclipse.kura.util.validation.Validator;
import org.eclipse.kura.util.validation.ValidatorOptions;
import org.osgi.service.cm.ConfigurationAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Path(value="/session/v1")
public class SessionRestService {
    private static final String AUDIT_FORMAT_STRING = "{} Rest - Failure - {}";
    private static final String INVALID_SESSION_MESSAGE = "Current session is not valid";
    private static final String BAD_USERNAME_OR_PASSWORD_MESSAGE = "Authentication failed as username or password not matching";
    private static final String PASSWORD_CHANGE_SAME_PASSWORD_MESSAGE = "Password change failed as previous password equals new one";
    private static final String IDENTITY_NOT_IN_ROLE_MESSAGE = "Identity does not have the required permissions";
    private static final Logger auditLogger = LoggerFactory.getLogger((String)"AuditLogger");
    private final UserAdminHelper userAdminHelper;
    private final RestSessionHelper restSessionHelper;
    private final ConfigurationAdmin configAdmin;
    private RestServiceOptions options;

    public SessionRestService(UserAdminHelper userAdminHelper, RestSessionHelper restSessionHelper, ConfigurationAdmin configurationAdmin) {
        this.userAdminHelper = userAdminHelper;
        this.restSessionHelper = restSessionHelper;
        this.configAdmin = configurationAdmin;
    }

    public void setOptions(RestServiceOptions options) {
        this.options = options;
    }

    @POST
    @Path(value="/login/password")
    @Consumes(value={"application/json"})
    @Produces(value={"application/json"})
    public AuthenticationResponseDTO authenticateWithUsernameAndPassword(UsernamePasswordDTO usernamePassword, @Context HttpServletRequest request) {
        if (!this.options.isSessionManagementEnabled() || !this.options.isPasswordAuthEnabled()) {
            throw new WebApplicationException(Response.Status.NOT_FOUND);
        }
        AuditContext auditContext = AuditContext.currentOrInternal();
        usernamePassword.validate();
        auditContext.getProperties().put(AuditConstants.KEY_IDENTITY.getValue(), usernamePassword.getUsername());
        try {
            this.userAdminHelper.verifyUsernamePassword(usernamePassword.getUsername(), usernamePassword.getPassword());
            HttpSession session = this.restSessionHelper.createNewAuthenticatedSession(request, usernamePassword.getUsername());
            AuthenticationResponseDTO response = this.buildAuthenticationResponse(usernamePassword.getUsername());
            if (response.isPasswordChangeNeeded()) {
                this.restSessionHelper.lockSession(session);
            }
            auditLogger.info("{} Rest - Success - Create session via password authentication succeeded", (Object)auditContext);
            return response;
        }
        catch (UserAdminHelper.AuthenticationException e) {
            this.handleAuthenticationException(e);
            throw new IllegalStateException("unreachable");
        }
    }

    @POST
    @Path(value="/login/certificate")
    @Produces(value={"application/json"})
    public AuthenticationResponseDTO authenticateWithCertificate(@Context HttpServletRequest request, @Context ContainerRequestContext requestContext) {
        if (!this.options.isSessionManagementEnabled() || !this.options.isCertificateAuthEnabled()) {
            throw new WebApplicationException(Response.Status.NOT_FOUND);
        }
        CertificateAuthenticationProvider certificateAuthProvider = new CertificateAuthenticationProvider(this.userAdminHelper);
        Optional<Principal> principal = certificateAuthProvider.authenticate(requestContext, "Create session via certificate authentication");
        if (!principal.isPresent()) {
            throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.UNAUTHORIZED, (String)"Certificate authentication failed");
        }
        this.restSessionHelper.createNewAuthenticatedSession(request, principal.get().getName());
        return this.buildAuthenticationResponse(principal.get().getName());
    }

    @GET
    @Path(value="/xsrfToken")
    public XsrfTokenDTO getXSRFToken(@Context HttpServletRequest request, @Context ContainerRequestContext requestContext) {
        if (!this.options.isSessionManagementEnabled()) {
            throw new WebApplicationException(Response.Status.NOT_FOUND);
        }
        Optional<HttpSession> session = this.restSessionHelper.getExistingSession(request);
        if (!session.isPresent()) {
            throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.UNAUTHORIZED, (String)INVALID_SESSION_MESSAGE);
        }
        if (!this.restSessionHelper.getCurrentPrincipal(requestContext).isPresent()) {
            throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.UNAUTHORIZED, (String)INVALID_SESSION_MESSAGE);
        }
        return new XsrfTokenDTO(this.restSessionHelper.getOrCreateXsrfToken(session.get()));
    }

    @POST
    @Path(value="/changePassword")
    public void updateUserPassword(@Context ContainerRequestContext requestContext, @Context HttpServletRequest request, UpdatePasswordDTO passwordUpdate) {
        passwordUpdate.validate();
        try {
            Optional username = this.restSessionHelper.getCurrentPrincipal(requestContext).flatMap(c -> Optional.ofNullable(c.getName()));
            if (!username.isPresent()) {
                throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.UNAUTHORIZED, (String)INVALID_SESSION_MESSAGE);
            }
            this.userAdminHelper.verifyUsernamePassword((String)username.get(), passwordUpdate.getCurrentPassword());
            String newPassword = passwordUpdate.getNewPassword();
            this.validatePasswordStrength(newPassword);
            this.userAdminHelper.changeUserPassword((String)username.get(), passwordUpdate.getNewPassword());
            Optional<HttpSession> session = this.restSessionHelper.getExistingSession(request);
            if (session.isPresent()) {
                this.restSessionHelper.unlockSession(session.get());
            }
        }
        catch (UserAdminHelper.AuthenticationException e) {
            this.handleAuthenticationException(e);
        }
    }

    @POST
    @Path(value="/logout")
    public void logout(@Context HttpServletRequest request, @Context HttpServletResponse response, @Context ContainerRequestContext requestContext) {
        if (!this.options.isSessionManagementEnabled()) {
            throw new WebApplicationException(Response.Status.NOT_FOUND);
        }
        if (!this.restSessionHelper.getCurrentPrincipal(requestContext).isPresent()) {
            throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.UNAUTHORIZED, (String)INVALID_SESSION_MESSAGE);
        }
        this.restSessionHelper.logout(request, response);
        auditLogger.info("{} Rest - Success - Logout succeeded", (Object)AuditContext.currentOrInternal());
    }

    @GET
    @Path(value="/currentIdentity")
    @Produces(value={"application/json"})
    public IdentityInfoDTO getCurrentIdentityInfo(@Context ContainerRequestContext requestContext, @Context HttpServletRequest request) {
        if (!this.options.isSessionManagementEnabled()) {
            throw new WebApplicationException(Response.Status.NOT_FOUND);
        }
        Optional<Principal> currentPrincipal = this.restSessionHelper.getCurrentPrincipal(requestContext);
        if (!currentPrincipal.isPresent()) {
            throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.UNAUTHORIZED, (String)INVALID_SESSION_MESSAGE);
        }
        String identityName = currentPrincipal.get().getName();
        Set permissions = this.userAdminHelper.getIdentityPermissions(identityName);
        boolean needsPasswordChange = this.userAdminHelper.isPasswordChangeRequired(identityName);
        return new IdentityInfoDTO(identityName, needsPasswordChange, permissions);
    }

    @GET
    @Path(value="/authenticationInfo")
    @Produces(value={"application/json"})
    public AuthenticationInfoDTO getAuthenticationMethodInfo() {
        boolean isPasswordAuthEnabled = this.options.isPasswordAuthEnabled();
        boolean isCertificateAuthenticationEnabled = this.options.isCertificateAuthEnabled();
        Map<String, Object> consoleConfig = ConfigurationAdminHelper.loadConsoleConfigurationProperties(this.configAdmin);
        String message = ConfigurationAdminHelper.getLoginMessage(consoleConfig).orElse(null);
        if (!isCertificateAuthenticationEnabled) {
            return new AuthenticationInfoDTO(isPasswordAuthEnabled, false, null, message);
        }
        Map<String, Object> httpServiceConfig = ConfigurationAdminHelper.loadHttpServiceConfigurationProperties(this.configAdmin);
        Set<Integer> httpsClientAuthPorts = ConfigurationAdminHelper.getHttpsMutualAuthPorts(httpServiceConfig);
        if (!httpsClientAuthPorts.isEmpty()) {
            return new AuthenticationInfoDTO(isPasswordAuthEnabled, true, httpsClientAuthPorts, message);
        }
        return new AuthenticationInfoDTO(isPasswordAuthEnabled, false, null, message);
    }

    private void validatePasswordStrength(String newPassword) {
        ValidatorOptions validationOptions = new ValidatorOptions(ConfigurationAdminHelper.loadConsoleConfigurationProperties(this.configAdmin));
        List validators = PasswordStrengthValidators.fromConfig((ValidatorOptions)validationOptions);
        ArrayList errors = new ArrayList();
        for (Validator validator : validators) {
            validator.validate((Object)newPassword, errors::add);
            if (errors.isEmpty()) continue;
            throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.BAD_REQUEST, (String)("The new password does not satisfy password strenght requirements: " + (String)errors.get(0)));
        }
    }

    private AuthenticationResponseDTO buildAuthenticationResponse(String username) {
        boolean needsPasswordChange = this.userAdminHelper.isPasswordChangeRequired(username);
        return new AuthenticationResponseDTO(needsPasswordChange);
    }

    private void handleAuthenticationException(UserAdminHelper.AuthenticationException e) {
        AuditContext auditContext = AuditContext.currentOrInternal();
        switch (e.getReason()) {
            case USER_NOT_FOUND: 
            case INCORRECT_PASSWORD: {
                auditLogger.warn(AUDIT_FORMAT_STRING, (Object)auditContext, (Object)BAD_USERNAME_OR_PASSWORD_MESSAGE);
                throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.UNAUTHORIZED, (String)BAD_USERNAME_OR_PASSWORD_MESSAGE);
            }
            case PASSWORD_CHANGE_WITH_SAME_PASSWORD: {
                auditLogger.warn(AUDIT_FORMAT_STRING, (Object)auditContext, (Object)PASSWORD_CHANGE_SAME_PASSWORD_MESSAGE);
                throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.BAD_REQUEST, (String)PASSWORD_CHANGE_SAME_PASSWORD_MESSAGE);
            }
            case USER_NOT_IN_ROLE: {
                auditLogger.warn(AUDIT_FORMAT_STRING, (Object)auditContext, (Object)IDENTITY_NOT_IN_ROLE_MESSAGE);
                throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.FORBIDDEN, (String)IDENTITY_NOT_IN_ROLE_MESSAGE);
            }
        }
        throw DefaultExceptionHandler.buildWebApplicationException((Response.Status)Response.Status.INTERNAL_SERVER_ERROR, (String)"An internal error occurred");
    }
}

