"use strict";
/*********************************************************************
 * Copyright (c) 2018 Ericsson and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const path = require("path");
const utils_1 = require("./utils");
const sinon_1 = require("sinon");
describe('Disassembly Test Suite', function () {
    let dc;
    let frame;
    const disProgram = path.join(utils_1.testProgramsDir, 'disassemble');
    const disSrc = path.join(utils_1.testProgramsDir, 'disassemble.c');
    const expectsGeneralDisassemble = (disassemble, length, ignoreEmptyInstructions) => {
        (0, chai_1.expect)(disassemble).not.eq(undefined);
        (0, chai_1.expect)(disassemble.body).not.eq(undefined);
        if (disassemble.body) {
            const instructions = disassemble.body.instructions;
            (0, chai_1.expect)(instructions).to.have.lengthOf(length);
            // the contents of the instructions are platform dependent, so instead
            // make sure we have read fully
            for (const i of instructions) {
                (0, chai_1.expect)(i.address).to.have.lengthOf.greaterThan(0);
                (0, chai_1.expect)(i.instruction).to.have.lengthOf.greaterThan(0);
                if (!ignoreEmptyInstructions) {
                    (0, chai_1.expect)(i.instructionBytes).to.have.lengthOf.greaterThan(0);
                }
            }
        }
    };
    beforeEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            dc = yield (0, utils_1.standardBeforeEach)();
            yield dc.hitBreakpoint((0, utils_1.fillDefaults)(this.currentTest, {
                program: disProgram,
            }), {
                path: disSrc,
                line: 2,
            });
            const threads = yield dc.threadsRequest();
            // On windows additional threads can exist to handle signals, therefore find
            // the real thread & frame running the user code. The other thread will
            // normally be running code from ntdll or similar.
            loop_threads: for (const thread of threads.body.threads) {
                const stack = yield dc.stackTraceRequest({ threadId: thread.id });
                if (stack.body.stackFrames.length >= 1) {
                    for (const f of stack.body.stackFrames) {
                        if (f.source && f.source.name === 'disassemble.c') {
                            frame = f;
                            break loop_threads;
                        }
                    }
                }
            }
            // Make sure we found the expected frame
            (0, chai_1.expect)(frame).not.eq(undefined);
        });
    });
    afterEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        });
    });
    it('can disassemble', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const disassemble = (yield dc.send('disassemble', {
                memoryReference: 'main',
                instructionCount: 100,
            }));
            expectsGeneralDisassemble(disassemble, 100);
        });
    });
    it('can disassemble with no source references', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // In this case we attempt to read from where there is no source,
            // GDB returns data in a different format in that case
            const disassemble = (yield dc.send('disassemble', {
                memoryReference: 'main+1000',
                instructionCount: 100,
            }));
            expectsGeneralDisassemble(disassemble, 100);
        });
    });
    it('can disassemble with negative offsets', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const disassemble = (yield dc.send('disassemble', {
                memoryReference: 'main',
                instructionOffset: -20,
                instructionCount: 20,
            }));
            expectsGeneralDisassemble(disassemble, 20, true);
        });
    });
    it('send error response handle on empty memory reference', function () {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield dc.send('disassemble', {
                    memoryReference: '',
                    instructionOffset: -20,
                    instructionCount: 20,
                });
                sinon_1.assert.fail('Should throw error!');
            }
            catch (e) {
                (0, chai_1.expect)(e).to.be.deep.equal(new Error('Target memory reference is not specified!'));
            }
        });
    });
    it('can disassemble with correct boundries', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const get = (disassemble, offset) => {
                var _a;
                const instruction = (_a = disassemble.body) === null || _a === void 0 ? void 0 : _a.instructions[offset];
                (0, chai_1.expect)(instruction).not.eq(undefined);
                // Instruction undefined already checked.
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                return instruction;
            };
            const expectsInstructionEquals = (instruction1, instruction2, message) => {
                (0, chai_1.expect)(instruction1.address).to.eq(instruction2.address, message);
            };
            const disassembleLower = (yield dc.send('disassemble', {
                memoryReference: 'main',
                instructionOffset: -20,
                instructionCount: 20,
            }));
            const disassembleMiddle = (yield dc.send('disassemble', {
                memoryReference: 'main',
                instructionOffset: -10,
                instructionCount: 20,
            }));
            const disassembleHigher = (yield dc.send('disassemble', {
                memoryReference: 'main',
                instructionOffset: 0,
                instructionCount: 20,
            }));
            expectsGeneralDisassemble(disassembleLower, 20, true);
            expectsGeneralDisassemble(disassembleMiddle, 20, true);
            expectsGeneralDisassemble(disassembleHigher, 20, true);
            // Current implementation have known edge cases, possibly instruction misaligning while
            // handling the negative offsets, please refer to the discussion at the following link:
            // https://github.com/eclipse-cdt-cloud/cdt-gdb-adapter/pull/341#discussion_r1857422980
            expectsInstructionEquals(get(disassembleLower, 15), get(disassembleMiddle, 5), 'lower[15] should be same with middle[5]');
            expectsInstructionEquals(get(disassembleMiddle, 15), get(disassembleHigher, 5), 'middle[15] should be same with higher[5]');
        });
    });
    it('return error at bad address', function () {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield dc.send('disassemble', {
                    memoryReference: '0x0',
                    instructionCount: 10,
                });
                sinon_1.assert.fail('Should throw error!');
            }
            catch (e) {
                (0, chai_1.expect)(e).to.be.deep.equal(new Error('Cannot access memory at address 0x0'));
            }
        });
    });
});
//# sourceMappingURL=diassemble.spec.js.map