"use strict";
/********************************************************************************
 * Copyright (c) 2021 STMicroelectronics and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 *******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const file_util_1 = require("./file-util");
const chai_1 = require("chai");
const paths = require("path");
chai_1.use(require('chai-string'));
/* eslint-disable no-unused-expressions */
describe('file-util', function () {
    describe('isFile', function () {
        it('handles files and directories that exist', function () {
            const fromFile = file_util_1.isFile('package.json');
            chai_1.expect(fromFile).to.be.true;
            const fromDirectory = file_util_1.isFile(fromPortablePath('src/util'));
            chai_1.expect(fromDirectory).to.be.false;
        });
        it('handles paths that do not exist', function () {
            const fromMissing = file_util_1.isFile(fromPortablePath('src/no/such/path'));
            chai_1.expect(fromMissing).to.be.false;
        });
    });
    describe('isDirectory', function () {
        it('handles files and directories that exist', function () {
            const fromFile = file_util_1.isDirectory('package.json');
            chai_1.expect(fromFile).to.be.false;
            const fromDirectory = file_util_1.isDirectory(fromPortablePath('src/util'));
            chai_1.expect(fromDirectory).to.be.true;
        });
        it('handles paths that do not exist', function () {
            const fromMissing = file_util_1.isDirectory(fromPortablePath('src/no/such/path'));
            chai_1.expect(fromMissing).to.be.false;
        });
    });
    describe('toConfigPath', function () {
        it('handles exact file', function () {
            const fromExactFile = file_util_1.toConfigPath(fromPortablePath('some/path/.clangd'));
            chai_1.expect(fromExactFile).to.equal(fromPortablePath('some/path/.clangd'));
        });
        // With a directory input, the result is resolved to an absolute path
        it('handles directory', function () {
            const fromDirectory = file_util_1.toConfigPath(fromPortablePath('some/path'));
            chai_1.expect(fromDirectory).to.endWith(fromPortablePath('/some/path/.clangd'));
            const fromDirectoryWithTrailingSlash = file_util_1.toConfigPath(fromPortablePath('some/path/'));
            chai_1.expect(fromDirectoryWithTrailingSlash).to.endWith(fromPortablePath('/some/path/.clangd'));
        });
        // With a directory input, the result is resolved to an absolute path
        it('handles path that happens to have .clangd extension', function () {
            const fromTrickyPath = file_util_1.toConfigPath(fromPortablePath('some/path/project.clangd'));
            chai_1.expect(fromTrickyPath).to.endWith(fromPortablePath('/some/path/project.clangd/.clangd'));
        });
    });
    describe('toPortablePath', function () {
        it('handles platform-specific path', function () {
            const pathSpec = 'this/is/a/path.c';
            const platformSpecificPath = fromPortablePath(pathSpec);
            chai_1.expect(file_util_1.toPortablePath(platformSpecificPath)).to.equal(pathSpec);
        });
    });
});
function fromPortablePath(path) {
    if (paths.sep !== '/') {
        return path.replace(/\/+/g, paths.sep);
    }
    return path;
}
//# sourceMappingURL=file-util.spec.js.map