/**
 * Copyright (c) 2010-2015, Denes Harmath, Istvan Rath and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.patternlanguage.emf.validation.whitelist;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.HashSet;
import org.apache.log4j.Logger;
import org.eclipse.viatra.query.patternlanguage.emf.validation.whitelist.IPureElementProvider;
import org.eclipse.viatra.query.patternlanguage.emf.validation.whitelist.IPureWhitelistExtensionProvider;
import org.eclipse.viatra.query.patternlanguage.emf.validation.whitelist.ServiceLoaderBasedWhitelistExtensionProvider;
import org.eclipse.viatra.query.runtime.util.ViatraQueryLoggingUtil;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * A whitelist that contains pure elements.
 * @since 2.0
 */
@Singleton
@SuppressWarnings("all")
public class PureWhitelist {
  /**
   * Declares that a method is pure or all methods in a class or package are pure.
   */
  @Data
  public static class PureElement {
    public enum Type {
      METHOD,
      
      CLASS,
      
      PACKAGE;
    }
    
    private final String fullyQualifiedName;
    
    private final PureWhitelist.PureElement.Type type;
    
    public boolean covers(final JvmOperation jvmOperation) {
      boolean _xifexpression = false;
      boolean _eIsProxy = jvmOperation.eIsProxy();
      if (_eIsProxy) {
        _xifexpression = false;
      } else {
        boolean _xblockexpression = false;
        {
          String _switchResult = null;
          final PureWhitelist.PureElement.Type type = this.type;
          if (type != null) {
            switch (type) {
              case METHOD:
                _switchResult = jvmOperation.getIdentifier();
                break;
              case CLASS:
                JvmDeclaredType _declaringType = jvmOperation.getDeclaringType();
                String _identifier = null;
                if (_declaringType!=null) {
                  _identifier=_declaringType.getIdentifier();
                }
                _switchResult = _identifier;
                break;
              case PACKAGE:
                JvmDeclaredType _declaringType_1 = jvmOperation.getDeclaringType();
                String _packageName = null;
                if (_declaringType_1!=null) {
                  _packageName=_declaringType_1.getPackageName();
                }
                _switchResult = _packageName;
                break;
              default:
                break;
            }
          }
          final String qualifiedNameToCheck = _switchResult;
          _xblockexpression = Objects.equal(qualifiedNameToCheck, this.fullyQualifiedName);
        }
        _xifexpression = _xblockexpression;
      }
      return _xifexpression;
    }
    
    public PureElement(final String fullyQualifiedName, final PureWhitelist.PureElement.Type type) {
      super();
      this.fullyQualifiedName = fullyQualifiedName;
      this.type = type;
    }
    
    @Override
    @Pure
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((this.fullyQualifiedName== null) ? 0 : this.fullyQualifiedName.hashCode());
      return prime * result + ((this.type== null) ? 0 : this.type.hashCode());
    }
    
    @Override
    @Pure
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      PureWhitelist.PureElement other = (PureWhitelist.PureElement) obj;
      if (this.fullyQualifiedName == null) {
        if (other.fullyQualifiedName != null)
          return false;
      } else if (!this.fullyQualifiedName.equals(other.fullyQualifiedName))
        return false;
      if (this.type == null) {
        if (other.type != null)
          return false;
      } else if (!this.type.equals(other.type))
        return false;
      return true;
    }
    
    @Override
    @Pure
    public String toString() {
      ToStringBuilder b = new ToStringBuilder(this);
      b.add("fullyQualifiedName", this.fullyQualifiedName);
      b.add("type", this.type);
      return b.toString();
    }
    
    @Pure
    public String getFullyQualifiedName() {
      return this.fullyQualifiedName;
    }
    
    @Pure
    public PureWhitelist.PureElement.Type getType() {
      return this.type;
    }
  }
  
  private final HashSet<PureWhitelist.PureElement> pureElements = CollectionLiterals.<PureWhitelist.PureElement>newHashSet();
  
  private boolean initialized = false;
  
  private final IPureWhitelistExtensionProvider extensionProvider;
  
  private final Logger logger;
  
  public PureWhitelist() {
    this.initialized = false;
    ServiceLoaderBasedWhitelistExtensionProvider _serviceLoaderBasedWhitelistExtensionProvider = new ServiceLoaderBasedWhitelistExtensionProvider();
    this.extensionProvider = _serviceLoaderBasedWhitelistExtensionProvider;
    this.logger = ViatraQueryLoggingUtil.getLogger(PureWhitelist.class);
  }
  
  /**
   * If loadExtensions is false, instead of loading the list of known extensions start with an empty whitelist
   */
  public PureWhitelist(final boolean loadExtensions) {
    this.initialized = (!loadExtensions);
    ServiceLoaderBasedWhitelistExtensionProvider _serviceLoaderBasedWhitelistExtensionProvider = new ServiceLoaderBasedWhitelistExtensionProvider();
    this.extensionProvider = _serviceLoaderBasedWhitelistExtensionProvider;
    this.logger = ViatraQueryLoggingUtil.getLogger(PureWhitelist.class);
  }
  
  @Inject
  public PureWhitelist(final IPureWhitelistExtensionProvider extensionProvider, final Logger logger) {
    this.initialized = false;
    this.extensionProvider = extensionProvider;
    this.logger = logger;
  }
  
  public boolean contains(final JvmOperation jvmOperation) {
    boolean _xblockexpression = false;
    {
      this.loadKnownExtensions();
      final Function1<PureWhitelist.PureElement, Boolean> _function = (PureWhitelist.PureElement it) -> {
        return Boolean.valueOf(it.covers(jvmOperation));
      };
      _xblockexpression = IterableExtensions.<PureWhitelist.PureElement>exists(this.pureElements, _function);
    }
    return _xblockexpression;
  }
  
  public void add(final PureWhitelist.PureElement pureElement) {
    this.pureElements.add(pureElement);
  }
  
  public void loadKnownExtensions() {
    if ((!this.initialized)) {
      Iterable<IPureElementProvider> _pureElementExtensions = this.extensionProvider.getPureElementExtensions();
      for (final IPureElementProvider provider : _pureElementExtensions) {
        try {
          this.pureElements.addAll(provider.getPureElements());
        } catch (final Throwable _t) {
          if (_t instanceof Exception) {
            final Exception e = (Exception)_t;
            String _name = provider.getClass().getName();
            String _plus = ("Error while loading extensions from provider " + _name);
            this.logger.error(_plus, e);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      }
      this.initialized = true;
    }
  }
}
