/**
 * Query.java
 *
 * This class implements the query Web service.
 * 
 * @author Arthur Ryman
 */

package com.leagueplanet;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Set;
import java.util.Vector;

import com.leagueplanet.message.query.GetScheduleRequest;
import com.leagueplanet.model.Game;
import com.leagueplanet.model.League;
import com.leagueplanet.model.Location;
import com.leagueplanet.model.Schedule;
import com.leagueplanet.model.Score;
import com.leagueplanet.model.Team;
import com.leagueplanet.resource.schedule.GameType;
import com.leagueplanet.resource.schedule.ResourceType;
import com.leagueplanet.resource.schedule.ScheduleContent;
import com.leagueplanet.resource.schedule.ScheduleType;
import com.leagueplanet.resource.schedule.ScoreType;
import com.leagueplanet.services.IceHockeyFacade;
import com.leagueplanet.services.LeagueFacade;

public class Query {

	// use the ice hockey implementation for this service
	private LeagueFacade facade = IceHockeyFacade.getLeagueFacade();

	public ScheduleContent getSchedule(GetScheduleRequest request) {

		long scheduleId = request.getScheduleId();
		Schedule schedule = facade.findSchedule(scheduleId);

		ScheduleType scheduleType;
		if (schedule == null) {
			scheduleType = makeUnknownScheduleType(scheduleId);
		} else {
			scheduleType = makeScheduleType(schedule);
		}

		return new ScheduleContent(scheduleType);
	}

	private ScheduleType makeUnknownScheduleType(long scheduleId) {
		ScheduleType scheduleType = new ScheduleType();
		scheduleType.setId(scheduleId);
		scheduleType.setName("unknown schedule");
		scheduleType.setLeague(new ResourceType("unknown league", 0));
		scheduleType.setGames(new GameType[0]);

		return scheduleType;
	}

	private ScheduleType makeScheduleType(Schedule schedule) {
		ScheduleType scheduleType = new ScheduleType();
		scheduleType.setId(schedule.getId());
		scheduleType.setName(schedule.getName());
		scheduleType.setLeague(makeResourceType(schedule.getLeague()));
		scheduleType.setGames(makeGamesType(schedule.getEvents()));

		return scheduleType;
	}

	private GameType[] makeGamesType(Set events) {
		Iterator eventIterator = events.iterator();
		Vector games = new Vector();
		while (eventIterator.hasNext()) {
			Object event = eventIterator.next();
			if (event instanceof Game) {
				Game game = (Game) event;
				GameType gameType = makeGameType(game);
				games.add(gameType);
			}
		}
		GameType[] gamesType = new GameType[games.size()];
		games.copyInto(gamesType);

		// sort the games by date
		Arrays.sort(gamesType, new Comparator() {
			public int compare(Object o1, Object o2) {
				long t1 = ((GameType) o1).getDateTime().getTimeInMillis();
				long t2 = ((GameType) o2).getDateTime().getTimeInMillis();
				if (t1 < t2)
					return -1;
				if (t1 > t2)
					return 1;
				return 0;
			}
		});

		return gamesType;
	}

	private GameType makeGameType(Game game) {
		GameType gameType = new GameType();
		gameType.setId(game.getId());
		gameType.setDateTime(game.toCalendar());
		gameType.setArena(makeResourceType(game.getLocation()));
		gameType.setVisitor(makeResourceType(game.getVisitor()));
		gameType.setHome(makeResourceType(game.getHome()));
		gameType.setScore(makeScoreType(game.getScore()));

		return gameType;
	}

	private ScoreType makeScoreType(Score score) {
		return new ScoreType(score.getVisitor(), score.getHome());
	}

	private ResourceType makeResourceType(long id, String name) {

		return new ResourceType(name, id);
	}

	private ResourceType makeResourceType(Team team) {
		return makeResourceType(team.getId(), team.getName());
	}

	private ResourceType makeResourceType(League league) {
		return makeResourceType(league.getId(), league.getName());
	}

	private ResourceType makeResourceType(Location location) {
		return makeResourceType(location.getId(), location.getName());
	}

}
