// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.openqa.selenium.docker.client;

import static org.assertj.core.api.Assertions.assertThat;
import static org.openqa.selenium.json.Json.MAP_TYPE;
import static org.openqa.selenium.remote.http.Contents.utf8String;

import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Set;
import org.junit.jupiter.api.Test;
import org.openqa.selenium.docker.Image;
import org.openqa.selenium.docker.ImageId;
import org.openqa.selenium.docker.internal.Reference;
import org.openqa.selenium.json.Json;
import org.openqa.selenium.remote.http.HttpHandler;
import org.openqa.selenium.remote.http.HttpResponse;

class ListImagesTest {

  @Test
  void shouldReturnImageIfTagIsPresentWithApi140() {
    testListImages("1.40");
  }

  @Test
  void shouldReturnImageIfTagIsPresentWithApi144() {
    testListImages("1.44");
  }

  @Test
  void shouldHandleApi144ResponseWithoutVirtualSize() {
    // API 1.44+ removes VirtualSize field
    HttpHandler handler =
        req -> {
          String filters = req.getQueryParameter("filters");
          String decoded = URLDecoder.decode(filters, StandardCharsets.UTF_8);
          Map<String, Object> raw = new Json().toType(decoded, MAP_TYPE);

          Map<?, ?> rawRef = (Map<?, ?>) raw.get("reference");
          assertThat(rawRef.get("selenium/standalone-firefox:latest")).isEqualTo(true);

          return new HttpResponse()
              .addHeader("Content-Type", "application/json")
              .setContent(
                  utf8String(
                      "[{\"Containers\":-1,\"Created\":1581716253,"
                          + "\"Id\":\"sha256:bc24341497a00a3afbf04c518cb4bf98834d933ae331d1c5d3cd6f52c079049e\","
                          + "\"Labels\":{\"authors\":\"SeleniumHQ\"},\"ParentId\":\"\","
                          + "\"RepoDigests\":null,"
                          + "\"RepoTags\":[\"selenium/standalone-firefox:latest\"],"
                          + "\"SharedSize\":-1,\"Size\":765131593}]")); // No VirtualSize field
        };

    Reference reference = Reference.parse("selenium/standalone-firefox:latest");

    // Test with API 1.44 - should handle missing VirtualSize gracefully
    Set<Image> images = new ListImages(handler, "1.44").apply(reference);

    assertThat(images.size()).isEqualTo(1);
    Image image = images.iterator().next();

    assertThat(image.getId())
        .isEqualTo(
            new ImageId("sha256:bc24341497a00a3afbf04c518cb4bf98834d933ae331d1c5d3cd6f52c079049e"));
  }

  private void testListImages(String apiVersion) {

    HttpHandler handler =
        req -> {
          String filters = req.getQueryParameter("filters");
          String decoded = URLDecoder.decode(filters, StandardCharsets.UTF_8);
          Map<String, Object> raw = new Json().toType(decoded, MAP_TYPE);

          Map<?, ?> rawRef = (Map<?, ?>) raw.get("reference");
          assertThat(rawRef.get("selenium/standalone-firefox:latest")).isEqualTo(true);

          return new HttpResponse()
              .addHeader("Content-Type", "application/json")
              .setContent(
                  utf8String(
                      "[{\"Containers\":-1,\"Created\":1581716253,"
                          + "\"Id\":\"sha256:bc24341497a00a3afbf04c518cb4bf98834d933ae331d1c5d3cd6f52c079049e\","
                          + "\"Labels\":{\"authors\":\"SeleniumHQ\"},\"ParentId\":\"\","
                          + "\"RepoDigests\":null,"
                          + "\"RepoTags\":[\"selenium/standalone-firefox:latest\"],"
                          + "\"SharedSize\":-1,\"Size\":765131593,\"VirtualSize\":765131593}]"));
        };

    Reference reference = Reference.parse("selenium/standalone-firefox:latest");

    // Test with specified API version
    Set<Image> images = new ListImages(handler, apiVersion).apply(reference);

    assertThat(images.size()).isEqualTo(1);
    Image image = images.iterator().next();

    assertThat(image.getId())
        .isEqualTo(
            new ImageId("sha256:bc24341497a00a3afbf04c518cb4bf98834d933ae331d1c5d3cd6f52c079049e"));
  }
}
