import type { AnyDesc, DescEnum, DescExtension, DescFile, DescMessage } from "@bufbuild/protobuf";
import type { ImportSymbol } from "./import-symbol.js";
import type { RuntimeImports } from "./runtime-imports.js";
import type { JSDocBlock } from "./jsdoc.js";
import type { ExportDeclaration, LiteralProtoInt64, LiteralString, RefDescEnum, RefDescMessage } from "./opaque-printables.js";
/**
 * All types that can be passed to GeneratedFile.print()
 */
export type Printable = string | number | boolean | bigint | Uint8Array | ImportSymbol | ExportDeclaration | JSDocBlock | LiteralString | LiteralProtoInt64 | RefDescMessage | RefDescEnum | DescMessage | DescEnum | DescExtension | Printable[];
/**
 * FileInfo represents an intermediate type using for transpiling TypeScript internally.
 */
export interface FileInfo {
    name: string;
    content: string;
    preamble?: string | undefined;
}
/**
 * Represents a JavaScript, TypeScript, or TypeScript declaration file.
 */
export interface GeneratedFile {
    /**
     * Create a standard preamble the includes comments at the top of the
     * protobuf source file (like a license header), as well as information
     * about the code generator and its version.
     *
     * The preamble is always placed at the very top of the generated file,
     * above import statements.
     *
     * A file with a preamble but no other content is still considered empty,
     * and will not be generated unless the plugin option keep_empty_files=true
     * is set.
     */
    preamble(file: DescFile): void;
    /**
     * Add a line of code to the file.
     *
     * - string: Prints the string verbatim.
     * - number or boolean: Prints a literal.
     * - bigint: Prints an expression using protoInt64.parse().
     * - Uint8Array: Prints an expression that re-created the array.
     * - ImportSymbol: Adds an import statement and prints the name of the symbol.
     * - DescMessage or DescEnum: Imports the type if necessary, and prints the name.
     */
    print(...printables: Printable[]): void;
    /**
     * Add a line of code to the file with tagged template literal and
     * an optional array of Printables.
     * See print(Printable[]) for behavior when printing Printable items.
     */
    print(fragments: TemplateStringsArray, ...printables: Printable[]): void;
    /**
     * @deprecated Please use createImportSymbol() from @bufbuild/protoplugin/ecmascript instead
     */
    export(name: string): ImportSymbol;
    /**
     * Create a string literal.
     */
    string(string: string): Printable;
    /**
     * Create a JSDoc comment block with the given text. Line breaks and white-space
     * stay intact.
     */
    jsDoc(text: string, indentation?: string): JSDocBlock;
    /**
     * Create a JSDoc comment block for the given message, enumeration, or other
     * descriptor. The comment block will contain the original comments from the
     * protobuf source, and annotations such as `@generated from message MyMessage`.
     */
    jsDoc(desc: Exclude<AnyDesc, DescFile>, indentation?: string): JSDocBlock;
    /**
     * Create a printable export statement. For example:
     *
     * ```ts
     * f.print(f.exportDecl("abstract class", "MyClass"), " {}")
     * ```
     *
     * Will generate as:
     * ```ts
     * export abstract class MyClass {}
     * ```
     *
     * Using this method is preferred over a calling print() with a literal export
     * statement. If the plugin option `js_import_style=legacy_commonjs` is set,
     * exports will automatically be generated for CommonJS.
     */
    exportDecl(declaration: string, name: string | DescMessage | DescEnum | DescExtension): Printable;
    /**
     * Import a message or enumeration generated by protoc-gen-es.
     */
    import(type: DescMessage | DescEnum | DescExtension): ImportSymbol;
    /**
     * Import any symbol from a file or package.
     *
     * The import path can point to a package, for example `@foo/bar/baz.js`, or
     * to a file, for example `./bar/baz.js`.
     *
     * Note that while paths to a file begin with a `./`, they must be
     * relative to the project root. The import path is automatically made
     * relative to the current file.
     */
    import(name: string, from: string): ImportSymbol;
    /**
     * In case you need full control over exports and imports, use print() and
     * formulate your own imports and exports based on this property.
     */
    readonly jsImportStyle: "module" | "legacy_commonjs";
}
export interface GeneratedFileController extends GeneratedFile {
    getFileInfo(): FileInfo;
}
type CreateTypeImportFn = (desc: DescMessage | DescEnum | DescExtension) => ImportSymbol;
type RewriteImportPathFn = (path: string) => string;
type CreatePreambleFn = (descFile: DescFile) => string;
export declare function createGeneratedFile(name: string, importPath: string, jsImportStyle: "module" | "legacy_commonjs", rewriteImportPath: RewriteImportPathFn, createTypeImport: CreateTypeImportFn, runtimeImports: RuntimeImports, createPreamble: CreatePreambleFn): GeneratedFileController;
export {};
