/*
 * Copyright (C) 1999 Accessible Linux Group (Rishi Dubey, Aaron Klish,
 *                                            Kristian Rickert, Shane Smith)
 * Copyright (C) 2000-2002 Daniel Linder / UIUC DRES
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *   USA
 *
 */

#include <X11/Xlib.h>
#include <X11/XKBlib.h>
#include "Access.h"

//
// Access()
//

Access::Access()
{
  this->enabled_ctrls = 0;               /* xkb->ctrls->enabled_ctrls   */
  this->accessx_delay = 0;               /* xkb->ctrls->ax_timeout      */

  this->mk_delay = 0;                    /* xkb->ctrls->mk_delay        */
  this->mk_interval = 0;                 /* xkb->ctrls->mk_interval     */
  this->mk_time_to_max = 0;              /* xkb->ctrls->mk_time_to_max  */
  this->mk_max_speed = 0;                /* xkb->ctrls->mk_max_speed    */
  this->mk_curve = 0;                    /* xkb->ctrls->mk_curve        */

  this->slow_keys_delay = 0;             /*xkb->ctrls->slow_keys_delay  */

  this->enabled_sticky_key_options = 0;  /* xkb->ctrls->ax_options      */

  this->debounce_delay = 0;              /* xkb->ctrls->debounce_delay  */

  this->repeat_delay = 0;                /* xkb->ctrls->repeat_delay    */
  this->repeat_interval = 0;             /* xkb->ctrls->repeat_interval */

}

//
// ~Access
//

Access::~Access()
{
} 

//
// ShowDiagnostics()
//

void Access::ShowDiagnostics()
{
  cout << "AccessX Timeout:       " << onoroff(GetAccessXTOState()) << endl;
  cout << "AccessX Timeout Value: " << GetAccessXDelay() << endl;

  cout << endl;

  cout << "MouseKeys:             " << onoroff(GetMouseKeysState()) << endl;
  cout << "  - Delay:             " << GetMouseKeysDelay() << endl;
  cout << "  - Interval:          " << GetMouseKeysInterval() << endl;
  cout << "  - T to Max:          " << GetMouseKeysTimeToMax() << endl;
  cout << "  - Max Speed:         " << GetMouseKeysMaxSpeed() << endl;
  cout << "  - Curve:             " << GetMouseKeysCurve() << endl;

  cout << endl;

  cout << "SlowKeys:              " << onoroff(GetSlowKeysState()) << endl;
  cout << "  - Delay:             " << GetSlowKeysDelay() << endl;

  cout << endl;

  cout << "StickyKeys:            " << onoroff(GetStickyKeysState()) << endl;
  cout << "  - Latch/Lock:        " << onoroff(GetStickyKeysLatchToLockState())
       << endl;
  cout << "  - 2 Key Dis:         " <<
    onoroff(GetStickyKeysTwoKeyDisableState()) << endl;

  cout << endl;

  cout << "BounceKeys:            " << onoroff(GetBounceKeysState()) << endl;
  cout << "  - Delay:             " << GetBounceKeysDelay() << endl;

  cout << endl;

  cout << "RepeatKeys:            " << onoroff(GetRepeatKeysState()) << endl;
  cout << "  - Delay:             " << GetRepeatKeysDelay() << endl;
  cout << "  - Interval:          " << GetRepeatKeysInterval() << endl;

  cout << endl;
}

//
// QueryServer()
//

int Access::QueryServer()
{
  // compile time lib major version in, server major version out
  int major_in_out = XkbMajorVersion;

  // compile time lib minor version in, server minor version out
  int minor_in_out = XkbMinorVersion;

  // backfilled with the extension base event code
  int event_rtrn = 0;

  // backfilled with the extension base error code
  int error_rtrn = 0;

  // backfilled with a status code
  int reason_rtrn = 0;

  //connection to XServer
  Display *display;

  // Open a connection with an X server, check for a compatible version of the
  // XKB extension in both the library and the server, and initialize the
  // extension for use.

  display = XkbOpenDisplay(NULL, &event_rtrn, &error_rtrn, &major_in_out,
			   &minor_in_out, &reason_rtrn);

  if (reason_rtrn == XkbOD_BadLibraryVersion)
    {
      return 1; /*AccessBadLibraryVersion;*/
    }
  else if (reason_rtrn == XkbOD_ConnectionRefused)
    {
      return 2; /*AccessConnectionRefused;*/
    }
  else if (reason_rtrn == XkbOD_BadServerVersion)
    {
      return 3; /*AccessBadServerVersion;*/
    }
  else if (reason_rtrn == XkbOD_NonXkbServer)
    {
      return 4; /*AccessNonXkbServer;*/
    }
  else {

    //Get the state of the keyboard.

    XkbDescPtr xkb = XkbGetMap(display, 0, XkbUseCoreKbd);
    if ((int)xkb == BadAlloc || xkb == NULL)
      {
	return 5; /*AccessKeyboardQueryFailure;*/
      }
    else
      {
	//Get the current state of the keyboard controls.
	if (Status S =
	    XkbGetControls(display,XkbAllControlsMask,xkb) != Success)
	  {
	    XkbFreeKeyboard(xkb,XkbAllControlsMask,True);
	    XCloseDisplay(display);
	    return 5; /*AccessKeyboardQueryFailure;*/
	  }
	else
	  {
	    //If the controls are valid, copy their settings to local vars

	    if (xkb->ctrls)
	      {
		enabled_ctrls              = xkb->ctrls->enabled_ctrls;
	        accessx_delay              = xkb->ctrls->ax_timeout;

		mk_delay                   = xkb->ctrls->mk_delay;
		mk_interval                = xkb->ctrls->mk_interval;
		mk_time_to_max             = xkb->ctrls->mk_time_to_max;
		mk_max_speed               = xkb->ctrls->mk_max_speed;
		mk_curve                   = xkb->ctrls->mk_curve;

		slow_keys_delay            = xkb->ctrls->slow_keys_delay;

		enabled_sticky_key_options = xkb->ctrls->ax_options;

		debounce_delay             = xkb->ctrls->debounce_delay;

		repeat_delay               = xkb->ctrls->repeat_delay;
	        repeat_interval            = xkb->ctrls->repeat_interval;
    
		// Free the local copy of the keyboard state
		XkbFreeKeyboard(xkb,XkbAllControlsMask,True);

		// Close the display, flushing the XLib Request Buffer
		XCloseDisplay(display);

		return 0; /* Success */
	      }
	    else
	      {
		XkbFreeKeyboard(xkb,XkbAllControlsMask,True);
		XCloseDisplay(display);
		return 6; /*AccessFailure;*/
	      }
	  }
      }
  }
}

//
// FlushNewSettings()
//

int Access::FlushNewSettings()
{
  // compile time lib major version in, server major version out
  int major_in_out = XkbMajorVersion;

  // compile time lib minor version in, server minor version out
  int minor_in_out = XkbMinorVersion;

  // backfilled with the extension base event code
  int event_rtrn = 0;

  // backfilled with the extension base error code
  int error_rtrn = 0;

  // backfilled with a status code
  int reason_rtrn = 0;

  //connection to XServer
  Display *display;

  // Open a connection with an X server, check for a compatible version of the
  // XKB extension in both the library and the server, and initialize the
  // extension for use.

  display = XkbOpenDisplay(NULL, &event_rtrn, &error_rtrn, &major_in_out,
			   &minor_in_out, &reason_rtrn);

  if (reason_rtrn == XkbOD_BadLibraryVersion)
    {
      return 1; /*AccessBadLibraryVersion;*/
    }
  else if (reason_rtrn == XkbOD_ConnectionRefused)
    {
      return 2; /*AccessConnectionRefused;*/
    }
  else if (reason_rtrn == XkbOD_BadServerVersion)
    {
      return 3; /*AccessBadServerVersion;*/
    }
  else if (reason_rtrn == XkbOD_NonXkbServer)
    {
      return 4; /*AccessNonXkbServer;*/
    }
  else {

    //Get the state of the keyboard.

    XkbDescPtr xkb = XkbGetMap(display, 0, XkbUseCoreKbd);
    if ((int)xkb == BadAlloc || xkb == NULL)
      {
	return 5; /*AccessKeyboardQueryFailure;*/
      }
    else
      {

	//Get the current state of the keyboard controls.

	if (Status S =
	    XkbGetControls(display,XkbAllControlsMask,xkb) != Success)
	  {
	    XkbFreeKeyboard(xkb,XkbAllControlsMask,True);
	    XCloseDisplay(display);
	    return 5; /*AccessKeyboardQueryFailure;*/
	  }
	else
	  {
	    //If the controls are valid, apply the new settings

	    if (xkb->ctrls)
	      {
		xkb->ctrls->enabled_ctrls   = enabled_ctrls;

	        xkb->ctrls->ax_timeout      = accessx_delay;

		xkb->ctrls->mk_delay        = mk_delay;
		xkb->ctrls->mk_interval     = mk_interval;
		xkb->ctrls->mk_time_to_max  = mk_time_to_max;
		xkb->ctrls->mk_max_speed    = mk_max_speed;
		xkb->ctrls->mk_curve        = mk_curve;

		xkb->ctrls->slow_keys_delay = slow_keys_delay;
		/*XkbSetSlowKeysDelay(display, XkbUseCoreKbd,
		  slow_keys_delay); */

		xkb->ctrls->ax_options      = enabled_sticky_key_options;
		/*XkbSetStickyKeysOptions(display, XkbUseCoreKbd,
					XkbAX_TwoKeysMask |
					XkbAX_LatchToLockMask,
					enabled_sticky_key_options);*/

		xkb->ctrls->debounce_delay  = debounce_delay;
		/*XkbSetBounceKeysDelay(display, XkbUseCoreKbd,
		  debounce_delay);*/

		xkb->ctrls->repeat_delay    = repeat_delay;
	        xkb->ctrls->repeat_interval = repeat_interval;

		// Set the modified controls at the X server.
		XkbSetControls(display,
			       XkbAllControlsMask,
			       xkb);
    
		// Free the local copy of the keyboard state
		XkbFreeKeyboard(xkb,XkbAllControlsMask,True);

		// Close the display, flushing the XLib Request Buffer
		XCloseDisplay(display);

		return 0; /* Success */
	      }
	    else
	      {
		XkbFreeKeyboard(xkb,XkbAllControlsMask,True);
		XCloseDisplay(display);
		return 6; /*AccessFailure;*/
	      }
	  }
      }
  }
}

//
// GetAccessXTOState()
//

int Access::GetAccessXTOState()
{
  if ((enabled_ctrls & XkbAccessXTimeoutMask) == 0)
    return 0;
  else
    return 1;
}

//
// SetAccessXTOState()
//

void Access::SetAccessXTOState(int state)
{
  unsigned long enabled = 0;
    
  enabled |= (state ? XkbAccessXTimeoutMask : 0);
    
  enabled_ctrls &= ~XkbAccessXTimeoutMask;
  enabled_ctrls |= enabled;
}

//
// GetAccessXDelay()
//

unsigned short Access::GetAccessXDelay()
{
  return accessx_delay;
}

//
// SetAccessXDelay()
//

void Access::SetAccessXDelay(unsigned short delay)
{
  accessx_delay = delay;
}

//
// GetMouseKeysState()
//

int Access::GetMouseKeysState()
{
  if ((enabled_ctrls & XkbMouseKeysMask) == 0)
    return 0;
  else
    return 1;
}

//
// SetMouseKeysState()
//

void Access::SetMouseKeysState(int state)
{
  unsigned long enabled = 0;
    
  enabled |= (state ? XkbMouseKeysMask : 0);
    
  enabled_ctrls &= ~XkbMouseKeysMask;
  enabled_ctrls |= enabled;
}

//
// GetMouseKeysDelay()
//

unsigned short Access::GetMouseKeysDelay()
{
  return mk_delay;
}

//
// SetMouseKeysDelay()
//

void Access::SetMouseKeysDelay(unsigned short delay)
{
  mk_delay = delay;
}

//
// GetMouseKeysInterval()
//

unsigned short Access::GetMouseKeysInterval()
{
  return mk_interval;
}

//
// SetMouseKeysInterval()
//

void Access::SetMouseKeysInterval(unsigned short interval)
{
  mk_interval = interval;
}

//
// GetMouseKeysTimeToMax()
//

unsigned short Access::GetMouseKeysTimeToMax()
{
  return mk_time_to_max;
}

//
// SetMouseKeysTimeToMax()
//

void Access::SetMouseKeysTimeToMax(unsigned short timetomax)
{
  mk_time_to_max = timetomax;
}


//
// GetMouseKeysMaxSpeed()
//

unsigned short Access::GetMouseKeysMaxSpeed()
{
  return mk_max_speed;
}

//
// SetMouseKeysMaxSpeed()
//

void Access::SetMouseKeysMaxSpeed(unsigned short maxspeed)
{
  mk_max_speed = maxspeed;
}


//
// GetMouseKeysCurve()
//

short Access::GetMouseKeysCurve()
{
  return mk_curve;
}

//
// SetMouseKeysCurve()
//

void Access::SetMouseKeysCurve(short curve)
{
  mk_curve = curve;
}

//
// GetSlowKeysState()
//

int Access::GetSlowKeysState()
{
  if ((enabled_ctrls & XkbSlowKeysMask) == 0)
    return 0;
  else
    return 1;
}

//
// SetSlowKeysState()
//

void Access::SetSlowKeysState(int state)
{
  unsigned long enabled = 0;
    
  enabled |= (state ? XkbSlowKeysMask : 0);
    
  enabled_ctrls &= ~XkbSlowKeysMask;
  enabled_ctrls |= enabled;
}

//
// GetSlowKeysDelay()
//

unsigned short Access::GetSlowKeysDelay()
{
  return slow_keys_delay;
}

//
// SetSlowKeysDelay()
//

void Access::SetSlowKeysDelay(unsigned short delay)
{
  slow_keys_delay = delay;
}

//
// GetStickyKeysState()
//

int Access::GetStickyKeysState()
{
  if ((enabled_ctrls & XkbStickyKeysMask) == 0)
    return 0;
  else
    return 1;
}

//
// SetStickyKeysState()
//

void Access::SetStickyKeysState(int state)
{   
  unsigned long enabled = 0;
    
  enabled |= (state ? XkbStickyKeysMask : 0);
    
  enabled_ctrls &= ~XkbStickyKeysMask;
  enabled_ctrls |= enabled;
}

//
// GetStickyKeysLatchToLockState()
//

int Access::GetStickyKeysLatchToLockState()
{
  if ((enabled_sticky_key_options & XkbAX_LatchToLockMask) == 0)
    return 0;
  else
    return 1;
}

//
// SetStickyKeysLatchToLockState()
//

void Access::SetStickyKeysLatchToLockState(int state)
{
  unsigned long enabled = 0;
    
  enabled |= (state ? XkbAX_LatchToLockMask : 0);
    
  enabled_sticky_key_options &= ~XkbAX_LatchToLockMask;
  enabled_sticky_key_options |= enabled; 
}

//
// GetStickyKeysTwoKeyDisableState()
//

int Access::GetStickyKeysTwoKeyDisableState()
{
  if ((enabled_sticky_key_options & XkbAX_TwoKeysMask) == 0)
    return 0;
  else
    return 1;
}

//
// SetStickyKeysTwoKeyDisableState()
//

void Access::SetStickyKeysTwoKeyDisableState(int state)
{
  unsigned long enabled = 0;
    
  enabled |= (state ? XkbAX_TwoKeysMask : 0);
    
  enabled_sticky_key_options &= ~XkbAX_TwoKeysMask;
  enabled_sticky_key_options |= enabled; 
}

//
// GetBounceKeysState()
//

int Access::GetBounceKeysState()
{
  if ((enabled_ctrls & XkbBounceKeysMask) == 0)
    return 0;
  else
    return 1;
}

//
// SetBounceKeysState()
//

void Access::SetBounceKeysState(int state)
{
  unsigned long enabled = 0;
    
  enabled |= (state ? XkbBounceKeysMask : 0);
    
  enabled_ctrls &= ~XkbBounceKeysMask;
  enabled_ctrls |= enabled;
}

//
// GetBounceKeysDelay()
//

unsigned short Access::GetBounceKeysDelay()
{
  return debounce_delay;
}

//
// SetBounceKeysDelay()
//

void Access::SetBounceKeysDelay(unsigned short delay)
{
  debounce_delay = delay;
}

//
// GetRepeatKeysState()
//

int Access::GetRepeatKeysState()
{
  if ((enabled_ctrls & XkbRepeatKeysMask) == 0)
    return 0;
  else
    return 1;
}

//
// SetRepeatKeysState()
//

void Access::SetRepeatKeysState(int state)
{
  unsigned long enabled = 0;
    
  enabled |= (state ? XkbRepeatKeysMask : 0);
    
  enabled_ctrls &= ~XkbRepeatKeysMask;
  enabled_ctrls |= enabled;
}

//
// GetRepeatKeysDelay()
//

unsigned short Access::GetRepeatKeysDelay() 
{
  return repeat_delay;
}

//
// SetRepeatKeysDelay()
//

void Access::SetRepeatKeysDelay(unsigned short delay)
{
  repeat_delay = delay;
}

//
// GetRepeatKeysInterval()
//

unsigned short Access::GetRepeatKeysInterval()
{
  return repeat_interval;
}

//
// SetRepeatKeysInterval()
//

void Access::SetRepeatKeysInterval(unsigned short interval)
{
  repeat_interval = interval;
}

