<?php

/**
 * @license GPL-2.0-or-later
 * @file
 */

namespace MediaWiki\Tests\Site;

use Exception;
use MediaWiki\Site\MediaWikiSite;
use MediaWiki\Site\Site;
use MediaWiki\Site\SiteImporter;
use MediaWiki\Site\SiteList;
use MediaWiki\Site\SiteStore;
use MediaWikiIntegrationTestCase;
use Psr\Log\LoggerInterface;

/**
 * @ingroup Site
 * @ingroup Test
 *
 * @group Site
 *
 * @covers \MediaWiki\Site\SiteImporter
 *
 * @author Daniel Kinzler
 */
class SiteImporterTest extends MediaWikiIntegrationTestCase {

	private function newSiteImporter( array $expectedSites, $errorCount ) {
		$store = $this->createMock( SiteStore::class );

		$store->expects( $this->once() )
			->method( 'saveSites' )
			->willReturnCallback( function ( $sites ) use ( $expectedSites ) {
				$this->assertSitesEqual( $expectedSites, $sites );
			} );

		$store->method( 'getSites' )
			->willReturn( new SiteList() );

		$errorHandler = $this->createMock( LoggerInterface::class );
		$errorHandler->expects( $this->exactly( $errorCount ) )
			->method( 'error' );

		$importer = new SiteImporter( $store );
		$importer->setExceptionCallback( [ $errorHandler, 'error' ] );

		return $importer;
	}

	public function assertSitesEqual( $expected, $actual, $message = '' ) {
		$this->assertEquals(
			$this->getSerializedSiteList( $expected ),
			$this->getSerializedSiteList( $actual ),
			$message
		);
	}

	public static function provideImportFromXML() {
		$foo = Site::newForType( Site::TYPE_UNKNOWN );
		$foo->setGlobalId( 'Foo' );

		$acme = Site::newForType( Site::TYPE_UNKNOWN );
		$acme->setGlobalId( 'acme.com' );
		$acme->setGroup( 'Test' );
		$acme->addLocalId( Site::ID_INTERWIKI, 'acme' );
		$acme->setPath( Site::PATH_LINK, 'http://acme.com/' );

		$dewiki = Site::newForType( Site::TYPE_MEDIAWIKI );
		$dewiki->setGlobalId( 'dewiki' );
		$dewiki->setGroup( 'wikipedia' );
		$dewiki->setForward( true );
		$dewiki->addLocalId( Site::ID_INTERWIKI, 'wikipedia' );
		$dewiki->addLocalId( Site::ID_EQUIVALENT, 'de' );
		$dewiki->setPath( Site::PATH_LINK, 'http://de.wikipedia.org/w/' );
		$dewiki->setPath( MediaWikiSite::PATH_PAGE, 'http://de.wikipedia.org/wiki/' );
		$dewiki->setSource( 'meta.wikimedia.org' );

		return [
			'empty' => [
				'<sites></sites>',
				[],
			],
			'no sites' => [
				'<sites><Foo><globalid>Foo</globalid></Foo><Bar><quux>Bla</quux></Bar></sites>',
				[],
			],
			'minimal' => [
				'<sites>' .
					'<site><globalid>Foo</globalid></site>' .
				'</sites>',
				[ $foo ],
			],
			'full' => [
				'<sites>' .
					'<site><globalid>Foo</globalid></site>' .
					'<site>' .
						'<globalid>acme.com</globalid>' .
						'<localid type="interwiki">acme</localid>' .
						'<group>Test</group>' .
						'<path type="link">http://acme.com/</path>' .
					'</site>' .
					'<site type="mediawiki">' .
						'<source>meta.wikimedia.org</source>' .
						'<globalid>dewiki</globalid>' .
						'<localid type="interwiki">wikipedia</localid>' .
						'<localid type="equivalent">de</localid>' .
						'<group>wikipedia</group>' .
						'<forward/>' .
						'<path type="link">http://de.wikipedia.org/w/</path>' .
						'<path type="page_path">http://de.wikipedia.org/wiki/</path>' .
					'</site>' .
				'</sites>',
				[ $foo, $acme, $dewiki ],
			],
			'skip' => [
				'<sites>' .
					'<site><globalid>Foo</globalid></site>' .
					'<site><barf>Foo</barf></site>' .
					'<site>' .
						'<globalid>acme.com</globalid>' .
						'<localid type="interwiki">acme</localid>' .
						'<silly>boop!</silly>' .
						'<group>Test</group>' .
						'<path type="link">http://acme.com/</path>' .
					'</site>' .
				'</sites>',
				[ $foo, $acme ],
				1
			],
		];
	}

	/**
	 * @dataProvider provideImportFromXML
	 */
	public function testImportFromXML( $xml, array $expectedSites, $errorCount = 0 ) {
		$importer = $this->newSiteImporter( $expectedSites, $errorCount );
		$importer->importFromXML( $xml );
	}

	public function testImportFromXML_malformed() {
		$this->expectException( Exception::class );

		$store = $this->createMock( SiteStore::class );
		$importer = new SiteImporter( $store );
		$importer->importFromXML( 'THIS IS NOT XML' );
	}

	public function testImportFromFile() {
		$foo = Site::newForType( Site::TYPE_UNKNOWN );
		$foo->setGlobalId( 'Foo' );

		$acme = Site::newForType( Site::TYPE_UNKNOWN );
		$acme->setGlobalId( 'acme.com' );
		$acme->setGroup( 'Test' );
		$acme->addLocalId( Site::ID_INTERWIKI, 'acme' );
		$acme->setPath( Site::PATH_LINK, 'http://acme.com/' );

		$dewiki = Site::newForType( Site::TYPE_MEDIAWIKI );
		$dewiki->setGlobalId( 'dewiki' );
		$dewiki->setGroup( 'wikipedia' );
		$dewiki->setForward( true );
		$dewiki->addLocalId( Site::ID_INTERWIKI, 'wikipedia' );
		$dewiki->addLocalId( Site::ID_EQUIVALENT, 'de' );
		$dewiki->setPath( Site::PATH_LINK, 'http://de.wikipedia.org/w/' );
		$dewiki->setPath( MediaWikiSite::PATH_PAGE, 'http://de.wikipedia.org/wiki/' );
		$dewiki->setSource( 'meta.wikimedia.org' );

		$importer = $this->newSiteImporter( [ $foo, $acme, $dewiki ], 0 );

		$file = __DIR__ . '/SiteImporterTest.xml';
		$importer->importFromFile( $file );
	}

	/**
	 * @param Site[] $sites
	 *
	 * @return array[]
	 */
	private function getSerializedSiteList( $sites ) {
		$serialized = [];

		foreach ( $sites as $site ) {
			$key = $site->getGlobalId();
			$data = unserialize( serialize( $site ) );

			$serialized[$key] = $data;
		}

		return $serialized;
	}
}
