# $Id: PicStart.pm,v 1.6 2005/08/27 18:22:10 kissg Exp $

###############################################################################
# Frontend
#
# This is a pseudoclass. All methods (except new) are inherited from
# version specific subclasses and the backend.
###############################################################################
package Pista::Programmer::PicStart;

use strict;
use warnings;
use Carp;
our @ISA = qw(Pista::Programmer::PicStart::BackEnd);	# initial value
use Dumpvalue;
my $dumper = new Dumpvalue('quoteHighBit'=>1, 'arrayDepth'=>64);

sub new {
	my $pkg = shift;
	my $line = shift;

	my %args = @_;
	my $self = Pista::Programmer::PicStart::BackEnd->
								new($line, $args{debug}) or return;
	$self->{progress} = $args{progress}; # Flag: Progress report required.
	undef %args;

	# Push version dependent plugins
	my $ver = $self->{ver}[0]*1_000_000 +
		  $self->{ver}[1]*1_000 +
		  $self->{ver}[2];

	if ($ver > 0) {
		add_feature($pkg.'::Features::Over128kB');
		add_feature($pkg.'::Features::Verify');
		add_feature($pkg.'::Features::EE_Blankcheck');
		if ($ver < 4_000_000) {
			add_feature($pkg.'::Features::NoFlash');
		}
		if ($ver <= 999_999_999) {
			add_feature($pkg.'::Features::Protect');
			add_feature($pkg.'::Features::DevID');
		}

		$self->{infolen} = 44;	# This may vary in different firmwares.
		$self->{conflen} = $ver >= 3_000_000 ? 8 : 4;

		printf "220 PICSTART Plus detected, ".
			"firmware version %d.%d.%d\n",
			@{$self->{ver}};
	}
	else {
		print "221-PICSTART Plus detected but FLASH image is corrupted.\n",
			"221 You have to download a valid firmware\n";
	}

	bless $self, $pkg;
}

# Inserts a new layer below us.
sub add_feature {
	my $new = shift;
	no strict 'refs';
	# Lower layer inherits our @ISA
	@{"${new}::ISA"} = @ISA unless @{"${new}::ISA"};
	# Set our new @ISA
	@ISA = $new;
}

###############################################################################
# Plugins
#
# Version specific methods. They replaces the default methods of
# the backend class.
###############################################################################
package Pista::Programmer::PicStart::Features::Over128kB;

# PS+ protocol does not allow to read/write blocks of 64k words

use strict;
use warnings;
our @ISA = ();		# initial value

sub write_prog {
	my ($self, $device, $start, $end, $values) = @_;
	$self->check_firmware($device) or return;
	my $len = $end-$start+1;

	my $status;
	my $mtu = 0x1fff0 / $device->{prog}->{width};
	my @values = @$values;
	while ($len > 0) {
		my $chunksize = $len;
		$chunksize = $mtu if $chunksize > $mtu;
		my @sublist = splice(@values, 0, $chunksize);
		$status = $self->SUPER::write_prog($device,
										$start, $start+$chunksize-1,
										\@sublist);
		$status or return;
		$start += $chunksize;
		$len -= $chunksize;
	}
	return $status;
}

sub read_prog {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;
	my $len = $end-$start+1;

	my @retval;
	my $mtu = 0x1fffe / $device->{prog}->{width};
	while ($len > 0) {
		my $chunksize = $len;
		$chunksize = $mtu if $chunksize > $mtu;
		my $content = $self->SUPER::read_prog($device,
										$start, $start+$chunksize-1);
		$content or return;
		push(@retval, @$content);
		$start += $chunksize;
		$len -= $chunksize;
	}
	return \@retval;
}

###############################################################################
package Pista::Programmer::PicStart::Features::DevID;

# Some versions of PICSTART Plus do not support "Read Device ID" command.
# This workaround method replaces the default one.

use strict;
use warnings;
use Pista::Util qw(dup CCtov);
our @ISA = ();		# initial value

# Alternative way of retrieving Device ID in case of different architectures:
my @DevID = qw(
	none   UserID UserID  UserID
	UserID None   None    Program
	UserID UserID None    Program
	UserID UserID Program UserID
	UserID UserID
);

sub read_devid {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;

	if ($DevID[$device->{picstart}->{arch}] eq 'UserID') {
		# 0x01 0x02 0x03 0x04 0x08 0x09 0x0c 0x0d 0x0f 0x10 0x11
		$device = dup($device);
		$device->{userid}->{len} = $end - $device->{userid}->{addr} + 1;
		my $devidref = $self->read_userid($device,
									$device->{userid}->{addr}, $end);
		splice(@$devidref, 0, $start - $device->{userid}->{addr});
		return $devidref;
	}
	elsif ($DevID[$device->{picstart}->{arch}] eq 'Program') {
		# 0x07
		# 0x0b 0x0e
		return $self->read_prog($device, $start, $end);

	}
	# 0x05 0x06 0x0a
	return undef;
}

###############################################################################
package Pista::Programmer::PicStart::Features::Verify;

use strict;
use warnings;
use Pista::Util qw(dup);
our @ISA = ();		# initial value

# Newer devices are written in multiword blocks however original PS+ protocol
# expect single words to write then returns verified word.
# Therefore verifying is just simulated during write operation.
# The actual verification must be done separately.

sub write_prog {
	my ($self, $device, $start, $end, $values) = @_;
	$self->check_firmware($device) or return;
	my $status = $self->SUPER::write_prog($device, $start, $end, $values);
	return $status unless $status;							# write failed
	return $status if $device->{prog}->{blocksize} == 1;	# verified
	my $content = $self->read_prog($device, $start, $end);
	return $self->_verify($device, 'prog', $values, $content);
}

sub write_userid {
	my ($self, $device, $start, $end, $values) = @_;
	$self->check_firmware($device) or return;
	my $status = $self->SUPER::write_userid($device, $start, $end, $values);
	return $status unless $status;							# write failed
	return $status if $device->{userid}->{blocksize} == 1;	# verified
	my $content = $self->read_userid($device, $start, $end);
	return $self->_verify($device, 'userid', $values, $content);
}

###############################################################################
package Pista::Programmer::PicStart::Features::NoFlash;

# Original PICSTART Plus has a PIC17C44 inside.
# Newer ones are equipped with PIC18F6720 and their firmware can
# be upgraded on the fly.
sub write_firmware {
	print "526 Your hardware does not support firmware download\n";
	return;
}

###############################################################################
package Pista::Programmer::PicStart::Features::Protect;

use strict;
use warnings;
use Pista::Util qw(dup);
our @ISA = ();		# initial value

# Available erase methods on different architectures
my @Erasure = qw(
	none      None      Protected Universal	
	Universal None      None      None
	None      None      None      None
	Protected Universal Universal Universal
	Protected Protected
);

# Some versions of PICSTART Plus do not care during erase if device is
# code protected or not.
# We must do code protection before bulk erasing chip.

sub _protect {
	my ($self, $device) = @_;
	$self->check_firmware($device) or return;

	return 1 unless $Erasure[$device->{picstart}->{arch}] eq 'Protected';
	
# "We don't know where code protection bits are."
# "No problemo" the Terminator said. "We set all config bits to zero."
	
	print "129 Setting code protection\n";
	my @zeroconf;
	if (exists $device->{conf}->{preserve}) {
		@zeroconf = @{ $self->read_conf($device,
									$device->{conf}->{addr}, 
									$device->{conf}->{end}) };
		for my $i (0..$#{$device->{conf}->{blank}}) {
			$zeroconf[$i] &= $device->{conf}->{preserve}->[$i];
		}
	}
	else {
		for my $i (0..$#{$device->{conf}->{blank}}) {
			$zeroconf[$i] = 0;
		}
	}
	return $self->write_conf($device, $device->{conf}->{addr},
					  $device->{conf}->{end}, \@zeroconf);
}

###############################################################################
package Pista::Programmer::PicStart::Features::EE_Blankcheck;

use strict;
use warnings;
use Pista::Util qw(dup);
our @ISA = ();		# initial value

# Internal EEPROM blank check routine
my @EEblankcheck = qw(
	none  OK    Buggy Buggy	
	Buggy OK    OK    OK
	Buggy OK    OK    OK
	Buggy Buggy OK    Buggy
	Buggy Buggy OK    Buggy
	Buggy Buggy Buggy Buggy
);

# Some versions of PICSTART Plus cannot check if EEPROM is blank or not.
# We must read EEPROM content and check it manually.
sub blankcheck {
	my ($self, $device) = @_;
	my $retval = $self->SUPER::blankcheck($device);
	defined $retval or return;
	return $retval
		unless $EEblankcheck[$device->{picstart}->{arch}] eq 'Buggy';
	$retval->{eeprom} = 1;
	my $content = $self->SUPER::read_eeprom($device,
					$device->{eeprom}->{addr},
					$device->{eeprom}->{addr} + $device->{eeprom}->{len}-1);
	for (my $i=0; $i<=$#{$content}; $i++) {
		next if $device->{eeprom}->{blank}->[0] ==
			($content->[$i] & $device->{eeprom}->{mask}->[0]);
		return $retval;
	}
	$retval->{eeprom} = 0;
	return $retval;
}

###############################################################################
# Backend
#
# Default (fallback) methods here
###############################################################################
package Pista::Programmer::PicStart::BackEnd;

use strict;
use warnings;
use POSIX qw(:termios_h :fcntl_h);
use FileHandle;
use Pista::Util qw(dup CCtov vtoCC);

use constant SPEED			=> 19200;

use constant BLANK_CHECK_FULL			=> 0x42;
use constant DIAGNOSTICS				=> 0x43;
use constant BLANK_CHECK_RANGE			=> 0x44;
use constant WRITE_PROGRAM				=> 0x51;
use constant READ_PROGRAM				=> 0x54;
use constant READ_CAL					=> 0x63;
use constant READ_EEPROM				=> 0x64;
use constant READ_USERID				=> 0x65;
use constant READ_CONFIG				=> 0x66;
use constant WRITE_CONFIG				=> 0x67;
use constant WRITE_USERID				=> 0x68;
use constant WRITE_EEPROM				=> 0x69;
use constant WRITE_CAL					=> 0x71;
use constant GET_INFO_LEN				=> 0x80;
use constant LOAD_PROC_INFO				=> 0x81;
use constant LOAD_EXTENDED_CONFIG		=> 0x82;
use constant DOWNLOAD_FIRMWARE			=> 0x87;	# 4.0.0+
use constant FW_MAGIC					=> 0x55aa;	# 4.0.0+
use constant ASK_HARDWARE				=> 0x88;
use constant  PICSTART_PLUS				=> 0xab;
use constant GET_VERSION				=> 0x8d;
use constant SET_RANGE					=> 0x8e;
use constant ERASE						=> 0x8f;
use constant COMPUTE_PROGRAM_CHKSUM		=> 0x90;
use constant COMPUTE_CONF_CHKSUM		=> 0x91;
use constant COMPUTE_USERID_CHKSUM		=> 0x92;
use constant COMPUTE_EEPROM_CHKSUM		=> 0x93;
use constant COMPUTE_CAL_CHKSUM			=> 0x94;
use constant READ_DEVID					=> undef;	# unsupported by 4.10.6

use constant PROG_NOTBLANK				=> 0x01;
use constant CONF_NOTBLANK				=> 0x02;
use constant USERID_NOTBLANK			=> 0x04;
use constant EEPROM_NOTBLANK			=> 0x08;
use constant CAL_NOTBLANK				=> 0x10;
use constant PARITY_ERROR				=> 0x20;

BEGIN {
	eval "sub CRTSCTS { 020000000000; }" # POSIX.pm does not define this
		unless defined (&CRTSCTS);
	1;
}

sub new {
	my $pkg = shift;
	my $self = {};
	bless $self, $pkg;
	$self->{line} = shift;
	$self->{debug_serial} = shift;
	$self->{cached_info} = '';
	my ($reply);

	# Open serial line
	$self->{fh} = FileHandle->new();
	if (!$self->{fh}->open($self->{line}, O_RDWR)) {
		$self->{fh} = undef;
		print "520 Cannot open line $self->{line} $!\n";
		return;
	}

	# Setup serial line
	my $termios = POSIX::Termios->new();
	my $fd=$self->{fh}->fileno();
	$termios->getattr($fd);
	$termios->setcc(VMIN, 1);
	$termios->setcc(VTIME, 0);
	$termios->setiflag(0);
	$termios->setoflag(0);
	$termios->setlflag(0);
	$termios->setcflag(CS8 | HUPCL | CREAD | CLOCAL | CRTSCTS);
	$termios->setispeed(B0);
	$termios->setospeed(B0);
	$termios->setattr($fd, TCSANOW);		# DTR down: reset PICSTART Plus
	print "120 Resetting programmer\n";
	sleep 1;
	my $posixspeed = eval "B".SPEED;
	$termios->setispeed($posixspeed);
	$termios->setospeed($posixspeed);
	$termios->setattr($fd, TCSANOW);		# DTR up

	select((select($self->{fh}),$|=1)[0]);	# make handle unbuffered
	tcflush($fd, TCIFLUSH);					# discard any pending input

	# Identify programmer
	$self->_send_serial('C', ASK_HARDWARE);
	$reply = $self->_receive_serial('C',1);
	if (!defined $reply) {
		print "522 Programmer not present\n";
		return;
	}
	if ($reply != PICSTART_PLUS) {
		print "522 No PICSTART Plus detected\n";
		return;
	}

	$self->_send_serial('C', GET_VERSION);
	($reply, @{$self->{ver}}) = $self->_receive_serial('CC',1); 
	if (!defined $reply) {
		print "524 Timeout\n";
		return;
	}
	if ($reply != GET_VERSION) {
		print "524 Protocol error\n";
		return;
	}
	if ($self->{ver}[0] == 0xff) {
		# Corrupted firmware detected
		@{$self->{ver}} = (0,0,0);
		return $self;
	}
	(@{$self->{ver}}[1..2]) = $self->_receive_serial('C2',1); 
	if (!defined $self->{ver}[1]) {
		print "524 Timeout\n";
		return;
	}
	return $self;
}

# Add programmer specific parts to device structure
sub grupid {
	my $self = shift;
	my $device = shift;
	return $device if exists $device->{programmer} and
			exists $device->{programmer}->{picstart};

	for my $element (@{$device->{deferred}->{programmer}->{content}}) {
		next unless $element->{name} eq 'picstart_plus';
		$device->{picstart} = $element->{attrib};
# TODO:
# Check if actual programmer version supports this PIC.
# Display notes about required adapter (if any)
		last;
	}
#$dumper->dumpValue($device);
	return $device;
}

sub show {
	my $self = shift;
	my $version = join('.',@{$self->{ver}});
	my $debug = $self->{debug_serial} ? 'enabled' : 'disabled';
	my $progress = $self->{progress} ? 'enabled' : 'disabled';
	print <<EOT;
123 Selected programmer
Type:              PICSTART Plus
Version:           $version
Line:              $self->{line}
Progress report:   $progress
Debug:             $debug
223 Programmer info finished
EOT
}

sub read_conf {
	my ($self, $device, $start, $end) = @_;
	my $len = _words($start, $end, $device->{prog}->{width});
	$device = dup($device);
	$self->_download_processor_info($device) or return;
	my $content = $self->_read(READ_CONFIG, $len, 2);
	$content = vtoCC($content) if ($device->{conf}->{width} == 1);
	pop(@$content) if $#$content > $end-$start;
	return $content;
}

sub write_conf {
	my ($self, $device, $start, $end, $values) = @_;
	my $masks = [];
	$self->check_firmware($device) or return;
#$dumper->dumpValue($device);
	my $relstart = $start - $device->{conf}->{addr};
	for my $i (0..$#{$values}) {
		$masks->[$i] = $device->{conf}->{mask}->[$relstart+$i];
		next if defined $values->[$i];
		$values->[$i] = $device->{conf}->{blank}->[$relstart+$i];
	}
	if ($device->{conf}->{width} == 1) {
		$values = CCtov($values);
	}
	my $len = _words($start, $end, $device->{conf}->{width});
	$device = dup($device);
	my $sect = $device->{conf};
	$sect->{addr} = $start;
	$sect->{end} = $end;
	$sect->{len} = $end - $start + 1;
	$self->_download_processor_info($device) or return;
	return $self->_write(WRITE_CONFIG, $len, 2, $values, $masks);
}

sub blankcheck {
	my ($self, $device) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	$self->_download_processor_info($device) or return;
	my $status = $self->_read(BLANK_CHECK_FULL);
	return unless defined $status;
	return {
		prog	=> $status & PROG_NOTBLANK,
		conf	=> $status & CONF_NOTBLANK,
		userid	=> $status & USERID_NOTBLANK,
		eeprom	=> $status & EEPROM_NOTBLANK,
		cal		=> $status & CAL_NOTBLANK,
	};
}

sub read_prog {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;
	my $len = _words($start, $end, $device->{prog}->{width});
	$device = dup($device);
	$self->_download_processor_info($device) or return;
	$self->_set_range($start, $len) or return;
	my $content = $self->_read(READ_PROGRAM, $len, 2);
	$content = vtoCC($content) if ($device->{prog}->{width} == 1);
	pop(@$content) if $#$content > $end-$start;
	return $content;
}

sub write_prog {
	my ($self, $device, $start, $end, $values) = @_;
	$self->check_firmware($device) or return;
	$values = dup($values);
	for (@$values) {
		$_ = $device->{prog}->{blank}->[0] unless defined $_;
	}
	if ($device->{prog}->{width} == 1) {
# TODO: Write to any position and in any length
		if ($start%8) {
			print "531 Program block to write must start on 8 byte boundary\n";
			return;
		}
		if (($end+1)%8) {
			print "531 Program area can be written in 8 byte blocks only\n";
			return;
		}
		$values = CCtov($values);
	}
	my $len = _words($start, $end, $device->{prog}->{width});
	$device = dup($device);
	$self->_download_processor_info($device) or return;
	$self->_set_range($start, $len) or return;
	return $self->_write(WRITE_PROGRAM, $len, 2, $values);
}

# address   0000                            2000                  2003
#            |                              base    start  end     |
#            |                               |        |     |      |
#            +-------------------------------=========*******------+
# address  000000                          200000                 200007
#            |                              base    start  end     |
#            |                               |        |     |      |
#            +-------------------------------=========*******------+
sub read_userid {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	my $sect = $device->{userid};
	my $base = $sect->{addr};
	$sect->{end} = $end;
	$sect->{len} = $end - $base + 1;
	$self->_download_processor_info($device) or return;
	my $content = $self->_read(READ_USERID,
				_words($base, $end, $sect->{width}), 2 );
	$content = vtoCC($content) if ($device->{userid}->{width} == 1);
	splice(@$content, 0, $start-$base);
	return $content;
}

sub write_userid {
	my ($self, $device, $start, $end, $values) = @_;
	$self->check_firmware($device) or return;
	for (@$values) {
		$_ = $device->{prog}->{blank}->[0] unless defined $_;
	}
	$device = dup($device);
	my $sect = $device->{userid};
	$sect->{end} = $end;
	if ($device->{userid}->{width} == 1) {
# TODO: Write to any position and in any length
		if ($start%8) {
			print "531 Userid block to write must start on 8 byte boundary\n";
			return;
		}
		if (($end+1)%8) {
			print "531 Userid area can be written in 8 byte blocks only\n";
			return;
		}
		$values = CCtov($values);
		my $len = _words($start, $end, 1);
		$sect->{addr} = $start;
		$sect->{len} = $end - $start + 1;
		$self->_download_processor_info($device) or return;
		return $self->_write(WRITE_USERID, $len, 2, $values);
	}
	else {
		my $base = $sect->{addr};
		$sect->{len} = $end - $base + 1;
		$self->_download_processor_info($device) or return;
		if ($start != $base) {
			my $content = $self->_read(READ_USERID, $sect->{len},2);
			splice(@$content,$start-$base, $end-$start+1, @$values);
			$values = $content;
		}
		return $self->_write(WRITE_USERID, $sect->{len}, 2, $values);
	}
}

# logical   0000                           f00000   lstart lend f000xx
# physical                                   0000   pstart pend   00xx
#            |                               |        |     |      |
#            +-------------------------------=========*******------+
#
# logical   0000                            2100    lstart lend  21xx
# physical                                  0000    pstart pend  00xx
#            |                               |        |     |      |
#            +-------------------------------=========*******------+
# We want to read the **** area. However some chip algorithms (0x0b, 0x0e)
# cannot use other than 0000 start address. Therefore we always
# read from 0000 till pend (=== plus ****) then we drop the unwanted bytes.
sub read_eeprom {
	my ($self, $device, $lstart, $lend) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	my $sect = $device->{eeprom};

	# logical -> physical  address translation
	my $lbase = $sect->{addr};	# base of logical addressing
	my $pbase = 0;			# base of physical addressing
	my $offset = $lbase-$pbase;
	$sect->{addr} -= $offset;	# I.e. pbase
	my $len = $lend - $lbase + 1;	# bytes to read
	$sect->{len} = $len;
	$sect->{end} = $sect->{addr} + $len - 1;

	$self->_download_processor_info($device) or return;
	my $content = $self->_read(READ_EEPROM, $len, 1);
	splice(@$content, 0, $lstart-$lbase) if ($lstart > $lbase);
	return $content;
}

sub write_eeprom {
	my ($self, $device, $lstart, $lend, $values) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	my $sect = $device->{eeprom};

	# logical -> physical  address translation
	my $lbase = $sect->{addr};	# base of logical addressing
	my $pbase = 0;			# base of physical addressing
	my $offset = $lbase-$pbase;
	my $pstart = $lstart - $offset;
	$sect->{addr} -= $offset;	# I.e. pbase
	my $len = $lend - $lbase + 1;	# bytes to read
	$sect->{len} = $len;
	$sect->{end} = $sect->{addr} + $len - 1;

	$values = dup($values);
	for (@$values) {
		next if defined $_;
		$_ = $sect->{blank}->[0];
	}
	$self->_download_processor_info($device) or return;
	my $content = $self->_read(READ_EEPROM, $len, 1);
	splice(@$content, $pstart, $lend-$lstart+1, @$values);
	return $self->_write(WRITE_EEPROM, $len, 1, $content);
}

sub read_cal {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;
	my $len = _words($start, $end, $device->{cal}->{width});
	$device = dup($device);
	my $sect = $device->{cal};
	$sect->{addr} = $start;
	$sect->{end} = $end;
	$sect->{len} = $end - $start + 1;
	$self->_download_processor_info($device) or return;
	my $content = $self->_read(READ_CAL, $len, 2);
	$content = vtoCC($content) if ($device->{cal}->{width} == 1);
	pop(@$content) if $#$content > $end-$start;
	return $content;
}

#TODO:
#do not execute on 0x15 architecture
sub write_cal {
	my ($self, $device, $start, $end, $values) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	my $sect = $device->{cal};
	$sect->{addr} = $start;
	$sect->{end} = $end;
	$sect->{len} = $end - $start + 1;
	$self->_download_processor_info($device) or return;
	return $self->_write(WRITE_CAL, $sect->{len}, $sect->{width}, $values);
}

# This method is overridden by
# Pista::Programmer::PicStart::Features::DevID::read_devid
# because current versions of PICSTART Plus do not support Device ID.
sub read_devid {
	die "Internal program error: Pista::Programmer::PicStart::BackEnd::read_devid()\n";
}

sub chksum_conf {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	$self->_download_processor_info($device) or return;
	return $self->_read(COMPUTE_CONF_CHKSUM, 1, 2);
}

sub chksum_userid {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	$self->_download_processor_info($device) or return;
	return $self->_read(COMPUTE_USERID_CHKSUM, 1, 2);
}

sub chksum_prog {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	$self->_download_processor_info($device) or return;
	return $self->_read(COMPUTE_PROGRAM_CHKSUM, 1, 2);
}

sub chksum_cal {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	$self->_download_processor_info($device) or return;
	return $self->_read(COMPUTE_CAL_CHKSUM, 1, 2);
}

sub chksum_eeprom {
	my ($self, $device, $start, $end) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	$self->_download_processor_info($device) or return;
	return $self->_read(COMPUTE_EEPROM_CHKSUM, 1, 2);
}

# This method is overridden by
# Pista::Programmer::PicStart::Features::Protect::_protect
# because current versions of PICSTART Plus do not care CP bits when erase
sub _protect { 1; }

# Warning: This routine does not restore words/bits to be
# preserved (calibration, bandgap)
sub bulk_erase {
	my ($self, $device) = @_;
	$self->check_firmware($device) or return;
	$device = dup($device);
	$self->_protect($device) or return;	# set code protection
	$self->_download_processor_info($device) or return;
	return $self->_read(ERASE);
}

sub _read {
	my ($self, $rdcmd, $len, $width) = @_;
	{
		no warnings qw(uninitialized);
		if ($len < 0) {
			die "Internal program error: len == $len";
		}
	}
	my $status;

	$self->_send_serial('C', $rdcmd);
	$status = $self->_receive_serial('C', 1);
	if (!defined $status) {
		print "524 Timeout\n";
		return;
	}
	if ($status != $rdcmd) {
		print "524 Protocol error\n";
		return;
	}

	my @retval;
	if ($len) {
		my $template = sprintf "%s%d", $width==2 ? 'n' : 'C', $len;
		my $timeout = int($len*$width/2*(10/SPEED + 0.01)) + 5;
#print "template=$template timeout=$timeout\n";
		@retval = $self->_receive_serial($template,$timeout);
	}
	$status = $self->_receive_serial('C',3);	# 3 sec for blankcheck
	if (!defined $status) {
		print "524 Timeout\n";
		return;
	}
#print "status=$status\n";
	return $len ? \@retval : $status;
}

sub write_firmware {
	my ($self, $device, $start, $end, $values) = @_;
	if ($device->{dev} ne '18f6720') {
		print "527 Device 18f6720 required.\n";
		return;
	}
	if ($start != 0x1000) {
		print "528 Bad start address.\n";
		return;
	}
	my $len = $end-$start+1;
	if ($len % 8) {
		print "528 Bad length.\n";
		return;
	}
	$values = dup($values);
	for (@$values) {
		$_ = $device->{prog}->{blank}->[0] unless defined $_;
	}
	my ($reply, $sig, $status);

	$self->_send_serial('C', DOWNLOAD_FIRMWARE);
	($reply) = $self->_receive_serial('C', 1);
	if (!defined $reply) {
		print "524 Timeout\n";
		return;
	}
	if ($reply != DOWNLOAD_FIRMWARE) {
		print "524 Protocol error\n";
		return;
	}

	$self->_send_serial('n', FW_MAGIC);
	($reply) = $self->_receive_serial('C', 1);
	if (!defined $reply) {
		print "524 Timeout\n";
		return;
	}
	if ($reply != 0x00) {
		print "524 Protocol error\n";
		return;
	}

	$self->_send_serial('C3', $len&0xff, ($len>>8)&0xff, ($len>>16)&0xff);
	# Dirty trick: we expect 3 byte length and a 0x00 status.
	# That is just a 'long' in little endian order.
	$status = $self->_receive_serial('V', 1);
	if (!defined $status) {
		print "524 Timeout\n";
		return;
	}
	if ($status != $len) {
		print "524 Protocol error\n";
		return;
	}

	my $lastprogress = time;
	for (my $i=0; $i<$len; $i+=8) {
		my $chunk = $self->_send_serial('C8',splice(@$values,0,8));
		$self->_send_serial('n', unpack('%16C*', $chunk));
		$reply = $self->_receive_serial('C',1);
		if (!defined $reply) {
			print "524 Timeout\n";
			return;
		}
		if ($reply != 0x00) {
			print "530 Unsuccesful download\n";
			return;
		}
		next unless $self->{progress};
		next unless $len > 255;
		next unless time - $lastprogress >= 2;
		print "122 ", int(100*$i/$len), "% written\n";
		$lastprogress = time;
	}
}

sub _write {
	my ($self, $wrcmd, $len, $width, $values, $masks) = @_;
	if ($len < 0) {
		die "Internal program error: len == $len";
	}
	my $status;
	my $errors = 0;

	$self->_send_serial('C', $wrcmd);
	$status = $self->_receive_serial('C', 1);
	if (!defined $status) {
		print "524 Timeout\n";
		return;
	}
	if ($status != $wrcmd) {
		print "524 Protocol error\n";
		return;
	}

	my $template = $width==2 ? 'n' : 'C';
#print "template=$template\n";
	my $lastprogress = time;
	for my $i (0..$len-1) {
		$self->_send_serial($template,$values->[$i]);
		my $echo = $self->_receive_serial($template,1);
		if (!defined $echo) {
			print "524 Timeout\n";
			return;
		}
		if (defined $masks) {
			$errors++ if ($echo&$masks->[$i]) != ($values->[$i]&$masks->[$i]);
		}
		else {
			$errors++ if $echo != $values->[$i];
		}
		next unless $self->{progress};
		next unless $len > 255;
		next unless time - $lastprogress >= 2;
		print "122 ", int(100*$i/$len), "% written\n";
		$lastprogress = time;
	}
	$status = $self->_receive_serial('C',1);
#print "errors=$errors status=$status\n";
	if ($status) {
		print "532 Write error\n";
		return;
	}
	if ($errors) {
		print "532 Verification failed. $errors words differ\n";
		return;
	}
	return 1;
}

sub _set_range {
	my ($self, $start, $words) = @_;
	my $status;

	for (SET_RANGE, ($start>>16)&0xff, ($start>>8)&0xff, $start&0xff,
					   ($words>>8)&0xff, $words&0xff) {
		$self->_send_serial('C', $_);
		$status = $self->_receive_serial('C', 1);
		if (!defined $status) {
			print "524 Timeout\n";
			return;
		}
		if ($status != $_) {
			print "524 Protocol error\n";
			return;
		}
	}
	return 1;
}

sub _download_processor_info {
	my $self = shift;
	my $device = shift;
	my ($reply, $infolen, $info);

	if ($device->{addressing} eq 'byte') {
		# Byte -> word addressing mode conversion

		$device = dup($device);
		no warnings qw(uninitialized);
		foreach (qw(prog cal userid devid conf)) {
			next unless defined $device->{$_};
			my $sect = $device->{$_};
			$sect->{len}   /= 2;
			$sect->{end}   -= $sect->{len};
			$sect->{width}  = 2;
			if ($#{$sect->{mask}} > 0) {
				$sect->{mask}  = CCtov($sect->{mask});
				$sect->{blank} = CCtov($sect->{blank});
			}
			else {
				$sect->{mask}->[0]  |= $sect->{mask}->[0]<<8;
				$sect->{blank}->[0] |= $sect->{blank}->[0]<<8;
			}
		}
	}

	{ no warnings qw(uninitialized);
	  $info = pack('n14CnCn4C4',
		$device->{prog}->{len},
		$device->{prog}->{blank}->[0],	$device->{prog}->{mask}->[0],
		$device->{userid}->{blank}->[0],$device->{userid}->{mask}->[0],
		$device->{conf}->{blank}->[0],	$device->{conf}->{mask}->[0],
		$device->{eeprom}->{blank}->[0],$device->{eeprom}->{mask}->[0],
		$device->{cal}->{blank}->[0],	$device->{cal}->{mask}->[0],
		$device->{prog}->{addr},	$device->{prog}->{len},
		$device->{userid}->{addr},	$device->{userid}->{len},
		$device->{conf}->{addr},	$device->{conf}->{len},
		$device->{eeprom}->{addr},	$device->{eeprom}->{len},
		$device->{cal}->{addr},		$device->{cal}->{len},
		$device->{picstart}->{overprog},
		$device->{picstart}->{maxpulses},
		$device->{picstart}->{arch},
		($device->{features}->{rewritable} ? 0x01 : 0) |
		($device->{features}->{userid}     ? 0x02 : 0) |
		($device->{features}->{conf}       ? 0x04 : 0) |
		($device->{features}->{eeprom}     ? 0x08 : 0) |
		($device->{features}->{cal}        ? 0x10 : 0) |
		($device->{features}->{parity}     ? 0x20 : 0)
	);}
	return 1 if $self->{cached_info} eq $info;	# already downloaded

	$self->_send_serial('C', GET_INFO_LEN);
	($reply, $infolen) = $self->_receive_serial('CC',1); 
	if (!defined $reply) {
		print "524 Timeout\n";
		return;
	}
	if ($reply != GET_INFO_LEN) {
		print "524 Protocol error\n";
		return;
	}
	if ($infolen != $self->{infolen}) {
		print "524 Unexpected procinfo length $infolen\n";
		return;
	}

	$self->_send_serial('C', LOAD_PROC_INFO);
	$reply = $self->_receive_serial('C',1); 
	if (!defined $reply) {
		print "524 Timeout\n";
		return;
	}
	if ($reply != LOAD_PROC_INFO) {
		print "524 Protocol error\n";
		return;
	}

	$self->_send_serial($info);
	$self->_send_serial('C', unpack("%8C*",$info));	# checksum
	$reply = $self->_receive_serial('C',1); 
	if (!defined $reply) {
		print "524 Timeout\n";
		return;
	}
	if ($reply != 0x00) {
		print "530 Unsuccesful download\n";
		return;
	}
	$self->{cached_info} = $info;

# Theoretically the rest of this routine needed by 18xxx chips only
# but I've experienced that PS+ 3.11.0 may get totally confused
# if no extended config info downloaded.

	$self->_send_serial('C', LOAD_EXTENDED_CONFIG);
	$reply = $self->_receive_serial('C',1); 
	if (!defined $reply) {
		print "524 Timeout\n";
		return;
	}
	if ($reply != LOAD_EXTENDED_CONFIG) {
		print "524 Protocol error\n";
		return;
	}

	$info = $self->_send_serial('n*',
		@{$device->{conf}->{blank}}[0..$self->{conflen}-1],
		@{$device->{conf}->{mask}}[0..$self->{conflen}-1]
	);

	$self->_send_serial('C', unpack("%8C*",$info));	# checksum
	$reply = $self->_receive_serial('C',1); 
	if (!defined $reply) {
		print "524 Timeout\n";
		return;
	}
	if ($reply != 0x00) {
		print "530 Unsuccesful download\n";
		return;
	}

	return 1;
}

sub _verify {
	my $self = shift;
	my ($device, $sect, $buffer, $content) = @_;
	print "121 Verifying $sect section\n";
	my $errors = 0;
	for (my $i=0; $i<=$#$buffer; $i++) {
		my $value = $buffer->[$i];
		$value = $device->{$sect}->{blank}->[0] unless defined $value;
		$errors++ if $value != $content->[$i];
	}
	return 1 unless $errors;
	print "532 Verification failed. $errors ",
		$device->{$sect}->{width} == 1 ? 'bytes' : 'words', " differ\n";
	return;
}

sub _receive_serial {
	my $self = shift;
	my ($template, $timeout) = @_;
	my $len = length(pack($template, unpack('C*', 0x00 x 1024)));
	my $wanted = $len;
	my ($nread, @values);
	my $buffer = '';
	my $lastprogress = time;
	eval {
		local $SIG{ALRM} = sub { die "alarm\n" };
		alarm $timeout;
		for (; $wanted>0; $wanted-=$nread) {
			$nread = $self->{fh}->sysread($buffer, $wanted,
							length($buffer));
#print "nread=$nread of $wanted\n";
			if ($self->{debug_serial}) {
				(my $string = substr($buffer,-$nread,$nread)) =~
					s/(.)/sprintf(" %.2X",ord($1))/seg;
				print "<< $string\n";
			}
			next unless $self->{progress};
			next unless $len > 1023;
			next unless time - $lastprogress >= 1;
			print "122 ", int(100*($len-$wanted)/$len), "% read\n";
			$lastprogress = time;
		}
		alarm 0;
	};
#$dumper->dumpValue(\$buffer);
	if ($@) {
#print "input error\n";
		die unless $@ eq "alarm\n";
		# Wire break. Flush tty driver otherwise line cannot be closed.
		tcflush($self->{fh}->fileno, TCOFLUSH);
		return undef;
	}
	@values = unpack($template, $buffer);
#$dumper->dumpValue(['receive_serial: values',@values]);
#print (wantarray ? "array\n" : "scalar\n");
	return wantarray ? @values : shift @values;
}

sub _send_serial {
	my $self = shift;
	my $template = shift;
	my $buffer;
	if (@_) {
		no warnings qw(uninitialized);
		$buffer = pack($template, @_);
	}
	else {
		$buffer = $template;
	}
	if ($self->{debug_serial}) {
		(my $string = $buffer) =~ s/(.)/sprintf(" %.2X",ord($1))/seg;
		print ">> $string\n";
	}
#$dumper->dumpValue($buffer);
	$self->{fh}->syswrite($buffer) == length($buffer) or
		die "transmit error: $!\n";
	return $buffer;
}

sub _words {
	my ($start, $end, $width) = @_;
	my $len = $end - $start + 1;
	return $len if ($width == 2);
	$len += $len%2;
	return $len/2;
}

# Check minimal required firmware version for each architecture
sub check_firmware {
	my $self = shift;
	my $device = shift;
	my $available_firmware = $self->{ver}[0]<<16 |
				 $self->{ver}[1]<<8 |
				 $self->{ver}[2];
	my $required_firmware = (
		0,        0x020100, 0x020100, 0x020100,	# ---- 0x01 0x02 0x03
		0x020100, 0x020100, 0x020100, 0x020100,	# 0x04 0x05 0x06 0x07
		0x020a01, 0x020100, 0x020100, 0x020100,	# 0x08 0x09 0x0a 0x0b
		0x020a01, 0x020a01, 0x021e01, 0x030007,	# 0x0c 0x0d 0x0e 0x0f
		0x030006, 0x030007, 0x030007, 0x040a00,	# 0x10 0x11 0x12 0x13
		0x040a00, 0x040a00, 0x040a01, 0x040a01,	# 0x14 0x15 0x16 0x17
		0x040a01,
	)[$device->{picstart}->{arch}];
	if (!defined $required_firmware) {
		print "525-This chip is not supported by known versions";
		print "525 of PICSTART Plus firmware (up to 4.10.6).";
		return;
	}
	return 1 if $available_firmware >= $required_firmware;
	my $v = sprintf("%d.%d.%d", ($required_firmware>>16)&0xff,
				    ($required_firmware>>8)&0xff,
				    ($required_firmware)&0xff);
	print "525 This device requires firmware version $v or higher.";
	return 0;
}

1;

# vi: ts=4 sw=4
