-- noinspection SqlResolveForFile


---- QueryCurrentSessionInfo ----
select sign(charindex('Azure', @@version))                     as is_azure,
       db_id()                                                 as db_id,
       db_name()                                               as db_name,
       schema_id()                                             as schema_id,
       schema_name()                                           as schema_name
;


---- QueryFingerprint ----
select cast(database_id as varchar(20)) + '.' + cast(cast(create_date as bigint) as varchar(20)) + '/' + cast(service_broker_guid as varchar(60)) + '@' + cast(serverproperty('ServerName') as varchar(20)) as fingerprint
from sys.databases
where #AZURE: name = db_name() #: database_id = db_id() #.
;


---- ListDatabases ----
select database_id as id, name, cast(databasepropertyex(name,'collation') as char) as collation_name, state
from sys.databases
order by case when #AZURE: name = db_name() #: database_id = db_id() #. then 1 else 0 end desc, database_id
;


---- ListSchemas ----
select S.schema_id as id,
       S.name,
       cast((case
               when P.default_schema_name is not null then 1
               else 0
             end) as bit) as curr
from #CAT.schemas S
       left outer join #CAT.database_principals P
                       on S.name = P.default_schema_name
                         and #CLASSIC: P.sid = suser_sid(suser_sname()) #: P.name = user_name() #.
order by curr desc, schema_id
;


---- ListExistentObjects ----
select object_id
from #CAT.all_objects
where schema_id = :schema_id
;


---- ListExistentIndices ----
select cast(I.object_id as bigint) * 100000 + (I.index_id % 30000) as Combi_Id
from #CAT.all_objects T,
     #CAT.indexes I
where T.schema_id = :schema_id
  and T.type in ('U','S','IT','V','TT')
  and T.object_id = I.object_id
;


---- ListExistentFulltextIndices ----
select F.object_id
from #CAT.fulltext_indexes F,
     #CAT.all_objects T
where T.schema_id = :schema_id
  and T.object_id = F.object_id
;


---- RetrieveStates ----
select object_id, modify_date
from #CAT.all_objects
where object_id in (:[*object_ids])
;


---- CalculateCustomTypesCheckSums ----
with CT as ( select cast(1 + #ISTABLETYPE + is_assembly_type * 2 as tinyint) as type_kind,
                    abs(cast(checksum(user_type_id, name) as bigint)) as c_s
             from #CAT.types
             where schema_id = :schema_id
               and is_user_defined >= 1 )
select type_kind, sum(c_s) as check_sum
from CT
group by type_kind
order by type_kind
;


---- RetrieveAliasTypes ----
with A as ( select user_type_id as type_id, system_type_id, name,
                   max_length, precision, scale, is_nullable
            from #CAT.types
            where schema_id = :schema_id
              and is_user_defined >= 1
              and not (is_assembly_type >= 1)
              and not (#ISTABLETYPE >= 1) )
select A.type_id, A.name,
       B.name as base_name,
       A.max_length, A.precision, A.scale, A.is_nullable
from A left outer join #CAT.types B on A.system_type_id = B.user_type_id
order by A.type_id
;


---- RetrieveTableTypes ----
with T as ( select user_type_id as type_id,
                   type_table_object_id as object_id,
                   name
            from #CAT.table_types
            where schema_id = :schema_id
              and is_user_defined = 1
              and #ISTABLETYPE = 1 )
select T.type_id, O.object_id, T.name,
       O.create_date, O.modify_date
from T join #CAT.all_objects O on T.object_id = O.object_id
order by create_date, type_id
;


---- RetrieveTablesExtra ----
select T.object_id as object_id,
       P.start_column_id as start_column_id,
       P.end_column_id as end_column_id,
       T.temporal_type as temporal_type,
       T.history_table_id as history_table_id,
       #V16: T.ledger_type #: 0 #. as ledger_type,
       #V16: T.ledger_view_id #: 0 #. as ledger_view_id,
       #V16: T.is_dropped_ledger_table #: 0 #. as is_dropped_ledger_table,
       T.history_retention_period as history_retention_period,
       T.history_retention_period_unit as history_retention_period_unit,
       T.lob_data_space_id as lob_data_space_id,
       T.filestream_data_space_id as filestream_data_space_id
from #CAT.tables T left outer join #CAT.periods P on T.object_id = P.object_id
where T.schema_id = :schema_id
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and modify_date >= cast(:since as datetime)
order by create_date
;

---- RetrievePartitions ----
select P.partition_id as partition_id,
       P.object_id as object_id,
       P.index_id as index_id,
       P.partition_number as partition_number,
       P.data_compression as data_compression,
       #V16: P.xml_compression #: 0 #. as xml_compression
from #CAT.all_objects T inner join #CAT.partitions P on T.object_id = P.object_id
where T.schema_id = :schema_id
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and T.modify_date >= cast(:since as datetime)
order by T.create_date
;


---- RetrieveViewsExtra ----
select V.object_id as object_id,
       V.ledger_view_type as is_ledger_view,
       V.is_dropped_ledger_view as is_dropped_ledger_view
from #CAT.views V
where V.schema_id = :schema_id
                     #FRAG> and V.name in ( :[*f_names] )
  #INC> and modify_date >= cast(:since as datetime)
order by create_date
;


---- RetrieveObjects ----
select object_id,
       name,
       rtrim(type) as type,
       create_date,
       modify_date,
       is_ms_shipped
from #CAT.all_objects
where type in ('U','S','IT','V','FN','IF','P','X','PC','TF','TT','R','SO','SN')
  and schema_id = :schema_id
  #FRAG> and name in ( :[*f_names] )
  #INC> and modify_date >= cast(:since as datetime)
order by create_date
;


---- RetrieveColumns ----
with T as ( select object_id, create_date
            from #CAT.all_objects
            where schema_id = :schema_id
              #FRAG> and name in ( :[*f_names] )
              #INC> and modify_date >= cast(:since as datetime)
              and type in ('U','S','IT','V','TT','IF','FT','TF'))
select T.create_date,
       C.object_id, C.column_id as column_position, C.name,
       Y.name as type_name,
       C.max_length, C.precision, C.scale, C.collation_name,
       C.is_nullable, C.is_computed,
       cast(I.seed_value as numeric(38, 0)) as identity_seed_value,
       cast(I.increment_value as numeric(38, 0)) as identity_increment_value,
       cast(I.last_value as numeric(38, 0)) as identity_last_value,
       D.name as default_constraint_name,
       D.is_system_named as default_constraint_name_is_surrogate,
       D.object_id as default_constraint_object_id,
       D.modify_date default_constraint_modify_date,
       coalesce(D.definition, X.definition) as default_expression,
       C.is_filestream as is_filestream,
       C.is_rowguidcol as is_rowguidcol,
       #V13: C.is_hidden #: 0 #. as is_hidden,
       #V13: C.is_sparse #: 0 #. as is_sparse,
       #V13: C.is_column_set #: 0 #. as is_column_set,
       #V13: C.generated_always_type #: 0 #. as generated_always_type,
       #V13: masking_function #: null #. as masking_function,
       #V13: C.encryption_type #: null #. as encryption_type,
       #V13: C.encryption_algorithm_name #: null #. as encryption_algorithm_name,
       #V13: C.column_encryption_key_id #: null #. as column_encryption_key_id,
       #V13: C.column_encryption_key_database_name #: null #. as column_encryption_key_database_name,
       #V16: C.is_data_deletion_filter_column #: null #. as is_data_deletion_filter_column
from T join #CAT.all_columns C on T.object_id = C.object_id
       left outer join #CAT.types Y on C.system_type_id = Y.system_type_id and C.user_type_id = Y.user_type_id
       left outer join #CAT.identity_columns I on C.object_id = I.object_id and C.column_id = I.column_id
       left outer join #CAT.default_constraints D on C.object_id = D.parent_object_id and C.column_id = D.parent_column_id
       left outer join #CAT.computed_columns X on C.object_id = X.object_id and C.column_id = X.column_id
       #V13> left outer join #CAT.periods P on C.object_id = P.object_id
       #V13> left outer join #CAT.masked_columns MC on MC.object_id = C.object_id and MC.column_id = C.column_id
order by T.create_date, C.object_id, C.column_id
;


---- RetrieveTypeColumns ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              #FRAG> and name in ( :[*f_names] )
              and is_user_defined = 1 )
select C.object_id, C.column_id as column_position, C.name,
       Y.name as type_name,
       C.max_length, C.precision, C.scale, C.collation_name,
       C.is_nullable, C.is_computed,
       cast(I.seed_value as bigint) as identity_seed_value,
       cast(I.increment_value as bigint) as identity_increment_value,
       cast(I.last_value as bigint) as identity_last_value,
       coalesce(D.definition, X.definition) as default_expression
from T join #CAT.all_columns C on T.object_id = C.object_id
       left outer join #CAT.types Y on C.system_type_id = Y.system_type_id and C.user_type_id = Y.user_type_id
       left outer join #CAT.identity_columns I on C.object_id = I.object_id and C.column_id = I.column_id
       left outer join #CAT.default_constraints D on C.object_id = D.parent_object_id and C.column_id = D.parent_column_id
       left outer join #CAT.computed_columns X on C.object_id = X.object_id and C.column_id = X.column_id
order by C.object_id, C.column_id
;


---- RetrieveIndices ----
select T.object_id,
       I.index_id % 30000 as position,
       I.name,
       I.type,
       #V10: I.filter_definition, #.
       I.is_unique,
       I.is_disabled,
       I.is_padded,
       I.ignore_dup_key,
       I.allow_row_locks,
       I.allow_page_locks,
       I.fill_factor,
       I.data_space_id
from #CAT.all_objects T,
     #CAT.indexes I
where T.schema_id = :schema_id
  and T.type in ('U','S','IT','V','TT')
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and T.modify_date >= cast(:since as datetime)
  and T.object_id = I.object_id
order by T.create_date, I.object_id, I.index_id
;


---- RetrieveTypeIndices ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              #FRAG> and name in ( :[*f_names] )
              and is_user_defined = 1 )
select T.object_id,
       I.index_id % 30000 as position,
       I.name,
       I.type,
       #V10: I.filter_definition, #.
       I.is_unique,
       I.is_disabled
from T join #CAT.indexes I on T.object_id = I.object_id
where I.type != 0
order by I.object_id, I.index_id
;


---- RetrieveFulltextIndices ----
select F.object_id,
       F.unique_index_id
from #CAT.all_objects T,
     #CAT.fulltext_indexes F
where T.schema_id = :schema_id
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and T.modify_date >= cast(:since as datetime)
  and T.object_id = F.object_id
;


---- RetrieveIndexColumns ----
select T.object_id,
       X.index_id % 30000 as index_position,
       C.name,
       X.is_descending_key as is_desc,
       X.is_included_column,
       X.column_id,
       X.partition_ordinal
from #CAT.all_objects T,
     #CAT.indexes I,
     #CAT.index_columns X,
     #CAT.all_columns C
where T.schema_id = :schema_id
  and T.type in ('U','S','IT','V','TT')
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and T.modify_date >= cast(:since as datetime)
  and T.object_id = I.object_id
  and I.object_id = X.object_id
  and I.index_id = X.index_id
  and X.object_id = C.object_id
  and X.column_id = C.column_id
order by T.create_date, X.object_id, X.index_id, X.key_ordinal, X.index_column_id
;


---- RetrieveTypeIndexColumns ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              #FRAG> and name in ( :[*f_names] )
              and is_user_defined = 1 )
select T.object_id,
       X.index_id % 30000 as index_position,
       C.name,
       X.is_descending_key as is_desc,
       X.is_included_column
from T,
     #CAT.indexes I,
     #CAT.index_columns X,
     #CAT.all_columns C
where T.object_id = I.object_id
  and I.object_id = X.object_id
  and I.index_id = X.index_id
  and X.object_id = C.object_id
  and X.column_id = C.column_id
order by X.object_id, X.index_id, X.index_column_id
;


---- RetrieveFulltextIndexColumns ----
select T.object_id,
       C.name
from #CAT.all_objects T,
     #CAT.fulltext_index_columns X,
     #CAT.all_columns C
where T.schema_id = :schema_id
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and T.modify_date >= cast(:since as datetime)
  and T.object_id = X.object_id
  and X.object_id = C.object_id
  and X.column_id = C.column_id
;


---- RetrieveKeys ----
select T.object_id as table_id,
       K.object_id as key_id,
       K.unique_index_id % 30000 as index_position,
       K.name,
       case when K.type = 'PK' then 1 else 0 end as is_primary,
       K.is_system_named,
       K.create_date,
       K.modify_date
from #CAT.all_objects T
     join #CAT.key_constraints K on T.object_id = K.parent_object_id and T.schema_id = K.schema_id
where T.schema_id = :schema_id
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and T.modify_date >= cast(:since as datetime)
order by T.create_date, T.object_id, is_primary desc, K.create_date, K.object_id
;


---- RetrieveTypeKeys ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              #FRAG> and name in ( :[*f_names] )
              and is_user_defined = 1 )
select T.object_id as table_id,
       K.object_id as key_id,
       K.unique_index_id % 30000 as index_position,
       K.name,
       case when K.type = 'PK' then 1 else 0 end as is_primary,
       K.is_system_named,
       K.create_date,
       K.modify_date
from T join #CAT.key_constraints K on T.object_id = K.parent_object_id
order by T.object_id, is_primary desc, K.create_date, K.object_id
;


---- RetrieveForeignKeys ----
select T.object_id as table_id,
       F.object_id as constraint_id,
       F.name,
       F.is_system_named,
       F.referenced_object_id,
       X.name as referenced_object_name,
       X.schema_id as referenced_schema_id,
       F.key_index_id,
       F.is_disabled,
       F.update_referential_action,
       F.delete_referential_action,
       F.create_date,
       F.modify_date
from #CAT.all_objects T
     join #CAT.foreign_keys F on T.object_id = F.parent_object_id and T.schema_id = F.schema_id
     left join #CAT.objects X on F.referenced_object_id = X.object_id
where T.schema_id = :schema_id
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and T.modify_date >= cast(:since as datetime)
order by T.create_date, T.object_id
;


---- RetrieveForeignKeyColumns ----
select K.constraint_object_id as constraint_id,
       CD.name as domestic_column_name,
       CR.name as reference_column_name
from #CAT.all_objects T
     join #CAT.foreign_keys F on T.object_id = F.parent_object_id and T.schema_id = F.schema_id
     join #CAT.foreign_key_columns K on F.object_id = K.constraint_object_id
     left outer join #CAT.all_columns CD on K.parent_object_id = CD.object_id and K.parent_column_id = CD.column_id
     left outer join #CAT.all_columns CR on K.referenced_object_id = CR.object_id and K.referenced_column_id = CR.column_id
where T.schema_id = :schema_id
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and T.modify_date >= cast(:since as datetime)
  and CD.name is not null
  and CR.name is not null
order by T.create_date, T.object_id, K.constraint_object_id, K.constraint_column_id
;


---- RetrieveCheckConstraints ----
select T.object_id as table_id,
       X.object_id as constraint_id,
       X.name,
       X.is_system_named,
       X.is_disabled,
       C.name as column_name,
       X.definition,
       X.create_date,
       X.modify_date
from #CAT.all_objects T
     join #CAT.check_constraints X on T.object_id = X.parent_object_id and T.schema_id = X.schema_id
     left outer join #CAT.all_columns C on X.parent_object_id = C.object_id and X.parent_column_id = C.column_id
where T.schema_id = :schema_id
  #FRAG> and T.name in ( :[*f_names] )
  #INC> and T.modify_date >= cast(:since as datetime)
order by T.create_date, T.object_id, X.parent_column_id, X.create_date, X.object_id
;


---- RetrieveTypeCheckConstraints ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              #FRAG> and name in ( :[*f_names] )
              and is_user_defined = 1 )
select T.object_id as table_id,
       X.object_id as constraint_id,
       X.name,
       X.is_system_named,
       C.name as column_name,
       X.definition,
       X.create_date,
       X.modify_date
from T
     join #CAT.check_constraints X on T.object_id = X.parent_object_id
     left outer join #CAT.all_columns C on X.parent_object_id = C.object_id and X.parent_column_id = C.column_id
order by T.object_id, X.parent_column_id, X.create_date, X.object_id
;


---- RetrieveTriggers ----
select O.parent_object_id as table_id,
       O.object_id as trigger_id,
       O.name,
       O.create_date,
       O.modify_date,
       ( select sum(power(2,type-1))
         from #CAT.trigger_events
         where object_id = O.object_id
           and type between 1 and 3
         group by object_id ) as events,
       ( select sum(power(2,type-1))
         from #CAT.trigger_events
         where object_id = O.object_id
           and is_first = 1
           and type between 1 and 3
         group by object_id ) as first_events,
       ( select sum(power(2,type-1))
         from #CAT.trigger_events
         where object_id = O.object_id
           and is_last = 1
           and type between 1 and 3
         group by object_id ) as last_events,
       T.is_instead_of_trigger,
       T.is_not_for_replication,
       T.is_disabled,
       C.encrypted
from #CAT.objects O
     inner join #CAT.triggers T on O.object_id = T.object_id
     left outer join #CAT.syscomments C on O.object_id = C.id
where O.schema_id = :schema_id
  #FRAG> and parent_id in (select object_id from #CAT.objects where schema_id = :schema_id and type in ('U','V') and name in ( :[*f_names] ))
  #INC> and O.modify_date >= cast(:since as datetime)
  and O.type in ('TA','TR')
;


---- RetrieveSynonyms ----
select object_id,
       name,
       create_date,
       modify_date,
       parsename(base_object_name, 3) as origin_db_name,
       parsename(base_object_name, 2) as origin_schema_name,
       parsename(base_object_name, 1) as origin_object_name,
       object_id(base_object_name)    as origin_id
from #CAT.synonyms
where schema_id = :schema_id
  #FRAG> and name in ( :[*f_names] )
  #INC> and modify_date >= cast(:since as datetime)
;


---- RetrieveSequences ----
select Q.name,
       Q.object_id,
       Q.create_date,
       Q.modify_date,
       Q.start_value,
       #V14: Q.last_used_value #: current_value #. as last_value,
       Q.increment,
       Q.minimum_value,
       Q.maximum_value,
       Q.is_cycling,
       Q.is_cached,
       Q.cache_size,
       TS.name as system_type_name,
       T.name as type_name,
       S.name as type_schema_name,
       Q.precision
from #CAT.sequences Q
     join #CAT.types TS on TS.system_type_id = Q.system_type_id
     join #CAT.types T on T.user_type_id = Q.user_type_id
     join #CAT.schemas S on S.schema_id = T.schema_id
where Q.schema_id = :schema_id
  #FRAG> and Q.name in ( :[*f_names] )
  and TS.is_user_defined = 0
;


---- RetrieveRules ----
select object_id,
       name,
       create_date,
       modify_date
from #CAT.objects
where schema_id = :schema_id
  and type = 'R'
  #FRAG> and name in ( :[*f_names] )
;


---- CalculateSchemaPermissionsChecksum ----
select sum(abs(cast(checksum(X.major_id, X.minor_id, X.grantee_principal_id, X.type, X.state) as bigint))) as checksum
from #CAT.database_permissions X
     left join #CAT.all_objects O
        on O.object_id = X.major_id
       and O.type in ('U','S','IT','V','FN','IF','P','X','PC','TF','TT','C','F','D','PK','SN','SO','UQ','TR', 'R')
       and X.class <> 6
       and X.class <> 3
            and X.class <> 6
            and X.class <> 3
     left join #CAT.types T
        on T.user_type_id = X.major_id
       and X.class = 6
where coalesce(O.schema_id, T.schema_id, case when X.class = 3 then X.major_id else null end) = :schema_id
  and X.class not in (0, 3, 4)
;


---- CalculateDbPermissionsChecksum ----
select sum(abs(cast(checksum(X.major_id, X.minor_id, X.grantee_principal_id, X.type, X.state) as bigint))) as checksum
from #CAT.database_permissions X
where X.class in (0, 3, 4)
;


---- CalculateServerPermissionsChecksum ----
select sum(abs(cast(checksum(X.major_id, X.minor_id, X.grantee_principal_id, X.type, X.state) as bigint))) as checksum
from #CAT.server_permissions X
;


---- RetrieveSchemaPermissions ----
select X.major_id object_id,
       X.class kind,
       case when X.class = 7 then -1 else 1 end*(X.minor_id % 30000) as position,
       X.grantee_principal_id,
       X.permission_name,
       X.state
from #CAT.database_permissions X
       left join #CAT.all_objects O
                 on O.object_id = X.major_id
                   and O.type in ('U','S','IT','V','FN','IF','P','X','PC','TF','TT','C','F','D','PK','SN','SO','UQ','TR', 'R')
                   and X.class <> 6
                   and X.class <> 3
       left join #CAT.types T
                 on T.user_type_id = X.major_id
                   and X.class = 6
where (coalesce(O.schema_id, T.schema_id, case when X.class = 3 then X.major_id else null end) = :schema_id)
  and X.class not in (0, 3, 4)
order by kind, object_id, grantee_principal_id, permission_name, state
;


---- RetrieveDbPermissions ----
select X.major_id object_id,
       X.class kind,
       -1 as position,
       X.grantee_principal_id,
       X.permission_name,
       X.state
from #CAT.database_permissions X
where X.class in (0, 3, 4)
;


---- RetrieveServerPermissions ----
select X.major_id object_id,
       X.class kind,
       -1 as position,
       X.grantee_principal_id,
       X.permission_name,
       X.state
from #CAT.server_permissions X
;


---- CalculateExtendedPropertiesChecksum ----
with CS as (select abs(cast(checksum(X.major_id, X.minor_id, cast(X.value as nvarchar(max))) as bigint)) + 1 as checksum
            from #CAT.all_objects O,
                 #CAT.extended_properties X
            where O.schema_id = :schema_id
              and O.object_id = X.major_id
              and O.type in ('U', 'S', 'IT', 'V', 'FN', 'IF', 'P', 'X', 'PC', 'TF', 'TT', 'C', 'F', 'D', 'PK', 'SN', 'SO', 'UQ', 'TR', 'R')
              and X.class not in (3, 6)
              and X.value is not null
            union all
            select abs(cast(checksum(Y.major_id, Y.minor_id, cast(Y.value as nvarchar(max))) as bigint))*3 + 2 as checksum
            from #CAT.types T,
                 #CAT.extended_properties Y
            where T.schema_id = :schema_id
              and T.user_type_id = Y.major_id
              and Y.class = 6
              and Y.value is not null
            union all
            select abs(cast(checksum(Z.major_id, 0, cast(Z.value as nvarchar(max))) as bigint))*19 + 3 as checksum
            from #CAT.extended_properties Z
            where Z.class = 3
              and Z.major_id = :schema_id)
select sum(checksum) as checksum
from CS
;


---- RetrieveExtendedProperties ----
select X.class as object_class, O.object_id,
       X.minor_id % 30000 as position,
       X.name as property_name,
       cast(X.value as nvarchar(max)) as value,
       SQL_VARIANT_PROPERTY(X.value, 'BaseType') as type
from #CAT.all_objects O,
     #CAT.extended_properties X
where O.schema_id = :schema_id
  #FRAG> and O.name in ( :[*f_names] )
  and O.object_id = X.major_id
  and O.type in ('U','S','IT','V','FN','IF','P','X','PC','TF','TT','C','F','D','PK','SN','SO','UQ','TR','R')
  and X.class not in (3, 6)
  and X.value is not null
--
union all
--
select Y.class as object_class, T.user_type_id as object_id,
       Y.minor_id % 30000 as position,
       Y.name as property_name,
       cast(Y.value as nvarchar(max)) as value,
       SQL_VARIANT_PROPERTY(Y.value, 'BaseType') as type
from #CAT.types T,
     #CAT.extended_properties Y
where T.schema_id = :schema_id
  #FRAG> and T.name in ( :[*f_names] )
  and T.user_type_id = Y.major_id
  and Y.class = 6
  and Y.value is not null
--
union all
--
select Z.class as object_class, Z.major_id as object_id,
       0 as position,
       Z.name as property_name,
       cast(Z.value as nvarchar(max)) as value,
       SQL_VARIANT_PROPERTY(Z.value, 'BaseType') as type
from #CAT.extended_properties Z
where Z.class = 3
  and Z.major_id = :schema_id
  #FRAG> and :schema_name in ( :[*f_names] )
;


---- RetrieveDbExtendedProperties ----
select X.major_id object_id,
       X.minor_id as position,
       X.name as property_name, 
       cast(X.value as nvarchar(max)) as value, SQL_VARIANT_PROPERTY(X.value, 'BaseType') as type
from #CAT.extended_properties X
where X.class = 0
  and X.value is not null
;


---- RetrieveSources ----
select O.parent_object_id as parent_id,
       M.object_id,
       0 as is_trigger,
       M.definition as text
from #CAT.sql_modules M
     inner join #CAT.objects O on M.object_id = O.object_id
where O.schema_id = :schema_id
  and not (O.type = 'TR' or O.type = 'TA')
  #FRAG> and O.name in ( :[*f_names] )
  #INC> and O.modify_date >= cast(:src_since as datetime)
union all
select O.parent_object_id as parent_id,
       M.object_id,
       1 as is_trigger,
       M.definition as text
from #CAT.sql_modules M
       inner join #CAT.objects O on M.object_id = O.object_id
       inner join #CAT.objects P on P.object_id = O.parent_object_id
where O.schema_id = :schema_id
  and (O.type = 'TR' or O.type = 'TA')
  #FRAG> and P.name in ( :[*f_names] )
  #INC> and O.modify_date >= cast(:src_since as datetime)
;


---- RetrieveSourcesWithSys ----
select O.parent_object_id as parent_id,
       M.object_id,
       0 as is_trigger,
       M.definition as text
from #CAT.all_sql_modules M
     inner join #CAT.all_objects O on M.object_id = O.object_id
where O.schema_id = :schema_id
  and not(O.type = 'TR' or O.type = 'TA')
  #FRAG> and O.name in ( :[*f_names] )
  #INC> and O.modify_date >= cast(:src_since as datetime)
union all
select O.parent_object_id as parent_id,
       M.object_id,
       1 as is_trigger,
       M.definition as text
from #CAT.all_sql_modules M
       inner join #CAT.all_objects O on M.object_id = O.object_id
       inner join #CAT.all_objects P on P.object_id = O.parent_object_id
where O.schema_id = :schema_id
  and (O.type = 'TR' or O.type = 'TA')
  #FRAG> and P.name in ( :[*f_names] )
  #INC> and O.modify_date >= cast(:src_since as datetime)
;


---- RetrieveArguments ----
select P.object_id,
       P.parameter_id % 30000 as position,
       P.name,
       T.name as type_name,
       P.max_length,
       P.precision,
       P.scale,
       P.is_output,
       P.is_cursor_ref,
       P.has_default_value,
       #V13: P.is_nullable #: cast(0 as bit) #. as is_nullable
from #CAT.all_objects R
       join #CAT.all_parameters P on R.object_id = P.object_id
       left outer join #CAT.types T on P.system_type_id = T.system_type_id and P.user_type_id = T.user_type_id
where R.schema_id = :schema_id
  and R.type in ('FN','IF','TF','P','X','PC')
  #FRAG> and R.name in ( :[*f_names] )
  #INC> and R.modify_date >= cast(:since as datetime)
order by R.create_date, P.object_id, P.parameter_id
;


---- RetrieveSecurityPolicies ----
select object_id,
       name,
       create_date,
       modify_date,
       is_enabled,
       is_not_for_replication,
       is_schema_bound
from #CAT.security_policies
where schema_id = :schema_id
;


---- RetrieveSecurityPredicates ----
select pr.object_id,
       pr.security_predicate_id,
       pr.target_object_id,
       pr.predicate_definition,
       pr.predicate_type,
       pr.operation
from #CAT.security_predicates pr
    inner join #CAT.security_policies po on pr.object_id = po.object_id
where po.schema_id = :schema_id
order by pr.object_id, pr.target_object_id;


---- RetrieveLinkedServers ----
exec sp_linkedservers;


---- RetrieveCollations ----
select name, description from fn_helpcollations();


---- RetrieveLogins ----
select principal_id, name, type, is_disabled, default_database_name, #V11: #: null #. owning_principal_id
from sys.server_principals
;


---- RetrieveRoles ----
select principal_id, name, type, default_schema_name, owning_principal_id,  #V11: #: null #. authentication_type
from #CAT.database_principals
;


---- RetrieveColumnMasterKeys ----
select K.name,
       K.column_master_key_id,
       K.key_store_provider_name,
       K.key_path,
       #V15: K.signature #: null #. as signature
from #CAT.column_master_keys K
where 1=1
  #FRAG> and K.name in ( :[*f_names] )
order by K.column_master_key_id
;


---- RetrieveColumnEncryptionKeys ----
select K.name,
       K.column_encryption_key_id
from #CAT.column_encryption_keys K
where 1=1
  #FRAG> and K.name in ( :[*f_names] )
order by K.column_encryption_key_id
;


---- RetrieveColumnEncryptionKeyValues ----
select V.column_encryption_key_id,
       V.column_master_key_id,
       V.encrypted_value,
       V.encryption_algorithm_name
from #CAT.column_encryption_key_values V
where 1=1
  #FRAG> and V.name in ( :[*f_names] )
order by V.column_encryption_key_id, V.column_master_key_id
;


---- RetrieveAssemblies ----
select A.name,
       A.assembly_id,
       A.is_visible
from #CAT.assemblies A
where 1=1
  #FRAG> and V.name in ( :[*f_names] )
order by A.assembly_id
;


---- RetrievePartitionFunctions ----
select PF.function_id as function_id,
       PF.name as name,
       PF.create_date as create_date,
       PF.modify_date as modify_date,
       PF.boundary_value_on_right as boundary_value_on_right,
       T.name as type_name,
       PP.max_length as max_length,
       PP.precision as precision,
       PP.scale as scale,
       PP.collation_name as collation_name
from #CAT.partition_functions PF
     inner join #CAT.partition_parameters PP on PP.function_id = PF.function_id
     left outer join #CAT.types T on PP.system_type_id = T.system_type_id and PP.user_type_id = T.user_type_id
where 1 = 1
  #INC> and modify_date >= cast(:since as datetime)
;

---- RetrievePartitionFunctionRanges ----
with ranges as (select function_id,
                       cast(sql_variant_property(value, 'BaseType') as varchar) as type,
                       value
                from #CAT.partition_range_values)
select function_id,
       case
         when type like '%binary' then value
         when type like '%date%' or
              type like '%time%' or
              type like '%varchar' then '''' + cast(value as varchar) + ''''
         when type like '%char' then '''' + rtrim(cast(value as varchar)) + ''''
         when type like 'uniqueidentifier' then cast(value as char(36))
         else cast(value as varchar)
       end as boundary_value
from ranges
;

---- RetrieveFileGroups ----
select name,
       data_space_id,
       type,
       is_default,
       #V11: is_system #: 0 #. as is_system,
       is_read_only,
       #V13: is_autogrow_all_files #: 0 #. as is_autogrow_all_files
from #CAT.filegroups
where not type = 'PS'

---- RetrievePartitionSchemes ----
select name,
       data_space_id,
       function_id
from #CAT.partition_schemes

---- RetrievePartitionSchemeMappings ----
select partition_scheme_id,
       data_space_id
from #CAT.destination_data_spaces
order by partition_scheme_id, destination_id