/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * GtkCTree widget for GTK+
 * Copyright (C) 1998 Lars Hamann and Stefan Jeske
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include "gtkctree.h"
#include "opened.xpm"
#include "closed.xpm"
#include "mini-doc1.xpm"
#include "link.xpm"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <limits.h>
#include <malloc.h>
#include <string.h>
#include <grp.h>
#include <pwd.h>

GdkPixmap *pixmap1;
GdkPixmap *pixmap2;
GdkPixmap *pixmap3;
GdkPixmap *pixmap4;
GdkBitmap *mask1;
GdkBitmap *mask2;
GdkBitmap *mask3;
GdkBitmap *mask4;
static gint dirs = 0;
static gint files = 0;
static GtkWidget *dir_label;
static GtkWidget *file_label;
static GtkWidget *sel_label;
static GtkWidget *vis_label;
static GtkWidget *popup;
static GtkWidget *omenu;
static GtkWidget *omenu2;
static GtkWidget *omenu3;
static char fullpath[1024];
static GtkTooltips *tooltips;

GdkColor col_fg;
GdkColor col_bg;


#define FTW_F   1
#define FTW_D   2
#define FTW_DNR 3
#define FTW_NS  4

#define RADIOMENUTOGGLED(_rmi_, __i) { \
  GSList * __g; \
  __i = 0; \
  __g = gtk_radio_menu_item_group(_rmi_); \
  while( __g  && !((GtkCheckMenuItem *)(__g->data))->active) { \
    __g = __g->next; \
    __i++; \
  }\
}

#define RADIOBUTTONTOGGLED(_rb_, __i) { \
  GSList * __g; \
  __i = 0; \
  __g = gtk_radio_button_group(_rb_); \
  while( __g  && !((GtkToggleButton *)(__g->data))->active) { \
    __g = __g->next; \
    __i++; \
  }\
}

GList * insert_item (char *pathname, const struct stat *statptr, int type,
		     GtkCTree *ctree, GList *parent);


void
dopath (GtkCTree *ctree, GList *parent)
{
  struct stat    statbuf;
  struct dirent *dirp;
  DIR  *dp;
  char *ptr;

  if (lstat(fullpath, &statbuf) < 0)
    {
      insert_item (fullpath, &statbuf, FTW_NS, ctree, parent);
      return;
    }

  if (S_ISDIR (statbuf.st_mode) == 0)
    {
      insert_item (fullpath, &statbuf, FTW_F, ctree, parent);
      return;
    }
  
  parent  = insert_item (fullpath, &statbuf, FTW_D, ctree, parent);

  ptr = fullpath + strlen (fullpath);
  *ptr++ = '/';
  *ptr = 0;
  
  if ((dp = opendir (fullpath)) == NULL)
    {
      gtk_ctree_set_foreground(ctree, parent, &col_fg);
      gtk_ctree_set_background(ctree, parent, &col_bg);
      gtk_ctree_set_text (ctree, parent, 2, "not readable");
      return;
    }

  
  while ((dirp = readdir(dp)) != NULL)
    {
      if (strcmp (dirp->d_name, ".") == 0 ||
	  strcmp (dirp->d_name, "..") == 0)
	continue;
      strcpy (ptr, dirp->d_name);
      dopath (ctree, parent);
    }
  
  ptr[-1] = 0;
  if (closedir (dp) < 0)
    {
      g_print ("can't close dir %s", fullpath);
      return;
    }
  
}

char * my_basename (char *s) 
{
  char *r;
  gint len;

  len = strlen (s);
  if (s[len-1] == '/')
    {
      if (len == 1)
	return s;
      else
	s[len-1] = 0;
    }
  r = strrchr(s, '/');
  return r ? r + 1 : s;
}

GList * insert_item (char *pathname, const struct stat *statptr, int type,
		     GtkCTree *ctree, GList *parent)
{
  GList *child = NULL;
  gint level = 0;
  gchar bufsize[256];
  gchar bufowner[256];
  gchar bufname[256];
  gchar * text [3];
  struct passwd *pwd;
  struct group  *gr;
  gint num;

  sprintf (bufsize, "%d", (int)statptr->st_size);
  text[2] = bufsize;

  pwd = getpwuid (statptr->st_uid);
  if (pwd)
    num = sprintf (bufowner, "%s:", pwd->pw_name);
  else
    num = sprintf (bufowner, "%d:", statptr->st_uid);

  gr = getgrgid (statptr->st_gid);
  if (gr)
    sprintf (&bufowner[num], "%s", gr->gr_name);
  else
    sprintf (&bufowner[num], "%d", statptr->st_gid);
  text[0] = bufowner;

  switch (type)
    {
    case FTW_F:
      switch (statptr->st_mode & S_IFMT)
	{
	case S_IFLNK:
	  num = sprintf (bufname, "%s -> ",my_basename(pathname));
	  num += readlink (pathname, &bufname[num], sizeof(bufname)-num);
	  bufname[num] = 0;
	  text[1] = bufname;
	  child = gtk_ctree_insert  (ctree, parent,  NULL, text, 5, pixmap4,
				     mask4, NULL, NULL, TRUE, FALSE);
	  GTK_CTREE_ROW (child)->row.data = 0;
	  files++;
	  break;
	case S_IFBLK:
	case S_IFCHR:
	case S_IFIFO:
	case S_IFSOCK:
	case S_IFREG:
	  sprintf (bufname, "%s", my_basename(pathname));
	  text[1] = bufname;
	  child = gtk_ctree_insert (ctree, parent,  NULL, text,5,
				    pixmap3, mask3, NULL, NULL, TRUE, FALSE);
	  
	  GTK_CTREE_ROW (child)->row.data = 0;
	  files++;
	  break;
	case S_IFDIR:
	  g_print ("for S_IFDIR for %s", pathname);
	  exit (1);
	}
      break;
    case FTW_D:
      if (parent)
	level = GTK_CTREE_ROW (parent)->level;
      sprintf (bufname, "%s", my_basename(pathname));
      text[1] = bufname;
      child = gtk_ctree_insert  (ctree, parent,  NULL, text, 5,
				 pixmap1, mask1, pixmap2, mask2, FALSE, 
				 (level == 0));
      GTK_CTREE_ROW (child)->row.data = (gpointer) 1;
      dirs++;
      break;
    case FTW_NS:
      return NULL;
    default:
      g_print ("unknown type %d for pathname %s", type, pathname);
      exit (1);
    }
  return child;
}


gint
my_list_find (GList *list, gpointer data)
{
  int count = 0;

  while (list)
    {
      if (list->data == data)
        break;
      count++;
      list = list->next;
    }
  return count;
}

void after_press (GtkCTree *ctree, gpointer data)
{
  char buf[80];

  sprintf (buf, "%d", g_list_length (GTK_CLIST (ctree)->selection));
  gtk_label_set (GTK_LABEL (sel_label), buf);

  sprintf (buf, "%d", g_list_length (GTK_CLIST (ctree)->row_list));
  gtk_label_set (GTK_LABEL (vis_label), buf);

  sprintf (buf, "%d", dirs);
  gtk_label_set (GTK_LABEL (dir_label), buf);

  sprintf (buf, "%d", files);
  gtk_label_set (GTK_LABEL (file_label), buf);
}

void before_move (GtkCTree *ctree, GList *child, GList *parent, 
		  GList *sibling, gpointer data)
{
  g_print ("Moving \"%s\" to \"%s\" with sibling \"%s\".\n", 
	   GTK_CELL_PIXTEXT (((GtkCListRow*) 
			      (GTK_CTREE_ROW (child)))->cell[1])->text,
    (parent) ?
    GTK_CELL_PIXTEXT (((GtkCListRow*) 
		       (GTK_CTREE_ROW (parent)))->cell[1])->text : "nil",
    (sibling) ?
    GTK_CELL_PIXTEXT (((GtkCListRow*) 
		       (GTK_CTREE_ROW (sibling)))->cell[1])->text : "nil");
}

gint button_press (GtkCTree *ctree, GdkEventButton *event, gpointer data)
{
  gint row;
  gint column;
  gint num;
  GList *work;
  gint res;
  
  res = gtk_clist_get_selection_info (GTK_CLIST (ctree), event->x, event->y, 
				      &row, &column);
  if (!res && event->button != 3)
    return FALSE;

  work = g_list_nth (GTK_CLIST (ctree)->row_list, row);

  switch (event->button)
    {
    case 1:
      if (GTK_CLIST (ctree)->selection_mode == GTK_SELECTION_MULTIPLE &&
	  event->state & GDK_SHIFT_MASK)
	{
	  gtk_clist_freeze (GTK_CLIST (ctree));
	  if (GTK_CTREE_ROW (work)->row.state == GTK_STATE_SELECTED) 
	    gtk_ctree_unselect_recursive (ctree, work);
	  else
	    gtk_ctree_select_recursive (ctree, work);
	  gtk_clist_thaw (GTK_CLIST (ctree));
	  after_press (ctree, NULL);
	  gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree), 
					"button_press_event");
	  return FALSE;
	}
      break;
    case  2:
      num = my_list_find (GTK_CLIST (ctree)->row_list, GTK_CTREE_ROW (work));
      if (GTK_CTREE_ROW (work)->children && 
	  gtk_ctree_is_hot_spot (ctree, GTK_CTREE_ROW (work), num, event->x,
				 event->y))
	{
	  gtk_clist_freeze (GTK_CLIST (ctree));
	  if (GTK_CTREE_ROW (work)->expanded)
	    gtk_ctree_collapse_recursive (ctree, work);
	  else
	    gtk_ctree_expand_recursive (ctree, work);
	  gtk_clist_thaw (GTK_CLIST (ctree));
	  after_press (ctree, NULL);
	  gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree), 
					"button_press_event");
	  return FALSE;
	}
      break;
    case 3:
      gtk_menu_popup (GTK_MENU(popup), NULL, NULL, NULL, NULL, 3, event->time);
      gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree), "button_press_event");
      return TRUE;
    default:
      break;
    }
  return FALSE;
}

void count_items (GtkCTree *ctree, GList *list)
{
  if (GTK_CTREE_ROW (list)->row.data)
    dirs--;
  else
    files--;
}

void expand_all (GtkWidget *widget, GtkCTree *ctree)
{
  GList *work;

  gtk_clist_freeze (GTK_CLIST (ctree));
  work = GTK_CLIST (ctree)->row_list;
  while (work)
    {
      gtk_ctree_expand_recursive (ctree, work);
      work = GTK_CTREE_ROW (work)->sibling;
    }
  gtk_clist_thaw (GTK_CLIST (ctree));
  after_press (ctree, NULL);
}

void collapse_all (GtkWidget *widget, GtkCTree *ctree)
{
  GList *work;

  gtk_clist_freeze (GTK_CLIST (ctree));
  work = GTK_CLIST (ctree)->row_list;
  while (work)
    {
      gtk_ctree_collapse_recursive (ctree, work);
      work = GTK_CTREE_ROW (work)->sibling;
    }
  gtk_clist_thaw (GTK_CLIST (ctree));
  after_press (ctree, NULL);
}

void select_all (GtkWidget *widget, GtkCTree *ctree)
{
  GList *work;

  gtk_clist_freeze (GTK_CLIST (ctree));
  work = GTK_CLIST (ctree)->row_list;
  while (work)
    {
      gtk_ctree_select_recursive (ctree, work);
      work = GTK_CTREE_ROW (work)->sibling;
    }
  gtk_clist_thaw (GTK_CLIST (ctree));
  after_press (ctree, NULL);
}

void unselect_all (GtkWidget *widget, GtkCTree *ctree)
{
  GList *work;

  gtk_clist_freeze (GTK_CLIST (ctree));
  work = GTK_CLIST (ctree)->row_list;
  while (work)
    {
      gtk_ctree_unselect_recursive (ctree, work);
      work = GTK_CTREE_ROW (work)->sibling;
    }
  gtk_clist_thaw (GTK_CLIST (ctree));
  after_press (ctree, NULL);
}

void remove_selection (GtkWidget *widget, GtkCTree *ctree)
{
  GList *work;
  GList *selection;
  GList *tmp;
  GList *new_sel;

  selection = GTK_CLIST (ctree)->selection;
  new_sel = NULL;

  gtk_clist_freeze (GTK_CLIST (ctree));

  while (selection)
    {
      work = selection->data;
      if (GTK_CTREE_ROW (work)->row.data)
	gtk_ctree_post_recursive (ctree, work, 
				  (GtkCTreeFunc) count_items, NULL);
      else
	files--;

      if (GTK_CLIST (ctree)->selection_mode == GTK_SELECTION_BROWSE)
	{
	  tmp = gtk_ctree_find_row (ctree, GTK_CLIST (ctree)->row_list, 
				    GTK_CTREE_ROW (work));
	  if (GTK_CTREE_ROW (tmp)->children)
	    {
	      new_sel = GTK_CTREE_ROW (tmp)->sibling;
	      if (!new_sel)
		new_sel = tmp->prev;
	    }
	  else
	    {
	      if (tmp->next)
		new_sel = tmp->next;
	      else
		new_sel = tmp->prev;
	    }
	}

      gtk_ctree_remove (ctree, work);
      selection = GTK_CLIST (ctree)->selection;
    }

  if (new_sel)
    gtk_ctree_select (ctree, new_sel);

  gtk_clist_thaw (GTK_CLIST (ctree));
  after_press (ctree, NULL);
}

void detacher (GtkWidget *widget, GtkMenu *menu)
{
  popup = NULL;
}

void window_destroy (GtkWidget *widget, GtkWidget *menu)
{
  gtk_widget_destroy (menu);
}

void window_close (GtkWidget *widget, GtkWidget *window)
{
  gtk_widget_destroy (window);
  gtk_main_quit ();
}

void change_indent (GtkWidget *widget, GtkCTree *ctree)
{
  gtk_ctree_set_indent (ctree, GTK_ADJUSTMENT(widget)->value);
}

void toggle_reorderable (GtkWidget *widget, GtkCTree *ctree)
{
  gtk_ctree_set_reorderable (ctree, GTK_TOGGLE_BUTTON (widget)->active);
}

void toggle_line_style (GtkWidget *widget, GtkCTree *ctree)
{
  gint i;

  if (!GTK_WIDGET_MAPPED (widget))
    return;

  RADIOMENUTOGGLED ((GtkRadioMenuItem *)
		    (((GtkOptionMenu *)omenu2)->menu_item),i);

  gtk_ctree_set_line_style (ctree, (GtkCTreeLineStyle) (2-i));
}

void toggle_justify (GtkWidget *widget, GtkCTree *ctree)
{
  gint i;

  if (!GTK_WIDGET_MAPPED (widget))
    return;

  RADIOMENUTOGGLED ((GtkRadioMenuItem *)
		    (((GtkOptionMenu *)omenu3)->menu_item),i);

  gtk_clist_set_column_justification (GTK_CLIST (ctree), 1, 
				      (GtkJustification) (1-i));
}

void toggle_sel_mode (GtkWidget *widget, GtkCTree *ctree)
{
  gint i;

  if (!GTK_WIDGET_MAPPED (widget))
    return;

  RADIOMENUTOGGLED ((GtkRadioMenuItem *)
		    (((GtkOptionMenu *)omenu)->menu_item), i);

  gtk_ctree_set_selection_mode (ctree, (GtkSelectionMode) (3-i));
  after_press (ctree, NULL);
}

void rebuild_tree (GtkWidget *widget, GtkCTree *ctree)
{
  GList *work;

  if (!strlen(GTK_ENTRY (widget)->text))
    return;
  gtk_clist_freeze (GTK_CLIST (ctree));
  work = GTK_CLIST (ctree)->row_list;
  while (work)
    {
      gtk_ctree_remove (ctree, work);
      work = GTK_CTREE_ROW (work)->sibling;
    }
  dirs = files = 0;
  strcpy (fullpath, GTK_ENTRY (widget)->text);
  dopath (ctree, NULL);
  gtk_clist_thaw (GTK_CLIST (ctree));
  after_press (ctree, NULL);
}

void create_ctree ()
{
  static GtkWidget *window = NULL;
  GtkCTree *ctree;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *hbox2;
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *entry;
  GtkWidget *menu_item;
  GtkWidget *menu;
  GtkWidget *submenu;
  GtkWidget *check;
  GtkAdjustment *adj;
  GtkWidget *spinner;
  GSList *group;
  GdkColor transparent;

  char *title[] = { "Owner" , "Files" , "Size" };
  char buf[80];

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  tooltips = gtk_tooltips_new ();

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);

  hbox = gtk_hbox_new (FALSE, 5);
  gtk_container_border_width (GTK_CONTAINER (hbox), 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);
  
  label = gtk_label_new ("Path :");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

  entry = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 5);
  gtk_entry_set_text (GTK_ENTRY (entry), fullpath);
  
  button = gtk_button_new_with_label ("Close");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (window_close), window);
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, TRUE, 0);

  ctree = GTK_CTREE (gtk_ctree_new_with_titles (3, 1, title));
  gtk_ctree_set_reorderable (ctree, TRUE);
  gtk_signal_connect (GTK_OBJECT (ctree), "button_press_event",
		      GTK_SIGNAL_FUNC (button_press), NULL);
  gtk_signal_connect_after (GTK_OBJECT (ctree), "button_press_event",
			    GTK_SIGNAL_FUNC (after_press), NULL);
  gtk_signal_connect (GTK_OBJECT (ctree), "tree_move",
		      GTK_SIGNAL_FUNC (before_move), NULL);
  gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET(ctree), TRUE, TRUE, 0);
  gtk_clist_column_titles_passive (GTK_CLIST (ctree));
  gtk_clist_set_column_justification (GTK_CLIST (ctree), 2, GTK_JUSTIFY_RIGHT);
  gtk_clist_set_selection_mode (GTK_CLIST (ctree), GTK_SELECTION_EXTENDED);
  gtk_clist_set_policy (GTK_CLIST (ctree), 
			GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  gtk_clist_set_column_width (GTK_CLIST (ctree), 0, 70);
  gtk_clist_set_column_width (GTK_CLIST (ctree), 1, 290);
  gtk_clist_set_column_width (GTK_CLIST (ctree), 2, 80);

  /*gtk_clist_set_row_height (GTK_CLIST (ctree), 30);*/

  col_fg.red   = 65535;
  col_fg.green = 65535;
  col_fg.blue  = 0;
  gdk_color_alloc (gtk_widget_get_colormap (GTK_WIDGET(ctree)), &col_fg);
  col_bg.red   = 65535;
  col_bg.green = 0;
  col_bg.blue  = 0;
  gdk_color_alloc (gtk_widget_get_colormap (GTK_WIDGET(ctree)), &col_bg);

  hbox = gtk_hbox_new (TRUE, 5);
  gtk_container_border_width (GTK_CONTAINER (hbox), 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);

  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), hbox2, FALSE, TRUE, 0);

  label = gtk_label_new ("Indent :");
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, TRUE, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new (20, 0, 60, 1, 10, 0);
  spinner = gtk_spin_button_new (adj, 0, 0);
  gtk_tooltips_set_tip (tooltips, spinner, "Tree indentation.", NULL);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (change_indent), ctree);
  gtk_box_pack_start (GTK_BOX (hbox2), spinner, FALSE, TRUE, 5);

  check = gtk_check_button_new_with_label ("Reorderable");
  gtk_tooltips_set_tip (tooltips, check,
			"Tree items can be reordered by dragging.", NULL);
  gtk_signal_connect (GTK_OBJECT (check), "clicked",
		      GTK_SIGNAL_FUNC (toggle_reorderable), ctree);
  gtk_box_pack_start (GTK_BOX (hbox), check, FALSE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (check), TRUE);

  omenu2 = gtk_option_menu_new ();
  gtk_tooltips_set_tip (tooltips, omenu2, "The tree's line style.", NULL);

  menu = gtk_menu_new ();
  submenu = NULL;
  group = NULL;
      
  menu_item = gtk_radio_menu_item_new_with_label (group, "Solid");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                      GTK_SIGNAL_FUNC (toggle_line_style), ctree);
  group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (menu_item), TRUE);
  gtk_widget_show (menu_item);

  menu_item = gtk_radio_menu_item_new_with_label (group, "Dotted");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                      GTK_SIGNAL_FUNC (toggle_line_style), ctree);
  group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_radio_menu_item_new_with_label (group, "No lines");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                      GTK_SIGNAL_FUNC (toggle_line_style), ctree);
  group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (omenu2), menu);
  gtk_box_pack_start (GTK_BOX (hbox), omenu2, FALSE, TRUE, 0);

  gtk_option_menu_set_history (GTK_OPTION_MENU (omenu2), 0);

  omenu3 = gtk_option_menu_new ();
  gtk_tooltips_set_tip (tooltips, omenu3, "The tree's justification.", NULL);

  menu = gtk_menu_new ();
  submenu = NULL;
  group = NULL;
      
  menu_item = gtk_radio_menu_item_new_with_label (group, "Left");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                      GTK_SIGNAL_FUNC (toggle_justify), ctree);
  group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (menu_item), TRUE);
  gtk_widget_show (menu_item);

  menu_item = gtk_radio_menu_item_new_with_label (group, "Right");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                      GTK_SIGNAL_FUNC (toggle_justify), ctree);
  group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (omenu3), menu);
  gtk_box_pack_start (GTK_BOX (hbox), omenu3, FALSE, TRUE, 0);

  gtk_option_menu_set_history (GTK_OPTION_MENU (omenu3), 0);

  omenu = gtk_option_menu_new ();
  gtk_tooltips_set_tip (tooltips, omenu, "The list's selection mode.", NULL);

  menu = gtk_menu_new ();
  submenu = NULL;
  group = NULL;
      
  menu_item = gtk_radio_menu_item_new_with_label (group, "Single");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                      GTK_SIGNAL_FUNC (toggle_sel_mode), ctree);
  group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_radio_menu_item_new_with_label (group, "Browse");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                      GTK_SIGNAL_FUNC (toggle_sel_mode), ctree);
  group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_radio_menu_item_new_with_label (group, "Multiple");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                      GTK_SIGNAL_FUNC (toggle_sel_mode), ctree);
  group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_radio_menu_item_new_with_label (group, "Extended");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                      GTK_SIGNAL_FUNC (toggle_sel_mode), ctree);
  group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (menu_item), TRUE);
  gtk_widget_show (menu_item);
  
  gtk_option_menu_set_menu (GTK_OPTION_MENU (omenu), menu);
  gtk_box_pack_start (GTK_BOX (hbox), omenu, FALSE, TRUE, 0);

  gtk_option_menu_set_history (GTK_OPTION_MENU (omenu), 3);

  popup = gtk_menu_new ();
  gtk_menu_attach_to_widget (GTK_MENU (popup), GTK_WIDGET (window), detacher);

  menu_item = gtk_menu_item_new_with_label ("Remove selection");
  gtk_menu_append (GTK_MENU (popup), menu_item);
  gtk_widget_show (menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item),"activate",
		      GTK_SIGNAL_FUNC (remove_selection), ctree);
  menu_item = gtk_menu_item_new ();
  gtk_menu_append (GTK_MENU (popup), menu_item);
  gtk_widget_show (menu_item);
  menu_item = gtk_menu_item_new_with_label ("Expand all");
  gtk_menu_append (GTK_MENU (popup), menu_item);
  gtk_widget_show (menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item),"activate",
		      GTK_SIGNAL_FUNC (expand_all), ctree);
  menu_item = gtk_menu_item_new_with_label ("Collapse all");
  gtk_menu_append (GTK_MENU (popup), menu_item);
  gtk_widget_show (menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item),"activate",
		      GTK_SIGNAL_FUNC (collapse_all), ctree);
  menu_item = gtk_menu_item_new ();
  gtk_menu_append (GTK_MENU (popup), menu_item);
  gtk_widget_show (menu_item);
  menu_item = gtk_menu_item_new_with_label ("Select all");
  gtk_menu_append (GTK_MENU (popup), menu_item);
  gtk_widget_show (menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item),"activate",
		      GTK_SIGNAL_FUNC (select_all), ctree);
  menu_item = gtk_menu_item_new_with_label ("Unselect all");
  gtk_menu_append (GTK_MENU (popup), menu_item);
  gtk_widget_show (menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item),"activate",
		      GTK_SIGNAL_FUNC (unselect_all), ctree);

  gtk_widget_realize (window);

  pixmap1 = gdk_pixmap_create_from_xpm_d (window->window, &mask1, 
                                         &transparent, closed);
  pixmap2 = gdk_pixmap_create_from_xpm_d (window->window, &mask2, 
                                         &transparent, opened);
  pixmap3 = gdk_pixmap_create_from_xpm_d (window->window, &mask3, 
                                         &transparent, mini_doc1);
  pixmap4 = gdk_pixmap_create_from_xpm_d (window->window, &mask4, 
                                         &transparent, link_xpm);

  gtk_clist_freeze (GTK_CLIST (ctree));
  dopath (ctree, NULL);
  gtk_clist_thaw (GTK_CLIST (ctree));

  gtk_widget_set_usize (GTK_WIDGET(ctree), 500, 400);

  frame = gtk_frame_new (NULL);
  gtk_container_border_width (GTK_CONTAINER (frame), 0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, TRUE, 0);

  hbox = gtk_hbox_new (TRUE, 2);
  gtk_container_border_width (GTK_CONTAINER (hbox), 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, TRUE, 0);

  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox2);

  label = gtk_label_new ("Directories :");
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, TRUE, 0);

  sprintf (buf, "%d", dirs);
  dir_label = gtk_label_new (buf);
  gtk_box_pack_end (GTK_BOX (hbox2), dir_label, FALSE, TRUE, 5);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, TRUE, 0);

  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox2);

  label = gtk_label_new ("Files :");
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, TRUE, 0);
  
  sprintf (buf, "%d", files);
  file_label = gtk_label_new (buf);
  gtk_box_pack_end (GTK_BOX (hbox2), file_label, FALSE, TRUE, 5);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, TRUE, 0);

  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox2);

  label = gtk_label_new ("Selected :");
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, TRUE, 0);

  sprintf (buf, "%d", g_list_length (GTK_CLIST (ctree)->selection));
  sel_label = gtk_label_new (buf);
  gtk_box_pack_end (GTK_BOX (hbox2), sel_label, FALSE, TRUE, 5);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, TRUE, 0);

  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox2);

  label = gtk_label_new ("Visible :");
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, TRUE, 0);

  sprintf (buf, "%d", g_list_length (GTK_CLIST (ctree)->row_list));
  vis_label = gtk_label_new (buf);
  gtk_box_pack_end (GTK_BOX (hbox2), vis_label, FALSE, TRUE, 5);

  gtk_signal_connect (GTK_OBJECT (entry),"activate",
		      GTK_SIGNAL_FUNC (rebuild_tree), ctree);
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
                      GTK_SIGNAL_FUNC (window_destroy), popup);
  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
                      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

  gtk_widget_show_all (window);
}

int main (int argc, char *argv[])
{
  gtk_init (&argc, &argv);
  if (argc > 2 || (argc == 2 && !strcmp(argv[1], "--help")))
    {
      g_print ("usage : %s [directory]\n", my_basename(argv[0]));
      exit (1);
    }
  if (argc == 2)
    strcpy (fullpath, argv[1]);
  else
    strcpy (fullpath, ".");
    
  gtk_rc_parse ("testctreerc");
  create_ctree ();
  gtk_main ();
  return 0;
}
