#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <glib.h>
#include "region.h"
#include "at_type.h"

AudioRegion *
region_new (size_t length, gint bps)
{
  AudioRegion *new;

  int fd;

  new = (AudioRegion *) g_malloc (sizeof (AudioRegion));

  if (new == NULL)
    g_error ("Unable to allocate AudioRegion");

  if (length < MIN_MMAP_LENGTH) 
    {
      new->type = REGION_MALLOC;
      new->filename = g_strdup ("none");
      new->offset = 0;
      new->length = length;
      new->bps = bps;
      new->data = g_malloc (length * bps);
    }
  else
    {
      new->type = REGION_MMAP;
      new->filename = tempnam ("./", "atech");
      new->offset = 0;
      new->bps = bps;
      new->length = length;

      if ((fd = open(new->filename, O_RDWR | O_CREAT | O_TRUNC, 
		     S_IRUSR | S_IWUSR)) < 0)
	g_error ("can't create %s for writting: %s", 
		 new->filename,
		 g_strerror (errno));

      if (lseek(fd, length * bps -1, SEEK_SET) == -1)
	g_warning ("lseek: %s", g_strerror (errno));
      if (write(fd, "\0", 1) != 1)
	g_warning ("write: %s", g_strerror (errno));

      if ((new->data = mmap (0, length * bps, PROT_READ | PROT_WRITE,
			     MAP_FILE | MAP_SHARED, fd, 0)) == (void *) -1)
	g_error ("mmap: %s", g_strerror (errno));
      
      new->fd = fd;
    }

  g_print ("new region - bps = %d, filename = '%s'\n", new->bps, new->filename);

  return new;
}

/* this is completely useless... oh well.... */
AudioRegion *
region_new_from_data (void *data, size_t length, gint bps)
{
  AudioRegion *region;

  region = region_new (length, bps);
  
  memcpy (region->data, data, length * bps);

  return region;
}

AudioRegion *
region_new_from_file (char *filename, gint bps) 
{
  struct stat statbuf;
  AudioRegion *new = NULL;
  
  new = (AudioRegion *) g_malloc (sizeof (AudioRegion));

  g_print ("region_new_from_file (char *filename = '%s', gint bps = %d)\n",
	   filename, bps);
   
  new->type = REGION_MMAP;      
  new->filename = g_strdup (filename);
  new->offset = 0;
  new->bps = bps;
  
  if ((new->fd = open(new->filename, O_RDWR, 
		 S_IRUSR | S_IWUSR)) < 0)
    {
      g_warning ("can't create %s for writting: %s", 
		 new->filename,
		 g_strerror (errno));

      g_free (new->filename);
      g_free (new);
      return (NULL);
    }

  if (fstat(new->fd, &statbuf) < 0)
    {
      g_warning ("fstat error");
      g_free (new->filename);
      g_free (new);
      return (NULL);
    }
  new->length = statbuf.st_size / bps; 

  new->data = mmap (0, statbuf.st_size, PROT_READ | PROT_WRITE,
		    MAP_FILE | MAP_SHARED, new->fd, 0);

  if (new->data == (void *) -1)
    {
      g_warning ("mmap error for output");
      g_free (new->filename);
      g_free (new);
      return (NULL);
    }

  return (new);

}

  
void 
region_delete (AudioRegion *del)
{
  g_return_if_fail (del != NULL);

  g_print ("region_delete - filename is '%s'\n", del->filename);

  switch (del->type)
    {
    case REGION_MALLOC:

      g_free (del->filename);
      if (del->data) 
	g_free (del->data);
      g_free (del);
      break;

    case REGION_MMAP:
      /* 
	 if (msync (del->data, del->length * del->bps , MS_SYNC | MS_INVALIDATE) == -1)
	g_warning ("msync: %s", g_strerror (errno));
	*/
      if (munmap (del->data, del->length * del->bps) == -1)
	g_warning ("munmap: %s", g_strerror (errno));
      if (close (del->fd) == -1) 
	g_warning ("failed to close file: %s : %s", 
		   del->filename, g_strerror (errno));
      if (unlink (del->filename) == -1)
	g_warning ("failed to unlink file: %s : %s",
		   del->filename, g_strerror (errno));

      g_free (del->filename);
      g_free (del);
      break;

    default:
      g_warning ("trying to delete region with unknown type");
    }
}

void
region_sync (AudioRegion *dest, gint type)
{
  switch (dest->type)
    {
    case REGION_MMAP:
#if 0
      if (type == AT_SYNC)
	{
	  if (msync (dest->data, dest->length * dest->bps, 
		     MS_SYNC | MS_INVALIDATE) == -1)
	    g_warning ("msync: %s", g_strerror (errno));
	}
      else
#endif
	if (msync (dest->data, dest->length * dest->bps,
		   MS_ASYNC | MS_INVALIDATE) == -1)
	  g_warning ("msync: %s", g_strerror (errno));
      
  break;
    }
}

void 
region_copy (AudioRegion *src, AudioRegion *dest,
		  off_t s_offset, off_t d_offset,
		  size_t length)
{
   void *dest_data;
   void *src_data;

   g_return_if_fail (src != NULL);
   g_return_if_fail (dest != NULL);

   /* we could check the lengths too... */
   /* FIXME: need to handle negative offsets in a sane way... */

   dest_data = dest->data + d_offset;
   src_data = src->data + s_offset;
   
   if (length > src->length)
     {
       g_warning ("stupido");
       return;
     }

   memcpy (dest_data, src_data, length * src->bps);
}

AudioRegion *
region_duplicate (AudioRegion *src, off_t offset, size_t length)
{
  AudioRegion *new;
  void *src_data;

  /* at some point this might be usefull... could do a cow maping etc..*/

  new = region_new (length, src->bps);
  src_data = src->data + offset * src->bps;

  memcpy (new->data, src_data, length * src->bps);
  
  return new;
} 


void
region_resize (AudioRegion *src, size_t offset, size_t length)
{
  void *new_data;
  AudioRegion *dest;
  AudioRegion tmp;
  gint copy_length = 0;

  g_return_if_fail (offset >= 0);


  /* this could be done better.... */
  switch (src->type)
    {
    case REGION_MMAP:

#define OPTIMIZE_RESIZE
#ifdef OPTIMIZE_RESIZE
      if ((offset == 0) && (length > src->length))
	{ 
	  region_sync (src, AT_ASYNC);
	  if (lseek(src->fd, length * src->bps -1, SEEK_SET) == -1)
	    g_warning ("lseek: %s", g_strerror (errno));
	  if (write(src->fd, "\0", 1) != 1)
	    g_warning ("write: %s", g_strerror (errno));  
	  if ((new_data = mmap (0, length * src->bps, PROT_READ | PROT_WRITE,
				MAP_FILE | MAP_SHARED, src->fd, 0)
	       ) == (void *) -1)
	    g_error ("mmap: %s", g_strerror (errno));
	  /* this is not strictly needed in this case I think... */
	  //memmove (new_data, src->data, src->length * src->bps);
	  if (munmap (src->data, src->length * src->bps) == -1)
	    g_warning ("munmap: %s", g_strerror(errno));
	  src->data = new_data;
	  src->length = length;
	  break;
	}
#endif OPTIMIZE_RESIZE

    case REGION_MALLOC:
      dest = region_new (length, src->bps);
      /* FIXME: need to handle negative offsets in a reasonable way.... */
      /* FIXME: this is ugly anyway.... */
      copy_length = MIN (length, src->length);
      region_copy (src, dest, 0, offset, copy_length);
      tmp = *src;
      *src = *dest;
      *dest = tmp;

      region_delete (dest);
      
      break;
    default:
      g_warning ("region_resize: unable resize region of unknown type");
    }
}

void 
region_zero (AudioRegion *dest)
{
  memset (dest->data, 0, dest->length * dest->bps);
}

/* ignore this it is just a test function */
AudioRegion *
region_combine (AudioRegion *a,
		AudioRegion *b)
		
{
  AudioRegion *dest;
  size_t comb_length;
  gint16 *a_16, *b_16, *dest_16;
  gint i;
  gint bps;

  if ((a == NULL) || (b == NULL))
    return (NULL);
  
  /* 
   * FIXME PLEASE... this is serverely broken don't even try to use it
   * it's old cruft, I just don't want to remove it right now
   */

  comb_length = a->length < b->length ? a->length : b->length;
  bps = MAX (a->bps, b->bps);
  dest = region_new (comb_length, bps);
  
  a_16 = a->data;
  b_16 = b->data;
  dest_16 = dest->data;

  g_print("hmmm... mixing segment of length %d", a->length);

  while ((comb_length - bps) > 0)  
    {
      for (i = 0; i < 2; i++)
	dest_16[i] = (gint16)(a_16[i] * 0.5 + b_16[i] * 0.5 + .00001);
    
      dest_16 += 2;
      a_16 += 2;
      b_16 += 2;
    }

  g_print("hmmm... looking bad :) a->len: %d b->len: %d\n", a->length, b->length);

  return (dest);
}
  
/* 
   this need to be changed if I ever fix the mapping stuff... 
   it should eventually ref the map region possibly checking the
   map cache if I decide to go that way? 
   */

void
tmp_region_init (TmpRegion *dest,
		 AudioRegion *src,
		 size_t offset,
		 size_t length)
{

  dest->bps = src->bps;
  
  dest->type = src->type;
  dest->length = length;

  dest->data = ((atdata *)src->data) + dest->bps * offset;
}



