#!/usr/bin/env python

# GMSK modulation and demodulation test.
#
#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 
# $Id: gmsk-test.py,v 1.1 2005/03/29 22:15:45 eb Exp $

import sys
import os
from math import pi
from time import time
from optparse import OptionParser

from gnuradio import gr
from gnuradio.blks import gmsk_mod, gmsk_demod
from gnuradio.eng_option import eng_option

if __name__ == '__main__':

	parser = OptionParser(option_class = eng_option)
	parser.add_option("-s", "--samples-per-symbol", type = "intx",
	   default = 8, help = "samples per symbol (integer)")
	parser.add_option("-r", "--symbols-per-second", type = "eng_float",
	   default = 1625000.0 / 6.0, help = "symbols per second")
	parser.add_option("-p", "--packet-size", type = "intx", default = 1024,
	   help = "data block size")
	parser.add_option("-i", "--if-freq", type = "eng_float",
	   default = 7.5e5, help = "intermediate frequency")
	parser.add_option("-w", "--filter-width", type = "eng_float",
	   default = 2.5e5, help = "low pass filter width")
	parser.add_option("-t", "--transition-width", type = "eng_float",
	   default = 5e5, help = "transition width of low pass filter")
	parser.add_option("-n", "--noise", type = "eng_float",
	   default = 0.0, help = "add AWGN noise (mag)")
	parser.add_option("-f", "--filename", type = "string",
	   default = "./payload.dat", help = "file name of data to transmit")
	parser.add_option("-o", "--output", type = "string", default = "",
	   help = "file for output data")
	parser.add_option("-a", "--air", action = "store_true", default = False,
	   help = "simulate actual transmit")
	(options, args) = parser.parse_args()

	sps = options.samples_per_symbol
	symbol_rate = options.symbols_per_second
	sample_rate = sps * symbol_rate
	p_size = options.packet_size
	lo_freq = options.if_freq
	lp_cutoff = options.filter_width
	lp_tw = options.transition_width
	noise_mag = options.noise
	filename = options.filename
	output_filename = options.output

	print "sps:\t\t" + str(sps)
	print "symbol_rate:\t" + str(symbol_rate)
	print "sample_rate:\t" + str(sample_rate)
	print "p_size:\t\t" + str(p_size)
	if options.air:
		print "lo_freq:\t" + str(lo_freq)
		print "lp_cutoff:\t" + str(lp_cutoff)
		print "lp_tw:\t\t" + str(lp_tw)
		print "noise_mag:\t" + str(noise_mag)

	fg = gr.flow_graph()

	src_data_file = file(filename, "r")
	src_data = src_data_file.read()
	src_data_len = len(src_data)
	src = gr.file_source(1, filename, 0)

	# GMSK modulate input bytes to baseband
	mod = gmsk_mod(fg, sps, symbol_rate, 0.3, p_size)

	if options.air:
		# Mix to IF
		lo = gr.sig_source_c(sample_rate, gr.GR_SIN_WAVE, lo_freq,
		   1.0, 0)
		mixer = gr.multiply_cc()
		fg.connect(lo, (mixer, 1))

		# Take real part as transmit
		ctof = gr.complex_to_float()

		# Simulate noise in the channel
		noise = gr.noise_source_f(gr.GR_GAUSSIAN, noise_mag)
		air_noise = gr.add_ff()
		fg.connect(noise, (air_noise, 1))

		# Mix to baseband
		chan_taps = gr.firdes.low_pass(1.0, sample_rate, lp_cutoff,
		   lp_tw, gr.firdes.WIN_HAMMING)
		chan_filter = gr.freq_xlating_fir_filter_fcf(1, chan_taps,
		   -lo_freq, sample_rate)

	# GMSK demodulate baseband to bytes
	demod = gmsk_demod(fg, sps, symbol_rate, p_size)

	# Sink
#	dst = gr.vector_sink_b()
#	dst = gr.null_sink(gr.sizeof_char)
	if output_filename == "":
		dst = gr.null_sink(gr.sizeof_char)
	else:
		dst = gr.file_sink(1, output_filename)

	# Connect
	fg.connect(src, mod.head)
	if options.air:
		fg.connect(mod.tail, mixer, ctof, air_noise, chan_filter,
		   demod.head)
	else:
		fg.connect(mod.tail, demod.head)
	fg.connect(demod.tail, dst)

	start = time()
	fg.run()
	finish = time()
	elapsed = finish - start
	print "tx data: " + str(src_data_len)
	tBs = src_data_len / elapsed
	print "elapsed time: %f seconds, %dkb/s" % (elapsed, 8 * tBs / 1000)

# vim:ts=8
