/*
 * Utility functions
 */

#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <unistd.h>

#include "util.h"

void abortmsg(const char *fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    fputs("\033[1;31mchg: abort: ", stderr);
    vfprintf(stderr, fmt, args);
    fputs("\033[m\n", stderr);
    va_end(args);

    exit(255);
}

static int debugmsgenabled = 0;

void enabledebugmsg(void)
{
    debugmsgenabled = 1;
}

void debugmsg(const char *fmt, ...)
{
    if (!debugmsgenabled) return;

    va_list args;
    va_start(args, fmt);
    fputs("\033[1;30mchg: debug: ", stderr);
    vfprintf(stderr, fmt, args);
    fputs("\033[m\n", stderr);
    va_end(args);
}

/*!
 * Get terminal width if available; otherwise return 80 as default
 *
 * This implementation is based on mercurial/posix.py.
 */
unsigned int gettermwidth(void)
{
    const char *columnsenv = getenv("COLUMNS");
    if (columnsenv) {
        int n = atoi(columnsenv);
        if (n > 0) return (unsigned int) n;
    }

    static const int fds[] = {STDERR_FILENO, STDOUT_FILENO, STDIN_FILENO};
    for (size_t i = 0; i < sizeof(fds) / sizeof(fds[0]); ++i) {
        if (!isatty(fds[i])) continue;
        struct winsize ws;
        if (ioctl(fds[i], TIOCGWINSZ, &ws) < 0) {
            debugmsg("error while getting termwidth from %d (errno = %d)",
                     fds[i], errno);
            continue;
        }
        if (ws.ws_col > 0)
            return ws.ws_col;
    }
    return 80;
}
