module mrtindex_sec_science
  use gkernel_interfaces
  !---------------------------------------------------------------------
  ! Support module for science section in MRTCAL index
  !---------------------------------------------------------------------
  !
  public :: sec_science_id
  public :: sec_science_len
  public :: sec_science_t
  public :: sec_science_array_t
  private
  !
  ! Section Science: reserved for science scans.
  integer(kind=4), parameter :: sec_science_id=3
  integer(kind=8), parameter :: sec_science_len=4
  type sec_science_t
    ! Save the reference(s) of the calibration scan(s) used:
    integer(kind=4) :: caldobs  ! [gag_date]  Date of the associated calibr. scan
    integer(kind=4) :: calscan  ! [        ]  Scan
    integer(kind=4) :: calback  ! [    code]  Backend
    integer(kind=4) :: calvers  ! [        ]  Version
  contains
    procedure, public :: variable => science_variable
  end type sec_science_t
  !
  type sec_science_array_t
    integer(kind=4)              :: nent        ! Used size of 'ent' dimension (allocation may be larger)
    integer(kind=4), allocatable :: caldobs(:)  ! [nent]
    integer(kind=4), allocatable :: calscan(:)  ! [nent]
    integer(kind=4), allocatable :: calback(:)  ! [nent]
    integer(kind=4), allocatable :: calvers(:)  ! [nent]
  contains
    procedure, public :: reallocate => science_array_reallocate
    procedure, public :: set        => science_array_set
    procedure, public :: variable   => science_array_variable
  end type sec_science_array_t
  !
contains
  !
  subroutine science_variable(sci,struct,ro,error)
    !---------------------------------------------------------------------
    ! @ private
    !---------------------------------------------------------------------
    class(sec_science_t), intent(in)    :: sci     !
    character(len=*),     intent(in)    :: struct  ! Structure name
    logical,              intent(in)    :: ro      ! Read-Only?
    logical,              intent(inout) :: error   ! Logical error flag
    !
    logical :: userreq
    character(len=32) :: str
    !
    userreq = .false.
    str = trim(struct)//'%SCI'
    !
    call sic_delvariable(str,userreq,error)
    call sic_defstructure(str,.true.,error)
    if (error)  return
    !
    call sic_def_inte(trim(str)//'%CALDOBS',sci%caldobs,0,0,ro,error)
    call sic_def_inte(trim(str)//'%CALSCAN',sci%calscan,0,0,ro,error)
    call sic_def_inte(trim(str)//'%CALBACK',sci%calback,0,0,ro,error)
    call sic_def_inte(trim(str)//'%CALVERS',sci%calvers,0,0,ro,error)
    if (error)  return
  end subroutine science_variable
  !
  subroutine science_array_reallocate(sci,nent,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(sec_science_array_t), intent(inout) :: sci
    integer(kind=8),            intent(in)    :: nent
    logical,                    intent(inout) :: error
    !
    character(len=*), parameter :: rname='VARIABLE'
    logical :: realloc
    integer(kind=4) :: ier
    !
    sci%nent = nent
    !
    if (allocated(sci%caldobs)) then
      ! Lower than is fine for trailing dimension
      realloc = ubound(sci%caldobs,1).lt.sci%nent
      if (sci%nent.le.0 .or. realloc) then
        deallocate(sci%caldobs,sci%calscan,sci%calback,sci%calvers)
      endif
    else
      realloc = sci%nent.gt.0
    endif
    !
    if (realloc) then
      allocate(sci%caldobs(sci%nent),sci%calscan(sci%nent),stat=ier)
      if (failed_allocate(rname,'MDX%HEAD%SCI arrays',ier,error))  return
      allocate(sci%calback(sci%nent),sci%calvers(sci%nent),stat=ier)
      if (failed_allocate(rname,'MDX%HEAD%SCI arrays',ier,error))  return
    endif
  end subroutine science_array_reallocate
  !
  subroutine science_array_set(varsci,ient,sec,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(sec_science_array_t), intent(inout) :: varsci
    integer(kind=8),            intent(in)    :: ient
    type(sec_science_t),        intent(in)    :: sec
    logical,                    intent(inout) :: error
    !
    varsci%caldobs(ient) = sec%caldobs
    varsci%calscan(ient) = sec%calscan
    varsci%calback(ient) = sec%calback
    varsci%calvers(ient) = sec%calvers
  end subroutine science_array_set
  !
  subroutine science_array_variable(sci,str,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(sec_science_array_t), intent(in)    :: sci
    character(len=*),           intent(in)    :: str
    logical,                    intent(inout) :: error
    !
    logical, parameter :: ro=.true.
    character(len=32) :: topstr
    integer(kind=index_length) :: dims(sic_maxdims)
    !
    topstr = trim(str)//'%sci'
    call sic_defstructure(topstr,.true.,error)
    if (error)  return
    dims(1) = sci%nent
    call sic_def_inte(trim(topstr)//'%CALDOBS',sci%caldobs,1,dims,ro,error)
    call sic_def_inte(trim(topstr)//'%CALSCAN',sci%calscan,1,dims,ro,error)
    call sic_def_inte(trim(topstr)//'%CALBACK',sci%calback,1,dims,ro,error)
    call sic_def_inte(trim(topstr)//'%CALVERS',sci%calvers,1,dims,ro,error)
    if (error)  return
  end subroutine science_array_variable
end module mrtindex_sec_science
