#define _POSIX_C_SOURCE 200112L
#include <ctype.h>
#include <errno.h>
#include <getopt.h>
#include <stdio.h>
#include <string.h>

static void print_usage(const char *name) {
	fprintf(stdout, "Usage: %s -o output.c [flags...] [assets...]\n", name);
}

int main(int argc, char **argv) {
	const char *deinit = "koio_unload_assets";
	const char *init = "koio_load_assets";
	const char *output = NULL;

	int c;
	while ((c = getopt(argc, argv, "hd:i:o:")) != -1) {
		switch (c) {
		case 'h':
			print_usage(argv[0]);
			return 0;
		case 'd':
			deinit = optarg;
			break;
		case 'i':
			init = optarg;
			break;
		case 'o':
			output = optarg;
			break;
		default:
			print_usage(argv[0]);
			return 1;
		}
	}

	if (!output) {
		fprintf(stderr, "Must specify an output file.\n");
		return 1;
	}

	FILE *out;
	if (strcmp(output, "-") == 0) {
		out = stdout;
	} else {
		out = fopen(output, "w");
		if (!out) {
			fprintf(stderr, "Unable to open '%s' for writing: %s\n",
					output, strerror(errno));
			return 1;
		}
	}

	fprintf(out, "#include <koio.h>\n\n");
	fprintf(out,
			"static struct {\n"
			"	const char *path;\n"
			"	size_t len;\n"
			"	char *data;\n"
			"} files[] = {\n");

	for (int i = optind; i < argc; ++i) {
		char *real_path = argv[i],
			*virt_path = argv[i];
		char *colon = strchr(argv[i], ':');
		if (colon) {
			*colon = '\0';
			virt_path = colon + 1;
		}
		FILE *in = fopen(real_path, "r");
		if (!in) {
			fprintf(stderr, "Unable to open '%s' for reading: %s\n",
					real_path, strerror(errno));
		}
		fseek(in, 0, SEEK_END);
		size_t len = ftell(in);
		fseek(in, 0, SEEK_SET);
		fprintf(out,
				"	{\n"
				"		.path = \"%s\",\n"
				"		.len = %zu,\n"
				"		.data = \"\"", virt_path, len);
		int cols = 1;
		int c;
		while ((c = fgetc(in)) != EOF) {
			if (cols == 1) {
				fprintf(out, "\n\"");
			}
			switch (c) {
			case '\\':
				cols += fprintf(out, "\\\\");
				break;
			case '\"':
				cols += fprintf(out, "\\\"");
				break;
			case '\n':
				cols += fprintf(out, "\\n");
				break;
			case '\t':
				cols += fprintf(out, "\\t");
				break;
			default:
				if (isprint(c)) {
					cols += fprintf(out, "%c", c);
				} else {
					cols += fprintf(out, "\\x%02X", c);
				}
				break;
			}
			if (cols >= 79) {
				cols = 1;
				fprintf(out, "\"");
			}
		}
		if (cols != 1) {
			fprintf(out, "\"");
		}
		fprintf(out,
				",\n"
				"	},\n");
	}
	fprintf(out, "};\n\n");

	fprintf(out, "void %s(void) {\n", init);
	for (int i = 0; i < argc - optind; ++i) {
		fprintf(out, "	ko_add_file(files[%d].path, "
				"files[%d].data, files[%d].len);\n", i, i, i);
	}
	fprintf(out, "}\n\n");

	fprintf(out, "void %s(void) {\n", deinit);
	for (int i = 0; i < argc - optind; ++i) {
		fprintf(out, "	ko_del_file(files[%d].path);\n", i);
	}
	fprintf(out, "}\n");

	fclose(out);
	return 0;
}
