// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import mergeWith from 'lodash.mergewith';
import { Dialog, showDialog } from '@jupyterlab/apputils';
import { nullTranslator } from '@jupyterlab/translation';
import { Signal } from '@lumino/signaling';
const createButton = Dialog.createButton;
/**
 * The values should follow the https://microsoft.github.io/language-server-protocol/specification guidelines
 */
const MIME_TYPE_LANGUAGE_MAP = {
    'text/x-rsrc': 'r',
    'text/x-r-source': 'r',
    // currently there are no LSP servers for IPython we are aware of
    'text/x-ipython': 'python'
};
/**
 * Foreign code: low level adapter is not aware of the presence of foreign languages;
 * it operates on the virtual document and must not attempt to infer the language dependencies
 * as this would make the logic of inspections caching impossible to maintain, thus the WidgetAdapter
 * has to handle that, keeping multiple connections and multiple virtual documents.
 */
export class WidgetLSPAdapter {
    // note: it could be using namespace/IOptions pattern,
    // but I do not know how to make it work with the generic type T
    // (other than using 'any' in the IOptions interface)
    constructor(widget, options) {
        this.widget = widget;
        this.options = options;
        /**
         * Signal emitted when the adapter is connected.
         */
        this._adapterConnected = new Signal(this);
        /**
         * Signal emitted when the active editor have changed.
         */
        this._activeEditorChanged = new Signal(this);
        /**
         * Signal emitted when an editor is changed.
         */
        this._editorAdded = new Signal(this);
        /**
         * Signal emitted when an editor is removed.
         */
        this._editorRemoved = new Signal(this);
        /**
         * Signal emitted when the adapter is disposed.
         */
        this._disposed = new Signal(this);
        this._isDisposed = false;
        this._virtualDocument = null;
        this._connectionManager = options.connectionManager;
        this._isConnected = false;
        this._trans = (options.translator || nullTranslator).load('jupyterlab');
        // set up signal connections
        this.widget.context.saveState.connect(this.onSaveState, this);
        this.connectionManager.closed.connect(this.onConnectionClosed, this);
        this.widget.disposed.connect(this.dispose, this);
    }
    /**
     * Check if the adapter is disposed
     */
    get isDisposed() {
        return this._isDisposed;
    }
    /**
     * Check if the document contains multiple editors
     */
    get hasMultipleEditors() {
        return this.editors.length > 1;
    }
    /**
     * Get the ID of the internal widget.
     */
    get widgetId() {
        return this.widget.id;
    }
    /**
     * Get the language identifier of the document
     */
    get language() {
        // the values should follow https://microsoft.github.io/language-server-protocol/specification guidelines,
        // see the table in https://microsoft.github.io/language-server-protocol/specification#textDocumentItem
        if (MIME_TYPE_LANGUAGE_MAP.hasOwnProperty(this.mimeType)) {
            return MIME_TYPE_LANGUAGE_MAP[this.mimeType];
        }
        else {
            let withoutParameters = this.mimeType.split(';')[0];
            let [type, subtype] = withoutParameters.split('/');
            if (type === 'application' || type === 'text') {
                if (subtype.startsWith('x-')) {
                    return subtype.substring(2);
                }
                else {
                    return subtype;
                }
            }
            else {
                return this.mimeType;
            }
        }
    }
    /**
     * Signal emitted when the adapter is connected.
     */
    get adapterConnected() {
        return this._adapterConnected;
    }
    /**
     * Signal emitted when the active editor have changed.
     */
    get activeEditorChanged() {
        return this._activeEditorChanged;
    }
    /**
     * Signal emitted when the adapter is disposed.
     */
    get disposed() {
        return this._disposed;
    }
    /**
     * Signal emitted when the an editor is changed.
     */
    get editorAdded() {
        return this._editorAdded;
    }
    /**
     * Signal emitted when the an editor is removed.
     */
    get editorRemoved() {
        return this._editorRemoved;
    }
    /**
     * The virtual document is connected or not
     */
    get isConnected() {
        return this._isConnected;
    }
    /**
     * The LSP document and connection manager instance.
     */
    get connectionManager() {
        return this._connectionManager;
    }
    /**
     * The translator provider.
     */
    get trans() {
        return this._trans;
    }
    /**
     * Promise that resolves once the document is updated
     */
    get updateFinished() {
        return this._updateFinished;
    }
    /**
     * Internal virtual document of the adapter.
     */
    get virtualDocument() {
        return this._virtualDocument;
    }
    /**
     * Callback on connection closed event.
     */
    onConnectionClosed(_, { virtualDocument }) {
        if (virtualDocument === this.virtualDocument) {
            this.dispose();
        }
    }
    /**
     * Dispose the adapter.
     */
    dispose() {
        if (this._isDisposed) {
            return;
        }
        this._isDisposed = true;
        this.disconnect();
        this._virtualDocument = null;
        this._disposed.emit();
        Signal.clearData(this);
    }
    /**
     * Disconnect virtual document from the language server.
     */
    disconnect() {
        var _a, _b;
        const uri = (_a = this.virtualDocument) === null || _a === void 0 ? void 0 : _a.uri;
        const { model } = this.widget.context;
        if (uri) {
            this.connectionManager.unregisterDocument(uri);
        }
        model.contentChanged.disconnect(this._onContentChanged, this);
        // pretend that all editors were removed to trigger the disconnection of even handlers
        // they will be connected again on new connection
        for (let { ceEditor: editor } of this.editors) {
            this._editorRemoved.emit({
                editor: editor
            });
        }
        (_b = this.virtualDocument) === null || _b === void 0 ? void 0 : _b.dispose();
    }
    /**
     * Update the virtual document.
     */
    updateDocuments() {
        if (this._isDisposed) {
            console.warn('Cannot update documents: adapter disposed');
            return Promise.reject('Cannot update documents: adapter disposed');
        }
        return this.virtualDocument.updateManager.updateDocuments(this.editors);
    }
    /**
     * Callback called on the document changed event.
     */
    documentChanged(virtualDocument, document, isInit = false) {
        if (this._isDisposed) {
            console.warn('Cannot swap document: adapter disposed');
            return;
        }
        // TODO only send the difference, using connection.sendSelectiveChange()
        let connection = this.connectionManager.connections.get(virtualDocument.uri);
        if (!(connection === null || connection === void 0 ? void 0 : connection.isReady)) {
            console.log('Skipping document update signal: connection not ready');
            return;
        }
        connection.sendFullTextChange(virtualDocument.value, virtualDocument.documentInfo);
    }
    // equivalent to triggering didClose and didOpen, as per syncing specification,
    // but also reloads the connection; used during file rename (or when it was moved)
    reloadConnection() {
        // ignore premature calls (before the editor was initialized)
        if (this.virtualDocument === null) {
            return;
        }
        // disconnect all existing connections (and dispose adapters)
        this.disconnect();
        // recreate virtual document using current path and language
        // as virtual editor assumes it gets the virtual document at init,
        // just dispose virtual editor (which disposes virtual document too)
        // and re-initialize both virtual editor and document
        this.initVirtual();
        // reconnect
        this.connectDocument(this.virtualDocument, true).catch(console.warn);
    }
    /**
     * Callback on document saved event.
     */
    onSaveState(context, state) {
        // ignore premature calls (before the editor was initialized)
        if (this.virtualDocument === null) {
            return;
        }
        if (state === 'completed') {
            // note: must only be send to the appropriate connections as
            // some servers (Julia) break if they receive save notification
            // for a document that was not opened before, see:
            // https://github.com/jupyter-lsp/jupyterlab-lsp/issues/490
            const documentsToSave = [this.virtualDocument];
            for (let virtualDocument of documentsToSave) {
                let connection = this.connectionManager.connections.get(virtualDocument.uri);
                if (!connection) {
                    continue;
                }
                connection.sendSaved(virtualDocument.documentInfo);
                for (let foreign of virtualDocument.foreignDocuments.values()) {
                    documentsToSave.push(foreign);
                }
            }
        }
    }
    /**
     * Connect the virtual document with the language server.
     */
    async onConnected(data) {
        let { virtualDocument } = data;
        this._adapterConnected.emit(data);
        this._isConnected = true;
        try {
            await this.updateDocuments();
        }
        catch (reason) {
            console.warn('Could not update documents', reason);
            return;
        }
        // refresh the document on the LSP server
        this.documentChanged(virtualDocument, virtualDocument, true);
        data.connection.serverNotifications['$/logTrace'].connect((connection, message) => {
            console.log(data.connection.serverIdentifier, 'trace', virtualDocument.uri, message);
        });
        data.connection.serverNotifications['window/logMessage'].connect((connection, message) => {
            console.log(connection.serverIdentifier + ': ' + message.message);
        });
        data.connection.serverNotifications['window/showMessage'].connect((connection, message) => {
            void showDialog({
                title: this.trans.__('Message from ') + connection.serverIdentifier,
                body: message.message
            });
        });
        data.connection.serverRequests['window/showMessageRequest'].setHandler(async (params) => {
            const actionItems = params.actions;
            const buttons = actionItems
                ? actionItems.map(action => {
                    return createButton({
                        label: action.title
                    });
                })
                : [createButton({ label: this.trans.__('Dismiss') })];
            const result = await showDialog({
                title: this.trans.__('Message from ') + data.connection.serverIdentifier,
                body: params.message,
                buttons: buttons
            });
            const choice = buttons.indexOf(result.button);
            if (choice === -1) {
                return null;
            }
            if (actionItems) {
                return actionItems[choice];
            }
            return null;
        });
    }
    /**
     * Opens a connection for the document. The connection may or may
     * not be initialized, yet, and depending on when this is called, the client
     * may not be fully connected.
     *
     * @param virtualDocument a VirtualDocument
     * @param sendOpen whether to open the document immediately
     */
    async connectDocument(virtualDocument, sendOpen = false) {
        virtualDocument.foreignDocumentOpened.connect(this.onForeignDocumentOpened, this);
        const connectionContext = await this._connect(virtualDocument).catch(console.error);
        if (connectionContext && connectionContext.connection) {
            virtualDocument.changed.connect(this.documentChanged, this);
            if (sendOpen) {
                connectionContext.connection.sendOpenWhenReady(virtualDocument.documentInfo);
            }
        }
    }
    /**
     * Create the virtual document using current path and language.
     */
    initVirtual() {
        var _a;
        const { model } = this.widget.context;
        (_a = this._virtualDocument) === null || _a === void 0 ? void 0 : _a.dispose();
        this._virtualDocument = this.createVirtualDocument();
        model.contentChanged.connect(this._onContentChanged, this);
    }
    /**
     * Handler for opening a document contained in a parent document. The assumption
     * is that the editor already exists for this, and as such the document
     * should be queued for immediate opening.
     *
     * @param host the VirtualDocument that contains the VirtualDocument in another language
     * @param context information about the foreign VirtualDocument
     */
    async onForeignDocumentOpened(_, context) {
        const { foreignDocument } = context;
        await this.connectDocument(foreignDocument, true);
        foreignDocument.foreignDocumentClosed.connect(this._onForeignDocumentClosed, this);
    }
    /**
     * Callback called when a foreign document is closed,
     * the associated signals with this virtual document
     * are disconnected.
     */
    _onForeignDocumentClosed(_, context) {
        const { foreignDocument } = context;
        foreignDocument.foreignDocumentClosed.disconnect(this._onForeignDocumentClosed, this);
        foreignDocument.foreignDocumentOpened.disconnect(this.onForeignDocumentOpened, this);
        foreignDocument.changed.disconnect(this.documentChanged, this);
    }
    /**
     * Detect the capabilities for the document type then
     * open the websocket connection with the language server.
     */
    async _connect(virtualDocument) {
        let language = virtualDocument.language;
        let capabilities = {
            textDocument: {
                synchronization: {
                    dynamicRegistration: true,
                    willSave: false,
                    didSave: true,
                    willSaveWaitUntil: false
                }
            },
            workspace: {
                didChangeConfiguration: {
                    dynamicRegistration: true
                }
            }
        };
        capabilities = mergeWith(capabilities, this.options.featureManager.clientCapabilities());
        let options = {
            capabilities,
            virtualDocument,
            language,
            hasLspSupportedFile: virtualDocument.hasLspSupportedFile
        };
        let connection = await this.connectionManager.connect(options);
        if (connection) {
            await this.onConnected({ virtualDocument, connection });
            return {
                connection,
                virtualDocument
            };
        }
        else {
            return undefined;
        }
    }
    /**
     * Handle content changes and update all virtual documents after a change.
     *
     * #### Notes
     * Update to the state of a notebook may be done without a notice on the
     * CodeMirror level, e.g. when a cell is deleted. Therefore a
     * JupyterLab-specific signal is watched instead.
     *
     * While by not using the change event of CodeMirror editors we lose an easy
     * way to send selective (range) updates this can be still implemented by
     * comparison of before/after states of the virtual documents, which is
     * more resilient and editor-independent.
     */
    async _onContentChanged(_) {
        // Update the virtual documents.
        // Sending the updates to LSP is out of scope here.
        const promise = this.updateDocuments();
        if (!promise) {
            console.warn('Could not update documents');
            return;
        }
        this._updateFinished = promise.catch(console.warn);
        await this.updateFinished;
    }
}
//# sourceMappingURL=adapter.js.map