"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventManager = void 0;
const coreutils_1 = require("@jupyterlab/coreutils");
const polling_1 = require("@lumino/polling");
const signaling_1 = require("@lumino/signaling");
const serverconnection_1 = require("../serverconnection");
/**
 * The url for the jupyter-server events service.
 */
const SERVICE_EVENTS_URL = 'api/events';
/**
 * The events API service manager.
 */
class EventManager {
    /**
     * Create a new event manager.
     */
    constructor(options = {}) {
        var _a;
        this._socket = null;
        this.serverSettings =
            (_a = options.serverSettings) !== null && _a !== void 0 ? _a : serverconnection_1.ServerConnection.makeSettings();
        // If subscription fails, the poll attempts to reconnect and backs off.
        this._poll = new polling_1.Poll({ factory: () => this._subscribe() });
        this._stream = new signaling_1.Stream(this);
        // Subscribe to the events socket.
        void this._poll.start();
    }
    /**
     * Whether the event manager is disposed.
     */
    get isDisposed() {
        return this._poll.isDisposed;
    }
    /**
     * An event stream that emits and yields each new event.
     */
    get stream() {
        return this._stream;
    }
    /**
     * Dispose the event manager.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        // Clean up poll.
        this._poll.dispose();
        // Clean up socket.
        const socket = this._socket;
        if (socket) {
            this._socket = null;
            socket.onopen = () => undefined;
            socket.onerror = () => undefined;
            socket.onmessage = () => undefined;
            socket.onclose = () => undefined;
            socket.close();
        }
        // Clean up stream.
        signaling_1.Signal.clearData(this);
        this._stream.stop();
    }
    /**
     * Post an event request to be emitted by the event bus.
     */
    async emit(event) {
        const { serverSettings } = this;
        const { baseUrl, token } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const url = coreutils_1.URLExt.join(baseUrl, SERVICE_EVENTS_URL) +
            (token ? `?token=${token}` : '');
        const init = { body: JSON.stringify(event), method: 'POST' };
        const response = await makeRequest(url, init, serverSettings);
        if (response.status !== 204) {
            throw new ResponseError(response);
        }
    }
    /**
     * Subscribe to event bus emissions.
     */
    _subscribe() {
        return new Promise((_, reject) => {
            if (this.isDisposed) {
                return;
            }
            const { token, WebSocket, wsUrl } = this.serverSettings;
            const url = coreutils_1.URLExt.join(wsUrl, SERVICE_EVENTS_URL, 'subscribe') +
                (token ? `?token=${encodeURIComponent(token)}` : '');
            const socket = (this._socket = new WebSocket(url));
            const stream = this._stream;
            socket.onclose = () => reject(new Error('EventManager socket closed'));
            socket.onmessage = msg => msg.data && stream.emit(JSON.parse(msg.data));
        });
    }
}
exports.EventManager = EventManager;
//# sourceMappingURL=index.js.map