// Copyright (c) NetXS Group.
// Licensed under the MIT license.

/**
 * unidata.hpp, autogenerated on 2021-12-02 21:36:09.050928
 *
 * Provides fastest access to the Unicode Character Database.
 * Properties of a single Unicode character
 * are accessed by its code point value.
 *
 * Available properties:
 *  See struct 'uniprops'
 *
 * Project location
 *  D:\netxs\unidata
 *
 * Format conventions: https://www.unicode.org/reports/tr44/
 *
 * Character presentation width rules
 *  EAW:    https://www.unicode.org/reports/tr11
 *  Emoji:  https://www.unicode.org/reports/tr51
 *
 * Boundaries rules
 *  Grapheme Cluster: https://www.unicode.org/reports/tr29
 *
 * Unicode Character Database properties
 *  https://www.unicode.org/reports/tr44/#Property_Index
 *
 * All emoji characters are treated 'East Asian Wide'
 *  Current practice is for emoji to have a square aspect ratio,
 *  deriving from their origin in Japanese.
 *  https://www.unicode.org/reports/tr51/#Display
 *
 * VS16/15
 *  Emoji presentation sequences behave as though they were East Asian Wide,
 *  regardless of their assigned East_Asian_Width property value.
 *  https://www.unicode.org/reports/tr11/#Recommendations
 *
 * Ambiguous characters
 *  Ambiguous characters behave like wide or narrow characters
 *  depending on the context.
 *  If the context cannot be established reliably,
 *  they should be treated as narrow characters by default.
 *  https://www.unicode.org/reports/tr11/#Recommendations
 *
 * Categories of the character width
 *  0 - non-printable
 *  1 - Halfwidth
 *  2 - Fullwidth
 *
 * C0 controls 0x00..0x1F
 *  Since C0 codes are based on bytes, they are excluded from
 *  the property list of controls.
 *  They are analyzed in static tables before decoding UTF-8.
 *
 * C1 controls 0x80..0x9F
 *  The C1 codes require two bytes to be encoded in UTF-8
 *  (for instance CSI at U+009B is encoded as the bytes 0xC2, 0x9B in UTF-8),
 *  so there is no advantage to using them rather than the equivalent
 *  two-byte ESC+letter sequence, so the C1 controls are represented
 *  as C0 escaped sequences.
 *  The C1 characters appear outdated now.
 *  https://en.wikipedia.org/wiki/C0_and_C1_control_codes#Unicode
 *
 * Soft Hyphen
 *  Two variants:
 *    1. interpret it as a command and divide the text
 *       strings into two independent once
 *    2. append it to the last grapheme cluster
 *
 * Printable format characters
 *  A 'Prepend' characters always have the width 'Narrow' to be
 *  the basis of the grapheme cluster.
 *  https://www.unicode.org/reports/tr29/#Table_Combining_Char_Sequences_and_Grapheme_Clusters
 *  https://www.unicode.org/reports/tr29/#GB9b
 *
 * Names and Aliases
 *  Character name aliases are immutable.
 *  https://www.unicode.org/versions/Unicode12.0.0/ch04.pdf#page=24
 *
 *  -del- Invisible math operators
 *  -del- All of science and technology uses formulas,
 *  -del- equations, and mathematical notation as part of
 *  -del- the language of the subject.
 *  -del- Nevertheless, I suppose that invisible math operators should be dropped,
 *  -del- because there is no way to apply them using a cellular/monospaced display.
 *  -del- https://unicode.org/reports/tr25/#page=23
 *
 * https://www.unicode.org/cldr/utility/bidi.jsp
 *
 * Unicode Bidirectional Algorithm
 *  https://unicode.org/reports/tr9/
 *
 *
 *  control (should be enumerated with ascending)
 *    command: <NON_CONTROL (possible cause the paragraph endings)
 *            c0
 *            c1
 *            \u2029 PARAGRAPH SEPARATOR
 *    visible: =NON_CONTROL - non control chars
 *     format: >NON_CONTROL
 *             all other enumarated controls
 *
 *
 *
 *
 **/

#ifndef NETXS_UNIDATA_HPP
#define NETXS_UNIDATA_HPP

#include <cstdint>
#include <vector>
#include <iterator>

#ifndef faux
    #define faux (false)
#endif

namespace netxs::unidata
{
    namespace widths
    {
        enum type : unsigned char
        {
            zero,  // non-printable
            slim,  // narrow
            wide,  // fullwidth
        };
    }

    namespace gbreak
    {
        enum type : unsigned char
        {
            ANY     ,  // Other
            CR      ,  // CR
            LF      ,  // LF
            CTRL    ,  // Control
            EXT     ,  // Extend or Emoji_Modifier_Base
            L       ,  // HANGUL CHOSEONG
            V       ,  // HANGUL JUNGSEONG
            T       ,  // HANGUL JUNGSEONG
            LV      ,  // HANGUL SYLLABLE
            LVT     ,  // HANGUL SYLLABLE
            RI      ,  // Regional_Indicator
            SM      ,  // SpacingMark
            PREP    ,  // Prepend
            ZWJ     ,  // ZERO WIDTH JOINER
            EP      ,  // Extended_Pictographic
            COMBO   ,  // EP + ZWJ
        };
    }

    namespace cntrls
    {
        enum type : unsigned char
        {
            NUL                                       ,  //   0     0 NULL
            SOH                                       ,  //   1     1 START OF HEADING
            STX                                       ,  //   2     2 START OF TEXT
            ETX                                       ,  //   3     3 END OF TEXT
            EOT                                       ,  //   4     4 END OF TRANSMISSION
            ENQ                                       ,  //   5     5 ENQUIRY
            ACK                                       ,  //   6     6 ACKNOWLEDGE
            BEL                                       ,  //   7     7 ALERT
            BS                                        ,  //   8     8 BACKSPACE
            TAB                                       ,  //   9     9 CHARACTER TABULATION
            EOL                                       ,  //  10     A LINE FEED
            VT                                        ,  //  11     B LINE TABULATION
            FF                                        ,  //  12     C FORM FEED
            CR                                        ,  //  13     D CARRIAGE RETURN
            SO                                        ,  //  14     E SHIFT OUT
            SI                                        ,  //  15     F SHIFT IN
            DLE                                       ,  //  16    10 DATA LINK ESCAPE
            DC1                                       ,  //  17    11 DEVICE CONTROL ONE
            DC2                                       ,  //  18    12 DEVICE CONTROL TWO
            DC3                                       ,  //  19    13 DEVICE CONTROL THREE
            DC4                                       ,  //  20    14 DEVICE CONTROL FOUR
            NAK                                       ,  //  21    15 NEGATIVE ACKNOWLEDGE
            SYN                                       ,  //  22    16 SYNCHRONOUS IDLE
            ETB                                       ,  //  23    17 END OF TRANSMISSION BLOCK
            CAN                                       ,  //  24    18 CANCEL
            EOM                                       ,  //  25    19 END OF MEDIUM
            SUB                                       ,  //  26    1A SUBSTITUTE
            ESC                                       ,  //  27    1B ESCAPE
            FS                                        ,  //  28    1C INFORMATION SEPARATOR FOUR
            GS                                        ,  //  29    1D INFORMATION SEPARATOR THREE
            RS                                        ,  //  30    1E INFORMATION SEPARATOR TWO
            US                                        ,  //  31    1F INFORMATION SEPARATOR ONE
            DEL                                       ,  //  32    7F DELETE
            PAD                                       ,  //  33    80 PADDING CHARACTER
            HOP                                       ,  //  34    81 HIGH OCTET PRESET
            BPH                                       ,  //  35    82 BREAK PERMITTED HERE
            NBH                                       ,  //  36    83 NO BREAK HERE
            IND                                       ,  //  37    84 INDEX
            NEL                                       ,  //  38    85 NEXT LINE
            SSA                                       ,  //  39    86 START OF SELECTED AREA
            ESA                                       ,  //  40    87 END OF SELECTED AREA
            HTS                                       ,  //  41    88 CHARACTER TABULATION SET
            HTJ                                       ,  //  42    89 CHARACTER TABULATION WITH JUSTIFICATION
            VTS                                       ,  //  43    8A LINE TABULATION SET
            PLD                                       ,  //  44    8B PARTIAL LINE FORWARD
            PLU                                       ,  //  45    8C PARTIAL LINE BACKWARD
            RI                                        ,  //  46    8D REVERSE LINE FEED
            SS2                                       ,  //  47    8E SINGLE SHIFT TWO
            SS3                                       ,  //  48    8F SINGLE SHIFT THREE
            DCS                                       ,  //  49    90 DEVICE CONTROL STRING
            PU1                                       ,  //  50    91 PRIVATE USE ONE
            PU2                                       ,  //  51    92 PRIVATE USE TWO
            STS                                       ,  //  52    93 SET TRANSMIT STATE
            CCH                                       ,  //  53    94 CANCEL CHARACTER
            MW                                        ,  //  54    95 MESSAGE WAITING
            SPA                                       ,  //  55    96 START OF GUARDED AREA
            EPA                                       ,  //  56    97 END OF GUARDED AREA
            SOS                                       ,  //  57    98 START OF STRING
            SGC                                       ,  //  58    99 SINGLE GRAPHIC CHARACTER INTRODUCER
            SCI                                       ,  //  59    9A SINGLE CHARACTER INTRODUCER
            CSI                                       ,  //  60    9B CONTROL SEQUENCE INTRODUCER
            ST                                        ,  //  61    9C STRING TERMINATOR
            OSC                                       ,  //  62    9D OPERATING SYSTEM COMMAND
            PM                                        ,  //  63    9E PRIVACY MESSAGE
            APC                                       ,  //  64    9F APPLICATION PROGRAM COMMAND
            PARAGRAPH_SEPARATOR                       ,  //  65  2029 PARAGRAPH SEPARATOR
            NON_CONTROL                               ,  //  66    -1 NON CONTROL
            SHY                                       ,  //  67    AD SOFT HYPHEN
            ALM                                       ,  //  68   61C ARABIC LETTER MARK
            MVS                                       ,  //  69  180E MONGOLIAN VOWEL SEPARATOR
            ZWSP                                      ,  //  70  200B ZERO WIDTH SPACE
            ZWNJ                                      ,  //  71  200C ZERO WIDTH NON-JOINER
            LRM                                       ,  //  72  200E LEFT-TO-RIGHT MARK
            RLM                                       ,  //  73  200F RIGHT-TO-LEFT MARK
            LINE_SEPARATOR                            ,  //  74  2028 LINE SEPARATOR
            LRE                                       ,  //  75  202A LEFT-TO-RIGHT EMBEDDING
            RLE                                       ,  //  76  202B RIGHT-TO-LEFT EMBEDDING
            PDF                                       ,  //  77  202C POP DIRECTIONAL FORMATTING
            LRO                                       ,  //  78  202D LEFT-TO-RIGHT OVERRIDE
            RLO                                       ,  //  79  202E RIGHT-TO-LEFT OVERRIDE
            WJ                                        ,  //  80  2060 WORD JOINER
            FUNCTION_APPLICATION                      ,  //  81  2061 FUNCTION APPLICATION
            INVISIBLE_TIMES                           ,  //  82  2062 INVISIBLE TIMES
            INVISIBLE_SEPARATOR                       ,  //  83  2063 INVISIBLE SEPARATOR
            INVISIBLE_PLUS                            ,  //  84  2064 INVISIBLE PLUS
            LRI                                       ,  //  85  2066 LEFT-TO-RIGHT ISOLATE
            RLI                                       ,  //  86  2067 RIGHT-TO-LEFT ISOLATE
            FSI                                       ,  //  87  2068 FIRST STRONG ISOLATE
            PDI                                       ,  //  88  2069 POP DIRECTIONAL ISOLATE
            INHIBIT_SYMMETRIC_SWAPPING                ,  //  89  206A INHIBIT SYMMETRIC SWAPPING
            ACTIVATE_SYMMETRIC_SWAPPING               ,  //  90  206B ACTIVATE SYMMETRIC SWAPPING
            INHIBIT_ARABIC_FORM_SHAPING               ,  //  91  206C INHIBIT ARABIC FORM SHAPING
            ACTIVATE_ARABIC_FORM_SHAPING              ,  //  92  206D ACTIVATE ARABIC FORM SHAPING
            NATIONAL_DIGIT_SHAPES                     ,  //  93  206E NATIONAL DIGIT SHAPES
            NOMINAL_DIGIT_SHAPES                      ,  //  94  206F NOMINAL DIGIT SHAPES
            ZWNBSP                                    ,  //  95  FEFF ZERO WIDTH NO-BREAK SPACE
            INTERLINEAR_ANNOTATION_ANCHOR             ,  //  96  FFF9 INTERLINEAR ANNOTATION ANCHOR
            INTERLINEAR_ANNOTATION_SEPARATOR          ,  //  97  FFFA INTERLINEAR ANNOTATION SEPARATOR
            INTERLINEAR_ANNOTATION_TERMINATOR         ,  //  98  FFFB INTERLINEAR ANNOTATION TERMINATOR
            EGYPTIAN_HIEROGLYPH_VERTICAL_JOINER       ,  //  99 13430 EGYPTIAN HIEROGLYPH VERTICAL JOINER
            EGYPTIAN_HIEROGLYPH_HORIZONTAL_JOINER     ,  // 100 13431 EGYPTIAN HIEROGLYPH HORIZONTAL JOINER
            EGYPTIAN_HIEROGLYPH_INSERT_AT_TOP_START   ,  // 101 13432 EGYPTIAN HIEROGLYPH INSERT AT TOP START
            EGYPTIAN_HIEROGLYPH_INSERT_AT_BOTTOM_START,  // 102 13433 EGYPTIAN HIEROGLYPH INSERT AT BOTTOM START
            EGYPTIAN_HIEROGLYPH_INSERT_AT_TOP_END     ,  // 103 13434 EGYPTIAN HIEROGLYPH INSERT AT TOP END
            EGYPTIAN_HIEROGLYPH_INSERT_AT_BOTTOM_END  ,  // 104 13435 EGYPTIAN HIEROGLYPH INSERT AT BOTTOM END
            EGYPTIAN_HIEROGLYPH_OVERLAY_MIDDLE        ,  // 105 13436 EGYPTIAN HIEROGLYPH OVERLAY MIDDLE
            EGYPTIAN_HIEROGLYPH_BEGIN_SEGMENT         ,  // 106 13437 EGYPTIAN HIEROGLYPH BEGIN SEGMENT
            EGYPTIAN_HIEROGLYPH_END_SEGMENT           ,  // 107 13438 EGYPTIAN HIEROGLYPH END SEGMENT
            SHORTHAND_FORMAT_LETTER_OVERLAP           ,  // 108 1BCA0 SHORTHAND FORMAT LETTER OVERLAP
            SHORTHAND_FORMAT_CONTINUING_OVERLAP       ,  // 109 1BCA1 SHORTHAND FORMAT CONTINUING OVERLAP
            SHORTHAND_FORMAT_DOWN_STEP                ,  // 110 1BCA2 SHORTHAND FORMAT DOWN STEP
            SHORTHAND_FORMAT_UP_STEP                  ,  // 111 1BCA3 SHORTHAND FORMAT UP STEP
            MUSICAL_SYMBOL_BEGIN_BEAM                 ,  // 112 1D173 MUSICAL SYMBOL BEGIN BEAM
            MUSICAL_SYMBOL_END_BEAM                   ,  // 113 1D174 MUSICAL SYMBOL END BEAM
            MUSICAL_SYMBOL_BEGIN_TIE                  ,  // 114 1D175 MUSICAL SYMBOL BEGIN TIE
            MUSICAL_SYMBOL_END_TIE                    ,  // 115 1D176 MUSICAL SYMBOL END TIE
            MUSICAL_SYMBOL_BEGIN_SLUR                 ,  // 116 1D177 MUSICAL SYMBOL BEGIN SLUR
            MUSICAL_SYMBOL_END_SLUR                   ,  // 117 1D178 MUSICAL SYMBOL END SLUR
            MUSICAL_SYMBOL_BEGIN_PHRASE               ,  // 118 1D179 MUSICAL SYMBOL BEGIN PHRASE
            MUSICAL_SYMBOL_END_PHRASE                 ,  // 119 1D17A MUSICAL SYMBOL END PHRASE
            COUNT                                     ,  // 120
        };
    }

    struct unidata;
    inline unidata const& select(uint32_t cp);

    struct unidata
    {
        widths::type  ucwidth;
        gbreak::type  brgroup;
        cntrls::type  control;
        unsigned char padding = {};

        constexpr
        unidata()
            : ucwidth (widths::slim),
              brgroup (gbreak::ANY),
              control (cntrls::NON_CONTROL)
        { }

        constexpr
        unidata(widths::type ucwidth,
                gbreak::type brgroup,
                cntrls::type control)
            : ucwidth (ucwidth),
              brgroup (brgroup),
              control (control)
        { }

        unidata(uint32_t cp)
            : unidata(select(cp))
        { }

        unidata(unidata const&) = default;

        bool is_cmd()
        {
            return control < cntrls::NON_CONTROL;
        }

        // Unicode 13.0.0 UAX #29 https://www.unicode.org/reports/tr29/#Grapheme_Cluster_Boundary_Rules
        bool allied(gbreak::type& l) const
        {
            gbreak::type const& r = brgroup;
            auto result =
                (  l == gbreak::CR    &&  r == gbreak::LF   )  ? true: // GB3

                (  l >= gbreak::CR    &&  l <= gbreak::CTRL )  ? faux: // GB4

                (  r >= gbreak::CR    &&  r <= gbreak::CTRL )  ? faux: // GB5

                (  l == gbreak::L     && (r == gbreak::L
                                      ||  r == gbreak::V
                                      ||  r == gbreak::LV
                                      ||  r == gbreak::LVT  )) ? true: // GB6

                (( l == gbreak::LV    ||  l == gbreak::V    )
              && ( r == gbreak::V     ||  r == gbreak::T    )) ? true: // GB7

                (( l == gbreak::LVT   ||  l == gbreak::T    )
                                      &&  r == gbreak::T    )  ? true: // GB8

                (  l == gbreak::PREP  ||  r == gbreak::ZWJ
                                      ||  r == gbreak::SM
                                      ||  r == gbreak::EXT  )  ? true: // GB9,a,b

                (  l == gbreak::COMBO &&  r == gbreak::EP   )  ? true: // GB11

                (  l == gbreak::RI    &&  r == gbreak::RI   )  ? true: // GB12,13
                                                                 faux; // GB999
            if (l == gbreak::EP)
            {
                l = (r == gbreak::EXT) ? gbreak::EP    :
               	    (r == gbreak::ZWJ) ? gbreak::COMBO : r;
            }
            else
            {
                l = (l == gbreak::RI && r == gbreak::RI) ? gbreak::ANY : r;
            }
            return result;
        }
    };

    struct base
    {
        static constexpr size_t  blocks_size = 4352;
        static constexpr int32_t blocks_pack[] =
        {
              0,  -2, 256, 512, 768, 1024, 1280, 1536, 1792, 2048,
            2304, 2560, 2816, 3072, 3328, 3584, 3840, 4096, 4352, 4608,
             -2, 256, 4864, 5120, 5376, 5632, 5888, 6144, 6400, 6656,
            256, 6912, 7168, 7424, 256, 7680, 7936, 8192, 8448, 8704,
            256, 8960, 256, 9216, 9472, 9728, 9984, 10240, 10496, 10752,
            11008, -26, 11264, 11520, -86, 11264, 11776, 256, 12032, 12288,
            12544, 12800, 13056, 13312, 13568, 13824, 14080, 14336, 14592, 14848,
            15104, 13568, 13824, 14080, 14336, 14592, 14848, 15104, 13568, 13824,
            14080, 14336, 14592, 14848, 15104, 13568, 13824, 14080, 14336, 14592,
            14848, 15104, 13568, 13824, 14080, 14336, 14592, 14848, 15104, 13568,
            13824, 14080, 14336, 14592, 14848, 15104, 13568, 15360,  -8, 15616,
            -25, 256, 11264, 15872, 16128, 256, 16384, 16640, 16896, 17152,
            17408, 17664, 17920, 18176, 18432, 256, 18688, 18944, 19200, 19456,
            19712, 19968, 20224, 20480, 20736, 20992, 21248, 21504, 21760, 22016,
            22272, 22528, 22784, 23040, 23296, 23552, 15616, 23808, 24064, 24320,
            24576,  -3, 256, 24832, 25088, 25344,  -9, 15616, 25600,  -4,
            256, 25856, -15, 15616,  -2, 256, 26112, -33, 15616,  -2,
            256, 26368, 26624,  -2, 15616, 26880, 27136, -23, 11264, 27392,
             -4, 11264, 27648, 27904, -33, 15616, 28160, 11264, 28416, 28672,
             -9, 15616, 28928, -18, 15616, 29184, 29440, 29696, 29952, 30208,
            30464, 30720, 30976, 31232,  -2, 256, 31488,  -4, 15616, 31744,
            32000, 32256, 32512,  -4, 15616, 32768, 33024, 33280,  -2, 15616,
            33536, 33792, 34048, 15616, 34304, 34560, 34816, 35072, 35328, 35584,
            35840, 36096, 36352, 36608, 36864, 37120,  -3, 37376, 37632, -166,
            11264, 37888, -16, 11264, 38144, 38400, -21, 11264, 38656, -28,
            11264, 38912, -12, 15616,  -2, 11264, 39168,  -5, 15616, -19,
            11264, 39424, -2796, 15616, 39680, 39936, -14, 40192, -240, 15616,
            -255, 256, 40448, -255, 256, 40448
        };

        static constexpr size_t  offset_size = 40704;
        static constexpr int32_t offset_pack[] =
        {
              1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16,
             17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32,
            -95,  0, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46,
             47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62,
             63, 64, 65, -9,  0, 66, -3,  0, 67, 66,-337,  0,-112, 68, -8,  0,
             -2, 69, -6,  0, -4, 69, -7,  0, 69,  0, 69,-20,  0, 69,-224,  0,
             -7, 68,-166,  0, 69,-38,  0, -2, 69,-50,  0, -2, 69, -3,  0, 69,
            -45, 68,  0, 68,  0, -2, 68,  0, -2, 68,  0, 68, -8, 69,-27,  0,
             -4, 69, -6,  0,-11, 69, -6, 70,-10,  0,-11, 68,  0, 71,-46,  0,
            -21, 68,-16,  0, 68,-101,  0, -7, 68, 70,  0, -6, 68, -2,  0, -2,
             68,  0, -4, 68,-32,  0, 69, 70,  0, 68,-30,  0,-27, 68, -2, 69,
            -89,  0,-11, 68,  0,-14, 69,-43,  0, -9, 68, -7,  0, -2, 69, 68,
            -24,  0, -4, 68,  0, -9, 68,  0, -3, 68,  0, -5, 68, -2, 69,-15,
              0, 69,-25,  0, -3, 68, -2, 69,  0, 69,-11,  0, -5, 69,-31,  0,
             69, -2, 70, -6, 69, -8, 68,-42,  0,-24, 68, 70,-32, 68, 72,-54,
              0, 68, 72, 68,  0, -3, 72, -8, 68, -4, 72, 68, -2, 72,  0, -7,

             68,-10,  0, -2, 68,-29,  0, 68, -2, 72, 69, -8,  0, -2, 69, -2,
              0, -2, 69,-22,  0, 69, -7,  0, 69,  0, -3, 69, -4,  0, -2, 69,
             68,  0, 68, -2, 72, -4, 68, -2, 69, -2, 72, -2, 69, -2, 72, 68,
              0, -8, 69, 68, -4, 69, -2,  0, 69, -3,  0, -2, 68, -2, 69,-24,
              0, 68, -2, 69, -2, 68, 72, 69, -6,  0, -4, 69, -2,  0, -2, 69,
            -22,  0, 69, -7,  0, 69, -2,  0, 69, -2,  0, 69, -2,  0, -2, 69,
             68, 69, -3, 72, -2, 68, -4, 69, -2, 68, -2, 69, -3, 68, -3, 69,
             68, -7, 69, -4,  0, 69,  0, -7, 69,-10,  0, -2, 68, -3,  0, 68,
              0,-10, 69, -2, 68, 72, 69, -9,  0, 69, -3,  0, 69,-22,  0, 69,
             -7,  0, 69, -2,  0, 69, -5,  0, -2, 69, 68,  0, -3, 72, -5, 68,
             69, -2, 68, 72, 69, -2, 72, 68, -2, 69,  0,-15, 69, -2,  0, -2,
             68, -2, 69,-12,  0, -7, 69,  0, -6, 68, 69, 68, -2, 72, 69, -8,
              0, -2, 69, -2,  0, -2, 69,-22,  0, 69, -7,  0, 69, -2,  0, 69,
             -5,  0, -2, 69, 68,  0, -2, 68, 72, -4, 68, -2, 69, -2, 72, -2,
             69, -2, 72, 68, -7, 69, -3, 68, -4, 69, -2,  0, 69, -3,  0, -2,
             68, -2, 69,-18,  0,-10, 69, 68,  0, 69, -6,  0, -3, 69, -3,  0,

             69, -4,  0, -3, 69, -2,  0, 69,  0, 69, -2,  0, -3, 69, -2,  0,
             -3, 69, -3,  0, -3, 69,-12,  0, -4, 69, 68, 72, 68, -2, 72, -3,
             69, -3, 72, 69, -3, 72, 68, -2, 69,  0, -6, 69, 68,-14, 69,-21,
              0, -5, 69, 68, -3, 72, 68, -8,  0, 69, -3,  0, 69,-23,  0, 69,
            -16,  0, -2, 69, 68,  0, -3, 68, -4, 72, 69, -3, 68, 69, -4, 68,
             -7, 69, -2, 68, 69, -3,  0, -2, 69,  0, -2, 69, -2,  0, -2, 68,
             -2, 69,-10,  0, -7, 69,-10,  0, 68, -2, 72, -9,  0, 69, -3,  0,
             69,-23,  0, 69,-10,  0, 69, -5,  0, -2, 69, 68,  0, 72, 68, -2,
             72, 68, -2, 72, 69, 68, -2, 72, 69, -2, 72, -2, 68, -7, 69, -2,
             68, -6, 69, -2,  0, 69, -2,  0, -2, 68, -2, 69,-10,  0, 69, -2,
              0,-13, 69, -2, 68, -2, 72, -9,  0, 69, -3,  0, 69,-41,  0, -2,
             68,  0, 68, -2, 72, -4, 68, 69, -3, 72, 69, -3, 72, 68, 70,  0,
             -4, 69, -3,  0, 68,-10,  0, -2, 68, -2, 69,-26,  0, 69, 68, -2,
             72, 69,-18,  0, -3, 69,-24,  0, 69, -9,  0, 69,  0, -2, 69, -7,
              0, -3, 69, 68, -4, 69, 68, -2, 72, -3, 68, 69, 68, 69, -7, 72,
             68, -6, 69,-10,  0, -2, 69, -2, 72,  0,-12, 69,-48,  0, 68,  0,

             73, -7, 68, -4, 69, -8,  0, -8, 68,-13,  0,-37, 69, -2,  0, 69,
              0, 69, -5,  0, 69,-24,  0, 69,  0, 69,-10,  0, 68,  0, 73, -9,
             68,  0, -2, 69, -5,  0, 69,  0, 69, -6, 68, -2, 69,-10,  0, -2,
             69, -4,  0,-32, 69,-24,  0, -2, 68,-27,  0, 68,  0, 68,  0, 68,
             -4,  0, -2, 72, -8,  0, 69,-36,  0, -4, 69,-14, 68, 72, -5, 68,
              0, -2, 68, -5,  0,-11, 68, 69,-36, 68, 69, -8,  0, 68, -6,  0,
             69,-13,  0,-37, 69,-43,  0, -2, 69, -4, 68, 72, -6, 68, 69, -2,
             68, -2, 72, -2, 68,-23,  0, -2, 72, -2, 68, -4,  0, -3, 68,  0,
             -3, 69, -2,  0, -7, 69, -3,  0, -4, 68,-13,  0, 68, 69, 72, -2,
             68, -6, 69, 68,  0, 69,-10,  0, -3, 69, 68,-40,  0, 69,  0, -5,
             69,  0, -2, 69,-48,  0,-96, 74,-72, 75,-88, 76,-73,  0, 69, -4,
              0, -2, 69, -7,  0, 69,  0, 69, -4,  0, -2, 69,-41,  0, 69, -4,
              0, -2, 69,-33,  0, 69, -4,  0, -2, 69, -7,  0, 69,  0, 69, -4,
              0, -2, 69,-15,  0, 69,-57,  0, 69, -4,  0, -2, 69,-67,  0, -2,
             69, -3, 68,-29,  0, -3, 69,-26,  0, -6, 69,-86,  0, -2, 69, -6,
              0, -2, 69,-157,  0, -3, 69,-89,  0, -7, 69,-18,  0, -3, 68, 72,

             -9, 69,-19,  0, -2, 68, 72, -2,  0, -9, 69,-18,  0, -2, 68,-12,
             69,-13,  0, 69, -3,  0, 69, -2, 68,-12, 69,-52,  0, -2, 68, 72,
             -7, 68, -8, 72, 68, -2, 72,-11, 68, -9,  0, 68, -2, 69,-10,  0,
             -6, 69,-10,  0, -6, 69,-11,  0, -3, 68, 77, 68,-10,  0, -6, 69,
            -89,  0, -7, 69, -5,  0, -2, 68,-34,  0, 68,  0, -5, 69,-70,  0,
            -10, 69,-31,  0, 69, -3, 68, -4, 72, -2, 68, -3, 72, -4, 69, -2,
             72, 68, -6, 72, -3, 68, -4, 69,  0, -3, 69,-42,  0, -2, 69, -5,
              0,-11, 69,-44,  0, -4, 69,-26,  0, -6, 69,-11,  0, -3, 69,-57,
              0, -2, 68, -2, 72, 68, -2, 69,-55,  0, 72, 68, 72, -7, 68, 69,
             68, 69, 68, -2, 69, -8, 68, -6, 72,-10, 68, -2, 69, 68,-10,  0,
             -6, 69,-10,  0, -6, 69,-14,  0, -2, 69,-31, 68,-49, 69, -4, 68,
             72,-47,  0, -7, 68, 72, 68, -5, 72, 68, -2, 72, -8,  0, -3, 69,
            -27,  0, -9, 68,-11,  0, 69, -2, 68, 72,-30,  0, 72, -4, 68, -2,
             72, -2, 68, 72, -3, 68,-56,  0, 68, 72, -2, 68, -3, 72, 68, 72,
             -3, 68, -2, 72, -8, 69,-40,  0, -8, 72, -8, 68, -2, 72, -2, 68,
             -3, 69,-15,  0, -3, 69,-60,  0, -7, 69,-43,  0, -2, 69,-11,  0,

             -8, 69, -3, 68,  0,-13, 68, 72, -7, 68, -4,  0, 68, -6,  0, 68,
             -2,  0, 72, -2, 68,  0, -5, 69,-192,  0,-64, 68,-22,  0, -2, 69,
             -6,  0, -2, 69,-38,  0, -2, 69, -6,  0, -2, 69, -8,  0, 69,  0,
             69,  0, 69,  0, 69,-31,  0, -2, 69,-53,  0, 69,-15,  0, 69,-14,
              0, -2, 69, -6,  0, 69,-19,  0, -2, 69, -3,  0, 69, -9,  0, 69,
            -11,  0, 78, 79, 80, 81, 82,-24,  0, 83, 84, 85, 86, 87, 88, 89,
            -13,  0, 66,-12,  0, 66,-22,  0, 90, 91, 92, 93, 94, 95, 96, 97,
             98, 99,100,101,102,103,104,105, -2,  0, -2, 69,-27,  0, 69,-13,
              0, -3, 69,-33,  0,-15, 69,-33, 68,-15, 69,-34,  0, 66,-22,  0,
             66,-82,  0, -4, 69, -4,  0, -6, 66,-15,  0, -2, 66,-111,  0, -2,
            106,-12,  0, 66, -2,107,-93,  0, 66,-70,  0, 66,-25,  0, -4,106,
             -3, 66,106, -2, 66,106, -4,  0, -3, 66,-44,  0,-25, 69,-11,  0,
            -21, 69,-98,  0, 66,-231,  0, -2, 66,-10,  0, 66, -9,  0, 66,-58,
              0, -2, 66, -2,106,  0, -6, 66,  0,-12, 66,  0, -2,106,-50, 66,
            -12,106,-43, 66,106, -6, 66,-10,  0, -3, 66,106,-13, 66,106, -8,
             66, -2,106,-17, 66, -2,106, -5, 66, -2,106, -8, 66,106, -5, 66,

            106,-21, 66,106, -7, 66, -2,106, 66,106, -4, 66,106, -2, 66,106,
             -7, 66,106, -2,  0, -2, 66, -2,106, -7, 66,  0, 66,  0, 66, -6,
              0, 66, -3,  0, 66, -6,  0,106,-10,  0, -2, 66,-15,  0, 66, -2,
              0, 66, -4,  0,106,  0,106, -4,  0, -3,106,  0,106,-11,  0, -5,
             66,-45,  0, -3,106, -9,  0, 66,-14,  0,106,-14,  0,106,-116,  0,
             -2, 66,-207,  0, -3, 66,-19,  0, -2,106,-51,  0,106, -4,  0,106,
            -30,  0, -2, 69,-32,  0, 69,-344,  0, -3, 68, -2,  0, -5, 69,-45,
              0, 69,  0, -5, 69,  0, -2, 69,-56,  0, -7, 69, -2,  0,-14, 69,
             68,-23,  0, -9, 69, -7,  0, 69, -7,  0, 69, -7,  0, 69, -7,  0,
             69, -7,  0, 69, -7,  0, 69, -7,  0, 69, -7,  0, 69,-32, 68,-94,
              0,-34, 69,-26,107, 69,-89,107,-12, 69,-214,107,-26, 69,-12,107,
             -4, 69,-42,107, -6, 68,106,-12,107,106,107,  0, 69,-86,107, -2,
             69, -2, 68,-101,107, -5, 69,-43,107, 69,-94,107, 69,-84,107,-12,
             69,-47,107, 69,-40,107, -8,  0,-71,107,106,107,106,-550,107,-64,
              0,-141,107, -3, 69,-55,107, -9, 69,-92,  0,-20, 69,-47,  0, -4,
             68,  0,-10, 68,-32,  0, -2, 68,-80,  0, -2, 68, -6,  0, -8, 69,

            -203,  0, -5, 69, -2,  0, 69,  0, 69, -5,  0,-24, 69,-16,  0, 68,
             -3,  0, 68, -4,  0, 68,-23,  0, -2, 72, -2, 68, 72, -4,  0, 68,
             -3, 69,-10,  0, -6, 69,-56,  0, -8, 69, -2, 72,-50,  0,-16, 72,
             -2, 68, -8, 69,-12,  0, -6, 69,-18, 68,-13,  0, 68,-38,  0, -8,
             68,-25,  0,-11, 68, -2, 72,-11, 69,  0,-29, 74, -3, 69, -3, 68,
             72,-47,  0, 68, -2, 72, -4, 68, -2, 72, -2, 68, -3, 72,-13,  0,
             69,-11,  0, -4, 69, -7,  0, 68,-25,  0, 69,-41,  0, -6, 68, -2,
             72, -2, 68, -2, 72, -2, 68, -9, 69, -3,  0, 68, -8,  0, 68, 72,
             -2, 69,-10,  0, -2, 69,-31,  0, 69, 68, 69,-50,  0, 68,  0, -3,
             68, -2,  0, -2, 68, -5,  0, -2, 68,  0, 68,  0,-24, 69,-16,  0,
             72, -2, 68, -2, 72, -5,  0, 72, 68,-10, 69, -6,  0, -2, 69, -6,
              0, -2, 69, -6,  0, -9, 69, -7,  0, 69, -7,  0, 69,-60,  0, -4,
             69,-115,  0, -2, 72, 68, -2, 72, 68, -2, 72,  0, 72, 68, -2, 69,
            -10,  0, -6, 69,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,
            -27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,

            109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,
            -27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,
            109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,
            -27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,
            109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,
            -27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,
            109,108,-51,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,-12, 69,-23, 75, -4, 69,-49, 76,-260, 69,-110,107, -2,
             69,-106,107,-38, 69, -7,  0,-12, 69, -5,  0, -5, 69,  0, 68,-24,
              0, 69, -5,  0, 69,  0, 69, -2,  0, 69, -2,  0, 69,-125,  0,-16,
             69,-189,  0, -2, 69,-54,  0, -7, 69,  0,-32, 69,-16,  0,-16, 68,
            -10,107, -6, 69,-16, 68,-35,107, 69,-19,107, 69, -4,107, -4, 69,
             -5,  0, 69,-135,  0, -2, 69,110, 69,-96,107,-61,  0, -2,111,-31,

              0, -3, 69, -6,  0, -2, 69, -6,  0, -2, 69, -6,  0, -2, 69, -3,
              0, -3, 69, -7,107, 69, -7,  0, 69, -9, 95,112,113,114, -2,  0,
             -2, 69,-12,  0, 69,-26,  0, 69,-19,  0, 69, -2,  0, 69,-15,  0,
             -2, 69,-14,  0,-34, 69,-123,  0, -5, 69, -3,  0, -4, 69,-45,  0,
             -3, 69,-88,  0, 69,-13,  0, -3, 69,  0,-47, 69,-45,  0, 68,-130,
             69,-29,  0, -3, 69,-49,  0,-15, 69, 68,-27,  0, -4, 69,-36,  0,
             -9, 69,-30,  0, -5, 69,-38,  0, -5, 68, -5, 69,-30,  0, 69,-37,
              0, -4, 69,-14,  0,-42, 69,-158,  0, -2, 69,-10,  0, -6, 69,-36,
              0, -4, 69,-36,  0, -4, 69,-40,  0, -8, 69,-52,  0,-11, 69,-12,
              0, 69,-15,  0, 69, -7,  0, 69, -2,  0, 69,-11,  0, 69,-15,  0,
             69, -7,  0, 69, -2,  0,-67, 69,-55,  0, -9, 69,-22,  0,-10, 69,
             -8,  0,-24, 69, -6,  0, 69,-42,  0, 69, -9,  0,-69, 69, -6,  0,
             -2, 69,  0, 69,-44,  0, 69, -2,  0, -3, 69,  0, -2, 69,-23,  0,
             69,-72,  0, -8, 69, -9,  0,-48, 69,-19,  0, 69, -2,  0, -5, 69,
            -33,  0, -3, 69,-27,  0, -5, 69,  0,-64, 69,-56,  0, -4, 69,-20,
              0, -2, 69,-47,  0, -3, 68, 69, -2, 68, -5, 69, -4, 68, -4,  0,

             69, -3,  0, 69,-29,  0, -2, 69, -3, 68, -4, 69, 68, -9,  0, -7,
             69, -9,  0, -7, 69,-64,  0,-32, 69,-37,  0, -2, 68, -4, 69,-12,
              0, -9, 69,-54,  0, -3, 69,-29,  0, -2, 69,-27,  0, -5, 69,-26,
              0, -7, 69, -4,  0,-12, 69, -7,  0,-80, 69,-73,  0,-55, 69,-51,
              0,-13, 69,-51,  0, -7, 69,-42,  0, -4, 68, -8, 69,-10,  0,-294,
             69,-31,  0, 69,-42,  0, 69, -2, 68,  0, -2, 69, -2,  0,-78, 69,
            -40,  0, -8, 69,-22,  0,-11, 68, -9,  0,-22, 69,-18,  0, -4, 68,
             -4,  0,-38, 69,-28,  0,-20, 69,-23,  0, -9, 69, 72, 68, 72,-53,
              0,-15, 68, -7,  0, -4, 69,-30,  0, 68, -2,  0, -2, 68,  0, -9,
             69, -3, 68, 72,-45,  0, -3, 72, -4, 68, -2, 72, -2, 68, -2,  0,
             70, -4,  0, 68,-10, 69, 70, -2, 69,-25,  0, -7, 69,-10,  0, -6,
             69, -3, 68,-36,  0, -5, 68, 72, -8, 68, 69,-15,  0, -2, 72,  0,
             -8, 69,-35,  0, 68, -3,  0, -9, 69, -2, 68, 72,-48,  0, -3, 72,
             -9, 68, -2, 72,  0, -2, 70, -5,  0, -4, 68,  0, 72, 68,-16,  0,
             69,-20,  0,-11, 69,-18,  0, 69,-25,  0, -3, 72, -3, 68, -2, 72,
             68, 72, -2, 68, -6,  0, 68,-65, 69, -7,  0, 69,  0, 69, -4,  0,

             69,-15,  0, 69,-11,  0, -6, 69,-47,  0, 68, -3, 72, -8, 68, -5,
             69,-10,  0, -6, 69, -2, 68, -2, 72, 69, -8,  0, -2, 69, -2,  0,
             -2, 69,-22,  0, 69, -7,  0, 69, -2,  0, 69, -5,  0, 69, -2, 68,
              0, 68, 72, 68, -4, 72, -2, 69, -2, 72, -2, 69, -3, 72, -2, 69,
              0, -6, 69, 68, -5, 69, -5,  0, -2, 72, -2, 69, -7, 68, -3, 69,
             -5, 68,-139, 69,-53,  0, -3, 72, -8, 68, -2, 72, -3, 68, 72, 68,
            -21,  0, 69,  0, 68, -3,  0,-30, 69,-48,  0, 68, -2, 72, -6, 68,
             72, 68, -2, 72, 68, 72, -2, 68, 72, -2, 68, -4,  0, -8, 69,-10,
              0,-166, 69,-47,  0, 68, -2, 72, -4, 68, -2, 69, -4, 72, -2, 68,
             72, -2, 68,-27,  0, -2, 68,-34, 69,-48,  0, -3, 72, -8, 68, -2,
             72, 68, 72, -2, 68, -4,  0,-11, 69,-10,  0, -6, 69,-13,  0,-19,
             69,-43,  0, 68, 72, 68, -2, 72, -6, 68, 72, 68, -2,  0, -6, 69,
            -10,  0,-54, 69,-27,  0, -2, 69, -3, 68, -2, 69, -4, 68, 72, -5,
             68, -4, 69,-23,  0,-185, 69,-44,  0, -3, 72, -9, 68, 72, -2, 68,
              0,-100, 69,-83,  0,-12, 69, -8,  0, -2, 69,  0, -2, 69, -8,  0,
             69, -2,  0, 69,-24,  0, 68, -5, 72, 69, -2, 72, -2, 69, -2, 68,

             72, 68, 70, 72, 70, 72, 68, -3,  0, -9, 69,-10,  0,-70, 69, -8,
              0, -2, 69,-39,  0, -3, 72, -4, 68, -2, 69, -2, 68, -4, 72, 68,
             -3,  0, 72,-27, 69,  0,-10, 68,-40,  0, -6, 68, 72, 70, -4, 68,
             -8,  0, 68, -8, 69,  0, -6, 68, -2, 72, -3, 68,-40,  0, -6, 70,
            -13, 68, 72, -2, 68, -9,  0,-13, 69,-73,  0, -7, 69, -9,  0, 69,
            -37,  0, 72, -7, 68, 69, -6, 68, 72, 68, -6,  0,-10, 69,-29,  0,
             -3, 69,-32,  0, -2, 69,-22, 68, 69, 72, -7, 68, 72, -2, 68, 72,
             -2, 68,-73, 69, -7,  0, 69, -2,  0, 69,-38,  0, -6, 68, -3, 69,
             68, 69, -2, 68, 69, -7, 68, 70, 68, -8, 69,-10,  0, -6, 69, -6,
              0, 69, -2,  0, 69,-32,  0, -5, 72, 69, -2, 68, 69, -2, 72, 68,
             72, 68,  0, -7, 69,-10,  0,-310, 69,-19,  0, -2, 68, -2, 72, -2,
              0,-183, 69,  0,-15, 69,-50,  0,-13, 69,-155,  0,-102, 69,-111,  0,
             69, -5,  0,-11, 69,-196,  0,-332, 69,-99,  0,-13, 69,-47,  0, 69,
            115,116,117,118,119,120,121,122,123,-199, 69,-71,  0,-185, 69,-57,
              0, -7, 69,-31,  0, 69,-10,  0, -4, 69,-81,  0, 69,-10,  0, -6,
             69,-30,  0, -2, 69, -5, 68,  0,-10, 69,-48,  0, -7, 68,-15,  0,

            -10, 69,-10,  0, 69, -7,  0, 69,-21,  0, -5, 69,-19,  0,-176, 69,
            -91,  0,-101, 69,-75,  0, -4, 69, 68,  0,-55, 72, -7, 69, -4, 68,
            -13,  0,-64, 69, -4,107, 68,-11, 69, -2, 72,-14, 69,-248,107, -8,
             69,-214,107,-42, 69, -9,107,-487, 69, -4,107, 69, -7,107, 69, -2,
            107, 69,-35,107,-45, 69, -3,107,-17, 69, -4,107, -8, 69,-396,107,
             -4, 69,-107,  0, -5, 69,-13,  0, -3, 69, -9,  0, -7, 69,-10,  0,
             -2, 69,  0, -2, 68,  0,124,125,126,127,-92, 69,-46, 68, -2, 69,
            -23, 68, -9, 69,-116,  0,-60, 69,-246,  0,-10, 69,-39,  0, -2, 69,
            -60,  0, 68, 72, -3, 68, -3,  0, 72, -5, 68,128,129,130,131,132,
            133,134,135, -8, 68, -2,  0, -7, 68,-30,  0, -4, 68,-61,  0,-21,
             69,-66,  0, -3, 68,  0,-154, 69,-20,  0,-12, 69,-87,  0, -9, 69,
            -25,  0,-135, 69,-85,  0, 69,-71,  0, 69, -2,  0, -2, 69,  0, -2,
             69, -2,  0, -2, 69, -4,  0, 69,-12,  0, 69,  0, 69, -7,  0, 69,
            -65,  0, 69, -4,  0, -2, 69, -8,  0, 69, -7,  0, 69,-28,  0, 69,
             -4,  0, 69, -5,  0, 69,  0, -3, 69, -7,  0, 69,-340,  0, -2, 69,
            -292,  0, -2, 69,-50,  0,-55, 68, -4,  0,-50, 68, -8,  0, 68,-14,

              0, 68, -7,  0,-15, 69, -5, 68, 69,-15, 68,-80, 69,-31,  0,-225,
             69, -7, 68, 69,-17, 68, -2, 69, -7, 68, 69, -2, 68, 69, -5, 68,
            -213, 69,-45,  0, -3, 69, -7, 68, -7,  0, -2, 69,-10,  0, -4, 69,
             -2,  0,-320, 69,-30,  0, 68,-17, 69,-44,  0, -4, 68,-10,  0, -5,
             69,  0,-224, 69, -7,  0, 69, -4,  0, 69, -2,  0, 69,-15,  0, 69,
            -197,  0, -2, 69, -9,  0, -7, 68,-41, 69,-68,  0, -7, 68,  0, -4,
             69,-10,  0, -4, 69, -2,  0,-273, 69,-68,  0,-76, 69,-61,  0,-194,
             69, -4,  0, 69,-27,  0, 69, -2,  0, 69,  0, -2, 69,  0, 69,-10,
              0, 69, -4,  0, 69,  0, 69,  0, -6, 69,  0, -4, 69,  0, 69,  0,
             69,  0, 69, -3,  0, 69, -2,  0, 69,  0, -2, 69,  0, 69,  0, 69,
              0, 69,  0, 69,  0, 69, -2,  0, 69,  0, -2, 69, -4,  0, 69, -7,
              0, 69, -4,  0, 69, -4,  0, 69,  0, 69,-10,  0, 69,-17,  0, -5,
             69, -3,  0, 69, -5,  0, 69,-17,  0,-52, 69, -2,  0,-14, 69, -4,
             66,106,-39, 66, -4,136,-100, 66,-12,136,-15, 66, -2,136,-15, 66,
            136,-14, 66,106,136,-37, 66,-10,136,-13,  0, -3, 66,-31,  0, 66,
            -60,  0, -6, 66,-12,  0, -2, 66,-14,  0,106, -2,  0,-10,106,-18,

              0, 66,-56,136,-26,137,107, -2,106,-13,136,-10,107,106,-20,107,
            106, -2,107, -9,106,107, -4,136, -9,107, -7,136, -2,106,-14,136,
             -6,106,-154,136,-33,106,-12, 66, -9,106, 66,-70,106, 66,-22,106,
            -12, 66,-43,106, -4, 66, -5,106,-12, 66,-17,106, -3, 66,106, -3,
             66, -3,106, -5,138,-63,106, 66,106, 66,-187,106, -2, 66,-63,106,
             -8,  0, -5, 66, -4,106, 66,-24,106,-18, 66,106,-26, 66, -2,106,
            -13, 66,106,-86, 66,-85,106,-48,  0,-70,106, -6, 66,106, -3, 66,
             -3,106, -2, 66, -3,106, -5,136, -3,106,-11, 66, -2,106, -3,136,
             -4, 66, -9,106, -3,136,-116,  0,-12,136,-85,  0, -4, 66, -7,136,
            -12,106, -4,136,106,-15,136,-12,  0, -4,136,-56,  0, -8,136,-10,
              0, -6,136,-40,  0, -8,136,-30,  0, -2,136, -2, 66,-78,136,-12,
              0,-47,106,  0,-10,106,  0,-185,106,-84, 66,-12,136,-14, 66, -2,
            136, -5,106, -3,136, -5,106, -3,136, -7,106, -9,136,-29,106, -3,
            136,-11,106, -5,136, -6,106,-10,136,-10,106, -6,136, -8,106, -8,
            136, -7,106, -9,136,-147,  0, 69,-55,  0,-37, 69,-10,  0, -6, 69,
            -510,136, -2, 69,-224,107,-32, 69,-57,107, -7, 69,-222,107, -2, 69,

            -386,107,-14, 69,-305,107,-31, 69,-30,107,-226, 69,-75,107,-181, 69,
            -32, 95,-96, 68,-128, 95,-240, 68,-272, 95,-254,  0, -2, 69
        };

        static constexpr unidata ucspec[] =
        {
            { widths::slim, gbreak::ANY , cntrls::NON_CONTROL            },  //   0
            { widths::zero, gbreak::CTRL, cntrls::NON_CONTROL            },  //   1
            { widths::zero, gbreak::CTRL, cntrls::SOH                    },  //   2
            { widths::zero, gbreak::CTRL, cntrls::STX                    },  //   3
            { widths::zero, gbreak::CTRL, cntrls::ETX                    },  //   4
            { widths::zero, gbreak::CTRL, cntrls::EOT                    },  //   5
            { widths::zero, gbreak::CTRL, cntrls::ENQ                    },  //   6
            { widths::zero, gbreak::CTRL, cntrls::ACK                    },  //   7
            { widths::zero, gbreak::CTRL, cntrls::BEL                    },  //   8
            { widths::zero, gbreak::CTRL, cntrls::BS                     },  //   9
            { widths::zero, gbreak::CTRL, cntrls::TAB                    },  //  10
            { widths::zero, gbreak::LF  , cntrls::EOL                    },  //  11
            { widths::zero, gbreak::CTRL, cntrls::VT                     },  //  12
            { widths::zero, gbreak::CTRL, cntrls::FF                     },  //  13
            { widths::zero, gbreak::CR  , cntrls::CR                     },  //  14
            { widths::zero, gbreak::CTRL, cntrls::SO                     },  //  15
            { widths::zero, gbreak::CTRL, cntrls::SI                     },  //  16
            { widths::zero, gbreak::CTRL, cntrls::DLE                    },  //  17
            { widths::zero, gbreak::CTRL, cntrls::DC1                    },  //  18
            { widths::zero, gbreak::CTRL, cntrls::DC2                    },  //  19
            { widths::zero, gbreak::CTRL, cntrls::DC3                    },  //  20
            { widths::zero, gbreak::CTRL, cntrls::DC4                    },  //  21
            { widths::zero, gbreak::CTRL, cntrls::NAK                    },  //  22
            { widths::zero, gbreak::CTRL, cntrls::SYN                    },  //  23
            { widths::zero, gbreak::CTRL, cntrls::ETB                    },  //  24
            { widths::zero, gbreak::CTRL, cntrls::CAN                    },  //  25
            { widths::zero, gbreak::CTRL, cntrls::EOM                    },  //  26
            { widths::zero, gbreak::CTRL, cntrls::SUB                    },  //  27
            { widths::zero, gbreak::CTRL, cntrls::ESC                    },  //  28
            { widths::zero, gbreak::CTRL, cntrls::FS                     },  //  29
            { widths::zero, gbreak::CTRL, cntrls::GS                     },  //  30
            { widths::zero, gbreak::CTRL, cntrls::RS                     },  //  31
            { widths::zero, gbreak::CTRL, cntrls::US                     },  //  32
            { widths::zero, gbreak::CTRL, cntrls::DEL                    },  //  33
            { widths::zero, gbreak::CTRL, cntrls::PAD                    },  //  34
            { widths::zero, gbreak::CTRL, cntrls::HOP                    },  //  35
            { widths::zero, gbreak::CTRL, cntrls::BPH                    },  //  36
            { widths::zero, gbreak::CTRL, cntrls::NBH                    },  //  37
            { widths::zero, gbreak::CTRL, cntrls::IND                    },  //  38
            { widths::zero, gbreak::CTRL, cntrls::NEL                    },  //  39
            { widths::zero, gbreak::CTRL, cntrls::SSA                    },  //  40
            { widths::zero, gbreak::CTRL, cntrls::ESA                    },  //  41
            { widths::zero, gbreak::CTRL, cntrls::HTS                    },  //  42
            { widths::zero, gbreak::CTRL, cntrls::HTJ                    },  //  43
            { widths::zero, gbreak::CTRL, cntrls::VTS                    },  //  44
            { widths::zero, gbreak::CTRL, cntrls::PLD                    },  //  45
            { widths::zero, gbreak::CTRL, cntrls::PLU                    },  //  46
            { widths::zero, gbreak::CTRL, cntrls::RI                     },  //  47
            { widths::zero, gbreak::CTRL, cntrls::SS2                    },  //  48
            { widths::zero, gbreak::CTRL, cntrls::SS3                    },  //  49
            { widths::zero, gbreak::CTRL, cntrls::DCS                    },  //  50
            { widths::zero, gbreak::CTRL, cntrls::PU1                    },  //  51
            { widths::zero, gbreak::CTRL, cntrls::PU2                    },  //  52
            { widths::zero, gbreak::CTRL, cntrls::STS                    },  //  53
            { widths::zero, gbreak::CTRL, cntrls::CCH                    },  //  54
            { widths::zero, gbreak::CTRL, cntrls::MW                     },  //  55
            { widths::zero, gbreak::CTRL, cntrls::SPA                    },  //  56
            { widths::zero, gbreak::CTRL, cntrls::EPA                    },  //  57
            { widths::zero, gbreak::CTRL, cntrls::SOS                    },  //  58
            { widths::zero, gbreak::CTRL, cntrls::SGC                    },  //  59
            { widths::zero, gbreak::CTRL, cntrls::SCI                    },  //  60
            { widths::zero, gbreak::CTRL, cntrls::CSI                    },  //  61
            { widths::zero, gbreak::CTRL, cntrls::ST                     },  //  62
            { widths::zero, gbreak::CTRL, cntrls::OSC                    },  //  63
            { widths::zero, gbreak::CTRL, cntrls::PM                     },  //  64
            { widths::zero, gbreak::CTRL, cntrls::APC                    },  //  65
            { widths::slim, gbreak::EP  , cntrls::NON_CONTROL            },  //  66
            { widths::zero, gbreak::CTRL, cntrls::SHY                    },  //  67
            { widths::zero, gbreak::EXT , cntrls::NON_CONTROL            },  //  68
            { widths::zero, gbreak::ANY , cntrls::NON_CONTROL            },  //  69
            { widths::slim, gbreak::PREP, cntrls::NON_CONTROL            },  //  70
            { widths::zero, gbreak::CTRL, cntrls::ALM                    },  //  71
            { widths::zero, gbreak::SM  , cntrls::NON_CONTROL            },  //  72
            { widths::slim, gbreak::SM  , cntrls::NON_CONTROL            },  //  73
            { widths::wide, gbreak::L   , cntrls::NON_CONTROL            },  //  74
            { widths::slim, gbreak::V   , cntrls::NON_CONTROL            },  //  75
            { widths::slim, gbreak::T   , cntrls::NON_CONTROL            },  //  76
            { widths::zero, gbreak::CTRL, cntrls::MVS                    },  //  77
            { widths::zero, gbreak::CTRL, cntrls::ZWSP                   },  //  78
            { widths::zero, gbreak::EXT , cntrls::ZWNJ                   },  //  79
            { widths::zero, gbreak::ZWJ , cntrls::NON_CONTROL            },  //  80
            { widths::zero, gbreak::CTRL, cntrls::LRM                    },  //  81
            { widths::zero, gbreak::CTRL, cntrls::RLM                    },  //  82
            { widths::zero, gbreak::CTRL, cntrls::LINE_SEPARATOR         },  //  83
            { widths::zero, gbreak::CTRL, cntrls::PARAGRAPH_SEPARATOR    },  //  84
            { widths::zero, gbreak::CTRL, cntrls::LRE                    },  //  85
            { widths::zero, gbreak::CTRL, cntrls::RLE                    },  //  86
            { widths::zero, gbreak::CTRL, cntrls::PDF                    },  //  87
            { widths::zero, gbreak::CTRL, cntrls::LRO                    },  //  88
            { widths::zero, gbreak::CTRL, cntrls::RLO                    },  //  89
            { widths::zero, gbreak::CTRL, cntrls::WJ                     },  //  90
            { widths::zero, gbreak::CTRL, cntrls::FUNCTION_APPLICATION   },  //  91
            { widths::zero, gbreak::CTRL, cntrls::INVISIBLE_TIMES        },  //  92
            { widths::zero, gbreak::CTRL, cntrls::INVISIBLE_SEPARATOR    },  //  93
            { widths::zero, gbreak::CTRL, cntrls::INVISIBLE_PLUS         },  //  94
            { widths::zero, gbreak::CTRL, cntrls::NON_CONTROL            },  //  95
            { widths::zero, gbreak::CTRL, cntrls::LRI                    },  //  96
            { widths::zero, gbreak::CTRL, cntrls::RLI                    },  //  97
            { widths::zero, gbreak::CTRL, cntrls::FSI                    },  //  98
            { widths::zero, gbreak::CTRL, cntrls::PDI                    },  //  99
            { widths::zero, gbreak::CTRL, cntrls::INHIBIT_SYMMETRIC_SWAPPING },  // 100
            { widths::zero, gbreak::CTRL, cntrls::ACTIVATE_SYMMETRIC_SWAPPING },  // 101
            { widths::zero, gbreak::CTRL, cntrls::INHIBIT_ARABIC_FORM_SHAPING },  // 102
            { widths::zero, gbreak::CTRL, cntrls::ACTIVATE_ARABIC_FORM_SHAPING },  // 103
            { widths::zero, gbreak::CTRL, cntrls::NATIONAL_DIGIT_SHAPES  },  // 104
            { widths::zero, gbreak::CTRL, cntrls::NOMINAL_DIGIT_SHAPES   },  // 105
            { widths::wide, gbreak::EP  , cntrls::NON_CONTROL            },  // 106
            { widths::wide, gbreak::ANY , cntrls::NON_CONTROL            },  // 107
            { widths::wide, gbreak::LV  , cntrls::NON_CONTROL            },  // 108
            { widths::wide, gbreak::LVT , cntrls::NON_CONTROL            },  // 109
            { widths::zero, gbreak::CTRL, cntrls::ZWNBSP                 },  // 110
            { widths::slim, gbreak::EXT , cntrls::NON_CONTROL            },  // 111
            { widths::zero, gbreak::CTRL, cntrls::INTERLINEAR_ANNOTATION_ANCHOR },  // 112
            { widths::zero, gbreak::CTRL, cntrls::INTERLINEAR_ANNOTATION_SEPARATOR },  // 113
            { widths::zero, gbreak::CTRL, cntrls::INTERLINEAR_ANNOTATION_TERMINATOR },  // 114
            { widths::zero, gbreak::CTRL, cntrls::EGYPTIAN_HIEROGLYPH_VERTICAL_JOINER },  // 115
            { widths::zero, gbreak::CTRL, cntrls::EGYPTIAN_HIEROGLYPH_HORIZONTAL_JOINER },  // 116
            { widths::zero, gbreak::CTRL, cntrls::EGYPTIAN_HIEROGLYPH_INSERT_AT_TOP_START },  // 117
            { widths::zero, gbreak::CTRL, cntrls::EGYPTIAN_HIEROGLYPH_INSERT_AT_BOTTOM_START },  // 118
            { widths::zero, gbreak::CTRL, cntrls::EGYPTIAN_HIEROGLYPH_INSERT_AT_TOP_END },  // 119
            { widths::zero, gbreak::CTRL, cntrls::EGYPTIAN_HIEROGLYPH_INSERT_AT_BOTTOM_END },  // 120
            { widths::zero, gbreak::CTRL, cntrls::EGYPTIAN_HIEROGLYPH_OVERLAY_MIDDLE },  // 121
            { widths::zero, gbreak::CTRL, cntrls::EGYPTIAN_HIEROGLYPH_BEGIN_SEGMENT },  // 122
            { widths::zero, gbreak::CTRL, cntrls::EGYPTIAN_HIEROGLYPH_END_SEGMENT },  // 123
            { widths::zero, gbreak::CTRL, cntrls::SHORTHAND_FORMAT_LETTER_OVERLAP },  // 124
            { widths::zero, gbreak::CTRL, cntrls::SHORTHAND_FORMAT_CONTINUING_OVERLAP },  // 125
            { widths::zero, gbreak::CTRL, cntrls::SHORTHAND_FORMAT_DOWN_STEP },  // 126
            { widths::zero, gbreak::CTRL, cntrls::SHORTHAND_FORMAT_UP_STEP },  // 127
            { widths::zero, gbreak::CTRL, cntrls::MUSICAL_SYMBOL_BEGIN_BEAM },  // 128
            { widths::zero, gbreak::CTRL, cntrls::MUSICAL_SYMBOL_END_BEAM },  // 129
            { widths::zero, gbreak::CTRL, cntrls::MUSICAL_SYMBOL_BEGIN_TIE },  // 130
            { widths::zero, gbreak::CTRL, cntrls::MUSICAL_SYMBOL_END_TIE },  // 131
            { widths::zero, gbreak::CTRL, cntrls::MUSICAL_SYMBOL_BEGIN_SLUR },  // 132
            { widths::zero, gbreak::CTRL, cntrls::MUSICAL_SYMBOL_END_SLUR },  // 133
            { widths::zero, gbreak::CTRL, cntrls::MUSICAL_SYMBOL_BEGIN_PHRASE },  // 134
            { widths::zero, gbreak::CTRL, cntrls::MUSICAL_SYMBOL_END_PHRASE },  // 135
            { widths::zero, gbreak::EP  , cntrls::NON_CONTROL            },  // 136
            { widths::slim, gbreak::RI  , cntrls::NON_CONTROL            },  // 137
            { widths::wide, gbreak::EXT , cntrls::NON_CONTROL            },  // 138
        };
    };

    template<class T, class D>
    auto unpack(D const& pack, size_t size)
    {
        auto data = std::vector<T>{};
        data.reserve(size);
        auto iter = pack;
        auto tail = pack + std::size(pack);
        while (iter < tail)
        {
            auto n = *iter++;
            if (n < 0) data.insert(data.end(), -n, *iter++);
            else       data.push_back(n);
        }
        return data;
    }

    inline unidata const& select(uint32_t cp)
    {
        using blocks_t = uint16_t;
        using offset_t = uint8_t;
        static std::vector<offset_t> offset = unpack<offset_t>(base::offset_pack, base::offset_size);
        static std::vector<blocks_t> blocks = unpack<blocks_t>(base::blocks_pack, base::blocks_size);

        return cp > 0x10FFFF
            ? base::ucspec[0]
            : base::ucspec[offset[blocks[cp >> 8] + (cp & 0xFF)]];
    }
}

#endif // NETXS_UNIDATA_HPP