/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.store;

import java.io.EOFException;
import java.io.IOException;
import java.lang.foreign.Arena;
import java.lang.foreign.MemorySegment;
import java.lang.foreign.ValueLayout;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Objects;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.RandomAccessInput;
import org.apache.lucene.util.ArrayUtil;

/*
 * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
 */
abstract class MemorySegmentIndexInput
extends IndexInput
implements RandomAccessInput {
    static final ValueLayout.OfByte LAYOUT_BYTE = ValueLayout.JAVA_BYTE;
    static final ValueLayout.OfShort LAYOUT_LE_SHORT = ValueLayout.JAVA_SHORT_UNALIGNED.withOrder(ByteOrder.LITTLE_ENDIAN);
    static final ValueLayout.OfInt LAYOUT_LE_INT = ValueLayout.JAVA_INT_UNALIGNED.withOrder(ByteOrder.LITTLE_ENDIAN);
    static final ValueLayout.OfLong LAYOUT_LE_LONG = ValueLayout.JAVA_LONG_UNALIGNED.withOrder(ByteOrder.LITTLE_ENDIAN);
    static final ValueLayout.OfFloat LAYOUT_LE_FLOAT = ValueLayout.JAVA_FLOAT_UNALIGNED.withOrder(ByteOrder.LITTLE_ENDIAN);
    final long length;
    final long chunkSizeMask;
    final int chunkSizePower;
    final Arena arena;
    final MemorySegment[] segments;
    int curSegmentIndex = -1;
    MemorySegment curSegment;
    long curPosition;

    public static MemorySegmentIndexInput newInstance(String resourceDescription, Arena arena, MemorySegment[] segments, long length, int chunkSizePower) {
        assert (Arrays.stream(segments).map(MemorySegment::scope).allMatch(arena.scope()::equals));
        if (segments.length == 1) {
            return new SingleSegmentImpl(resourceDescription, arena, segments[0], length, chunkSizePower);
        }
        return new MultiSegmentImpl(resourceDescription, arena, segments, 0L, length, chunkSizePower);
    }

    private MemorySegmentIndexInput(String resourceDescription, Arena arena, MemorySegment[] segments, long length, int chunkSizePower) {
        super(resourceDescription);
        this.arena = arena;
        this.segments = segments;
        this.length = length;
        this.chunkSizePower = chunkSizePower;
        this.chunkSizeMask = (1L << chunkSizePower) - 1L;
        this.curSegment = segments[0];
    }

    void ensureOpen() {
        if (this.curSegment == null) {
            throw this.alreadyClosed(null);
        }
    }

    RuntimeException handlePositionalIOOBE(RuntimeException unused, String action, long pos) throws IOException {
        if (pos < 0L) {
            return new IllegalArgumentException(action + " negative position (pos=" + pos + "): " + this);
        }
        throw new EOFException(action + " past EOF (pos=" + pos + "): " + this);
    }

    AlreadyClosedException alreadyClosed(RuntimeException e) {
        if (this.curSegment == null) {
            return new AlreadyClosedException("Already closed: " + this);
        }
        if (e instanceof IllegalStateException && e.getMessage() != null && e.getMessage().contains("closed")) {
            return new AlreadyClosedException("Already closed: " + this, e);
        }
        throw e;
    }

    @Override
    public final byte readByte() throws IOException {
        try {
            byte v = this.curSegment.get(LAYOUT_BYTE, this.curPosition);
            ++this.curPosition;
            return v;
        }
        catch (IndexOutOfBoundsException e) {
            do {
                ++this.curSegmentIndex;
                if (this.curSegmentIndex >= this.segments.length) {
                    throw new EOFException("read past EOF: " + this);
                }
                this.curSegment = this.segments[this.curSegmentIndex];
                this.curPosition = 0L;
            } while (this.curSegment.byteSize() == 0L);
            byte v = this.curSegment.get(LAYOUT_BYTE, this.curPosition);
            ++this.curPosition;
            return v;
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public final void readBytes(byte[] b, int offset, int len) throws IOException {
        try {
            MemorySegment.copy(this.curSegment, LAYOUT_BYTE, this.curPosition, b, offset, len);
            this.curPosition += (long)len;
        }
        catch (IndexOutOfBoundsException e) {
            this.readBytesBoundary(b, offset, len);
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    private void readBytesBoundary(byte[] b, int offset, int len) throws IOException {
        try {
            long curAvail = this.curSegment.byteSize() - this.curPosition;
            while ((long)len > curAvail) {
                MemorySegment.copy(this.curSegment, LAYOUT_BYTE, this.curPosition, b, offset, (int)curAvail);
                len = (int)((long)len - curAvail);
                offset = (int)((long)offset + curAvail);
                ++this.curSegmentIndex;
                if (this.curSegmentIndex >= this.segments.length) {
                    throw new EOFException("read past EOF: " + this);
                }
                this.curSegment = this.segments[this.curSegmentIndex];
                this.curPosition = 0L;
                curAvail = this.curSegment.byteSize();
            }
            MemorySegment.copy(this.curSegment, LAYOUT_BYTE, this.curPosition, b, offset, len);
            this.curPosition += (long)len;
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public void readInts(int[] dst, int offset, int length) throws IOException {
        try {
            MemorySegment.copy(this.curSegment, LAYOUT_LE_INT, this.curPosition, dst, offset, length);
            this.curPosition += 4L * (long)length;
        }
        catch (IndexOutOfBoundsException iobe) {
            super.readInts(dst, offset, length);
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public void readLongs(long[] dst, int offset, int length) throws IOException {
        try {
            MemorySegment.copy(this.curSegment, LAYOUT_LE_LONG, this.curPosition, dst, offset, length);
            this.curPosition += 8L * (long)length;
        }
        catch (IndexOutOfBoundsException iobe) {
            super.readLongs(dst, offset, length);
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public void readFloats(float[] dst, int offset, int length) throws IOException {
        try {
            MemorySegment.copy(this.curSegment, LAYOUT_LE_FLOAT, this.curPosition, dst, offset, length);
            this.curPosition += 4L * (long)length;
        }
        catch (IndexOutOfBoundsException iobe) {
            super.readFloats(dst, offset, length);
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public final short readShort() throws IOException {
        try {
            short v = this.curSegment.get(LAYOUT_LE_SHORT, this.curPosition);
            this.curPosition += 2L;
            return v;
        }
        catch (IndexOutOfBoundsException e) {
            return super.readShort();
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public final int readInt() throws IOException {
        try {
            int v = this.curSegment.get(LAYOUT_LE_INT, this.curPosition);
            this.curPosition += 4L;
            return v;
        }
        catch (IndexOutOfBoundsException e) {
            return super.readInt();
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public final long readLong() throws IOException {
        try {
            long v = this.curSegment.get(LAYOUT_LE_LONG, this.curPosition);
            this.curPosition += 8L;
            return v;
        }
        catch (IndexOutOfBoundsException e) {
            return super.readLong();
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public long getFilePointer() {
        this.ensureOpen();
        return ((long)this.curSegmentIndex << this.chunkSizePower) + this.curPosition;
    }

    @Override
    public void seek(long pos) throws IOException {
        this.ensureOpen();
        int si = (int)(pos >> this.chunkSizePower);
        try {
            if (si != this.curSegmentIndex) {
                MemorySegment seg = this.segments[si];
                this.curSegmentIndex = si;
                this.curSegment = seg;
            }
            this.curPosition = Objects.checkIndex(pos & this.chunkSizeMask, this.curSegment.byteSize() + 1L);
        }
        catch (IndexOutOfBoundsException e) {
            throw this.handlePositionalIOOBE(e, "seek", pos);
        }
    }

    @Override
    public byte readByte(long pos) throws IOException {
        try {
            int si = (int)(pos >> this.chunkSizePower);
            return this.segments[si].get(LAYOUT_BYTE, pos & this.chunkSizeMask);
        }
        catch (IndexOutOfBoundsException ioobe) {
            throw this.handlePositionalIOOBE(ioobe, "read", pos);
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    private void setPos(long pos, int si) throws IOException {
        try {
            MemorySegment seg = this.segments[si];
            this.curPosition = pos & this.chunkSizeMask;
            this.curSegmentIndex = si;
            this.curSegment = seg;
        }
        catch (IndexOutOfBoundsException ioobe) {
            throw this.handlePositionalIOOBE(ioobe, "read", pos);
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public short readShort(long pos) throws IOException {
        int si = (int)(pos >> this.chunkSizePower);
        try {
            return this.segments[si].get(LAYOUT_LE_SHORT, pos & this.chunkSizeMask);
        }
        catch (IndexOutOfBoundsException ioobe) {
            this.setPos(pos, si);
            return this.readShort();
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public int readInt(long pos) throws IOException {
        int si = (int)(pos >> this.chunkSizePower);
        try {
            return this.segments[si].get(LAYOUT_LE_INT, pos & this.chunkSizeMask);
        }
        catch (IndexOutOfBoundsException ioobe) {
            this.setPos(pos, si);
            return this.readInt();
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public long readLong(long pos) throws IOException {
        int si = (int)(pos >> this.chunkSizePower);
        try {
            return this.segments[si].get(LAYOUT_LE_LONG, pos & this.chunkSizeMask);
        }
        catch (IndexOutOfBoundsException ioobe) {
            this.setPos(pos, si);
            return this.readLong();
        }
        catch (IllegalStateException | NullPointerException e) {
            throw this.alreadyClosed(e);
        }
    }

    @Override
    public final long length() {
        return this.length;
    }

    @Override
    public final MemorySegmentIndexInput clone() {
        MemorySegmentIndexInput clone = this.buildSlice(null, 0L, this.length);
        try {
            clone.seek(this.getFilePointer());
        }
        catch (IOException ioe) {
            throw new AssertionError((Object)ioe);
        }
        return clone;
    }

    @Override
    public final MemorySegmentIndexInput slice(String sliceDescription, long offset, long length) {
        if (offset < 0L || length < 0L || offset + length > this.length) {
            throw new IllegalArgumentException("slice() " + sliceDescription + " out of bounds: offset=" + offset + ",length=" + length + ",fileLength=" + this.length + ": " + this);
        }
        return this.buildSlice(sliceDescription, offset, length);
    }

    MemorySegmentIndexInput buildSlice(String sliceDescription, long offset, long length) {
        this.ensureOpen();
        long sliceEnd = offset + length;
        int startIndex = (int)(offset >>> this.chunkSizePower);
        int endIndex = (int)(sliceEnd >>> this.chunkSizePower);
        MemorySegment[] slices = ArrayUtil.copyOfSubArray(this.segments, startIndex, endIndex + 1);
        slices[slices.length - 1] = slices[slices.length - 1].asSlice(0L, sliceEnd & this.chunkSizeMask);
        offset &= this.chunkSizeMask;
        String newResourceDescription = this.getFullSliceDescription(sliceDescription);
        if (slices.length == 1) {
            return new SingleSegmentImpl(newResourceDescription, null, slices[0].asSlice(offset, length), length, this.chunkSizePower);
        }
        return new MultiSegmentImpl(newResourceDescription, null, slices, offset, length, this.chunkSizePower);
    }

    @Override
    public final void close() throws IOException {
        if (this.curSegment == null) {
            return;
        }
        if (this.arena != null) {
            while (this.arena.scope().isAlive()) {
                try {
                    this.arena.close();
                    break;
                }
                catch (IllegalStateException e) {
                    Thread.onSpinWait();
                }
            }
        }
        this.curSegment = null;
        Arrays.fill(this.segments, null);
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    static final class MultiSegmentImpl
    extends MemorySegmentIndexInput {
        private final long offset;

        MultiSegmentImpl(String resourceDescription, Arena arena, MemorySegment[] segments, long offset, long length, int chunkSizePower) {
            super(resourceDescription, arena, segments, length, chunkSizePower);
            this.offset = offset;
            try {
                this.seek(0L);
            }
            catch (IOException ioe) {
                throw new AssertionError((Object)ioe);
            }
            assert (this.curSegment != null && this.curSegmentIndex >= 0);
        }

        @Override
        RuntimeException handlePositionalIOOBE(RuntimeException unused, String action, long pos) throws IOException {
            return super.handlePositionalIOOBE(unused, action, pos - this.offset);
        }

        @Override
        public void seek(long pos) throws IOException {
            assert (pos >= 0L) : "negative position";
            super.seek(pos + this.offset);
        }

        @Override
        public long getFilePointer() {
            return super.getFilePointer() - this.offset;
        }

        @Override
        public byte readByte(long pos) throws IOException {
            return super.readByte(pos + this.offset);
        }

        @Override
        public short readShort(long pos) throws IOException {
            return super.readShort(pos + this.offset);
        }

        @Override
        public int readInt(long pos) throws IOException {
            return super.readInt(pos + this.offset);
        }

        @Override
        public long readLong(long pos) throws IOException {
            return super.readLong(pos + this.offset);
        }

        @Override
        MemorySegmentIndexInput buildSlice(String sliceDescription, long ofs, long length) {
            return super.buildSlice(sliceDescription, this.offset + ofs, length);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    static final class SingleSegmentImpl
    extends MemorySegmentIndexInput {
        SingleSegmentImpl(String resourceDescription, Arena arena, MemorySegment segment, long length, int chunkSizePower) {
            super(resourceDescription, arena, new MemorySegment[]{segment}, length, chunkSizePower);
            this.curSegmentIndex = 0;
        }

        @Override
        public void seek(long pos) throws IOException {
            this.ensureOpen();
            try {
                this.curPosition = Objects.checkIndex(pos, this.length + 1L);
            }
            catch (IndexOutOfBoundsException e) {
                throw this.handlePositionalIOOBE(e, "seek", pos);
            }
        }

        @Override
        public long getFilePointer() {
            this.ensureOpen();
            return this.curPosition;
        }

        @Override
        public byte readByte(long pos) throws IOException {
            try {
                return this.curSegment.get(LAYOUT_BYTE, pos);
            }
            catch (IndexOutOfBoundsException e) {
                throw this.handlePositionalIOOBE(e, "read", pos);
            }
            catch (IllegalStateException | NullPointerException e) {
                throw this.alreadyClosed(e);
            }
        }

        @Override
        public short readShort(long pos) throws IOException {
            try {
                return this.curSegment.get(LAYOUT_LE_SHORT, pos);
            }
            catch (IndexOutOfBoundsException e) {
                throw this.handlePositionalIOOBE(e, "read", pos);
            }
            catch (IllegalStateException | NullPointerException e) {
                throw this.alreadyClosed(e);
            }
        }

        @Override
        public int readInt(long pos) throws IOException {
            try {
                return this.curSegment.get(LAYOUT_LE_INT, pos);
            }
            catch (IndexOutOfBoundsException e) {
                throw this.handlePositionalIOOBE(e, "read", pos);
            }
            catch (IllegalStateException | NullPointerException e) {
                throw this.alreadyClosed(e);
            }
        }

        @Override
        public long readLong(long pos) throws IOException {
            try {
                return this.curSegment.get(LAYOUT_LE_LONG, pos);
            }
            catch (IndexOutOfBoundsException e) {
                throw this.handlePositionalIOOBE(e, "read", pos);
            }
            catch (IllegalStateException | NullPointerException e) {
                throw this.alreadyClosed(e);
            }
        }
    }
}

