# ===========================================================================
# File: debug.tcl
#                        Created: 2010-09-08 07:27:55
#              Last modification: 2010-10-31 08:24:37
# Author: Bernard Desgraupes
# e-mail: <bdesgraupes@users.sourceforge.net>
# (c) Copyright: Bernard Desgraupes 2010
# All rights reserved.
# Description: default Aida settings
# ===========================================================================


namespace eval aida {}


## 
 # ------------------------------------------------------------------------
 # 
 # "aida::verbose" --
 # 
 # Print 'txt' in file 'name' located in directory 'dir' (by default, the
 # user's Logs directory).
 # 
 # ------------------------------------------------------------------------
 ##
proc aida::verbose {level txt} {
	global aida_verbosity
	
	if {$level <= $aida_verbosity} {
		puts stderr $txt
	} 
}


## 
 # ------------------------------------------------------------------------
 # 
 # "aida::log" --
 # 
 # Print 'txt' in file 'name' located in directory 'dir' (by default, the
 # user's Logs directory). There are 1 to 3 args:
 # - if only one argument, it is the message and the name is a default one
 #   based on the date of the day
 # - if two arguments, the first is the name and the second is the message
 # - if there is a third argument, it is a directory to use instead of the
 #   default one (~/Library/Logs/Aida).
 # 
 # Examples:
 #     aida::log "some text"
 #     aida::log somefile "some text"
 #     aida::log somefile "some text" /some/dir
 # 
 # ------------------------------------------------------------------------
 ##
proc aida::log {args} {
	set len [llength $args]
	set name ""
	if {$len == 0} {
		error "wrong number of arguments in aida::log"
	} elseif {$len == 1} {
		set txt [lindex $args 0]
	} elseif {$len > 1} {
		set name [lindex $args 0]
		set txt [lindex $args 1]
	} 
	set dir [lindex $args 2]
	if {$dir eq ""} {
		set dir [file normalize [file join ~ Library Logs Aida]]
	} 
	if {![file exists $dir]} {
		file mkdir $dir
	} 
	if {$name eq ""} {
		set date [clock format [clock seconds] -format "%Y%m%d"]
		set name "aida_${date}.log"
	} 
	regsub -all ":" $name "-" name
	set f [file join $dir ${name}]
	set fid [open $f a+]
	fconfigure $fid -translation lf -encoding macRoman
	puts $fid $txt
	close $fid
}


## 
 # ------------------------------------------------------------------------
 # 
 # "aida::stampMsg" --
 # 
 # Print a time stamp in logfile 'name' and any additional arguments.
 # 
 # Examples:
 #     aida::stampMsg "stamps" "from core.tcl"
 #     aida::stampMsg "postConvertHook"
 #     aida::stampMsg "aida_open" "some text" [encoding system] $aida_library
 # 
 # ------------------------------------------------------------------------
 ##
proc aida::stampMsg {name args} {
	set stamp "----- [clock format [clock seconds] -format "%Y-%m-%d %H:%M:%S"] -----"
	aida::log $name $stamp
	foreach str $args {
		aida::log $name $str
	} 	
} 


# # TAGGED FOR REMOVAL
proc aida::dbgVars {} {
	global aida_library aida_version aida_verbosity
	aida::stampMsg "core" 
	aida::log "core" "aida_library: $aida_library"
	aida::log "core" "aida_version: $aida_version"
	aida::log "core" "aida_verbosity: $aida_verbosity"
	aida::log "core" "name of executable: [info nameofexecutable]"
	aida::log "core" "pwd: [pwd]"
}


# # TEMPORARY: call it now
# aida::dbgVars


proc aida::dbgHeader {} {
	aida::log "aida_dbgHeader" "[string repeat "-" 70]\n[clock format [clock seconds]]"
	aida::log "aida_dbgHeader" [exec aida info variables]
}


