// //////////////////////////////////////////////////////////////////////////////
// / DISCLAIMER
// /
// /
// / Licensed under the Apache License, Version 2.0 (the "License")
// / you may not use this file except in compliance with the License.
// / You may obtain a copy of the License at
// /
// /     http://www.apache.org/licenses/LICENSE-2.0
// /
// / Unless required by applicable law or agreed to in writing, software
// / distributed under the License is distributed on an "AS IS" BASIS,
// / WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// / See the License for the specific language governing permissions and
// / limitations under the License.
// /
// / Copyright holder is ArangoDB GmbH, Cologne, Germany
// /
// / @author Heiko Kernbach
// / @author Lars Maier
// / @author Markus Pfeiffer
// / @author Copyright 2020, ArangoDB GmbH, Cologne, Germany
// //////////////////////////////////////////////////////////////////////////////

// Example graphs for pregel testing

const internal = require("internal");
const db = internal.db;
const _ = require("lodash");

// ee/ce check + gm selection
const isEnterprise = require("internal").isEnterprise();
const graphModule = isEnterprise? require("@arangodb/smart-graph") : require("@arangodb/general-graph");

// graph
//
// name
// vertices list of objects with properties, will turn into documents
// edges pairs of integers referring to docs in the list of vertices

// add a sequence of edges specified by a sequence of vertices
function add_arc(graph, vids) {
  for (var i = 0; i < vids.length - 1; i++) {
    graph.edges.push([vids[i], vids[i+1]]);
  }
}
exports.add_arc = add_arc;

// circle
function add_circle(graph, vids) {
  add_arc(graph, [...vids, vids[0]]);
}
exports.add_circle = add_circle;

// complete graph
function add_complete(graph, vids) {
  for (let i = 0; i < vids.length; i++) {
    for (let j = 0; j < vids.length; j++) {
      if (i !== j) {
        graph.edges.push([vids[i], vids[j]]);
      }
    }
  }
}
exports.add_complete = add_complete;

function add_binary_tree(graph, vids) {
  let i = 0;

  while (i < vids.length) {
    if (2 * i + 1 < vids.length) {
      graph.edges.push([vids[i], vids[2 * i + 1]]);
    }
    if (2 * i + 2 < vids.length) {
      graph.edges.push([vids[i], vids[2 * i + 2]]);
    }
    i = i + 1;
  }
}
exports.add_binary_tree = add_binary_tree;

function create_graph_spec(name, smartGraphAttribute, numberOfShards) {
  return {
    name: name,
    vname: name + "_V",
    ename: name + "_E",
    numberOfShards: numberOfShards,
    smartGraphAttribute: smartGraphAttribute,
    vertices: [],
    edges: []
  };
}
exports.create_graph_spec = create_graph_spec;

function create_graph(graphSpec) {
  try {
    graphModule._drop(graphSpec.name, true);
  } catch (e) {}
  graphModule._create(
    graphSpec.name,
    [graphModule._relation(graphSpec.ename, graphSpec.vname, graphSpec.vname)],
    [],
    { smartGraphAttribute: graphSpec.smartGraphAttribute, numberOfShards: graphSpec.numberOfShards }
  );

  let vids = db._collection(graphSpec.vname).save(graphSpec.vertices);

  // TODO: edge weights/costs/docs?
  const ecoll = db._collection(graphSpec.ename);
  graphSpec.edges.forEach( function(e) {
    let edgeDoc = {_from: vids[e[0]]._id, _to: vids[e[1]]._id};
    ecoll.save(edgeDoc);
  });
}
exports.create_graph = create_graph;

function create_disjoint_circle_and_complete_graph(name, shards) {
  var spec = create_graph_spec(name, "id", shards);

  for (let i = 0; i < 10; i++) {
    spec.vertices.push({ "id": "A" + i});
  }
  add_circle(spec, _.range(10));

  for (let i = 0; i < 10; i++) {
    spec.vertices.push({ "id": "B" + i});
  }
  add_complete(spec, _.range(10,20));

  create_graph(spec);

  return spec;
}
exports.create_disjoint_circle_and_complete_graph = create_disjoint_circle_and_complete_graph;

function create_circle_graph(name, shards) {
  var spec = create_graph_spec(name, "id", shards);

  for (var i = 0; i < 10; i++) {
    spec.vertices.push({ "id": "A" + i});
  }
  add_circle(spec, _.range(10));
  create_graph(spec);

  return spec;
}
exports.create_circle_graph = create_circle_graph;

function create_complete_graph(name, shards, numVertices = 100) {
  var spec = create_graph_spec(name, "id", shards);

  for (var i = 0; i < numVertices; i++) {
    spec.vertices.push({ "id": "A" + i});
  }
  add_complete(spec, _.range(numVertices));
  create_graph(spec);

  return spec;
}
exports.create_complete_graph = create_complete_graph;

exports.create_tadpole_graph = function(name, shards) {
  var spec = create_graph_spec(name, "id", shards);

  for (var i = 0; i < 10; i++) {
    spec.vertices.push({ "id": "A" + i});
  }
  add_arc(spec, [0,1,2,3,4]);
  add_circle(spec, [4,5,6,7,8,9]);
  create_graph(spec);

  return spec;
};

// This should maybe also be using other example graphs from
// for example the traversal suite.
function createCircle(graphName, numberOfVertices, numberOfShards) {
  const vname = graphName + "_V";
  const ename = graphName + "_E";
  try {
    graphModule._drop(graphName, true);
  } catch (e) {}
  graphModule._create(
      graphName,
      [graphModule._relation(ename, vname, vname)],
      [],
      { smartGraphAttribute: "id", numberOfShards: numberOfShards }
  );

  var vs = [];
  var vids;

  for (let i = 0; i < numberOfVertices; i++) {
    vs.push({ id: "" + i });
  }
  vids = db._collection(vname).save(vs);

  var es = [];
  for (let i = 0; i < numberOfVertices; i++) {
    es.push({
      _from: vids[i]._id,
      _to: vids[(i+1)%numberOfVertices]._id,
      cost: 1,
    });
  };

  db._collection(ename).save(es);

  return { vname: vname, ename: ename };
}

function writeToLeaf(obj, path) {
  path = path.split('.');
  let res = obj;

  for (var i=0; i < path.length; i++) {
    if (res[path[i]]) {
      res = res[path[i]];
    } else {
      res[path[i]] = {};
    }
  }
}

function objectInsertHelper(arr, valueToInsert) {
  let obj = {};
  arr.forEach((key) => {
    if (typeof key === "string") {
      obj[key] = valueToInsert;
    } else {
      // array expected (toplevel -> n-level)
      _.set(obj, key, valueToInsert);
    }
  });
  obj.id = valueToInsert;
  return obj;
}

function createLineGraph(graphName, numberOfVertices, numberOfShards, vertexKeysToInsert) {
  const vname = graphName + "_V";
  const ename = graphName + "_E";

  try {
    graphModule._drop(graphName, true);
  } catch (e) {}
  graphModule._create(
    graphName,
    [graphModule._relation(ename, vname, vname)],
    [],
    { smartGraphAttribute: "id", numberOfShards: numberOfShards }
  );

  var vs = [];
  var vids;

  // vertexKeysToInsert can be either array of strings (simple path)
  // ["a", "b", "c"]
  // or also array of arrays of strings (nested path)
  // [["a", "b"], ["a", "c"], "d", "e"]

  for (let i = 0; i < numberOfVertices; i++) {
    if (vertexKeysToInsert) {
      let toInsert = objectInsertHelper(vertexKeysToInsert, i);
      vs.push(toInsert);
    } else {
      vs.push({ id: "" + i });
    }
  }
  vids = db._collection(vname).save(vs);

  var es = [];
  for (let i = 0; i < numberOfVertices - 1; i++) {
    es.push({
      _from: vids[i]._id,
      _to: vids[i+1]._id,
      cost: 1,
    });
  };

  db._collection(ename).save(es);

  return { name: graphName, vname: vname, ename: ename };
}

function createStarGraph(graphName, numberOfVertices, numberOfShards) {
  const vname = graphName + "_V";
  const ename = graphName + "_E";

  try {
    graphModule._drop(graphName, true);
  } catch (e) {}
  graphModule._create(
      graphName,
      [graphModule._relation(ename, vname, vname)],
      [],
      { smartGraphAttribute: "id", numberOfShards: numberOfShards }
  );

  var vs = [];
  var vids;

  for (let i = 0; i < numberOfVertices; i++) {
    vs.push({ id: "" + i });
  }
  vids = db._collection(vname).save(vs);

  var es = [];
  for (let i = 1; i < numberOfVertices - 1; i++) {
    es.push({
      _from: vids[0]._id,
      _to: vids[i+1]._id,
      cost: 1,
    });
  }

  db._collection(ename).save(es);

  return { name: graphName, vname: vname, ename: ename };
}

function createPageRankGraph(graphName, numberOfVertices, numberOfShards) {
  const vname = graphName + "_V";
  const ename = graphName + "_E";
  try {
    graphModule._drop(graphName, true);
  } catch (e) {}
  graphModule._create(
      graphName,
      [graphModule._relation(ename, vname, vname)],
      [],
      { smartGraphAttribute: "id", numberOfShards: numberOfShards }
  );

  var vs = [];
  var vids;

  // vertices data
  let vertices = ['A', 'B', 'C', 'D', 'E'];
  vertices.forEach(function(v) {
    vs.push({
      name: v
    });
  });
  vids = db._collection(vname).save(vs);

  let storeEdge = function (from, to, customSeed) {
    db._collection(ename).save({
      _from: from,
      _to: to,
      customSeed: customSeed
    });
  };
  // A = 0, B = 1, C = 2, D = 3, E = 4

  // A -> B
  storeEdge(vids[0]._id, vids[1]._id, 1);

  // A -> C
  storeEdge(vids[0]._id, vids[2]._id, 1);

  // B -> C
  storeEdge(vids[1]._id, vids[2]._id, 1);

  // C -> E
  storeEdge(vids[2]._id, vids[4]._id, 1);

  // D -> B
  storeEdge(vids[3]._id, vids[1]._id, 1);

  // D -> C
  storeEdge(vids[3]._id, vids[2]._id, 1);

  // D -> E
  storeEdge(vids[3]._id, vids[4]._id, 1);

  // E -> E
  storeEdge(vids[4]._id, vids[4]._id, 1);
  return { name: graphName, vname: vname, ename: ename };
}

function createWikiVoteGraph(graphName, numberOfShards) {
  const vname = graphName + "_V";
  const ename = graphName + "_E";
  const N = 889;
  const socialNetwork = [
    [8, 1],
    [19, 1],
    [36, 1],
    [3, 2],
    [4, 2],
    [6, 2],
    [8, 2],
    [12, 2],
    [16, 2],
    [20, 2],
    [21, 2],
    [4, 3],
    [8, 3],
    [11, 3],
    [16, 3],
    [20, 3],
    [23, 3],
    [24, 3],
    [60, 3],
    [88, 5],
    [118, 5],
    [164, 5],
    [306, 5],
    [369, 5],
    [447, 5],
    [608, 5],
    [657, 5],
    [8, 6],
    [28, 6],
    [48, 6],
    [9, 7],
    [16, 7],
    [18, 7],
    [26, 7],
    [9, 8],
    [11, 8],
    [17, 8],
    [18, 8],
    [19, 8],
    [20, 8],
    [21, 8],
    [23, 8],
    [26, 8],
    [27, 8],
    [30, 8],
    [32, 8],
    [33, 8],
    [34, 8],
    [37, 8],
    [39, 8],
    [40, 8],
    [42, 8],
    [44, 8],
    [45, 8],
    [46, 8],
    [16, 9],
    [20, 9],
    [32, 9],
    [34, 9],
    [49, 9],
    [55, 9],
    [58, 9],
    [59, 9],
    [88, 9],
    [117, 9],
    [23, 10],
    [50, 12],
    [16, 13],
    [28, 13],
    [33, 13],
    [41, 13],
    [64, 13],
    [71, 13],
    [78, 13],
    [95, 13],
    [96, 13],
    [127, 13],
    [163, 13],
    [166, 13],
    [190, 13],
    [196, 13],
    [207, 13],
    [230, 13],
    [20, 14],
    [47, 14],
    [55, 14],
    [56, 14],
    [63, 14],
    [88, 14],
    [98, 14],
    [110, 14],
    [20, 15],
    [60, 16],
    [34, 18],
    [24, 20],
    [25, 20],
    [33, 20],
    [66, 20],
    [79, 20],
    [33, 21],
    [37, 21],
    [273, 21],
    [36, 22],
    [40, 22],
    [67, 22],
    [70, 22],
    [88, 22],
    [98, 22],
    [104, 22],
    [123, 22],
    [158, 22],
    [207, 22],
    [306, 22],
    [408, 22],
    [536, 22],
    [555, 22],
    [617, 22],
    [33, 23],
    [37, 23],
    [32, 24],
    [45, 24],
    [53, 25],
    [55, 25],
    [75, 25],
    [88, 25],
    [90, 25],
    [100, 25],
    [112, 25],
    [124, 25],
    [146, 25],
    [150, 25],
    [151, 25],
    [31, 26],
    [52, 26],
    [50, 29],
    [64, 29],
    [87, 31],
    [45, 32],
    [43, 35],
    [98, 35],
    [106, 35],
    [148, 35],
    [195, 35],
    [212, 35],
    [257, 35],
    [273, 35],
    [199, 38],
    [50, 39],
    [50, 42],
    [50, 45],
    [69, 45],
    [50, 47],
    [54, 47],
    [60, 47],
    [67, 47],
    [82, 47],
    [91, 47],
    [97, 47],
    [66, 49],
    [88, 49],
    [90, 49],
    [121, 49],
    [161, 49],
    [190, 49],
    [193, 49],
    [76, 50],
    [82, 50],
    [86, 50],
    [87, 50],
    [202, 51],
    [207, 51],
    [212, 51],
    [214, 51],
    [230, 51],
    [231, 51],
    [253, 51],
    [311, 51],
    [342, 51],
    [415, 51],
    [420, 51],
    [482, 51],
    [496, 51],
    [59, 53],
    [105, 53],
    [114, 53],
    [117, 53],
    [123, 53],
    [125, 53],
    [129, 53],
    [166, 53],
    [167, 53],
    [60, 54],
    [69, 54],
    [59, 55],
    [61, 55],
    [88, 55],
    [105, 55],
    [135, 55],
    [137, 55],
    [140, 55],
    [141, 55],
    [74, 56],
    [91, 56],
    [100, 56],
    [101, 56],
    [59, 57],
    [61, 57],
    [93, 57],
    [94, 57],
    [98, 57],
    [104, 57],
    [108, 57],
    [114, 57],
    [121, 57],
    [122, 57],
    [123, 57],
    [125, 57],
    [143, 57],
    [156, 57],
    [158, 57],
    [160, 57],
    [163, 57],
    [170, 57],
    [171, 57],
    [176, 57],
    [179, 57],
    [181, 57],
    [95, 58],
    [104, 58],
    [105, 58],
    [125, 58],
    [161, 58],
    [63, 59],
    [74, 59],
    [93, 59],
    [105, 59],
    [106, 59],
    [114, 59],
    [123, 59],
    [125, 59],
    [138, 59],
    [145, 59],
    [163, 59],
    [170, 59],
    [91, 60],
    [121, 61],
    [151, 61],
    [163, 61],
    [114, 62],
    [78, 63],
    [135, 63],
    [151, 63],
    [156, 63],
    [211, 63],
    [204, 65],
    [396, 65],
    [83, 67],
    [204, 68],
    [266, 68],
    [301, 68],
    [99, 70],
    [110, 70],
    [202, 70],
    [216, 70],
    [258, 70],
    [164, 72],
    [230, 72],
    [496, 72],
    [98, 73],
    [308, 73],
    [431, 73],
    [121, 74],
    [126, 74],
    [130, 74],
    [137, 74],
    [172, 74],
    [187, 74],
    [95, 75],
    [112, 75],
    [81, 76],
    [560, 77],
    [96, 78],
    [166, 78],
    [229, 78],
    [230, 78],
    [81, 80],
    [90, 84],
    [196, 84],
    [152, 85],
    [89, 88],
    [90, 88],
    [96, 88],
    [107, 88],
    [111, 88],
    [115, 88],
    [116, 88],
    [118, 88],
    [121, 88],
    [108, 90],
    [162, 90],
    [170, 90],
    [415, 91],
    [98, 92],
    [123, 93],
    [142, 93],
    [118, 95],
    [230, 96],
    [104, 98],
    [105, 98],
    [108, 98],
    [109, 98],
    [118, 98],
    [123, 98],
    [137, 98],
    [138, 98],
    [163, 98],
    [170, 98],
    [195, 98],
    [109, 102],
    [160, 102],
    [195, 103],
    [114, 104],
    [123, 104],
    [156, 104],
    [162, 104],
    [167, 104],
    [170, 104],
    [171, 104],
    [118, 105],
    [123, 105],
    [128, 105],
    [169, 105],
    [178, 106],
    [189, 106],
    [109, 108],
    [114, 108],
    [121, 108],
    [122, 108],
    [141, 108],
    [158, 108],
    [171, 108],
    [183, 108],
    [189, 108],
    [121, 109],
    [122, 109],
    [123, 109],
    [183, 109],
    [189, 109],
    [212, 109],
    [214, 109],
    [224, 109],
    [230, 109],
    [121, 110],
    [123, 110],
    [130, 110],
    [170, 110],
    [223, 110],
    [306, 110],
    [328, 110],
    [115, 111],
    [135, 111],
    [326, 113],
    [413, 113],
    [431, 113],
    [479, 113],
    [515, 113],
    [546, 113],
    [551, 113],
    [562, 113],
    [570, 113],
    [634, 113],
    [637, 113],
    [666, 113],
    [690, 113],
    [707, 113],
    [709, 113],
    [715, 113],
    [738, 113],
    [121, 114],
    [171, 114],
    [176, 114],
    [123, 115],
    [128, 115],
    [123, 118],
    [170, 118],
    [178, 118],
    [212, 118],
    [298, 118],
    [447, 118],
    [471, 118],
    [606, 118],
    [223, 119],
    [238, 119],
    [159, 120],
    [259, 120],
    [311, 120],
    [317, 120],
    [350, 120],
    [364, 120],
    [126, 121],
    [184, 121],
    [185, 121],
    [186, 121],
    [123, 122],
    [124, 123],
    [125, 123],
    [163, 123],
    [169, 123],
    [171, 123],
    [172, 123],
    [174, 123],
    [125, 124],
    [146, 124],
    [168, 124],
    [130, 128],
    [143, 128],
    [162, 128],
    [170, 128],
    [171, 128],
    [191, 128],
    [195, 128],
    [205, 128],
    [178, 129],
    [131, 130],
    [170, 130],
    [197, 130],
    [224, 130],
    [229, 130],
    [233, 130],
    [237, 130],
    [277, 130],
    [278, 130],
    [204, 132],
    [273, 132],
    [361, 132],
    [385, 132],
    [392, 132],
    [399, 132],
    [410, 132],
    [416, 132],
    [442, 132],
    [448, 132],
    [453, 132],
    [455, 132],
    [456, 132],
    [461, 132],
    [482, 132],
    [488, 132],
    [504, 132],
    [506, 132],
    [523, 132],
    [536, 132],
    [560, 132],
    [562, 132],
    [608, 132],
    [617, 132],
    [143, 133],
    [170, 133],
    [187, 133],
    [205, 133],
    [224, 133],
    [239, 133],
    [240, 133],
    [259, 133],
    [170, 134],
    [247, 134],
    [289, 134],
    [331, 134],
    [351, 134],
    [690, 136],
    [822, 136],
    [832, 136],
    [834, 136],
    [855, 136],
    [231, 139],
    [239, 139],
    [324, 139],
    [431, 139],
    [437, 139],
    [447, 139],
    [453, 139],
    [455, 139],
    [456, 139],
    [458, 139],
    [230, 140],
    [328, 140],
    [156, 141],
    [220, 141],
    [163, 143],
    [187, 143],
    [205, 143],
    [207, 143],
    [217, 143],
    [240, 143],
    [254, 143],
    [356, 144],
    [557, 147],
    [717, 147],
    [163, 148],
    [199, 148],
    [239, 148],
    [254, 148],
    [257, 148],
    [267, 148],
    [273, 148],
    [305, 148],
    [309, 148],
    [314, 148],
    [328, 149],
    [156, 152],
    [157, 152],
    [189, 152],
    [193, 152],
    [205, 152],
    [214, 152],
    [215, 152],
    [216, 152],
    [216, 153],
    [248, 153],
    [225, 154],
    [227, 154],
    [306, 154],
    [188, 155],
    [199, 155],
    [225, 155],
    [273, 155],
    [337, 155],
    [163, 156],
    [189, 156],
    [196, 156],
    [208, 156],
    [210, 156],
    [215, 157],
    [170, 158],
    [189, 158],
    [204, 158],
    [273, 158],
    [322, 158],
    [166, 160],
    [170, 160],
    [166, 161],
    [230, 161],
    [327, 161],
    [328, 161],
    [173, 162],
    [195, 162],
    [234, 162],
    [205, 163],
    [230, 164],
    [374, 164],
    [380, 164],
    [381, 164],
    [399, 164],
    [466, 164],
    [496, 164],
    [170, 165],
    [337, 165],
    [351, 165],
    [170, 166],
    [175, 166],
    [178, 166],
    [194, 166],
    [215, 166],
    [224, 166],
    [230, 166],
    [248, 166],
    [217, 168],
    [191, 170],
    [192, 170],
    [193, 170],
    [195, 170],
    [199, 170],
    [202, 170],
    [203, 170],
    [204, 170],
    [205, 170],
    [209, 170],
    [212, 170],
    [214, 170],
    [216, 170],
    [226, 170],
    [233, 170],
    [244, 170],
    [247, 170],
    [258, 170],
    [259, 170],
    [260, 170],
    [263, 170],
    [264, 170],
    [271, 170],
    [272, 170],
    [273, 170],
    [275, 170],
    [277, 170],
    [278, 170],
    [280, 170],
    [285, 170],
    [289, 170],
    [298, 170],
    [299, 170],
    [303, 170],
    [306, 170],
    [308, 170],
    [317, 170],
    [322, 170],
    [324, 170],
    [330, 170],
    [334, 170],
    [335, 170],
    [341, 170],
    [349, 170],
    [356, 170],
    [360, 170],
    [361, 170],
    [363, 170],
    [364, 170],
    [365, 170],
    [367, 170],
    [247, 173],
    [263, 173],
    [267, 173],
    [318, 173],
    [178, 174],
    [198, 174],
    [200, 175],
    [205, 175],
    [240, 175],
    [389, 177],
    [230, 178],
    [273, 179],
    [289, 179],
    [334, 179],
    [237, 180],
    [264, 180],
    [265, 180],
    [256, 181],
    [356, 182],
    [186, 185],
    [252, 186],
    [224, 187],
    [207, 189],
    [214, 189],
    [221, 189],
    [193, 190],
    [236, 190],
    [218, 191],
    [266, 191],
    [273, 191],
    [322, 191],
    [399, 191],
    [402, 191],
    [236, 193],
    [271, 193],
    [280, 193],
    [333, 193],
    [344, 193],
    [351, 193],
    [356, 193],
    [359, 193],
    [233, 196],
    [237, 196],
    [245, 196],
    [213, 197],
    [247, 197],
    [273, 197],
    [285, 197],
    [317, 197],
    [335, 197],
    [204, 199],
    [206, 199],
    [212, 199],
    [225, 199],
    [233, 199],
    [239, 199],
    [247, 199],
    [256, 199],
    [259, 199],
    [260, 199],
    [266, 199],
    [269, 199],
    [273, 199],
    [281, 199],
    [285, 199],
    [299, 199],
    [316, 199],
    [317, 199],
    [325, 199],
    [328, 199],
    [337, 199],
    [351, 199],
    [205, 201],
    [273, 202],
    [222, 203],
    [209, 204],
    [230, 204],
    [231, 204],
    [237, 204],
    [239, 204],
    [255, 204],
    [266, 204],
    [273, 204],
    [306, 204],
    [343, 204],
    [347, 204],
    [356, 204],
    [361, 204],
    [364, 204],
    [370, 204],
    [374, 204],
    [380, 204],
    [387, 204],
    [388, 204],
    [392, 204],
    [393, 204],
    [399, 204],
    [404, 204],
    [407, 204],
    [410, 204],
    [416, 204],
    [421, 204],
    [431, 204],
    [433, 204],
    [437, 204],
    [447, 204],
    [448, 204],
    [452, 204],
    [453, 204],
    [455, 204],
    [456, 204],
    [458, 204],
    [461, 204],
    [471, 204],
    [488, 204],
    [491, 204],
    [508, 204],
    [526, 204],
    [531, 204],
    [536, 204],
    [543, 204],
    [555, 204],
    [560, 204],
    [565, 204],
    [209, 205],
    [214, 205],
    [215, 205],
    [237, 205],
    [254, 205],
    [258, 205],
    [259, 205],
    [260, 205],
    [261, 205],
    [262, 205],
    [263, 205],
    [306, 206],
    [380, 206],
    [221, 207],
    [230, 207],
    [337, 209],
    [355, 209],
    [233, 210],
    [257, 210],
    [258, 210],
    [260, 210],
    [261, 210],
    [228, 211],
    [237, 211],
    [223, 212],
    [224, 212],
    [254, 212],
    [260, 212],
    [266, 212],
    [281, 212],
    [293, 212],
    [294, 212],
    [328, 212],
    [332, 212],
    [373, 212],
    [399, 212],
    [486, 212],
    [216, 213],
    [221, 213],
    [227, 214],
    [230, 215],
    [224, 216],
    [233, 216],
    [237, 216],
    [248, 216],
    [258, 216],
    [259, 216],
    [270, 216],
    [275, 216],
    [286, 216],
    [239, 218],
    [536, 218],
    [710, 218],
    [230, 219],
    [247, 219],
    [267, 219],
    [282, 219],
    [286, 220],
    [233, 223],
    [260, 223],
    [266, 223],
    [285, 223],
    [303, 223],
    [323, 223],
    [353, 223],
    [368, 223],
    [404, 223],
    [453, 223],
    [233, 224],
    [237, 224],
    [240, 224],
    [247, 224],
    [254, 224],
    [258, 224],
    [260, 224],
    [267, 224],
    [273, 224],
    [285, 224],
    [227, 225],
    [281, 225],
    [341, 225],
    [227, 226],
    [297, 226],
    [341, 226],
    [237, 228],
    [273, 228],
    [324, 228],
    [361, 228],
    [386, 228],
    [415, 228],
    [416, 228],
    [474, 228],
    [232, 229],
    [281, 230],
    [301, 230],
    [328, 230],
    [336, 230],
    [358, 230],
    [386, 230],
    [393, 230],
    [395, 230],
    [396, 230],
    [410, 230],
    [433, 230],
    [448, 230],
    [460, 230],
    [483, 230],
    [486, 230],
    [503, 230],
    [509, 230],
    [513, 230],
    [430, 231],
    [431, 231],
    [455, 231],
    [492, 231],
    [278, 232],
    [335, 232],
    [237, 233],
    [247, 233],
    [248, 233],
    [257, 233],
    [259, 233],
    [260, 233],
    [266, 233],
    [270, 233],
    [273, 233],
    [284, 233],
    [285, 233],
    [304, 233],
    [261, 234],
    [323, 235],
    [319, 236],
    [344, 236],
    [351, 236],
    [247, 237],
    [260, 237],
    [279, 237],
    [290, 237],
    [311, 237],
    [324, 237],
    [389, 237],
    [404, 237],
    [455, 237],
    [464, 237],
    [491, 237],
    [508, 237],
    [611, 238],
    [242, 239],
    [255, 239],
    [266, 239],
    [273, 239],
    [317, 239],
    [320, 239],
    [337, 239],
    [349, 239],
    [372, 239],
    [420, 239],
    [431, 239],
    [436, 239],
    [260, 240],
    [341, 241],
    [342, 241],
    [344, 241],
    [357, 241],
    [245, 242],
    [431, 242],
    [396, 243],
    [543, 243],
    [608, 243],
    [617, 243],
    [671, 243],
    [715, 243],
    [317, 244],
    [433, 245],
    [502, 245],
    [266, 246],
    [386, 246],
    [258, 247],
    [260, 247],
    [273, 247],
    [275, 247],
    [280, 247],
    [288, 247],
    [290, 247],
    [250, 249],
    [251, 249],
    [257, 249],
    [267, 249],
    [251, 250],
    [263, 254],
    [291, 254],
    [361, 255],
    [614, 255],
    [630, 255],
    [638, 255],
    [665, 255],
    [751, 255],
    [260, 256],
    [272, 256],
    [281, 256],
    [290, 256],
    [325, 256],
    [258, 257],
    [267, 257],
    [273, 257],
    [288, 257],
    [290, 257],
    [309, 257],
    [270, 258],
    [275, 258],
    [285, 258],
    [260, 259],
    [269, 259],
    [270, 259],
    [275, 259],
    [281, 259],
    [289, 259],
    [290, 259],
    [317, 259],
    [319, 259],
    [336, 259],
    [350, 259],
    [351, 259],
    [377, 259],
    [399, 259],
    [410, 259],
    [263, 260],
    [266, 260],
    [275, 260],
    [286, 260],
    [290, 260],
    [299, 260],
    [315, 260],
    [317, 260],
    [274, 261],
    [550, 262],
    [296, 263],
    [285, 266],
    [289, 266],
    [299, 266],
    [317, 266],
    [322, 266],
    [337, 266],
    [399, 266],
    [405, 266],
    [464, 266],
    [270, 268],
    [285, 271],
    [291, 271],
    [319, 271],
    [337, 271],
    [351, 271],
    [273, 272],
    [280, 273],
    [285, 273],
    [290, 273],
    [300, 273],
    [308, 273],
    [310, 273],
    [317, 273],
    [322, 273],
    [347, 273],
    [348, 273],
    [361, 273],
    [364, 273],
    [370, 273],
    [380, 273],
    [381, 273],
    [385, 273],
    [386, 273],
    [388, 273],
    [389, 273],
    [392, 273],
    [396, 273],
    [398, 273],
    [399, 273],
    [408, 273],
    [415, 273],
    [416, 273],
    [426, 273],
    [428, 273],
    [429, 273],
    [430, 273],
    [431, 273],
    [434, 273],
    [441, 273],
    [442, 273],
    [447, 273],
    [448, 273],
    [449, 273],
    [458, 273],
    [460, 273],
    [462, 273],
    [463, 273],
    [465, 273],
    [471, 273],
    [475, 273],
    [483, 273],
    [487, 273],
    [492, 273],
    [494, 273],
    [496, 273],
    [499, 273],
    [505, 273],
    [519, 273],
    [527, 273],
    [528, 273],
    [536, 273],
    [538, 273],
    [547, 273],
    [548, 273],
    [550, 273],
    [552, 273],
    [560, 273],
    [568, 273],
    [570, 273],
    [586, 273],
    [617, 273],
    [691, 273],
    [730, 273],
    [736, 273],
    [750, 273],
    [751, 273],
    [770, 273],
    [834, 273],
    [284, 274],
    [298, 275],
    [299, 275],
    [299, 276],
    [317, 276],
    [314, 277],
    [301, 278],
    [335, 278],
    [285, 280],
    [289, 280],
    [337, 280],
    [372, 280],
    [282, 281],
    [290, 281],
    [292, 281],
    [299, 281],
    [317, 281],
    [328, 281],
    [342, 281],
    [372, 283],
    [364, 284],
    [416, 284],
    [431, 284],
    [322, 285],
    [337, 285],
    [344, 285],
    [351, 285],
    [358, 285],
    [364, 285],
    [370, 285],
    [379, 285],
    [381, 285],
    [386, 285],
    [399, 285],
    [376, 287],
    [319, 289],
    [322, 289],
    [347, 289],
    [364, 289],
    [372, 289],
    [395, 289],
    [399, 289],
    [299, 290],
    [317, 290],
    [350, 290],
    [468, 292],
    [636, 295],
    [416, 297],
    [448, 297],
    [482, 297],
    [518, 297],
    [333, 298],
    [300, 299],
    [317, 299],
    [336, 299],
    [347, 299],
    [350, 299],
    [364, 299],
    [395, 299],
    [410, 299],
    [303, 301],
    [322, 301],
    [386, 301],
    [362, 302],
    [344, 303],
    [356, 306],
    [432, 306],
    [315, 307],
    [389, 307],
    [309, 308],
    [317, 308],
    [358, 308],
    [500, 310],
    [322, 311],
    [408, 311],
    [415, 311],
    [436, 311],
    [485, 311],
    [313, 312],
    [542, 312],
    [319, 314],
    [337, 314],
    [351, 314],
    [356, 314],
    [358, 314],
    [375, 314],
    [361, 316],
    [417, 316],
    [460, 316],
    [536, 316],
    [568, 316],
    [590, 316],
    [344, 317],
    [351, 317],
    [364, 317],
    [362, 318],
    [419, 318],
    [351, 319],
    [356, 319],
    [364, 319],
    [431, 321],
    [546, 321],
    [550, 321],
    [557, 321],
    [604, 321],
    [350, 322],
    [372, 322],
    [386, 322],
    [399, 322],
    [416, 322],
    [421, 322],
    [428, 322],
    [438, 322],
    [439, 322],
    [445, 322],
    [386, 324],
    [453, 324],
    [462, 324],
    [709, 326],
    [772, 326],
    [802, 326],
    [395, 327],
    [417, 327],
    [418, 327],
    [420, 327],
    [373, 328],
    [404, 329],
    [408, 330],
    [421, 330],
    [356, 331],
    [338, 333],
    [344, 334],
    [370, 334],
    [410, 336],
    [490, 336],
    [347, 337],
    [351, 337],
    [356, 337],
    [359, 337],
    [383, 337],
    [410, 337],
    [431, 339],
    [594, 339],
    [405, 340],
    [342, 341],
    [492, 343],
    [610, 343],
    [709, 343],
    [351, 344],
    [356, 344],
    [652, 345],
    [801, 345],
    [804, 345],
    [807, 345],
    [482, 346],
    [356, 347],
    [431, 348],
    [458, 348],
    [560, 348],
    [613, 348],
    [365, 350],
    [356, 351],
    [382, 351],
    [606, 352],
    [639, 352],
    [368, 353],
    [380, 353],
    [380, 354],
    [448, 354],
    [504, 354],
    [511, 354],
    [373, 356],
    [399, 356],
    [406, 356],
    [386, 358],
    [478, 358],
    [426, 361],
    [431, 361],
    [433, 361],
    [474, 361],
    [396, 362],
    [510, 362],
    [513, 362],
    [539, 362],
    [424, 363],
    [370, 364],
    [399, 364],
    [424, 364],
    [437, 364],
    [548, 366],
    [399, 367],
    [625, 367],
    [377, 368],
    [416, 368],
    [431, 368],
    [471, 368],
    [484, 368],
    [485, 368],
    [499, 368],
    [502, 368],
    [508, 368],
    [408, 369],
    [448, 369],
    [548, 369],
    [550, 369],
    [388, 370],
    [393, 370],
    [425, 370],
    [431, 370],
    [447, 370],
    [466, 370],
    [482, 370],
    [494, 370],
    [570, 370],
    [586, 370],
    [655, 370],
    [399, 371],
    [425, 371],
    [717, 371],
    [386, 372],
    [399, 372],
    [412, 372],
    [433, 372],
    [437, 372],
    [441, 372],
    [447, 372],
    [470, 372],
    [383, 375],
    [513, 375],
    [560, 375],
    [641, 376],
    [675, 376],
    [396, 377],
    [415, 377],
    [447, 377],
    [502, 377],
    [525, 377],
    [415, 378],
    [412, 379],
    [441, 379],
    [391, 380],
    [416, 380],
    [431, 380],
    [435, 380],
    [438, 380],
    [456, 380],
    [459, 380],
    [462, 380],
    [484, 380],
    [504, 380],
    [506, 380],
    [509, 380],
    [511, 380],
    [518, 380],
    [534, 380],
    [538, 380],
    [540, 380],
    [545, 380],
    [547, 380],
    [548, 380],
    [557, 380],
    [568, 380],
    [598, 380],
    [609, 380],
    [611, 380],
    [615, 380],
    [626, 380],
    [657, 380],
    [663, 380],
    [386, 381],
    [399, 381],
    [412, 381],
    [425, 381],
    [447, 381],
    [568, 381],
    [606, 381],
    [617, 381],
    [462, 384],
    [491, 384],
    [510, 384],
    [434, 385],
    [449, 385],
    [455, 385],
    [458, 385],
    [504, 385],
    [554, 385],
    [563, 385],
    [405, 386],
    [410, 386],
    [412, 386],
    [417, 386],
    [430, 386],
    [437, 386],
    [447, 386],
    [471, 386],
    [490, 386],
    [496, 386],
    [506, 386],
    [413, 387],
    [536, 387],
    [431, 388],
    [467, 388],
    [570, 388],
    [399, 389],
    [480, 389],
    [552, 390],
    [567, 390],
    [673, 390],
    [392, 391],
    [536, 391],
    [619, 391],
    [393, 392],
    [410, 392],
    [431, 392],
    [447, 392],
    [455, 392],
    [456, 392],
    [458, 392],
    [466, 392],
    [498, 392],
    [539, 392],
    [399, 393],
    [408, 393],
    [415, 393],
    [416, 393],
    [417, 393],
    [426, 393],
    [428, 393],
    [449, 393],
    [451, 393],
    [458, 393],
    [464, 393],
    [470, 393],
    [482, 393],
    [496, 393],
    [498, 393],
    [508, 393],
    [523, 393],
    [536, 393],
    [503, 394],
    [411, 396],
    [416, 396],
    [431, 396],
    [453, 396],
    [474, 396],
    [490, 396],
    [496, 396],
    [497, 396],
    [534, 397],
    [408, 398],
    [425, 398],
    [450, 398],
    [471, 398],
    [499, 398],
    [506, 398],
    [531, 398],
    [536, 398],
    [549, 398],
    [553, 398],
    [586, 398],
    [619, 398],
    [416, 399],
    [417, 399],
    [421, 399],
    [427, 399],
    [428, 399],
    [431, 399],
    [434, 399],
    [436, 399],
    [438, 399],
    [442, 399],
    [448, 399],
    [453, 399],
    [458, 399],
    [462, 399],
    [464, 399],
    [465, 399],
    [466, 399],
    [469, 399],
    [487, 399],
    [490, 399],
    [494, 399],
    [504, 399],
    [506, 399],
    [509, 399],
    [519, 399],
    [526, 399],
    [531, 399],
    [536, 399],
    [538, 399],
    [555, 399],
    [560, 399],
    [562, 399],
    [566, 399],
    [568, 399],
    [576, 399],
    [613, 399],
    [621, 399],
    [417, 400],
    [431, 401],
    [526, 401],
    [545, 403],
    [550, 403],
    [606, 403],
    [627, 403],
    [683, 403],
    [685, 403],
    [697, 403],
    [416, 404],
    [428, 404],
    [429, 404],
    [431, 404],
    [448, 404],
    [449, 404],
    [504, 404],
    [508, 404],
    [511, 404],
    [518, 404],
    [520, 404],
    [431, 405],
    [440, 405],
    [425, 406],
    [536, 406],
    [538, 406],
    [550, 406],
    [570, 406],
    [636, 406],
    [691, 406],
    [697, 406],
    [431, 407],
    [482, 407],
    [484, 407],
    [542, 407],
    [598, 407],
    [417, 408],
    [434, 408],
    [447, 408],
    [453, 408],
    [471, 408],
    [482, 408],
    [513, 408],
    [536, 408],
    [548, 408],
    [606, 408],
    [625, 408],
    [419, 409],
    [430, 410],
    [433, 410],
    [448, 410],
    [449, 410],
    [451, 410],
    [453, 410],
    [463, 410],
    [471, 410],
    [490, 410],
    [503, 410],
    [513, 410],
    [568, 410],
    [446, 411],
    [529, 411],
    [509, 413],
    [550, 413],
    [663, 413],
    [687, 413],
    [741, 413],
    [772, 413],
    [557, 414],
    [671, 414],
    [418, 415],
    [428, 415],
    [433, 415],
    [442, 415],
    [462, 415],
    [469, 415],
    [485, 415],
    [508, 415],
    [421, 416],
    [428, 416],
    [430, 416],
    [431, 416],
    [442, 416],
    [448, 416],
    [449, 416],
    [451, 416],
    [453, 416],
    [455, 416],
    [456, 416],
    [459, 416],
    [461, 416],
    [463, 416],
    [464, 416],
    [466, 416],
    [482, 416],
    [484, 416],
    [486, 416],
    [488, 416],
    [499, 416],
    [503, 416],
    [504, 416],
    [506, 416],
    [510, 416],
    [511, 416],
    [512, 416],
    [517, 416],
    [518, 416],
    [523, 416],
    [536, 416],
    [539, 416],
    [551, 416],
    [562, 416],
    [584, 416],
    [585, 416],
    [426, 417],
    [428, 417],
    [469, 418],
    [481, 418],
    [482, 418],
    [556, 418],
    [558, 418],
    [612, 418],
    [433, 419],
    [730, 419],
    [745, 419],
    [426, 420],
    [431, 421],
    [447, 421],
    [449, 421],
    [453, 421],
    [458, 421],
    [461, 421],
    [464, 421],
    [529, 421],
    [538, 421],
    [525, 422],
    [439, 423],
    [504, 423],
    [550, 423],
    [555, 423],
    [587, 423],
    [654, 423],
    [471, 425],
    [536, 425],
    [548, 425],
    [566, 425],
    [606, 425],
    [609, 425],
    [637, 425],
    [702, 425],
    [770, 425],
    [433, 426],
    [453, 426],
    [458, 426],
    [468, 426],
    [481, 426],
    [491, 426],
    [471, 427],
    [536, 427],
    [568, 427],
    [445, 428],
    [468, 428],
    [431, 430],
    [438, 430],
    [442, 430],
    [445, 430],
    [450, 430],
    [440, 431],
    [444, 431],
    [447, 431],
    [448, 431],
    [452, 431],
    [455, 431],
    [456, 431],
    [460, 431],
    [462, 431],
    [463, 431],
    [466, 431],
    [472, 431],
    [481, 431],
    [487, 431],
    [488, 431],
    [492, 431],
    [493, 431],
    [498, 431],
    [499, 431],
    [501, 431],
    [504, 431],
    [524, 431],
    [526, 431],
    [529, 431],
    [532, 431],
    [536, 431],
    [538, 431],
    [540, 431],
    [544, 431],
    [546, 431],
    [547, 431],
    [548, 431],
    [550, 431],
    [558, 431],
    [561, 431],
    [575, 431],
    [586, 431],
    [591, 431],
    [593, 431],
    [596, 431],
    [604, 431],
    [606, 431],
    [607, 431],
    [613, 431],
    [619, 431],
    [622, 431],
    [672, 431],
    [690, 431],
    [691, 431],
    [697, 431],
    [707, 431],
    [736, 431],
    [738, 431],
    [750, 431],
    [757, 431],
    [762, 431],
    [763, 431],
    [765, 431],
    [781, 431],
    [785, 431],
    [799, 431],
    [803, 431],
    [811, 431],
    [820, 431],
    [822, 431],
    [823, 431],
    [826, 431],
    [834, 431],
    [837, 431],
    [840, 431],
    [843, 431],
    [853, 431],
    [855, 431],
    [868, 431],
    [869, 431],
    [437, 433],
    [443, 433],
    [447, 433],
    [449, 433],
    [453, 433],
    [455, 433],
    [458, 433],
    [463, 433],
    [485, 433],
    [496, 433],
    [515, 433],
    [517, 433],
    [528, 433],
    [544, 433],
    [602, 433],
    [437, 434],
    [447, 434],
    [448, 434],
    [465, 434],
    [471, 434],
    [490, 434],
    [496, 434],
    [503, 434],
    [506, 434],
    [523, 434],
    [538, 434],
    [543, 434],
    [560, 434],
    [655, 435],
    [679, 435],
    [464, 436],
    [461, 437],
    [465, 437],
    [477, 437],
    [490, 437],
    [496, 437],
    [465, 438],
    [536, 438],
    [568, 438],
    [573, 439],
    [660, 439],
    [709, 439],
    [710, 439],
    [723, 439],
    [548, 440],
    [465, 441],
    [466, 441],
    [460, 442],
    [492, 442],
    [506, 442],
    [585, 442],
    [804, 443],
    [530, 444],
    [492, 445],
    [574, 445],
    [642, 445],
    [453, 447],
    [454, 447],
    [456, 447],
    [458, 447],
    [463, 447],
    [465, 447],
    [466, 447],
    [482, 447],
    [483, 447],
    [502, 447],
    [503, 447],
    [504, 447],
    [506, 447],
    [511, 447],
    [517, 447],
    [527, 447],
    [449, 448],
    [453, 448],
    [456, 448],
    [458, 448],
    [459, 448],
    [464, 448],
    [482, 448],
    [497, 448],
    [498, 448],
    [503, 448],
    [504, 448],
    [506, 448],
    [511, 448],
    [523, 448],
    [529, 448],
    [543, 448],
    [548, 448],
    [562, 448],
    [568, 448],
    [574, 448],
    [584, 448],
    [593, 448],
    [597, 448],
    [451, 449],
    [455, 449],
    [456, 449],
    [458, 449],
    [460, 449],
    [463, 449],
    [483, 449],
    [496, 449],
    [499, 449],
    [504, 449],
    [519, 449],
    [521, 449],
    [526, 449],
    [536, 449],
    [539, 449],
    [561, 449],
    [578, 449],
    [536, 450],
    [458, 451],
    [466, 451],
    [471, 451],
    [513, 451],
    [539, 451],
    [560, 451],
    [561, 451],
    [562, 451],
    [579, 452],
    [659, 452],
    [668, 452],
    [687, 452],
    [762, 452],
    [785, 452],
    [786, 452],
    [793, 452],
    [820, 452],
    [825, 452],
    [839, 452],
    [850, 452],
    [851, 452],
    [858, 452],
    [863, 452],
    [872, 452],
    [886, 452],
    [456, 453],
    [463, 453],
    [475, 453],
    [489, 453],
    [499, 453],
    [523, 453],
    [531, 453],
    [536, 453],
    [538, 453],
    [556, 453],
    [558, 453],
    [555, 454],
    [461, 455],
    [463, 455],
    [504, 455],
    [519, 455],
    [560, 455],
    [461, 456],
    [472, 456],
    [506, 456],
    [550, 457],
    [704, 457],
    [466, 458],
    [498, 458],
    [499, 458],
    [536, 458],
    [543, 458],
    [544, 458],
    [504, 459],
    [511, 459],
    [523, 459],
    [536, 459],
    [550, 459],
    [584, 459],
    [597, 459],
    [471, 460],
    [504, 460],
    [496, 461],
    [502, 461],
    [504, 461],
    [500, 462],
    [502, 462],
    [518, 463],
    [490, 464],
    [529, 464],
    [539, 464],
    [486, 465],
    [490, 465],
    [560, 465],
    [499, 466],
    [503, 466],
    [504, 466],
    [517, 466],
    [536, 466],
    [538, 466],
    [539, 466],
    [543, 466],
    [657, 467],
    [525, 469],
    [697, 469],
    [702, 469],
    [810, 469],
    [482, 470],
    [536, 470],
    [538, 470],
    [550, 470],
    [702, 470],
    [483, 471],
    [489, 471],
    [513, 471],
    [536, 471],
    [619, 471],
    [652, 471],
    [550, 473],
    [543, 475],
    [539, 476],
    [492, 477],
    [593, 477],
    [635, 477],
    [697, 477],
    [643, 479],
    [730, 479],
    [779, 479],
    [506, 481],
    [536, 481],
    [548, 481],
    [562, 481],
    [568, 481],
    [574, 481],
    [582, 481],
    [605, 481],
    [492, 482],
    [504, 482],
    [506, 482],
    [509, 482],
    [523, 482],
    [526, 482],
    [536, 482],
    [543, 482],
    [545, 482],
    [546, 482],
    [549, 482],
    [550, 482],
    [562, 482],
    [568, 482],
    [584, 482],
    [588, 482],
    [591, 482],
    [592, 482],
    [597, 482],
    [620, 482],
    [636, 482],
    [536, 484],
    [540, 484],
    [635, 484],
    [660, 484],
    [666, 484],
    [702, 484],
    [704, 484],
    [710, 484],
    [502, 485],
    [508, 485],
    [509, 485],
    [506, 486],
    [531, 486],
    [538, 486],
    [538, 487],
    [540, 487],
    [543, 487],
    [548, 487],
    [562, 487],
    [568, 487],
    [572, 487],
    [582, 487],
    [613, 487],
    [715, 488],
    [783, 488],
    [871, 488],
    [531, 490],
    [536, 490],
    [538, 490],
    [574, 490],
    [562, 494],
    [762, 495],
    [503, 496],
    [560, 496],
    [539, 499],
    [569, 499],
    [580, 499],
    [528, 500],
    [557, 500],
    [730, 500],
    [736, 500],
    [746, 500],
    [530, 503],
    [560, 503],
    [506, 504],
    [508, 504],
    [511, 504],
    [513, 504],
    [518, 504],
    [523, 504],
    [529, 504],
    [536, 504],
    [544, 504],
    [568, 504],
    [582, 504],
    [584, 504],
    [597, 504],
    [507, 505],
    [550, 505],
    [635, 505],
    [536, 506],
    [548, 506],
    [559, 506],
    [511, 508],
    [514, 509],
    [515, 509],
    [523, 511],
    [584, 511],
    [597, 511],
    [536, 513],
    [559, 513],
    [697, 513],
    [786, 513],
    [819, 513],
    [550, 515],
    [583, 515],
    [588, 515],
    [617, 515],
    [631, 515],
    [665, 516],
    [536, 518],
    [575, 518],
    [592, 521],
    [617, 522],
    [724, 522],
    [526, 523],
    [568, 523],
    [584, 523],
    [624, 523],
    [568, 524],
    [550, 525],
    [666, 525],
    [769, 525],
    [529, 526],
    [536, 526],
    [562, 526],
    [568, 526],
    [592, 526],
    [545, 528],
    [617, 528],
    [646, 528],
    [681, 528],
    [538, 529],
    [543, 529],
    [596, 529],
    [536, 531],
    [543, 531],
    [586, 531],
    [557, 532],
    [820, 532],
    [822, 532],
    [842, 532],
    [786, 533],
    [540, 534],
    [550, 534],
    [538, 535],
    [625, 535],
    [538, 536],
    [539, 536],
    [545, 536],
    [546, 536],
    [548, 536],
    [550, 536],
    [562, 536],
    [566, 536],
    [568, 536],
    [570, 536],
    [572, 536],
    [575, 536],
    [576, 536],
    [577, 536],
    [579, 536],
    [592, 536],
    [593, 536],
    [601, 536],
    [606, 536],
    [609, 536],
    [619, 536],
    [630, 536],
    [639, 536],
    [650, 536],
    [736, 537],
    [762, 537],
    [781, 537],
    [823, 537],
    [839, 537],
    [864, 537],
    [880, 537],
    [548, 538],
    [558, 538],
    [562, 538],
    [568, 538],
    [571, 538],
    [572, 538],
    [575, 538],
    [593, 538],
    [619, 538],
    [624, 538],
    [632, 538],
    [540, 539],
    [545, 540],
    [635, 540],
    [645, 540],
    [626, 541],
    [550, 543],
    [619, 543],
    [550, 544],
    [546, 545],
    [547, 545],
    [549, 545],
    [550, 545],
    [551, 545],
    [588, 545],
    [611, 545],
    [613, 545],
    [619, 545],
    [620, 545],
    [626, 545],
    [629, 545],
    [639, 545],
    [646, 545],
    [653, 545],
    [660, 545],
    [672, 545],
    [683, 545],
    [547, 546],
    [550, 546],
    [557, 546],
    [584, 546],
    [588, 546],
    [597, 546],
    [617, 546],
    [626, 546],
    [550, 547],
    [588, 547],
    [596, 547],
    [611, 547],
    [613, 547],
    [617, 547],
    [627, 547],
    [639, 547],
    [645, 547],
    [647, 547],
    [653, 547],
    [655, 547],
    [660, 547],
    [562, 548],
    [572, 548],
    [602, 548],
    [609, 548],
    [611, 548],
    [647, 548],
    [650, 548],
    [596, 549],
    [611, 549],
    [551, 550],
    [552, 550],
    [557, 550],
    [562, 550],
    [566, 550],
    [579, 550],
    [584, 550],
    [587, 550],
    [601, 550],
    [617, 550],
    [622, 550],
    [625, 550],
    [635, 550],
    [654, 550],
    [660, 550],
    [665, 550],
    [681, 550],
    [682, 550],
    [690, 550],
    [691, 550],
    [697, 550],
    [702, 550],
    [705, 550],
    [709, 550],
    [726, 550],
    [738, 550],
    [556, 551],
    [596, 551],
    [610, 551],
    [638, 551],
    [653, 551],
    [606, 552],
    [617, 552],
    [635, 552],
    [639, 552],
    [660, 552],
    [666, 552],
    [670, 552],
    [735, 552],
    [736, 552],
    [753, 552],
    [776, 552],
    [777, 552],
    [794, 552],
    [802, 552],
    [662, 553],
    [573, 554],
    [602, 554],
    [562, 555],
    [646, 555],
    [558, 556],
    [564, 557],
    [608, 557],
    [617, 557],
    [651, 557],
    [655, 557],
    [696, 557],
    [710, 557],
    [736, 557],
    [560, 558],
    [568, 559],
    [568, 562],
    [572, 562],
    [574, 562],
    [575, 562],
    [576, 562],
    [581, 562],
    [586, 562],
    [588, 562],
    [589, 562],
    [593, 562],
    [600, 562],
    [616, 562],
    [619, 562],
    [622, 562],
    [629, 562],
    [635, 564],
    [659, 564],
    [665, 564],
    [678, 564],
    [687, 564],
    [697, 564],
    [702, 564],
    [735, 564],
    [736, 564],
    [738, 564],
    [745, 564],
    [751, 564],
    [772, 564],
    [635, 567],
    [692, 567],
    [697, 567],
    [699, 567],
    [763, 567],
    [771, 567],
    [572, 568],
    [575, 568],
    [579, 568],
    [581, 568],
    [588, 568],
    [604, 568],
    [619, 568],
    [575, 570],
    [586, 570],
    [770, 570],
    [842, 570],
    [573, 571],
    [617, 571],
    [655, 571],
    [690, 571],
    [768, 571],
    [602, 573],
    [610, 573],
    [634, 573],
    [635, 573],
    [666, 573],
    [681, 573],
    [642, 574],
    [582, 575],
    [593, 575],
    [619, 575],
    [635, 579],
    [586, 582],
    [588, 584],
    [597, 584],
    [678, 587],
    [751, 587],
    [597, 588],
    [606, 588],
    [609, 588],
    [617, 588],
    [651, 588],
    [653, 588],
    [660, 588],
    [661, 588],
    [672, 588],
    [681, 588],
    [708, 588],
    [615, 592],
    [645, 592],
    [637, 593],
    [652, 593],
    [611, 595],
    [653, 597],
    [676, 597],
    [666, 599],
    [739, 599],
    [691, 601],
    [702, 601],
    [627, 602],
    [650, 602],
    [617, 603],
    [609, 606],
    [620, 606],
    [627, 606],
    [628, 606],
    [678, 606],
    [691, 606],
    [750, 606],
    [770, 606],
    [622, 607],
    [714, 608],
    [627, 609],
    [630, 609],
    [640, 609],
    [617, 610],
    [660, 610],
    [681, 610],
    [627, 611],
    [738, 613],
    [762, 613],
    [646, 614],
    [758, 614],
    [856, 614],
    [656, 615],
    [646, 617],
    [662, 617],
    [678, 617],
    [681, 617],
    [692, 617],
    [702, 617],
    [705, 617],
    [709, 617],
    [710, 617],
    [727, 617],
    [735, 617],
    [736, 617],
    [750, 617],
    [751, 617],
    [771, 617],
    [675, 618],
    [625, 622],
    [666, 623],
    [634, 624],
    [636, 624],
    [655, 626],
    [675, 626],
    [640, 627],
    [682, 627],
    [661, 629],
    [665, 629],
    [663, 630],
    [636, 633],
    [646, 635],
    [659, 635],
    [666, 635],
    [668, 635],
    [677, 635],
    [681, 635],
    [690, 635],
    [698, 635],
    [702, 635],
    [705, 635],
    [706, 635],
    [707, 635],
    [710, 635],
    [714, 635],
    [718, 635],
    [723, 635],
    [725, 635],
    [735, 635],
    [736, 635],
    [738, 635],
    [755, 635],
    [762, 635],
    [769, 635],
    [777, 635],
    [780, 635],
    [786, 635],
    [787, 635],
    [647, 637],
    [660, 637],
    [679, 637],
    [687, 637],
    [697, 637],
    [748, 637],
    [665, 638],
    [645, 639],
    [660, 639],
    [889, 641],
    [769, 643],
    [771, 643],
    [692, 644],
    [729, 644],
    [655, 645],
    [668, 645],
    [678, 645],
    [655, 646],
    [666, 646],
    [690, 646],
    [697, 646],
    [707, 646],
    [709, 646],
    [723, 646],
    [736, 646],
    [749, 646],
    [763, 646],
    [764, 646],
    [800, 646],
    [789, 648],
    [669, 649],
    [705, 649],
    [666, 651],
    [674, 651],
    [660, 652],
    [673, 652],
    [694, 652],
    [717, 652],
    [763, 652],
    [834, 652],
    [671, 654],
    [697, 654],
    [723, 654],
    [679, 657],
    [710, 657],
    [734, 657],
    [709, 658],
    [660, 659],
    [679, 659],
    [690, 659],
    [712, 659],
    [736, 659],
    [778, 659],
    [791, 659],
    [818, 659],
    [841, 659],
    [845, 659],
    [853, 659],
    [856, 659],
    [673, 660],
    [683, 660],
    [691, 660],
    [710, 660],
    [735, 660],
    [744, 660],
    [746, 660],
    [672, 664],
    [736, 665],
    [758, 665],
    [813, 665],
    [668, 666],
    [678, 666],
    [702, 666],
    [704, 666],
    [710, 666],
    [731, 666],
    [736, 666],
    [754, 666],
    [762, 666],
    [784, 666],
    [683, 667],
    [670, 668],
    [678, 668],
    [681, 668],
    [717, 668],
    [720, 668],
    [736, 668],
    [745, 668],
    [750, 668],
    [802, 668],
    [819, 668],
    [723, 669],
    [728, 669],
    [720, 670],
    [738, 670],
    [762, 670],
    [771, 670],
    [693, 671],
    [735, 671],
    [697, 673],
    [702, 673],
    [729, 673],
    [776, 673],
    [795, 673],
    [681, 677],
    [691, 677],
    [697, 677],
    [722, 677],
    [706, 678],
    [718, 678],
    [736, 678],
    [753, 678],
    [839, 680],
    [690, 684],
    [738, 686],
    [688, 687],
    [690, 687],
    [697, 687],
    [709, 687],
    [743, 687],
    [751, 687],
    [829, 687],
    [851, 687],
    [884, 688],
    [736, 689],
    [709, 690],
    [757, 690],
    [762, 690],
    [763, 690],
    [769, 690],
    [773, 690],
    [807, 690],
    [817, 690],
    [818, 690],
    [822, 690],
    [832, 690],
    [845, 690],
    [853, 690],
    [856, 690],
    [692, 691],
    [705, 691],
    [735, 691],
    [756, 691],
    [770, 691],
    [812, 691],
    [707, 692],
    [736, 692],
    [750, 692],
    [752, 692],
    [784, 692],
    [790, 692],
    [795, 692],
    [804, 692],
    [817, 692],
    [743, 693],
    [769, 693],
    [825, 693],
    [832, 693],
    [839, 693],
    [853, 693],
    [743, 694],
    [738, 695],
    [710, 697],
    [714, 697],
    [720, 697],
    [736, 697],
    [738, 697],
    [750, 697],
    [758, 697],
    [763, 697],
    [769, 697],
    [777, 697],
    [784, 697],
    [802, 697],
    [805, 697],
    [816, 697],
    [817, 697],
    [818, 697],
    [819, 697],
    [829, 697],
    [832, 697],
    [837, 697],
    [846, 697],
    [852, 697],
    [700, 699],
    [779, 699],
    [723, 701],
    [708, 702],
    [714, 702],
    [738, 702],
    [709, 703],
    [710, 705],
    [738, 706],
    [709, 707],
    [727, 707],
    [736, 707],
    [762, 707],
    [763, 707],
    [781, 707],
    [804, 707],
    [822, 707],
    [839, 707],
    [842, 707],
    [845, 707],
    [852, 707],
    [856, 707],
    [860, 707],
    [877, 707],
    [719, 708],
    [724, 709],
    [736, 709],
    [750, 709],
    [763, 709],
    [769, 709],
    [773, 709],
    [792, 709],
    [802, 709],
    [804, 709],
    [807, 709],
    [721, 710],
    [724, 710],
    [725, 710],
    [819, 711],
    [713, 712],
    [771, 715],
    [783, 715],
    [736, 716],
    [820, 716],
    [724, 717],
    [773, 717],
    [776, 717],
    [736, 718],
    [769, 718],
    [724, 719],
    [856, 722],
    [745, 724],
    [791, 725],
    [826, 725],
    [836, 725],
    [761, 729],
    [781, 729],
    [791, 729],
    [734, 730],
    [750, 732],
    [740, 733],
    [787, 735],
    [738, 736],
    [749, 736],
    [750, 736],
    [753, 736],
    [762, 736],
    [763, 736],
    [764, 736],
    [769, 736],
    [773, 736],
    [776, 736],
    [777, 736],
    [780, 736],
    [786, 736],
    [789, 736],
    [801, 736],
    [803, 736],
    [804, 736],
    [816, 736],
    [822, 736],
    [837, 736],
    [843, 736],
    [856, 737],
    [878, 737],
    [750, 738],
    [753, 738],
    [771, 738],
    [802, 738],
    [804, 738],
    [827, 738],
    [763, 740],
    [803, 740],
    [807, 740],
    [846, 740],
    [750, 742],
    [762, 742],
    [773, 742],
    [777, 742],
    [831, 742],
    [835, 742],
    [885, 743],
    [816, 747],
    [827, 747],
    [760, 749],
    [762, 750],
    [763, 750],
    [777, 750],
    [791, 750],
    [801, 750],
    [802, 750],
    [804, 750],
    [806, 750],
    [807, 750],
    [816, 750],
    [825, 750],
    [791, 751],
    [798, 751],
    [762, 753],
    [763, 753],
    [777, 753],
    [829, 753],
    [803, 755],
    [758, 757],
    [807, 757],
    [837, 757],
    [840, 757],
    [841, 757],
    [856, 757],
    [781, 759],
    [784, 761],
    [795, 761],
    [763, 762],
    [765, 762],
    [768, 762],
    [769, 762],
    [773, 762],
    [774, 762],
    [777, 762],
    [781, 762],
    [785, 762],
    [800, 762],
    [801, 762],
    [802, 762],
    [807, 762],
    [816, 762],
    [819, 762],
    [822, 762],
    [823, 762],
    [827, 762],
    [832, 762],
    [833, 762],
    [839, 762],
    [841, 762],
    [843, 762],
    [845, 762],
    [846, 762],
    [853, 762],
    [855, 762],
    [856, 762],
    [874, 762],
    [769, 763],
    [803, 763],
    [815, 763],
    [784, 764],
    [778, 765],
    [803, 765],
    [828, 765],
    [839, 765],
    [859, 765],
    [867, 765],
    [873, 765],
    [807, 766],
    [819, 767],
    [844, 767],
    [769, 768],
    [822, 768],
    [771, 769],
    [781, 769],
    [796, 769],
    [803, 769],
    [807, 769],
    [822, 769],
    [829, 769],
    [837, 769],
    [840, 769],
    [807, 771],
    [813, 771],
    [838, 771],
    [830, 772],
    [781, 773],
    [840, 773],
    [815, 774],
    [825, 774],
    [807, 775],
    [777, 776],
    [816, 776],
    [878, 776],
    [802, 777],
    [829, 777],
    [839, 778],
    [838, 779],
    [781, 780],
    [816, 780],
    [819, 780],
    [788, 781],
    [807, 781],
    [817, 781],
    [822, 781],
    [841, 781],
    [843, 781],
    [853, 781],
    [856, 781],
    [807, 782],
    [791, 784],
    [805, 784],
    [807, 784],
    [809, 784],
    [838, 785],
    [845, 785],
    [856, 785],
    [858, 785],
    [864, 785],
    [873, 785],
    [875, 785],
    [856, 786],
    [864, 786],
    [874, 786],
    [803, 788],
    [829, 789],
    [819, 794],
    [803, 795],
    [798, 797],
    [853, 799],
    [864, 799],
    [829, 800],
    [856, 800],
    [803, 801],
    [807, 801],
    [840, 801],
    [829, 802],
    [804, 803],
    [818, 803],
    [820, 803],
    [823, 803],
    [826, 803],
    [833, 803],
    [839, 803],
    [865, 803],
    [816, 804],
    [821, 804],
    [859, 805],
    [807, 806],
    [831, 806],
    [814, 807],
    [819, 807],
    [840, 807],
    [809, 808],
    [818, 816],
    [828, 818],
    [859, 818],
    [820, 819],
    [848, 819],
    [856, 819],
    [847, 820],
    [859, 820],
    [862, 822],
    [863, 822],
    [864, 822],
    [840, 823],
    [843, 823],
    [845, 824],
    [888, 824],
    [840, 826],
    [853, 826],
    [840, 829],
    [847, 829],
    [849, 829],
    [851, 829],
    [840, 832],
    [843, 832],
    [856, 832],
    [842, 839],
    [843, 839],
    [853, 839],
    [854, 839],
    [855, 839],
    [858, 839],
    [860, 839],
    [861, 839],
    [847, 841],
    [850, 841],
    [853, 841],
    [855, 841],
    [856, 841],
    [858, 841],
    [859, 841],
    [865, 841],
    [882, 841],
    [853, 843],
    [854, 843],
    [869, 845],
    [879, 845],
    [881, 847],
    [887, 850],
    [870, 851],
    [853, 852],
    [857, 852],
    [872, 852],
    [854, 853],
    [856, 853],
    [866, 853],
    [872, 853],
    [858, 856],
    [859, 856],
    [864, 856],
    [866, 856],
    [874, 856],
    [876, 856],
    [882, 856],
    [883, 856],
    [869, 859],
    [873, 859],
    [872, 861],
    [882, 862],
    [870, 864],
    [874, 864],
    [885, 884],
  ];

  try {
    graphModule._drop(graphName, true);
  } catch (e) {}
  graphModule._create(
    graphName,
    [graphModule._relation(ename, vname, vname)],
    [],
    { smartGraphAttribute: "id", numberOfShards: numberOfShards }
  );

  var vs = [];
  var vids;

  for (let i = 0; i < N; i++) {
    vs.push({ id: "" + i });
  }
  vids = db._collection(vname).save(vs);

  var es = [];
  socialNetwork.forEach(function (item, index) {
    es.push({
      _from: vids[item[0] - 1]._id,
      _to: vids[item[1] - 1]._id,
      cost: Math.random() * 10000,
    });
  });

  db._collection(ename).save(es);

  return { name: graphName, vname: vname, ename: ename };
}

exports.create_wiki_vote_graph = createWikiVoteGraph;
exports.create_line_graph = createLineGraph;
exports.create_pagerank_graph = createPageRankGraph;
exports.create_circle = createCircle;
