#*******************************************************************************
# E.S.O. - VLT project
#
# "@(#) $Id: configure.tcl 290518 2016-11-24 15:14:57Z pbaksai $"
#
# who       when        what
# --------  ----------  --------------------------------------------------------
# jpritcha  2016/10/01  Removed spurious debugging terminal output left in after 
#                       previous update
# pbaksai   2015/02/07  CCB-000755: Support for 64 bits.
#                       - Fixed font XLFD names.
# rschmutz  2004-07-16  PPRS 13423: always pop-up "Argus Settings" panel.
# lsanzana  11/06/04    Verify checksum of PAf files.
# rschmutz  02/02/03    PPRS 8632: DoList(): CompressUnalloc() deleted.
# rschmutz  29/01/03    PPRS 8670: enforce validity check over HA range
#                       (+ Francesca easter egg in expert mode)
# rschmutz  29/01/03    PPRS 8668: DoOpenPAF: set SelectedGuideName.
# rschmutz  28/01/03    PPRS 8648: if manual fibre allocation fails, then
#		        call DeallocateFib to keep correct fibre statistics.
# rschmutz  28/01/03    PPRS 8651: display CMM version instead of date.
# akaufer   26/01/03    proc DoList: Guide star added to the parameter list 
#                                    for skycatListing
# rschmutz  24/11/02    this header created - previous history see below.
#
#*******************************************************************************
#
#                           C o n f i g u r e . t c l
#   Filename:
#       configure.tcl
#
#   Function:
#       Tcl script for the AAO fibre instrument configuration user interface.
#
#   Description:
#       The 'configure' program is part of the observation preparation
#       software used by a number of instruments built by the AAO. These
#       include the original '2dF', the more recent '6dF' and the 'OzPoz'
#       fibre positioner built for the 'FLAMES' instrument for ESO. All of
#       these use a robotic fibre positioner to place optical fibres in the
#       focal plane of the telescope so that light from a number of target
#       objects may be fed to spectrographs or other instruments. These all
#       need a configuration phase in which the fibres are allocated to
#       targets in such a way that the fibre configuration is physically
#       possible (fibres and the 'buttons' that carry them do not collide,
#       and fibres are not bent past their allowed tolerances, for example),
#       and the best coverage possible is achieved (as few unallocated 
#       targets as possible, without any strange artifacts in the distribution
#       of allocated targets, for example).
#
#   Overview:
#       The 'configure' program has a number of distinct sections:
#
#       There is an underlying allocation 'engine', written in C. This has
#       two separate layers, an algorithm-dependent layer that does the 
#       actual allocation of fibres to targets, and a 'housekeeping' 
#       layer that provides an interface to the higher levels of the program
#       and provides a number of utility functions that are needed by any
#       allocation algorithm. This split is supposed to make it relatively
#       easy to modify or completly change the algorithm used to do the
#       allocation. It has the side-effect that the higher levels know very
#       little of the details of the algorithm. For example, although all
#       algorithms can make use of parameters set by the higher levels (and
#       ultimately by the user), the prompts and details of these parameters
#       vary with the algorithm used, and so the higher levels have to 
#       determine such things as parameter defaults, prompts, and types
#       through enquiry functions supplied, ultimately, by the algorithm-
#       specific code. This makes some of the interface code seem rather
#       abstract, since it is working with parameters that are only known
#       to it in a very indirect way.
#
#       There is a user-interface, written in Tcl/Tk. This exists to allow
#       the user to control the application of the allocation algorithm, for
#       example by selecting the configuration files to be processed and by
#       providing dialogues that allow the user to set the various parameters
#       used by the allocation algorithm, and to see the results. After seeing
#       the results of an allocation, the user may want to modify various
#       algorithm parameters and repeat the allocation, or may even want to
#       exercise more direct control, for example by explicitly selecting a
#       a target and fibre and insisting that that fibre be allocated to that
#       particular target. An important part of the user interface is the
#       'mimic' display that provides a graphical representation of the
#       target field and shows the fibres allocated to the various targets.
#
#       Acting as an interface between the Tcl/Tk user-interface code, and
#       the 'pure' C of the allocation algorithm layers, is a layer of C
#       code that implements a relatively large number of new Tcl commands.
#       These commands allow a Tcl program to access the allocation code and
#       provide a large number of additional utilities.
#
#       This particular file contains the Tcl code for the main part of the
#       user interface.
#
#       This script is loaded on startup of the 'configure' program and 
#       contains most of the code for the configuration user interface. 
#       The code makes use of a large number of additional Tcl commands,
#       defined in the configure.c file. The user interface maintains a
#       'mimic' window which shows the positions of all the target objects
#       and the fibres. Most of the code for the mimic window is contained
#       in the mimic.tcl file.
#
#   Authors:
#       Jeremy Bailey, AAO. (JAB)
#       Gavin Dalton, University of Oxford (GBD)
#       Tony Farrell, AAO (TJF)
#       Keith Shortridge, AAO (KS)  
#
#   Copyright (c)  Anglo-Australian Telescope Board, 1995 - 2002.
#   Permission granted for use for non-commercial purposes.
#
#   Sccs Id:     configure.tcl, Release 1.41, 11/21/02
#
# History:
#     The original version of this program (only supporting 2dF) dates from
#     1993. The history section of the code that dates from 1993 to 2001
#     is long and - given the substantial rewriting to support multiple
#     instruments such as 6dF and FLAMES - somewhat irrelevant. It has been
#     removed from this file and saved in the file configure.history, which
#     can be consulted as a historical document. The following section
#     begins afresh with the first proper OzPoz release.
#
#     26-Sep-2001 -  KS - First release with proper OzPoz support. Commenting
#                         revised.
#     31-Oct-2001 -  KS - Modified ApplyArgusSettings{} to restore Argus
#                         settings if rejected, and InvalidateAlloc{} to
#                         test properly for allocated fibres. Same change made
#                         to MoveCenterDialogue{}
#      5-Nov-2001 -  KS - Introduced NullAction{} to prevent closing of the
#                         handholder basic sequence window. Changing fibre
#                         combination now invalidates any current allocation.
#                         Expert mode only available if 'expert allowed' flag
#                         is set at the C level.
#     19-Nov-2001 -  KS - The allocation dialogue now presents the user with
#                         entry boxes to specify the number of fibres to be
#                         allocated to sky targets on the basis of individual
#                         fibre type.
#     26-Nov-2001 -  KS - Re-selecting the same fibre combination as that
#                         used for the last allocation now ticks the button
#                         in the basic sequence dialogue.
#     28-Nov-2001 -  KS - "Allocate Sky Grid" is now "Generate Sky Grid" and
#                         does not attempt to allocate the new sky targets.
#      5-Dec-2001 -  KS - Wavelength is no longer displayed in the main
#                         control panel for FLAMES. Changing the ARGUS angle
#                         clears any VLT guide probe selection. Basic
#                         sequence now varies slightly depending on instrument
#                         being used. Wavelengths are now displayed in nm
#                         for FLAMES, Angstroms for other instruments.
#      3-Jan-2002 -  KS - Manual fibre allocation is now checked using the
#                         same constraints as automatic allocation.
#     30-Jan-2002 -  KS - Added Skycat output option to List dialogue.
#      5-Feb-2002 -  KS - Reworked the change field center code so it provides
#                         a much more sensible interface.
#     12-Feb-2002 -  KS - Removed all use of DtcluFDialog - used the standard
#                         tk_getSaveFile and tk_getOpenFile instead.
#     11-Apr-2002 -  KS - Added support for -use_all_fibres command line
#                         option.
#     16-Apr-2002 -  KS - Fixed some bugs connected with the 'Save As SDS File'
#                         menu option. Added the -Z command line option, and
#                         made both it and -z work properly.
#      4-Jun-2002 -  KS - Modified use of PAFFileName.
#      1-Aug-2002 -  KS - Check for shadowed targets now performed after a 
#                         sky grid is generated.
#      5-Aug-2002 -  KS - Expert mode now always allowed in the 6dF/2dF cases.
#      8-Aug-2002 -  KS - WavelengthCheck now initialised properly.
#     22-Aug-2002 -  KS - Counts updated properly after sky position or grid
#                         added - ensures VLT guide probe obscuration is
#                         handled properly for the new sky positions.
#     28-Aug-2002 -  KS - ReadPAF now sets the fibre combination.
#      3-Sep-2002 -  KS - GetFibreStats{} now returns available object and sky
#                         counts.
#      5-Sep-2002 -  KS - Available target statistics now included in main
#                         panel. New global SelectedGuideName now used to
#                         identify guide star to WritePAF{}. Changing the field
#                         center now updates the main panel position display.
#                         Introduced current directory into all open/save file
#                         dialogues.
#     10-Sep-2002 -  KS - AllocSky1 and AllocSky2 removed from code - they are
#                         no longer relevant and can confuse the total count
#                         of allocated fibres.
#     25-Sep-2002 -  KS - Fixed bug causing DoOpenSds{} to fail. Introduced
#                         separate input and output directories.
#     02-Oct-2002 -  KS - Sky counts now correct after sky grid allocated.
#     23-Oct-2002 -  KS - VLT guide star selection now maintained through
#                         change of ARGUS angle.
#     04-Nov-2002 -  KS - Added ESO logo to menubar.
#     07-Nov-2002 -  KS - Target positions now updated when fibre
#                         combination is changed.
#     18-Nov-2002 -  KS - Mimic now redrawn with ARGUS settings as soon as
#                         an ARGUS combination is selected.
#     21-Nov-2002 -  KS - For FLAMES, setting the wavelength or the time of
#                         the observation are now expert mode options.
#     22-Nov-2002 -  AK - Paranalization:
#                         ESO and AAO logos removed 
#                         Menu and Button labels renamed:
#                            Open             -> Open input file
#                            Open file        -> Open input file
#                            Save as PAF file -> Save Target Setup file
#                            Open PAF file    -> Open Target Setup file
#                            Allocate         -> Allocate fibres
#                            Input lists      -> Input target list
#                         Window titles changed:
#                            AAO Fibre Configuration - [Instrument] 
#                                -> FPOSS: [Instrument] Observation Support Software
#                            CONFIGURE -> FPOSS: CONFIGURE
#                            Basic Sequence -> FPOSS: Basic Sequence
#                            Fibre Configuration PAF File -> Target Setup File 
#                         Report only filename without path in 'Input File' field                  #
#-------------------------------------------------------------------------- 
#
#                          G l o b a l s
#
#  This section should list all the global variables used in this file. This 
#  section isn't necessary for the program to work, but it provides a 
#  convenient place to document (briefly) the use to which these variables are
#  put. Each procedure that uses globals should have global declarations that
#  duplicate some of the comments here, but usually only the first line is
#  included. So this section can contain rather more details about a given
#  global than are included in the procedures that use it. If the use of a 
#  global is unclear from the actual procedure code, looking here may help.

global AllocationStatus   ;# Describes current allocation status.
                          ;# This is used to provide feedback to the basic
                          ;# sequence display panel.
global AllocGui           ;# The number of allocated guide targets.
global AllocObj           ;# Number of allocated objects
global AllocSky           ;# Number of sky targets allocated
global ArrowHighlighting  ;# Flag set if arrow highlighting is enabled
global ArgusInUse         ;# Argus status text
global ArgusErrorText     ;# Error text used in ARGUS settings display
global ArgusAngle         ;# Argus position angle in degrees. Note that this
                          ;# is for the user interface, so is the value used
                          ;# by FLAMES for the actual angle of ARGUS itself,
                          ;# not the plate - see configure.c for details, and
                          ;# note that this Tcl ArgusAngle is not the same as
                          ;# the C global ArgusAngle - it has the offset applied
global ArgusScale         ;# Argus scale - 1:1 or 1:1.67
global ArgusStatus        ;# Summary of ARGUS settings
global BatchMode          ;# True if the program is running in batch mode.
                          ;# In batch mode, we assume there is no display
                          ;# available, and Tk is not initialised and so
                          ;# no Tk operations are allowed. The flag is set
                          ;# on initialisation on the basis of the command
                          ;# line options. 
global BufferedDisplay    ;# True if the display output is being buffered
global CheckMode          ;# Non-zero if a check should attempt to reallocate.
                          ;# This is tied to a check button on the hour angle
                          ;# range check dialogue.
global Cfid               ;# Sds Id for the top level of the current
                          ;# configuration structure. 
                          ;# This is the structure containing the "fieldData",
                          ;# "unallocGuide", "unallocObject" and "objects" 
                          ;# sub-structures.
global ConfigDirDef       ;# Directory used for configuration files
global ConfigFile         ;# Currently displayed configuration file (or plate)
global ConfigFilesDef     ;# Directory containing configuration files, such
                          ;# as tdFdistortion.sds.
global config_font_1      ;# Font used in dialogues
global config_font_2      ;# Font (smaller) used in dialogues
global ConfigPlate        ;# Plate currently being configured (0 or 1)
global CurrentFibreCombo  ;# Describes current fibre combination.
global CurrentOutDirectory;# Current directory for saving files
global CurrentInDirectory ;# Current directory for reading files
global DSSOutMode         ;# Set if DSS output mode has been selected
                          ;# (this is set if the -z command line option is used)
global DoAllocation       ;# Indicates the -s option was specified.
global Edited             ;# True once the configuration has been changed
global errorCode          ;# Error description (generated by Tcl)
global errorInfo          ;# Stack trace describing an error (generated by Tcl)
global ExpertMode         ;# True if the system has been set to 'expert mode'.
                          ;# This is normally false, but can be set true by
                          ;# the 'expert mode' check box in the 'options' menu.
                          ;# In expert mode, additional options are made
                          ;# available, and some dialogues are modified.
                          ;# features appear in GUI
global FibreCombination   ;# Fibre combination selected for use. This is an
                          ;# index value. -1 indicates that what has been
                          ;# selected is not a known configuration. Values from
                          ;# 0 up are index values that must match the order
                          ;# returned by [FibreCombos].
global FibreComboMode     ;# Mode keyword for Fibre combination selected.
                          ;# This is written into the PAF file for FLAMES use.
                          ;# At present other instruments do not use this.
global FibreTypeFlags     ;# Array.  Indicates if each fibre type
                          ;# is enabled. See SelectFibreType for details.
global Gtext              ;# Text displayed in a 'progress' bar window
global HighlightClick     ;# True if clicked position is to be highlit.
global HighMagEntry       ;# High magnitude as given in dialogue. This is the
                          ;# entry variable associated with the dialogue box.
global HighMagLimit       ;# High magnitude limit applied to input lists. This
                          ;# is the value used when input lists are read, and
                          ;# is set to HighMagEntry when 'Apply' is clicked.
global ImportFile         ;# Name of import file if such a file is used,
                          ;# (set if the -i command line option is used)
global ImportMode         ;# True if import mode is used (-i on command line)
global InitialPlate       ;# Initial plate number - specified on command
                          ;# line with -p option.
global LowMagEntry        ;# Low magnitude as given in dialogue. This is the
                          ;# entry variable associated with the dialogue box.
global LowMagLimit        ;# Low magnitude limit applied to input lists. This
                          ;# is the value used when input lists are read, and
                          ;# is set to LowMagEntry when 'Apply' is clicked.
global MagFiltering       ;# State of magnitude filtering: "On" or "Off".
global MagFilterText      ;# Describes magnitude filter limits. A line of text
                          ;# that describes the current magnitude filter.
global MagErrorText       ;# Describes any dialogue entry error. This is a line
                          ;# of text describing any error in the limits.
global MimicClicked       ;# Indicates that mimic display was clicked upon.
global SaveFile           ;# Name of the file to be saved
global SelectedFibre      ;# Currently selected fibre (pivot) number - starts
                          ;# from 1 - zero => no fibre selected.
global SelectedObject     ;# Index of selected target object - starts at one.
                          ;# This is the index in one of the sub-structures
                          ;# "unallocGuide", "unallocObject" or "objects" that
                          ;# are contained in the configuration structure whose
                          ;# SdsId is held in Cfid.
global SelectedGuideName  ;# Name of selected guide star.
global SelectedGuidePivot ;# Indicates which of the two possible guide pivot
                          ;# positions has been selected.
                          ;# Zero if no pivot selected, 1 or 2 when a
                          ;# selection has been made.
global SelectedGuideObject;# Which guide object is selected ( 0=> none).
                          ;# This is the index (starting from 1) of the
                          ;# selected target in the "unallocGuide"
                          ;# section of the overall SDS structure whose SDS id
                          ;# is held in Cfid.
global SelectedGuideOrient;# Guide probe orientation - "POS" or "NEG".
global SelectedTag        ;# Tag associated with selected target object.
                          ;# This is one of "uguide", "uobj", or "obj" depending
                          ;# on whether the target is currently held in the
                          ;# "unallocGuide", "unallocObject" or "objects"
                          ;# sub-structure of the configuration structure.
global ShowEmptyFlag      ;# True if empty fibre positions are to be shown.
global TargetCriteriaFlags;# Flags set to show target selection criteria.
                          ;# Elements 0 through TargetNumCriteria-1 are the
                          ;# instrument-dependent criteria, the next element
                          ;# (TargetNumCriteria) is the 'by priority' flag,
                          ;# and element TargetNumCriteria+1 is the 'by
                          ;# allocation' flag.
global TargetHighPriority ;# Entry variable for selected top priority
global TargetLowPriority  ;# Entry variable for selected low priority
global TargetByAllocation ;# Entry variable used to select by allocation
                          ;# Set to 1 is selection is 'allocated', to zero
                          ;# if 'unallocated' was selected.
global TargetResult       ;# Number of selected targets
global TargetNumCriteria  ;# No. of instrument-dependent selection criteria
global TargetScale        ;# Scale factor used to magnify targets.
global TickDetails        ;# Details of ticks in the basic sequence dialogue.
                          ;# This is an array with a first index that identifies
                          ;# the element in the dialogue "MagFilter", "OpenFile"
                          ;# etc, and a second index that is one of:
                          ;# widget    - the widget used to display the tick
                          ;# displayed - flags if this element is in use
                          ;# level     - level in the basic sequence
                          ;# ticked    - set if this element has been ticked
                          ;# next      - 1st index value for next entry down
                          ;# The 'ticked' fields can be used to see what stages
                          ;# of the sequence have been completed so far.
global TkAvailable        ;# Set if Tk operations are permitted.
                          ;# This is simply the inverse of the BatchMode global
                          ;# but its use makes the point clearer.
global UnallocGui         ;# The number of unallocated guide targets.
                          ;# This is the number of items in the "unallocGuide" 
                          ;# sub-structure of the configuration structure.
global UnallocObj         ;# The number of unallocated non-guide, non-sky,
                          ;# targets.
global UnallocSky         ;# The number of unallocated sky targets.
                          ;# UnallocObj + UnallocSky gives the number of items
                          ;# in the "unallocObject" sub-structure of the 
                          ;# configuration structure.
global UsePAFFiles        ;# True if the default output file format is PAF.
global UtDateSpec         ;# Initial UT date/time specified on command line.
global WAVELENGTH         ;# Observing wavelength in Angstroms.
global WavelengthInA      ;# True if wavelength is displayed in Angstroms.
global WavelengthShown    ;# The wavelength displayed (in Angstroms or nm).
global WavelengthCheck    ;# Non-zero if wavelength ranges are to be checked.
global zoom               ;# Current zoom factor - 1 is normal.
global CheckStatus        ;# Describes current check status

# -----------------------------------------------------------------------------
#
#                             S e t u p
#
#   Procedure Setup{} creates the main window, the menus and the mimic
#   display canvas widget and performs other initialization. It is run
#   as part of the initialisation code when this file is loaded.
  
proc Setup {} {

    #   Global variables used:

    global AllocGui        ;# Number of guide targets allocated
    global AllocObj        ;# Number of ordinary targets allocated
    global AllocSky        ;# Number of sky targets allocated
    global BufferedDisplay ;# True if the display output is being buffered
    global CDEC            ;# Tied to central Dec field in main control panel
    global CHA             ;# Tied to Hour angle field in main control panel
    global ConfigDirDef    ;# Directory used for configuration files
    global ConfigFile      ;# Currently displayed configuration file (or plate)
    global ConfigPlate     ;# Plate currently being configured (0 or 1)
    global CRA             ;# Tied to central RA field in main control panel
    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure.
    global CZD             ;# Tied to ZD field in main control panel
    global DSSOutMode      ;# Set if DSS output mode has been selected
    global Edited          ;# True once the configuration has been changed
    global errorCode       ;# Error description (generated by Tcl)
    global errorInfo       ;# Stack trace describing an error (generated by Tcl)
    global ExpertMode      ;# True if the system has been set to 'expert mode'.
    global FieldName       ;# The label associated with the current field
    global Gtext           ;# Text displayed in a 'progress' bar window
    global ImportFile      ;# Name of import file if such a file is used,
    global ImportMode      ;# True if import mode is used (-i on command line)
    global InputFile       ;# The name of the file containing target data
    global OutputFileStatus;# Description of output file status
    global OutputFileName  ;# Output file name
    global SelectedFibre   ;# Currently selected fibre (pivot) number - starts
                           ;# from 1 - zero => no fibre selected.
    global TargetScale     ;# Scale factor used to magnify targets.
    global TkAvailable     ;# Set if Tk operations are permitted.
    global InitialPlate    ;# Initial plate number - specified on command
                           ;# line with -p option.
    global DoAllocation    ;# Indicates the -s option was specified.
    global UnallocGui      ;# Number of guide targets unallocated
    global UnallocObj      ;# Number of ordinary targets unallocated
    global UnallocSky      ;# Number of sky targets unallocated
    global UsePAFFiles     ;# True if the default output file format is PAF.
    global UtDateSpec      ;# Initial UT date/time specified on command line.
    global UTDATE          ;# Tied to UT date field in main control panel
    global UTTIME          ;# Tied to UT time field in main control panel
    global WAVELENGTH      ;# Observing wavelength in Angstroms.
    global WavelengthInA   ;# True if wavelength is displayed in Angstroms.
    global WavelengthShown ;# The wavelength displayed (in Angstroms or nm).
    global validHourAngle  ;# Validated hour angle range.
    global CheckStatus     ;# Describes current check status
    global ListSaved       ;# Flag: current config. list is saved
    global ListTargetFile  ;# List file saved: proposed target setup file name.
        
    if { !$TkAvailable } return
    
    set ExpertMode 0
    set validHourAngle 0
    set CheckStatus "Not checked"
    set ListSaved 0
    
    set Cfid 0
    set TargetScale 1.0
    set Gtext " "
    
    #  We use DtcluCreateMain{} to create the main frame for the program.
    #  This creates a main window containing three frames: .menubar,
    #  .mainframe, and .messframe.
    #  It also packs three menu buttons into .menubar:
    #     [.file.menu] containing an Exit buttons
    #     [.commands.menu] containing nothing
    #     [.options.menu] containing nothing.

    DtcluCreateMain

    # If the main window is deleted, use this to try application exit.
    
    wm protocol . WM_DELETE_WINDOW ExitApp
    
    pack forget .options .commands .file
    
    #  Some initial values that depend on the instrument
    
    set WAVELENGTH [GetWavelength]
    set UsePAFFiles 0
    set WavelengthInA 1
    set WavelengthShown $WAVELENGTH
    if { [Instrument] == "FLAMES"} { 
        set UsePAFFiles 1
        set WavelengthInA 0
        set WavelengthShown [expr $WAVELENGTH / 10]
    }

    #  Create the menu bar.  Note that the menus created here only contain 
    #  the non-expert mode options. Selecting the Expert Mode option invokes
    #  the procedure SetExpertMode{} which adds the expert mode commands when
    #  expert mode is selected and deletes them when non-expert mode is 
    #  selected. Assorted events can cause the menu items to be disabled or
    #  enabled.

    #  Create the File menu (contains Exit and Print commands). We try to keep
    #  the options down to a minimum for FLAMES, since that's what ESO want. It
    #  would be nice not to have to use an explicit test for FLAMES to do
    #  this, however.

    # no logos in ESO GUIs
    #if { [Instrument] == "FLAMES"} { 
    #   label .esologo -bitmap @[TranslateName CONFIGURE_DIR -file eso-logo.bmp \
    #                  -default [file join $ConfigDirDef eso-logo.bmp]]
    #   pack .esologo -side left -in .menubar
    #}

    menubutton .menubar.file -text File -underline 0
    pack .menubar.file -side left -padx 2m
    menu .menubar.file.menu -postcommand FilePost
    .menubar.file.menu add command -label "Open input file ... (^O)" -command OpenAscii
    
    if { $UsePAFFiles } {
        .menubar.file.menu add command -label "Open Target Setup file..." -command OpenPAF
        .menubar.file.menu add command -label "Save Target Setup file..." \
                                      -command SaveAsPAF -state disabled
    } else {
        .menubar.file.menu add command -label "Open SDS..." -command OpenSDS
        .menubar.file.menu add command -label "Merge..." -command MergeFiles
        .menubar.file.menu add command -label "Save (^S)" -command SaveFile \
            -state disabled
        .menubar.file.menu add command -label "Save As Sds file..." \
                               -command SaveAsSDS -state disabled
        .menubar.file.menu add command -label "Save Alloc to UnAlloc..." \
                    -command SaveAllocToUnalloc -state disabled
    }
    .menubar.file.menu add command -label "List..." -command List -state \
        disabled
    .menubar.file.menu add command -label "Print..." -command PrintField \
        -state disabled
    .menubar.file.menu add command -label "Exit ^E" -command ExitApp
    .menubar.file configure -menu .menubar.file.menu
    
    #  Create the View menu (controls what is displayed in the mimic display)
    #  All of this is now relegated to the 'Select targets' dialogue box, and
    #  all the view menu contains is the option to show this box.
    
    menubutton .menubar.display -text View -underline 0
    pack .menubar.display -side left -padx 2m
    menu .menubar.display.menu
    .menubar.display.menu add command -label "Select targets..." \
        -command HighlightTargets -state disabled
    .menubar.display.menu add command -label "Use grey shades for display" \
        -command "SetGreyColours 1" -state normal
    .menubar.display.menu add command -label "Use colours for display" \
        -command "SetGreyColours 0" -state disabled
    if { [Instrument] == "FLAMES"} { 
       .menubar.display.menu add command -label "Show ARGUS settings" \
                                 -command "ArgusSettings" -state normal
    }
    .menubar.display configure -menu .menubar.display.menu
       
    #  Create the Options menu (used for parameter setting )

    menubutton .menubar.options -text Options -underline 0
    pack .menubar.options -side left -padx 2m
    menu .menubar.options.menu
    
    .menubar.options.menu configure -postcommand OptionsPost
    if { [NumFields] > 1 } {
        .menubar.options.menu add command -label "Set Field Plate..." \
            -command SetFieldPlate
    }
    .menubar.options.menu add command -label "Set magnitude filter..." \
        -command MagnitudeFilter -state normal
    .menubar.options.menu add command -label "Change Field Label" \
        -command RenameField -state disabled
    if { [Instrument] != "FLAMES"} { 
        .menubar.options.menu add command -label "Change Observation Date" \
            -command ChangeObsDate -state disabled
    }
    .menubar.options.menu add command -label "Select Fibre Combination" \
        -command SelectFibreCombo -state disabled
    if { [ExpertAllowed] || ([Instrument] != "FLAMES") } {
        .menubar.options.menu add checkbutton -label "Expert" \
            -variable ExpertMode -offvalue 0 -onvalue 1 -command SetExpertMode
    }
    
    .menubar.options configure -menu .menubar.options.menu
    

    menubutton .menubar.commands -text Commands -underline 0
    pack .menubar.commands -in .menubar -side left -padx 2m
    menu .menubar.commands.menu
    
    .menubar.commands.menu configure -postcommand CommandsPost
    
    .menubar.commands.menu add command -label "Allocate... (F4)" \
        -command Allocate -state disabled
    .menubar.commands.menu add command -label "Auto-Reallocate..." \
        -command ReAllocate -state disabled
    .menubar.commands.menu add command -label "Check Allocation" \
        -command CheckAlloc -state disabled
    .menubar.commands.menu add command -label "Check over HA range...(F2)" \
        -command Check -state disabled
    .menubar.commands.menu add command -label "Generate Sky Grid..." \
        -command GenerateSkyGrid -state disabled
    .menubar.commands.menu add command -label "Uncross Fibres..." \
        -command UncrossFibres -state disabled
    .menubar.commands.menu add command -label "Set Hour Angle..." \
        -command SetHA -state disabled
    if { [Instrument] != "FLAMES"} { 
       .menubar.commands.menu add command -label "Set Wavelength..." \
             -command SetWave -state disabled
    }
    .menubar.commands.menu add command -label "Remove Allocations..." \
        -command RemoveAlloc -state disabled
    .menubar.commands.menu add command -label "Allocate Fibre (F3)" \
        -command AllocateFib -state disabled
    .menubar.commands.menu add command -label "Deallocate Fibre (Del)" \
        -command DeallocateFib -state disabled
    .menubar.commands.menu add command -label "Show Fibre Info..." \
        -command ShowFibre -state disabled
    .menubar.commands.menu add command -label "Set field center..." \
        -command MoveFieldCenter -state disabled
    .menubar.commands.menu add command -label "Deallocate Broken Fibres" \
        -command DeallocateBrokenFibres -state disabled
    .menubar.commands.menu add command -label "Deallocate Sky Fibres" \
        -command DeallocateSkyFibres -state disabled
    .menubar.commands configure -menu .menubar.commands.menu
    
    #  Create the Zoom menu (Sets the mimic display zoom factor)
    
    menubutton .menubar.zoom -text Zoom -underline 0
    pack .menubar.zoom -side left -padx 2m
    menu .menubar.zoom.menu -postcommand ZoomPost
    .menubar.zoom.menu add command -label " x 0.5" -command "ZoomMimic 0.5" \
        -state disabled
    .menubar.zoom.menu add command -label " x 0.8" -command "ZoomMimic 0.8" \
        -state disabled
    .menubar.zoom.menu add command -label "Normal" -command "ZoomMimic 1" \
        -state disabled
    .menubar.zoom.menu add command -label " x 2 " -command "ZoomMimic 2" \
        -state disabled
    .menubar.zoom.menu add command -label " x 4 " -command "ZoomMimic 4" \
        -state disabled
    .menubar.zoom.menu add command -label " x 8 " -command "ZoomMimic 8" \
        -state disabled
    .menubar.zoom configure -menu .menubar.zoom.menu
    

    #  Set up the menu bar for keyboard traversal

    #tk_menuBar .menubar .menubar.file .menubar.display .menubar.options \
    #    .menubar.commands .menubar.zoom

    #  no logos in ESO GUIs
    #  Put the AAO logo in the right hand end of the menu bar
    # label .logo -bitmap @[TranslateName CONFIGURE_DIR -file aao_logo.bmp \
    #                   -default [file join $ConfigDirDef aao_logo.bmp]]
    # pack .logo -side right -in .menubar

    #  Create the top section of the display

    frame .ts1
    label .ts1.flabel -text "Input File:" -width 12 -anchor e
    pack .ts1.flabel -side left
    label .ts1.ftext -textvariable InputFile -width 50 -relief sunken \
        -borderwidth 2
    pack .ts1.ftext -side left
    
    pack .ts1 -in .mainframe -side top -padx 4m -pady 2m -anchor w
    
    frame .ts3
    label .ts3.flabel -text "Field Name:" -width 12 -anchor e
    pack .ts3.flabel -side left
    label .ts3.ftext -textvariable FieldName -width 50 -relief sunken \
        -borderwidth 2
    pack .ts3.ftext -side left
    
    pack .ts3 -in .mainframe -side top -padx 4m -pady 2m -anchor w
    
    #  ESO doesn't want FLAMES users to have to worry about the UT date
    #  and time, nor about the wavelength - the configurations should be
    #  insensitive to changes in central wavelength and UT.
    
    if { [Instrument] != "FLAMES"} {
    
        frame .ts7
        label .ts7.flabel -text "UT Date:" -width 20 -anchor e
        pack .ts7.flabel -side left
        label .ts7.ftext1 -textvariable UTDATE -width 14 -relief sunken \
            -borderwidth 2
        pack .ts7.ftext1 -side left
        label .ts7.lab2 -text "Time:" -width 6
        pack .ts7.lab2 -side left
        label .ts7.ftext2 -textvariable UTTIME -width 14 -relief sunken \
            -borderwidth 2
        pack .ts7.ftext2 -side left
    
        pack .ts7 -in .mainframe -side top -padx 4m -pady 2m -anchor w
    
        frame .ts9
        label .ts9.flabel -text "Config Wavelength:" -width 20 -anchor e
        pack .ts9.flabel -side left
        label .ts9.ftext1 -textvariable WavelengthShown -width 14 \
                                           -relief sunken -borderwidth 2
        pack .ts9.ftext1 -side left
    
        pack .ts9 -in .mainframe -side top -padx 4m -pady 2m -anchor w
    }
    
    frame .ts2
    label .ts2.flabel -text "Field Centre (J2000):" -width 22 -anchor e
    pack .ts2.flabel -side left
    label .ts2.lab1 -text "RA:" -width 4
    pack .ts2.lab1 -side left
    label .ts2.ftext1 -textvariable CRA -width 14 -relief sunken \
        -borderwidth 2
    pack .ts2.ftext1 -side left
    label .ts2.lab2 -text "Dec:" -width 6
    pack .ts2.lab2 -side left
    label .ts2.ftext2 -textvariable CDEC -width 14 -relief sunken \
        -borderwidth 2
    pack .ts2.ftext2 -side left
    
    pack .ts2 -in .mainframe -side top -padx 4m -pady 2m -anchor w
    
    frame .ts8
    label .ts8.flabel -text "  " -width 22 -anchor e
    pack .ts8.flabel -side left
    
    #  For FLAMES, we always use the non-existent plate 0, which is a
    #  combination of the actual plates 1 and 2 - since FLAMES configurations
    #  are supposed to be plate-independent. So we don't show the plate.
    
    if { [Instrument] != "FLAMES"} {
        label .ts8.lab0 -text "Plate:" -width 7
        pack .ts8.lab0 -side left
        label .ts8.ftext0 -textvariable ConfigPlate -width 1 -relief sunken \
            -borderwidth 2
        pack .ts8.ftext0 -side left
    }
    
    label .ts8.lab1 -text "HA:" -width 4
    pack .ts8.lab1 -side left
    label .ts8.ftext1 -textvariable CHA -width 14 -relief sunken -borderwidth 2
    pack .ts8.ftext1 -side left
    label .ts8.lab2 -text "  ZD:" -width 6
    pack .ts8.lab2 -side left
    label .ts8.ftext2 -textvariable CZD -width 14 -relief sunken -borderwidth 2
    pack .ts8.ftext2 -side left
    
    pack .ts8 -in .mainframe -side top -padx 4m -pady 2m -anchor w
    
    #  The allocation statistics are put in a separate frame that is packed
    #  in . rather than in .mainframe simply so that I could get the raised
    #  effect I wanted - I wanted it to look like a separate section of the
    #  control panel, rather than just another part of the mainframe - mainly
    #  because it can get so big.
    #
    #  There are two parts of the statistics panel, which change size together
    #  depending on the number of active fibre types. These give the details
    #  of the number of fibres allocated (in .allocstats) and the number of
    #  targets that fibres of the given type can be assigned to (in .availstats)
 
    frame .stats   
    pack .stats -in . -after .mainframe -side top -anchor w
    frame .allocstats -relief raised -borderwidth 1
    pack .allocstats -in .stats -side left -anchor e
    frame .availstats -relief raised -borderwidth 1
    pack .availstats -in .stats -side left -anchor e
    
    label .allocstats.title -text "Allocation statistics"
    pack .allocstats.title -side top -anchor c -pady 1m
    label .availstats.title -text "Available targets"
    pack .availstats.title -side top -anchor c -pady 1m

    frame .allocstats.header
    pack .allocstats.header -side top -anchor w
    frame .availstats.header
    pack .availstats.header -side top -anchor w

    label .allocstats.header.name -width 11 -anchor e -text "Fibre type"
    pack .allocstats.header.name -side left -padx 2m -pady 2m
    label .allocstats.header.fibres -width 7 -anchor e -text "Total"
    pack .allocstats.header.fibres -side left -padx 2m -pady 2m
    label .allocstats.header.skies -width 7 -anchor e -text "Sky"
    pack .allocstats.header.skies -side left -padx 2m -pady 2m
    label .allocstats.header.objects -width 7 -anchor e -text "Objects"
    pack .allocstats.header.objects -side left -padx 2m -pady 2m
    label .allocstats.header.unalloc -width 7 -anchor e -text "Unalloc"
    pack .allocstats.header.unalloc -side left -padx 2m -pady 2m
    
    label .availstats.header.targets -width 7 -anchor e -text "Objects"
    pack .availstats.header.targets -side left -padx 2m -pady 2m
    label .availstats.header.skytargets -width 7 -anchor e -text "Sky"
    pack .availstats.header.skytargets -side left -padx 2m -pady 2m

    frame .allocstats.details
    pack .allocstats.details -side top -anchor w
    
    frame .availstats.details
    pack .availstats.details -side top -anchor w

    #  The size of the statistics panel changes depending on the
    #  number of fibre types currently active. So creating it is delegated
    #  to a separate routine. (Then calling ReformatFibreStats{} changes its
    #  size when the current fibre combination changes.)
    
    FormatFibreStats .allocstats.details .availstats.details
        
    #  Create the main mimic display
    
    toplevel .mimic
    
    wm title .mimic "FPOSS: CONFIGURE" 
    
    CreateMimic
    
    wm geometry .mimic -0+0
    
    .mimic.scv.f1.field configure -closeenough 2
    
    #  Allow window manager resizing

    wm minsize . 1 1

    #  Set position on screen
    
    wm geometry . +0+0
    
    #  Initialize global variables
    
    #  Set the window title
    
    wm title . "FPOSS: [Instrument] Observation Support Software (Version: [ConfigVerNum])" 
    
    set Edited 0
    set BufferedDisplay 1
    set SelectedFibre 0
    
    bind .mimic.scv.f1.field <Control-Button-2> {
        DoAddSkyPosition %x %y
    }
    bind .mimic.scv.f1.field <F1> {
        if {$SelectedFibre != 0} {
            AttemptRecoverFib
        }
    }
    bind .mimic.scv.f1.field <F2> {
        Check
    }
    
    bind .mimic.scv.f1.field <Delete> {
        if {$SelectedFibre != 0} {
            DeallocateFib
        }
    }
    bind .mimic.scv.f1.field <F3> {
        if {$SelectedFibre != 0} {
            AllocateFib
        }
    }

    bind all <F4> {
        Allocate
    }

    bind all <Control-e> {
        ExitApp
    }
    bind all <Control-s> {
        SaveFile
    }
     bind all <Control-o> {
        OpenAscii
    }
    
    set ConfigPlate $InitialPlate
    SetPlateOk
    DtcluDialogPosSet .
    
    #  Set the default colour/grey display scheme
    
    SetGreyColours [UseGrey]
    
    MsgOut "Initial File: " $InitialPlate

    # Open the initial file, if any.
    
    if {[info exists ConfigFile]} {
        
        #  We try it as an SDS file.  If we fail, we attempt to classify it
        #  as either an ASCII file - a .fld text input file - or a PAF file -
        #  and then try to read it in on that basis.
        
        if [catch "SdsRead $ConfigFile" id] {
            set InitialSDSFile 0
            set FileType [ClassifyFile $ConfigFile]
            
            if { $FileType == "FLD" } {
                if [catch "OpenAsciiOk $ConfigFile" msg] {
                    ErsFlush
                    ErsOut \
                      "Error reading Initial object file $ConfigFile\n\n$msg"
                } else {
                    puts "Opened ASCII file $ConfigFile"
                }
                
            } elseif { $FileType == "PAF" } {
                if [catch "DoOpenPAF $ConfigFile" msg] {
                    ErsFlush
                    ErsOut \
                      "Error reading Initial PAF file $ConfigFile\n\n$msg"
                } else {
                    puts "Opened PAF file $ConfigFile"
                }
            }
            
        } else {
        
            #  This appears to be an SDS file.
            
            set InitialSDSFile 1
            SdsReadFree $id
            SdsFreeId $id
            if [catch "DoOpenSDS $ConfigFile" msg] {
                ErsFlush
                ErsOut "Error reading Initial object SDS $ConfigFile\n\n$msg"

            } else {
                puts "Opened SDS file $ConfigFile"
            }
        }

        if {$ImportMode == 1} {
            SetImportOnlyMode 1
            set ImportFile [ImportFileName $ConfigFile]
            ImportDataOnlyOk $ImportFile
            SetImportOnlyMode 0
            SaveFile
            exit
        }
        if { $UtDateSpec != "" } {
            #puts "UT date $UtDateSpec"
            ApplyUTDate $UtDateSpec
        }
        SetTick OpenFile
        
    } else {
        if {$ImportMode == 1} {
            ErsOut "ImportMode (-i option) selected but no config file \
                  (-f option) has been specified."
        } elseif {$DoAllocation == 1} {
            ErsOut "You have requested to start an allocation \
                      (-s option) selected but no config file (-f option) \
                      has been specified."
        } elseif {$DSSOutMode == 1} {
            ErsOut "You have requested to make a DSS listing (-z option) \
                  selected but no config file (-f option) has been specified."
        }
        if { $UtDateSpec != "" } {
            ErsOut "You have specified a utdate/time (-uttime option) \
                   but no config file (-f option) has been specified."
        }
        set ConfigFile " "
        set UtDateSpec ""
    }
    
    if { $DoAllocation} {

        if { ($DoAllocation == 2) && ($InitialSDSFile == 1)} {
            ReAllocate
            set SaveFile [NewSdsFileName $ConfigFile]
             # DoAutoOverAllocate
        } else {
            WorkingDialog .work "Performing Allocation..."
            set Edited 1
	    set ListSaved 0
            set OutputFileStatus "Not saved"
            
            set start [clock seconds]
            DoAllocation $Cfid
            MsgOut "Allocation completed in [expr [clock seconds]-$start] \
                                                                    seconds."
            destroy .work
            SetTick Allocate

            if { $InitialSDSFile == 1 } {
                set SaveFile [NewSdsFileName $ConfigFile]
            } else {
                set SaveFile [SdsFileName $ConfigFile]
            }
        }
        GenerateSkyGrid
        DoSaveAs $SaveFile
        puts "Allocation generated in file $SaveFile, checking over HA range"
        CheckOk1
        exit
    }
    
    #  Start up the basic sequence window
    
    Handholder
        
} ;# Setup.

#  -----------------------------------------------------------------------------
#
#                        N u l l   A c t i o n
#
#   A procedure that deliberately does nothing. This is used as a 'close
#   window' handler for the main windows that we don't want the user to
#   be able to close (because there's no easy way to recreate them).

proc NullAction {} {
}

# -----------------------------------------------------------------------------
#
#                      S e t   E x p e r t  M o d e
#
#  SetExpertMode{} enables or disables the various menu options that are
#  only provided in expert mode.

proc SetExpertMode {} {

    global BufferedDisplay  ;# True if the display output is being buffered
    global ExpertMode       ;# True if the system has been set to 'expert mode'.
    global ShowEmptyFlag    ;# True if empty fibre positions are to be shown.
    global UsePAFFiles      ;# True if the default output file format is PAF.

    #  The File menu setup depends on whether or not FLAMES is in use - 
    #  we provide a more limited set of options if FLAMES is being used,
    #  and then enable the others only in expert mode. FLAMES is the only
    #  configuration that uses PAF files by default.
        
    if {$ExpertMode == 1} {
        .menubar.file.menu delete "Exit ^E"
        if { $UsePAFFiles } {
            .menubar.file.menu add command -label "Open SDS..." \
                                                             -command OpenSDS
            .menubar.file.menu add command -label "Merge..." \
                                                          -command MergeFiles
            .menubar.file.menu add command -label "Save (^S)" \
                                                            -command SaveFile
            .menubar.file.menu add command -label "Save As Sds file..." \
                                                            -command SaveAsSDS
            .menubar.file.menu add command -label "Save Alloc to UnAlloc..." \
                                                  -command SaveAllocToUnalloc
        } else {
            .menubar.file.menu add command -label "Open Target Setup file..." \
                                                             -command OpenPAF
            .menubar.file.menu add command -label "Save Target Setup file..." \
                                                           -command SaveAsPAF
        }
        .menubar.file.menu add command -label "Exit ^E" -command ExitApp
        if {$BufferedDisplay == 1} {
            .menubar.options.menu add command -label "Buffer Display Update" \
                -command BuffDisplay -state disabled
            .menubar.options.menu add command -label "Full Display Update" \
                -command FullDisplay -state normal
        } else {
            .menubar.options.menu add command -label "Buffer Display Update" \
                -command BuffDisplay -state normal
            .menubar.options.menu add command -label "Full Display Update" \
                -command FullDisplay -state disabled
        }
        if { $UsePAFFiles } {
            .menubar.options.menu add command -label "Change Observation Date" \
                -command ChangeObsDate -state disabled
        }
        .menubar.options.menu add command -label "Select Fibre Type" \
                -command SelectFibreType -state disabled
        .menubar.options.menu add command -label "Refraction Parameters..." \
                  -command SetRefraction -state normal
        .menubar.options.menu add command -label "Telescope Parameters..." \
                  -command SetTelescope -state normal      

        if { $UsePAFFiles } {
           .menubar.commands.menu add command -label "Set Wavelength..." \
                 -command SetWave -state disabled
        }
        .menubar.commands.menu add command -label "Rotate Button..." \
                 -command RotateButton -state disabled
        .menubar.commands.menu add command -label "Import Allocations..." \
                -command ImportData -state disabled
        .menubar.commands.menu add command -label "Tcl Command" \
                -command DtcluCommand
                
        .menubar.display.menu add command -label "Show empty fibre positions" \
                -command "ShowEmpty 1"
        .menubar.display.menu add command -label "Hide empty fibre positions" \
                -command "ShowEmpty 0"
        if { $ShowEmptyFlag } {
            .menubar.display.menu entryconfigure "Show empty fibre positions" \
                                                              -state disabled
            .menubar.display.menu entryconfigure "Hide empty fibre positions" \
                                                              -state normal
        } else {
            .menubar.display.menu entryconfigure "Show empty fibre positions" \
                                                              -state normal
            .menubar.display.menu entryconfigure "Hide empty fibre positions" \
                                                              -state disabled
        }
    
    }
    if {$ExpertMode == 0} {
        if { $UsePAFFiles } {
            .menubar.file.menu delete "Open SDS..."
            .menubar.file.menu delete "Merge..."
            .menubar.file.menu delete "Save (^S)" 
            .menubar.file.menu delete "Save As Sds file..."
            .menubar.file.menu delete "Save Alloc to UnAlloc..."
        } else {
            .menubar.file.menu delete "Open Target Setup file..."
            .menubar.file.menu delete "Save Target Setup file..."
        }
        .menubar.options.menu delete "Buffer Display Update"
        .menubar.options.menu delete "Full Display Update"
        if { $UsePAFFiles } {
            .menubar.options.menu delete "Change Observation Date"
        }
        .menubar.options.menu delete "Select Fibre Type"
        .menubar.options.menu delete "Refraction Parameters..."
        .menubar.options.menu delete "Telescope Parameters..."

        if { $UsePAFFiles } {
            .menubar.commands.menu delete "Set Wavelength..."
        }
        .menubar.commands.menu delete "Rotate Button..."
        .menubar.commands.menu delete "Import Allocations..." 
        .menubar.commands.menu delete "Tcl Command"
         
        .menubar.display.menu delete "Show empty fibre positions"
        .menubar.display.menu delete "Hide empty fibre positions"
               
    }
    
    #  Invoking MenuUpdate here makes sure the various menu items are
    #  enabled and/or disabled properly
    
    MenuUpdate
}

#-------------------------------------------------------------------------------

#                        S h o w  E m p t y
#
#   ShowEmpty{} is used to set the global flag that determines whether or not
#   empty fibre positions are displayed.

proc ShowEmpty { On } {

    global ShowEmptyFlag   ;# True if empty fibre positions are to be shown.
    global ExpertMode      ;# True if the system has been set to 'expert mode'.
    global zoom            ;# Current zoom factor - 1 is normal.

    #  We set the global flag, and enable/disable the appropriate menu options.
    #  Note that these are expert-mode only menu options. If the setting
    #  hasn't changed, we do nothing.
    
    if { $On != $ShowEmptyFlag } {
        set ShowEmptyFlag $On
        if { $ExpertMode } {
            if { $ShowEmptyFlag } {
                .menubar.display.menu entryconfigure \
                           "Show empty fibre positions" -state disabled
                .menubar.display.menu entryconfigure \
                           "Hide empty fibre positions" -state normal
            } else {
                .menubar.display.menu entryconfigure \
                           "Show empty fibre positions" -state normal
                .menubar.display.menu entryconfigure \
                           "Hide empty fibre positions" -state disabled
            }
        }
    }
    
    #  Now redraw the display

    ZoomMimic $zoom   
}       

#-------------------------------------------------------------------------------

#                        B u f f  D i s p l a y
#
#   BuffDisplay{} sets the system into a buffered display mode and switches the
#   menu options connected with display buffering appropriately.

proc BuffDisplay {} {

   global BufferedDisplay    ;# True if the display output is being buffered

   set BufferedDisplay 1
   .menubar.options.menu entryconfigure "Buffer Display Update" -state disabled
   .menubar.options.menu entryconfigure "Full Display Update" -state normal
}

#-------------------------------------------------------------------------------

#                          F u l l  D i s p l a y
#
#   FullDisplay{} sets the system into a non-buffered display mode and 
#   switches the  menu options connected with display buffering appropriately.

proc FullDisplay {} {

   global BufferedDisplay    ;# True if the display output is being buffered

   set BufferedDisplay 0
   .menubar.options.menu entryconfigure "Buffer Display Update" -state normal
   .menubar.options.menu entryconfigure "Full Display Update" -state disabled
}

# -----------------------------------------------------------------------------
#
#                          M e n u  U p d a t e
#
#   MenuUpdate{} handles any changes needed to the various menus as a result
#   of a change to the open file status. The important thing is that
#   a number of options are only meaningful if the SDS structure that
#   contains the details of the field and the unallocated and allocated
#   targets exist. This is determined by the global variable Cfid - if 
#   this is null, then this structure does not exist. It invokes all the 
#   menu 'post' routines in turn.

proc MenuUpdate {} {

    global TkAvailable     ;# Set if Tk operations are permitted.
    
    if { !$TkAvailable } return
    
    FilePost
    OptionsPost
    CommandsPost
    ZoomPost

}
    
# -----------------------------------------------------------------------------
#
#                          F i l e  P o s t
#
#   FilePost{} handles any changes needed to the various menus as a result of
#   an option in the file menu being selected. The important thing is that
#   a number of options are only meaningful if the SDS structure that
#   contains the details of the field and the unallocated and allocated
#   targets exist. This is determined by the global variable Cfid - if 
#   this is null, then this structure does not exist. (The name refers to
#   'post'-processing needed on the 'file' menu, and the procedure is set
#   as the post-processing option associated with the file menu.)
#
#   Just using this as the menu post-processing routine isn't enough - it
#   will be invoked as a result of an Open being selected, for example,
#   before the file is actually opened. So we also invoke it at crucial
#   points such as after a file is actually opened or closed - this is the
#   reason for having MenuUpdate{}.

proc FilePost {} {

  #  Global variables used by this routine:

    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure.
    global ExpertMode      ;# True if the system has been set to 'expert mode'.
    global UsePAFFiles     ;# True if the default output file format is PAF.

    if {$Cfid == 0} {

        #  Structure does not exist, so disable those menu options that
        #  make use of it.
        
        if { $UsePAFFiles || $ExpertMode } {
        
            .menubar.file.menu entryconfigure "Save Target Setup file..." \
                                                            -state disabled
        }
        if { $UsePAFFiles == 0 || $ExpertMode } {
            .menubar.file.menu entryconfigure "Save (^S)" -state disabled
            .menubar.file.menu entryconfigure "Save As Sds file..." \
                                                          -state disabled
            .menubar.file.menu entryconfigure "Save Alloc to UnAlloc..." \
                                                        -state disabled
        }
        .menubar.file.menu entryconfigure "List..." -state disabled
        .menubar.file.menu entryconfigure "Print..." -state disabled
        .menubar.display.menu entryconfigure "Select targets..." \
                                                           -state disabled

    } else {

        #  Structure does exist, enable the relevant menu options.
        
        if { $UsePAFFiles || $ExpertMode } {
        
            .menubar.file.menu entryconfigure "Save Target Setup file..." \
                                                            -state normal
        }
        if { $UsePAFFiles == 0 || $ExpertMode } {
            .menubar.file.menu entryconfigure "Save (^S)" -state normal
            .menubar.file.menu entryconfigure "Save As Sds file..." \
                                                             -state normal
            .menubar.file.menu entryconfigure "Save Alloc to UnAlloc..." \
                                                          -state normal
        }
        .menubar.file.menu entryconfigure "List..." -state normal
        .menubar.file.menu entryconfigure "Print..." -state normal
        .menubar.display.menu entryconfigure "Select targets..." \
                                                           -state normal
    }
}

# -----------------------------------------------------------------------------
#
#                          O p t i o n s  P o s t
#
#   OptionsPost{} handles any changes needed to the various menus as a result 
#   of a menu selection in the options menu. This is similar to FilePost{},
#   but is set as the post-processing option associated with the Options menu.

proc OptionsPost {} {

    #  Global variables used by this routine:

    global Cfid             ;# Sds Id for the top level of the current
                            ;# configuration structure.
    global ExpertMode       ;# True if the system has been set to 'expert mode'.
    global UsePAFFiles      ;# True if the default output file format is PAF.

    if {$Cfid == 0} {
        .menubar.options.menu entryconfigure "Change Field Label" \
                                                         -state disabled
        if {!$UsePAFFiles || $ExpertMode } {
           .menubar.options.menu entryconfigure "Change Observation Date" \
                                                         -state disabled
        }
        if {$ExpertMode==1} {
            .menubar.options.menu entryconfigure "Select Fibre Type" \
                                                         -state disabled
        }
        .menubar.options.menu entryconfigure "Select Fibre Combination" \
                                                         -state disabled
    } else {
        .menubar.options.menu entryconfigure "Change Field Label" -state normal
        if { !$UsePAFFiles || $ExpertMode } {
            .menubar.options.menu entryconfigure "Change Observation Date" \
                                                                  -state normal
        }
        if {$ExpertMode==1} {
            .menubar.options.menu entryconfigure "Select Fibre Type" \
                                                                  -state normal
        }
        .menubar.options.menu entryconfigure "Select Fibre Combination" \
                                                                  -state normal
    }
   
}   

# -----------------------------------------------------------------------------
#
#                          C o m m a n d s  P o s t
#
#   CommandsPost{} handles any changes needed to the various menus as a result
#   of selecting an option in the commands menu. This is similar to FilePost{},
#   but also has to take into account the SelectedFibre and SelectedObject 
#   globals and the ExpertMode setting. It is set as the post-processing 
#   option associated with the commands menu.

proc CommandsPost {} {

    #  Global variables used by this routine:

    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure.
    global SelectedFibre   ;# Currently selected fibre (pivot) number - starts
                           ;# from 1 - zero => no fibre selected.
    global SelectedObject  ;# Index of selected target object - starts at one.
    global ExpertMode      ;# True if the system has been set to 'expert mode'.
    global UsePAFFiles     ;# True if the default output file format is PAF.
  
    if {$Cfid == 0} {
        .menubar.commands.menu entryconfigure "Allocate... (F4)" \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Auto-Reallocate..." \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Check Allocation" \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Check over HA range...(F2)" \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Generate Sky Grid..." \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Uncross Fibres..." \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Allocate Fibre (F3)" \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Deallocate Fibre (Del)" \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Deallocate Broken Fibres" \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Deallocate Sky Fibres" \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Show Fibre Info..." \
                                                             -state disabled
        .menubar.commands.menu entryconfigure "Set field center..." \
                                                             -state disabled
        if {$ExpertMode==1} {
           .menubar.commands.menu entryconfigure "Rotate Button..." \
                                                             -state normal
        }
        .menubar.commands.menu entryconfigure "Set Hour Angle..." \
                                                             -state disabled
        if {!$UsePAFFiles || $ExpertMode } {
           .menubar.commands.menu entryconfigure "Set Wavelength..." \
                                                             -state disabled
        }
        .menubar.commands.menu entryconfigure "Remove Allocations..." \
                                                             -state disabled
        if {$ExpertMode==1} {
            .menubar.commands.menu entryconfigure "Import Allocations..." \
                                                             -state disabled
        }
   } else {
        .menubar.commands.menu entryconfigure "Allocate... (F4)" -state normal
        .menubar.commands.menu entryconfigure "Auto-Reallocate..." -state normal
        .menubar.commands.menu entryconfigure "Check Allocation" -state normal
        .menubar.commands.menu entryconfigure "Check over HA range...(F2)" \
                                                                -state normal
        .menubar.commands.menu entryconfigure "Generate Sky Grid..." \
                                                                -state normal
        .menubar.commands.menu entryconfigure "Uncross Fibres..." \
                                                                -state normal
        .menubar.commands.menu entryconfigure "Set Hour Angle..." \
                                                                -state normal
        if {!$UsePAFFiles || $ExpertMode } {
           .menubar.commands.menu entryconfigure "Set Wavelength..." \
                                                                -state normal
        }
        .menubar.commands.menu entryconfigure "Remove Allocations..." \
                                                                -state normal
        .menubar.commands.menu entryconfigure "Deallocate Broken Fibres" \
                                                                -state normal
        .menubar.commands.menu entryconfigure "Deallocate Sky Fibres" \
                                                                -state normal
        .menubar.commands.menu entryconfigure "Set field center..." \
                                                                -state normal
        if {$ExpertMode==1} {
            .menubar.commands.menu entryconfigure "Import Allocations..." \
                                                                -state normal
        }
        if {$SelectedFibre != 0} {
            .menubar.commands.menu entryconfigure "Show Fibre Info..." \
                                                                -state normal
            if {$ExpertMode==1} {
                .menubar.commands.menu entryconfigure "Rotate Button..." \
                                                                -state normal 
            }
            .menubar.commands.menu entryconfigure "Deallocate Fibre (Del)" \
                                                                -state normal
            if {[info exists SelectedObject]} {
               .menubar.commands.menu entryconfigure "Allocate Fibre (F3)" \
                                                                -state normal
            } else {
               .menubar.commands.menu entryconfigure "Allocate Fibre (F3)" \
                                                                -state disabled
            }
        } else {
            .menubar.commands.menu entryconfigure "Show Fibre Info..." \
                                                                -state disabled
            if {$ExpertMode==1} {
                .menubar.commands.menu entryconfigure "Rotate Button..." \
                                                                -state normal 
            }
            .menubar.commands.menu entryconfigure "Deallocate Fibre (Del)" \
                                                                -state disabled
            .menubar.commands.menu entryconfigure "Allocate Fibre (F3)" \
                                                                -state disabled
        }
    }
}


# -----------------------------------------------------------------------------
#
#                          Z o o m  P o s t
#
#   ZoomPost{} handles any changes needed to the various menus as a result
#   of selecting an option in the zoom menu. This is similar to FilePost{}.

proc ZoomPost {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.

    if {$Cfid == 0} {
        .menubar.zoom.menu entryconfigure " x 0.5" -state disabled
        .menubar.zoom.menu entryconfigure " x 0.8" -state disabled
        .menubar.zoom.menu entryconfigure "Normal" -state disabled
        .menubar.zoom.menu entryconfigure " x 2 " -state disabled
        .menubar.zoom.menu entryconfigure " x 4 " -state disabled
        .menubar.zoom.menu entryconfigure " x 8 " -state disabled
    } else {
        .menubar.zoom.menu entryconfigure " x 0.5" -state normal
        .menubar.zoom.menu entryconfigure " x 0.8" -state normal
        .menubar.zoom.menu entryconfigure "Normal" -state normal
        .menubar.zoom.menu entryconfigure " x 2 " -state normal
        .menubar.zoom.menu entryconfigure " x 4 " -state normal
        .menubar.zoom.menu entryconfigure " x 8 " -state normal
    }
}


# -----------------------------------------------------------------------------
#
#                          S a v e  C h e c k
#
#   Should be invoked when we are about to exit or read in a new file.
#   If the existing file has been edited, will prompt for saving of
#   changes.  Returns 1 if user select yes/no to that prompt, and 0 if
#   the user selects cancel.
#
proc SaveCheck { } {

    global Cfid           ;# Sds Id for the top level of the current
                          ;# configuration structure.
    global config_font_1  ;# Font used in dialogues
    global Edited         ;# True once the configuration has been changed
    global FibreComboMode ;# Mode keyword for Fibre combination selected.
    global ConfigFile     ;# Currently displayed configuration file (or plate)
    global SaveFile       ;# Name of the file to be saved

    #  This routine cheats a little with the SaveFile global. It uses it first
    #  as the name of the file to be saved, and then uses it as a convenient 
    #  global to set to 1 or 0 depending on the option selected by the user.

    set UsingPAF 0
    if { [Instrument] == "FLAMES" } { set UsingPAF 1 }
    
    if {$Edited != 0} {
        set SaveFile 0
        set w ".savePrompt"
        toplevel $w
        wm title $w "Save Changes ?"
        if { $UsingPAF } {
            set SaveFile [PAFFileName $Cfid $FibreComboMode]
        } else {
            set SaveFile [SdsFileName $ConfigFile]
        }
        frame $w.top -relief raised -borderwidth 1
        pack $w.top -side top
        message $w.top.msg -width 100m -text "Save Changes to $SaveFile" \
            -font $config_font_1
        pack $w.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
        label $w.top.bitmap -bitmap question
        pack $w.top.bitmap -side left -padx 2m -pady 5m
        pack $w.top -side top
        frame $w.buttons -relief raised -borderwidth 1
        button $w.buttons.yes -command "destroy $w; set SaveFile 1" \
            -text Yes -width 6
        pack $w.buttons -side bottom -fill both
        frame $w.buttons.default -relief sunken -bd 1
        raise $w.buttons.yes $w.buttons.default
        pack $w.buttons.default -side left -expand 1 -padx 3m -pady 2m
        pack $w.buttons.yes -in $w.buttons.default -padx 2m -pady 2m \
            -ipadx 2m -ipady 1m
        button $w.buttons.no -command "destroy $w;set SaveFile 0" \
            -text No -width 6
        pack $w.buttons.no -side left -expand 1 -padx 2m -pady 3m \
            -ipadx 2m -ipady 1m
        button $w.buttons.cancel -command "destroy $w; set SaveFile -1" \
            -text Cancel -width 6
        pack $w.buttons.cancel -side left -expand 1 -padx 2m -pady 3m \
            -ipadx 2m -ipady 1m
        bind $w <Return> "$w.buttons.yes flash; destroy $w; set SaveFile 1"

        # Wait until the variable SaveFile is changed. 

        vwait SaveFile
        if { $SaveFile == 1 } {
            # Value is 1, we save the file and return true
            SaveFile
            return 1
        } elseif { $SaveFile == 0 } {
            # Value is 0, don't save file, but return true
            return 1
        } else {   ;# SaveFile == -1
            # Value is -1, cancel, return false.
            return 0
        }
    } else {
        return 1
    }
}

# -----------------------------------------------------------------------------
#
#                             E x i t  A p p
#
#   This procedure is tied to the Exit option in the file menu. It closes
#   down the program, first checking if any modified file is to be saved.

proc ExitApp {} {

    if [SaveCheck] {
        destroy .
    }
}

# -----------------------------------------------------------------------------
#
#                       S e t  G r e y  C o l o u r s
#
#   Used to switch the display in and out of grey-scale mode. In grey-scale
#   mode a set of greys of different hues is used, rather than using colours
#   to distinguish between the various items in the display.

proc SetGreyColours { UseGreyFlag } {

    global zoom            ;# Current zoom factor - 1 is normal.
    
    SetGrey $UseGreyFlag
    
    if { $UseGreyFlag } {
        .menubar.display.menu entryconfigure "Use grey shades for display" \
                                                              -state disabled
        .menubar.display.menu entryconfigure "Use colours for display" \
                                                              -state normal
    } else {
        .menubar.display.menu entryconfigure "Use grey shades for display" \
                                                              -state normal
        .menubar.display.menu entryconfigure "Use colours for display" \
                                                              -state disabled
    }
       
    ZoomMimic $zoom
}

# -----------------------------------------------------------------------------
#
#                       M o v e  F i e l d  C e n t e r
#
#   Invoked by the "Commands->Set Field Center" menu item.  It resets the
#   field details in the SDS configuration structure so that the field
#   center is now the position last clicked on in the mimic display.

proc MoveFieldCenter {} {

    global Cfid             ;# Sds Id for the top level of the current
                            ;# configuration structure.
    global config_font_1    ;# Font used in dialogues
    global CDEC             ;# Tied to central Dec field in main control panel
    global CRA              ;# Tied to central RA field in main control panel
    global CHA              ;# Tied to Hour angle field in main control panel
    global CZD              ;# Tied to ZD field in main control panel
    global HighlightClick   ;# True if clicked position is to be highlit.
    global LastClickRa      ;# The mean RA of the point last clicked upon.
    global LastClickDec     ;# The mean Dec of the point last clicked upon.
    global MimicClicked     ;# Indicates that mimic display was clicked upon.
    global zoom             ;# Current zoom factor - 1 is normal.
    
    if { $Cfid } {
    
        set fid [SdsFind $Cfid fieldData]
        set CurrentRa [ArgGet $fid "cenRa"]
        set CurrentDec [ArgGet $fid "cenDec"]
        
        #  We're going to put up an explanatory dialogue, and wait for one
        #  of two things. Either the 'cancel' button in the dialogue will
        #  be clicked, or a point in the mimic display will be clicked on
        #  to indicate the required new field center. In both cases, the
        #  global variable MimicClicked will be set (because SetClickPosition{}
        #  sets it, and because the cancel button in the dialogue is tied to
        #  code that sets it. We can tell which of the two, because they set
        #  it to different values.  First we make sure that clicked positions
        #  in the mimic display are highlit, and that any existing such
        #  highlighting is cleared.
        
        RemoveClickHighlights
        set HighlightClick 1
        
        #  Now we put up the explanatory dialogue.
        
        catch {destroy .preMoveCenter}
        toplevel .preMoveCenter
        wm title .preMoveCenter "Move field center"
        wm geom .preMoveCenter +100+0 

        frame .preMoveCenter.top
        label .preMoveCenter.top.msg \
           -text "Click on the required new field center position in the
                  mimic display."  \
           -font $config_font_1
        pack .preMoveCenter.top.msg -padx 5m -pady 2m
        pack .preMoveCenter.top -side top
    
        frame .preMoveCenter.buttons
        pack .preMoveCenter.buttons -side bottom -fill both
        button .preMoveCenter.buttons.cancel -command "set MimicClicked 0" \
                                                        -text "Cancel"
        frame .preMoveCenter.buttons.default -relief sunken -bd 1
        raise .preMoveCenter.buttons.cancel .preMoveCenter.buttons.default
        pack .preMoveCenter.buttons.default -side left -expand 1 -padx 3m \
                                                                      -pady 2m
        pack .preMoveCenter.buttons.cancel -in .preMoveCenter.buttons.default \
                                    -padx 2m -pady 2m -ipadx 2m -ipady 1m
        bind .preMoveCenter <Return> "set MimicClicked 0"

        #  And now we wait until the cancel button is selected or a point
        #  in the mimic display is selected. When it is, we remove the
        #  preliminary dialogue.
        
        tkwait variable MimicClicked
        
        destroy .preMoveCenter
        
        set HighlightClick 0
        
        #  Now if the cancel was selected, MimicClicked will be zero.
        
        if { $MimicClicked } {
        
            #  This was a click in the mimic display. Confirm the change of
            #  field center.
        
            if { [MoveCenterDialogue $CurrentRa $CurrentDec $LastClickRa \
                                                $LastClickDec ] == 0 } {
        
                #  Remove any current allocations. Clear any selected guide
                #  pivot.
    
                RemoveAllocOk
                ClearTicks Allocate
                DeselectGuidePivot

                #  We need to set the mean and apparent field center Ra,Dec
                #  values in the "fieldData" component of the structure.
        
                ArgPutd $fid cenRa $LastClickRa
                ArgPutd $fid cenDec $LastClickDec
                MeanToAppRaDec $Cfid $LastClickRa $LastClickDec AppRa AppDec
                ArgPutd $fid appRa $AppRa
                ArgPutd $fid appDec $AppDec
        
                #  Now we invoke ConvertXy to recalculate all the plate
                #  positions for the various items in the structure.
        
                ConvertXy $Cfid
        
                #  Update the fields displayed in the main panel for the
                #  field center.
                
                set CRA [Ra2string $LastClickRa]
                set CDEC [Dec2string $LastClickDec]
                set CHA [RaDec2Ha $AppRa $AppDec]
                set CZD [RaDec2Zd $AppRa $AppDec]
                
                #  ZoomMimic forces a redisplay of the shifted field.
        
                ZoomMimic $zoom
            
                #  And now recognise that this invalidates any guide star
                #  selection.
            
                ClearTicks GuideStar
            }
        }
    }
}
        
    
#------------------------------------------------------------------------------ 

#                    M o v e  C e n t e r  D i a l o g u e
#
#   A dialogue put up when the user invokes the "Commands->Set Field Center" 
#   menu item. 

proc MoveCenterDialogue { CurrentRa CurrentDec ClickedRa ClickedDec } {

    #  Global variables used by this procedure :
    
    global AllocObj         ;# Number of ordinary targets allocated
    global AllocSky         ;# Number of sky targets allocated
    global AllocGui         ;# Number of guide targets allocated
    global config_font_1    ;# Font used in dialogues
    global TickDetails      ;# Details of ticks in the basic sequence dialogue.
    global LastClickRa      ;# The mean RA of the point last clicked upon.
    global LastClickDec     ;# The mean Dec of the point last clicked upon.
    
    #  Global variables set by this procedure :
    
    global MoveCenterFlag;  ;# Set when the dialogue buttons are pressed.
                  
    catch {destroy .moveCenter}
    toplevel .moveCenter
    wm title .moveCenter "Move field center"
    wm geom .moveCenter +100+0 

    #  If there is a current allocation, warn that it will be invalidated.

    if { ($AllocObj + $AllocSky + $AllocGui) > 0 } {
        frame .moveCenter.top
        label .moveCenter.top.msg \
           -text "Changing the field center invalidates the current allocation.
                  If you do so all allocated fibres will be deallocated."  \
           -font $config_font_1
        pack .moveCenter.top.msg -padx 5m -pady 2m
        pack .moveCenter.top -side top
    }
    
    #  Display current and proposed Ra,Dec
    
    frame .moveCenter.current
    set RaString [Ra2string $CurrentRa]
    set DecString [Dec2string $CurrentDec]
    set Current "Current  field center: $RaString $DecString"
    label .moveCenter.current.msg \
           -text $Current \
           -font $config_font_1
    pack .moveCenter.current.msg -padx 5m -pady 2m
    pack .moveCenter.current -side top
    

    frame .moveCenter.proposed
    set RaString [Ra2string $ClickedRa]
    set DecString [Dec2string $ClickedDec]
    set Proposed "Proposed field center: $RaString $DecString"
    label .moveCenter.proposed.msg \
           -text $Proposed \
           -font $config_font_1
    pack .moveCenter.proposed.msg -padx 5m -pady 2m
    pack .moveCenter.proposed -side top
    
    #  And now the 'Cancel' and 'Continue' buttons.

    frame .moveCenter.buttons
    pack .moveCenter.buttons -side bottom -fill both
    button .moveCenter.buttons.continue \
          -command "set MoveCenterFlag 0; destroy .moveCenter" \
                                                        -text "Continue"
    frame .moveCenter.buttons.default -relief sunken -bd 1
    raise .moveCenter.buttons.continue .moveCenter.buttons.default
    pack .moveCenter.buttons.default -side left -expand 1 -padx 3m -pady 2m
    pack .moveCenter.buttons.continue -in .moveCenter.buttons.default \
                                    -padx 2m -pady 2m -ipadx 2m -ipady 1m
    button .moveCenter.buttons.cancel \
          -command "set MoveCenterFlag 1; destroy .moveCenter" \
                                            -text "Cancel"
    pack .moveCenter.buttons.cancel -side left -expand 1 \
                            -padx 2m -pady 3m -ipadx 2m -ipady 1m
    bind .moveCenter <Return> "set MoveCenterFlag 0; destroy .moveCenter"

    tkwait window .moveCenter
        
    return $MoveCenterFlag
}
   

# -----------------------------------------------------------------------------
#
#                             A l l o c a t e
#
#   Invoked by the "Commands->Allocate" menu item to create the dialog used
#   to initiate an allocation.  Note that the handling of the various 
#   allocation parameters is all abstracted in order to handle the different
#   requirements of different algorithms.  ParameterDetails{} and
#   NumberParameters{} (implemented in C, in configure.c which makes use of
#   the allocation method parameter enquiry interface routines) are used
#   to get details of the parameters and these are combined to construct the
#   dialogue presented to the user.

proc Allocate {} {

   global config_font_1  ;# Font used in dialogues
   
   #  Before we do anything else, perform any necessary checks and abort if
   #  there is a problem the user chooses not to override.

   if [PreAllocateCheck] return

   #  Make sure any shaddowed pivots are disabled. Doing it now makes sure 
   #  the allocation makes use of the latest shaddowing settings.
   
   DisableShaddowedPivots

   #  We must display fields for each of the parameters required by the
   #  tdFConfig library.  NumberParameters is a command which returns
   #  the number of such parameters.
   #
   #  We put the frame with the parameters on it in a scrolled canvas 
   #  widget, so that we can pack more parameters then can be listed on the
   #  screen.
   
   set num [NumberParameters]

   #  Create the top level window.
   
   toplevel .all
   wm title .all "Allocation"
   wm geometry .all +200+70

   #  Create a window title
   #
   #  (For some reason, this is being scrolled, I don't understand why,
   #   TJF, 17-Apr-2000)

   label .all.title -text "Set Allocation Parameters and Hit OK to Allocate" \
       -pady 10 -font $config_font_1
   pack .all.title -expand 1 -fill x 

   #  Create the main window frame

   frame .all.mf -bd 2 -relief ridge
   pack .all.mf -expand 1 -fill both

   #  Create a frame for the parameters.
   
   frame .all.f -bd 0

   #  Create a scrolling canvas widget.

   canvas .all.mf.canvas \
       -yscrollcommand ".all.mf.yscroll set"

   scrollbar .all.mf.yscroll -orient vertical -command ".all.mf.canvas yview"

   pack .all.mf.yscroll -side right -fill y -in .all.mf
   pack .all.mf.canvas -side left -fill both -expand true -in .all.mf

    #  Put our frame into the canvas
    
   .all.mf.canvas create window 0 0 -anchor nw -window .all.f
   
   #  Now create an entry for each parameter.  We use a grid geometry
   #  manager to manage the parameters within the frame ".all.f".   The
   #  variable "gridrow" keeps track of the row number we are working
   #  on (since some parameters are not displayed in expert mode)

   set i 1   
   set gridrow 0
   while {$i <= $num} {

      # ParameterDetails returns details about the specified parameter
      
      ParameterDetails $i name type low high nvalues values

      # We create a name for the frame to be used for the parameter entry box.

      set pname [format "p%s" $i]

      # Create the parameter entry box, based on the type of the parameter.
      
      switch $type {
 
        L  { LParam $pname $name gridrow }
        I  { IParam $pname $name $low $high  gridrow }
        F  { FParam $pname $name $low $high  gridrow }
        C  { CParam $pname $name $nvalues $values gridrow }
      }
      incr i
   }
   
   SkyCountParams gridrow

   DrawButtons .all AllocateOK

   # Wait for the child windows to become visible and then set up the
   # scroll region based on the requested size of the frame
   # and set scroll instrument.

   tkwait visibility .all.buttons.ok

   set bbox   [grid bbox .all.f 0 0]
   set incr   [lindex $bbox 3]
   set width  [winfo reqwidth .all.f]
   set height [winfo reqheight .all.f]
   
   .all.mf.canvas configure -scrollregion "0 0 $width $height"
   .all.mf.canvas configure -yscrollincrement $incr
   .all.mf.canvas configure -width $width
   
   #  If we can display all the window on the screen, then we will resize
   #  to do so and do away with the scrollbar.  Otherwise, we must chose
   #  an appropriate size for the window, say 80% of the screen, but allow
   #  about 20 pixels for the title label.
   #
   #  Note, we do not handle use specific window resizing, so we disable
   #  resizing in the Y direction (width is ok).  We would need a resize
   #  binding to handle a height resize correctly, such that we could
   #  enable or disable the scrollbar as appropiate.

   set winheight [expr [winfo screenheight .] * 0.8 - 20]
   if { $height < $winheight } {
   
       #  We have sufficent space for all parameters, Resize and
       #  do away with the scrollbar.

       .all.mf.canvas configure -height $height
       pack forget .all.mf.yscroll
       
   } else {
   
       #  Use what space we have
       
       .all.mf.canvas configure -height $winheight
   }
   
   wm resizable .all 1 0

   #  Center the window.
   
   Centre .all
}

# -----------------------------------------------------------------------------
#
#                             L  P a r a m
#
#
# Create a parameter entry box in the .all.f window for a Logical parameter
#
# Sets the global variable ${pname}_val to the value of the parameter.
#
# pname -> Parameter entry box name
# name  -> Parameter name.
# row   -> Row in the grid geometry manage to use.  This is a variable
#          name, use upvar to access and the value should be incremented
#          if a value is added.
#
#
proc LParam {pname name row } {

    upvar $row gridrow

    #  Create the global variable used to save the value

    set name_val [format "%s_val" $pname]
    global $name_val

    #  Set initial value of the parameter - will be linked to the checkbutton
    
    set $name_val [GetParameter "$name"]

    #  Create a check button to use to set this value.
    
    checkbutton .all.f.$pname -variable $name_val -text "$name" \
         -onvalue T -offvalue F -anchor w

    #  Place the windows
    
    grid .all.f.$pname -row $gridrow -column 1 -sticky we -padx 10
    grid rowconfigure .all.f $gridrow -pad 10
    incr gridrow

}

# -----------------------------------------------------------------------------
#
#                             I  P a r a m
#
# Create a parameter entry box in the .all.f window for an Integer parameter
#
# Sets the global variable ${pname}_val to the value of the parameter.
#
# pname -> Parameter entry box name
# name  -> Parameter name.
# low   -> Lowest allowed value
# high  -> Highest allowed value
# row   -> Row in the grid geometry manage to use.  This is a variable
#          name, use upvar to access and the value should be incremented
#          if a value is added.
#
# Sets the global variable ${pname}_val to the value of the parameter.

proc IParam {pname name low high row } {

    global ExpertMode      ;# True if the system has been set to 'expert mode'.

    upvar $row gridrow

    #  Create the global variable used to save the value
    
    set name_val [format "%s_val" $pname]
    global $name_val

    #  Set the initial value of the parameter - will be linked to the entry
    
    set $name_val [GetParameter "$name"]

    #  Parameters whose names end in "(expert)" are only made available in 
    #  ExpertMode.

    set ParameterName $name
    set ExpertIndex [string first "(expert)" $ParameterName]
    if { $ExpertIndex >= 0 } {
        set LastIndex [expr $ExpertIndex - 1]
        set ParameterName [string range $ParameterName 0 $LastIndex]
    }   
    if {$ExpertMode == 0 && $ExpertIndex >= 0} { return }
    
    #  Create the label and entry.
    
    label .all.f.${pname}_l -text "$ParameterName"  -anchor w 
    entry .all.f.${pname}_e -textvariable $name_val -relief sunken \
        -borderwidth 2 -width 20

    #  Bind the entry to ensure the range is correct
    
    DtcluBindEntryInt .all.f.${pname}_e $low $high

    #  Place the windows.
    
    grid .all.f.${pname}_l -row $gridrow -column 0 -sticky we -padx 10
    grid .all.f.${pname}_e -row $gridrow -column 1            -padx 10
    grid rowconfigure .all.f $gridrow -pad 10
    incr gridrow

}


# -----------------------------------------------------------------------------
#
#                             F  P a r a m
#
# Create a parameter entry box in the .all.f window for a Floating point
#  parameter
#
# pname -> Parameter entry box name
# name  -> Parameter name.
# low   -> Lowest allowed value
# high  -> Highest allowed value
# row   -> Row in the grid geometry manage to use.  This is a variable
#          name, use upvar to access and the value should be incremented
#          if a value is added.
#
# Sets the global variable ${pname}_val to the vsalue of the parameter.
#
proc FParam {pname name low high row} {

    global ExpertMode      ;# True if the system has been set to 'expert mode'.

    upvar 1 $row gridrow

    #  Create the global variable used to save the value
    
    set name_val [format "%s_val" $pname]
    global $name_val

    #  Set the initial value of the parameter - will be linked to the entry
    
    set $name_val [GetParameter "$name"]
    
    #  Parameters whose names end in "(expert)" are only made available in 
    #  ExpertMode.

    set ParameterName $name
    set ExpertIndex [string first "(expert)" $ParameterName]
    if { $ExpertIndex >= 0 } {
        set LastIndex [expr $ExpertIndex - 1]
        set ParameterName [string range $ParameterName 0 $LastIndex]
    }   
    if {$ExpertMode == 0 && $ExpertIndex >= 0} { return }

    #  CReate the label and entry.
     
    label .all.f.${pname}_l -text "$ParameterName"  -anchor w 
    entry .all.f.${pname}_e -textvariable $name_val -relief sunken \
        -borderwidth 2 -width 20
        
    #  Bind the entry to ensure the range is correct
    
    DtcluBindEntryReal .all.f.${pname}_e "neg"

    #  Place the windows.
    
    grid .all.f.${pname}_l -row $gridrow -column 0 -sticky we -padx 10
    grid .all.f.${pname}_e -row $gridrow -column 1 -padx 10
    grid rowconfigure .all.f $gridrow -pad 10
    incr gridrow

}

# -----------------------------------------------------------------------------
#
#                             C  P a r a m
#
#
# Create a parameter entry box in the .all.f window for a Text/Enumerated 
# parameter
#
# pname   -> Parameter entry box name
# name    -> Parameter name.
# nvalues -> Number of possible values
# values  -> A list of the possible values.
# row   -> Row in the grid geometry manage to use.  This is a variable
#          name, use upvar to access and the value should be incremented
#          if a value is added.
#
# if nvalues > 0, we use radio widgets to selected from enumerate values.
#
# if nvalues is 0, we don't have enumerated values and use a entry widget to
# accept any value
#
# Sets the global variable ${pname}_val to the value of the parameter.
#
proc CParam {pname name nvalues values row} {

    global ExpertMode      ;# True if the system has been set to 'expert mode'.

    upvar $row gridrow

    #  Create the global variable used to save the value
    
    set name_val [format "%s_val" $pname]
    global $name_val
    
    #  Set the initial value of the parameter - will be linked to the entry
    #  radiobuttons.
    
    set $name_val [GetParameter "$name"]
 
    #  Menus whose names end in "(expert)" are only made available in 
    #  ExpertMode.

    set MenuName $name
    set ExpertIndex [string first "(expert)" $MenuName]
    if { $ExpertIndex >= 0 } {
        set LastIndex [expr $ExpertIndex - 1]
        set MenuName [string range $MenuName 0 $LastIndex]
    }   
    if {$ExpertMode == 0 && $ExpertIndex >= 0} { return }

    label .all.f.${pname}_l -text "$MenuName" -anchor w  
    if {$nvalues == 0} {
        entry .all.f.${pname}_e -textvariable $name_val -relief sunken \
            -borderwidth 2 -anchor w
            
        #  Place the windows.
        
        grid .all.f.${pname}_l -row $gridrow -column 0 -sticky we -padx 10
        grid .all.f.${pname}_e -row $gridrow -column 1 -sticky we -padx 10
        grid rowconfigure .all.f $gridrow -pad 10
        incr gridrow
    } else {

        #  Where we are have an enumerated value.  We set up a series
        #  of radio-buttons to select the value.
        #
        #  First create a frame (_rad) to put the check button into.

        frame .all.f.${pname}_rad -relief ridge -borderwidth 2 

        #  Place the windows.
        
        grid .all.f.${pname}_l   -row $gridrow -column 0 -sticky we -padx 10
        grid .all.f.${pname}_rad -row $gridrow -column 1 -sticky we -padx 10
        grid rowconfigure .all.f $gridrow -pad 10
        incr gridrow       

        #  Create the radio buttons.  Note that values that end with the
        #  string "(expert)" are only made available in expert mode, and we
        #  strip off the (expert) before we display them.

        set n 0
        while {$n < $nvalues} {

           set ButtonName [lindex $values $n]
           set ExpertIndex [string first "(expert)" $ButtonName]
           if { $ExpertIndex >= 0 } {
               set LastIndex [expr $ExpertIndex - 1]
               set ButtonName [string range $ButtonName 0 $LastIndex]
           }
           if { $ExpertMode == 1 || $ExpertIndex <= 0 } {
               radiobutton .all.f.${pname}_rad.$n -variable $name_val \
                   -value $ButtonName -text $ButtonName \
                   -anchor w 
               pack .all.f.${pname}_rad.$n -side top -padx 10m -ipadx 2m \
                                                                    -fill x
           }
           incr n
       }
    }
}

# -----------------------------------------------------------------------------
#
#                       S k y  C o u n t  P a r a m s
#
#  This routine adds to the standard allocation parameters being built up
#  by the allocation dialogue a set of additional parameters used to specify
#  the number of fibres of each type to be allocated to sky targets. Doing
#  this is a bit moot. It means that these are the only parameters that are
#  not handled via the parameter mechanism built into the configuration
#  algorithms, and as such require quite a lot of code to get their values
#  down through the various system layers. However, the parameter mechanism
#  is over-worked at the moment and has been bypassed in rather ugly ways
#  by other routines at the Tcl layer that need to know the meaning of the
#  various parameters, so this seems rather better. What we really need is a
#  more flexible way to get information such as this to the algorithms,
#  together with a way to allow the Tcl layer to request services from the
#  algorithm layer.
#
#  This routine adds lines to the grid set up by the allocation dialogue.
#  The 'row' parameter is the row in the grid geometry manager to use.  This
#  is a variable name, so we use upvar to access it and the value should be
#  incremented each time a value is added.

proc SkyCountParams { row } {

    #  Global variables used:
    #
    #  Declared dynamically in the code, global variables named SkyCount_{n}
    #  are used as the associated variables for the entry widgets.
    
    #  Make sure we have an up-to-date list of enabled fibre types
    
    UpdateFibreCounts
    
    upvar $row gridrow
    
    #  Look at each fibre type in use
    
    set FibreTypeList [FibreTypes]
    set NumFibreTypes [llength $FibreTypeList]
    for { set Index 0 } { $Index < $NumFibreTypes } { incr Index } {
        GetFibreStats $Index Name Enabled Fibres Skies Guides \
                            Objects Unalloc SameAs MayShare Targets SkyTargets
         
        #  The fibre types we're interested in are those that are currently
        #  enabled and share sky and non-sky targets (other fibre types don't
        #  need this sort of control). We also have to allow for the fact that
        #  some types (UVES_7 and UVES_8 in FLAMES, for example) are treated
        #  for this purpose as equivalent to a lower-indexed fibre type (UVES
        #  in the case of UVES_7 and UVES_8)
        
        if { $Enabled && $MayShare && ($SameAs < 0) } {
        
            set ValueName "SkyCount_$Index"
            global $ValueName
            if { [info exists $ValueName] == 0 } {
            
                #  This value variable doesn't exist yet, so it needs an
                #  initial value for the number of fibres. The expression
                #  below is taken from Matthew Colless' talk at the FLAMES
                #  workshop reported in the ESO Messenger #105 (Sept 2001)
                
                set $ValueName [expr round(pow($Fibres/2,.66))]
            }
            set LabelText "Number of $Name fibres to assign to sky"
            label .all.f.lab_$ValueName -text $LabelText -anchor w 
            entry .all.f.ent_$ValueName -textvariable $ValueName \
                                   -relief sunken -borderwidth 2 -width 3

            #  Bind the entry to ensure the range is correct
    
            DtcluBindEntryInt .all.f.ent_$ValueName 0 $Fibres

            #  Place the windows.
    
            grid .all.f.lab_$ValueName -row $gridrow -column 0 \
                                                      -sticky we -padx 10
            grid .all.f.ent_$ValueName -row $gridrow -column 1 -sticky w \
                                                            -padx 10
            grid rowconfigure .all.f $gridrow -pad 10
            incr gridrow
        }
    }   
}

# -----------------------------------------------------------------------------
#
#                    S e t  S k y  C o u n t  P a r a m s
#
#  Applies the sky count parameters specified in the allocation dialogue. See
#  SkyCountParms{} for more details.

proc SetSkyCountParams { } {

    #  Global variables used:
    #
    #  Declared dynamically in the code, global variables named SkyCount_{n}
    #  are used as the associated variables for the entry widgets.
    
    #  SetSkyCounts{} with no arguments is used to clear the counts for
    #  every fibre type. Then we call it again with the sky counts for
    #  the types that were specified.
    
    SetSkyCounts
    set FibreTypeList [FibreTypes]
    set NumFibreTypes [llength $FibreTypeList]
    for { set Index 0 } { $Index < $NumFibreTypes } { incr Index } {
        GetFibreStats $Index Name Enabled Fibres Skies Guides \
                           Objects Unalloc SameAs MayShare Targets SkyTargets
        if { $Enabled && $MayShare && ($SameAs < 0) } {
            set ValueName "SkyCount_$Index"
            global $ValueName
            set SkyCount [set $ValueName]
            if { $SkyCount != "" } {
                SetSkyCounts $Index $SkyCount
            }
        }
    }   
}

# -----------------------------------------------------------------------------
#
#                             A l l o c a t e  O K
#
# AllocateOk is invoked when the Ok button is hit in the Allocate window.
#
# It starts an allocation going.

proc AllocateOK {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global OutputFileStatus  ;# Description of output file status
    global zoom              ;# Current zoom factor - 1 is normal.
    global Edited            ;# True once the configuration has been changed
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved

    set num [NumberParameters]
    set i 1
    while {$i <= $num} {

        set name_val [format "p%s_val" $i]
        ParameterDetails $i name type low high nvalues values
        global $name_val

        SetParameter "$name" "[set $name_val]"
        incr i
    }
    
    WorkingDialog .work "Performing Allocation..."
    set validHourAngle 0
    set Edited 1
    set ListSaved 0
    set OutputFileStatus "Not saved"
    set CheckStatus "Not checked"
    SetSkyCountParams
    set start [clock seconds]
    DoAllocation $Cfid
    CompressUnalloc $Cfid
    AddIndex $Cfid
    SetTick Allocate
    MsgOut "Allocation completed in [expr [clock seconds]-$start] seconds."
   
    destroy .work
    SetTick Allocate
    ZoomMimic $zoom

}

# -----------------------------------------------------------------------------
#
#                             R e  A l l o c a t e
#
#   Runs the auto-reallocate algorithm. Note that this is supported by the
#   'Oxford' algorithm but not by the 'Taylor' algorithm.  This is one of
#   the routines that makes use of knowledge about the parameters of the
#   allocation. Ideally there should be a proper interface to the allocation
#   algorithms that will allow this to be done more elegantly.
#

proc ReAllocate {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global Edited            ;# True once the configuration has been changed
    global OutputFileStatus  ;# Description of output file status
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved
    
    # Set parameters to appropriate values

    # Target Selection = auto-reallocate
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    set p5_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 5]"

    # Tenative deallocation = never
    
    set name_val "p6_val"
    ParameterDetails 6 name type low high nvalues values
    global $name_val
    set p6_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 0]"

    # Fibre uncrossing = when allocation complete
    
    set name_val "p7_val"
    ParameterDetails 7 name type low high nvalues values
    global $name_val
    set p7_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 2]"

    # Number of fibres to leave for sky = 0
    
    ParameterDetails 8 name type low high nvalues values
    global $name_val
    set p8_value [GetParameter "$name"]
    SetParameter "$name" 0

    # Set up field.
    
    WorkingDialog .work "Auto-Reallocating in Progress...."
    set valiHourAngle 0
    set Edited 1
    set ListSaved 0
    set OutputFileStatus "Not saved"
    set CheckStatus "Not checked"
    CompressUnalloc $Cfid
    AddIndex $Cfid
    ConvertXy $Cfid

    # Do allocation
    
    set start [clock seconds]
    DoAllocation $Cfid

    # Finish.
    
    CompressUnalloc $Cfid
    AddIndex $Cfid
    MsgOut "Allocation completed in [expr [clock seconds]-$start] seconds."
    destroy .work
    SetTick Allocate

    # Have to put parameters back
    
    # Target Selection 
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p5_value
    
    # Tenative deallocation
    
    set name_val "p6_val"
    ParameterDetails 6 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p6_value

    # Fibre uncrossing 
    
    set name_val "p7_val"
    ParameterDetails 7 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p7_value

    # Number of fibres to leave for sky
     
    ParameterDetails 8 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p8_value

}

# -----------------------------------------------------------------------------
#
#                 R e  A l l o c a t e  N o  U n c r o s s
#
#  This routine is invoked if an HA check is performed with the 'reallocate
#  invalid fibres' option selected, although comments in an earlier version
#  of this file indicated that this routine was no longer used. It may be
#  that this option doesn't work! (KS).
#
#  This is one of the routines that makes use of knowledge about the 
#  parameters used by the allocation algorithm. Ideally there should be a
#  proper interface to the allocation algorithms that will allow this to be
#  done more elegantly.

proc ReAllocateNoUncross { } {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global OutputFileStatus  ;# Description of output file status
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved

    # Target Selection = auto-reallocate
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 5]"
    
    # Tenative deallocation = never
    
    set name_val "p6_val"
    ParameterDetails 6 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 0]"
    
    # Fibre uncrossing = never
    
    set name_val "p7_val"
    ParameterDetails 7 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 0]"
    
    # Number of fibres to leave for sky = 0
    
    ParameterDetails 8 name type low high nvalues values
    global $name_val
    SetParameter "$name" 0
    
    WorkingDialog .work "Recovery in Progress...."
    set validHourAngle 0
    set Edited 1
    set ListSaved 0
    set OutputFileStatus "Not saved"
    set CheckStatus "Not checked"
    set start [clock seconds]
    DoAllocation $Cfid
    CompressUnalloc $Cfid
    AddIndex $Cfid
    MsgOut "Allocation completed in [expr [clock seconds]-$start] seconds."
    destroy .work
    SetTick Allocate
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 4]"
    set name_val "p6_val"
    ParameterDetails 6 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 0]"
    set name_val "p7_val"
    ParameterDetails 7 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 2]"
}

# -----------------------------------------------------------------------------
#
#                             A l l  U p d a t e
#
#  This routine is invoked when we get a progress report on an allocation.
# 
#  description = Description of what the allocation algorithm is doing.
#  percent     = Percentage complete. 
# 
#  This used to be a much more complicated routine, since it handled the
#  display of allocated fibres as well, but that was removed and now it is
#  only used to report progress and allow the user to cancel whatever is
#  going on. 

proc AllUpdate {description percent} {

   UpdateCounts
   set cancel [UpdateProgress .work $description $percent]
   update
    
   return $cancel
}

# -----------------------------------------------------------------------------
#
#                             A l l o c  R p t
#
#  This routine is invoked to report allocation/deallocation of fibre
# 
#  alloc       = 1/0 indicating if we are allocating or deallocating a fibre
#  fibre       = Pivot number (1 based)
#  object      = Index into unallocated guide or unallocate object arrays 
#                indicating the object in question.  The tag indicates which 
#                one of these arrays applies.
#  tag         = "uguide" if the object is a guide object, "uobj" if it
#                is a program object
#  theta       = Button twist angle
#  
#
proc AllocRpt {type fibre object tag theta} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.

    if {$type} {

        # Object has been allocated.
        
        if {$tag == "uguide"} {
            set type guide
        } else {
            set type object
        }
        # Update the structures to indicate the fibre has been allocated
        
        set status [AllocateFibre $Cfid [PivotsId] $fibre $type $object]
        
        # And set the angle
        
        SetButtonAngle $Cfid [PivotsId] $fibre $theta
        if {$status == "OK"} {
           .mimic.scv.f1.field delete lin$fibre
           .mimic.scv.f1.field delete fib$fibre 
           DrawAFibre $fibre
           RedrawOnAllocate $fibre $object $tag
        }
    } else {
        set status [DeallocateFibre $Cfid [PivotsId] $fibre tag index]
        if {$status == "OK"} {
           .mimic.scv.f1.field delete lin$fibre
           .mimic.scv.f1.field delete fib$fibre
           DrawAFibre $fibre
           if {$index != -1} {RedrawOnDeallocate $fibre $index $tag}
        }  
    }

}


# -----------------------------------------------------------------------------
#
#                          R e m o v e  A l l o c
#
#   This procedure is invoked in response to the "Remove allocations" menu
#   option.  It puts up a confirmation dialogue which will invoke the
#   procedure RemoveAllocOk if confirmed.

proc RemoveAlloc {} {

    global config_font_1  ;# Font used in dialogues

    toplevel .remove
    wm title .remove "Remove Allocations"
    frame .remove.top -relief raised -borderwidth 1
    pack .remove.top -side top
    message .remove.top.msg -width 90m \
            -text "Do you really want to remove all fibre allocations" \
            -font $config_font_1
    pack .remove.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
    label .remove.top.bitmap -bitmap question
    pack .remove.top.bitmap -side left -padx 2m -pady 5m
    pack .remove.top -side top
    frame .remove.buttons -relief raised -borderwidth 1
    button .remove.buttons.yes -command "destroy .remove;RemoveAllocOk" \
                                                       -text Yes -width 8
    pack .remove.buttons -side bottom -fill both
    frame .remove.buttons.default -relief sunken -bd 1
    raise .remove.buttons.yes .remove.buttons.default
    pack .remove.buttons.default -side left -expand 1 -padx 3m -pady 2m
    pack .remove.buttons.yes -in .remove.buttons.default -padx 2m -pady 2m \
                    -ipadx 2m -ipady 1m
    button .remove.buttons.no -command "destroy .remove" -text No -width 8
    pack .remove.buttons.no -side left -expand 1 -padx 2m -pady 3m \
        -ipadx 2m -ipady 1m
    bind .remove <Return> \
               ".remove.buttons.yes flash;destroy .remove;RemoveAllocOk"
}

# -----------------------------------------------------------------------------
#
#                       R e m o v e  A l l o c  O k
#
#   Deallocates all allocated fibres that aren't explicitly flagged as
#   not to be modified.  This is invoked when the user clicks on the OK 
#   button in the dialog put up by RemoveAlloc{}

proc RemoveAllocOk {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global CheckStatus       ;# Describes current check status

   set fibre 1
   set obj [SdsFind $Cfid objects]
   WorkingDialog .work "Removing Allocations"
   while {$fibre <= [NumPivots]} {
       set ModsAllowed [lindex [GetFibre $Cfid $fibre] 8]
       if {$ModsAllowed} {
           GetObjectData $obj $fibre name ra dec type spect prio mag pid \
                                                  comment text xposn yposn
           if {$type != [UnallocatedType]} {
               DeallocateFibre $Cfid [PivotsId] $fibre tag index
               .mimic.scv.f1.field delete lin$fibre
               .mimic.scv.f1.field delete fib$fibre
               DrawAFibre $fibre
               if {$index != -1} {RedrawOnDeallocate $fibre $index $tag}
               set percent [expr {$fibre*100.0/[NumPivots]}]
               set cancel [UpdateProgress .work "Removing Allocations" $percent]
               if {$cancel != 0} break
               if {$fibre % 20 ==0} { # Update display every so often
                   UpdateCounts
                   update
               }
           }
       }
       incr fibre
   }
   UpdateCounts
   update
   CompressUnalloc $Cfid
   AddIndex $Cfid
   destroy .work
   set CheckStatus "Not checked"
   ClearTicks Allocate
}

# -----------------------------------------------------------------------------
#
#                          H a n d h o l d e r
#
#   This procedure puts up the 'handholding' basic sequence guide.
#

proc Handholder {} {

    global AllocationStatus   ;# Describes current allocation status
    global ConfigFile         ;# Currently displayed configuration file
    global config_font_1      ;# Font used in dialogues
    global CurrentFibreCombo  ;# Describes current fibre combination.
    global MagFilterText      ;# Describes magnitude filter limits
    global SelectedGuideDescr ;# Describes guide star selection
    global OutputFileStatus   ;# Description of output file status
    global TickDetails        ;# Details of ticks in the basic sequence dialogue
    global CheckStatus        ;# Describes current check status
    
    #  Note that this code originated with FLAMES. This section tries
    #  to make it a little more abstract to handle the other supported
    #  instruments. 
    
    set UsingPAF 0
    set UsingVLTGuider 0
    if { [Instrument] == "FLAMES" } {
        set UsingPAF 1
        set UsingVLTGuider 1
    }
    
    #  TickDetails is used to record the state of the tick marks following
    #  each button, and forms a linked list that can be used to clear all
    #  the ticks from a named entry down. The 'displayed' entry is supposed
    #  to handle cases where an item is not used for a specific instrument,
    #  but the entry is still needed to maintain the linked list.
       
    toplevel .handhold
    wm title .handhold "FPOSS: Basic sequence"
    
    set level 1
    
    #  Magnitude filter
    
    frame .handhold.selmag -relief raised -borderwidth 1
    button .handhold.selmag.button -command MagnitudeFilter -width 22 \
                                                     -text "Magnitude filter"
    label .handhold.selmag.tick -bitmap tick -foreground red
    label .handhold.selmag.filter -textvariable MagFilterText -width 50 \
                                                  -font $config_font_1
    pack .handhold.selmag.button -side left -fill both
    pack .handhold.selmag.tick -side left -fill both
    pack .handhold.selmag.filter -side left -fill both
    raise .handhold.selmag.button
    pack .handhold.selmag
    
    set TickDetails(MagFilter,widget) .handhold.selmag.tick
    set TickDetails(MagFilter,displayed) 1
    set TickDetails(MagFilter,next) OpenFile
    set TickDetails(MagFilter,level) $level
    incr level
    
    #  Open file
    
    frame .handhold.open -relief raised -borderwidth 1
    button .handhold.open.button -command OpenAscii -width 22 \
                                                     -text "Open input file"
    label .handhold.open.tick -bitmap tick -foreground red
    label .handhold.open.filename -textvariable ConfigFile -width 50 \
                                                  -font $config_font_1
    pack .handhold.open.button -side left -fill both
    pack .handhold.open.tick -side left -fill both
    pack .handhold.open.filename -side left -fill both
    raise .handhold.open.button
    pack .handhold.open
    
    set TickDetails(OpenFile,widget) .handhold.open.tick
    set TickDetails(OpenFile,displayed) 1
    set TickDetails(OpenFile,next) GuideStar
    set TickDetails(OpenFile,level) $level
    incr level
    
    #  Select guide star - only used for FLAMES
    
    if { $UsingVLTGuider } {
        frame .handhold.gstar -relief raised -borderwidth 1
        button .handhold.gstar.button -command ExplainGstar -width 22 \
                                               -text "Select guide star"
        label .handhold.gstar.tick -bitmap tick -foreground red
        label .handhold.gstar.starname -textvariable SelectedGuideDescr \
                                          -width 50 -font $config_font_1
        pack .handhold.gstar.button -side left -fill both
        pack .handhold.gstar.tick -side left -fill both
        pack .handhold.gstar.starname -side left -fill both
        raise .handhold.gstar.button
        pack .handhold.gstar
    }
    
    set TickDetails(GuideStar,widget) .handhold.gstar.tick
    set TickDetails(GuideStar,displayed) $UsingVLTGuider
    set TickDetails(GuideStar,next) FibreCombo
    set TickDetails(GuideStar,level) $level
    
    #  Fibre combination (at same level as Guide star)
    
    frame .handhold.combo -relief raised -borderwidth 1
    button .handhold.combo.button -command "SelectFibreCombo" -width 22 \
                                               -text "Fibre combination"
    label .handhold.combo.tick -bitmap tick -foreground red
    label .handhold.combo.name -textvariable CurrentFibreCombo -width 50 \
                                                  -font $config_font_1
    pack .handhold.combo.button -side left -fill both
    pack .handhold.combo.tick -side left -fill both
    pack .handhold.combo.name -side left -fill both
    raise .handhold.combo.button
    pack .handhold.combo
    
    set TickDetails(FibreCombo,widget) .handhold.combo.tick
    set TickDetails(FibreCombo,displayed) 1
    set TickDetails(FibreCombo,next) Allocate
    set TickDetails(FibreCombo,level) $level
    incr level
    
    #  Allocate
    
    frame .handhold.alloc -relief raised -borderwidth 1
    button .handhold.alloc.button -command "Allocate" -width 22 \
                                               -text "Allocate fibres"
    label .handhold.alloc.tick -bitmap tick -foreground red
    label .handhold.alloc.status -textvariable AllocationStatus -width 50 \
                                                  -font $config_font_1
    pack .handhold.alloc.button -side left -fill both
    pack .handhold.alloc.tick -side left -fill both
    pack .handhold.alloc.status -side left -fill both
    raise .handhold.alloc.button
    pack .handhold.alloc
    
    set TickDetails(Allocate,widget) .handhold.alloc.tick
    set TickDetails(Allocate,displayed) 1
    set TickDetails(Allocate,next) Check
    set TickDetails(Allocate,level) $level
    incr level

    #  Allocate
    
    frame .handhold.check -relief raised -borderwidth 1
    button .handhold.check.button -command "Check" -width 22 \
                                               -text "Check HA range"
    label .handhold.check.tick -bitmap tick -foreground red
    label .handhold.check.status -textvariable CheckStatus -width 50 \
                                                  -font $config_font_1
    pack .handhold.check.button -side left -fill both
    pack .handhold.check.tick -side left -fill both
    pack .handhold.check.status -side left -fill both
    raise .handhold.check.button
    pack .handhold.check
    
    set TickDetails(Check,widget) .handhold.check.tick
    set TickDetails(Check,displayed) 1
    set TickDetails(Check,next) SaveFile
    set TickDetails(Check,level) $level
    incr level
    
    #  Save as PAF file (for FLAMES) or as SDS file
    
    frame .handhold.write -relief raised -borderwidth 1
    if { $UsingPAF } {
        button .handhold.write.button -command "SaveAsPAF" -width 22 \
                                               -text "Save Target Setup file"
    #                                           -text "Save as PAF file"
    } else {
        button .handhold.write.button -command "SaveAsSDS" -width 22 \
                                               -text "Save as SDS file"
    }
    label .handhold.write.tick -bitmap tick -foreground red
    label .handhold.write.filename -textvariable OutputFileStatus -width 50 \
                                                  -font $config_font_1
    pack .handhold.write.button -side left -fill both
    pack .handhold.write.tick -side left -fill both
    pack .handhold.write.filename -side left -fill both
    raise .handhold.write.button
    pack .handhold.write
    
    set TickDetails(SaveFile,widget) .handhold.write.tick
    set TickDetails(SaveFile,displayed) 1
    set TickDetails(SaveFile,next) Exit
    set TickDetails(SaveFile,level) $level
    incr level
        
    #  Exit
    
    frame .handhold.exit -relief raised -borderwidth 1
    button .handhold.exit.button -command "ExitApp" -width 22 \
                                               -text "Exit"
    label .handhold.exit.tick -bitmap tick -foreground red
    label .handhold.exit.blank -text "" -width 50 \
                                                  -font $config_font_1
    pack .handhold.exit.button -side left -fill both
    pack .handhold.exit.tick -side left -fill both
    pack .handhold.exit.blank -side left -fill both
    raise .handhold.exit.button
    pack .handhold.exit
    
    set TickDetails(Exit,widget) .handhold.exit.tick
    set TickDetails(Exit,displayed) 1
    set TickDetails(Exit,next) ""
    set TickDetails(Exit,level) $level
        
    wm geometry .handhold +0+0
    
    #  Don't allow the user to close the window
    
    wm protocol .handhold WM_DELETE_WINDOW NullAction
    
    #  This works to clear all the ticks - if there were other entries at
    #  the same level as MagFilter we'd need some other ploy, as this
    #  would leave those other entries ticked.
    
    ClearTicks MagFilter

}

# -----------------------------------------------------------------------------
#
#                          C l e a r  T i c k s
#
#  This routine clears the tick from one of the entries in the basic sequence
#  dialogue and for all entries at a lower level. It is passed one of the 
#  names used for the entries in the TickDetails global array - see 
#  Handholder{} for more details. It clears the entry specified, and then
#  any other entries at a higher level. (So it doesn't clear other entries
#  at the same level as the named entry.)
#
#  This was a lot more straightforward until it was realised that in some
#  cases clearing an entry in the basic sequence set up by Handholder{} does
#  not always imply that all subsequent entries should be cleared. For 
#  example, for FLAMES, selecting the guide probe and selecting the fibre
#  combination are independent and changing one does not invalidate the other.
#  These are really at the same 'level' in the sequence, which is why the
#  'level' concept was added to this code.

proc ClearTicks { name } {

    #  GLobal variables used:

    global TickDetails      ;# Details of ticks in the basic sequence dialogue
   
    #  Allow for the possibility that we are being run before Handholder{}
    #  has set up the TickDetails array.
    
    if { [info exists TickDetails] } {
            
        set level $TickDetails($name,level)
        
        #  Clear the tick for this entry.
        
        ClearOneTick $name
        
        #  Work through all the entries further on in the sequence,
        #  clearing the ticks for any at a higher level.
        
        while { $TickDetails($name,next) != "" } {
           set NewName $TickDetails($name,next)
           set name $NewName
           if { $TickDetails($name,level) > $level } { ClearOneTick $name }
        }
    }
}

# -----------------------------------------------------------------------------
#
#                         C l e a r  O n e  T i c k 
#
#  This routine clears the tick from one of the entries in the basic sequence,
#  by setting the colour for the tick to be the same colour as its background.
#  It is passed one of the names used for the entries in the TickDetails 
#  global array - see Handholder{} for more details.

proc ClearOneTick { name } {

    #  GLobal variables used:
    
    global TickDetails      ;# Details of ticks in the basic sequence dialogue

    if { $TickDetails($name,displayed) } {           
        set widget $TickDetails($name,widget)
        set bgcolour [$widget cget -background]
        $widget configure -foreground $bgcolour
    }
    set TickDetails($name,ticked) 0
}        

# -----------------------------------------------------------------------------
#
#                            S e t  T i c k
#
#  This routine ticks one of the entries in the basic sequence dialogue.
#  It should be called whenever the action associated with the entry has
#  taken place (whether initiated from the dialogue or not). It ticks the
#  entry itself and then clears all the ticks from subsequent entries - on
#  the assumption that a change earlier on in the sequence invalidates
#  any later actions. It is passed one of the names used for the entries
#  in the TickDetails global array - see Handholder{} for more details.

proc SetTick { name } {

    #  GLobal variables used:
    
    global TickDetails      ;# Details of ticks in the basic sequence dialogue

    #  Allow for the possibility that we are being run before Handholder{}
    #  has set up the TickDetails array.
    
    if { [info exists TickDetails] } {
    
        #  If the entry is being displayed, enable the tick by changing
        #  its colour to red.
        
        if { $TickDetails($name,displayed) } {   
            set widget $TickDetails($name,widget)
            $widget configure -foreground red
        }
        set TickDetails($name,ticked) 1
        
        #  If there are entries further on in the sequence, clear their ticks.
        #  Note that we clear everything at a level greater than that of the
        #  entry whose tick we just set.
        
        set level $TickDetails($name,level)
        
        while { $TickDetails($name,next) != "" } {
           set newName $TickDetails($name,next)
           set name $newName
           if { $TickDetails($name,level) > $level } { ClearOneTick $name }
        }
    }
}

# -----------------------------------------------------------------------------
#
#                        E x p l a i n  G s t a r
#
#  This routine runs when the 'select guide star' button in the basic
#  sequence display is pressed.  A guide star cannot be selected by the
#  program in reponse to this, so all it can do is try to explain to the
#  user what is needed.

proc ExplainGstar {} {

    #  GLobal variables used:
    
    global SelectedGuideDescr ;# Describes guide star selection

    set SelectedGuideDescr "Double click on a VLT guide star, then select side"
}

# -----------------------------------------------------------------------------
#
#                             C e n t r e
#
#   This routine moves the specified window to the center of the display.

proc Centre {w} {

  wm withdraw $w
  update idletasks
  set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
            - [winfo vrootx [winfo parent $w]]]
  set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
            - [winfo vrooty [winfo parent $w]]]
  wm geom $w +$x+$y
  wm deiconify $w
}


# -----------------------------------------------------------------------------
#
#                               L i s t
#
#   This routine is tied to the 'List' item in the file menu. It puts up
#   a dialog box that allows the user to select the sort of list to be 
#   created. When the user gives the go-ahead, ListOk{} will be invoked.

proc List {} {

    #  GLobal variables used:
    
    global ListItems      ;# Variable tied to the dialog radio buttons
    global config_font_1  ;# Font used in dialogues

    catch {destroy .list}
    toplevel .list
    wm title .list "List"

    frame .list.top

    message .list.top.msg -width 90m -text "Output Listing of..." \
            -font $config_font_1
    pack .list.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
    label .list.top.bitmap -bitmap question
    pack .list.top.bitmap -side left -padx 2m -pady 5m
    pack .list.top -side top
  
    frame .list.items
    pack .list.items -side top -padx 5m -pady 5m

    set ListItems skycat

    radiobutton .list.items.alloc -value alloc -variable ListItems \
       -text "Allocations" -anchor w
    radiobutton .list.items.obj -value obj -variable ListItems \
       -text "Unallocated Objects" -anchor w
    radiobutton .list.items.all -value all -variable ListItems \
       -text "All objects" -anchor w
    radiobutton .list.items.dssall -value dssall -variable ListItems \
       -text "Allocated as DSS input" -anchor w
    radiobutton .list.items.dsssly -value dsssky -variable ListItems \
       -text "Allocated Sky as DSS input" -anchor w
    radiobutton .list.items.skycat -value skycat -variable ListItems \
       -text "Allocated objects in Skycat format" -anchor w
    pack .list.items.alloc -side top -padx 10m -ipadx 2m -fill x
    pack .list.items.obj    -side top -padx 10m -ipadx 2m -fill x
    pack .list.items.all    -side top -padx 10m -ipadx 2m -fill x
    pack .list.items.dssall -side top -padx 10m -ipadx 2m -fill x
    pack .list.items.dsssly -side top -padx 10m -ipadx 2m -fill x
    pack .list.items.skycat -side top -padx 10m -ipadx 2m -fill x


    DrawButtons .list ListOk

    Centre .list
}


# -----------------------------------------------------------------------------
#
#                            F i l e  N a m e
#
#  Given a file name, this procedure returns the name minus the directory

proc FileName { file } {

   #  Use file split to get at the components
   
   set name [file split $file]

   # The last element of the list is the one we want.

   set length [llength $name]
   incr length -1

   return [lindex $name $length]

}

# -----------------------------------------------------------------------------
#
#                            L i s t  O k
#
#  This routine is invoked when the user selects a list type in the dialogue
#  put up by List{} and presses the OK button. It puts up an output file
#  selection dialogue which will invoke DoList{} when a file is selected.

proc ListOk {} {

    global ListItems        ;# Variable tied to the dialog radio buttons
    global ListFile         ;# Initial suggestion for listing file - does
                            ;# This really need to be global?
    global ConfigFile       ;# Currently displayed configuration file (or plate)
    global CurrentOutDirectory;# Current directory for saving files
    global OutputFileStatus   ;# Description of output file status
    global Cfid             ;# Sds Id for the top level of the current
                            ;# configuration structure.
    global FibreComboMode   ;# Mode keyword for Fibre combination selected
    global Edited           ;# True once the configuration has been changed
    global ListTargetFile   ;# List file saved: proposed target setup file name.

   set types {
        {{All Files}      *}
   }

   if { $OutputFileStatus != "Not saved" && $Edited == 0 } {
       set ListTargetFile $OutputFileStatus
   } elseif { [file extension $ConfigFile] == ".ins" && $Edited == 0 } {
       set ListTargetFile [file tail $ConfigFile]
   } else {
       set ListTargetFile [PAFFileName $Cfid $FibreComboMode]
   }
   set tag [lindex $ListItems 0]
   if { ($tag == "dssall") || ($tag == "dsssky")} {
       set ListFile [DSSFileName $ListTargetFile]
       set File [tk_getSaveFile    \
           -filetypes $types \
           -initialdir $CurrentOutDirectory \
           -initialfile $ListFile \
           -title "DSS File"]
       if {$File != ""} {
           set CurrentOutDirectory [file dirname $File]
           DoList $File
       }
   } else {
       set ListFile [ListFileName $ListTargetFile]
       set File [tk_getSaveFile    \
           -filetypes $types \
           -initialdir $CurrentOutDirectory \
           -initialfile $ListFile \
           -title "Listing File"]
       if {$File != ""} {
           set CurrentOutDirectory [file dirname $File]
           DoList $File
       }
   }
}

# -----------------------------------------------------------------------------
#
#                             D o  L i s t
#
#   Procedure called from the file save dialogues put up by ListOk{}. Most
#   of the real work is done by the C-implemented MakeDSSListing{} or
#   MakeListing{}.
#
#   2003-01-27: Guide Star added to the parameter list for SkycatListing, AKA

proc DoList {file} {

    #  GLobal variables used:
    
    global ListItems           ;# Variable tied to the dialog radio buttons
    global Cfid                ;# Sds Id for the top level of the current
                               ;# configuration structure.
    global SelectedGuideObject ;# Which guide object is selected ( 0=> none).
    global SelectedGuideName   ;# Name of selected guide star.
    global ListSaved           ;# Flag: current config. list is saved
    global ListTargetFile      ;# List file saved: proposed target setup file name.

    set tag [lindex $ListItems 0]
    if { ($tag == "dssall") || ($tag == "dsssky")} {
        MakeDSSListing $Cfid $file $ListItems
    } elseif { ($tag == "skycat")} {
        SkycatListing $Cfid $file $SelectedGuideObject $SelectedGuideName
    } else {
        MakeListing $Cfid $file $ListItems
    }
    set ListTargetFile "[file rootname [file tail $file]].ins"
    set ListSaved 1
}

# -----------------------------------------------------------------------------
#
#                         R o t a t e  B u t t o n
#
#  This procedure is tied to the "Rotate button.." item in the Commands
#  menu. It puts up a dialogue with an OK button that will allow the new
#  angle to be input and will trigger AngleOK{} when the OK button is
#  pressed. 
#
#  Note that although button rotation is supported by the 2dF instrument
#  and by it's control software, it is not properly supported by the
#  configure program - the gains from allowing button rotation appeared
#  to be minimal, and the complexity is scary. 6dF and the OzPoz
#  positioner used by FLAMES have circular buttons and do not support
#  rotation.

proc RotateButton {} {

    #  GLobal variables used:
    
    global SelectedFibre     ;# Currently selected fibre (pivot) number -
                             ;# starts from 1 - zero => no fibre selected.
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.

    if { $SelectedFibre == 0 } {
        ErsOut "No fibre selected"
        return
    }

    set xy [GetFibre $Cfid $SelectedFibre]
    set Theta [lindex $xy 2]
    set Theta [expr {int($Theta*180/3.1415926)}]

    catch {destroy .rot}
    toplevel .rot
    wm title .rot "Set Button Angle"
    scale .rot.scale -label "Angle (degrees)" -from 0 -to 360 -length 10c \
        -orient horizontal -command NewAngle
    .rot.scale set $Theta
    pack .rot.scale -side top
    Centre .rot
    button .rot.ok -command {destroy .rot; AngleOK} -text OK -width 4
    pack .rot.ok -side top -padx 10m -pady 10m -fill x
                                                        }


# -----------------------------------------------------------------------------
#
#                             N e w  A n g l e
#
#   This routine is tied to the scale widget in the dialogue put up by
#   RotaterButton{}. It changes the angle for the fibre and re-draws
#   it at the new angle.
#

proc NewAngle {theta} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelectedFibre     ;# Currently selected fibre (pivot) number -
                             ;# starts from 1 - zero => no fibre selected.
                             
    SetButtonAngle $Cfid [PivotsId] $SelectedFibre [expr {$theta*3.1415926/180}]
    .mimic.scv.f1.field delete fib$SelectedFibre
    .mimic.scv.f1.field delete lin$SelectedFibre
    DrawAFibre $SelectedFibre
}
  
# -----------------------------------------------------------------------------
#
#                             A n g l e  O K
#
#   This procedure is invoked when the user presses the OK button in the
#   dialogue put up by RotateButton(). It checks that the specified angle
#   results in a valid configuration.
#
proc AngleOK {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelectedFibre     ;# Currently selected fibre (pivot) number -
                             ;# starts from 1 - zero => no fibre selected.
    global Edited            ;# True once the configuration has been changed
    global OutputFileStatus  ;# Description of output file status
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved
    
    WorkingDialog .work "Checking Field..."
    set status [DoCheck $Cfid]
    destroy .work
    if {$status == "OK"} {
        .mimic.scv.f1.field delete lin$SelectedFibre
        .mimic.scv.f1.field delete fib$SelectedFibre
        DrawAFibre $SelectedFibre
	set validHourAngle 0
        set Edited 1
        set ListSaved 0
        set OutputFileStatus "Not saved"
        set CheckStatus "Not checked"
        OpenSDSOk
    } else {
        SetButtonAngle $Cfid [PivotsId] $SelectedFibre radial
        StartUpdate $Cfid [PivotsId]
        .mimic.scv.f1.field delete lin$SelectedFibre
        .mimic.scv.f1.field delete fib$SelectedFibre
        DrawAFibre $SelectedFibre
    }
}

# -----------------------------------------------------------------------------
#
#                      A t t e m p t  R e c o v e r  F i b
#
#   Proc to try to recover an allocation with a different fibre 
#   e.g. if it's failed an hour angle check.

proc AttemptRecoverFib {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelectedFibre     ;# Currently selected fibre (pivot) number - starts
                             ;# from 1 - zero => no fibre selected.
    global SelectedTag       ;# Tag associated with selected target object.
    global Edited            ;# True once the configuration has been changed
    global OutputFileStatus  ;# Description of output file status
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved

    set xy [GetFibre $Cfid $SelectedFibre]
    set Type [lindex $xy 3]
    set status "OK"
    puts stdout "TCL: trying to recover $SelectedFibre from state $Type"
    if {$Type != [UnallocatedType]} {
        if {[TypeIsSky $Type]} {
            puts stdout "TCL: Deallocated Sky Fibre $SelectedFibre"
            DeallocateFib
            set status "OK"
        }
        # Note, this may leak if an error is returned by ReallocateGuide.
        set save [SdsCopy $Cfid]
        if {[TypeIsGuide $type]} {
            puts stdout "TCL: collision involving Guide Fibre $SelectedFibre"
            ReallocateGuide
        } else {
            puts stdout "TCL: collision involving Object Fibre $SelectedFibre"
            # set status [ReallocateObject $Cfid [PivotsId] $SelectedFibre]
        }
        if {$status == "OK"} {
	    set validHourAngle 0
            set Edited 1
            set ListSaved 0
            set OutputFileStatus "Not saved"
            set CheckStatus "Not checked"
            OpenSDSOk
            ArgDelete $save
        } else {
            if { $Cfid != 0 } {
                ArgDelete $Cfid
            }
            set Cfid $save
            StartUpdate $Cfid [PivotsId]
        }
    }
}
 


# -----------------------------------------------------------------------------
#
#                            A l l o c a t e  F i b
#
#   Explicitly allocates the selected fibre to the selected target object.
#
proc AllocateFib {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelectedFibre     ;# Currently selected fibre (pivot) number - starts
                             ;# from 1 - zero => no fibre selected.
    global SelectedTag       ;# Tag associated with selected target object.
    global Edited            ;# True once the configuration has been changed
    global SelectedObject    ;# Index of selected target object - starts at one.
    global OutputFileStatus  ;# Description of output file status
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved

    # Deallocate if already allocated (i.e. allow a move)
    
    set xy [GetFibre $Cfid $SelectedFibre]
    set Type [lindex $xy 3]
    if {$Type != [UnallocatedType]} { DeallocateFib } 

    if {$SelectedTag == "uguide"} {set type guide} else {set type object}

    #   Save a copy of the current configuration file.
    
    set save [SdsCopy $Cfid]
    set status [AllocateFibre $Cfid [PivotsId] $SelectedFibre \
                                                  $type $SelectedObject]
    if {$status == "OK"} {
        WorkingDialog .work "Checking Field..."
        set status [DoCheck $Cfid 0]
        destroy .work
        if {$status == "OK"} {
            .mimic.scv.f1.field delete lin$SelectedFibre
            .mimic.scv.f1.field delete fib$SelectedFibre
            DrawAFibre $SelectedFibre
            RedrawOnAllocate $SelectedFibre $SelectedObject $SelectedTag
	    set validHourAngle 0
            set Edited 1
            set ListSaved 0
            set OutputFileStatus "Not saved"
            set CheckStatus "Not checked"
            OpenSDSOk
        
            # Delete the copy, we no longer need it.
            ArgDelete $save
        } else {

	    # Deallocate fibre to keep correct statistics
	    DeallocateFib

            # We need to revert to the saved copy.  Must delete the current
            # configuration.
            ArgDelete $Cfid
            set Cfid $save
            StartUpdate $Cfid [PivotsId]
        }
    } else {
        # Delete the copy
        ArgDelete $save
    }
}

# -----------------------------------------------------------------------------
#
#                       G e n e r a t e  S k y  G r i d
#
#   Invoked to generate a sky grid. This routine used to allocate the
#   grid after generating it, but that doesn't work well with the new
#   finer control provided over sky allocations in the allocation dialogue,
#   and was inelegant into the bargain, so that has been dropped and this
#   routine is now back to simply generating a sky grid that can be allocated
#   later.
#

proc GenerateSkyGrid {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global config_font_1     ;# Font used in dialogues
    global zoom              ;# Current zoom factor - 1 is normal.

    #   Create a display for the pause while we generate the grid.
    
    toplevel .work
    wm title .work "Please Wait..."
    wm geometry .work +100+200
    frame .work.top
    pack .work.top -side top
    message .work.top.msg -width 90m \
        -text "Please Wait while I set up a grid..." \
        -font $config_font_1
    bind .work.top.msg <Return> 
    pack .work.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
        
    puts stdout "Generating Sky Grid..."
    
    #  All the work of generating the new targets is done by SetSkyGrid{},
    #  and then ConvertXy{} handles the coordinate conversions. We also
    #  check for guide probe shadowing.
    
    SetSkyGrid $Cfid
    UpdateCounts
    ConvertXy $Cfid
    DisableShaddowedPivots
    ZoomMimic $zoom
    destroy .work
    
   #  Update the statistics showing target allocation by fibre type
   
   UpdateFibreStats $Cfid
   UpdateFibreCounts
}


# -----------------------------------------------------------------------------
#
#                       R e a l l o c a t e  G u i d e
#
#   This is inovked by AttemptRecoverFib, to attempt to allocate
#   guide fibres after having recovered broken fibres.   This is one of
#   the routines that makes use of knowledge about the parameters of the
#   allocation. Ideally there should be a proper interface to the allocation
#   algorithms that will allow this to be done more elegantly.
#
#
proc ReallocateGuide {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelectedFibre     ;# Currently selected fibre (pivot) number -
                             ;# starts from 1 - zero => no fibre selected.
    global OutputFileStatus  ;# Description of output file status
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved
    
    ConvertXy $Cfid
    ZoomMimic 1

    # Set parameters to appropiate values
    
    # Target Selection = "recover a single fibre"
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    set p5_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 9]"


    # Tenative deallocation = never
    
    set name_val "p6_val"
    ParameterDetails 6 name type low high nvalues values
    global $name_val
    set p6_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 0]"

    # Fibre uncrossing = when allocation complete
    
    set name_val "p7_val"
    ParameterDetails 7 name type low high nvalues values
    global $name_val
    set p7_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 0]"

    # Fibre to recover
    
    set name_val "p9_val"
    ParameterDetails 9 name type low high nvalues values
    global $name_val
    SetParameter "$name" $SelectedFibre

    WorkingDialog .work "Searching the Tree...."
    set validHourAngle 0
    set Edited 1
    set ListSaved 0
    set OutputFileStatus "Not saved"
    set CheckStatus "Not checked"
    CompressUnalloc $Cfid
    AddIndex $Cfid

    # Do the allocation
    
    set start [clock seconds]
    DoAllocation $Cfid
    CompressUnalloc $Cfid
    AddIndex $Cfid
    MsgOut "Allocation completed in [expr [clock seconds]-$start] seconds."
    destroy .work
    SetTick Allocate

    # Have to put parameters back
    
    # Target Selection 
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p5_value
    
    # Tenative deallocation
    
    set name_val "p6_val"
    ParameterDetails 6 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p6_value
}

# -----------------------------------------------------------------------------
#
#                    D o  A u t o  O v e r  A l l o c a t e
#
#   This routine is currently not used. It remains in the code, but it's
#   believed that it doesn't reset the parameters correctly.  This is one of
#   the routines that makes use of knowledge about the parameters of the
#   allocation, and at the moment it isn't resetting the parameters it
#   modified properly. See ReAllocate{} to see how it should be done.
#

proc DoAutoOverAllocate {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global OutputFileStatus  ;# Description of output file status
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved
    
    ConvertXy $Cfid
    ZoomMimic 1
    
    # Target Selection = only allocate sky
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 6]"
    
    # Tenative deallocation = never
    
    set name_val "p6_val"
    ParameterDetails 6 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 0]"
    
    # Fibre uncrossing = when allocation completed
    
    set name_val "p7_val"
    ParameterDetails 7 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 2]"
    
    WorkingDialog .work "Allocating Extra Objects...."
    set validHourAngle 0
    set Edited 1
    set ListSaved 0
    set OutputFileStatus "Not saved"
    set CheckStatus "Not checked"
    CompressUnalloc $Cfid
    AddIndex $Cfid
    set start [clock seconds]
    DoAllocation $Cfid
    CompressUnalloc $Cfid
    AddIndex $Cfid
    MsgOut "Allocation completed in [expr [clock seconds]-$start] seconds."
    destroy .work
    SetTick Allocate
    UncrossFibres
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    SetParameter "$name" "[lindex $values 4]"
    set name_val "p6_val"
}

# -----------------------------------------------------------------------------
#
#                         U n c r o s s  F i b r e s
#
#   Set up an allocation run that just uncrosses the fibres. This is one of
#   the  routines that makes use of knowledge about the parameters of the
#   allocation. Ideally there should be a proper interface to the allocation
#   algorithms that will allow this to be done more elegantly. This is a
#   function that is supported by the 'Oxford' algorithm but not by the
#   'Taylor' algorithm, which is why the iterative target selection option
#   has to be imposed.
#

proc UncrossFibres {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global Edited            ;# True once the configuration has been changed
    global OutputFileStatus  ;# Description of output file status
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved
    
    # Set parameters to appropiate values

    # Target Selection = "iterative target allocation" (ie, Oxford algorithm)
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    set p5_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 4]"

    # Tenative deallocation = never
    
    set name_val "p6_val"
    ParameterDetails 6 name type low high nvalues values
    global $name_val
    set p6_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 0]"

    # Fibre uncrossing = uncross only
    
    set name_val "p7_val"
    ParameterDetails 7 name type low high nvalues values
    global $name_val
    set p7_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 3]"

    # Set up field.
    
    WorkingDialog .work "Uncrossing Fibres...."
    set validHourAngle 0
    set Edited 1
    set ListSaved 0
    set OutputFileStatus "Not saved"
    set CheckStatus "Not checked"
    CompressUnalloc $Cfid
    AddIndex $Cfid

    # Do allocation
    
    set start [clock seconds]
    DoAllocation $Cfid

    # Tidy up.
    
    CompressUnalloc $Cfid
    AddIndex $Cfid
    MsgOut "Allocation completed in [expr [clock seconds]-$start] seconds."
    destroy .work
    SetTick Allocate
    ConvertXy $Cfid
    ZoomMimic 1

    # Have to put parameters back
    
    # Target Selection 
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p5_value
    
    # Tenative deallocation
    
    set name_val "p6_val"
    ParameterDetails 6 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p6_value

    # Fibre uncrossing 
    
    set name_val "p7_val"
    ParameterDetails 7 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p7_value

}

# -----------------------------------------------------------------------------
#
#                     D o  A d d  S k y  P o s i t i o n
#
#  Adds a sky position to the list of targets at the position given by
#  the passed x y coordinates. This is bound to control-button-2 in the 
#  mimic display, and the x and y values passed are those given by 
#  %x and %y in the tcl bind statement.

proc DoAddSkyPosition {x y} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelectedFibre     ;# Currently selected fibre (pivot) number -
                             ;# starts from 1 - zero => no fibre selected.
    global SelectedObject    ;# Index of selected target object - starts at one.
    global SelectedTag       ;# Tag associated with selected target object.
    global Edited            ;# True once the configuration has been changed
    global OutputFileStatus  ;# Description of output file status
    global zoom              ;# Current zoom factor - 1 is normal.
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved

  # Add a new sky position to the object list
  
    AddSkyPosition $Cfid $x $y object $zoom \
        [lindex [.mimic.scv.f1.hscroll get] 0] \
        [lindex [.mimic.scv.vscroll get] 0]
    UpdateCounts

    # Select this object
    
    set SelectedObject $object
    set SelectedTag "uobj"
    set OutputFileStatus "Not saved"
    set CheckStatus "Not checked"
    set validHourAngle 0
    set Edited 1
    set ListSaved 0

    # Convert positions to x/y, and allow for guide probe shadowing
    
    ConvertXy $Cfid
    DisableShaddowedPivots

    # Find the unallocated ojbects and draw this object.
    
    set id [SdsFind $Cfid unallocObject]
    DrawOneObject $id $object $SelectedTag
    SdsFreeId $id

    # If we have a selected fibre, we want to allocate it to this object.
    
    if {$SelectedFibre != 0} {
        set xy [GetFibre $Cfid $SelectedFibre]
        set Type [lindex $xy 3]
        # Deallocate the fibre,  if it is already allocated
        if {$Type != [UnallocatedType]} { DeallocateFib } 
        set xy [GetFibre $Cfid $SelectedFibre]
        set Type [lindex $xy 3]
        if {$Type == [UnallocatedType]} {
            AllocateFib
        } 
    } 

    #  The following line doesn't appear to be necessary, and have the
    #  effect of confusing the index number - DrawOneTarget assumes that
    #  the structures aren't going to change under it, and this can happen
    #  if CompressUnalloc actually finds allocated objects and AddIndex
    #  then puts them back. This is a rather odd sequence, in fact, and
    #  I'm sure it's wrong.  Eventually, this can be deleted.
    
    #CompressUnalloc $Cfid
    
    AddIndex $Cfid
    OpenSDSOk
}

# -----------------------------------------------------------------------------
#
#                         D e a l l o c a t e  F i b
#
#   Deallocates the currently selected fibre.

proc DeallocateFib {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelectedFibre     ;# Currently selected fibre (pivot) number -
                             ;# starts from 1 - zero => no fibre selected.
    global SelectedObject    ;# Index of selected target object - starts at one.
    global SelectedTag       ;# Tag associated with selected target object.
    global OutputFileStatus  ;# Description of output file status
    global Edited            ;# True once the configuration has been changed
    global ListSaved         ;# Flag: current config. list is saved
    
    set status [DeallocateFibre $Cfid [PivotsId] $SelectedFibre tag index]
    if {$status == "OK"} {
        .mimic.scv.f1.field delete lin$SelectedFibre
        .mimic.scv.f1.field delete fib$SelectedFibre
        DrawAFibre $SelectedFibre
        set OutputFileStatus "Not saved"
        set Edited 1
        set ListSaved 0
        OpenSDSOk
        if {$index != -1} {RedrawOnDeallocate $SelectedFibre $index $tag}
    }
}

# -----------------------------------------------------------------------------
#
#                  D e a l l o c a t e  B r o k e n  F i b r e s
#
#
#  This routine deallocates all broken fibres. It is tied to the 'Deallocate
#  broken fibres' option in the commands menu.

proc DeallocateBrokenFibres {} {
  
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.

    FixBreakages [PivotsId] $Cfid
    StartUpdate $Cfid [PivotsId]

}

# -----------------------------------------------------------------------------
#
#                  D e a l l o c a t e  S k y  F i b r e s
#
#  This routine deallocates all sky fibres. It is tied to the 'Deallocate
#  sky fibres' option in the commands menu. Note that the name is misleading -
#  there are really sky targets, not sky fibres. This routine deallocated any
#  fibre that has been allocated to a sky target.

proc DeallocateSkyFibres {} {
  
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
                             
    RemoveSky [PivotsId] $Cfid
    StartUpdate $Cfid [PivotsId]

}

# -----------------------------------------------------------------------------
#
#                  D u m m y  D e a l l o c a t e  F i b
#
#   Invoked by FixBreakges or RemoveSky Tcl command C implementations to
#   actually deallocate the specified fibre. The name of this routine is
#   misleading - it isn't a dummy at all, it's a perfectly good routine that
#   looks rather like DeallocateFib{}, except that it takes a fibre number
#   argument rather than deallocating the currently selected fibre, making it
#   more suitable for calling from C code.
#
proc DummyDeallocateFib { FibreNum } {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelectedFibre     ;# Currently selected fibre (pivot) number -
                             ;# starts from 1 - zero => no fibre selected.
    global SelectedObject    ;# Index of selected target object - starts at one.
    global SelectedTag       ;# Tag associated with selected target object.
    global OutputFileStatus  ;# Description of output file status
    global Edited            ;# True once the configuration has been changed
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved
    
    set SelectedFibre $FibreNum
    set status [DeallocateFibre $Cfid [PivotsId] $SelectedFibre tag index]
    if {$status == "OK"} {
        .mimic.scv.f1.field delete lin$SelectedFibre
        .mimic.scv.f1.field delete fib$SelectedFibre
        DrawAFibre $SelectedFibre
	set validHourAngle 0
        set Edited 1
        set ListSaved 0
        set OutputFileStatus "Not saved"
        set CheckStatus "Not checked"
        OpenSDSOk
        if {$index != -1} {RedrawOnDeallocate $SelectedFibre $index $tag}
    }
}


# -----------------------------------------------------------------------------
#
#                              O p e n  S D S
#
#   Procedure called as a result of the "Open SDS..." selection from the
#   File menu. Put up a file selection dialog to get the file name
#   and then call DoOpenSDS{} to display it if the OK button is selected.

proc OpenSDS {} {

   #  Global variables used by this routine:
   
   global CurrentInDirectory ;# Current directory for reading files
   
   # First check if we have to save the existing file.  If this returns
   # false, the user cancelled the operation.
   
   if { [ SaveCheck ] == 0 } {
       return
   }

   set types {
        {{SDS files} {*.sds}}
        {{All Files}      *}
   }

   set File [tk_getOpenFile    \
       -filetypes $types \
       -initialdir $CurrentInDirectory \
       -title "File with Input Target List"]
   if {$File != ""} {
      set CurrentInDirectory [file dirname $File]
      DoOpenSDS $File
   }
      
}


# -----------------------------------------------------------------------------
#
#                        D o  O p e n  S D S
#
#   Read and open the new Sds file. This can be called directly from code
#   such as the batch mode main line code that opens a file specified on the
#   command line, but is usually called in response to the file open dialogue
#   put up by OpenSDS{}. The specified SDS file should contain a copy of the
#   internal SDS configuration structure manipulated by the configuration
#   program, as saved by a previous run of the program.

proc DoOpenSDS {file} {

    global Cfid           ;# Sds Id for the top level of the current
                          ;# configuration structure.
    global ConfigFile     ;# Currently displayed configuration file (or plate)
    global Edited         ;# True once the configuration has been changed
    global errorCode      ;# Error description (generated by Tcl)
    global errorInfo      ;# Stack trace describing an error (generated by Tcl)
    global SelectedFibre  ;# Currently selected fibre (pivot) number - starts
                          ;# from 1 - zero => no fibre selected.
    global SelectedObject ;# Index of selected target object - starts at one.
    global OutputFileStatus;# Description of output file status
    global TkAvailable    ;# Set if Tk operations are permitted.
    global validHourAngle ;# Validated hour angle range.
    global CheckStatus    ;# Describes current check status

   # Read the file.
   
   set temp [SdsRead $file]

   # Save a copy of the Sds structure (means we have an internal SDS
   # structure which we can modify, rather then an external Sds structure
   
   set temp2 [SdsCopy $temp]

   # Tidy up the read file's memory.
   
   SdsReadFree $temp
   SdsFreeId $temp

   # Access the structure.
   
   SetUTMeridian $temp2
   MakeAllocated $temp2 [PivotsId]
   ConvertXy $temp2
   
   # AddIndex may fail here if we have duplicated objects.  If that
   # happends, we need to restore the old file, if any, and delete
   # the new one.
   
   if [catch "AddIndex $temp2" msg] {
       set savedInfo $errorInfo
       set savedCode $errorCode
       ArgDelete $temp2
       error $msg $savedInfo $savedCode
   } else {

       # Free up the old configuration.
       
       if { $Cfid != 0 } {
           ArgDelete $Cfid
       }

       # Setup the global variable.
       
       set ConfigFile $file
       set Cfid $temp2
       SetTick OpenFile

       # File has not been edited.
       
       set validHourAngle 0
       set Edited 0
       set OutputFileStatus "Not saved"
       set CheckStatus "Not checked"
       OpenSDSOk

       # Display the field

       if { $TkAvailable } { wm title .mimic $ConfigFile }
       catch {set SelectedFibre 0}
       catch {unset SelectedObject}
       DeselectGuidePivot
       ZoomMimic 1
   }

   #  Update the statistics showing target allocation by fibre type
   
   UpdateFibreStats $Cfid
   
   ReformatFibreStats .allocstats.details .availstats.details
   
}

# -----------------------------------------------------------------------------
#
#                              O p e n  P A F
#
#   Procedure called as a result of the "Open PAF..." selection from the
#   File menu. Put up a file selection dialog to get the file name
#   and then call DoOpenPAF{} to display it if the OK button is selected

proc OpenPAF {} {

   #  Global variables used by this routine:
   
   global CurrentInDirectory ;# Current directory for reading files
   
   # First check if we have to save the existing file.  If this returns
   # false, the user cancelled the operation.
   
   if { [ SaveCheck ] == 0 } {
       return
   }

   set types {
        {{Target Setup files} {*.ins}}
        {{All Files}      *}
   }

   set File [tk_getOpenFile    \
       -filetypes $types \
       -initialdir $CurrentInDirectory \
       -title "Target Setup File"]
   if {$File != ""} {
      set CurrentInDirectory [file dirname $File]
      DoOpenPAF $File
   }


}

# -----------------------------------------------------------------------------
#
#                           D o  O p e n  P A F
#
#   Read and open the a new PAF file. This can be called directly from code
#   such as the batch mode main line code that opens a file specified on the
#   command line, but is usually called in response to the file open dialogue
#   put up by OpenPAF{}. The specified PAF file will almost certainly be one
#   written by an earlier run of this program. The contents of the PAF file
#   are used to recreate the SDS configuration structure mainipulated by the
#   configuration program, and also to set some other things not stored in
#   the SDS structure, such as the ARGUS position angle and the VLT guide
#   star that has been selected. PAF format originated at ESO, and so will
#   usually by used only be used in the context of the FLAMES instrument, but
#   there is nothing to stop it being used for other instruments.

proc DoOpenPAF {file} {

    global Cfid               ;# Sds Id for the top level of the current
                              ;# configuration structure.
    global ConfigFile         ;# Currently displayed configuration file 
    global Edited             ;# True once the configuration has been changed
    global errorCode          ;# Error description (generated by Tcl)
    global errorInfo          ;# Stack trace describing an error 
    global OutputFileStatus   ;# Description of output file status
    global SelectedFibre      ;# Currently selected fibre (pivot) number - 
                              ;# starts from 1 - zero => no fibre selected.
    global SelectedObject     ;# Index of selected target object - starts at 1
    global SelectedGuidePivot ;# Indicates which of the two possible guide pivot
                              ;# positions has been selected.
    global SelectedGuidePosn  ;# Position of guider with selected pivot.
    global SelectedGuideState ;# State of guider position selection and display.
    global SelectedGuideObject;# Which guide object is selected
    global SelectedGuideOrient;# Guide probe orientation - "POS" or "NEG".
    global SelectedGuideName  ;# Guide probe name
    global FibreCombination   ;# Fibre combination selected for use
    global FibreComboMode     ;# Mode keyword for Fibre combination selected
    global FibreTypeFlags     ;# Array.  Indicates if each fibre type
                              ;# is enabled. See SelectFibreType for details.
    global TkAvailable        ;# Set if Tk operations are permitted.
    global validHourAngle     ;# Validated hour angle range.
    global CheckStatus        ;# Describes current check status
    global CurrentOutDirectory ;# Current directory for saving files
    global ListTargetFile     ;# List file saved: proposed target setup file name.
    global CurrentOutDirectory ;# Current directory for saving files

   #  Reading a PAF file is actually quite similar to reading an existing
   #  SDS file, except that we use ReadPAF, which creates a new SDS structure
   #  that recreates the one we had when we wrote the PAF file. So it's
   #  almost as if we had read that file instead of creating it from the
   #  PAF file, and most of what has to happen next (except for the business
   #  with the VLT guide probe position) is the same as for DoOpenSDS{}.

   # Read the file.
   
    set chkvalue [PafChkVerify $file] 

    if { $chkvalue != 0 } {
	switch $chkvalue {
	    "1" {set output "Error has ocurred on checksum verification!"} 
            "2" {set output "PAF file has an incorrect checksum!"}
            "4" {set output "PAF file does not have a checksum!"}
            default {set output "No indication $chkvalue !!"}
	}
	set answer [tk_messageBox -message "$output. Do you want to load this file?" -type okcancel -icon question]
            switch -- $answer {
                ok { set chkvalue 0 }
                cancel {return 1}
            }

    }
   
    set temp2 [ReadPAF $file SelGuideIndex Posneg ModeCode validHourAngle]
   
    # Access the structure.
    
    SetUTMeridian $temp2
    ConvertXy $temp2
   
    # Calculate the button angles for all the fibres - these aren't saved
    # in the PAF file.
   
    set Fibre 1
    while {$Fibre <= [NumPivots]} {
	SetButtonAngle $temp2 [PivotsId] $Fibre radial
	incr Fibre
    }
	
    # AddIndex may fail here if we have duplicated objects.  If that
    # happends, we need to restore the old file, if any, and delete
    # the new one.
   
    if [catch "AddIndex $temp2" msg] {
	set savedInfo $errorInfo
	set savedCode $errorCode
	ArgDelete $temp2
	error $msg $savedInfo $savedCode
    } else {
	
	# Free up the old configuration.
	if { $Cfid != 0 } {
	    ArgDelete $Cfid
	}
	    
	#  Setup the global variable.
	
	set ConfigFile $file
	set Cfid $temp2
	SetTick OpenFile
	
	#  File has not been edited.
	
	set Edited 0
	set OutputFileStatus "Not saved"
	set CurrentOutDirectory [file dirname $file]
	set ListTargetFile [file tail $file]
	if { $validHourAngle > 0 } {
	    set CheckStatus "Checked ok for +/- $validHourAngle hours"
	} else {
	    set CheckStatus "Not checked"
	}
	OpenSDSOk
	
	#  If the fibre combination could be determined, set that up
	
	if { $ModeCode >= 0 } {
	    set FibreCombination $ModeCode
	    set Fibres [FibreTypes]
	    set i 0
	    foreach fibre $Fibres {
		set FibreTypeFlags($i) 0
		incr i
	    }
	    set TypeCodeList [FibreComboTypes $FibreCombination]
	    foreach TypeCode $TypeCodeList {
		set FibreTypeFlags($TypeCode) 1
	    }
	    set FibreComboMode [FibreComboMode $FibreCombination]
	    CurrentFibreCombo
	    MarkFibreType FibreTypeFlags
	    SetTick FibreCombo
	}
	
	#  Display the field
	
	if { $TkAvailable } { wm title .mimic $ConfigFile }
	catch {set SelectedFibre 0}
	catch {unset SelectedObject}
	
	#  And now we have to deal with the selected VLT guide probe
	#  position, if any. 
	
	if { $SelGuideIndex } {
	    set id [SdsFind $Cfid unallocGuide]
	    VLTGuider $id $SelGuideIndex posn1 posn2 pospos
	    set SelectedGuideObject $SelGuideIndex
	    if { $Posneg == "POS" } {
		set PivNo $pospos
	    } else {
		if { $pospos == 1 } {
		    set PivNo 2
		} else {
		    set PivNo 1
		}
	    }
	    set SelectedGuidePivot $PivNo
	    set SelectedGuideOrient $Posneg
	    set SelectedGuideState 0
	    if { $PivNo == 1 } {
		set SelectedGuidePosn $posn1
	    } else {
		set SelectedGuidePosn $posn2
	    }
	    
	    #  We need the object name to identify the guide object 
	    #  unambiguously - the index number will change after an
	    #  allocation, the position (which was used by earlier versions
	    #  of the code) can change if the observation date or other
	    #  astrometric quantities are changed.
	    
	    GetObjectData $id $SelectedGuideObject obj_name obj_ra obj_dec type \
		spect  obj_prio obj_mag obj_pid obj_comment obj_text obj_x obj_y
	    set SelectedGuideName $obj_name
	    SdsFreeId $id
	    ShowGuidePivot
	    DisableShaddowedPivots
	    SetTick GuideStar
	} else {
	    DeselectGuidePivot
	}
	ZoomMimic 1
    }
    
    #  Update the statistics showing target allocation by fibre type
    
    UpdateFibreStats $Cfid
   
    ReformatFibreStats .allocstats.details .availstats.details
}


# -----------------------------------------------------------------------------
#
#                           M e r g e  F i l e s
#
#  This procedure implements a command which merges the "allocated objects"
#  component of an existing allocation with the unallocated components of
#  another field. 
 
proc MergeFiles {} {

   #  Global variables used by this routine:
   
   global CurrentInDirectory ;# Current directory for reading files
   
   # First check if we have to save the existing file.  If this returns
   # false, the user cancelled the operation.
   
   if { [ SaveCheck ] == 0 } {
       return
   }

   # Grab an SDS file name

    set types {
        {{SDS files} {*.sds}}
        {{All Files}      *}
   }

   set File [tk_getOpenFile    \
       -filetypes $types \
       -initialdir $CurrentInDirectory \
       -title "Allocated Fibre Configuration File"]
   if {$File != ""} {
      set CurrentInDirectory [file dirname $File]
      DoMergeOpenSds $File
   }

}

# -----------------------------------------------------------------------------
#
#                       D o  M e r g e  O p e n  S d s
#
#  This procedure is invoked when we have an allocated fibre merge file 
#  to open.

proc DoMergeOpenSds { filename } {

    #  Global variables used by this routine:
   
    global CurrentInDirectory  ;# Current directory for reading files
    global MergeSdsFilename    ;# SDS file to be merged  
    global ConfigFile          ;# Currently displayed configuration file
    
    set MergeSdsFilename $filename

    # Do a test read of the file before we proceed.  We should
    # also be checking if it has an "objects" array.
    
    set temp [SdsRead $MergeSdsFilename]
    set oid  [SdsFind $temp "objects"]
    
    # Tidy up the read file's memory.
    
    SdsFreeId $oid
    SdsReadFree $temp
    SdsFreeId $temp

    # If we are here ok, then prompt for the other file, which can
    # be either SDS or Ascii.
    
    set types {
        {{Ascii files} {*.fld}}
        {{Sds files} {*.sds}}
        {{All Files}      *}
   }

   # We need the file containing the unallocated objects to merge in

   set File [tk_getOpenFile    \
       -filetypes $types \
       -initialdir $CurrentInDirectory \
       -title "Source of unallocated objects"]
   if {$File != ""} {
      set CurrentInDirectory [file dirname $File]
      DoMergeOpenOther $File
   }
   
}

# -----------------------------------------------------------------------------
#
#                     D o  M e r g e  O p e n  O t h e r
#
#  This procedure is invoked from the file open dialogue put up by 
#  DoMergeOpenSds{}. It opens the other file and performs the merge.

proc DoMergeOpenOther { filename } {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global ConfigFile        ;# Currently displayed configuration file
    global MergeSdsFilename  ;# SDS file to be merged  
    global OutputFileStatus  ;# Description of output file status
    global validHourAngle    ;# Validated hour angle range.
    global CheckStatus       ;# Describes current check status
    global ListSaved         ;# Flag: current config. list is saved
    
    # We don't know at this point if we have an Sds file or an Ascii
    # (.fld) file.  Try to open it as SDS, if that files, try to
    # convert from Ascii.

    if [catch "SdsRead $filename" OtherFileId] {
        set SdsReadOk 0
        set OtherFileId [ConvertAscii fracDayFlag $filename]
        
    } else {
        set SdsReadOk 1
    }

    # The unallocated file is now open.  Reopen the SDS file containing
    # the allocated objects.
    
    set AllocFileId [SdsRead $MergeSdsFilename]

    # We have both files now.
      
    SdsList $OtherFileId
    SdsList $AllocFileId

    # Validate field centers
    
    if [ MergeCheckCentersOk $AllocFileId $OtherFileId] {

        # Merge the fields.
        
        set NewId [MergeFields $AllocFileId $OtherFileId]

        # Ready to make this the new field. Free up the old configuration.
        
        if { $Cfid != 0 } {
            ArgDelete $Cfid
        }

        # Generate a filename.
        
        set ConfigFile [NewSdsFileName [pwd]/MergeFile]
        set Cfid $NewId
        SetTick OpenFile
        
        # Access the structure.
        
        if { $fracDayFlag == 0 } { SetUTMeridian $Cfid }
        MakeAllocated $Cfid [PivotsId]
        ConvertXy $Cfid
        AddIndex $Cfid
	set validHourAngle 0
        set Edited 1
        set ListSaved 0
        set OutputFileStatus "Not saved"
        set CheckStatus "Not checked"
        OpenSDSOk

        # Display the field

        wm title .mimic $ConfigFile
        catch {set SelectedFibre 0}
        catch {unset SelectedObject}
        ZoomMimic 1
       
    }

    if { $SdsReadOk } {
        SdsReadFree $OtherFileId
    }
    SdsReadFree $OtherFileId
    SdsReadFree $AllocFileId
    SdsReadFree $AllocFileId

   #  Update the statistics showing target allocation by fibre type
   
   UpdateFibreStats $Cfid
   
   ReformatFibreStats .allocstats.details .availstats.details
   
}

# -----------------------------------------------------------------------------
#
#                  M e r g e  C h e c k  C e n t e r s  O k
#
#  This procedure checks the field centers in the two files that are to be
#  merged (their top level SDS Ids are passed as the two arguments). It
#  returns 1 (true) if the field centers match and zero (false) if they
#  do not.

proc MergeCheckCentersOk { AllocFileId OtherFileId } {
    
    set FieldId [SdsFind $AllocFileId "fieldData"]
    set AllocCenRa  [ArgGet $FieldId "cenRa"]
    set AllocCenDec [ArgGet $FieldId "cenDec"]
    SdsFreeId $FieldId

    set FieldId [SdsFind $OtherFileId "fieldData"]
    set OtherCenRa  [ArgGet $FieldId "cenRa"]
    set OtherCenDec [ArgGet $FieldId "cenDec"]
    SdsFreeId $FieldId

    if { ($AllocCenRa  != $OtherCenRa) ||
         ($AllocCenDec != $OtherCenDec) } {
        ErsOut "Field centers do not match - can't merge"
        return 0
    }
    return 1
}

# -----------------------------------------------------------------------------
#
#                          M e r g e  F i e l d s
#
#  This procedure actually merges the data in the two files that are to be
#  merged (their top level SDS Ids are passed as the two arguments).

proc MergeFields { AllocFileId OtherFileId } {

    #  Copy the allocated field to get the new structure
    
    set NewFile [SdsCopy $AllocFileId]

    #  Now we want to delete the unallocGuide and unallocSky components

    if { ![catch "SdsFind $NewFile unallocGuide" tempid]} {
        MsgOut "Deleting unallocGuide structure in allocated field"
        SdsDelete $tempid
        SdsFreeId $tempid
    }
    if { ![catch "SdsFind $NewFile unallocObject" tempid]} {
        MsgOut "Deleting unallocObject structure in allocated field"
        SdsDelete $tempid
        SdsFreeId $tempid
    }

    #  Now find and copy the equivalent structures in the other file
    
    MsgOut "Merging in unallocGuide structure from unallocated field"
    set tempid1 [SdsFind $OtherFileId unallocGuide]
    set tempid2 [SdsCopy $tempid1]
    SdsInsert $NewFile $tempid2
    SdsFreeId $tempid1
    SdsFreeId $tempid2

    MsgOut "Merging in unallocObject structure from unallocated field"
    set tempid1 [SdsFind $OtherFileId unallocObject]
    set tempid2 [SdsCopy $tempid1]
    SdsInsert $NewFile $tempid2
    SdsFreeId $tempid1
    SdsFreeId $tempid2

    #  Should now be ready to go.
    
    MsgOut "Merge complete"
    return $NewFile
    
}

# -----------------------------------------------------------------------------
#
#                               S e t  H A
#
#   This procedure is tied to the "Set Hour Angle..." commands menu item.
#   It puts up a dialogue to get a new hour angle value and invokes SetHaOk{}
#   when its OK button is selected.

proc SetHA {} {

    global HA                ;# Tied to the entry variable in Set HA dialog
    global config_font_1     ;# Font used in dialogues

    catch {destroy .setha}
    toplevel .setha
    wm title .setha "Set Hour Angle"

    frame .setha.top

    set ex [info exists HA]
    if {$ex == 0} {set HA 0.0}

    message .setha.top.msg -width 90m \
            -text "Set Hour Angle to..." \
            -font $config_font_1
    pack .setha.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
    pack .setha.top -side top

    frame .setha.ha
    pack .setha.ha -side top -padx 5m -pady 3m
    label .setha.ha.label -text "H.A.:" -width 10 -anchor w
    pack .setha.ha.label -side left -anchor w
    entry .setha.ha.ftext1 -textvariable HA -width 10 \
        -relief sunken -borderwidth 2
    DtcluBindEntryReal .setha.ha.ftext1 "neg"
    pack .setha.ha.ftext1 -side left

    DrawButtons .setha SetHaOk
    bind .setha.ha.ftext1 <Return> \
        ".setha.buttons.ok flash; destroy .setha; SetHaOk"

    Centre .setha
}

# -----------------------------------------------------------------------------
#
#                               S e t  H a  O k
#
#  Procedure invoked when the OK button in the dialogue put up by SetHA{}
#  is selected. This applies the new hour angle specified in the dialogue.

proc SetHaOk {} {

    global HA                ;# Tied to the entry variable in Set HA dialog
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global Edited            ;# True once the configuration has been changed
    global OutputFileStatus  ;# Description of output file status
    global ListSaved         ;# Flag: current config. list is saved

    # Save a copy of the current configuration.
    
    set save [SdsCopy $Cfid]

    SetUTMeridian $Cfid $HA
    set status [ConvertXy $Cfid]
    if {$status == "OK"} { 
        OpenSDSOk
        set Edited 1
        set ListSaved 0
        set OutputFileStatus "Not saved"
        StartUpdate $Cfid [PivotsId]

        # Copy no longer needed.
        
        ArgDelete $save
    } else {

        # Revert to saved copy.
        
        ArgDelete $Cfid
        set Cfid $save
        ConvertXy $Cfid
        OpenSDSOk
        StartUpdate $Cfid [PivotsId]
    }
}
 
# -----------------------------------------------------------------------------
#
#                   S e l e c t  F i b r e  T y p e
#
# Puts up a dialog box which allows the user to select which fibre types
# are to be configured.

proc SelectFibreType {} {

    #  Global variables used:

    global config_font_1  ;# Font used in dialogues
    
    #  Global variables modified (indirectly, by being tied to the check
    #  buttons in the dialog).
    
    global FibreTypeFlags ;# Array.  Indicates if each fibre type
                          ;# is enabled.
    
    #  Note how FibreTypeFlags works. There is one entry in this array for
    #  each fibre type returned by the c-implemented Tcl command FibreTypes.
    #  This list of fibre types is ultimately returned by FpilConSpecInfo().
    #  It is important when we use fibre type index numbers to access
    #  this array that the index numbers match properly. The SelectFibreType
    #  dialog is created with entries based on what FibreTypes returns, so
    #  that will be fine, so long as we don't mess with the order. The other
    #  place this comes in is in the SelectFibreCombo code, where the index
    #  numbers are those returned by the FibreComboTypes Tcl command, which
    #  ultimately gets the indices from FpilComboFibreCodes(). It's important
    #  that these two Fpil routines use a consistent set of index numbers.

    catch {destroy .selectspec}
    toplevel .selectspec
    wm title .selectspec "Select Fibre Type"
    wm geom .selectspec +100+0 

    frame .selectspec.top

    #  Get the list of fibre types.

    set FibreTypeList [FibreTypes]  

    #  Create the window.

    message .selectspec.top.msg -width 90m \
            -text "Select fibre types to configure..." \
            -font $config_font_1
    pack .selectspec.top.msg -side right -fill x -padx 5m -pady 2m
    pack .selectspec.top -side top

    frame .selectspec.spec
    pack .selectspec.spec -side top -padx 5m -pady 1m

    #  Add a check button for each fibre type.

    set NumFibreTypes [llength $FibreTypeList]
    for {set i 0 } { $i < $NumFibreTypes } { incr i } {
       set name [lindex $FibreTypeList $i]     
       checkbutton .selectspec.spec.spec$i -text $name \
          -variable FibreTypeFlags($i) -width 20 -anchor w
       pack .selectspec.spec.spec$i -padx 2m -pady 2m

    }

    DrawButtons .selectspec SelectFibreTypeOk 
}

# -----------------------------------------------------------------------------
#
#                   S e l e c t  F i b r e  T y p e  O k
#
#  Invoked when the 'Select Fibre type' dialog's OK button is hit.
#

proc SelectFibreTypeOk {} {


    global FibreTypeFlags ;# Array.  Indicates if each fibre type
                          ;# is enabled. See SelectFibreType for details.

    MarkFibreType FibreTypeFlags
    DrawFibres
  
    #  See if this is a standard fibre combination.
    
    CurrentFibreCombo
    
    #  Tick 'fibre combination selected' in the basic sequence dialogue
    
    SetTick FibreCombo
    
    #  Update the allocation statistics layout to match new combination
    
    ReformatFibreStats .allocstats.details .availstats.details
    
    if { [IsArgusUsed] } {
        ArgusUsed
    } else {
        ArgusUnused
    }
}
 
# -----------------------------------------------------------------------------
#
#                       S e t  F i e l d  P l a t e
#
#   Prompt for field plate selection.  Should not be invoked if we have
#   only one field plate as the command should not be avaible in the menu.

proc SetFieldPlate {} {

    #  Global variables used:
   
    global ConfigPlate         ;# Plate currently being configured (0 or 1)
    global ConfigFile          ;# Currently displayed configuration file
    global config_font_1       ;# Font used in dialogues
   
    # First check if we have to save the existing file.  If this returns
    # false, the user cancelled the operation.
   
    if { [ SaveCheck ] == 0 } {
        return
    }

    catch {destroy .setplate}
    toplevel .setplate
    wm title .setplate "Set Field Plate"
    wm geom .setplate +100+0 

    frame .setplate.top

    set ConfigPlate [GetPlate]

    message .setplate.top.msg -width 90m \
            -text "Set Field Plate to..." \
            -font $config_font_1
    pack .setplate.top.msg -side right -fill x -padx 5m -pady 5m
    pack .setplate.top -side top

    frame .setplate.plate
    pack .setplate.plate -side top -padx 5m -pady 1m

    for { set i 0 } { $i < [NumFields] } { incr i} {
        radiobutton .setplate.plate.plate$i -text "Plate $i" \
          -variable ConfigPlate \
          -value $i -width 16 -anchor w
        pack .setplate.plate.plate$i -padx 2m -pady 2m

    }

    DrawButtons .setplate SetPlateOk 
}

# -----------------------------------------------------------------------------
#
#                       S e t  P l a t e  O k
#
#   Invoked when the OK button in the dialogue put up by SetFieldPlate{}
#   is selected.

proc SetPlateOk {} {

    global ConfigPlate       ;# Plate currently being configured (0 or 1)
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.

    SetPlate $ConfigPlate
    if {$Cfid != 0} {
        ConvertXy $Cfid
        ZoomMimic 1
    }
}
  

# -----------------------------------------------------------------------------
#
#                            S e t  W a v e
#
#   This procedure is tied to the "Set Wavelength..." item in the commands
#   menu.  It puts up a dialogue in which a new wavelength can be specified
#   and calls SetWaveOk{} when the OK button is selected.

proc SetWave {} {

    global WAVELENGTH        ;# Observing wavelength in Angstroms.
    global WavelengthInA     ;# True if wavelength is displayed in Angstroms.
    global WavelengthShown   ;# The wavelength displayed (in Angstroms or nm).
    global config_font_1     ;# Font used in dialogues

    catch {destroy .setwave}
    toplevel .setwave
    wm title .setwave "Set Wavelength"

    frame .setwave.top

    set ex [info exists WAVELENGTH]
    if {$ex == 0} {set WAVELENGTH [GetWavelength]}

    if { $WavelengthInA } {
        set WavelengthShown $WAVELENGTH
        message .setwave.top.msg -width 90m \
            -text "Set Configuration Wavelength (3000-10000 Angstroms) to..." \
            -font $config_font_1
    } else {
        set WavelengthShown [expr $WAVELENGTH / 10]
        message .setwave.top.msg -width 90m \
            -text "Set Configuration Wavelength (300-1000 nm) to..." \
            -font $config_font_1
    }
    pack .setwave.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
    pack .setwave.top -side top

    frame .setwave.wave
    pack .setwave.wave -side top -padx 5m -pady 3m
    entry .setwave.wave.ftext1 -textvariable WavelengthShown -width 10 \
        -relief sunken -borderwidth 2
    pack .setwave.wave.ftext1 -side left
    if { $WavelengthInA } {
        label .setwave.wave.label -text "Angstroms" -width 10 -anchor w
    } else {
        label .setwave.wave.label -text "nm" -width 10 -anchor w
    }
    pack .setwave.wave.label -side left -anchor w

    DrawButtons .setwave SetWaveOk

    Centre .setwave
}

# -----------------------------------------------------------------------------
#
#                            S e t  W a v e  O k
#
#   This procedure is invoked when the OK button in the dialogue put up by
#   SetWave{} is selected.

proc SetWaveOk {} {

    global WAVELENGTH        ;# Observing wavelength in Angstroms.
    global WavelengthInA     ;# True if wavelength is displayed in Angstroms.
    global WavelengthShown   ;# The wavelength displayed (in Angstroms or nm).
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.

    #  The variable associated with the wavelenth entry box is WavelengthShown
    #  which may be in Angstroms or nm. We use this to set WAVELENGTH, which
    #  in turn is used to set the wavelength used by the C level code. These
    #  need to be in Angstroms.
    
    if { $WavelengthInA } {
        set WAVELENGTH $WavelengthShown
    } else {
        set WAVELENGTH [expr $WavelengthShown * 10]
    }
    if {$WAVELENGTH < 3000 || $WAVELENGTH > 10000} { SetWave }

    SetWavelength $WAVELENGTH
    ConvertXy $Cfid
}
  
# -----------------------------------------------------------------------------
#
#                         S e t  R e f r a c t i o n
#
#   This procedure is tied to the "Refraction Parameters..." item in the 
#   commands menu.  It puts up a dialogue in which new values for
#   temperature, pressure and humidity can be specified and calls SetRefOk{}
#   when the OK button is selected.

proc SetRefraction {} {

    global TEMP           ;# Tied to the temperature in the parameters dialogue
    global PRESS          ;# Tied to the pressure in the parameters dialogue
    global HUMID          ;# Tied to the humidity in the parameters dialogue
    global config_font_1  ;# Font used in dialogues

    catch {destroy .setref}
    toplevel .setref
    wm title .setref "Set Refraction Parameters"

    frame .setref.top

    GetRefPars TEMP PRESS thumid
    set HUMID [expr {$thumid*100}]
  

    message .setref.top.msg -width 90m \
            -text "Set Refraction Parameters to..." \
            -font $config_font_1
    pack .setref.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
    pack .setref.top -side top

    frame .setref.temp
    label .setref.temp.label -text "Temperature (K):" -width 24 -anchor e
    pack .setref.temp.label -side left -anchor e
    pack .setref.temp -side top -padx 5m -pady 3m
    entry .setref.temp.ftext1 -textvariable TEMP -width 10 \
        -relief sunken -borderwidth 2
    DtcluBindEntryReal .setref.temp.ftext1 "non-neg"
    pack .setref.temp.ftext1 -side left

    frame .setref.press
    label .setref.press.label -text "Pressure (mb):" -width 24 -anchor e
    pack .setref.press.label -side left -anchor e
    pack .setref.press -side top -padx 5m -pady 3m
    entry .setref.press.ftext1 -textvariable PRESS -width 10 \
        -relief sunken -borderwidth 2
    DtcluBindEntryReal .setref.press.ftext1 "non-neg"
    pack .setref.press.ftext1 -side left

    frame .setref.humid
    label .setref.humid.label -text "Relative Humidity (%):" -width 24 -anchor e
    pack .setref.humid.label -side left -anchor e
    pack .setref.humid -side top -padx 5m -pady 3m
    entry .setref.humid.ftext1 -textvariable HUMID -width 10 \
        -relief sunken -borderwidth 2
    DtcluBindEntryReal .setref.humid.ftext1 "non-neg"
    pack .setref.humid.ftext1 -side left

    DrawButtons .setref SetRefOk


    Centre .setref
}

# -----------------------------------------------------------------------------
#
#                            S e t  R e f  O k
#
#   This procedure is invoked when the OK button in the dialogue put up by
#   SetRefraction{} is selected.

proc SetRefOk {} {

    global TEMP           ;# Tied to the temperature in the parameters dialogue
    global PRESS          ;# Tied to the pressure in the parameters dialogue
    global HUMID          ;# Tied to the humidity in the parameters dialogue
    global Cfid           ;# Sds Id for the top level of the current
                          ;# configuration structure.

    set thumid [expr {$HUMID/100}]
    SetRefPars $TEMP $PRESS $thumid
    if {$Cfid != 0} {ConvertXy $Cfid}
}
  
# -----------------------------------------------------------------------------
#
#                         S e t  T e l e s c o p e
#
#   This procedure is tied to the "Telescope Parameters..." item in the 
#   commands menu.  It puts up a dialogue in which new values for
#   the telescope model parameters can be specified and calls SetTelOk{}
#   when the OK button is selected. The details (and number) of such
#   parameters depend on the instrument (or at least, the telescope) in
#   question and these are ultimately supplied by an FPIL routine,
#   accessed through GetTelPars{} which is implemented in C in configure.c
 
proc SetTelescope {} {

    global config_font_1     ;# Font used in dialogues
    global TelNumParams      ;# Number of telescope parameters
    global TelParamArray     ;# Telescope parameter details
   
    catch {destroy .settel}
    toplevel .settel
    wm title .settel "Set Telescope Parameters"

    frame .settel.top

    # Get the telescope model parameter details.
    
    set ParamList [GetTelPars]

    # GetTelPars must return a three element list.
    
    if { [llength $ParamList] != 3 } {
        error "GetTelParams returned invalid list"
    }

    # Each element is a list, first the values, second the system names
    # and third the descrptive names.
    
    set ParamValues [lindex $ParamList 0]
    set ParamNames  [lindex $ParamList 1]
    set ParamDescrs [lindex $ParamList 2]

    # Get the number of parameters in the model.
    
    set TelNumParams [llength $ParamValues]
    if { $TelNumParams <= 0 } {
        ErsOut "The [Telescope] Telescope Model for [Instrument] \
                      has no telescope parameters which can be set"
        return
    }
    
    # Create message window.
    
    message .settel.top.msg -width 110m \
        -text "Set Telescope Pointing Parameters to..." \
        -font $config_font_1
    pack .settel.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
    pack .settel.top -side top

    # Get the maximum length of a parameter description.
    
    set MaxLength 0
    for { set i 0 } { $i < $TelNumParams } { incr i } {

        set Descr [lindex $ParamDescrs $i]
        set Length [string length $Descr]
        if { $Length > $MaxLength } {
            set MaxLength $Length
        }
        
    }

    # Allow for colon and space.
    
    incr MaxLength 3

    # For each telescope model parameter
    
    for { set i 0 } { $i < $TelNumParams } { incr i } {

        # Get the current value and the description.
        
        set Value [lindex $ParamValues $i]
        set Descr [lindex $ParamDescrs $i]

        # Save the value in the array
        
        set TelParamArray($i) $Value

        # Display the description and value.
        
        frame .settel.param$i
        label .settel.param${i}.label -text "$Descr :" \
            -width $MaxLength -anchor e
        pack  .settel.param${i}.label -side left -anchor e

        pack .settel.param${i} -side top -padx 5m -pady 3m

        entry .settel.param${i}.ftext1 -textvariable TelParamArray($i) \
            -width 10 -relief sunken -borderwidth 2
        DtcluBindEntryReal .settel.param${i}.ftext1 "neg"
        pack .settel.param${i}.ftext1 -side left

    }
    DrawButtons .settel SetTelOk
    Centre .settel
}

# -----------------------------------------------------------------------------
#
#                          S e t  T e l  O k
#
#   This procedure is invoked when the OK button in the dialogue put up by
#   SetTelescope{} is selected.
 
proc SetTelOk {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global TelNumParams      ;# Number of telescope parameters
    global TelParamArray     ;# Telescope parameter details
    global zoom              ;# Current zoom factor - 1 is normal.
    
    # Put each telescope model parameter value into a list.
    
    for { set i 0 } { $i < $TelNumParams } { incr i } {
        lappend params $TelParamArray($i)
    }

    # Set them.
    
    SetTelPars $params
    
    #  If we have a configuration structure, re-calculate the positions of
    #  the targets.

    if {$Cfid != 0} {ConvertXy $Cfid}
    
    #  Force a redisplay of the mimic with the new target positions.
    
    ZoomMimic $zoom
}
  

# -----------------------------------------------------------------------------
#
#                            C h e c k
#
#   This procedure is tied to the "Check over HA range..." item in the commands
#   menu. It puts up a dialogue that allows the user to specify the details
#   of the test to be performed, then calls CheckOk{} when the OK button is
#   clicked on. If the auto-reallocate option is selected, it calls
#   CheckReAllocateOk{} instead of CheckOk{}.

proc Check {} {

    global config_font_1   ;# Font used in dialogues
    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure.
    global CheckMode       ;# Non-zero if a check should attempt to reallocate.
    global ExpertMode      ;# True if the system has been set to 'expert mode'.
    global UTyear          ;# Tied to the year in the HA range dialogue
    global UTmonth         ;# Tied to the month in the HA range dialogue
    global UTday           ;# Tied to the day in the HA range dialogue
    global Nights          ;# Tied to the # of nights in the HA range dialogue
    global MaxHA           ;# Tied to the maximum HA in the HA range dialogue
    global WavelengthCheck ;# Non-zero if wavelength ranges are to be checked.

    set fid [SdsFind $Cfid fieldData]
    set mjd [ArgGet $fid configMjd]
    SdsFreeId $fid
    set UTDATE [Mjd2date $mjd]
    set UTyear [string range $UTDATE 0 3]
    set UTmonth [string range $UTDATE 5 6]
    set UTday [string range $UTDATE 8 9]

    set ex [info exists MaxHA]
    if {$ex == 0} {set MaxHA 4.0}
    set ex [info exists Nights]
    if {$ex == 0} {set Nights 1}
    
    catch {destroy .check}
    toplevel .check
    wm title .check "Checking Allocation"
      
    frame .check.top

    message .check.top.msg -width 90m \
              -text "Check Validity of Allocation for..." \
              -font $config_font_1
    pack .check.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
    pack .check.top -side top
      
    frame .check.date
    pack .check.date -side top -padx 5m -pady 3m
    label .check.date.label -text "Starting Date:" -width 16 -anchor w
    pack .check.date.label -side left
    label .check.date.lab1 -text "Year:" -width 6
    pack .check.date.lab1 -side left
    entry .check.date.ftext1 -textvariable UTyear -width 5 \
              -relief sunken -borderwidth 2
    DtcluBindEntryInt .check.date.ftext1 1950 2050
    pack .check.date.ftext1 -side left
    label .check.date.lab2 -text "Month:" -width 6
    pack .check.date.lab2 -side left
    entry .check.date.ftext2 -textvariable UTmonth -width 4 \
          -relief sunken -borderwidth 2
    DtcluBindEntryInt .check.date.ftext2 1 12
    pack .check.date.ftext2 -side left
    label .check.date.lab3 -text "Day:" -width 5
    pack .check.date.lab3 -side left
    entry .check.date.ftext3 -textvariable UTday -width 4 \
      -relief sunken -borderwidth 2
    DtcluBindEntryInt .check.date.ftext3 1 31
    pack .check.date.ftext3 -side left

    frame .check.days
    pack .check.days -side top -padx 5m -pady 3m
    label .check.days.label -text "Number of Nights:" -width 22 -anchor w
    pack .check.days.label -side left -anchor w
    entry .check.days.ftext1 -textvariable Nights -width 5 \
          -relief sunken -borderwidth 2
    DtcluBindEntryInt .check.days.ftext1 0 1000
    pack .check.days.ftext1 -side left

    frame .check.ha
    pack .check.ha -side top -padx 5m -pady 3m
    label .check.ha.label -text "Maximum H.A.:" -width 22 -anchor w
    pack .check.ha.label -side left -anchor w
    entry .check.ha.ftext1 -textvariable MaxHA -width 5 \
          -relief sunken -borderwidth 2
    DtcluBindEntryReal .check.ha.ftext1 "neg"
    pack .check.ha.ftext1 -side left

    set CheckMode 0
    frame .check.mode
    pack .check.mode -side top -padx 5m -pady 3m
    checkbutton .check.mode.button -variable CheckMode \
              -text "Attempt to reallocate invalid fibres" -anchor w
    pack .check.mode.button -side left

    set WavelengthCheck 0
    if { $ExpertMode } {
        frame .check.waves
        pack .check.waves -side top -padx 5m -pady 3m
        checkbutton .check.waves.button -variable WavelengthCheck \
              -text "Check all wavelength combinations" -anchor w
        pack .check.waves.button -side left
    }

    DrawButtons .check {if {$CheckMode == 0} {CheckOk} \
                       else { CheckReAllocateOk ; set CheckMode 0}}


    Centre .check

}


# -----------------------------------------------------------------------------
#
#                          C h e c k  O k
#
#   This procedure is invoked when the OK button in the dialogue put up by
#   Check{} is selected. It works through the HA range specified and runs a
#   field validity check at each step.
  
proc CheckOk {} {

    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure.
    global UTyear          ;# Tied to the year in the HA range dialogue
    global UTmonth         ;# Tied to the month in the HA range dialogue
    global UTday           ;# Tied to the day in the HA range dialogue
    global Nights          ;# Tied to the # of nights in the HA range dialogue
    global MaxHA           ;# Tied to the maximum HA in the HA range dialogue
    global WavelengthCheck ;# Non-zero if wavelength ranges are to be checked.
    global validHourAngle  ;# Validated hour angle range.
    global CheckStatus     ;# Describes current check status

    set copy [SdsCopy $Cfid]
    set save $Cfid
    set Cfid $copy
        
    #  If we are going to perform the check for different wavelength
    #  combinations, we get the details of the number of such combinations
    #  given the current fibre combination. If we are not checking wavelengths
    #  then we only use the first combination, which is always defined to
    #  be the current observing wavelength.
    
    set WavelengthComboList [WavelengthCombos]  
    if ($WavelengthCheck) {
        set NumWavelengthCombos [llength $WavelengthComboList]
    } else {
        set NumWavelengthCombos 1
    }
    
    set AllocError 0
    set i 1
    SetUT $Cfid $UTyear $UTmonth $UTday
    while {$i <= $Nights} {
        foreach ha "-$MaxHA 0 $MaxHA" {
            SetUTMeridian $Cfid $ha
            for {set iCombo 0} {$iCombo < $NumWavelengthCombos} {incr iCombo} {
                set ComboName [lindex $WavelengthComboList $iCombo]
                if { $WavelengthCheck } { MsgOut "Using: $ComboName" }
                SetWavelengthCombo $iCombo
                set status [ConvertXy $Cfid]
                if {$status == "OK"} {
                    OpenSDSOk
                    update
                    WorkingDialog .work "Checking Allocation..."
                    set status [DoCheck $Cfid]
                    destroy .work
                    if {$status == "OK"} {
                        MsgOut "Allocation OK"
                    } else {
                        MsgOut "Allocation Invalid"
                        set AllocError 1
                        break
                    }
                } else {
                    set AllocError 1
                    break
                }
            }
            if { $AllocError } break          
        }


        if { $AllocError } break          
        NextDay $Cfid
        incr i
    }
    if { $AllocError == 0 } {
    	set validHourAngle $MaxHA
	set CheckStatus "Checked ok for +/- $MaxHA hours"
	SetTick Check
    }
    
    #  Always make sure we end up using the default wavelength combination
    
    SetWavelengthCombo 0
    
    # Delete copy and restore Cfid
    
    ArgDelete $Cfid
    set Cfid $save

    ConvertXy $Cfid
    OpenSDSOk
}

# -----------------------------------------------------------------------------
#
#                  C h e c k  R e  A l l o c a t e  O k
#
#   This procedure is invoked when the OK button in the dialogue put up by
#   Check{} is selected and the auto-reallocate option has been selected in
#   the dialogue. It works through the HA range specified and runs a
#   field validity check with auto reallocation at each step.

proc CheckReAllocateOk {} {

    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure.
    global UTyear          ;# Tied to the year in the HA range dialogue
    global UTmonth         ;# Tied to the month in the HA range dialogue
    global UTday           ;# Tied to the day in the HA range dialogue
    global Nights          ;# Tied to the # of nights in the HA range dialogue
    global MaxHA           ;# Tied to the maximum HA in the HA range dialogue

    # Was this intended to allow us to revert on error or something? I don't
    # know, but it is not being used and is just causing a
    # memory leak (TJF, 21-Jun-1999)
    #   set copy [SdsCopy $Cfid]
    #   set save $Cfid
    #   set Cfid $copy

    global MaxHA
    set ex [info exists MaxHA]
    if {$ex == 0} {set MaxHA 4.0}
    set ex [info exists Nights]
    if {$ex == 0} {set Nights 1}

    set i 1
    SetUT $Cfid $UTyear $UTmonth $UTday
    while {$i <= $Nights} {
        foreach ha "-$MaxHA 0 $MaxHA" {
            SetUTMeridian $Cfid $ha
            set status [ConvertXy $Cfid]
            if {$status == "OK"} {
                OpenSDSOk
                update
                WorkingDialog .work "Checking Allocation..."
                set status [DoCheck $Cfid]
                destroy .work
                if {$status == "OK"} {
                    MsgOut "Allocation OK"
                } else {
                
                    #  Since we're now actually at the HA of interest, we can 
                    #  try to run the autoreallocate routine from here:
                                   
                    ReAllocateNoUncross
                    MsgOut "Allocation was Patched"
                    puts stdout "Allocation was Patched"
                    set status [DoCheck $Cfid]
                    set ha1 0
                    SetUT $Cfid $UTyear $UTmonth $UTday
                    SetUTMeridian $Cfid $ha1
                    set status [ConvertXy $Cfid]
                    OpenSDSOk
                    if {$status != "OK"} {
                        puts stdout "Problem Converting XY"
                    }
                    set i [expr {$i - 1}]
                    break
                }
            } else {
                set i [expr {$Nights + 1}]
                break
            }           
        }


        NextDay $Cfid
        incr i
    }
    set ha 0
    SetUT $Cfid $UTyear $UTmonth $UTday
    SetUTMeridian $Cfid $ha
    ConvertXy $Cfid
    OpenSDSOk
    ZoomMimic 1
}

# -----------------------------------------------------------------------------
#
#                          C h e c k  O k  1
#
#   This procedure is similar to CheckOk{}. but is intended to be used when
#   an initial allocation and check is run in batch mode. It works through
#   the HA range specified and runs a field validity check at each step.

proc CheckOk1 {} {

    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure.
    global UTyear          ;# Tied to the year in the HA range dialogue
    global UTmonth         ;# Tied to the month in the HA range dialogue
    global UTday           ;# Tied to the day in the HA range dialogue
    global Nights          ;# Tied to the # of nights in the HA range dialogue
    global MaxHA           ;# Tied to the maximum HA in the HA range dialogue
    global validHourAngle  ;# Validated hour angle range.
    global CheckStatus     ;# Describes current check status

#   Save the original configuration.

    set copy [SdsCopy $Cfid]
    set save $Cfid
    set Cfid $copy

    set ex [info exists MaxHA]
    if {$ex == 0} {
        set MaxHA 4.0
        set fid [SdsFind $Cfid fieldData]
        set mjd [ArgGet $fid configMjd]
        set UTDATE [Mjd2date $mjd]
        set UTyear [string range $UTDATE 0 3]
        set UTmonth [string range $UTDATE 5 6]
        set UTday [string range $UTDATE 8 9]
        set Nights 1
    }

    set i 1
    SetUT $Cfid $UTyear $UTmonth $UTday
    while {$i <= $Nights} {
        foreach ha "-$MaxHA 0 $MaxHA" {
            SetUTMeridian $Cfid $ha
            set status [ConvertXy $Cfid]
            if {$status == "OK"} {
                OpenSDSOk
                update
                WorkingDialog .work "Checking Allocation..."
                set status [DoCheck1 $Cfid]
                ##set status "OK"
                destroy .work
                
                #  Keep going here, so we pass through all the errors...
                
            } else {
                set i [expr {$Nights + 1}]
                break
            }           
        }


        NextDay $Cfid
        incr i
    }
    ArgDelete $Cfid
    set Cfid $save
    ConvertXy $Cfid
    OpenSDSOk

}

# -----------------------------------------------------------------------------
#
#                          C h e c k  A l l o c
#
#   This procedure is tied to the "Check Allocation" option in the commands
#   menu. It checks the current configuration for validity.

proc CheckAlloc {} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.

    WorkingDialog .work "Checking Allocation..."
    set status [DoCheck $Cfid]
    destroy .work
    if {$status == "OK"} {
        MsgOut "Allocation OK"
    } else {
        MsgOut "Allocation invalid"
    }
}

#------------------------------------------------------------------------------ 

#                    H i g h l i g h t  T a r g e t s
#
#   Puts up a dialog box which allows the user to select which targets are
#   to be highlit.  It puts up a dialog with a number of items that can be
#   used to control the selection, together with three buttons: an 'apply'
#   button, which causes the selected criteria to be applied, through the
#   procedure HighlightByCriteria{}; a 'clear' button that resets the 
#   criteria to default values, via the procedure HighlightTargetsClear{};
#   and a 'cancel' button that deletes the dialog.

proc HighlightTargets {} {

    #  Global variables used by this procedure :
    
    global TargetNumCriteria  ;# No. of instrument-dependent selection criteria
    global config_font_1      ;# Font used in dialogues
    global zoom               ;# Current zoom factor - 1 is normal.
    
    #  Global variables modified by this procedure (in the sense that they
    #  are associated with checkboxes and radiobuttons and entries in the
    #  dialog, and so can be set by the user interacting with the dialog):
    
    global ArrowHighlighting  ;# Flag set if arrow highlighting is enabled
    global TargetCriteriaFlags;# Flags set to show target selection criteria.
    global TargetHighPriority ;# Entry variable for selected top priority
    global TargetLowPriority  ;# Entry variable for selected low priority
    global TargetByAllocation ;# Entry variable used to select by allocation
    global TargetResult       ;# Number of selected targets
    global TargetScale        ;# Scale factor used to magnify targets.
      
    catch {destroy .highlightTargets}
    toplevel .highlightTargets
    wm title .highlightTargets "Highlight Targets"
    wm geom .highlightTargets +100+0 

    frame .highlightTargets.top

    #  Get the list of fibre types.
    
    set TargetCriteriaList [TargetCriteria]  

    #  Create the window.
    
    message .highlightTargets.top.msg -width 90m \
            -text "Select targets to highlight" \
            -font $config_font_1
    pack .highlightTargets.top.msg -side right -fill x -padx 5m -pady 2m
    pack .highlightTargets.top -side top

    frame .highlightTargets.criteria
    pack .highlightTargets.criteria -side top -padx 5m -pady 1m

    #  Then a box to display the number of selected objects
    
    frame .highlightTargets.criteria.result
    pack .highlightTargets.criteria.result -side top -anchor center -pady 5m
    set TargetResult 0
    label .highlightTargets.criteria.result.txt1 -textvariable TargetResult \
                                                    -width 4 -relief sunken
    pack .highlightTargets.criteria.result.txt1 -side left
    label .highlightTargets.criteria.result.txt2 -text " Targets selected"
    pack .highlightTargets.criteria.result.txt2 -side left

    #  Add a check button for each instrument-dependent criterion. These
    #  are tied to elements 0 through TargetNumCriteria-1 of the global array
    #  TargetCriteriaFlags.

    set Done 0
    set iCrit 0
    for {set ilev 0 } { !$Done } { incr ilev } {
        frame .highlightTargets.criteria.lev$ilev
        pack .highlightTargets.criteria.lev$ilev -side top -anchor w
        for { set ibutton 0 } { $ibutton < 2 } { incr ibutton } {
            if {($iCrit >= $TargetNumCriteria)} {
                set Done 1
            } else {
               set name [lindex $TargetCriteriaList $iCrit]     
               checkbutton .highlightTargets.criteria.lev$ilev.criterion$iCrit \
                          -text $name -variable TargetCriteriaFlags($iCrit) \
                                                        -width 20 -anchor w
               pack .highlightTargets.criteria.lev$ilev.criterion$iCrit \
                                         -side left -padx 2m -pady 2m
               incr iCrit
            }
        }
    }
    
    #  Then add the standard criteria.  First a priority range. The check
    #  button for this is tied to element TargetNumCriteria of the global array
    #  TargetCriteriaFlags, and the two priority calues are tied to
    #  the global variables TargetHighPriority and TargetLowPriority.

    frame .highlightTargets.criteria.prio
    pack .highlightTargets.criteria.prio -side top -anchor w
    checkbutton .highlightTargets.criteria.prio.txt1 \
                   -text "By priority: " -variable TargetCriteriaFlags($iCrit) \
                                                       -width 15 -anchor w
    pack .highlightTargets.criteria.prio.txt1 -side left -pady 2m -padx 2m
    entry .highlightTargets.criteria.prio.ent1 \
        -textvariable TargetLowPriority -width 2 -borderwidth 2 -relief sunken 
    pack .highlightTargets.criteria.prio.ent1 -side left -padx 2m -pady 2m
    label .highlightTargets.criteria.prio.txt2 -text " to " 
    pack .highlightTargets.criteria.prio.txt2 -side left -pady 2m -padx 2m
    entry .highlightTargets.criteria.prio.ent2 \
        -textvariable TargetHighPriority -width 2 -borderwidth 2 -relief sunken 
    pack .highlightTargets.criteria.prio.ent2 -side left -padx 2m -pady 2m
    incr iCrit

    #  Then the allocated and unallocated options. The check button for this
    #  is tied to element TargetNumCriteria+1 of the global array
    #  TargetCriteriaFlags, and the global variable TargetByAllocation will
    #  be set to 1 if Allocated is set and 0 if Unallocated is set.
    
    frame .highlightTargets.criteria.alloc
    pack .highlightTargets.criteria.alloc -side top -anchor w
    checkbutton .highlightTargets.criteria.alloc.txt1 \
               -text "By allocation: " -variable TargetCriteriaFlags($iCrit) \
                                                       -width 15 -anchor w
    pack .highlightTargets.criteria.alloc.txt1 -side left -pady 2m -padx 2m
    set TargetByAllocation 1
    radiobutton .highlightTargets.criteria.alloc.yes -text "Allocated" \
               -variable TargetByAllocation -value "1" -anchor w
    pack .highlightTargets.criteria.alloc.yes -side left -padx 2m -pady 2m
    radiobutton .highlightTargets.criteria.alloc.no -text "Unallocated" \
               -variable TargetByAllocation -value "0" -anchor w
    pack .highlightTargets.criteria.alloc.no -side left -padx 2m -pady 2m
    incr iCrit
    
    #  An entry box that allows the target drawings to be scaled up, bound
    #  to the global variable TargetScale.
    
    frame .highlightTargets.criteria.scale
    pack .highlightTargets.criteria.scale -side top -anchor c
    label .highlightTargets.criteria.scale.txt -text "Target scale factor: " 
    pack .highlightTargets.criteria.scale.txt -side left -pady 2m -padx 2m
    entry .highlightTargets.criteria.scale.ent \
        -textvariable TargetScale -width 3 -borderwidth 2 -relief sunken 
    pack .highlightTargets.criteria.scale.ent -side left -padx 2m -pady 2m
    button .highlightTargets.criteria.scale.redraw \
                -command "ZoomMimic \$zoom" -text Redraw -width 4 
    pack .highlightTargets.criteria.scale.redraw -side left -padx 2m -pady 2m
    
    #  Add the check button indicating that targets should be indicated
    #  by an arrow, as well as by being highlit. This is bound to the
    #  global variable ArrowHighlighting.
    
    checkbutton .highlightTargets.criteria.arrow \
          -text "Highlight using arrows" -variable ArrowHighlighting
    pack .highlightTargets.criteria.arrow -side top -padx 2m -pady 4m
                                         
        
    #  And now the 'Apply', 'Clear' and 'Close' buttons at the bottom. We
    #  can't do this using DrawButtons, because that has only a standard
    #  OK and Close button pair.
    
    frame .highlightTargets.buttons
    pack .highlightTargets.buttons -side bottom -fill both
    button .highlightTargets.buttons.apply -command "HighlightByCriteria" \
                                                         -text Apply -width 4
    frame .highlightTargets.buttons.default -relief sunken -bd 1
    raise .highlightTargets.buttons.apply .highlightTargets.buttons.default
    pack .highlightTargets.buttons.default -side left -expand 1 \
                                                             -padx 3m -pady 2m
    pack .highlightTargets.buttons.apply -in .highlightTargets.buttons.default \
                                       -padx 2m -pady 2m -ipadx 2m -ipady 1m
    button .highlightTargets.buttons.clear \
            -command "HighlightTargetsClear; HighlightByCriteria" \
                                                          -text Clear -width 4
    pack .highlightTargets.buttons.clear -side left -expand 1 -padx 2m \
                                                -pady 3m -ipadx 2m -ipady 1m
    button .highlightTargets.buttons.cancel \
                  -command "destroy .highlightTargets" -text Close -width 4
    pack .highlightTargets.buttons.cancel -side left -expand 1 \
                               -padx 2m -pady 3m -ipadx 2m -ipady 1m
    bind .highlightTargets <Return> \
             ".highlightTargets.buttons.apply flash; HighlightByCriteria"
}

#------------------------------------------------------------------------------ 

#                 H i g h l i g h t  T a r g e t s  C l e a r
#
#   Resets the global variables controlling the setting for the dialog set up
#   by HighlightTargets{} to their default values. This is invoked at
#   startup and by the 'clear' button in that dialog.

proc HighlightTargetsClear {} {

    #  Global variables set by this routine:
    
    global TargetCriteriaFlags;# Flags set to show target selection criteria.
    global TargetHighPriority ;# Entry variable for selected top priority
    global TargetLowPriority  ;# Entry variable for selected low priority
    global TargetByAllocation ;# Entry variable used to select by allocation
    global TargetResult       ;# Number of selected targets
    global TargetNumCriteria  ;# No. of instrument-dependent selection criteria
    
    set TargetByAllocation 1
    set TargetHighPriority " "
    set TargetLowPriority " "
    set TargetResult 0
    
    #  TargetNumCriteria is the number of instrument-dependent selection
    #  criteria. There are other general criteria, hence the +2 in the
    #  loop test.  See the detailed comments for the global variables.
    
    for { set i 0 } { $i < ($TargetNumCriteria + 2) } { incr i } {
        set TargetCriteriaFlags($i) 0
    }
      
}

#------------------------------------------------------------------------------ 

#                    P r e  A l l o c a t e  C h e c k
#
#   Performs a pre-allocation check. FLAMES, for example, should not be
#   configured without a VLT guide star selected. Most instruments should
#   not be configured with a non-standard fibre configuration selected.
#   This routine tests for these conditions and puts up a dialogue box
#   explaining the problem and giving the use the chance to cancel the
#   allocation or continue anyway. It returns true (1) if the allocation
#   should be abandoned, false (0) if it should be allowed to continue.

proc PreAllocateCheck {} {

    #  Global variables used by this procedure :
    
    global config_font_1      ;# Font used in dialogues
    global FibreCombination   ;# Fibre combination selected for use
    global SelectedGuideObject;# Which guide object is selected ( 0=> none).
    global SelectedGuidePivot ;# Indicates which of the two possible guide pivot
                              ;# positions has been selected.
        
    #  Global variables set by this procedure :
    
    global PreAllocFlag;
    
    #  First, we have to see if there are any problems to report.
    
    set PreAllocFlag 0
    
    set NoGuideObject 0
    set NoGuidePivot 0
    set NoStandardCombo 0
    
    if { [Instrument] == "FLAMES"} {
       if { $SelectedGuideObject == 0 } {
          set NoGuideObject 1
       } else {
          if { $SelectedGuidePivot == 0 } { set NoGuidePivot 1 }
       }
    }
    CurrentFibreCombo
    if { $FibreCombination < 0 } { set NoStandardCombo 1 }
    
    set Problems [expr $NoStandardCombo + $NoGuideObject + $NoGuidePivot]
    
    if { $Problems } {
     
       catch {destroy .preAllocate}
       toplevel .preAllocate
       wm title .preAllocate "Pre-allocation Check"
       wm geom .preAllocate +100+0 

       frame .preAllocate.top

       #  Create the window.
    
       if { $NoGuideObject } {
          label .preAllocate.top.msg1 \
                  -text "There is no VLT guide star selected" \
                  -font $config_font_1
          pack .preAllocate.top.msg1 -padx 5m -pady 2m
          label .preAllocate.top.msg2 \
             -text "(Double-click a VLT guide star, select a pivot position)"
          pack .preAllocate.top.msg2 -padx 5m -pady 1m
       }
       if { $NoGuidePivot } {
          label .preAllocate.top.msg3 \
                  -text "No pivot position for VLT guide probe selected" \
                  -font $config_font_1
          pack .preAllocate.top.msg3 -padx 5m -pady 2m
          label .preAllocate.top.msg4 \
             -text "(Click on one of the blue pivot positions)"
          pack .preAllocate.top.msg4 -padx 5m -pady 1m
       }
       if { $NoStandardCombo } {
          label .preAllocate.top.msg5 \
                  -text "Non-standard fibre combination selected" \
                  -font $config_font_1
          pack .preAllocate.top.msg5 -padx 5m -pady 2m
          label .preAllocate.top.msg6 \
             -text "(Use Select Fibre Combination in the Options menu)"
          pack .preAllocate.top.msg6 -padx 5m -pady 1m
       }
       pack .preAllocate.top -side top
    
       #  And now the 'Cancel allocation', 'Allocate anyway' buttons.
    
       frame .preAllocate.buttons
       pack .preAllocate.buttons -side bottom -fill both
       button .preAllocate.buttons.cancel \
             -command "set PreAllocFlag 1; destroy .preAllocate" \
                                       -text "Cancel allocation"
       frame .preAllocate.buttons.default -relief sunken -bd 1
       raise .preAllocate.buttons.cancel .preAllocate.buttons.default
       pack .preAllocate.buttons.default -side left -expand 1 -padx 3m -pady 2m
       pack .preAllocate.buttons.cancel -in .preAllocate.buttons.default \
                                       -padx 2m -pady 2m -ipadx 2m -ipady 1m
       button .preAllocate.buttons.allocate \
             -command "set PreAllocFlag 0; destroy .preAllocate" \
                                               -text "Allocate anyway"
       pack .preAllocate.buttons.allocate -side left -expand 1 \
                               -padx 2m -pady 3m -ipadx 2m -ipady 1m
       bind .preAllocate <Return> "set PreAllocFlag 1; destroy .preAllocate"
             
       tkwait window .preAllocate
    }
        
    return $PreAllocFlag
}   

#------------------------------------------------------------------------------ 

#                    I n v a l i d a t e  A l l o c
#
#   A dialogue put up when the user does something - such as changing a
#   guide probe selection - which will invalidate a current allocation.
#   It explains the position and offers the user a 'cancel' or a
#   'continue' option. It returns true (1) if the action should be
#   cancelled, false (0) if it should be allowed to continue. 

proc InvalidateAlloc { } {

    #  Global variables used by this procedure :
    
    global AllocObj         ;# Number of ordinary targets allocated
    global AllocSky         ;# Number of sky targets allocated
    global AllocGui         ;# Number of guide targets allocated
    global config_font_1    ;# Font used in dialogues
    global TickDetails      ;# Details of ticks in the basic sequence dialogue.
    
    #  Global variables set by this procedure :
    
    global InvalAllocFlag;  ;# Set when the dialogue buttons are pressed.
    
    #  If there is no current allocation, nothing to worry about.
    
    if { ($AllocObj + $AllocSky + $AllocGui) == 0 } { return 0 }
              
    catch {destroy .invalAlloc}
    toplevel .invalAlloc
    wm title .invalAlloc "Allocation invalidated"
    wm geom .invalAlloc +100+0 

    frame .invalAlloc.top

    #  Create the window.

    label .invalAlloc.top.msg \
        -text "This will invalidate the current allocation.
              If you continue all allocated fibres will be deallocated."  \
        -font $config_font_1
    pack .invalAlloc.top.msg -padx 5m -pady 2m
    pack .invalAlloc.top -side top

    #  And now the 'Cancel' and 'Continue' buttons.

    frame .invalAlloc.buttons
    pack .invalAlloc.buttons -side bottom -fill both
    button .invalAlloc.buttons.continue \
          -command "set InvalAllocFlag 0; destroy .invalAlloc" \
                                                        -text "Continue"
    frame .invalAlloc.buttons.default -relief sunken -bd 1
    raise .invalAlloc.buttons.continue .invalAlloc.buttons.default
    pack .invalAlloc.buttons.default -side left -expand 1 -padx 3m -pady 2m
    pack .invalAlloc.buttons.continue -in .invalAlloc.buttons.default \
                                    -padx 2m -pady 2m -ipadx 2m -ipady 1m
    button .invalAlloc.buttons.cancel \
          -command "set InvalAllocFlag 1; destroy .invalAlloc" \
                                            -text "Cancel"
    pack .invalAlloc.buttons.cancel -side left -expand 1 \
                            -padx 2m -pady 3m -ipadx 2m -ipady 1m
    bind .invalAlloc <Return> "set InvalAllocFlag 0; destroy .invalAlloc"

    tkwait window .invalAlloc
    
    #  If we are to invalidate all the allocations, clear them out.
    
    if { $InvalAllocFlag == 0 } { 
       RemoveAllocOk
       ClearTicks Allocate
    }
    
    return $InvalAllocFlag
}

#------------------------------------------------------------------------------ 

#                          O k  T o  S a v e
#
#   This procedure is called before a configuration is saved. It checks
#   that enough guide stars have been allocated and, if FLAMES is being
#   used, that a VLT guide star has been allocated. If the configuration
#   does not pass these tests, it puts up a dialogue explaining why the
#   save is being disallowed. In expert mode, it is possible to force the
#   save to continue. This routine returns true (1) if the save is to
#   continue, and zero if it is to be disallowed.

proc OkToSave { } {

    #  Global variables used by this procedure :
    
    global AllocGui           ;# Number of guide targets allocated
    global config_font_1      ;# Font used in dialogues
    global ExpertMode         ;# True if the system is set to 'expert mode'.
    global SelectedGuideObject;# Which guide object is selected ( 0=> none).
    global SelectedGuidePivot ;# Indicates which of the two possible guide pivot
                              ;# positions has been selected.
    global validHourAngle     ;# Validated hour angle range.
    global CheckStatus        ;# Describes current check status
    
    #  Global variables set by this procedure :
    
    global OkToSaveFlag;      ;# Set when the dialogue buttons are pressed.
        
    #  See if the allocation is valid. For FLAMES test to see if a VLT
    #  guide star has been allocated, for all systems test the number of
    #  guide stars. (We ought to use an FPIL routine to get the number
    #  of required guide stars, but this will do for the moment.)
    
    set OkToSaveFlag 1
    
    set NoGuideObject 0
    set NoGuidePivot 0
    set NotEnoughGuides 0
    set NoValidHourAngle 0
    
    UpdateCounts
    
    if { [Instrument] == "FLAMES" } { 
        if { $SelectedGuideObject == 0 } {
            set NoGuideObject 1
        } else {
            if { $SelectedGuidePivot == 0 } {
                set NoGuidePivot 1
            }
        }
    }
    if { $AllocGui < 3 } {
        set NotEnoughGuides 1
    }
    if { $validHourAngle == 0 } {
    	set NoValidHourAngle 1
    }
   
    set Problems [expr $NotEnoughGuides + $NoGuideObject + $NoGuidePivot]
    set Warnings $NoValidHourAngle
    
    if { $Problems || $Warnings } {
     
       catch {destroy .okCheck}
       toplevel .okCheck
       wm title .okCheck "Allocation validity check"
       wm geom .okCheck +100+0 

       frame .okCheck.top

       #  Create the window.
    
       if { $Problems } {
          label .okCheck.top.msg0 \
                  -text "Invalid configurations cannot be saved" \
                  -font $config_font_1
          pack .okCheck.top.msg0 -padx 5m -pady 2m
       } else {
          label .okCheck.top.msg0 \
                  -text "Configuration Warning" \
                  -font $config_font_1
          pack .okCheck.top.msg0 -padx 5m -pady 2m
       }
       if { $NoGuideObject } {
          label .okCheck.top.msg1 \
                  -text "There is no VLT guide star selected" \
                  -font $config_font_1
          pack .okCheck.top.msg1 -padx 5m -pady 2m
       }
       if { $NoGuidePivot } {
          label .okCheck.top.msg3 \
                  -text "No pivot position for VLT guide probe selected" \
                  -font $config_font_1
          pack .okCheck.top.msg3 -padx 5m -pady 2m
       }
       if { $NotEnoughGuides } {
          label .okCheck.top.msg5 \
                  -text "Fewer than 3 reference stars allocated" \
                  -font $config_font_1
          pack .okCheck.top.msg5 -padx 5m -pady 2m
       }
       if { $NoValidHourAngle } {
          if { $ExpertMode == 1 } {
              label .okCheck.top.msg6 \
                  -text "Hour angle range not determined. Any OB using this target setup file will be rejected.\nBe aware that you will receive an angry e-mail from Francesca.\nTo avoid this, use F2 to check hour angle range." \
                  -font $config_font_1
	  } else {
              label .okCheck.top.msg6 \
                  -text "Hour angle range not determined. Any OB using this target setup file will be rejected.\nUse F2 to check hour angle range." \
                  -font $config_font_1
	  }
          pack .okCheck.top.msg6 -padx 5m -pady 2m
       }
       pack .okCheck.top -side top
    
       #  And now the 'Cancel save', 'Save anyway' buttons. The 'Save
       #  anyway option is only provided in expert mode or if only
       #  warnings are reported.
    
       frame .okCheck.buttons
       pack .okCheck.buttons -side bottom -fill both
       button .okCheck.buttons.cancel \
             -command "set OkToSaveFlag 0; destroy .okCheck" \
                                       -text "Cancel save"
       frame .okCheck.buttons.default -relief sunken -bd 1
       raise .okCheck.buttons.cancel .okCheck.buttons.default
       pack .okCheck.buttons.default -side left -expand 1 -padx 3m -pady 2m
       pack .okCheck.buttons.cancel -in .okCheck.buttons.default \
                                       -padx 2m -pady 2m -ipadx 2m -ipady 1m
       if { $ExpertMode == 1 || $Problems == 0 } {
           button .okCheck.buttons.save \
             -command "set OkToSaveFlag 1; destroy .okCheck" \
                                               -text "Save anyway"
           pack .okCheck.buttons.save -side left -expand 1 \
                               -padx 2m -pady 3m -ipadx 2m -ipady 1m
       }
       bind .okCheck <Return> "set OkToSaveFlag 0; destroy .okCheck"
             
       tkwait window .okCheck
    }
        
    return $OkToSaveFlag
    
}

#------------------------------------------------------------------------------ 

#                    M a g n i t u d e   F i l t e r
#
#   Puts up a dialog box which allows the user to specify a low and high
#   magnitude limit to be used to filter input target lists. It puts
#   up a dialogue with two entry boxes for the two magnitude limits, together
#   with a 'cancel' button that deletes the dialog.

proc MagnitudeFilter {} {

    #  Global variables used by this procedure :
    
    global config_font_1         ;# Font used in dialogues
    global MagFiltering          ;# State of magnitude filtering: "On" or "Off".
    global MagFilterText         ;# Describes magnitude filter limits
    global MagErrorText          ;# Describes any dialogue entry error
    
    #  Global variables modified by this procedure (in the sense that they
    #  are associated with checkboxes and radiobuttons and entries in the
    #  dialog, and so can be set by the user interacting with the dialog):
    
    global LowMagEntry           ;# Low magnitude as given in dialogue
    global HighMagEntry          ;# High magnitude as given in dialogue
         
    catch {destroy .magnitudeFilter}
    toplevel .magnitudeFilter
    wm title .magnitudeFilter "Magnitude Filter"
    wm geom .magnitudeFilter +100+0 

    frame .magnitudeFilter.top

    #  Create the window.
    
    message .magnitudeFilter.top.msg -width 120m \
            -text "Magnitude limits to apply to input target lists" \
            -font $config_font_1
    pack .magnitudeFilter.top.msg -side right -fill x -padx 5m -pady 2m
    pack .magnitudeFilter.top -side top
    
    frame .magnitudeFilter.line1
    pack .magnitudeFilter.line1 -side top -padx 5m -pady 1m
    label .magnitudeFilter.line1.txt1 -text "Magnitude filtering is "
    pack .magnitudeFilter.line1.txt1 -side left -pady 2m -padx 2m
    label .magnitudeFilter.line1.txt2 -textvariable MagFiltering
    pack .magnitudeFilter.line1.txt2 -side left -pady 2m

    frame .magnitudeFilter.line2
    pack .magnitudeFilter.line2 -side top -padx 5m -pady 1m
    label .magnitudeFilter.line2.txt1 -textvariable MagFilterText -width 55
    pack .magnitudeFilter.line2.txt1 -side left -pady 1m -padx 2m
    
    frame .magnitudeFilter.limits
    pack .magnitudeFilter.limits -side top -padx 5m -pady 1m

    label .magnitudeFilter.limits.txt1 -text "Magnitude range: "
    pack .magnitudeFilter.limits.txt1 -side left -pady 2m -padx 2m
    entry .magnitudeFilter.limits.ent1 \
        -textvariable LowMagEntry -width 5 -borderwidth 2 -relief sunken 
    pack .magnitudeFilter.limits.ent1 -side left -padx 2m -pady 2m
    label .magnitudeFilter.limits.txt2 -text " to " 
    pack .magnitudeFilter.limits.txt2 -side left -pady 2m -padx 2m
    entry .magnitudeFilter.limits.ent2 \
        -textvariable HighMagEntry -width 5 -borderwidth 2 -relief sunken 
    pack .magnitudeFilter.limits.ent2 -side left -padx 2m -pady 2m
    
    frame .magnitudeFilter.errorline
    pack .magnitudeFilter.errorline -side top -padx 5m -pady 1m
    label .magnitudeFilter.errorline.txt1 -textvariable MagErrorText
    pack .magnitudeFilter.errorline.txt1 -side left -pady 1m -padx 2m
    
    #  And now the 'Apply', 'Clear' and 'Close' buttons at the bottom.
    
    frame .magnitudeFilter.buttons
    pack .magnitudeFilter.buttons -side bottom -fill both
    button .magnitudeFilter.buttons.ok -command "ApplyMagnitudeLimits" \
                                                         -text "Apply" -width 4
    frame .magnitudeFilter.buttons.default -relief sunken -bd 1
    raise .magnitudeFilter.buttons.ok .magnitudeFilter.buttons.default
    pack .magnitudeFilter.buttons.default -side left -expand 1 \
                                                             -padx 3m -pady 2m
    pack .magnitudeFilter.buttons.ok -in .magnitudeFilter.buttons.default \
                                       -padx 2m -pady 2m -ipadx 2m -ipady 1m
    button .magnitudeFilter.buttons.clear \
                  -command "ClearMagnitudeLimits" -text "Clear" -width 4
    pack .magnitudeFilter.buttons.clear -side left -expand 1 \
                               -padx 2m -pady 3m -ipadx 2m -ipady 1m
    button .magnitudeFilter.buttons.cancel \
                  -command "destroy .magnitudeFilter" -text "Close" -width 4
    pack .magnitudeFilter.buttons.cancel -side left -expand 1 \
                               -padx 2m -pady 3m -ipadx 2m -ipady 1m
    bind .magnitudeFilter <Return> \
             ".magnitudeFilter.buttons.ok flash; ApplyMagnitudeLimits"
}

#------------------------------------------------------------------------------ 

#                 C l e a r  M a g n i t u d e  L i m i t s
#
#   This procedure is invoked when the 'Clear' button is pressed in the
#   dialogue set up by MagnitudeFilter{}. It clears the global variables
#   linked to the magnitude entry boxes. This is usually more convenient 
#   than selecting the box text and delting it. Note that this does not
#   apply the new cleared values - the 'Apply' button still needs to be
#   pressed if that is the user's intention.

proc ClearMagnitudeLimits {} {

    #  Global variables set by this routine:
    
    global LowMagEntry           ;# Low magnitude as given in dialogue
    global HighMagEntry          ;# High magnitude as given in dialogue
    global MagErrorText          ;# Describes any dialogue entry error
    
    set LowMagEntry ""
    set HighMagEntry ""
    set MagErrorText ""
}
    
#------------------------------------------------------------------------------ 

#                 A p p l y  M a g n i t u d e  L i m i t s
#
#   This procedure is invoked when the 'Apply' button is pressed in the
#   dialogue set up by MagnitudeFilter{}. It validates the entries, makes
#   sure they are in order, and sets the global variables actually applied
#   to the input lists to the values specified in the dialogue.

proc ApplyMagnitudeLimits {} {

    #  Global variables modified by this routine:
    
    global LowMagEntry           ;# Low magnitude as given in dialogue
    global HighMagEntry          ;# High magnitude as given in dialogue
    global MagErrorText          ;# Describes any dialogue entry error
    
    #  Note that this routine calls SetMagnitudeLimits{} which sets most
    #  of the global variables used by the MagnitudeFilter dialogue and
    #  the input list routines.

    #  Validating entry values like this is a tedious business, but it
    #  does make for a nicer dialogue with the user.  First, what if the
    #  entries had been cleared. We assume having both entries clear
    #  indicates that the user wants to disable magnitude filtering. Having
    #  only one entry clear and something in the other is an error - we
    #  could support having only one limit, eg just a high limit, but we don't.
    
    set Text " "
    if { $LowMagEntry == ""  || $HighMagEntry == "" } {
        if { $LowMagEntry != "" || $HighMagEntry != "" } {
            set Text "You cannot have just one limit blank"
        } else {
            SetMagnitudeLimits 0 0 0
        }
    } else {
        
        #  If we got here, the entries both have something in them. We use
        #  a catch on and expr to see if they are valid numbers, then
        #  see if they are within (admittedly arbitrary) sensible limits.
         
        set HighInvalid [catch { expr $HighMagEntry }]
        if { $HighMagEntry > 50 || $HighMagEntry < -50 } {
            set HighInvalid 1
        }
        set LowInvalid [catch { expr $HighMagEntry }]
        if { $LowMagEntry > 50 || $LowMagEntry < -50 } {
            set LowInvalid 1
        }
        
        #  If we had an error, report it.
        
        if { $HighInvalid || $LowInvalid } {
            if { $HighInvalid && $LowInvalid } {
                set Text "Both magnitude limits are invalid"
            } elseif { $HighInvalid } {
                set Text "High magnitude limit is invalid"
            } else {
                set Text "Low magnitude limit is invalid"
            }
        } else {
        
            #  If we got here, both entries were valid numbers with sensible
            #  values. If they're wrong way round we allow that - it's 
            #  easy for the poor user to get them backwards. And then we
            #  make sure some non-zero range was specified.
             
            if { $HighMagEntry < $LowMagEntry } {
                set TempEntry $LowMagEntry
                set LowMagEntry $HighMagEntry  
                set HighMagEntry $TempEntry
            } elseif { $HighMagEntry == $LowMagEntry } {
                set Text "Both limits cannot be the same"
            } else {
            
               #  And, finally, if we get here, we've got a valid range that
               #  we can use.
               
               SetMagnitudeLimits 1 $LowMagEntry $HighMagEntry
            }  
        }
    }
    
    #  The text set into MagErrorText is displayed in the dialogue
    #  box.
    
    set MagErrorText $Text
}

#------------------------------------------------------------------------------ 

#              S e t  M a g n i t u d e  L i m i t s
#
#   This procedure sets the global text variables displayed in the dialogue
#   put up by MagnitudeFilter{} that describe the current magnitude filtering.
#   It also sets the global variables that are used by the input file routines
#   to control the actual filtering.
#   

proc SetMagnitudeLimits { On LowLimit HighLimit } {

    #  Global variables set by this routine:
    
    global LowMagLimit           ;# Low magnitude limit applied to input lists
    global HighMagLimit          ;# High magnitude limit applied to input lists
    global LowMagEntry           ;# Low magnitude as given in dialogue
    global HighMagEntry          ;# High magnitude as given in dialogue
    global MagFiltering          ;# State of magnitude filtering: "On" or "Off".
    global MagFilterText         ;# Describes magnitude filter limits

    if { $On } {
       set LowMagLimit $LowLimit
       set HighMagLimit $HighLimit
       set LowMagEntry $LowLimit
       set HighMagEntry $HighLimit
       set MagFiltering "ON"
       set MagFilterText \
              "Input target list will be filtered between $LowLimit and $HighLimit"
    } else {
       set LowMagLimit " "
       set HighMagLimit " "
       set LowMagEntry " "
       set HighMagEntry " "
       set MagFiltering "OFF"
       set MagFilterText "Input target list is not being filtered"
    }
    
    #  Tick the Mag Filter button in the basic sequence display
    
    SetTick MagFilter
       
}

#------------------------------------------------------------------------------ 

#                    S e l e c t  F i b r e  C o m b o
#
#   Puts up a dialog box which allows the user to select which fibre
#   combination is to be used. This is normally only used for FLAMES,
#   which has a complex set of possible fibre combinations, but can be
#   used for all instruments. It puts up a dialog with each supported
#   combination shown as a radio button, an OK button and a Close button
#   that deletes the dialog.

proc SelectFibreCombo {} {

    #  Global variables used by this procedure
    
    global FibreCombination   ;# Fibre combination selected for use
    global config_font_1      ;# Font used in dialogues
    global config_font_2      ;# Font (smaller) used in dialogues
    global CurrentFibreCombo  ;# Describes current fibre combination.

    catch {destroy .selectCombo}
    toplevel .selectCombo
    wm title .selectCombo "Select Fibre Combination"
    wm geom .selectCombo +100+0 

    #  See what the current combination is, if any
    
    CurrentFibreCombo
    
    frame .selectCombo.top

    #  Get the list of possible fibre combinations.

    set FibreComboList [FibreCombos]  

    #  Create the window.

    label .selectCombo.top.msg  \
            -text "Select fibre combination to configure..." \
            -font $config_font_1
    pack .selectCombo.top.msg -side right -fill x -padx 5m -pady 2m
    pack .selectCombo.top -side top

    frame .selectCombo.line1
    pack .selectCombo.line1 -side top -padx 5m -pady 1m
    label .selectCombo.line1.txt1 -text "Current combination is " \
                                                    -font $config_font_2
    pack .selectCombo.line1.txt1 -side left -pady 1m -padx 2m

    frame .selectCombo.line2
    pack .selectCombo.line2 -side top -padx 5m
    label .selectCombo.line2.txt1 -textvariable CurrentFibreCombo \
                                                    -font $config_font_1
    pack .selectCombo.line2.txt1 -side left -pady 1m -padx 2m
    
    #  Add a check button for each fibre combination.

    frame .selectCombo.combos
    pack .selectCombo.combos -side top -padx 5m -pady 1m

    set NumFibreCombos [llength $FibreComboList]
    for {set i 0 } { $i < $NumFibreCombos } { incr i } {
       set name [lindex $FibreComboList $i]     
       radiobutton .selectCombo.combos.combo$i -text $name \
          -variable FibreCombination -value $i -width 43 -anchor w \
                                                    -font $config_font_2
       pack .selectCombo.combos.combo$i -padx 2m -pady 2m

    }
    
    #  And now the 'Apply' and 'Close' buttons at the bottom.
    
    frame .selectCombo.buttons
    pack .selectCombo.buttons -side bottom -fill both
    button .selectCombo.buttons.apply -command "ApplySelectFibreCombo" \
                                                      -text "Apply"
    frame .selectCombo.buttons.default -relief sunken -bd 1
    raise .selectCombo.buttons.apply .selectCombo.buttons.default
    pack .selectCombo.buttons.default -side left -expand 1 \
                                                             -padx 3m -pady 2m
    pack .selectCombo.buttons.apply -in .selectCombo.buttons.default \
                                       -padx 2m -pady 2m -ipadx 2m -ipady 1m
    button .selectCombo.buttons.cancel \
                  -command "destroy .selectCombo" -text "Close"
    pack .selectCombo.buttons.cancel -side left -expand 1 \
                               -padx 2m -pady 3m -ipadx 2m -ipady 1m
    bind .selectCombo <Return> \
             ".selectCombo.buttons.apply flash; ApplySelectFibreCombo"

}

#------------------------------------------------------------------------------ 

#                 A p p l y  S e l e c t  F i b r e  C o m b o
#
#  Invoked when the 'Select fibre combination' dialog's 'Apply' button is hit.
#  This sets the appropriate elements of the global array FibreTypeFlags
#  to match the types in the selected combination.
#

proc ApplySelectFibreCombo {} {


    global Cfid               ;# Sds Id for the top level of the current
                              ;# configuration structure.
    global FibreCombination   ;# Fibre combination selected for use
    global FibreComboMode     ;# Mode keyword for Fibre combination selected
    global FibreTypeFlags     ;# Array.  Indicates if each fibre type
                              ;# is enabled. See SelectFibreType for details.

    #  Get list of fibre types and record the current flag settings and
    #  clear out all flags.
    
    set Fibres [FibreTypes]
    set i 0
    foreach fibre $Fibres {
        set FibreTypesWere($i) $FibreTypeFlags($i)
        set FibreTypeFlags($i) 0
        incr i
    }
    
    #  Set the flags for the fibre types in the selected combination
    
    set TypeCodeList [FibreComboTypes $FibreCombination]
    foreach TypeCode $TypeCodeList {
        set FibreTypeFlags($TypeCode) 1
    }
    
    #  See if this has actually changed the settings. If so, we force
    #  any allocated fibres to be deallocated (or allow the user to
    #  cancel the changes, in which case we revert to the previous settings.
    
    set Changed 0
    set i 0
    foreach fibre $Fibres {
        if { $FibreTypesWere($i) != $FibreTypeFlags($i) } { set Changed 1 }
        incr i
    }
    if { $Changed } {
        if { [InvalidateAlloc] == 1 } {
            set i 0
            foreach fibre $Fibres {
                set FibreTypeFlags($i) $FibreTypesWere($i)
                incr i
            }
            set Changed 0
        }
    }
    
    if { $Changed } {
    
        #  Implement the new combination. Redraw the display with the currently
        #  selected fibre combination. Note that in some cases changing the 
        #  combination changes the pointing wavelength for the telescope (eg
        #  if FLAMES is switched into a UVES only mode) so we need to 
        #  recalculate the target positions as well.
    
        if {$Cfid != 0} {
            ConvertXy $Cfid
        }
        set FibreComboMode [FibreComboMode $FibreCombination]
        CurrentFibreCombo
        MarkFibreType FibreTypeFlags
        DrawFibres
        ReformatFibreStats .allocstats.details .availstats.details
    
    }
    
    #  If the combination involves ARGUS, we have to consider the setting
    #  of the ARGUS angle.
    #
    #  PPRS13423: always popup "Argus Settings" panel when Argus is
    #  selected, even if selection not changed now (previously this
    #  code was inside the if { $Changed } condition).
    if { [IsArgusUsed] } {
        ArgusUsed
    } else {
        ArgusUnused
    }
    
    SetTick FibreCombo
  
}

#------------------------------------------------------------------------------ 

#                 C u r r e n t  F i b r e  C o m b o
#
#  Looks at the settings in the global array FibreTypeFlags to see if.
#  they match any of the supported fibre combinations.  It sets
#  the global string CurrentFibreCombo to the description of the
#  combination in question. It also sets the global FibreCombination 
#  (tied to the radio buttons in the select fibre combination
#  dialogue) to the appropriate value, or to -1 if the current selected
#  combination is invalid.
#

proc CurrentFibreCombo {} {

    global FibreCombination   ;# Fibre combination selected for use
    global FibreTypeFlags     ;# Array.  Indicates if each fibre type
                              ;# is enabled. See SelectFibreType for details.
    global CurrentFibreCombo  ;# Described current fibre combination.
                              
    set matchCombo -1
    set Fibres [FibreTypes]
    set FibreComboList [FibreCombos]
    set NumFibreCombos [llength $FibreComboList]
    
    SetFibreCombo -1
    
    #  We look at each combination in turn.
    
    for {set iCombo 0 } { $iCombo < $NumFibreCombos } { incr iCombo } {
        set TypeCodeList [FibreComboTypes $iCombo]
        set Match 1
        
        #  We now look at each fibre type in turn. We set isSelected if this
        #  fibre type is selected in the FibreTypeFlags array, and we
        #  set isInCombo if it is specified in the list of type for this
        #  combination. Match is set to zero as soon as we find a mismatch
        #  between isSelected and isInCombo.
        
        set iType 0;
        foreach fibre $Fibres {
            set isSelected 0
            if { $FibreTypeFlags($iType) == 1 } { set isSelected 1 }
            set isInCombo 0
            foreach TypeCode $TypeCodeList {
                if { $iType == $TypeCode } { set isInCombo 1 }
            }
            if { $isSelected != $isInCombo } {
                set Match 0
                break
            }
            incr iType
        }
        
        #  If we've tried each fibre type and have a match for each one,
        #  then this combination matches. We assume only one combination
        #  will match, and we quit at this point.
        
        if { $Match } {
           set CurrentFibreCombo [lindex $FibreComboList $iCombo]     
           set matchCombo $iCombo
           SetFibreCombo $iCombo
           break;
        }
    }
    
    #  At the end, if we don't have a match, try for two specific combinations
    #  that may be quite common - no fibres selected, all fibres selected.
    
    if { $matchCombo < 0 } {
        set iType 0
        set allSet 1
        set noneSet 1
        foreach fibre $Fibres {
            if { $FibreTypeFlags($iType) == 0 } {
               set allSet 0
            } else {
               set noneSet 0
            }
            incr iType
        }
        if { $allSet } {
            set CurrentFibreCombo "NON_STANDARD (all selected)"
        } elseif { $noneSet } {
            set CurrentFibreCombo "NON_STANDARD (none selected)"
        } else {
            set CurrentFibreCombo "NON_STANDARD fibre combination"
        }
    }
    set FibreCombination $matchCombo
}

#------------------------------------------------------------------------------ 

#                         I s  A r g u s  U s e d
#
#  Looks at the settings in the global array FibreTypeFlags to see if
#  the ARGUS fibres are being used. Returns 1 if it is and 0 if it is not.
#

proc IsArgusUsed {} {

    global ArgusInUse         ;# Argus status text
    global FibreTypeFlags     ;# Array.  Indicates if each fibre type
                              ;# is enabled. See SelectFibreType for details.
                              
    set Fibres [FibreTypes]
    set usesArgus 0
    
    #  We assume that the fibre description of any fibre associated with
    #  ARGUS will include the string "ARGUS" (and that other fibre types
    #  won't include ARGUS in their descriptions!). If we find such a fibre
    #  type flagged as selected, then we assume ARGUS is in use.
    
    set iType 0;
    foreach fibre $Fibres {
        if { [string first "ARGUS" $fibre] >= 0 } { 
            if { $FibreTypeFlags($iType) == 1 } { 
               set usesArgus 1
               break
            }
        }
        incr iType
    }
    if { $usesArgus } {
        set ArgusInUse "in use"
    } else {
        set ArgusInUse "unused"
    }
        
    return $usesArgus
}

#------------------------------------------------------------------------------ 

#                       A r g u s  S e t t i n g s
#
#   Puts up a dialog box which allows the user to specify the various settings
#   for ARGUS.

proc ArgusSettings {} {

    #  Global variables used by this procedure :
    
    global config_font_1        ;# Font used in dialogues
    global config_font_2        ;# Font (smaller) used in dialogues
    global ArgusInUse           ;# Argus status text
    global ArgusErrorText       ;# Error text used in ARGUS settings display
    global ArgusStatus          ;# Summary of ARGUS settings
    global ArgusScale           ;# Argus scale - 1:1 or 1:1.67
    global ArgusAngle           ;# Argus position angle in degrees
    
    #  Global variables modified by this procedure (in the sense that they
    #  are associated with checkboxes and radiobuttons and entries in the
    #  dialog, and so can be set by the user interacting with the dialog):
    
    global ArgusScaleEntry      ;# Argus scale from radiobuttons
    global ArgusAngleEntry      ;# Argus position angle in dialogue entry
    
    
    #  We update the Argus global variables to make sure they match the
    #  values maintained by the C layer.
    
    GetArgusData InUse ArgusAngle ArgusScale Offset
    set ArgusAngle [expr $ArgusAngle - $Offset]
    if { $InUse } { set ArgusInUse "in use" } else { set ArgusInUse "unused" }
    set ArgusStatus "Angle: $ArgusAngle Scale: $ArgusScale"
     
    catch {destroy .argusSettings}
    toplevel .argusSettings
    wm title .argusSettings "Argus Settings"
    wm geom .argusSettings +100+0 

    frame .argusSettings.top
    
    set ArgusScaleEntry $ArgusScale
    set ArgusAngleEntry $ArgusAngle

    #  Create the window.
    
    frame .argusSettings.line1
    pack .argusSettings.line1 -side top -padx 5m -pady 1m
    label .argusSettings.line1.txt1 -text "ARGUS is" -font $config_font_1
    pack .argusSettings.line1.txt1 -side left -pady 2m -padx 2m
    label .argusSettings.line1.txt2 -textvariable ArgusInUse \
                                                     -font $config_font_1
    pack .argusSettings.line1.txt2 -side left -pady 2m

    frame .argusSettings.status
    pack .argusSettings.status -side top -padx 5m -pady 1m
    label .argusSettings.status.txt -textvariable ArgusStatus \
                                       -width 30 -font $config_font_1
    pack .argusSettings.status.txt -side left -pady 2m
    
    frame .argusSettings.angle
    pack .argusSettings.angle -side top -padx 5m -pady 1m

    label .argusSettings.angle.txt1 -text "Set Angle: " -font $config_font_2
    pack .argusSettings.angle.txt1 -side left -pady 2m -padx 2m
    entry .argusSettings.angle.ent1 -font $config_font_2 \
        -textvariable ArgusAngleEntry -width 5 -borderwidth 2 -relief sunken 
    pack .argusSettings.angle.ent1 -side left -padx 2m -pady 2m
    label .argusSettings.angle.txt2 -text " degrees"  -font $config_font_2
    pack .argusSettings.angle.txt2 -side left -pady 2m -padx 2m
        
    frame .argusSettings.scale
    pack .argusSettings.scale -side top -padx 5m -pady 1m
    label .argusSettings.scale.txt1 -text "Set scale: " -font $config_font_2
    pack .argusSettings.scale.txt1 -side left -pady 2m -padx 2m
    radiobutton .argusSettings.scale.one -text "1:1" \
       -variable ArgusScaleEntry -value "1:1" -anchor w -font $config_font_2
    pack .argusSettings.scale.one -side left -padx 2m -pady 2m
    radiobutton .argusSettings.scale.oneto167 -text "1:1.67" \
       -variable ArgusScaleEntry -value "1:1.67" -anchor w -font $config_font_2
    pack .argusSettings.scale.oneto167 -side left -padx 2m -pady 2m
    
    frame .argusSettings.errorline
    pack .argusSettings.errorline -side top -padx 5m -pady 1m
    label .argusSettings.errorline.txt1 -textvariable ArgusErrorText
    pack .argusSettings.errorline.txt1 -side left -pady 1m -padx 2m
    
    #  And now the 'Apply' and 'Close' buttons at the bottom.
    
    frame .argusSettings.buttons
    pack .argusSettings.buttons -side bottom -fill both
    button .argusSettings.buttons.apply -command "ApplyArgusSettings" \
                                                         -text "Apply" -width 4
    frame .argusSettings.buttons.default -relief sunken -bd 1
    raise .argusSettings.buttons.apply .argusSettings.buttons.default
    pack .argusSettings.buttons.default -side left -expand 1 \
                                                             -padx 3m -pady 2m
    pack .argusSettings.buttons.apply -in .argusSettings.buttons.default \
                                       -padx 2m -pady 2m -ipadx 2m -ipady 1m
    button .argusSettings.buttons.close \
                  -command "destroy .argusSettings" -text "Close" -width 4
    pack .argusSettings.buttons.close -side left -expand 1 \
                               -padx 2m -pady 3m -ipadx 2m -ipady 1m
    bind .argusSettings <Return> \
             ".argusSettings.buttons.apply flash; ApplyArgusSettings"
             
    ApplyArgusSettings
}

#------------------------------------------------------------------------------ 

#                    A p p l y  A r g u s  S e t t i n g s
#
#   Invoked when the 'apply' button in the Argus settings dialogue is
#   clicked on.  It validates the entry values for the ARGUS settings, 
#   and, if they are OK, applies them.

proc ApplyArgusSettings {} {

    #  Global variables used by this procedure :
    
    global Cfid                 ;# Sds Id for the top level of the current
                                ;# configuration structure.
    global ArgusScaleEntry      ;# Argus scale from radiobuttons
    global ArgusAngleEntry      ;# Argus position angle in dialogue entry
    global SelectedGuidePivot   ;# Indicates which of the two possible guide
                                ;# pivot positions has been selected.
    global SelectedGuideObject  ;# Which guide object is selected
    global zoom                 ;# Current zoom factor - 1 is normal.

    #  Global variables set by this procedure :
    
    global ArgusErrorText       ;# Error text used in ARGUS settings display   
    global ArgusScale           ;# Argus scale - 1:1 or 1:1.67
    global ArgusStatus          ;# Summary of ARGUS settings
    global ArgusAngle           ;# Argus position angle in degrees

    GetArgusData InUse PreviousAngle PreviousScale Offset
        
    if { $ArgusAngleEntry < 0 || $ArgusAngleEntry > 360 } {
        set ArgusErrorText "Angle must be between 0 and 360"
    } else {
        set ArgusAngle $ArgusAngleEntry
        set ArgusErrorText ""
    }
    set ArgusScale $ArgusScaleEntry
    set ArgusStatus "Angle: $ArgusAngle Scale: $ArgusScale"
    
    #  Apply the position angle.
    
    GetArgusData WasInUse WasAngle WasScale WasOffset
    set Angle [expr $ArgusAngle + $Offset]
    SetArgusData $InUse $Angle $ArgusScale $Offset
    if { $Cfid != 0 } {
       if { [ParamsChanged $Cfid] } {
           if { [InvalidateAlloc] == 1 } {
               #  We need to reset the parameters.
               SetArgusData $WasInUse $WasAngle $WasScale $WasOffset
           } else {
               ConvertXy $Cfid
               ZoomMimic $zoom
               if { $SelectedGuidePivot } {
                  set GuidePivot $SelectedGuidePivot
                  set GuideObject $SelectedGuideObject
                  DeselectGuidePivot
                  SetGuidePivot $GuidePivot $GuideObject
               }
           }
       }
    }
}         

#------------------------------------------------------------------------------ 

#                     A r g u s  U n u s e d
#
#   Invoked whenever the fibre combination is changed to a non-ARGUS
#   combination. If ARGUS had previously been flagged as in use, the
#   argus angle will have been applied to rotate the field. This routine
#   resets things to the default angle.

proc ArgusUnused {} {

    #  Global variables used by this procedure :
    
    global Cfid                 ;# Sds Id for the top level of the current
                                ;# configuration structure.
    global SelectedGuidePivot   ;# Indicates which of the two possible guide
                                ;# pivot positions has been selected.
    global SelectedGuideObject  ;# Which guide object is selected
    global zoom                 ;# Current zoom factor - 1 is normal.

    #  Global variables set by this procedure :
    
    global ArgusInUse           ;# Argus status text
  
    #  Reset the the position angle if things have changed.
    
    GetArgusData WasInUse Angle Scale Offset
    if { $WasInUse } {
       SetArgusData 0 $Angle $Scale $Offset
       set ArgusInUse "unused"
       if {$Cfid != 0} {ConvertXy $Cfid}
       ZoomMimic $zoom
       if { $SelectedGuidePivot } {
          set GuidePivot $SelectedGuidePivot
          set GuideObject $SelectedGuideObject
          DeselectGuidePivot
          SetGuidePivot $GuidePivot $GuideObject
       }
    }
}         

#------------------------------------------------------------------------------ 

#                         A r g u s  U s e d
#
#   Invoked whenever the fibre combination is changed to an ARGUS
#   combination. Sets the ARGUS in use flags, and puts up the ARGUS
#   settings dialogue.

proc ArgusUsed {} {

    #  Global variables set by this procedure :
    
    global ArgusInUse           ;# Argus status text
      
    GetArgusData WasInUse Angle Scale Offset
    SetArgusData 1 $Angle $Scale $Offset
    set ArgusInUse "in use"
    ArgusSettings
}         


#------------------------------------------------------------------------------ 
#
#                             S a v e  A s  S D S
#
#   Procedure called as a result of the "Save As SDS file..." selection from
#   the File menu. Put up a file selection dialog to get the file name
#   and then write the output file

proc SaveAsSDS {} {

    global CurrentOutDirectory ;# Current directory for saving files
    global SaveFile            ;# Name of the file to be saved
    global ConfigFile          ;# Currently displayed configuration file

    #  Make sure the configuration to be saved is valid
    
    if { [OkToSave] == 0 } return 

    #  $SaveFile should be the default name.  But SdsFileName returns
    #  the full file name, we don't want the directory - use the FileName
    #  procedure to get the name component. 
    
    set SaveFile [FileName [SdsFileName $ConfigFile]]

    set types {
        {{SDS files} {*.sds}}
        {{All Files}      *}
    }

    set File [tk_getSaveFile    \
       -filetypes $types \
       -initialdir $CurrentOutDirectory \
       -initialfile $SaveFile \
       -title "Fibre Configuration SDS Output File"]
    if {$File != ""} {
       set CurrentOutDirectory [file dirname $File]
       DoSaveAsSDS $File
    }

}

#------------------------------------------------------------------------------ 
#
#                             S a v e  A s  P A F
#
#   Procedure called as a result of the "Save As PAF File..." selection from
#   the File menu. Put up a file selection dialog to get the file name
#   and then write the output file
#

proc SaveAsPAF {} {

    global Cfid                ;# Sds Id for the top level of the current
                               ;# configuration structure.
    global CurrentOutDirectory ;# Current directory for saving files
    global SaveFile            ;# Name of the file to be saved
    global FibreComboMode      ;# Mode keyword for Fibre combination selected.
    global OutputFileStatus    ;# Description of output file status
    global ListSaved           ;# Flag: current config. list is saved
    global ListTargetFile      ;# List file saved: proposed target setup file name.
    
    #  Make sure the configuration to be saved is valid
    
    if { [OkToSave] == 0 } return 

    #  $SaveFile should be the default name. We use PAFFileName to get this,
    #  set up according to the ESO preferred standard for FLAMES. 
    
    if { $ListSaved } {
    	set SaveFile $ListTargetFile
    } else {
        set SaveFile [PAFFileName $Cfid $FibreComboMode]
    }

    set types {
        {{PAF files} {*.ins}}
        {{All Files}      *}
    }

    set File [tk_getSaveFile    \
       -filetypes $types \
       -initialdir $CurrentOutDirectory \
       -initialfile $SaveFile \
       -title "Target Setup Output File"]
    if {$File != ""} {
       set CurrentOutDirectory [file dirname $File]
       DoSaveAsPAF $File
    }

}

#------------------------------------------------------------------------------ 
#
#                        R e n a m e  F i e l d
#
#  Invoked by the "Check Field Label" option menu item, to change the label
#  associated with a field. Simply changes the value of the "label" item in
#  the fieldData of the SDS structure.

proc RenameField {} {

    global FieldName         ;# The label associated with the current field
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global Edited            ;# True once the configuration has been changed
    global OutputFileStatus  ;# Description of output file status
    global ListSaved         ;# Flag: current config. list is saved
    
    set ex [info exists FieldName]
    if {$ex == 0} {return}
    set button [prompt_dialog .dialog "Change Field Label" \
                       "Enter New Field Name:" FieldName 0 "OK"] 
    set fid [SdsFind $Cfid fieldData]
    ArgPutString $fid label $FieldName
    set Edited 1
    set ListSaved 0
    set OutputFileStatus "Not saved"
}


#------------------------------------------------------------------------------ 
#
#                        C h a n g e  O b s  D a t e
#
#  Invoked by the "Change Observation Date" option menu item, to change
#  the observation date of a file. It puts up a dialogue that allows a new
#  date to be specified, and calls ChangeObsDateOK{} when the OK button is
#  selected.

proc ChangeObsDate { } {
    
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global config_font_1     ;# Font used in dialogues
    global ChObsYear         ;# Tied to the year in the change obs date dialog
    global ChObsMonth        ;# Tied to the month in the change obs date dialog
    global ChObsDay          ;# Tied to the day in the change obs date dialog
    
    #  Get the date for the current field.
    
    set fid [SdsFind $Cfid fieldData]
    set mjd [ArgGet $fid configMjd]
    SdsFreeId $fid
    set UTDATE [Mjd2date $mjd]
    set ChObsYear [string range $UTDATE 0 3]
    set ChObsMonth [string range $UTDATE 5 6]
    set ChObsDay [string range $UTDATE 8 9]

    # Create the prompt window.
    
    set w .obsDate
    catch "destroy $w"
    toplevel $w
    wm title $w "Checking Allocation"
      
    frame $w.top

    message $w.top.msg -width 90m \
              -text "Change Observation Date to..." \
              -font $config_font_1
    pack $w.top.msg -side right -expand 1 -fill x -padx 5m -pady 5m
    pack $w.top -side top
      
    frame $w.date
    pack $w.date -side top -padx 5m -pady 3m
    label $w.date.lab1 -text "Year:" -width 6
    pack $w.date.lab1 -side left
    entry $w.date.ftext1 -textvariable ChObsYear -width 5 \
        -relief sunken -borderwidth 2
    DtcluBindEntryInt $w.date.ftext1 1950 2050
    pack $w.date.ftext1 -side left
    label $w.date.lab2 -text "Month:" -width 6
    pack $w.date.lab2 -side left
    entry $w.date.ftext2 -textvariable ChObsMonth -width 4 \
        -relief sunken -borderwidth 2
    DtcluBindEntryInt $w.date.ftext2 1 12
    pack $w.date.ftext2 -side left
    label $w.date.lab3 -text "Day:" -width 5
    pack $w.date.lab3 -side left
    entry $w.date.ftext3 -textvariable ChObsDay -width 4 \
          -relief sunken -borderwidth 2
    DtcluBindEntryInt $w.date.ftext3 1 31
    pack $w.date.ftext3 -side left
      
    DrawButtons $w ChangeObsDateOK

    Centre $w
       
}

#------------------------------------------------------------------------------ 
#
#                    C h a n g e  O b s  D a t e  O K
#
#  Invoked when the OK button is selected in the dialogue put up by
#  ChangeObsDate{}.

proc ChangeObsDateOK { } {

    global Cfid           ;# Sds Id for the top level of the current
                          ;# configuration structure.
    global ChObsYear      ;# Tied to the year in the change obs date dialog
    global ChObsMonth     ;# Tied to the month in the change obs date dialog
    global ChObsDay       ;# Tied to the day in the change obs date dialog
    global errorCode      ;# Error description (generated by Tcl)
    global errorInfo      ;# Stack trace describing an error (generated by Tcl)
    global zoom           ;# Current zoom factor - 1 is normal.

    #  Create a copy of the field to work on (in case of errors)

    set fieldCopy [SdsCopy $Cfid]
    
    #  Change the date.  If we have an error, delete the structure
    #
    #  Likewise, convert the X/Y values, again catching any error.

    if [catch "SetUT $fieldCopy $ChObsYear $ChObsMonth $ChObsDay" msg] {
       set savedInfo $errorInfo
       set savedCode $errorCode
       ArgDelete $fieldCopy
       error $msg $savedInfo $savedCode
    } elseif [catch "ConvertXy $fieldCopy" msg] {
       set savedInfo $errorInfo
       set savedCode $errorCode
       ArgDelete $fieldCopy
       error $msg $savedInfo $savedCode
    } else {
    
        # We should now be safe to use the new file.  Delete the old
        # field and reference the new
        
        ArgDelete $Cfid
        set Cfid $fieldCopy
        
        # Update all the details and the screen.
        
        OpenSDSOk
        
        ZoomMimic $zoom
        
        # Check the allocation, if any.
        
        WorkingDialog .work "Checking Allocation..."
        set status [DoCheck $Cfid]
        destroy .work
        if {$status == "OK"} {
            MsgOut "Allocation OK after change of date"
        } else {
            ErsOut "Please reallocate to fix configuration problems \
                                                          for the new date"
        }
    }
}

#------------------------------------------------------------------------------ 
#
#                         A p p l y  U T  D a t e
#
#   Given a UT date specification in a string as "yyyy mm dd[.dd]", applies it
#   to the current configuration.

proc ApplyUTDate { UtDateSpec } {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global UTDATE            ;# Tied to UT date field in main control panel
    global UTTIME            ;# Tied to UT time field in main control panel

    #  We have to split up the spec into its three components, since they
    #  have to be passed separately to SetUT.
    
    set UtYear [lindex $UtDateSpec 0]
    set UtMonth [lindex $UtDateSpec 1]
    set UtDay [lindex $UtDateSpec 2]
    
    #  SetUT modifies the structure so that it has the specified UT date/time
    
    SetUT $Cfid $UtYear $UtMonth $UtDay
    
    #  Update the global variables that display the UT date and time
    
    set fid [SdsFind $Cfid fieldData]
    set mjd [ArgGet $fid configMjd]
    SdsFreeId $fid
    set UTDATE [Mjd2date $mjd]
    set UTTIME [Mjd2time $mjd]
    
    #  And ConvertXy modifies all the X,Y positions in the structure
    #  for the new date/time.
    
    ConvertXy $Cfid
    
    #  Note that this doesn't validate the modified structure. Maybe it
    #  should?
    
}


#------------------------------------------------------------------------------ 
#
#                         Do  S a v e  A s  S D S
#
#   Procedure called from SaveAsSDS{} to do the actual writing of the 
#   output file.

proc DoSaveAsSDS {file} {

    global ConfigFile        ;# Currently displayed configuration file
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global Edited            ;# True once the configuration has been changed
    global OutputFileStatus  ;# Description of output file status

    CompressUnalloc $Cfid
    MsgOut "Writing file $file"
    SdsWrite $Cfid $file
    AddIndex $Cfid
    set ConfigFile $file
    SetTick OpenFile
    wm title .mimic $ConfigFile
    set Edited 0
    set OutputFileStatus "SDS file: $file"
    SetTick SaveFile

}

#------------------------------------------------------------------------------ 
#
#                         Do  S a v e  A s  P A F
#
#   Procedure called from SaveAsPAF{} to do the actual writing of the 
#   output file.

proc DoSaveAsPAF {file} {

    global ConfigFile          ;# Currently displayed configuration file
    global Cfid                ;# Sds Id for the top level of the current
                               ;# configuration structure.
    global Edited              ;# True once the configuration has been changed
    global ExpertMode          ;# True if the system is set to 'expert mode'.
    global FibreComboMode      ;# Mode keyword for Fibre combination selected.
    global OutputFileStatus    ;# Description of output file status
    global SelectedGuideObject ;# Which guide object is selected ( 0=> none).
    global SelectedGuideOrient ;# Guide probe orientation - "POS" or "NEG".
    global SelectedGuideName   ;# Name of selected guide star.
    global WAVELENGTH          ;# Observing wavelength in Angstroms.
    global validHourAngle      ;# Validated hour angle range.
    global CheckStatus         ;# Describes current check status

    CompressUnalloc $Cfid
    MsgOut "Writing file $file"
    
    #  WritePAF - a C routine - does all the actual work. It needs a number
    #  of things known at this Tcl layer passed to it, such as guide probe
    #  details and the observing wavelength - actually, the wavelength is
    #  maintained both by the Tcl layer and the C layer. We need to make
    #  sure these are in step. It also needs the fibre combination keyword.
    
    if { $WAVELENGTH != [GetWavelength] } {
        puts "Warning - Tcl and C versions of wavelength differ"
        puts "$WAVELENGTH (Tcl) != [GetWavelength] (C)"
    }
    
    #  Note that it isn't enough to pass the value of SelectedGuideObject,
    #  because generally an allocation will have been performed between the
    #  setting of its value and the invocation of this routine. If that has
    #  happened, its value will be out of date, as the layout of the
    #  unallocated Guide section of the overall structure will have changed.
    #  (It is just an index into that section, and if targets get allocated
    #  the index values change.) So we pass the guide star name as well.    

    WritePAF $Cfid $file $SelectedGuideObject $SelectedGuideName \
                          $SelectedGuideOrient $FibreComboMode $ExpertMode \
			  $validHourAngle
    AddIndex $Cfid
    set Edited 0
    set OutputFileStatus [FileName $file]
    SetTick SaveFile

}

#------------------------------------------------------------------------------ 
#
#                   S a v e  A l l o c  T o  U n a l l o c
#
#   Procedure called as a result of the "Save Alloc to UnAlloc..." selection 
#   from the File menu. Put up a file selection dialog to get the file name
#   and then write the output file
#

proc SaveAllocToUnalloc {} {

    global SaveFile        ;# Name of the file to be saved
    global ConfigFile      ;# Currently displayed configuration file (or plate)
    global CurrentOutDirectory;# Current directory for saving files

    # $SaveFile should be the default name.  But SdsFileName returns
    # the full file name, we don't want the directory - use the FileName
    # procedure to get the name component. 
    
   set SaveFile [FileName [SdsFileName $ConfigFile]]

   set types {
        {{SDS files} {*.sds}}
        {{All Files}      *}
   }

   set File [tk_getSaveFile    \
       -filetypes $types \
       -initialdir $CurrentOutDirectory \
       -initialfile $SaveFile \
       -title "Allocated to Unallocated configuration file"]
   if {$File != ""} {
       set CurrentOutDirectory [file dirname $File]
       DoSaveAllocToUnalloc $File
   }

}

#------------------------------------------------------------------------------ 
#
#                   D o  S a v e  A l l o c  T o  U n a l l o c
#
#  Invoked when a file is selected in the dialog put up by SaveAllocToUnalloc{}.

proc DoSaveAllocToUnalloc {file} {

    global Cfid           ;# Sds Id for the top level of the current
                          ;# configuration structure.
    global errorCode      ;# Error description (generated by Tcl)
    global errorInfo      ;# Stack trace describing an error (generated by Tcl)

    #  Copy the current field

    set fieldCopy [SdsCopy $Cfid]

    #  Copy the allocated items to the unallocated part of the structure.
    #  Ensure we catch any error so we can tidy up the copy.

    if [catch "AllocToUnalloc $fieldCopy" msg] {
        set savedInfo $errorInfo
        set savedCode $errorCode
        ArgDelete $fieldCopy
        error $msg $savedInfo $savedCode
    } else {

        #  Write the structure out.  Ensure we catch any error so that
        #  we can tidy up.

        MsgOut "Writing Allocated objects to unallocated fields of $file"
        if [catch "SdsWrite $fieldCopy $file" msg] {
            set savedInfo $errorInfo
            set savedCode $errorCode
            ArgDelete $fieldCopy
            error $msg $savedInfo $savedCode
        } else {
            ArgDelete $fieldCopy
        }
    }
  
}

#------------------------------------------------------------------------------ 
#
#                          S a v e  F i l e
#
#   Procedure called as a result of the "Save" selection from the
#   File menu. It puts up a dialogue for either a PAF or an SDS file
#   name (depending on the instrument in use) and then invokes the 
#   appropriate output routine.

proc SaveFile {} {

    global SaveFile          ;# Name of the file to be saved
    global ConfigFile        ;# Currently displayed configuration file
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global FibreComboMode    ;# Mode keyword for Fibre combination selected.
    global OutputFileStatus  ;# Description of output file status

    #  Make sure the configuration to be saved is valid
    
    if { [OkToSave] == 0 } return 

    set UsingPAF 0
    if { [Instrument] == "FLAMES" } { set UsingPAF 1 }
    
    if { $UsingPAF } {
        set SaveFile [PAFFileName $Cfid $FibreComboMode]
    } else {
        set SaveFile [SdsFileName $ConfigFile]
    }

    MsgOut "Writing file $SaveFile"
    
    if { $UsingPAF } {
        DoSaveAsPAF $SaveFile
    } else {
        CompressUnalloc $Cfid
        SdsWrite $Cfid $SaveFile
        AddIndex $Cfid
    }
    set Edited 0
    set OutputFileStatus [FileName $SaveFile]
    SetTick SaveFile

}

#------------------------------------------------------------------------------ 
#
#                          I m p o r t  D a t a
#
#   Procedure tied to the "Import Allocations..." item in the commands
#   menu. It puts up a dialogue for a file containing a list of allocations,
#   and then invokes the routine ImportDataOk{}.
 
proc ImportData {} {

    #  Global variables used by this routine:
   
    global CurrentInDirectory ;# Current directory for reading files
   
    set l1 [list "Import Files" "*.imp"]
    set l2 [list "All Files" *]
    set types [list $l1 $l2]
    
    set File [tk_getOpenFile    \
        -filetypes $types \
        -initialdir $CurrentInDirectory \
        -title "Allocation Table to Import"]
    if {$File != ""} {
       set CurrentInDirectory [file dirname $File]
       ImportDataOk $File
    }
}

#------------------------------------------------------------------------------ 
#
#                       I m p o r t  D a t a  O k
#
#
#   Called when the file dialog put up by ImportData{} opens the file. It
#   operates by running the allocation algorithm with the target selection
#   parameter set to "import ascii allocation table". This is one of
#   the routines that makes use of knowledge about the parameters of the
#   allocation. Ideally there should be a proper interface to the allocation
#   algorithms that will allow this to be done more elegantly.

proc ImportDataOk {file} {

    global ImportFile        ;# Name of import file if such a file is used,
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global OutputFileStatus  ;# Description of output file status
    global ListSaved         ;# Flag: current config. list is saved

   
    set ImportFile $file

    SetImportFile $file
    toplevel .remove
    RemoveAllocOk

    # Target Selection = "import ascii allocation table"
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    set p5_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 7]"

    # Set up field.
    
    WorkingDialog .work "Importing Data...."
    set Edited 1
    set ListSaved 0
    set OutputFileStatus "Not saved"
    CompressUnalloc $Cfid
    AddIndex $Cfid
    ConvertXy $Cfid

    # Do allocation

    set start [clock seconds]
    DoAllocation $Cfid

    # Finish.
    
    CompressUnalloc $Cfid
    AddIndex $Cfid
    MsgOut "Allocation completed in [expr [clock seconds]-$start] seconds."
    destroy .work
    SetTick Allocate

    # Have to put parameter back
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p5_value

    # Check the allocation
    
    CheckAlloc
}

#------------------------------------------------------------------------------ 
#
#                  I m p o r t  D a t a  O n l y  O k
#
#   This imports an allocation file without doing any checks.  It is only
#   currently used by the "-i" program option. It operates by running the 
#   allocation algorithm with the target selection parameter set to 
#   "import with no checks". This is one of the routines that makes use of
#   knowledge about the parameters of the allocation. Ideally there should 
#   be a proper interface to the allocation algorithms that will allow this
#   to be done more elegantly.

proc ImportDataOnlyOk {file} {

    global ImportFile        ;# Name of import file if such a file is used,
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global OutputFileStatus  ;# Description of output file status
    global ListSaved         ;# Flag: current config. list is saved

    set ImportFile $file

    SetImportFile $file
    toplevel .remove
    RemoveAllocOk

    # Target Selection = "import with no checks"
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    set p5_value [GetParameter "$name"]
    SetParameter "$name" "[lindex $values 8]"

    WorkingDialog .work "Importing Data...."
    set Edited 1
    set ListSaved 0
    set OutputFileStatus "Not saved"
    CompressUnalloc $Cfid
    AddIndex $Cfid
    ConvertXy $Cfid

    # Do the allocation
    
    set start [clock seconds]
    DoAllocation $Cfid
    CompressUnalloc $Cfid
    AddIndex $Cfid
    MsgOut "Allocation completed in [expr [clock seconds]-$start] seconds."
    destroy .work
    SetTick Allocate

    # Have to put parameter back
    
    set name_val "p5_val"
    ParameterDetails 5 name type low high nvalues values
    global $name_val
    SetParameter "$name" $p5_value

}

#------------------------------------------------------------------------------ 
#
#                       O p e n  A s c i i
#
#   Procedure called as a result of the "Open..." selection from the
#   File menu. Puts up a file selection dialog to get the file name
#   and then call OpenAsciiOk to read it in it when the OK button is
#   selected.

proc OpenAscii {} {

    global ConfigFile         ;# Currently displayed configuration file
    global CurrentInDirectory ;# Current directory for reading files
   
    # First check if we have to save the existing file.  If this returns
    # false, the user cancelled the operation.
   
    if { [ SaveCheck ] == 0 } {
       return
    }

    set types {
        {{Ascii files} {*.fld}}
        {{All Files}      *}
    }

    # $ConfigFile should be the default name

    set File [tk_getOpenFile    \
       -filetypes $types \
       -initialdir $CurrentInDirectory \
       -title "File with Input Target List"]
    if {$File != ""} {
       set CurrentInDirectory [file dirname $File]
       OpenAsciiOk $File
    }

}

#------------------------------------------------------------------------------ 
#
#                       O p e n  A s c i i  O k
#
#   Called when the file dialog put up by OpenAscii{} opens the file. All the
#   heavy work is done by OpenAsciiFile{} and then all this has to do is
#   display the new field.

proc OpenAsciiOk {file} {

    global SelectedFibre     ;# Currently selected fibre (pivot) number -
                             ;# starts from 1 - zero => no fibre selected.
    global SelectedObject
    global ConfigFile        ;# Currently displayed configuration file

    #  Open the file.
    
    OpenAsciiFile $file

    #  Display the field

    wm title .mimic $ConfigFile
    ZoomMimic 1
    catch {set SelectedFibre 0}
    catch {unset Selectedobject}
    DeselectGuidePivot
   
    return

}

#------------------------------------------------------------------------------ 
#
#                       O p e n  A s c i i  F i l e
#
#   We have a field file in ascii format.  Open it.  Most of the work is
#   done by the ConvertAscii{} routine, which is implemented in C in
#   configure.c.

proc OpenAsciiFile { file } {

    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure.
    global ConfigFile      ;# Currently displayed configuration file (or plate)
    global Edited          ;# True once the configuration has been changed
    global errorCode       ;# Error description (generated by Tcl)
    global errorInfo       ;# Stack trace describing an error (generated by Tcl)
    global HighMagLimit    ;# High magnitude limit applied to input lists
    global LowMagLimit     ;# Low magnitude limit applied to input lists
    global MagFiltering    ;# State of magnitude filtering: "On" or "Off".
    global OutputFileStatus;# Description of output file status
    global ListSaved       ;# Flag: current config. list is saved
    
    #  Do an initial pass through the file to count the number of objects
    #  we will get. If this is excessive, we may want to modify the 
    #  magnitude filter.
    
    if { $MagFiltering == "ON" } {
        set count [CountAsciiObjects $file $LowMagLimit $HighMagLimit]
    } else {
        set count [CountAsciiObjects $file]
    }
    if { $count > 1500 } {
        if { [ObjectCountDialogue $count] == 0 } return
    }
        
    #  Convert the file from Ascii to Sds.  Note that we use a three argument
    #  form of ConvertAscii if magnitude filtering is being used.

    if { $MagFiltering == "ON" } {
        set temp [ConvertAscii $file fracDayFlag $LowMagLimit $HighMagLimit]
    } else {
        set temp [ConvertAscii $file fracDayFlag]
    }
    if {$temp != 0} {        

        if { $fracDayFlag == 0 } { SetUTMeridian $temp }
        MakeAllocated $temp [PivotsId]
        ConvertXy $temp

        # AddIndex may fail here if we have duplicated objects.  If that
        # happends, we need to restore the old file, if any, and delete
        # the new one.
        
        if [catch "AddIndex $temp" msg] {
            set savedInfo $errorInfo
            set savedCode $errorCode
            ArgDelete $temp
            error $msg $savedInfo $savedCode
        } else {
        
            # Looks ok. Free up the old configuration.
            
            if { $Cfid != 0 } {
                ArgDelete $Cfid
            }
            
            # Setup the global variable.
            
            set ConfigFile $file
            set Cfid $temp
            SetTick OpenFile

            # This file is considered edited. 
             
            set Edited 1
    	    set ListSaved 0
            set OutputFileStatus "Not saved"
            OpenSDSOk
        }
    } else {
        ErsFlush
    }

   #  Update the statistics showing target allocation by fibre type
   
   UpdateFibreStats $Cfid
   
   ReformatFibreStats .allocstats.details .availstats.details
   
}

#------------------------------------------------------------------------------ 

#                    O b j e c t  C o u n t  D i a l o g u e
#
#   A dialogue put up when the user attempts to read in an ascii file with
#   magnitude filter settings that seem to be letting in an excessive 
#   number of objects. 

proc ObjectCountDialogue { Count } {

    #  Global variables used by this procedure :
    
    global config_font_1    ;# Font used in dialogues
    global MagFilterText    ;# Describes magnitude filter limits. 
    
    #  Global variables set by this procedure :
    
    global ObjectCountFlag; ;# Set when the dialogue buttons are pressed.
                  
    catch {destroy .objectCount}
    toplevel .objectCount
    wm title .objectCount "Object count"
    wm geom .objectCount +100+0 
    
    #  Display details of problem
    
    frame .objectCount.filter
    label .objectCount.filter.msg \
           -text $MagFilterText \
           -font $config_font_1
    pack .objectCount.filter.msg -padx 5m -pady 2m
    pack .objectCount.filter -side top
    

    frame .objectCount.count
    set CountString "At this setting there will be $Count objects.
                     You may want to modify the magnitude filter."
    label .objectCount.count.msg \
           -text $CountString \
           -font $config_font_1 -anchor w
    pack .objectCount.count.msg -padx 5m -pady 2m
    pack .objectCount.count -side top
    
    #  And now the 'Cancel' and 'Continue' buttons.

    frame .objectCount.buttons
    pack .objectCount.buttons -side bottom -fill both
    button .objectCount.buttons.cancel \
          -command "set ObjectCountFlag 0; destroy .objectCount" \
                                                        -text "Cancel"
    frame .objectCount.buttons.default -relief sunken -bd 1
    raise .objectCount.buttons.cancel .objectCount.buttons.default
    pack .objectCount.buttons.default -side left -expand 1 -padx 3m -pady 2m
    pack .objectCount.buttons.cancel -in .objectCount.buttons.default \
                                    -padx 2m -pady 2m -ipadx 2m -ipady 1m
    button .objectCount.buttons.continue \
          -command "set ObjectCountFlag 1; destroy .objectCount" \
                                            -text "Continue"
    pack .objectCount.buttons.continue -side left -expand 1 \
                            -padx 2m -pady 3m -ipadx 2m -ipady 1m
    bind .objectCount <Return> "set ObjectCountFlag 0; destroy .objectCount"

    tkwait window .objectCount
        
    return $ObjectCountFlag
}

#------------------------------------------------------------------------------ 
#
#                       O p e n  S D S  O k
#
#   This is a utility routine used in a number of places in the code when
#   the SDS structure at the center of the configuration program has been
#   modified. Its role is to set the various global variables that are tied
#   to the control panel display, so that the panel is up to date. (The
#   name of the routine does not match the usual pattern - you might expect
#   it to be the routine called when the OK button put up by OpenSDS{} is
#   clicked, but in fact that's DoOpenSDS{}.)
# 

proc OpenSDSOk {} {

    global ConfigFile        ;# Currently displayed configuration file
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global CRA               ;# Tied to central RA field in main control panel
    global CDEC              ;# Tied to central Dec field in main control panel
    global CHA               ;# Tied to Hour angle field in main control panel
    global CZD               ;# Tied to ZD field in main control panel
    global UTDATE            ;# Tied to UT date field in main control panel
    global UTTIME            ;# Tied to UT time field in main control panel
    global WAVELENGTH        ;# Observing wavelength in Angstroms.
    global InputFile         ;# The name of the file containing target data
    global FieldName         ;# The label associated with the current field
    global AllocObj          ;# Number of ordinary targets allocated
    global UnallocObj        ;# Number of ordinary targets unallocated
    global AllocSky          ;# Number of sky targets allocated
    global UnallocSky        ;# Number of sky targets unallocated
    global AllocGui          ;# Number of guide targets allocated
    global UnallocGui        ;# Number of guide targets unallocated

    set Display $ConfigFile

    set fid [SdsFind $Cfid fieldData]

    set CRA [Ra2string [ArgGet $fid cenRa]]
    set CDEC [Dec2string [ArgGet $fid cenDec]]
    set mjd [ArgGet $fid configMjd]
    set ra [ArgGet $fid appRa]
    set dec [ArgGet $fid appDec]
    set CHA [RaDec2Ha $ra $dec]
    set CZD [RaDec2Zd $ra $dec]
    set UTDATE [Mjd2date $mjd]
    set UTTIME [Mjd2time $mjd]
    set WAVELENGTH [GetWavelength]
    # set InputFile $ConfigFile
    set InputFile [file tail $ConfigFile]
    set FieldName [ArgGet $fid label]
    MakeSkyvals $Cfid
    set stat [catch {ArgGet $fid allocObj} AllocObj]
    if {$stat != 0} {set AllocObj 0}
    set stat [catch {ArgGet $fid unallocObj} UnallocObj]
    if {$stat != 0} {set UnallocObj 0}
    set stat [catch {ArgGet $fid allocSky} AllocSky]
    if {$stat != 0} {set AllocSky 0}
    set stat [catch {ArgGet $fid unallocSky} UnallocSky]
    if {$stat != 0} {set UnallocSky 0}
    set stat [catch {ArgGet $fid allocGui} AllocGui]
    if {$stat != 0} {set AllocGui 0}
    set stat [catch {ArgGet $fid unallocGui} UnallocGui]
    if {$stat != 0} {set UnallocGui 0}

    SdsFreeId $fid
    
    #  Make sure the menus reflect the open status of the file
    
    MenuUpdate
    
    UpdateCounts
}

#------------------------------------------------------------------------------ 
#
#                       U p d a t e  C o u n t s
#
#   This is a utility routine used in a number of places in the code when
#   the allocations have changed. It updates the global variables tied
#   to the allocation fields in the main control panel.

proc UpdateCounts {} {

    global AllocationStatus  ;# Describes current allocation status.
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global AllocObj          ;# Number of ordinary targets allocated
    global UnallocObj        ;# Number of ordinary targets unallocated
    global AllocSky          ;# Number of sky targets allocated
    global UnallocSky        ;# Number of sky targets unallocated
    global AllocGui          ;# Number of guide targets allocated
    global UnallocGui        ;# Number of guide targets unallocated
    
    set fid [SdsFind $Cfid fieldData]
    set stat [catch {ArgGet $fid allocObj} AllocObj]
    if {$stat != 0} {set AllocObj 0}
    set stat [catch {ArgGet $fid unallocObj} UnallocObj]
    if {$stat != 0} {set UnallocObj 0}
    set stat [catch {ArgGet $fid allocSky} AllocSky]
    if {$stat != 0} {set AllocSky 0}
    set stat [catch {ArgGet $fid unallocSky} UnallocSky]
    if {$stat != 0} {set UnallocSky 0}
    set stat [catch {ArgGet $fid allocGui} AllocGui]
    if {$stat != 0} {set AllocGui 0}
    set stat [catch {ArgGet $fid unallocGui} UnallocGui]
    if {$stat != 0} {set UnallocGui 0}
    set TotalAllocated [expr $AllocObj + $AllocGui + $AllocSky]
    if { $TotalAllocated <= 0 } {
       set AllocationStatus "No fibres allocated"
    } else {
       set AllocationStatus "$TotalAllocated fibres allocated"
    }
    SdsFreeId $fid

    UpdateFibreCounts
}

#------------------------------------------------------------------------------ 
#
#                  U p d a t e  F i b r e  C o u n t s
#
#   This updates the global variables tied to the fibre type allocation
#   statistics displayed in the fibre allocation statistics panel. The C
#   layer maintains the allocation statistics, and this routine is used
#   to get the FibStats global Tcl variable tied to the statistics panel
#   into sync with the information in the C layer.

proc UpdateFibreCounts {} {

    #  Global variables set by this routine:

    global FibStats           ;# Array holding fibre allocation statistics
    
    set FibreTypeList [FibreTypes]
    set NumFibreTypes [llength $FibreTypeList]

    set TotalFibres 0
    set TotalSkies 0
    set TotalObjects 0
    set TotalUnalloc 0
    
    #  Most of the real work is done by the C-implemented GetFibreStats{}.
    #  The rest of the code in this routine is just to transfer the data
    #  it returns into the FibStats array. Note that some fibre types as
    #  classified for these purposes as the same as another fibre type.
    #  We used to distinguish between objects and guide in the displayed
    #  counts, but to save space we now just count them together - objects
    #  for guide fibres are always guide targets and objects for non-guide
    #  fibres are always non-guide targets, so there's no loss in information.
    
    for { set Index 0 } { $Index < $NumFibreTypes } { incr Index } {
        GetFibreStats $Index Name Enabled Fibres Skies Guides \
                          Objects Unalloc SameAs MayShare Targets SkyTargets
        incr Objects $Guides
        if { $SameAs < 0 } {
            set FibStats($Index,enabled) $Enabled
            set FibStats($Index,name) $Name
            set FibStats($Index,fibres) $Fibres
            set FibStats($Index,skies) $Skies
            set FibStats($Index,objects) $Objects
            set FibStats($Index,unalloc) $Unalloc
            set FibStats($Index,targets) $Targets
            set FibStats($Index,skyTargets) $SkyTargets
        } else {
        
            #  This is classified as the same as a lower indexed fibre type,
            #  so we treat this type as not-enabled and increment the counts
            #  for the lower-numbered type instead. We don't increment the
            #  target counts, since this just double-counts the targets.
            
            set FibStats($Index,enabled) 0
            incr FibStats($SameAs,fibres) $Fibres
            incr FibStats($SameAs,skies) $Skies
            incr FibStats($SameAs,objects) $Objects
            incr FibStats($SameAs,unalloc) $Unalloc
        }
        
        #  We work out the total statistics for enabled fibre types for
        #  ourselves from the data returned by GetFibreStats{} for each
        #  fibre type. Note that total target statistics are meaningless
        #  because some targets are compatible with more than one fibre
        #  type and so appear many times in the counts.
        
        if { $Enabled } {
           incr TotalFibres $Fibres
           incr TotalSkies $Skies
           incr TotalObjects $Objects
           incr TotalUnalloc $Unalloc
        }
    }
    set FibStats(Total,name) " "
    set FibStats(Total,fibres) $TotalFibres
    set FibStats(Total,skies) $TotalSkies
    set FibStats(Total,objects) $TotalObjects
    set FibStats(Total,unalloc) $TotalUnalloc
}        
        
#------------------------------------------------------------------------------ 
#
#                     F o r m a t  F i b r e  S t a t s  
#
#   This procedure creates the fibre allocation statistics panel. This 
#   procedure is expected to be called only once, when the main control
#   panel is set up. If the set of enabled fibres changes, ReformatFibreStats{}
#   can be called to change which fibre type data is actually displayed.
#   The two arguments give the names of the frames used for the allocation
#   and available target statistics respectively.

proc FormatFibreStats { Window Window2 } {

    #  Global variables used by this routine:

    global FibStats           ;# Array holding fibre allocation statistics
    
    #  Make sure we have an up to date set of fibre statistics
    
    UpdateFibreCounts
    
    #  For each fibre type we create a frame in the window that can be
    #  used to display its details. We build it up for display, but it
    #  may be disabled almost immediately by ReformatFibreStats{}. 
      
    set FibreTypeList [FibreTypes]
    set NumFibreTypes [llength $FibreTypeList]

    for { set Index 0 } { $Index < $NumFibreTypes } { incr Index } {

        if { $FibStats($Index,enabled) } {
            frame $Window.type$Index
            pack $Window.type$Index -side top -anchor w
            frame $Window2.type$Index
            pack $Window2.type$Index -side top -anchor w

            label $Window.type$Index.name -width 11 -anchor e \
                             -textvariable FibStats($Index,name)
            pack $Window.type$Index.name -side left -padx 2m -pady 1m
            label $Window.type$Index.fibres -width 7 -anchor e \
                           -textvariable FibStats($Index,fibres)
            pack $Window.type$Index.fibres -side left -padx 2m -pady 1m
            label $Window.type$Index.skies -width 7 -anchor e \
                            -textvariable FibStats($Index,skies)
            pack $Window.type$Index.skies -side left -padx 2m -pady 1m
            label $Window.type$Index.objects -width 7 -anchor e \
                          -textvariable FibStats($Index,objects)
            pack $Window.type$Index.objects -side left -padx 2m -pady 1m
            label $Window.type$Index.unalloc -width 7 -anchor e \
                          -textvariable FibStats($Index,unalloc)
            pack $Window.type$Index.unalloc -side left -padx 2m -pady 1m
            
            label $Window2.type$Index.targets -width 7 -anchor e \
                          -textvariable FibStats($Index,targets)
            pack $Window2.type$Index.targets -side left -padx 2m -pady 1m
            label $Window2.type$Index.skyTargets -width 7 -anchor e \
                          -textvariable FibStats($Index,skyTargets)
            pack $Window2.type$Index.skyTargets -side left -padx 2m -pady 1m
        }
    }
    
    #  Then we set up the totals line that is always shown at the bottom
    #  of the frame.
    
    frame $Window.totals
    pack $Window.totals -side top -anchor w

    label $Window.totals.name -width 11 -anchor e -text "Totals"
    pack $Window.totals.name -side left -padx 2m -pady 1m
    label $Window.totals.fibres -width 7 -anchor e -relief sunken \
                   -textvariable FibStats(Total,fibres)
    pack $Window.totals.fibres -side left -padx 2m -pady 1m
    label $Window.totals.skies -width 7 -anchor e -relief sunken \
                    -textvariable FibStats(Total,skies)
    pack $Window.totals.skies -side left -padx 2m -pady 1m
    label $Window.totals.objects -width 7 -anchor e -relief sunken \
                  -textvariable FibStats(Total,objects)
    pack $Window.totals.objects -side left -padx 2m -pady 1m
    label $Window.totals.unalloc -width 7 -anchor e -relief sunken \
                  -textvariable FibStats(Total,unalloc)
    pack $Window.totals.unalloc -side left -padx 2m -pady 1m
    
    frame $Window2.totals
    pack $Window2.totals -side top -anchor w
    label $Window2.totals.dummy -width 7 -anchor e \
                  -text " "
    pack $Window2.totals.dummy -side left -padx 2m -pady 1m
    
    ReformatFibreStats $Window $Window2
}

#------------------------------------------------------------------------------ 
#
#                  R e f o r m a t  F i b r e  S t a t s  
#
#   This procedure redisplays the variable part of the fibre allocation
#   statistics panel. It is usually invoked after the set of fibre types
#   being configured is changed. The two arguments give the names of the
#   frames used for the allocation and available target statistics
#   respectively.

proc ReformatFibreStats { Window Window2 } {
                                  
    #  Global variables used by this routine:

    global Cfid               ;# Sds Id for the top level of the current
                              ;# configuration structure.
    global FibStats           ;# Array holding fibre allocation statistics
    global TkAvailable        ;# Set if Tk operations are permitted.
            
    UpdateFibreCounts
    
    if { !$TkAvailable } return
    
    #  Work through all the different fibre types and see which are in use.
    #  Each fibre type has a frame in the window, and we either pack it
    #  or forget it (which removes it but doesn't destroy it) depending
    #  on whether that fibre type is enabled or not.
    
    set FibreTypeList [FibreTypes]
    set NumFibreTypes [llength $FibreTypeList]

    for { set Index 0 } { $Index < $NumFibreTypes } { incr Index } {

        set Enable 0
        if { $Cfid != 0 } {
            set Enable $FibStats($Index,enabled)
        }
        
        if { $Enable } {
            pack $Window.type$Index -before $Window.totals -side top -anchor w
            pack $Window2.type$Index -before $Window2.totals -side top -anchor w
        } else {
            pack forget $Window.type$Index
            pack forget $Window2.type$Index
        }
    }
}       

#------------------------------------------------------------------------------
#
#                     W o r k i n g  D i a l o g u e
#
#   This is a utility routine used to display a progress bar. It is passed
#   the name of the window to use and the header text to use. It puts
#   up a progress bar and an abort button. UpdateProgress{} will then
#   be used to update this display.

proc WorkingDialog {w text} {

   global Abort              ;# Tied to the working dialog's abort button
   global Gtext              ;# Tied to the text in the working dialog
   global config_font_1      ;# Font used in dialogues

   set Abort 0

   #  Create a toplevel window and set its title

   toplevel $w
   wm title $w "Action in Progress"
   wm geometry $w +0-50

   #  Split into top, mid and bottom sections

   frame $w.top -relief raised -bd 1
   pack $w.top -side top -fill both
   frame $w.mid -relief raised -bd 1
   pack $w.mid -side top -fill both -expand 1
   frame $w.bot -relief raised -bd 1
   pack $w.bot -side bottom -fill both

   #  Draw a canvas widget for the progress bar in the mid section
  
   label $w.mid.label1 -textvariable Gtext -width 40 \
         -relief sunken -borderwidth 2 \
         -font $config_font_1
   pack $w.mid.label1 -side top -padx 5m -pady 5m

   canvas $w.bar -height 22 -width 300 -borderwidth 2 -relief sunken
   pack $w.bar -side top -padx 12 -pady 2

   #  Draw the hourglass bitmap and message text in the top section

   label $w.top.bitmap -bitmap hourglass
   pack $w.top.bitmap -side left -padx 5m -pady 5m
   label $w.top.label -text $text \
         -font $config_font_1
   pack $w.top.label -side right -padx 5m -pady 5m

   #  Draw Abort button in the bottom section

   button $w.bot.abort -text Cancel \
       -command "set Abort 1"
   pack $w.bot.abort -side left -expand 1 -padx 5m -pady 2m \
          -ipadx 2m -ipady 1m

}

#------------------------------------------------------------------------- 
#
#                       U p d a t e  P r o g r e s s
#
#   This procedure is called to update the progress bar and other information
#   in the display created by WorkingDialog{}. It is passed the name of the
#   window to use, the percentage complete and a text string describing
#   what's going on. It returns the state of the abort button.

proc UpdateProgress {w text percent} {

    global Abort              ;# Tied to the working dialog's abort button
    global Gtext              ;# Tied to the text in the working dialog

    # This procedure appears to be invoked sometimes where
    # $w.bar does not exist, due to timing issues in automatic
    # sequences (such as F2).  Hence we don't complain if it
    # does not exist.

    if {$text != $Gtext} {
        catch "$w.bar delete bar"
    } else {
        set x2 [expr {$percent * 3}]
        catch "$w.bar create rectangle 0 0 $x2 25 -fill salmon2 -tag bar"
    }
    set Gtext $text

    return $Abort
}

#------------------------------------------------------------------------- 
#
#                       D r a w  B u t t o n s
#
#   This is utility routine that draws the standard OK and Cancel buttons at
#   the bottom of a dialog. It is passed the name of the dialog's top level
#   window and the command to be executed by the Ok button.

proc DrawButtons {w command} {

   frame $w.buttons
   button $w.buttons.ok -command "destroy $w; $command" -text OK -width 4
   pack $w.buttons -side bottom -fill both
   frame $w.buttons.default -relief sunken -bd 1
   raise $w.buttons.ok $w.buttons.default
   pack $w.buttons.default -side left -expand 1 -padx 3m -pady 2m
   pack $w.buttons.ok -in $w.buttons.default -padx 2m -pady 2m \
                    -ipadx 2m -ipady 1m
   button $w.buttons.cancel -command "destroy $w" -text Close -width 4
   pack $w.buttons.cancel -side left -expand 1 -padx 2m -pady 3m \
        -ipadx 2m -ipady 1m
   bind $w <Return> "$w.buttons.ok flash; destroy $w; $command"
}


#------------------------------------------------------------------------------ 
#
#                      A l l  U p d a t e  B a t c h
#
#   This routine used to be invoked by allocation algorithms running in
#   batch mode to report allocation/deallocation of fibres.. This is now
#   performed by AllocRpt{}, and this routine is now just a dummy.

proc AllUpdateBatch {type fibre object tag theta description percent} {
    
   return
}

#------------------------------------------------------------------------------ 

#                      M a i n - l i n e   C o d e
#
#  This is the code executed on initialization, when the configure program
#  is loaded and run.  It is this code that should be thought of as the
#  main program for the configuration program. It initialises all the
#  necessary Tcl global variables, locates and loads the other tcl files
#  used by the program then looks at the command line flags and
#  sets itself up to handle them. In some cases, it will complete the
#  requested batch mod eprocessing withing this routine. Usually, however,
#  it ends by calling Setup{} - defined at the start of this file - which
#  sets up the GUI used by the program. Once the GUI is displayed, this
#  program becomes an event-driven Tcl program, responding to events from
#  the GUI.

#   Global variables used in this section:

global ConfigDirDef       ;# Directory used for configuration files
global config_font_1      ;# Font used in dialogues
global DSSOutMode         ;# Set if DSS output mode has been selected
global ImportFile         ;# Name of import file if such a file is used,
global ImportMode         ;# True if import mode is used (-i on command line)

#  One important preliminary. If tcl version 8.0 or later is being used, 
#  SDS routines will be in the SDS namespace.  We must import these commands
#  into the global namespce.

if { [info tclversion] >= 8.0 } {
    namespace eval sds { namespace export * }
    namespace import sds::*
}

#  Set some of the basic global variables

set ImportMode 0
set DSSOutMode 0
set InitialPlate 0
set Instrument ""
set BatchMode 0
set DoAllocation 0
set FibreComboMode ""
set Cfid 0
set UtDateSpec ""

#  There are some things we need to know before we Initialise, so we do a 
#  first pass through the command line arguments at this point. In most cases
#  these are to do with being invoked in batch mode.
#
#  NOTE, Tcl/Tk options are processed before this.  They may be abbreviated
#  to the smallest unique length within the set (shown below).   Use "--"
#  to terminate Tcl/Tk interpretation of options.
#
#        -c/-colormap    Colormap for main window
#        -d/-display     Display to use
#        -g/-geometry    Initial geometry for main window
#        -n/-name        Name to use for application
#        -s/-sync        Use synchronous mode for display server
#        -v/visual       Visual for main window
#        -u/-use         Id of window in which to embed application

set argc [llength $argv]
set i 0
for { set i 0 } { $i < $argc } {incr i } {
   set arg [lindex $argv $i]
   if {($arg == "-a") || ($arg == "-d") } {
    
       # Processed both now and in second pass at options, since we need
       # it before prompting for the instrument.
       
       set BatchMode 1

   } elseif {$arg == "-s"} {
    
       # Indicates allocation should start automatically.
       
       set BatchMode 1
       set DoAllocation 1

   } elseif {$arg == "-r" } {
   
       # Indicates allocation should start automatically, but should reallocate
       
       set BatchMode 1
       set DoAllocation 2

   } elseif {$arg == "-f"} {
   
       # Set initial configuration file.
     
       incr i
       set ConfigFile [lindex $argv $i]

   } elseif {$arg == "-i"} {
   
      #  Reads in the ASCII field file specified with the -f option.
      #  The file of the same name but a file type of ".imp" will
      #  also be read.  This is an ASCII file with a set of fibre no, 
      #  fibre name pairs. This file will be used to perform a 
      #  specific set of allocations with no checks whatsoever. It 
      #  will then write an output file (.sds) and exit.  The instrument
      #  defaults to 2dF if not specified with -I.
      
      set ImportMode 1 

   } elseif { ($arg == "-z") || ($arg == "-Z") } {
   
       # Indicates we should generate a DSS or listing  file automatically.
       
       set BatchMode 1
       set DSSOutMode 1

   } elseif {$arg == "-p"} {
   
       # Set the initial configuration plate.
     
       incr i
       set InitialPlate [lindex $argv $i]

   } elseif {$arg == "-I"} {
   
       # Select the instrument to configure.
       
       incr i
       set Instrument [lindex $argv $i]
       
   } elseif { $arg == "-k" } {
   
       # Disable fibre type.  We ignore this at this stage.  See the
       # post initialise processing of the arguments
     
       incr i
     
   } elseif { ($arg == "-b") || ($arg == "-d") || ($arg == "-n") } {
   
       # Options to ignore - from old versions of the program.  All
       # these had arguments.
     
       incr i
     
   } elseif { $arg == "-utdate" } {
   
       # Specify UT date and (optionally) time.
     
       incr i
       set UtDateSpec [lindex $argv $i]
     
   } elseif { ($arg == "-tk_library") } {
   
       #  The program is being run only to report on the location of the
       #  Tk anciliary files.
     
       puts $tk_library
       exit
     
   } elseif { ($arg == "-tcl_library") } {
   
       #  The program is being run only to report on the location of the
       #  Tcl anciliary files.
     
       puts $tcl_library
       exit
     
   } elseif { ($arg == "-tcl_version") } {
   
       #  The program is being run only to report on the Tcl version
       #  it has been linked with.
     
       puts [info tclversion]
       exit
     
   } elseif { ($arg == "-tk_version") } {
   
       #  The program is being run only to report on the Tk version
       #  it has been linked with. This will have been reported by the
       #  C initialisation code - there isn't an [info tkversion].
     
       exit
     
   } elseif { ($arg == "-use_all_fibres") } {
   
       #  The use_all_fibres option is handled entirely by the C initialisation
       #  code at startup. We ignore it here.
            
   } elseif { ($arg == "-ch")||($arg == "-h") } {
   
       #  Produce option help.  Note that by default, -h is handled by
       #  Tcl/Tk, unless escaped.  So we also allow -ch
     
       puts "AAO Fibre Instrument configuration program"
       puts "Configure Options"
       puts "    -f filename    #Specify the input file (.fld or .sds or .paf)"
       puts "    -i             #Enable import mode."
       puts "    -z             #Write DSS listing file"
       puts "    -Z             #Write listing file of all objects"
       puts "    -p n           #Select the plate to configure, 0 or 1"
       puts "    -s             #Start configuration of -f file immediately"
       puts "    -r             #Start re-configuration of -f file immediately"
       puts "    -I 2df/6df     #Select instrument to configure"
       puts "    -k n           #Disable configuration of the specified "
       puts "                   # fibre type number.  Can list mutliple"
       puts "                   # in a quoted string"
       puts "    -a             #Invoke batch mode"
       puts "    -d             #Read in configuration file (.fld) and write"
       puts "                   # out the SDS file without configuring"
       puts "    -utdate date   #Set the UTDATE for the configuration"
       puts "    -ch            #Produce this list"
       puts "    -h             #Produce this list"
       puts ""
       puts "Tcl/Tk Options."
       puts "   These are processed before the above options"
       puts "   If there is a conflict, terminate them with \"--"
       puts "     e.g., if specifying configure's \"-h or \"-d, specify \"--\" \
                                                                         first."
       puts "   These may be abbreviated to the minimum unique string"
       puts ""
       puts "   -colormap map   # Colormap for main window"
       puts "   -display  disp  # Display to use"
       puts "   -geometry spec  # Initial geometry for window"
       puts "   -name name      # Name to use for application"
       puts "   -sync           # Use synchronous mode for display server"
       puts "   -visual vis     # Visual for main window"
       puts "   -use id         # Id of window in which to embed application"
       puts "    --             # Pass all remaining arguments to script"
       exit
     } else {
       global argv0
       puts stderr "configure:Invalid argument \"$arg\""
       puts stderr "configure:Should be one of -"
       puts stderr "    -f filename|-i|-z|-p n|-s|-r|-I 2df/6df|-k|-a|-d|-h"
       exit
     }
}

#  Determine the default location for out tcl and etc. files.  We
#  first look in CONFIGURE_DIR, but if that is not defined, we will
#  try the directory containing the exectuable.  

if { [string length [TranslateName CONFIGURE_DIR -default ""]] == 0 } {
    set ConfigDirDef [file join [ExeDir] ".." "lib"]
    if [file exists [file join $ConfigDirDef util.tcl]] {
        puts "Will pick up script files from $ConfigDirDef directory"
    } else {
        puts stderr "Warning, CONFIGURE_DIR environment variable not defined"
        puts stderr "  And scripts don't appear to be with executable"
        puts stderr "  May not be able to find script files"
    }
} else {
    set ConfigDirDef "."
}

#  We set the current directory as the default for opening and saving files

set CurrentInDirectory [pwd]
set CurrentOutDirectory [pwd]

#  Set the global batch mode variables.  In batch mode, we don't initialise
#  Tk at all - it can't run without a DISPLAY environment variable, and one
#  may not be set in batch mode.

if { $BatchMode == 0 } {
    set TkAvailable 1
} else {
    set TkAvailable 0
}

#  Dtclu.  Now found in the util script.  Note that the Dtclu routines are
#  all concerned with providing elaborate Tk facilities. If Tk isn't
#  available, they won't work - and util.tcl won't even execute. Which is
#  OK, because in batch mode we don't need them.

if { ($TkAvailable) } {
    source [TranslateName CONFIGURE_DIR -file util.tcl \
               -default [file join $ConfigDirDef util.tcl]]
}

#  The procs.tcl script defines most of the dialog boxes used by
#  the user interface

source [TranslateName CONFIGURE_DIR -file procs.tcl \
            -default [file join $ConfigDirDef procs.tcl]]

#  Positioner mimic display procedures

source [TranslateName CONFIGURE_DIR -file mimic.tcl \
            -default [file join $ConfigDirDef mimic.tcl]]

#  Give the user the option of select the instrument to configure for.
#  But in batch mode we default to the first in the list (since Tk isn't
#  available to put up the dialogue). If there's only one option, we use
#  that without prompting.
#

if { ($TkAvailable) &&($ImportMode == 0) && ($Instrument == "") } {
    set InstrumentList [Instruments]
    set listlength [llength $InstrumentList]
    if { $listlength <= 1 } {
        set button 0
    } else {
        wm withdraw .
        set button [eval tk_dialog .inst_select \
                    \"Configure Instrument Selector\" \
                    \"Please select the Fibre Instrument to configure for\" \
                    info 1 Cancel [Instruments]]
	#"
        if { $button == 0 }  {
            exit
        }
        incr button -1
    }
    set Instrument [lindex [Instruments] $button]
                    
}

#  Get the size of the screen - Initialise{} needs this to set up scaling
#  factors and window sizes.

set screenWidth 0
set screenHeight 0
if { ($TkAvailable) } {
   set screenWidth [winfo screenwidth .]
   set screenHeight [winfo screenheight .]
}

#  Invoke the initialisation code for the C routines contained in
#  configure.c This now needs to be after the args parsing bit above

Initialise $Instrument $InitialPlate $screenWidth $screenHeight

#  More Tcl global variable initialisation.

#  Set intial values for the FibreTypeFlags variable.

global FibreTypeFlags
set Fibres [FibreTypes]
set i 1
set FibreTypeFlags(0) 1
foreach fibre $Fibres {
    set FibreTypeFlags($i) 1
    incr i
}

#  Set intial values for the TargetCriteriaFlags variable.

global TargetCriteriaFlags
global TargetNumCriteria
set Criteria [TargetCriteria]
set TargetNumCriteria [llength $Criteria]
set ArrowHighlighting 0
HighlightTargetsClear

#  Set initial value for the input magnitude filter - no filtering

SetMagnitudeLimits 0 0 0

#  We don't display non-existent fibres.

set ShowEmptyFlag 0

#  Classify the current selected fibre combination

CurrentFibreCombo

#  Set the global variables for the Argus settings (only used for FLAMES)

set ArgusInUse "unused"
set ArgusScale "1:1"
set ArgusAngle 0
set ArgusStatus "Angle: $ArgusAngle Scale: $ArgusScale"

#  Set the global variables used for the basic sequence display

set CheckStatus "Not checked"
set AllocationStatus "No fibres allocated"
set OutputFileStatus "Not saved"

#  Second pass at arguments, things that can only be down after Initialise
#  has set up the C routines in configure.c

set i 0
foreach arg $argv {
   incr i
   if {$arg == "-k"} {
   
       # Handle selection of fibre types to configure.
       
       global FibreTypeFlags
       foreach item [lindex $argv $i] {
           set FibreTypeFlags($item) 0
       }
       MarkFibreType FibreTypeFlags
   }

   if {$arg == "-a"} {
   
       set FileType [ClassifyFile $ConfigFile]
       puts "File $ConfigFile is type $FileType"
       
       # Invoke batch mode
       
       if { $FileType == "PAF" } {
           DoOpenPAF $ConfigFile
       } elseif { $FileType == "SDS" } {
           DoOpenSDS $ConfigFile
       } elseif { $FileType == "FLD" } {
           OpenAsciiFile $ConfigFile
       } else {
           puts stderr "Cannot open $ConfigFile - type unknown"
       }
       if { $UtDateSpec != "" } {
           #puts "UT date $UtDateSpec"
           ApplyUTDate $UtDateSpec
       }
       DoAllocationBatch [SdsCopy $Cfid]
       set SaveFile [SdsFileName $ConfigFile]

       set ListFile [ListFileName $ConfigFile]
       MakeListing $Cfid $ListFile all
       CompressUnalloc $Cfid
       puts stdout "Writing file $SaveFile"
       SdsWrite $Cfid $SaveFile
       exit
   }
   if {$arg == "-d"} {
       set Cfid 0
       
       # Read in a .fld file and write out the equivalent .sds file.
       
       OpenAsciiFile $ConfigFile
       set SaveFile [SdsFileName $ConfigFile]
       puts stdout "Writing file $SaveFile"
       SdsWrite $Cfid $SaveFile
       exit
   }
   if { ($arg == "-z") || ($arg == "-Z") } {
       set Cfid 0
       
       #  Read in a .sds or .paf file and write out the equivalent .lis file,
       #  or .dss file, depending on whether -z or -Z was specified. 
       
       set FileType [ClassifyFile $ConfigFile]
       puts "File $ConfigFile is type $FileType"
       
       if { $FileType == "PAF" } {
           DoOpenPAF $ConfigFile
       } elseif { $FileType == "SDS" } {
           DoOpenSDS $ConfigFile
       } elseif { $FileType == "FLD" } {
           OpenAsciiFile $ConfigFile
       } else {
           puts stderr "Cannot open $ConfigFile - type unknown"
       }
       set Value 0
       if { $arg == "-z" } {
           set ListFile [DSSFileName $ConfigFile]
           MakeDSSListing $Cfid $ListFile $Value
       } else {
           set ListFile [ListFileName $ConfigFile]
           MakeListing $Cfid $ListFile alloc
       }
       exit
   }
}

#  Get the current field details for the configuring plate and draw
#  the mimic display for it.

global config_font_1 tcl_platform
if { ($tcl_platform(platform) == "windows") ||
           ($tcl_platform(platform) == "macintosh")} {
    set config_font_1 {Times 14 }
    set config_font_2 {Times 12 }
} else {
    set config_font_1 {Times 14 }
    set config_font_2 {Times 12 }
}

#  Create the main display window, menus and mimic display etc.

Setup

#  What happens after this is up to the user and what they do with the GUI.

#        1         2         3         4         5         6         7         8
#2345678901234567890123456789012345678901234567890123456789012345678901234567890
