/*
 * Decompiled with CFR 0.152.
 */
package org.kse.crypto.x509;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.CRLException;
import java.security.cert.CertPath;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.ResourceBundle;
import javax.security.auth.x500.X500Principal;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.util.encoders.Base64;
import org.kse.crypto.CryptoException;
import org.kse.crypto.SecurityProvider;
import org.kse.crypto.signing.SignatureType;
import org.kse.crypto.x509.X500NameUtils;
import org.kse.utilities.ArrayUtils;
import org.kse.utilities.StringUtils;
import org.kse.utilities.io.IOUtils;
import org.kse.utilities.pem.PemInfo;
import org.kse.utilities.pem.PemUtil;

public final class X509CertUtil {
    private static ResourceBundle res = ResourceBundle.getBundle("org/kse/crypto/x509/resources");
    private static final String X509_CERT_TYPE = "X.509";
    private static final String PKCS7_ENCODING = "PKCS7";
    private static final String PKI_PATH_ENCODING = "PkiPath";
    private static final String CERT_PEM_TYPE = "CERTIFICATE";
    private static final String PKCS7_PEM_TYPE = "PKCS7";
    public static final String BEGIN_CERTIFICATE = "-----BEGIN CERTIFICATE-----";
    public static final String END_CERTIFICATE = "-----END CERTIFICATE-----";
    public static final String BASE64_TESTER = "^([A-Za-z0-9+/]{4})*([A-Za-z0-9+/]{4}|[A-Za-z0-9+/]{3}=|[A-Za-z0-9+/]{2}==)$";

    private X509CertUtil() {
    }

    public static X509Certificate[] loadCertificates(byte[] certsBytes) throws CryptoException {
        try {
            certsBytes = X509CertUtil.fixCommonInputCertProblems(certsBytes);
            CertificateFactory cf = CertificateFactory.getInstance(X509_CERT_TYPE, SecurityProvider.BOUNCY_CASTLE.jce());
            Collection<? extends Certificate> certs = cf.generateCertificates(new ByteArrayInputStream(certsBytes));
            ArrayList<X509Certificate> loadedCerts = new ArrayList<X509Certificate>();
            for (X509Certificate x509Certificate : certs) {
                if (x509Certificate == null) continue;
                loadedCerts.add(x509Certificate);
            }
            return loadedCerts.toArray(new X509Certificate[loadedCerts.size()]);
        }
        catch (NoSuchProviderException ex) {
            throw new CryptoException(res.getString("NoLoadCertificate.exception.message"), ex);
        }
        catch (CertificateException ex) {
            try {
                return X509CertUtil.loadCertificatesPkiPath(new ByteArrayInputStream(certsBytes));
            }
            catch (CryptoException ex2) {
                throw new CryptoException(res.getString("NoLoadCertificate.exception.message"), ex);
            }
        }
    }

    private static X509Certificate[] loadCertificatesPkiPath(InputStream is) throws CryptoException {
        try {
            CertificateFactory cf = CertificateFactory.getInstance(X509_CERT_TYPE, SecurityProvider.BOUNCY_CASTLE.jce());
            CertPath certPath = cf.generateCertPath(is, PKI_PATH_ENCODING);
            List<? extends Certificate> certs = certPath.getCertificates();
            ArrayList<X509Certificate> loadedCerts = new ArrayList<X509Certificate>();
            for (X509Certificate x509Certificate : certs) {
                if (x509Certificate == null) continue;
                loadedCerts.add(x509Certificate);
            }
            X509Certificate[] x509CertificateArray = loadedCerts.toArray(new X509Certificate[loadedCerts.size()]);
            return x509CertificateArray;
        }
        catch (NoSuchProviderException | CertificateException e) {
            throw new CryptoException(res.getString("NoLoadPkiPath.exception.message"), e);
        }
        finally {
            IOUtils.closeQuietly(is);
        }
    }

    private static byte[] fixCommonInputCertProblems(byte[] certs) {
        String certsStr = new String(certs);
        if (certsStr.startsWith(BEGIN_CERTIFICATE)) {
            certsStr = certsStr.replaceAll(BEGIN_CERTIFICATE, "");
            certsStr = certsStr.replaceAll(END_CERTIFICATE, "!");
        }
        String[] splitCertsStr = certsStr.split("!");
        byte[] allDecoded = null;
        for (String singleCertB64 : splitCertsStr) {
            byte[] decoded = X509CertUtil.attemptBase64Decode(singleCertB64.trim());
            if (decoded == null) continue;
            allDecoded = ArrayUtils.add(allDecoded, decoded);
        }
        if (allDecoded != null) {
            return allDecoded;
        }
        return certs;
    }

    private static byte[] attemptBase64Decode(String toTest) {
        char[] base64 = new char[]{'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '+', '/', '='};
        toTest = toTest.replaceAll("\\s", "");
        StringBuffer sb = new StringBuffer();
        block2: for (int i = 0; i < toTest.length(); ++i) {
            char c = toTest.charAt(i);
            for (int j = 0; j < base64.length; ++j) {
                if (c == base64[j]) {
                    sb.append(c);
                    continue block2;
                }
                if (c == '\u0000') continue block2;
            }
            return null;
        }
        try {
            return Base64.decode((String)sb.toString());
        }
        catch (Exception exception) {
            return null;
        }
    }

    public static X509CRL loadCRL(byte[] crlData) throws CryptoException {
        try {
            CertificateFactory cf = CertificateFactory.getInstance(X509_CERT_TYPE);
            return (X509CRL)cf.generateCRL(new ByteArrayInputStream(crlData));
        }
        catch (CRLException | CertificateException ex) {
            throw new CryptoException(res.getString("NoLoadCrl.exception.message"), ex);
        }
    }

    public static X509Certificate[] convertCertificates(Certificate[] certsIn) throws CryptoException {
        if (certsIn == null) {
            return new X509Certificate[0];
        }
        X509Certificate[] certsOut = new X509Certificate[certsIn.length];
        for (int i = 0; i < certsIn.length; ++i) {
            certsOut[i] = X509CertUtil.convertCertificate(certsIn[i]);
        }
        return certsOut;
    }

    public static X509Certificate convertCertificate(Certificate certIn) throws CryptoException {
        try {
            CertificateFactory cf = CertificateFactory.getInstance(X509_CERT_TYPE, SecurityProvider.BOUNCY_CASTLE.jce());
            ByteArrayInputStream bais = new ByteArrayInputStream(certIn.getEncoded());
            return (X509Certificate)cf.generateCertificate(bais);
        }
        catch (NoSuchProviderException | CertificateException e) {
            throw new CryptoException(res.getString("NoConvertCertificate.exception.message"), e);
        }
    }

    public static X509Certificate[] orderX509CertChain(X509Certificate[] certs) {
        if (certs == null) {
            return new X509Certificate[0];
        }
        if (certs.length <= 1) {
            return certs;
        }
        ArrayList paths = new ArrayList();
        for (int i = 0; i < certs.length; ++i) {
            ArrayList<X509Certificate> path = new ArrayList<X509Certificate>();
            X509Certificate issuerCert = certs[i];
            path.add(issuerCert);
            X509Certificate newIssuer = null;
            while ((newIssuer = X509CertUtil.findIssuedCert(issuerCert, certs)) != null) {
                issuerCert = newIssuer;
                path.add(0, newIssuer);
            }
            paths.add(path);
        }
        ArrayList longestPath = (ArrayList)paths.get(0);
        for (int i = 1; i < paths.size(); ++i) {
            ArrayList path = (ArrayList)paths.get(i);
            if (path.size() <= longestPath.size()) continue;
            longestPath = path;
        }
        return longestPath.toArray(new X509Certificate[longestPath.size()]);
    }

    private static X509Certificate findIssuedCert(X509Certificate issuerCert, X509Certificate[] certs) {
        for (int i = 0; i < certs.length; ++i) {
            X509Certificate cert = certs[i];
            if (issuerCert.getSubjectX500Principal().equals(cert.getSubjectX500Principal()) && issuerCert.getIssuerX500Principal().equals(cert.getIssuerX500Principal()) || !issuerCert.getSubjectX500Principal().equals(cert.getIssuerX500Principal())) continue;
            return cert;
        }
        return null;
    }

    public static byte[] getCertEncodedX509(X509Certificate cert) throws CryptoException {
        try {
            return cert.getEncoded();
        }
        catch (CertificateException ex) {
            throw new CryptoException(res.getString("NoDerEncodeCertificate.exception.message"), ex);
        }
    }

    public static String getCertEncodedX509Pem(X509Certificate cert) throws CryptoException {
        PemInfo pemInfo = new PemInfo(CERT_PEM_TYPE, null, X509CertUtil.getCertEncodedX509(cert));
        return PemUtil.encode(pemInfo);
    }

    public static String getCertsEncodedX509Pem(X509Certificate[] certs) throws CryptoException {
        StringBuilder sb = new StringBuilder();
        for (X509Certificate cert : certs) {
            sb.append(X509CertUtil.getCertEncodedX509Pem(cert));
        }
        return sb.toString();
    }

    public static byte[] getCertEncodedPkcs7(X509Certificate cert) throws CryptoException {
        return X509CertUtil.getCertsEncodedPkcs7(new X509Certificate[]{cert});
    }

    public static byte[] getCertsEncodedPkcs7(X509Certificate[] certs) throws CryptoException {
        try {
            ArrayList encodedCerts = new ArrayList();
            Collections.addAll(encodedCerts, certs);
            CertificateFactory cf = CertificateFactory.getInstance(X509_CERT_TYPE, SecurityProvider.BOUNCY_CASTLE.jce());
            CertPath cp = cf.generateCertPath(encodedCerts);
            return cp.getEncoded("PKCS7");
        }
        catch (NoSuchProviderException | CertificateException e) {
            throw new CryptoException(res.getString("NoPkcs7Encode.exception.message"), e);
        }
    }

    public static String getCertEncodedPkcs7Pem(X509Certificate cert) throws CryptoException {
        return X509CertUtil.getCertsEncodedPkcs7Pem(new X509Certificate[]{cert});
    }

    public static String getCertsEncodedPkcs7Pem(X509Certificate[] certs) throws CryptoException {
        PemInfo pemInfo = new PemInfo("PKCS7", null, X509CertUtil.getCertsEncodedPkcs7(certs));
        return PemUtil.encode(pemInfo);
    }

    public static byte[] getCertEncodedPkiPath(X509Certificate cert) throws CryptoException {
        return X509CertUtil.getCertsEncodedPkiPath(new X509Certificate[]{cert});
    }

    public static byte[] getCertsEncodedPkiPath(X509Certificate[] certs) throws CryptoException {
        try {
            ArrayList encodedCerts = new ArrayList();
            Collections.addAll(encodedCerts, certs);
            CertificateFactory cf = CertificateFactory.getInstance(X509_CERT_TYPE, SecurityProvider.BOUNCY_CASTLE.jce());
            CertPath cp = cf.generateCertPath(encodedCerts);
            return cp.getEncoded(PKI_PATH_ENCODING);
        }
        catch (NoSuchProviderException | CertificateException e) {
            throw new CryptoException(res.getString("NoPkcs7Encode.exception.message"), e);
        }
    }

    public static boolean verifyCertificate(X509Certificate signedCert, X509Certificate signingCert) throws CryptoException {
        try {
            signedCert.verify(signingCert.getPublicKey());
            return true;
        }
        catch (InvalidKeyException | SignatureException ex) {
            return false;
        }
        catch (NoSuchAlgorithmException | NoSuchProviderException | CertificateException ex) {
            throw new CryptoException(res.getString("NoVerifyCertificate.exception.message"), ex);
        }
    }

    public static X509Certificate[] establishTrust(X509Certificate cert, KeyStore[] keyStores) throws CryptoException {
        ArrayList<X509Certificate> ksCerts = new ArrayList<X509Certificate>();
        for (int i = 0; i < keyStores.length; ++i) {
            ksCerts.addAll(X509CertUtil.extractCertificates(keyStores[i]));
        }
        return X509CertUtil.establishTrust(cert, ksCerts);
    }

    private static X509Certificate[] establishTrust(X509Certificate cert, List<X509Certificate> compCerts) throws CryptoException {
        for (int i = 0; i < compCerts.size(); ++i) {
            X509Certificate compCert = compCerts.get(i);
            if (!cert.getIssuerX500Principal().equals(compCert.getSubjectX500Principal()) || !X509CertUtil.verifyCertificate(cert, compCert)) continue;
            if (compCert.getSubjectX500Principal().equals(compCert.getIssuerX500Principal())) {
                return new X509Certificate[]{cert, compCert};
            }
            X509Certificate[] tmpChain = X509CertUtil.establishTrust(compCert, compCerts);
            if (tmpChain == null) continue;
            X509Certificate[] trustChain = new X509Certificate[tmpChain.length + 1];
            trustChain[0] = cert;
            System.arraycopy(tmpChain, 0, trustChain, 1, tmpChain.length);
            return trustChain;
        }
        return null;
    }

    private static List<X509Certificate> extractCertificates(KeyStore keyStore) throws CryptoException {
        try {
            ArrayList<X509Certificate> certs = new ArrayList<X509Certificate>();
            Enumeration<String> aliases = keyStore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                if (!keyStore.isCertificateEntry(alias)) continue;
                certs.add(X509CertUtil.convertCertificate(keyStore.getCertificate(alias)));
            }
            return certs;
        }
        catch (KeyStoreException ex) {
            throw new CryptoException(res.getString("NoExtractCertificates.exception.message"), ex);
        }
    }

    public static String matchCertificate(KeyStore keyStore, X509Certificate cert) throws CryptoException {
        try {
            Enumeration<String> aliases = keyStore.aliases();
            while (aliases.hasMoreElements()) {
                X509Certificate compCert;
                String alias = aliases.nextElement();
                if (!keyStore.isCertificateEntry(alias) || !cert.equals(compCert = X509CertUtil.convertCertificate(keyStore.getCertificate(alias)))) continue;
                return alias;
            }
            return null;
        }
        catch (KeyStoreException ex) {
            throw new CryptoException(res.getString("NoMatchCertificate.exception.message"), ex);
        }
    }

    public static String getCertificateAlias(X509Certificate cert) {
        X500Principal subject = cert.getSubjectX500Principal();
        X500Principal issuer = cert.getIssuerX500Principal();
        String subjectCn = X500NameUtils.extractCN(X500NameUtils.x500PrincipalToX500Name(subject));
        String issuerCn = X500NameUtils.extractCN(X500NameUtils.x500PrincipalToX500Name(issuer));
        if (StringUtils.isBlank(subjectCn)) {
            return "";
        }
        if (StringUtils.isBlank(issuerCn) || subjectCn.equals(issuerCn)) {
            return subjectCn;
        }
        return MessageFormat.format("{0} ({1})", subjectCn, issuerCn);
    }

    public static String getShortName(X509Certificate cert) {
        X500Name subject = X500NameUtils.x500PrincipalToX500Name(cert.getSubjectX500Principal());
        String shortName = X500NameUtils.extractCN(subject);
        if (StringUtils.isBlank(shortName)) {
            shortName = subject.toString();
        }
        return shortName;
    }

    public static String getCertificateSignatureAlgorithm(X509Certificate cert) {
        String algorithm = cert.getSigAlgName();
        SignatureType type = SignatureType.resolveJce(algorithm);
        if (type != null) {
            algorithm = type.friendly();
        } else {
            type = SignatureType.resolveOid(algorithm);
            if (type != null) {
                algorithm = type.friendly();
            }
        }
        return algorithm;
    }

    public static boolean isCertificateSelfSigned(X509Certificate cert) {
        return cert.getIssuerX500Principal().equals(cert.getSubjectX500Principal());
    }
}

