import * as React from 'react';
import * as ReactDOM from 'react-dom';
export const createPortalNode = () => {
    let initialProps = {};
    let parent;
    let lastPlaceholder;
    const portalNode = Object.assign(document.createElement('div'), {
        setPortalProps: (props) => {
            initialProps = props;
        },
        getInitialPortalProps: () => {
            return initialProps;
        },
        mount: (newParent, newPlaceholder) => {
            if (newPlaceholder === lastPlaceholder) {
                // Already mounted - noop.
                return;
            }
            portalNode.unmount();
            newParent.replaceChild(portalNode, newPlaceholder);
            parent = newParent;
            lastPlaceholder = newPlaceholder;
        },
        unmount: (expectedPlaceholder) => {
            if (expectedPlaceholder && expectedPlaceholder !== lastPlaceholder) {
                // Skip unmounts for placeholders that aren't currently mounted
                // They will have been automatically unmounted already by a subsequent mount()
                return;
            }
            if (parent && lastPlaceholder) {
                parent.replaceChild(lastPlaceholder, portalNode);
                parent = undefined;
                lastPlaceholder = undefined;
            }
        }
    });
    return portalNode;
};
export class InPortal extends React.PureComponent {
    constructor(props) {
        super(props);
        this.addPropsChannel = () => {
            Object.assign(this.props.node, {
                setPortalProps: (props) => {
                    // Rerender the child node here if/when the out portal props change
                    this.setState({ nodeProps: props });
                }
            });
        };
        this.state = {
            nodeProps: this.props.node.getInitialPortalProps()
        };
    }
    componentDidMount() {
        this.addPropsChannel();
    }
    componentDidUpdate() {
        this.addPropsChannel();
    }
    render() {
        const { children, node } = this.props;
        return ReactDOM.createPortal(React.Children.map(children, (child) => {
            if (!React.isValidElement(child))
                return child;
            return React.cloneElement(child, this.state.nodeProps);
        }), node);
    }
}
export class OutPortal extends React.PureComponent {
    constructor(props) {
        super(props);
        this.placeholderNode = React.createRef();
        this.passPropsThroughPortal();
    }
    passPropsThroughPortal() {
        const propsForTarget = Object.assign({}, this.props, { node: undefined });
        this.props.node.setPortalProps(propsForTarget);
    }
    componentDidMount() {
        const node = this.props.node;
        const placeholder = this.placeholderNode.current;
        const parent = placeholder.parentNode;
        node.mount(parent, placeholder);
        this.passPropsThroughPortal();
    }
    componentDidUpdate() {
        // We re-mount on update, just in case we were unmounted (e.g. by
        // a second OutPortal, which has now been removed)
        const node = this.props.node;
        const placeholder = this.placeholderNode.current;
        const parent = placeholder.parentNode;
        node.mount(parent, placeholder);
        this.passPropsThroughPortal();
    }
    componentWillUnmount() {
        const node = this.props.node;
        node.unmount(this.placeholderNode.current);
    }
    render() {
        // Render a placeholder to the DOM, so we can get a reference into
        // our location in the DOM, and swap it out for the portaled node.
        return React.createElement("div", { ref: this.placeholderNode });
    }
}
//# sourceMappingURL=index.js.map