"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _slicedToArray2 = _interopRequireDefault(require("@babel/runtime/helpers/slicedToArray"));

var _classCallCheck2 = _interopRequireDefault(require("@babel/runtime/helpers/classCallCheck"));

var _createClass2 = _interopRequireDefault(require("@babel/runtime/helpers/createClass"));

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

/**
 * minimal debounce function
 *
 * mostly for not spamming the server with requests when
 * searching with type ahead
 */
function debounce(func, wait) {
  var timeout;

  var debouncedFn = function debouncedFn() {
    var args = arguments;

    var later = function later() {
      func.apply(null, args);
    };

    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };

  debouncedFn.cancel = function () {
    if (timeout) {
      clearTimeout(timeout);
      timeout = null;
    }
  };

  return debouncedFn;
}

var DebounceManager = /*#__PURE__*/function () {
  function DebounceManager() {
    (0, _classCallCheck2.default)(this, DebounceManager);
    (0, _defineProperty2.default)(this, "debounceCache", {});
  }

  (0, _createClass2.default)(DebounceManager, [{
    key: "runWithDebounce",
    value:
    /**
     * Dynamically debounce and cache a debounced version of a function at the time of calling that function. This avoids
     * managing debounced version of functions locally.
     *
     * In other words, debounce usually works by debouncing based on
     * referential identity of a function. This works by comparing provided function names.
     *
     * This also has the ability to short-circuit a debounce all-together, if no wait
     * time is provided.
     *
     * Assumption:
     * Functions are debounced on a combination of unique function name and wait times. So debouncing won't work on
     * subsequent calls with different wait times or different functions. That also means that the debounce manager
     * can be used for different functions in parallel, and keep the two functions debounced separately.
     *
     * @param {number} wait Milliseconds to debounce. Executes immediately if falsey.
     * @param {function} fn Function to debounce
     * @param {function} functionName Name of function to debounce, used to create a unique key
     * @param {...any} parameters Parameters to pass to function
     */
    function runWithDebounce(wait, functionName, fn) {
      for (var _len = arguments.length, parameters = new Array(_len > 3 ? _len - 3 : 0), _key = 3; _key < _len; _key++) {
        parameters[_key - 3] = arguments[_key];
      }

      if (!wait) {
        return fn.apply(void 0, parameters);
      }

      var key = "".concat(functionName, "|").concat(wait.toString());
      var debounced = this.debounceCache[key];

      if (!debounced) {
        this.debounceCache[key] = debounce(fn, wait);
        debounced = this.debounceCache[key];
      }

      debounced.apply(void 0, parameters);
    }
    /**
     * Cancels existing debounced function calls.
     *
     * This will cancel any debounced function call, regardless of the debounce length that was provided.
     *
     * For example, making the following series of calls will create multiple debounced functions, because
     * they are cached by a combination of unique name and debounce length.
     *
     * runWithDebounce(1000, "_updateSearchResults", this._updateSearchResults)
     * runWithDebounce(500, "_updateSearchResults", this._updateSearchResults)
     * runWithDebounce(1000, "_updateSearchResults", this._updateSearchResults)
     *
     * Calling the following will cancel all of those, if they have not yet executed:
     *
     * cancelByName("_updateSearchResults")
     *
     * @param {string} functionName The name of the function that was debounced. This needs to match exactly what was provided
     * when runWithDebounce was called originally.
     */

  }, {
    key: "cancelByName",
    value: function cancelByName(functionName) {
      Object.entries(this.debounceCache).filter(function (_ref) {
        var _ref2 = (0, _slicedToArray2.default)(_ref, 1),
            cachedKey = _ref2[0];

        return cachedKey.startsWith("".concat(functionName, "|"));
      }) // eslint-disable-next-line no-unused-vars
      .forEach(function (_ref3) {
        var _ref4 = (0, _slicedToArray2.default)(_ref3, 2),
            _ = _ref4[0],
            cachedValue = _ref4[1];

        return cachedValue.cancel();
      });
    }
  }]);
  return DebounceManager;
}();
/**
 * Perform a standard debounce
 *
 * @param {number} wait Milliseconds to debounce. Executes immediately if falsey.
 * @param {function} fn Function to debounce
 */


DebounceManager.debounce = function (wait, fn) {
  return debounce(fn, wait);
};

var _default = DebounceManager;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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