import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _objectSpread from "@babel/runtime/helpers/objectSpread2";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { map, reduce, mapValues, has, get, keys, pickBy } from 'lodash';
var OPERANDS_IN_RANGE = 2;
var operators = {
  gt: '>',
  gte: '>=',
  lte: '<=',
  lt: '<'
};
var comparators = {
  gt: 'boolean gt(Supplier s, def v) {return s.get() > v}',
  gte: 'boolean gte(Supplier s, def v) {return s.get() >= v}',
  lte: 'boolean lte(Supplier s, def v) {return s.get() <= v}',
  lt: 'boolean lt(Supplier s, def v) {return s.get() < v}'
};
var dateComparators = {
  gt: 'boolean gt(Supplier s, def v) {return s.get().toInstant().isAfter(Instant.parse(v))}',
  gte: 'boolean gte(Supplier s, def v) {return !s.get().toInstant().isBefore(Instant.parse(v))}',
  lte: 'boolean lte(Supplier s, def v) {return !s.get().toInstant().isAfter(Instant.parse(v))}',
  lt: 'boolean lt(Supplier s, def v) {return s.get().toInstant().isBefore(Instant.parse(v))}'
};
/**
 * An interface for all possible range filter params
 * It is similar, but not identical to estypes.QueryDslRangeQuery
 * @public
 */

var hasRangeKeys = function hasRangeKeys(params) {
  return Boolean(keys(params).find(function (key) {
    return ['gte', 'gt', 'lte', 'lt', 'from', 'to'].includes(key);
  }));
};

/**
 * @param filter
 * @returns `true` if a filter is an `RangeFilter`
 *
 * @public
 */
export var isRangeFilter = function isRangeFilter(filter) {
  return has(filter, 'query.range');
};
/**
 *
 * @param filter
 * @returns `true` if a filter is a scripted `RangeFilter`
 *
 * @public
 */

export var isScriptedRangeFilter = function isScriptedRangeFilter(filter) {
  var params = get(filter, 'query.script.script.params', {});
  return hasRangeKeys(params);
};
/**
 * @internal
 */

export var getRangeFilterField = function getRangeFilterField(filter) {
  return filter.query.range && Object.keys(filter.query.range)[0];
};

var formatValue = function formatValue(params) {
  return map(params, function (val, key) {
    return get(operators, key) + val;
  }).join(' ');
};
/**
 * Creates a filter where the value for the given field is in the given range
 * params should be an object containing `lt`, `lte`, `gt`, and/or `gte`
 *
 * @param field
 * @param params
 * @param indexPattern
 * @param formattedValue
 * @returns
 *
 * @public
 */


export var buildRangeFilter = function buildRangeFilter(field, params, indexPattern, formattedValue) {
  params = mapValues(params, function (value) {
    return field.type === 'number' ? parseFloat(value) : value;
  });
  if ('gte' in params && 'gt' in params) throw new Error('gte and gt are mutually exclusive');
  if ('lte' in params && 'lt' in params) throw new Error('lte and lt are mutually exclusive');
  var totalInfinite = ['gt', 'lt'].reduce(function (acc, op) {
    var key = op in params ? op : "".concat(op, "e");
    var isInfinite = Math.abs(get(params, key)) === Infinity;

    if (isInfinite) {
      acc++; // @ts-ignore

      delete params[key];
    }

    return acc;
  }, 0);

  var meta = _objectSpread({
    index: indexPattern.id,
    params: {},
    field: field.name
  }, formattedValue ? {
    formattedValue: formattedValue
  } : {});

  if (totalInfinite === OPERANDS_IN_RANGE) {
    return {
      meta: meta,
      query: {
        match_all: {}
      }
    };
  } else if (field.scripted) {
    var scr = getRangeScript(field, params); // TODO: type mismatch enforced

    scr.script.params.value = formatValue(scr.script.params);
    return {
      meta: meta,
      query: {
        script: scr
      }
    };
  } else {
    return {
      meta: meta,
      query: {
        range: _defineProperty({}, field.name, params)
      }
    };
  }
};
/**
 * @internal
 */

export var getRangeScript = function getRangeScript(field, params) {
  var knownParams = mapValues(pickBy(params, function (val, key) {
    return key in operators;
  }), function (value) {
    return field.type === 'number' && typeof value === 'string' ? parseFloat(value) : value;
  });
  var script = map(knownParams, function (_, key) {
    return '(' + field.script + ')' + get(operators, key) + key;
  }).join(' && '); // We must wrap painless scripts in a lambda in case they're more than a simple expression

  if (field.lang === 'painless') {
    var comp = field.type === 'date' ? dateComparators : comparators;
    var currentComparators = reduce(knownParams, function (acc, val, key) {
      return acc.concat(get(comp, key));
    }, []).join(' ');
    var comparisons = map(knownParams, function (val, key) {
      return "".concat(key, "(() -> { ").concat(field.script, " }, params.").concat(key, ")");
    }).join(' && ');
    script = "".concat(currentComparators).concat(comparisons);
  }

  return {
    script: {
      source: script,
      params: knownParams,
      lang: field.lang
    }
  };
};