'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var react = require('react');

function useCombinedRefs(...refs) {
  return react.useMemo(() => node => {
    refs.forEach(ref => ref(node));
  }, // eslint-disable-next-line react-hooks/exhaustive-deps
  refs);
}

// https://github.com/facebook/react/blob/master/packages/shared/ExecutionEnvironment.js
const canUseDOM = typeof window !== 'undefined' && typeof window.document !== 'undefined' && typeof window.document.createElement !== 'undefined';

/**
 * A hook that resolves to useEffect on the server and useLayoutEffect on the client
 * @param callback {function} Callback function that is invoked when the dependencies of the hook change
 */

const useIsomorphicLayoutEffect = canUseDOM ? react.useLayoutEffect : react.useEffect;

function useInterval() {
  const intervalRef = react.useRef(null);
  const set = react.useCallback((listener, duration) => {
    intervalRef.current = setInterval(listener, duration);
  }, []);
  const clear = react.useCallback(() => {
    if (intervalRef.current !== null) {
      clearInterval(intervalRef.current);
      intervalRef.current = null;
    }
  }, []);
  return [set, clear];
}

function useLazyMemo(callback, dependencies) {
  const valueRef = react.useRef();
  return react.useMemo(() => {
    const newValue = callback(valueRef.current);
    valueRef.current = newValue;
    return newValue;
  }, // eslint-disable-next-line react-hooks/exhaustive-deps
  [...dependencies]);
}

function useNodeRef() {
  const node = react.useRef(null);
  const setNodeRef = react.useCallback(element => {
    node.current = element;
  }, []);
  return [node, setNodeRef];
}

let ids = {};
function useUniqueId(prefix, value) {
  return react.useMemo(() => {
    if (value) {
      return value;
    }

    const id = ids[prefix] == null ? 0 : ids[prefix] + 1;
    ids[prefix] = id;
    return `${prefix}-${id}`;
  }, [prefix, value]);
}

function createAdjustmentFn(modifier) {
  return (object, ...adjustments) => {
    return adjustments.reduce((accumulator, adjustment) => {
      const entries = Object.entries(adjustment);

      for (const [key, valueAdjustment] of entries) {
        const value = accumulator[key];

        if (value != null) {
          accumulator[key] = value + modifier * valueAdjustment;
        }
      }

      return accumulator;
    }, { ...object
    });
  };
}

const add = /*#__PURE__*/createAdjustmentFn(1);
const subtract = /*#__PURE__*/createAdjustmentFn(-1);

const CSS = /*#__PURE__*/Object.freeze({
  Translate: {
    toString(transform) {
      if (!transform) {
        return;
      }

      const {
        x,
        y
      } = transform;
      return `translate3d(${x ? Math.round(x) : 0}px, ${y ? Math.round(y) : 0}px, 0)`;
    }

  },
  Scale: {
    toString(transform) {
      if (!transform) {
        return;
      }

      const {
        scaleX,
        scaleY
      } = transform;
      return `scaleX(${scaleX}) scaleY(${scaleY})`;
    }

  },
  Transform: {
    toString(transform) {
      if (!transform) {
        return;
      }

      return [CSS.Translate.toString(transform), CSS.Scale.toString(transform)].join(' ');
    }

  },
  Transition: {
    toString({
      property,
      duration,
      easing
    }) {
      return `${property} ${duration}ms ${easing}`;
    }

  }
});

exports.CSS = CSS;
exports.add = add;
exports.canUseDOM = canUseDOM;
exports.subtract = subtract;
exports.useCombinedRefs = useCombinedRefs;
exports.useInterval = useInterval;
exports.useIsomorphicLayoutEffect = useIsomorphicLayoutEffect;
exports.useLazyMemo = useLazyMemo;
exports.useNodeRef = useNodeRef;
exports.useUniqueId = useUniqueId;
//# sourceMappingURL=utilities.cjs.development.js.map
