import _classCallCheck from "@babel/runtime/helpers/classCallCheck";
import _createClass from "@babel/runtime/helpers/createClass";
import { createBrowserHistory as createHistory } from "history";
import queryString from "./queryString";

function isNumericString(num) {
  return !isNaN(num);
}

function toSingleValue(val) {
  return Array.isArray(val) ? val[val.length - 1] : val;
}

function toSingleValueInteger(num) {
  return toInteger(toSingleValue(num));
}

function toInteger(num) {
  if (!isNumericString(num)) return;
  return parseInt(num, 10);
}

function parseFiltersFromQueryParams(queryParams) {
  return queryParams.filters;
}

function parseCurrentFromQueryParams(queryParams) {
  return toSingleValueInteger(queryParams.current);
}

function parseSearchTermFromQueryParams(queryParams) {
  return toSingleValue(queryParams.q);
}

function parseSortFromQueryParams(queryParams) {
  var sortField = toSingleValue(queryParams["sort-field"]);
  var sortDirection = toSingleValue(queryParams["sort-direction"]);
  if (sortField) return [sortField, sortDirection];
  return [];
}

function parseSizeFromQueryParams(queryParams) {
  return toSingleValueInteger(queryParams.size);
}

function paramsToState(queryParams) {
  var state = {
    current: parseCurrentFromQueryParams(queryParams),
    filters: parseFiltersFromQueryParams(queryParams),
    searchTerm: parseSearchTermFromQueryParams(queryParams),
    resultsPerPage: parseSizeFromQueryParams(queryParams),
    sortField: parseSortFromQueryParams(queryParams)[0],
    sortDirection: parseSortFromQueryParams(queryParams)[1]
  };
  return Object.keys(state).reduce(function (acc, key) {
    var value = state[key];
    if (value) acc[key] = value;
    return acc;
  }, {});
}

function stateToParams(_ref) {
  var searchTerm = _ref.searchTerm,
      current = _ref.current,
      filters = _ref.filters,
      resultsPerPage = _ref.resultsPerPage,
      sortDirection = _ref.sortDirection,
      sortField = _ref.sortField;
  var params = {};
  if (current > 1) params.current = current;
  if (searchTerm) params.q = searchTerm;
  if (resultsPerPage) params.size = resultsPerPage;

  if (filters && filters.length > 0) {
    params["filters"] = filters;
  }

  if (sortField) {
    params["sort-field"] = sortField;
    params["sort-direction"] = sortDirection;
  }

  return params;
}

function stateToQueryString(state) {
  return queryString.stringify(stateToParams(state));
}
/**
 * The URL Manager is responsible for synchronizing state between
 * SearchDriver and the URL. There are 3 main cases we handle when
 * synchronizing:
 *
 * 1. When the app loads, SearchDriver will need to
 * read the current state from the URL, in order to perform the search
 * expressed by the query string. `getStateFromURL` is used for this case.
 *
 * 2. When the URL changes as a result of `pushState` or `replaceState`,
 * SearchDriver will need to be notified and given the updated state, so that
 * it can re-run the current search. `onURLStateChange` is used for this case.
 *
 * 3. When state changes internally in the SearchDriver, as a result of an
 * Action, it will need to notify the URLManager of the change. `pushStateToURL`
 * is used for this case.
 */


var URLManager = /*#__PURE__*/function () {
  function URLManager() {
    _classCallCheck(this, URLManager);

    this.history = createHistory();
    this.lastPushSearchString = "";
  }
  /**
   * Parse the current URL into application state
   *
   * @return {Object} - The parsed state object
   */


  _createClass(URLManager, [{
    key: "getStateFromURL",
    value: function getStateFromURL() {
      return paramsToState(queryString.parse(this.history.location.search));
    }
    /**
     * Push the current state of the application to the URL
     *
     * @param {Object} state - The entire current state from the SearchDriver
     * @param {boolean} options
     * @param {boolean} options.replaceUrl - When pushing state to the URL, use history 'replace'
     * rather than 'push' to avoid adding a new history entry
     */

  }, {
    key: "pushStateToURL",
    value: function pushStateToURL(state) {
      var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
          _ref2$replaceUrl = _ref2.replaceUrl,
          replaceUrl = _ref2$replaceUrl === void 0 ? false : _ref2$replaceUrl;

      var searchString = stateToQueryString(state);
      this.lastPushSearchString = searchString;
      var navigationFunction = replaceUrl ? this.history.replace : this.history.push;
      navigationFunction({
        search: "?".concat(searchString)
      });
    }
    /**
     * Add an event handler to be executed whenever state is pushed to the URL
     *
     * @callback requestCallback
     * @param {Object} state - Updated application state parsed from the new URL
     *
     * @param {requestCallback} callback
     */

  }, {
    key: "onURLStateChange",
    value: function onURLStateChange(callback) {
      var _this = this;

      this.unlisten = this.history.listen(function (location) {
        // If this URL is updated as a result of a pushState request, we don't
        // want to notify that the URL changed.
        if ("?".concat(_this.lastPushSearchString) === location.search) return; // Once we've decided to return based on lastPushSearchString, reset
        // it so that we don't break back / forward button.

        _this.lastPushSearchString = "";
        callback(paramsToState(queryString.parse(location.search)));
      });
    }
  }, {
    key: "tearDown",
    value: function tearDown() {
      this.unlisten();
    }
  }]);

  return URLManager;
}();

export { URLManager as default };
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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